/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <assert.h>
#include <float.h>

#include "referencesystem.h"
#include "bestfitcircle.h"

TBestFitCircle::TBestFitCircle(void)
:d_Position(0,0,0),
 d_Axis(0,0,1),
 d_Radius(0),
 d_FormError(0)
{
}

TBestFitCircle::~TBestFitCircle(void)
{

}


bool TBestFitCircle::Fit(
	const TVector3						&Axis,
	const int 							NumPoints,
	const TVector3 						*Pnts)
{
	int                                 IterationCntr;
	int									Cntr;
    TVector3							InitSphereCenter;
	TVector3 							DifferenceVector;
	TVector3							AverageErrorVector;
	TVector3							InitialPosition;
	double								DifferenceLength;
	double								InvDifferenceLength;
	double								InvNumberOfPoints;
	TVector3 							*Points;
	TReferenceSystem 					Ref;
	static const int					MAX_ITERATIONS(10000);
	static const double					ZERO_EPSILON(0.000000001);


	if (NumPoints < 3)
	{
		return false;
	}

	assert(Pnts);
	
	InvNumberOfPoints = 1.0 / static_cast<double>(NumPoints);

	if(Axis.Length() < ZERO_EPSILON)
	{
		return false;
	}

	d_Axis = Axis;
	d_Axis.Normal();

	Ref.Skew1(d_Axis, TReferenceSystem::ZPLUS);

	Points = new TVector3[NumPoints];
	
	for (Cntr = 0; Cntr < NumPoints; ++Cntr)
	{
		Points[Cntr] = Ref.FromWorld(Pnts[Cntr]);
	}

	InitialPosition = TVector3(0,0,0);
	for (Cntr = 0; Cntr < NumPoints; Cntr++)
	{
		InitialPosition += Points[Cntr];
	}

	InitialPosition *= InvNumberOfPoints;

	d_Position = InitialPosition;

	for (IterationCntr = 0; IterationCntr < MAX_ITERATIONS; ++IterationCntr)
	{
		InitSphereCenter = d_Position;

		d_Radius = 0.0;
		AverageErrorVector = TVector3(0,0,0);

		for (Cntr = 0; Cntr < NumPoints; Cntr++)
		{
			DifferenceVector = Points[Cntr] - d_Position;
            DifferenceVector.z = 0.0;
            
			DifferenceLength = DifferenceVector.Length();

			if (DifferenceLength > ZERO_EPSILON)
			{
				d_Radius += DifferenceLength;
				InvDifferenceLength = static_cast<double>(1.0)/DifferenceLength;
				AverageErrorVector -= InvDifferenceLength * DifferenceVector;
			}
		}

		d_Radius *= InvNumberOfPoints;
		AverageErrorVector *= InvNumberOfPoints;

		d_Position = InitialPosition + d_Radius * AverageErrorVector;

		DifferenceVector = d_Position - InitSphereCenter;
        DifferenceVector.z = 0.0;

		if ((fabs(DifferenceVector.x) < ZERO_EPSILON) &&
		        (fabs(DifferenceVector.y) < ZERO_EPSILON))
		{
			break;
		}
	}

	delete[] Points;

	d_Position = Ref.ToWorld(d_Position);

	this->CalcFormError(NumPoints,Pnts);

	return true;
}

void TBestFitCircle::CalcFormError(
	const int 							NumPoints,
	const TVector3 						*Points)
{
	int 								Cntr;
	double 								Rad,RadMax(0),RadMin(0);
	TVector3 							Pr;

	for(Cntr = 0; Cntr < NumPoints; ++Cntr)
	{
		Pr = d_Axis * (d_Position - Points[Cntr]);

		Rad = Pr.Length();

		if (Cntr == 0)
		{
			RadMax = RadMin = Rad;
		}
		if (Rad > RadMax)
		{
			RadMax = Rad;
		}
		if (Rad < RadMin)
		{
			RadMin = Rad;
		}
	}

	d_FormError = RadMax - RadMin;
}
