/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <assert.h>
#include <float.h>
#include <string.h>

#include "referencesystem.h"
#include "vector3.h"
#include "bestfitline.h"

TBestFitLine::TBestFitLine(void)
:d_position(0,0,0),
 d_axis(0,0,1),
 d_length(0),
 d_form_error(0)
{
}

TBestFitLine::~TBestFitLine(void)
{
}

bool TBestFitLine::Fit(
	const int 							number_of_points,
	const TVector3 						*points)
{
	int 								Cntr;
	double 								InvQuantity;
	double 								Sum[6];
	TVector3 							Pnt;
	TVector3 							Diff;
	double 								eigen_values[3];
	double 								eigen_vectors[9];
	double 								MinL, MaxL, Len;
	static const double					ZERO_EPSILON(0.000001);

	d_position = TVector3(0, 0, 0);
	d_axis = TVector3(0,0,1);
	d_length = 0;
	d_form_error = 0;

	if(number_of_points < 2)
	{
		return false;
	}
	
	if(number_of_points == 2)
	{
		d_position = (points[1] + points[0]) / 2.0;
		d_axis = points[1] - points[0];

		if(d_axis.Length() > ZERO_EPSILON)
		{
			d_axis.Normal();
			return true;
		}
		
		return false;
	}
	
	for (Cntr = 0; Cntr < number_of_points; Cntr++)
	{
		d_position += points[Cntr];
	}

	InvQuantity = 1.0 / static_cast<double> (number_of_points);
	d_position *= InvQuantity;

	// Create error matrix
	memset(Sum, 0, sizeof(Sum));

	for (Cntr = 0; Cntr < number_of_points; Cntr++)
	{
		Diff = points[Cntr] - d_position;

		Sum[0] += Diff.x * Diff.x;
		Sum[1] += Diff.x * Diff.y;
		Sum[2] += Diff.y * Diff.y;
		Sum[3] += Diff.x * Diff.z;
		Sum[4] += Diff.y * Diff.z;
		Sum[5] += Diff.z * Diff.z;
	}

	// Find Eigen values from data
	Get_Eigen_Values(Sum, 3, eigen_vectors, eigen_values);

	// isotropic case (infinite number of directions)
	if (eigen_values[0] == eigen_values[1] && eigen_values[0]
	        == eigen_values[2])
	{
		return false;
	}

	// regular case
	// The first three (0,1,2) values represent the best fit line
	// The last three values (6,7,8) represent the best fit plane

	d_axis.x = eigen_vectors[0];
	d_axis.y = eigen_vectors[1];
	d_axis.z = eigen_vectors[2];

	// Find the length

	MinL = 0;
	MaxL = 0;

	for (Cntr = 0; Cntr < number_of_points; ++Cntr)
	{
		Pnt = (points[Cntr] - d_position);
		Len = d_axis.i * Pnt.x + d_axis.j * Pnt.y + d_axis.k * Pnt.z;

		if (Len < MinL)
		{
			MinL = Len;
		}
		else if (Len > MaxL)
		{
			MaxL = Len;
		}
	}

	d_length = MaxL - MinL;

	this->Calculate_Form_Error(number_of_points,points);

	return true;
}

void TBestFitLine::Calculate_Form_Error(
	int 								number_of_points,
	const TVector3 						*points)
{
	int 								Cntr;
	TVector3 							Diff;
	TVector3 							PRad, PRadMax;
	TVector3 							PRadMax1, PRadMax2, PRadMax3;
	TVector3 							MaxPntAxis, MaxPntCenter;
	int 								MaxPnt1Index(0);
	int									MaxPnt2Index(1);
	int									MaxPnt3Index(2);
	double 								MaxPntDiameter;
	double 								Rad, RadMax;
	static const double 				ZERO_EPSILON(0.000000000001);
	
	if(number_of_points < 3)
	{
		d_form_error = 0.0;
		return;
	}

	assert(points);

	// Find farthest point from center.  This represents LS form

	for (Cntr = 0; Cntr < number_of_points; ++Cntr)
	{
		PRad = d_axis * (points[Cntr] - d_position) * d_axis;

		Rad = PRad.Length();

		if (Cntr == 0)
		{
			RadMax = Rad;
			PRadMax = PRad;
			MaxPnt1Index = Cntr;
		}
		else
		{
			if (Rad > RadMax)
			{
				RadMax = Rad;
				PRadMax = PRad;
				MaxPnt1Index = Cntr;
			}
		}
	}

	PRadMax1 = points[MaxPnt1Index];

	// Find farthest point from PRadMax1

	for (Cntr = 0; Cntr < number_of_points; ++Cntr)
	{
		PRad = d_axis * (points[Cntr] - points[MaxPnt1Index]) * d_axis;

		Rad = PRad.Length();

		if (Cntr == 0)
		{
			RadMax = Rad;
			PRadMax = PRad;
			MaxPnt2Index = Cntr;

		}
		else
		{
			if (Rad > RadMax)
			{
				RadMax = Rad;
				PRadMax = PRad;
				MaxPnt2Index = Cntr;
			}
		}
	}

	PRadMax2 = points[MaxPnt2Index];

	MaxPntAxis = PRadMax2 - PRadMax1;
	MaxPntAxis = d_axis * MaxPntAxis * d_axis;

	MaxPntCenter = PRadMax2 + PRadMax1;
	MaxPntCenter /= 2;
	MaxPntDiameter = MaxPntAxis.Length();

	d_form_error = MaxPntDiameter;

	if (MaxPntAxis.Length() > ZERO_EPSILON)
	{
		MaxPntAxis.Normal();
	}
	else
	{
		return;
	}

	// Find farthest point from MaxPntAxis

	for (Cntr = 0; Cntr < number_of_points; ++Cntr)
	{
		PRad = d_axis * (points[Cntr] - MaxPntCenter) * d_axis;
		PRad = MaxPntAxis * PRad * MaxPntAxis;

		Rad = PRad.Length();

		if (Cntr == 0)
		{
			RadMax = Rad;
			PRadMax = PRad;
			MaxPnt3Index = Cntr;
		}
		else
		{
			if (Rad > RadMax)
			{
				RadMax = Rad;
				PRadMax = PRad;
				MaxPnt3Index = Cntr;
			}
		}
	}

	PRadMax3 = points[MaxPnt3Index];

	PRad = d_axis * (PRadMax3 - MaxPntCenter);
	Rad = PRad.Length();

	if (Rad > (MaxPntDiameter / 2))
	{
		// Point is outside MaxPntDiameter.  Must construct a containing circle
		Create_Circle(points[MaxPnt1Index],points[MaxPnt2Index],points[MaxPnt3Index],d_axis,&MaxPntCenter,&MaxPntDiameter);
	}

	d_form_error = MaxPntDiameter;
}

void TBestFitLine::Get_Eigen_Values(
	const double 						*mat,
	const int 							n,
	double 								*eigen_vectors,
	double 								*eigen_values)
{
	static const int 					MAX_ITER(100);
	static const double 				ZERO_EPSILON(0.000000000001);

	// number of entries in mat
	int nn = (n * (n + 1)) / 2;

	// copy matrix
	double *a = new double[nn];
	int ij;
	for (ij = 0; ij < nn; ij++)
	{
		a[ij] = mat[ij];
	}
	// Fortran-porting
	a--;

	// init diagonalization matrix as the unit matrix
	double *v = new double[n * n];
	ij = 0;
	int i;
	for (i = 0; i < n; i++)
	{
		for (int j = 0; j < n; j++)
		{
			if (i == j)
			{
				v[ij++] = 1.0;
			}
			else
			{
				v[ij++] = 0.0;
			}
		}
	}
	// Fortran-porting
	v--;

	// compute weight of the non diagonal terms
	ij = 1;
	double a_norm = 0.0;
	for (i = 1; i <= n; i++)
	{
		for (int j = 1; j <= i; j++)
		{
			if (i != j)
			{
				double a_ij = a[ij];
				a_norm += a_ij * a_ij;
			}
			ij++;
		}
	}

	if (a_norm != 0.0)
	{
		double a_normEPS = a_norm * ZERO_EPSILON;
		double thr = a_norm;

		// rotations
		int nb_iter = 0;
		while (thr > a_normEPS && nb_iter < MAX_ITER)
		{
			nb_iter++;
			double thr_nn = thr / nn;

			for (int l = 1; l < n; l++)
			{
				for (int m = l + 1; m <= n; m++)
				{
					// compute sinx and cosx
					int lq = (l * l - l) / 2;
					int mq = (m * m - m) / 2;

					int lm = l + mq;
					double a_lm = a[lm];
					double a_lm_2 = a_lm * a_lm;

					if (a_lm_2 < thr_nn)
					{
						continue;
					}

					int ll = l + lq;
					int mm = m + mq;
					double a_ll = a[ll];
					double a_mm = a[mm];

					double delta = a_ll - a_mm;

					double x;
					if (delta == 0.0)
					{
						x = (double) -3.1415926535897932384626433832795 / 4;
					}
					else
					{
						x = (double) (-atan((a_lm + a_lm) / delta) / 2.0);
					}

					double sinx = sin(x);
					double cosx = cos(x);
					double sinx_2 = sinx * sinx;
					double cosx_2 = cosx * cosx;
					double sincos = sinx * cosx;

					// rotate L and M columns
					int ilv = n * (l - 1);
					int imv = n * (m - 1);

					int i;
					for (i = 1; i <= n; i++)
					{
						if ((i != l) && (i != m))
						{
							int iq = (i * i - i) / 2;

							int im;
							if (i < m)
							{
								im = i + mq;
							}
							else
							{
								im = m + iq;
							}

							double a_im = a[im];

							int il;
							if (i < l)
							{
								il = i + lq;
							}
							else
							{
								il = l + iq;
							}
							double a_il = a[il];

							a[il] = a_il * cosx - a_im * sinx;
							a[im] = a_il * sinx + a_im * cosx;
						}

						ilv++;
						imv++;

						double v_ilv = v[ilv];
						double v_imv = v[imv];

						v[ilv] = cosx * v_ilv - sinx * v_imv;
						v[imv] = sinx * v_ilv + cosx * v_imv;
					}

					x = a_lm * sincos;
					x += x;

					a[ll] = a_ll * cosx_2 + a_mm * sinx_2 - x;
					a[mm] = a_ll * sinx_2 + a_mm * cosx_2 + x;
					a[lm] = 0.0;

					thr = fabs(thr - a_lm_2);
				}
			}
		}
	}

	// convert indices and copy eigen values
	a++;
	for (i = 0; i < n; i++)
	{
		int k = i + (i * (i + 1)) / 2;
		eigen_values[i] = a[k];
	}

	delete[] a;

	// sort eigen values and vectors
	int *index = new int[n];
	for (i = 0; i < n; i++)
	{
		index[i] = i;
	}

	for (i = 0; i < (n - 1); i++)
	{
		double x = eigen_values[i];
		int k = i;

		for (int j = i + 1; j < n; j++)
		{
			if (x < eigen_values[j])
			{
				k = j;
				x = eigen_values[j];
			}
		}
		eigen_values[k] = eigen_values[i];
		eigen_values[i] = x;

		int jj = index[k];
		index[k] = index[i];
		index[i] = jj;
	}

	// save eigen vectors
	v++; // back to C++
	ij = 0;
	for (int k = 0; k < n; k++)
	{
		int ik = index[k] * n;
		for (int i = 0; i < n; i++)
		{
			eigen_vectors[ij++] = v[ik++];
		}
	}

	delete[] v;
	delete[] index;
}

bool TBestFitLine::Create_Circle(
	const TVector3 						p1,
	const TVector3 						p2,
	const TVector3 						p3,
	const TVector3 						axis,
	TVector3 							*center,
	double 								*diameter) const
{
	TReferenceSystem 					Ref;
	TVector3 							ProPnt1, ProPnt2, ProPnt3;
	TVector3 							AxisMa, AxisMb, AlignAxis;
	TVector3 							Diff;
	double 								Ma, Mb;
	double 								X, Y;
	static const double					ZERO_EPSILON(0.000001);

	if (axis.Length() < ZERO_EPSILON)
	{
		return false;
	}

	Ref.Skew1(axis, TReferenceSystem::ZPLUS);

	ProPnt1 = Ref.FromWorld(p1);
	ProPnt2 = Ref.FromWorld(p2);
	ProPnt3 = Ref.FromWorld(p3);

	AxisMa = ProPnt1 - ProPnt2;
	AxisMb = ProPnt3 - ProPnt2;

	AxisMa.z = 0;
	AxisMb.z = 0;

	if (AxisMa.Length() < ZERO_EPSILON)
	{
		return false;
	}

	if (AxisMb.Length() < ZERO_EPSILON)
	{
		return false;
	}

	AlignAxis = AxisMa + AxisMb;
	AlignAxis.Normal();

	Ref.Skew2(AlignAxis, TReferenceSystem::XPLUS, TReferenceSystem::ZPLUS);

	ProPnt1 = Ref.FromWorld(p1);
	ProPnt2 = Ref.FromWorld(p2);
	ProPnt3 = Ref.FromWorld(p3);

	AxisMa = ProPnt2 - ProPnt1;
	AxisMb = ProPnt3 - ProPnt2;

	Ma = AxisMa.y / AxisMa.x;
	Mb = AxisMb.y / AxisMb.x;

	X = (Ma * Mb * (ProPnt1.y - ProPnt3.y) + Mb * (ProPnt1.x + ProPnt2.x) - Ma
	        * (ProPnt2.x + ProPnt3.x)) / (2 * (Mb - Ma));

	if (fabs(Ma) > ZERO_EPSILON)
	{
		Y = -(X - (ProPnt1.x + ProPnt2.x) / 2) / Ma + (ProPnt1.y + ProPnt2.y) / 2;
	}
	else
	{
		Y = -(X - (ProPnt2.x + ProPnt3.x) / 2) / Mb + (ProPnt2.y + ProPnt3.y) / 2;
	}

	center->x = X;
	center->y = Y;

	center->z = (ProPnt1.z + ProPnt2.z + ProPnt3.z) / 3;

	Diff = ProPnt1 - (*center);
	Diff.z = 0;

	*diameter = Diff.Length() * 2;
	*center = Ref.ToWorld(*center);

	return true;
}



