/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QWidget>
#include <QLabel>
#include <QFont>
#include <QFontMetrics>
#include <QPen>
#include <QColor>
#include <QVector>
#include <QPaintEvent>
#include <QPainter>
#include <QRegion>
#include <QRect>
#include <cmath>
#include <assert.h>

#include "graphwidget.h"

TGraphWidget::TGraphWidget(
	const QWidget						*parent,
	Qt::WindowFlags 					flags)
: QWidget(const_cast<QWidget*>(parent),flags)
{
	d_graph_type = TGraphWidget::LINE_GRAPH;

	d_vertical_legend_title = "V Title";
	d_horizontal_legend_title = "H Title";
	d_title = "Sample Graph";
	
	d_horizontal_text_total = 8;
	d_horizontal_text_precision = 4;
	d_vertical_text_total = 8;
	d_vertical_text_precision = 4;

	d_draw_mouse_rect = false;
	d_left_mouse_down = false;
	d_left_mouse_moving = false;
	d_right_mouse_down = false;
	d_right_mouse_moving = false;
	d_show_mouse_tracking = true;
	d_enable_direct_rendering = false;
	d_y_includes_zero = true;
	d_bidirectional_y_min_value = true;

	d_tracking_text = QString("%1,   %2").arg(0.0,0,'f',5).arg(0.0,0,'f',5);

	d_draw_border = true;
	d_process_scale_to_fit = false;
	d_process_zoom_out = false;
	
	this->setMouseTracking(true);
	this->setSizePolicy(QSizePolicy::MinimumExpanding,QSizePolicy::MinimumExpanding);
	this->setAttribute(Qt::WA_OpaquePaintEvent,true);

	d_horizontal_legend_mode = TGraphWidget::DEFAULT_LEGEND_PRECISION;
	d_vertical_legend_mode = TGraphWidget::DEFAULT_LEGEND_PRECISION;
	
	d_y_min_value = 0.000005;
	d_lock_aspect = false;
}

TGraphWidget::~TGraphWidget()
{
}

int TGraphWidget::DataSetSize(
	const int							set) const
{
	assert(set < static_cast<int>(d_data_sets.size()));

	return static_cast<int>(d_data_sets[set].dataset_points.size());
}

TVector2 TGraphWidget::GetPoint(
	const int							set,
	const int							index) const
{
	assert(set < static_cast<int>(d_data_sets.size()));
	assert(index < static_cast<int>(d_data_sets[set].dataset_points.size()));

	return d_data_sets[set].dataset_points[index];
}

bool TGraphWidget::DeleteDataSet(
	const int 							index)
{
	assert(index < static_cast<int>(d_data_sets.size()));

	d_data_sets.erase(d_data_sets.begin() + index);

	return true;
}

void TGraphWidget::DeleteDataSets(void)
{
	d_data_sets.clear();
}

int TGraphWidget::CreateDataSet(
	const QColor 						color,
	const TGraphWidget::TGraphType		dataset_type)
{
	TDataSet							data_set;

	data_set.dataset_color = color;
	data_set.dataset_graph_type = dataset_type;

	d_data_sets.push_back(data_set);

	return static_cast<int>(d_data_sets.size() - 1);
}

bool TGraphWidget::AddPoint(
	const int 							set,
	const TVector2						&pnt)
{
	assert(set < static_cast<int>(d_data_sets.size()));

	d_data_sets[set].dataset_points.push_back(pnt);

	return true;
}

bool TGraphWidget::AddPoints(
	const int							set,
	const std::vector<TVector2>			&points)
{
	assert(set < static_cast<int>(d_data_sets.size()));
	
	d_data_sets[set].dataset_points.insert(d_data_sets[set].dataset_points.end(),points.begin(),points.end());
	
	return true;
}

bool TGraphWidget::SetPoints(
	const int							set,
	const std::vector<TVector2>			&points)
{
	assert(set < static_cast<int>(d_data_sets.size()));
	
	d_data_sets[set].dataset_points = points;
	
	return true;
}

bool TGraphWidget::ModifyPoint(
	const int 							set,
	const int 							index,
	const TVector2						&new_pnt)
{
	assert(set < static_cast<int>(d_data_sets.size()));
	assert(index < static_cast<int>(d_data_sets[set].dataset_points.size()));

	d_data_sets[set].dataset_points[index] = new_pnt;

	return true;
}

bool TGraphWidget::DeletePoint(
	const int 							set,
	const int 							index)
{
	assert(set < static_cast<int>(d_data_sets.size()));
	assert(index < static_cast<int>(d_data_sets[set].dataset_points.size()));
	
	d_data_sets[set].dataset_points.erase(d_data_sets[set].dataset_points.begin() + index);
	
	return true;
}

bool TGraphWidget::DeletePoints(
	const int 							set)
{
	assert(set < static_cast<int>(d_data_sets.size()));
	
	d_data_sets[set].dataset_points.clear();
	
	return true;
}

void TGraphWidget::UpdateGraph(void)
{
	if(d_viewport_rect.height() < 1 || d_viewport_rect.width() < 1)
	{
		return;
	}

	if(!d_enable_direct_rendering)
	{
		this->Draw_Background_Pixmap();
	}
	
	this->update();
}

bool TGraphWidget::PaintToDevice(
	QPainter 							*painter,
	const QRect							&rect,
	const bool							enable_border)
{
	TViewportMat						viewport_mat;
	QRect								graph_rect;
	QRect								vertical_legend_rect;
	QRect								vertical_legend_title_rect;
	QRect								horizontal_legend_rect;
	QRect								horizontal_legend_tile_rect;
	QRect								title_rect;
	QRect								viewport_rect;
	bool								draw_border;
	int									line_height;
	int									X,Y;
	
	assert(painter);
	
	QFontMetrics						font_metrics(painter->font());
	
	line_height = font_metrics.height();
	
	viewport_mat = d_viewport_mat;
	graph_rect = d_graph_rect;
	vertical_legend_rect = d_vertical_legend_rect;
	vertical_legend_title_rect = d_vertical_legend_title_rect;
	horizontal_legend_rect = d_horizontal_legend_rect;
	horizontal_legend_tile_rect = d_horizontal_legend_title_rect;
	title_rect = d_title_rect;
	viewport_rect = d_viewport_rect;
	draw_border = d_draw_border;
	
	d_draw_border = enable_border;

	d_viewport_rect = rect;
	
	X = d_viewport_rect.width();
	Y = d_viewport_rect.height();

	d_vertical_legend_title_rect.setLeft(d_viewport_rect.left());
	d_vertical_legend_title_rect.setTop(d_viewport_rect.top());
	d_vertical_legend_title_rect.setRight(line_height * 2 + d_viewport_rect.left());
	d_vertical_legend_title_rect.setBottom(Y - line_height * 2);

	d_horizontal_legend_title_rect.setLeft(line_height * 2 + d_viewport_rect.left());
	d_horizontal_legend_title_rect.setTop(Y - line_height * 2);
	d_horizontal_legend_title_rect.setRight(X);
	d_horizontal_legend_title_rect.setBottom(Y);

	d_title_rect.setLeft(d_vertical_legend_title_rect.right() + 5 * line_height);
	d_title_rect.setTop(d_viewport_rect.top());
	d_title_rect.setRight(X);
	d_title_rect.setBottom(line_height * 2 + d_viewport_rect.top());

	d_vertical_legend_rect.setLeft(d_vertical_legend_title_rect.right());
	d_vertical_legend_rect.setTop(d_viewport_rect.top());
	d_vertical_legend_rect.setRight(d_vertical_legend_title_rect.right() + 4 * line_height);
	d_vertical_legend_rect.setBottom(d_horizontal_legend_title_rect.top());

	d_horizontal_legend_rect.setLeft(d_vertical_legend_rect.right());
	d_horizontal_legend_rect.setTop(d_horizontal_legend_title_rect.top() - line_height * 2);
	d_horizontal_legend_rect.setRight(X);
	d_horizontal_legend_rect.setBottom(d_horizontal_legend_title_rect.top());

	d_graph_rect.setLeft(d_vertical_legend_rect.right());
	d_graph_rect.setTop(d_title_rect.bottom());
	d_graph_rect.setRight(X - line_height * 2);
	d_graph_rect.setBottom(d_horizontal_legend_rect.top());

	d_vertical_legend_title_rect.setTop(d_graph_rect.top());
	d_vertical_legend_title_rect.setBottom(d_graph_rect.bottom());

	d_horizontal_legend_title_rect.setLeft(d_graph_rect.left());
	d_horizontal_legend_title_rect.setRight(d_graph_rect.right());
	
	this->Direct_Render(painter);

	d_viewport_mat = viewport_mat;
	d_graph_rect = graph_rect;
	d_vertical_legend_rect = vertical_legend_rect; 
	d_vertical_legend_title_rect = vertical_legend_title_rect;
	d_horizontal_legend_rect = horizontal_legend_rect;
	d_horizontal_legend_title_rect = horizontal_legend_tile_rect;
	d_title_rect = title_rect;
	d_viewport_rect= viewport_rect;
	d_draw_border = draw_border;
	
	return true;
}

bool TGraphWidget::PaintToDevice_GraphOnly(
	QPainter 							*painter,
	const QRect							&rect,
	const bool							enable_border)
{
	TViewportMat						viewport_mat;
	QRect								graph_rect;
	QRect								vertical_legend_rect;
	QRect								vertical_legend_title_rect;
	QRect								horizontal_legend_rect;
	QRect								horizontal_legend_tile_rect;
	QRect								title_rect;
	QRect								viewport_rect;
	QString								vertical_legend_title;
	QString								horizontal_legend_title;
	QString								title;
	bool								draw_border;
	int									line_height;
	int									X,Y;
	static const int					border(6);
	
	assert(painter);
	QFontMetrics						font_metrics(painter->font());
	
	line_height = font_metrics.height();

	viewport_mat = d_viewport_mat;
	graph_rect = d_graph_rect;
	vertical_legend_rect = d_vertical_legend_rect;
	vertical_legend_title_rect = d_vertical_legend_title_rect;
	horizontal_legend_rect = d_horizontal_legend_rect;
	horizontal_legend_tile_rect = d_horizontal_legend_title_rect;
	title_rect = d_title_rect;
	viewport_rect = d_viewport_rect;
	vertical_legend_title = d_vertical_legend_title;
	horizontal_legend_title = d_horizontal_legend_title;
	title = d_title;
	draw_border = d_draw_border;
	
	d_vertical_legend_title = QString();
	d_horizontal_legend_title = QString();
	d_title = QString();
	d_draw_border = enable_border;
	
	d_viewport_rect = rect;
	
	X = d_viewport_rect.width();
	Y = d_viewport_rect.height();
	
	d_vertical_legend_title_rect.setLeft(d_viewport_rect.left());
	d_vertical_legend_title_rect.setTop(d_viewport_rect.top());
	d_vertical_legend_title_rect.setRight(border + d_viewport_rect.left());
	d_vertical_legend_title_rect.setBottom(Y - border);
	
	d_horizontal_legend_title_rect.setLeft(border + d_viewport_rect.left());
	d_horizontal_legend_title_rect.setTop(Y - border);
	d_horizontal_legend_title_rect.setRight(X);
	d_horizontal_legend_title_rect.setBottom(Y);
	
	d_title_rect.setLeft(d_vertical_legend_title_rect.right() + border);
	d_title_rect.setTop(d_viewport_rect.top());
	d_title_rect.setRight(X);
	d_title_rect.setBottom(border + d_viewport_rect.top());
	
	d_vertical_legend_rect.setLeft(d_vertical_legend_title_rect.right());
	d_vertical_legend_rect.setTop(d_viewport_rect.top());
	d_vertical_legend_rect.setRight(d_vertical_legend_title_rect.right() + 4 * line_height);
	d_vertical_legend_rect.setBottom(d_horizontal_legend_title_rect.top());
	
	d_horizontal_legend_rect.setLeft(d_vertical_legend_rect.right());
	d_horizontal_legend_rect.setTop(d_horizontal_legend_title_rect.top() - line_height * 2);
	d_horizontal_legend_rect.setRight(X);
	d_horizontal_legend_rect.setBottom(d_horizontal_legend_title_rect.top());
	
	d_graph_rect.setLeft(d_vertical_legend_rect.right());
	d_graph_rect.setTop(d_title_rect.bottom());
	d_graph_rect.setRight(X - line_height * 2);
	d_graph_rect.setBottom(d_horizontal_legend_rect.top());
	
	d_vertical_legend_title_rect.setTop(d_graph_rect.top());
	d_vertical_legend_title_rect.setBottom(d_graph_rect.bottom());
	
	d_horizontal_legend_title_rect.setLeft(d_graph_rect.left());
	d_horizontal_legend_title_rect.setRight(d_graph_rect.right());
	
	this->Direct_Render(painter);
	
	d_viewport_mat = viewport_mat;
	d_graph_rect = graph_rect;
	d_vertical_legend_rect = vertical_legend_rect;
	d_vertical_legend_title_rect = vertical_legend_title_rect;
	d_horizontal_legend_rect = horizontal_legend_rect;
	d_horizontal_legend_title_rect = horizontal_legend_tile_rect;
	d_title_rect = title_rect;
	d_viewport_rect= viewport_rect;
	
	d_vertical_legend_title = vertical_legend_title;
	d_horizontal_legend_title = horizontal_legend_title;
	d_title = title;
	d_draw_border = draw_border;
	
	return true;
}

void TGraphWidget::SetScaleMinimumY(
	const double						&minimum_y,
	const bool							bidirectional)
{
	if(minimum_y > 0)
	{
		d_y_min_value = minimum_y;
	}
	else
	{
		d_y_min_value = 0.00005;
	}
	
	d_bidirectional_y_min_value = bidirectional;
}

void TGraphWidget::EnableYIncludesZero(
	const bool							state)
{
	d_y_includes_zero = state;
}

bool TGraphWidget::GenerateRGBColor(
	const int 							set,
	int 								*R,
	int 								*G,
	int 								*B)
{
    int                           		intensitity_offset;
    int                           		intensity_multiplier;
    int                           		table_offset;
    static const int					enable_color_table[] = {1,0,0,0,1,0,0,0,1,1,1,0,0,1,1,1,0,1};
    static const int					color_intensity_table[] = {255,128,192,64,160,224,96,32,144,112,176,80,208,48,240,16};   // 16 colors. Max colors = 16 * 6 = 96
	
	// default when more than 96 sets exist
	if(set > 96)
	{
		*R = 0;
		*G = 0;
		*B = 0;

		return false;
	}
     
     intensitity_offset = set / 6;

     if(intensitity_offset > 15)
     {
          intensity_multiplier = 0;
     }
     else
     {
          intensity_multiplier = color_intensity_table[intensitity_offset];
     }

     table_offset = set - (intensitity_offset * 6);
     table_offset *= 3;

     *R = enable_color_table[table_offset] * intensity_multiplier;
     *G = (enable_color_table[table_offset + 1] * intensity_multiplier)/2;
     *B = enable_color_table[table_offset + 2] * intensity_multiplier;

     return true;
}
	
void TGraphWidget::ZoomOut(void)
{
	d_process_zoom_out = true;
	
	this->update();
}

void TGraphWidget::ScaleToFit(void)
{
	d_process_scale_to_fit = true;
	
	this->update();
}

void TGraphWidget::paintEvent(
	QPaintEvent 						*event)
{
#if (QT_VERSION >= QT_VERSION_CHECK(5, 14, 0))
	QRegion::const_iterator				region_iter;
#else
	QVector<QRect>						rect_list = event->region().rects();
#endif
    QPainter 							painter(this);
	QBrush								white_brush(Qt::white);
	QPen								pen_light_grey(Qt::lightGray);
	QRect								mouse_tracking_rect;
 	QRegion								region;
	QSize								size;
	QRect								rect;
	QPoint								point;
	
	region = event->region();
	
	if(d_enable_direct_rendering)
	{
		painter.fillRect(d_viewport_rect,white_brush);
		
		this->Update_Viewport_To_Data();
		this->Calculate_Legend_Display_Precision(&painter);
		this->Draw_Legend(&painter);
		this->Draw_Graph(&painter);
	}
	else
	{
		if(d_process_scale_to_fit)
		{
			d_process_scale_to_fit = false;
			
			this->Draw_Background_Pixmap();
			update(d_viewport_rect.left(),
						d_viewport_rect.top(),
						d_viewport_rect.width(),
						d_viewport_rect.height());
				
			return;
		}
		
		if(d_process_zoom_out)
		{
			d_process_zoom_out = false;
			
			size = d_graph_rect.size();
			size *= 2;
		
			point = d_graph_rect.center();
			
			rect.setWidth(size.width());
			rect.setHeight(size.height());
		
			rect.moveCenter(point);
		
			this->Draw_Background_Scaled_Pixmap(rect);
			
			update(d_viewport_rect.left(),
						d_viewport_rect.top(),
						d_viewport_rect.width(),
						d_viewport_rect.height());
			
			return;
		}
		
#if (QT_VERSION >= QT_VERSION_CHECK(5, 14, 0))
		for(region_iter = region.begin();region_iter != region.end();++region_iter)
		{
			if(!(*region_iter).isEmpty())
			{
				painter.drawPixmap((*region_iter),d_widget_pixmap,(*region_iter));
			}
		}
#else
		for(int i=0;i < static_cast<int>(rect_list.size());++i)
		{
			painter.drawPixmap(rect_list[i],d_widget_pixmap,rect_list[i]);
		}
#endif
		if(d_draw_mouse_rect)
		{
			painter.drawRect(d_mouse_down_rect.normalized());
		}
	}

	if(d_show_mouse_tracking)
	{
		painter.setPen(pen_light_grey);
		
		rect = painter.boundingRect(0,0,0,0,0,d_tracking_text);
		
		mouse_tracking_rect.setLeft(d_graph_rect.right() - rect.width());
		mouse_tracking_rect.setTop(d_graph_rect.top());
		mouse_tracking_rect.setRight(d_graph_rect.right());
		mouse_tracking_rect.setBottom(d_graph_rect.top() + rect.height());
		
		painter.drawText(mouse_tracking_rect.left(),mouse_tracking_rect.bottom(),d_tracking_text);
	}
}

void TGraphWidget::resizeEvent(
	QResizeEvent 						*size)
{
	int									X,Y;
	int									line_height;
	
	QFontMetrics						font_metrics(this->font());
	line_height = font_metrics.height();
	
	X = size->size().width();
	Y = size->size().height();

	d_viewport_rect.setLeft(0);
	d_viewport_rect.setTop(0);
	d_viewport_rect.setRight(X - 1);
	d_viewport_rect.setBottom(Y - 1);

	d_vertical_legend_title_rect.setLeft(d_viewport_rect.left());
	d_vertical_legend_title_rect.setTop(d_viewport_rect.top());
	d_vertical_legend_title_rect.setRight(line_height * 2 + d_viewport_rect.left());
	d_vertical_legend_title_rect.setBottom(Y - line_height * 2);

	d_horizontal_legend_title_rect.setLeft(line_height * 2 + d_viewport_rect.left());
	d_horizontal_legend_title_rect.setTop(Y - line_height * 2);
	d_horizontal_legend_title_rect.setRight(X);
	d_horizontal_legend_title_rect.setBottom(Y);

	d_title_rect.setLeft(d_vertical_legend_title_rect.right() + 5 * line_height);
	d_title_rect.setTop(d_viewport_rect.top());
	d_title_rect.setRight(X);
	d_title_rect.setBottom(line_height * 2 + d_viewport_rect.top());

	d_vertical_legend_rect.setLeft(d_vertical_legend_title_rect.right());
	d_vertical_legend_rect.setTop(d_viewport_rect.top());
	d_vertical_legend_rect.setRight(d_vertical_legend_title_rect.right() + 4 * line_height);
	d_vertical_legend_rect.setBottom(d_horizontal_legend_title_rect.top());

	d_horizontal_legend_rect.setLeft(d_vertical_legend_rect.right());
	d_horizontal_legend_rect.setTop(d_horizontal_legend_title_rect.top() - line_height * 2);
	d_horizontal_legend_rect.setRight(X);
	d_horizontal_legend_rect.setBottom(d_horizontal_legend_title_rect.top());

	d_graph_rect.setLeft(d_vertical_legend_rect.right() + line_height);
	d_graph_rect.setTop(d_title_rect.bottom());
	d_graph_rect.setRight(X - line_height * 2);
	d_graph_rect.setBottom(d_horizontal_legend_rect.top());

	d_vertical_legend_title_rect.setTop(d_graph_rect.top());
	d_vertical_legend_title_rect.setBottom(d_graph_rect.bottom());

	d_horizontal_legend_title_rect.setLeft(d_graph_rect.left());
	d_horizontal_legend_title_rect.setRight(d_graph_rect.right());
	
	this->Draw_Background_Pixmap();
}

void TGraphWidget::mousePressEvent(
	QMouseEvent							*event)
{
	if(event->button() & Qt::LeftButton)
	{
		if(d_graph_rect.contains(event->pos()))
		{
			d_mouse_down_rect.setTopLeft(QPoint(event->pos()));
			d_mouse_down_rect.setBottomRight(QPoint(event->pos()));
		}
		else
		{
			d_mouse_down_rect.setTopLeft(QPoint(d_graph_rect.topLeft()));
			d_mouse_down_rect.setBottomRight(QPoint(d_graph_rect.topLeft()));
		}	
		
		d_left_mouse_down = true;
		d_left_mouse_moving = false;
	
	}
	
	if(event->button() & Qt::RightButton)
	{
		d_mouse_down_rect.setTopLeft(event->pos());
		d_mouse_down_rect.setBottomRight(event->pos());
		
		d_right_mouse_moving = false;
		d_right_mouse_down = true;
	
	}
}

void TGraphWidget::mouseMoveEvent(
	QMouseEvent							*event)
{
	QPoint								point;
	QRect								rect;
	double								dX,dY;

	point = event->pos();
	
	if(event->buttons() & Qt::LeftButton)
	{
		if(d_left_mouse_down)
		{
			if(d_left_mouse_moving)
			{
				this->Update_Left_Down_Box();
			}
			else
			{
				d_left_mouse_moving = true;				
				d_draw_mouse_rect = true;
			}
			
			d_mouse_down_rect.setBottomRight(point);
			this->Update_Left_Down_Box();
		}
	
	} 
	else if(event->buttons() & Qt::RightButton)
	{
		if(d_right_mouse_down)
		{
			d_mouse_down_rect.setBottomRight(point);
			
			rect = d_graph_rect;
			
			rect.adjust(
					d_mouse_down_rect.left() - d_mouse_down_rect.right(),
					d_mouse_down_rect.top() - d_mouse_down_rect.bottom(),
					d_mouse_down_rect.left() - d_mouse_down_rect.right(),
					d_mouse_down_rect.top() - d_mouse_down_rect.bottom());
					
			this->Draw_Background_Scaled_Pixmap(rect);
			this->repaint();
	
			d_mouse_down_rect.setTopLeft(point);
		}
			
		d_right_mouse_moving = true;
	}
	else if(d_graph_rect.contains(point) && d_show_mouse_tracking)
	{
		d_viewport_mat.Viewport_To_Position(point.x(),point.y(),&dX,&dY);
		
		if(d_horizontal_legend_mode == TGraphWidget::LOG10_LEGEND_PRECISION)
		{
			dX = this->Calculate_Log10_Value_Horizontal(point.x());
		}

		if(d_vertical_legend_mode == TGraphWidget::LOG10_LEGEND_PRECISION)
		{
			dY = this->Calculate_Log10_Value_Vertical(point.y());
		}

		d_tracking_text = QString("%1,   %2").arg(dX,0,'f',5).arg(dY,0,'f',5);
		
		this->repaint();
	}
}

void TGraphWidget::mouseReleaseEvent(
	QMouseEvent							*event)
{
	QPoint								point;
	QPoint								c1,c2;
	double								dX,dY;

	point = event->pos();

	if(event->button() & Qt::LeftButton)
	{
		// check for minimum size
		if(d_left_mouse_moving)
		{
			c1 = d_mouse_down_rect.topLeft();
			c2 = d_mouse_down_rect.bottomRight();
			c2 -= c1;
			
			if(c2.manhattanLength() < 20)	// move more likely accidental
			{
				d_left_mouse_moving = false;
			}
		}
		
		if(d_left_mouse_moving)
		{
			d_draw_mouse_rect = false;
			this->Update_Left_Down_Box();
			
			if(d_horizontal_legend_mode == TGraphWidget::LOG10_LEGEND_PRECISION)
			{
				d_mouse_down_rect.setLeft(this->Reverse_Log10_Position_Horizontal(d_mouse_down_rect.left()));
				d_mouse_down_rect.setRight(this->Reverse_Log10_Position_Horizontal(d_mouse_down_rect.right()));
			}

			if(d_vertical_legend_mode == TGraphWidget::LOG10_LEGEND_PRECISION)
			{
				d_mouse_down_rect.setTop(this->Reverse_Log10_Position_Vertical(d_mouse_down_rect.top()));
				d_mouse_down_rect.setBottom(this->Reverse_Log10_Position_Vertical(d_mouse_down_rect.bottom()));
			}

			this->Draw_Background_Scaled_Pixmap(d_mouse_down_rect);
				
			update(d_viewport_rect.left(),
						d_viewport_rect.top(),
						d_viewport_rect.width(),
						d_viewport_rect.height());
						
			d_left_mouse_moving = false;						
		}
		else
		{
			d_viewport_mat.Viewport_To_Position(point.x(),point.y(),&dX,&dY);
			
			if(d_horizontal_legend_mode == TGraphWidget::LOG10_LEGEND_PRECISION)
			{
				dX = this->Calculate_Log10_Value_Horizontal(point.x());
			}

			if(d_vertical_legend_mode == TGraphWidget::LOG10_LEGEND_PRECISION)
			{
				dY = this->Calculate_Log10_Value_Vertical(point.y());
			}

			emit Mouse_Click(dX,dY);
		}
		
		d_left_mouse_down = false;
	}
	
	if(event->button() & Qt::RightButton)
	{
		if(d_right_mouse_moving)
		{
			d_right_mouse_moving = false;						
		}
		else
		{
			this->ZoomOut();
		}
		
		d_right_mouse_down = false;
	}
}

void TGraphWidget::Draw_Background_Pixmap(void)
{
	if(d_viewport_rect.height() < 1 || d_viewport_rect.width() < 1)
	{
		return;
	}
	
	d_widget_pixmap = QPixmap(QSize(d_viewport_rect.width(),d_viewport_rect.height()));
	d_widget_pixmap.fill(Qt::white);
	
	QPainter painter(&d_widget_pixmap);

    painter.setViewport(d_viewport_rect.left(),
    			d_viewport_rect.top(),
    			d_viewport_rect.right(),
    			d_viewport_rect.bottom());
	
	this->Update_Viewport_To_Data();
    this->Calculate_Legend_Display_Precision(&painter);
	this->Draw_Legend(&painter);
	this->Draw_Graph(&painter);
}

void TGraphWidget::Draw_Background_Scaled_Pixmap(
	const QRect							&rect)
{
	if(d_widget_pixmap.height() < 1 || d_widget_pixmap.width() < 1)
	{
		return;
	}
	
	d_widget_pixmap.fill(Qt::white);

	QPainter painter(&d_widget_pixmap);

    painter.setViewport(d_viewport_rect.left(),
    			d_viewport_rect.top(),
    			d_viewport_rect.right(),
    			d_viewport_rect.bottom());
	
	this->Update_Viewport_To_Rect(rect);
    this->Calculate_Legend_Display_Precision(&painter);
	this->Draw_Legend(&painter);
	this->Draw_Graph(&painter);
}

void TGraphWidget::Direct_Render(
	QPainter							*painter)
{
	this->Update_Viewport_To_Data();
    this->Calculate_Legend_Display_Precision(painter);
	this->Draw_Legend(painter);
	this->Draw_Graph(painter);
}


void TGraphWidget::Update_Viewport_To_Data(void)
{
	std::vector<TDataSet>::iterator     iter_set;
	std::vector<TVector2>::iterator		iter_pnt;
    double                              min_x,max_x,min_y,max_y;
    TViewportMat                        W,S,V;
    bool                                init;
	double								dvalx;//,dvaly;
	double								dsclx,dscly;

 	max_x = 0.000005;
    min_x = -0.000005;
    max_y = 0.000005;
    min_y = -0.000005;
 
	init = true;
	
	for(iter_set = d_data_sets.begin();iter_set != d_data_sets.end();++iter_set)
	{
		for(iter_pnt = (*iter_set).dataset_points.begin();iter_pnt != (*iter_set).dataset_points.end();++iter_pnt)
		{			
			if(init)
			{
				min_x = max_x = (*iter_pnt).x;
				min_y = max_y = (*iter_pnt).y;

				init = false;
			}
			else
			{
				if((*iter_pnt).x > max_x)
				{
					max_x = (*iter_pnt).x;
				}

				if((*iter_pnt).x < min_x)
				{
					min_x = (*iter_pnt).x;
				}

				if((*iter_pnt).y > max_y)
				{
					max_y = (*iter_pnt).y;
				}

				if((*iter_pnt).y < min_y)
				{
					min_y = (*iter_pnt).y;
				}
			}
		}
	}
	
	if(d_y_includes_zero)
	{
		if(d_bidirectional_y_min_value)
		{
			if(min_y > (-1 * d_y_min_value))
			{
				min_y = -1 * d_y_min_value;
			}
		}
		
		if(max_y < d_y_min_value)
		{
			max_y = d_y_min_value;
		}
	}
	
	if(d_graph_type == TGraphWidget::BAR_GRAPH)
	{
		if(d_horizontal_legend_mode == TGraphWidget::LOG10_LEGEND_PRECISION)
		{
			dvalx = (max_x - min_x) / 40;	// bar sized reduced by 50% for LOG version
		}
		else
		{
			dvalx = (max_x - min_x) / 20;
		}
		max_x += dvalx;
		min_x -= dvalx;
	}

    if((max_x - min_x) < 0.00001)
    {
    	max_x+= 0.000005;
        min_x-= 0.000005;
    }
	
	// set x axis scale same as y axis
	if(d_lock_aspect)
	{
		dvalx = (max_x + min_x)/2.0;
//		dvaly = (max_y + min_y)/2.0;
		
		dscly = abs(d_graph_rect.top() - d_graph_rect.bottom());
		dsclx = abs(d_graph_rect.right() - d_graph_rect.left());
		
		if(dscly > 0)
		{
			dsclx = dsclx / dscly;
		}
		else
		{
			dsclx = 1.0;
		}
		
		max_x = dvalx + max_y * dsclx;
		min_x = dvalx + min_y * dsclx;
	}

    if((max_y - min_y) < 0.00001)
    {
    	max_y += 0.000005;
        min_y -= 0.000005;
    }

    d_data_range_x = max_x - min_x;
    d_data_range_y = max_y - min_y;

	if(d_horizontal_legend_mode == TGraphWidget::DEFAULT_LEGEND_PRECISION)
	{
		max_x += (d_data_range_x / 50);
		min_x -= (d_data_range_x / 50);
	}

	if(d_vertical_legend_mode == TGraphWidget::DEFAULT_LEGEND_PRECISION)
    {
		max_y += (d_data_range_y / 50);
		min_y -= (d_data_range_y / 50);
    }
	
	W.tx[0] = 1;
    W.tx[1] = 0;
    W.ty[0] = 0;
    W.ty[1] = 1;
    W.to[0] = -min_x;
    W.to[1] = -min_y;

    S.tx[0] = (d_graph_rect.right() - d_graph_rect.left()) / (max_x - min_x);
    S.tx[1] = 0;
    S.ty[0] = 0;
    S.ty[1] = (d_graph_rect.top() - d_graph_rect.bottom()) / (max_y - min_y);
    S.to[0] = 0;
    S.to[1] = 0;

    V.tx[0] = 1;
    V.tx[1] = 0;
    V.ty[0] = 0;
    V.ty[1] = 1;
    V.to[0] = d_graph_rect.left();
    V.to[1] = d_graph_rect.bottom();

    d_viewport_mat = W;
    d_viewport_mat *=S;
    d_viewport_mat *=V;    
}

void TGraphWidget::Update_Viewport_To_Rect(
	const QRect							&rect)
{
    double                              min_x,max_x,min_y,max_y;
    double								scale_limit;
	QRect								rect_normalized;
	TViewportMat                        W,S,V;

	rect_normalized = rect.normalized();

	d_viewport_mat.Viewport_To_Position(rect_normalized.left(),rect_normalized.top(),&min_x,&max_y);
	d_viewport_mat.Viewport_To_Position(rect_normalized.right(),rect_normalized.bottom(),&max_x,&min_y);

	if((max_x - min_x) > (d_data_range_x * 50))
	{
		scale_limit = (d_data_range_x * 50) / (max_x - min_x);

		max_x *= scale_limit;
		min_x *= scale_limit;
	}

	if((max_y - min_y) > (d_data_range_y * 50))
	{
		scale_limit = (d_data_range_y * 50) / (max_y - min_y);

		max_y *= scale_limit;
		min_y *= scale_limit;
	}

    if((max_x - min_x) < 0.00001)
    {
    	max_x+= 0.000005;
        min_x-= 0.000005;
    }

    if((max_y - min_y) < 0.00001)
    {
    	max_y += 0.000005;
        min_y -= 0.000005;
    }


	W.tx[0] = 1;
    W.tx[1] = 0;
    W.ty[0] = 0;
    W.ty[1] = 1;
    W.to[0] = -min_x;
    W.to[1] = -min_y;

    S.tx[0] = (d_graph_rect.right() - d_graph_rect.left()) / (max_x - min_x);
    S.tx[1] = 0;
    S.ty[0] = 0;
    S.ty[1] = (d_graph_rect.top() - d_graph_rect.bottom()) / (max_y - min_y);
    S.to[0] = 0;
    S.to[1] = 0;

    V.tx[0] = 1;
    V.tx[1] = 0;
    V.ty[0] = 0;
    V.ty[1] = 1;
    V.to[0] = d_graph_rect.left();
    V.to[1] = d_graph_rect.bottom();

    d_viewport_mat = W;
    d_viewport_mat *=S;
    d_viewport_mat *=V;
}

void TGraphWidget::Draw_Legend(
	QPainter							*painter)
{
	QString                       		text;
	QFont								font(painter->font());
	QRect								bounding_rect;
	QRect								rect;
    QPoint                         		text_position;
	QPen								pen_light_grey(Qt::lightGray);
	QPen								pen_dark_grey(Qt::darkGray);
	QPen								pen_black(Qt::black);
    double                        		label_increment;
	double								label_value;
	double                        		Xd,Yd;
	double								min_x,min_y;
	double								max_x,max_y;
	int	                        		label_position;
	int									last_label_position;
	int                           		count;
	int                           		cntr;
    int									Xi,Yi;
    int									max_int;
    int									min_int;
    int									last_int(0);
	int									line_length;
    bool								init;
	
	assert(painter);
/*
    // display all the rect's
 
	QPen								pen_red(Qt::red);

	painter->setPen(pen_red);

	painter->drawRect(d_graph_rect.adjusted(1,1,-1,-1));
	painter->drawRect(d_vertical_legend_rect.adjusted(1,1,-1,-1));
	painter->drawRect(d_vertical_legend_title_rect.adjusted(1,1,-1,-1));
	painter->drawRect(d_horizontal_legend_rect.adjusted(1,1,-1,-1));
	painter->drawRect(d_horizontal_legend_title_rect.adjusted(1,1,-1,-1));
	painter->drawRect(d_title_rect.adjusted(1,1,-1,-1));
*/
	
// horizontal legend
	line_length = d_horizontal_label_size.height() / 3;

	if(d_horizontal_legend_mode == TGraphWidget::DEFAULT_LEGEND_PRECISION)
	{
		count = d_graph_rect.width() / d_horizontal_label_size.width();
		label_increment = static_cast<double>(d_graph_rect.width()) / static_cast<double>(count - 1);
		
		text_position.setY(d_horizontal_legend_rect.top() + d_horizontal_label_size.height());

		for(cntr = 0;cntr < count;++cntr)
		{
			text_position.setX(d_graph_rect.left() + static_cast<int>(static_cast<double>(cntr) * label_increment));

			painter->setPen(pen_light_grey);
			
			painter->drawLine(text_position.x(),
						d_graph_rect.bottom(),
						text_position.x(),
						d_graph_rect.bottom() + line_length);

			if(d_horizontal_legend_mode != TGraphWidget::NO_LEGEND)
			{
				d_viewport_mat.Viewport_To_Position(text_position.x(),0,&Xd,&Yd);
				
				text = QString("%1").arg(Xd,d_horizontal_text_total,'f',d_horizontal_text_precision);

				painter->setPen(pen_black);
				painter->drawText(
							text_position.x() - d_horizontal_label_size.width() / 2,
							text_position.y() + d_horizontal_label_size.height() / 2,
							text);
			}
		}
	}
	else if(d_horizontal_legend_mode == TGraphWidget::INTEGER_LEGEND_PRECISION)
	{
		// Find range of integer values that fit inside graph
		d_viewport_mat.Viewport_To_Position(d_graph_rect.left(),0,&Xd,&Yd);
		min_int = Xd;
		last_int = min_int;

		d_viewport_mat.Viewport_To_Position(d_graph_rect.right(),0,&Xd,&Yd);
		max_int = Xd;
		
		init = true;
		
		text_position.setY(d_horizontal_legend_rect.top() + d_horizontal_label_size.height());
		
		for(cntr = min_int;cntr <= max_int;cntr++)
		{
			d_viewport_mat.Position_To_Viewport(static_cast<double>(cntr),0,&Xi,&Yi);
			
			text_position.setX(Xi);
			
			if(abs(Xi - last_int) > d_horizontal_label_size.width() || init == true)
			{
				init = false;
				
				painter->setPen(pen_light_grey);
				painter->drawLine(text_position.x(),
								  d_graph_rect.bottom(),
								  text_position.x(),
								  d_graph_rect.bottom() + line_length);
				
				text = QString("%1").arg(cntr);				
				
				rect.setRect(text_position.x() - d_horizontal_label_size.width() / 2,
							 text_position.y() - d_horizontal_label_size.height() / 2,
							 d_horizontal_label_size.width(),
							 d_horizontal_label_size.height());
				
				
				painter->setPen(pen_black);
				painter->drawText(rect,Qt::AlignCenter | Qt::AlignVCenter | Qt::TextDontClip,text);

				last_int = Xi;
			}
		}
	}
	else if(d_horizontal_legend_mode == TGraphWidget::LOG10_LEGEND_PRECISION)
	{
		d_viewport_mat.Viewport_To_Position(
				  d_graph_rect.left(),
				  d_graph_rect.top(),
				  &min_x,
				  &max_y);

		 d_viewport_mat.Viewport_To_Position(
				  d_graph_rect.right(),
				  d_graph_rect.bottom(),
				  &max_x,
				  &min_y);
		
		count = d_graph_rect.width() / d_horizontal_label_size.width();
		label_increment = ceil((max_x - min_x) / static_cast<double>(count - 1));
		
		label_value = min_x;
		text_position.setY(d_horizontal_legend_rect.top() + d_horizontal_label_size.height());

		for(cntr = 0;cntr < count;++cntr)
		{
			label_position = Calculate_Log10_Position_Horizontal(label_value);
			
			if(cntr)
			{
				if((label_position - last_label_position) < d_horizontal_label_size.width())
				{
					label_increment *= 1.5;
				}
			}
			
			last_label_position = label_position;
			text_position.setX(label_position);

			painter->setPen(pen_light_grey);
			
			painter->drawLine(text_position.x(),
						d_graph_rect.bottom(),
						text_position.x(),
						d_graph_rect.bottom() + line_length);
			
			text = QString("%1").arg(label_value,d_horizontal_text_total,'f',d_horizontal_text_precision);
			
			painter->setPen(pen_black);
			painter->drawText(
						text_position.x() - d_horizontal_label_size.width() / 2,
						text_position.y() + d_horizontal_label_size.height() / 2,
						text);
			
			if(label_value > (max_x - label_increment))
			{
				break;
			}

			label_value += label_increment;
		}
 
	}

// Vertical Axis
	line_length = d_vertical_label_size.height() / 2;

    if(d_vertical_legend_mode == TGraphWidget::DEFAULT_LEGEND_PRECISION)
    {
		count = d_graph_rect.height() / d_vertical_label_size.height();;
		label_increment = static_cast<double>(d_graph_rect.height()) / static_cast<double>(count - 1);

		for(cntr = 0;cntr < count;++cntr)
		{
			text_position.setY(d_graph_rect.top() + static_cast<int>(cntr * label_increment));

			painter->setPen(pen_light_grey);
			painter->drawLine(d_graph_rect.left() - line_length,
						text_position.y(),
						d_graph_rect.left(),
						text_position.y());

			d_viewport_mat.Viewport_To_Position(0,text_position.y(),&Xd,&Yd);
			
			rect.setRect(d_vertical_legend_rect.right() - d_vertical_label_size.width(),
						 text_position.y() - d_vertical_label_size.height() / 2,
						 d_vertical_label_size.width(),
						 d_vertical_label_size.height());

			text = QString("%1").arg(Yd,d_vertical_text_total,'f',d_vertical_text_precision);

			painter->setPen(pen_black);
			painter->drawText(rect,Qt::AlignRight | Qt::AlignVCenter | Qt::TextDontClip,text);
		}
    }
    else if(d_vertical_legend_mode == TGraphWidget::INTEGER_LEGEND_PRECISION)
    {
    	// Find range of integer values that fit inside graph
		d_viewport_mat.Viewport_To_Position(0,d_graph_rect.top() + d_graph_rect.height(),&Xd,&Yd);
		min_int = Yd;
		last_int = min_int;

		d_viewport_mat.Viewport_To_Position(0,d_graph_rect.top(),&Xd,&Yd);
		max_int = Yd;

		init = true;

		for(cntr = min_int;cntr <= max_int;cntr++)
		{
			d_viewport_mat.Position_To_Viewport(0,static_cast<double>(cntr),&Xi,&Yi);

			text_position.setY(Yi);

			if(abs(Yi - last_int) > d_vertical_label_size.height() || init == true)
			{
				init = false;

				painter->setPen(pen_light_grey);
				painter->drawLine(d_graph_rect.left() - line_length,
							text_position.y(),
							d_graph_rect.left(),
							text_position.y());

				rect.setRect(d_vertical_legend_rect.right() - d_vertical_label_size.width(),
							 text_position.y() - d_vertical_label_size.height() / 2,
							 d_vertical_label_size.width(),
							 d_vertical_label_size.height());

				text = QString("%1 ").arg(cntr);	// one space shift on right side

				painter->setPen(pen_black);
				painter->drawText(rect,Qt::AlignRight | Qt::AlignVCenter | Qt::TextDontClip,text);

				last_int = Yi;
			}
		}
    }
	if(d_vertical_legend_mode == TGraphWidget::LOG10_LEGEND_PRECISION)
	{
		d_viewport_mat.Viewport_To_Position(
				  d_graph_rect.left(),
				  d_graph_rect.top(),
				  &min_x,
				  &max_y);

		 d_viewport_mat.Viewport_To_Position(
				  d_graph_rect.right(),
				  d_graph_rect.bottom(),
				  &max_x,
				  &min_y);
		
		count = d_graph_rect.height() / d_vertical_label_size.height();;
		label_increment = (max_y - min_y) / static_cast<double>(count - 1);

		label_value = min_y;
		text_position.setX(d_vertical_legend_rect.left() + d_vertical_label_size.width());

		for(cntr = 0;cntr < count;++cntr)
		{
			label_position = Calculate_Log10_Position_Vertical(label_value);
			
			if(cntr)
			{
				if((label_position - last_label_position) < d_vertical_label_size.height())
				{
					label_increment *= 1.5;
				}
			}
			
			last_label_position = label_position;
			text_position.setY(label_position);

			painter->setPen(pen_light_grey);
			painter->drawLine(d_graph_rect.left() - line_length,
						text_position.y(),
						d_graph_rect.left(),
						text_position.y());

			Yd = this->Calculate_Log10_Value_Vertical(text_position.y());

			rect.setRect(d_vertical_legend_rect.right() - d_vertical_label_size.width(),
						 text_position.y() - d_vertical_label_size.height() / 2,
						 d_vertical_label_size.width(),
						 d_vertical_label_size.height());

			text = QString("%1").arg(Yd,d_vertical_text_total,'f',d_vertical_text_precision);
			
			painter->setPen(pen_black);
			painter->drawText(rect,Qt::AlignRight | Qt::AlignVCenter | Qt::TextDontClip,text);
			
			if(label_value > max_y)
			{
				break;
			}

			label_value += label_increment;
		}
	}

// Draw a line at graph zero if within the graph viewport
	Xd = 0;
	Yd = 0;
	d_viewport_mat.Position_To_Viewport(Xd,Yd,&Xi,&Yi);
	
	if(d_horizontal_legend_mode == TGraphWidget::LOG10_LEGEND_PRECISION)
	{
		Xi = this->Calculate_Log10_Position_Horizontal(Xd);
	}

	if(d_vertical_legend_mode == TGraphWidget::LOG10_LEGEND_PRECISION)
	{
		Yi = this->Calculate_Log10_Position_Vertical(Yd);
	}
	
	painter->setPen(pen_light_grey);
	
	if(Xi > d_graph_rect.left() && Xi < d_graph_rect.right())
	{
		painter->drawLine(
					Xi,
					d_graph_rect.top(),
					Xi,
					d_graph_rect.bottom());
	}
	
	if(Yi > d_graph_rect.top() && Yi < d_graph_rect.bottom())
	{
		painter->drawLine(
					d_graph_rect.left(),
					Yi,
					d_graph_rect.right(),
					Yi);
	}
	
	painter->setPen(pen_dark_grey);
	
// border around the graph.
	painter->drawRect(d_graph_rect.left(),d_graph_rect.top(),d_graph_rect.width(),d_graph_rect.height());

	painter->setPen(pen_black);

// border around the graph widget.
	if(d_draw_border)
	{
		painter->drawRect(0,0,d_viewport_rect.width()-1,d_viewport_rect.height()-1);
	}

// title
	font.setPointSize(12);
	font.setBold(true);
	painter->setFont(font);

	painter->drawText(d_title_rect,Qt::AlignCenter | Qt::AlignVCenter,d_title);

// vertical legend
	font.setPointSize(10);
	painter->setFont(font);

	painter->drawText(d_horizontal_legend_title_rect,Qt::AlignCenter,d_horizontal_legend_title);
	
	bounding_rect = painter->boundingRect(0,0,0,0,Qt::AlignLeft,d_vertical_legend_title);
	
	if(bounding_rect.width() < 1)
	{
		bounding_rect.setWidth(1);
	}
	
	if(bounding_rect.height() < 1)
	{
		bounding_rect.setHeight(1);
	}

	painter->save();
	painter->translate(static_cast<double>(d_vertical_legend_title_rect.center().x()),static_cast<double>(d_vertical_legend_title_rect.center().y()));
	painter->rotate(-90);

	painter->drawText(-bounding_rect.width()/2,0,d_vertical_legend_title);
	painter->restore();
}

void TGraphWidget::Draw_Graph(
	QPainter							*painter)
{
    int									X,Y,X2,Y2,LastX,LastY;
    int									bar_width;
	bool								init;
	std::vector<TDataSet>::const_iterator	set_iter;
	std::vector<TVector2>::const_iterator	pnt_iter;

	assert(painter);
	
	if(!d_data_sets.empty())
    {
		painter->setClipRect(d_graph_rect);
		
		for(set_iter = d_data_sets.begin();set_iter != d_data_sets.end();set_iter++)
		{
			if((*set_iter).dataset_points.size())
			{
				if((d_graph_type == TGraphWidget::BAR_GRAPH && (*set_iter).dataset_graph_type == TGraphWidget::DEFAULT_GRAPH) ||
				   (*set_iter).dataset_graph_type == TGraphWidget::BAR_GRAPH)
				{
					painter->setPen((*set_iter).dataset_color);
					
					bar_width = d_graph_rect.width();
					
					if(d_horizontal_legend_mode == TGraphWidget::LOG10_LEGEND_PRECISION)
					{
						bar_width /= (static_cast<int>((*set_iter).dataset_points.size()) * 4);
					}
					else
					{
						bar_width /= (static_cast<int>((*set_iter).dataset_points.size()) * 2);
					}
					
					bar_width -= 1;
					
					// make sure bar width is at least one pixel
					if(bar_width < 1)
					{
						bar_width = 1;
					}
					
					for(pnt_iter = (*set_iter).dataset_points.begin();pnt_iter != (*set_iter).dataset_points.end();pnt_iter++)
					{
						d_viewport_mat.Position_To_Viewport((*pnt_iter).x,(*pnt_iter).y,&X,&Y);
						d_viewport_mat.Position_To_Viewport((*pnt_iter).x,0.0,&X2,&Y2);
																		
						if(d_horizontal_legend_mode == TGraphWidget::LOG10_LEGEND_PRECISION)
						{
							X = this->Calculate_Log10_Position_Horizontal((*pnt_iter).x);
							X2 = X;
						}

						if(d_vertical_legend_mode == TGraphWidget::LOG10_LEGEND_PRECISION)
						{
							Y = this->Calculate_Log10_Position_Vertical((*pnt_iter).y);
							Y2 = this->Calculate_Log10_Position_Vertical(0.0);
						}
						
						painter->fillRect(X - bar_width,Y,2 * bar_width,Y2 - Y,(*set_iter).dataset_color);
					}
				}
				else if((d_graph_type == TGraphWidget::LINE_GRAPH && (*set_iter).dataset_graph_type == TGraphWidget::DEFAULT_GRAPH) ||
				   (*set_iter).dataset_graph_type == TGraphWidget::LINE_GRAPH ||
				   d_graph_type == TGraphWidget::DEFAULT_GRAPH)
				{
					painter->setPen((*set_iter).dataset_color);
					
					init = true;
					for(pnt_iter = (*set_iter).dataset_points.begin();pnt_iter != (*set_iter).dataset_points.end();pnt_iter++)
					{
						if(init)
						{
							d_viewport_mat.Position_To_Viewport((*pnt_iter).x,(*pnt_iter).y,&LastX,&LastY);
							init = false;

							if(d_horizontal_legend_mode == TGraphWidget::LOG10_LEGEND_PRECISION)
							{
								LastX = this->Calculate_Log10_Position_Horizontal((*pnt_iter).x);
							}

							if(d_vertical_legend_mode == TGraphWidget::LOG10_LEGEND_PRECISION)
							{
								LastY = this->Calculate_Log10_Position_Vertical((*pnt_iter).y);
							}

						}
						else
						{
							d_viewport_mat.Position_To_Viewport((*pnt_iter).x,(*pnt_iter).y,&X,&Y);
							
							
							if(d_horizontal_legend_mode == TGraphWidget::LOG10_LEGEND_PRECISION)
							{
								X = this->Calculate_Log10_Position_Horizontal((*pnt_iter).x);
							}

							if(d_vertical_legend_mode == TGraphWidget::LOG10_LEGEND_PRECISION)
							{
								Y = this->Calculate_Log10_Position_Vertical((*pnt_iter).y);
							}

							painter->drawLine(LastX,LastY,X,Y);
							LastX = X;
							LastY = Y;
						}
					}
				}
				else if((d_graph_type == TGraphWidget::POINT_GRAPH && (*set_iter).dataset_graph_type == TGraphWidget::DEFAULT_GRAPH) ||
						(*set_iter).dataset_graph_type == TGraphWidget::POINT_GRAPH)
				{						
					painter->setPen((*set_iter).dataset_color);
					
					for(pnt_iter = (*set_iter).dataset_points.begin();pnt_iter != (*set_iter).dataset_points.end();pnt_iter++)
					{
						d_viewport_mat.Position_To_Viewport((*pnt_iter).x,(*pnt_iter).y,&X,&Y);
						
						if(d_horizontal_legend_mode == TGraphWidget::LOG10_LEGEND_PRECISION)
						{
							X = this->Calculate_Log10_Position_Horizontal((*pnt_iter).x);
						}

						if(d_vertical_legend_mode == TGraphWidget::LOG10_LEGEND_PRECISION)
						{
							Y = this->Calculate_Log10_Position_Vertical((*pnt_iter).y);
						}
						
						painter->drawLine(X-3,Y-3,X+3,Y+3);
						painter->drawLine(X-3,Y+3,X+3,Y-3);
					}
				}
				else if((d_graph_type == TGraphWidget::DOT_GRAPH && (*set_iter).dataset_graph_type == TGraphWidget::DEFAULT_GRAPH) ||
						(*set_iter).dataset_graph_type == TGraphWidget::DOT_GRAPH)
				{
					painter->setPen((*set_iter).dataset_color);
					
					for(pnt_iter = (*set_iter).dataset_points.begin();pnt_iter != (*set_iter).dataset_points.end();pnt_iter++)
					{
						d_viewport_mat.Position_To_Viewport((*pnt_iter).x,(*pnt_iter).y,&X,&Y);
												
						if(d_horizontal_legend_mode == TGraphWidget::LOG10_LEGEND_PRECISION)
						{
							X = this->Calculate_Log10_Position_Horizontal((*pnt_iter).x);
						}

						if(d_vertical_legend_mode == TGraphWidget::LOG10_LEGEND_PRECISION)
						{
							Y = this->Calculate_Log10_Position_Vertical((*pnt_iter).y);
						}
						
						painter->fillRect(X-3,Y-3,6,6,(*set_iter).dataset_color);
					}
				}
			}
		}
		
		painter->setClipping(false);
	}
}

void TGraphWidget::Calculate_Legend_Display_Precision(
	QPainter							*painter)
{
	double								max_x,max_y;
	double								min_x,min_y;
    double								largest_x,largest_y;
    double								range_x,range_y;
    QString								text;
    QRect								bounding_rect;

    d_viewport_mat.Viewport_To_Position(
    			d_graph_rect.left(),
				d_graph_rect.top(),
				&min_x,
				&max_y);

	d_viewport_mat.Viewport_To_Position(
				d_graph_rect.right(),
				d_graph_rect.bottom(),
				&max_x,
				&min_y);

	if(fabs(max_x) > fabs(min_x))
    {
		largest_x = fabs(max_x);
	}
    else
    {
    	largest_x = fabs(min_x);
    }

    if(fabs(max_y) > fabs(min_y))
    {
    	largest_y = fabs(max_y);
    }
    else
    {
    	largest_y = fabs(min_y);
    }

    range_x = max_x - min_x;
    range_y = max_y - min_y;

    if(range_x < 1)
    {
    	range_x = 1;
    }

    if(range_y < 1)
    {
    	range_y = 1;
    }
	
	d_horizontal_text_total = 8;
	d_horizontal_text_precision = 4;

    if(d_horizontal_legend_mode == TGraphWidget::DEFAULT_LEGEND_PRECISION)
    {
		if(largest_x > 1)
		{
			d_horizontal_text_precision = 4 - static_cast<int>(log10(range_x));

			if(d_horizontal_text_precision < 0)
			{
				d_horizontal_text_precision = 0;
			}
			
			d_horizontal_text_total = static_cast<int>(log10(largest_x)) + d_horizontal_text_precision + 2;	// add decimal and sign
		}
    }
	else if(d_horizontal_legend_mode == TGraphWidget::LOG10_LEGEND_PRECISION)
	{
		d_horizontal_text_total = 8;
		d_horizontal_text_precision = 1;
	}
    else
    {
		d_horizontal_text_total = 5;
		d_horizontal_text_precision = 0;
    }

	text = QString("  %1  ").arg(-0.5,d_horizontal_text_total,'f',d_horizontal_text_precision);	// extra spaces uncrowds the horizontal axis

	bounding_rect = painter->boundingRect(0,0,0,0,0,text);

	d_horizontal_label_size = QSize(bounding_rect.width(),bounding_rect.height());
	
	if(d_horizontal_label_size.width() < 1)
	{
		d_horizontal_label_size.setWidth(1);
	}
	
	if(d_horizontal_label_size.height() < 1)
	{
		d_horizontal_label_size.setHeight(1);
	}
	
	d_vertical_text_total = 8;
	d_vertical_text_precision = 4;
	
    if(d_vertical_legend_mode == TGraphWidget::DEFAULT_LEGEND_PRECISION)
    {
		if(largest_y > 1)
		{
			d_vertical_text_precision = 4 - static_cast<int>(log10(range_y));

			if(d_vertical_text_precision < 0)
			{
				d_vertical_text_precision = 0;
			}
			
			d_vertical_text_total = static_cast<int>(log10(largest_y)) + d_vertical_text_precision + 2;	// add decimal and sign
		}
    }
	else if(d_vertical_legend_mode == TGraphWidget::LOG10_LEGEND_PRECISION)
	{
		d_vertical_text_total = 6;
		d_vertical_text_precision = 1;
	}
    else
    {
		d_vertical_text_total = 5;
		d_vertical_text_precision = 0;
	}

	text = QString("  %1").arg(-0.5,d_vertical_text_total,'f',d_vertical_text_precision);

	bounding_rect = painter->boundingRect(0,0,0,0,0,text);
	d_vertical_label_size = QSize(bounding_rect.width(),bounding_rect.height());
	
	if(d_vertical_label_size.width() < 1)
	{
		d_vertical_label_size.setWidth(1);
	}
	
	if(d_vertical_label_size.height() < 1)
	{
		d_vertical_label_size.setHeight(1);
	}
}

void TGraphWidget::Update_Left_Down_Box(void)
{
	QRect								rect(d_mouse_down_rect.normalized());

	update(rect.left(),rect.top(),rect.width(),1);
	update(rect.left(),rect.top(),1,rect.height());
	update(rect.left(),rect.bottom()+1,rect.width(),1);
	update(rect.right()+1,rect.top(),1,rect.height());
}

int TGraphWidget::Forward_Log10_Position_Horizontal(
	const int 							x) const
{
	double								log_in;
	double								log_result;
	int									x_in;
	int									range_x;
	int									result;
	
	range_x = d_graph_rect.width();
	x_in = x - d_graph_rect.left();

	if(!range_x)
	{
		return x;
	}
	
	if(x_in < 0 || x_in > range_x)
	{
		return x;
	}
	
	log_in = static_cast<double>(x_in) / static_cast<double>(range_x);
	log_in *= 9.0;
	log_in += 1.0;
	
	log_result = log10(log_in);
	
	result = static_cast<int>(log_result * static_cast<double>(range_x) + 0.5);
	result += d_graph_rect.left();

	return result;
}

int TGraphWidget::Forward_Log10_Position_Vertical(
	const int 							y) const
{
	double								log_in;
	double								log_result;
	int									y_in;
	int									range_y;
	int									result;

	range_y = d_graph_rect.height();
	y_in = y - d_graph_rect.top();

	if(!range_y)
	{
		return y;
	}
	
	if(y_in < 0 || y_in > range_y)
	{
		return y;
	}
	
	log_in = static_cast<double>(range_y - y_in) / static_cast<double>(range_y);
	log_in *= 9.0;
	log_in += 1.0;
	
	log_result = log10(log_in);
	
	result = range_y - static_cast<int>(log_result * static_cast<double>(range_y) + 0.5);
	result += d_graph_rect.top();

	return result;
}

int TGraphWidget::Reverse_Log10_Position_Horizontal(
	const int 							x) const
{
	double								log_in;
	double								log_result;
	int									x_in;
	int									range_x;
	int									result;
	
	range_x = d_graph_rect.width();
	x_in = x - d_graph_rect.left();

	if(!range_x)
	{
		return x;
	}
	
	if(x_in < 0 || x_in > range_x)
	{
		return x;
	}
	
	log_in = static_cast<double>(range_x - x_in) / static_cast<double>(range_x);
	log_in *= 9.0;
	log_in += 1.0;
	
	log_result = log10(log_in);
	
	result = range_x - static_cast<int>(log_result * static_cast<double>(range_x) + 0.5);
	result += d_graph_rect.left();

	return result;
}

int TGraphWidget::Reverse_Log10_Position_Vertical(
	const int 							y) const
{
	double								log_in;
	double								log_result;
	int									y_in;
	int									range_y;
	int									result;

	range_y = d_graph_rect.height();
	y_in = y - d_graph_rect.top();

	if(!range_y)
	{
		return y;
	}
	
	if(y_in < 0 || y_in > range_y)
	{
		return y;
	}
	
	log_in = static_cast<double>(y_in) / static_cast<double>(range_y);
	log_in *= 9.0;
	log_in += 1.0;
	
	log_result = log10(log_in);
	
	result = static_cast<int>(log_result * static_cast<double>(range_y) + 0.5);
	result += d_graph_rect.top();

	return result;
}

int TGraphWidget::Calculate_Log10_Position_Horizontal(
	const double 						&x) const
{
	double								max_x,max_y;
	double								min_x,min_y;
	double								scale;
	double								value;
	double								range_x;
	double								x_in;
	int									x_pos,y_pos;
	
	if(!d_graph_rect.width())
	{
		return 0;
	}

	d_viewport_mat.Viewport_To_Position(
			 d_graph_rect.left(),
			 d_graph_rect.top(),
			 &min_x,
			 &max_y);

	d_viewport_mat.Viewport_To_Position(
			 d_graph_rect.right(),
			 d_graph_rect.bottom(),
			 &max_x,
			 &min_y);
		
	if(x < min_x)
	{
		return d_graph_rect.left();
	}
	
	if(x > max_x)
	{
		return d_graph_rect.right();
	}
	
	range_x = max_x - min_x;
	x_in = x - min_x;
	
	scale = (range_x - x_in) / range_x;
			
	value = pow(10,scale);
	value -= 1.0;
	value /= 9.0;
	value = 1.0 - value;
	value *= range_x;
	value += min_x;
	
	d_viewport_mat.Position_To_Viewport(value,0.0,&x_pos,&y_pos);

	return x_pos;
}

double TGraphWidget::Calculate_Log10_Value_Horizontal(
	const int 							x) const
{
	double								xval,yval;
	int									x_in;
		
	x_in = Reverse_Log10_Position_Horizontal(x);

	d_viewport_mat.Viewport_To_Position(x_in,0,&xval,&yval);
	return xval;
}

int TGraphWidget::Calculate_Log10_Position_Vertical(
	const double 						&y) const
{
	double								max_x,max_y;
	double								min_x,min_y;
	double								scale;
	double								value;
	double								range_y;
	double								y_in;
	int									x_pos,y_pos;

	if(!d_graph_rect.height())
	{
		return 0;
	}

	d_viewport_mat.Viewport_To_Position(
			 d_graph_rect.left(),
			 d_graph_rect.top(),
			 &min_x,
			 &max_y);

	d_viewport_mat.Viewport_To_Position(
			 d_graph_rect.right(),
			 d_graph_rect.bottom(),
			 &max_x,
			 &min_y);
		
	if(y < min_y)
	{
		return d_graph_rect.bottom();
	}
	
	if(y > max_y)
	{
		return d_graph_rect.top();
	}
	
	range_y = max_y - min_y;
	y_in = y - min_y;

	scale = (range_y - y_in) / range_y;

	value = pow(10,scale);
	value -= 1.0;
	value /= 9.0;
	value = 1.0 - value;
	value *= range_y;
	value += min_y;

	d_viewport_mat.Position_To_Viewport(0.0,value,&x_pos,&y_pos);

	return y_pos;
}

double TGraphWidget::Calculate_Log10_Value_Vertical(
	const int 							y) const
{
	double								xval,yval;
	int									y_in;
	
	y_in = Reverse_Log10_Position_Vertical(y);

	d_viewport_mat.Viewport_To_Position(0,y_in,&xval,&yval);
	return yval;
}
