/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <assert.h>

#include "referencesystem.h"

TReferenceSystem::TReferenceSystem(void)
{

}

TReferenceSystem::TReferenceSystem(
     const TReferenceSystem						&rhs)
{
	d_origin = rhs.d_origin;
	d_rotation_mat = rhs.d_rotation_mat;
	d_rotation_mat_inv = rhs.d_rotation_mat_inv;
}

TReferenceSystem::~TReferenceSystem(void)
{

}

TReferenceSystem& TReferenceSystem::operator=(
	const TReferenceSystem						&rhs)
{
	if (&rhs != this)
	{
		d_origin = rhs.d_origin;
		d_rotation_mat = rhs.d_rotation_mat;
		d_rotation_mat_inv = rhs.d_rotation_mat_inv;
	}

	return *this;
}

void TReferenceSystem::Clear_Alignment(void)
{
	d_origin.Set(0,0,0);
	d_rotation_mat.Identity();
	d_rotation_mat_inv.Identity();
}

void TReferenceSystem::Skew1(
	const TVector3						&axis,
	const TAxis							axis_name)
{
	TVector3							primary_axis(axis);
	TVector3							secondary_axis_a;
	TVector3							secondary_axis_b;

	if(primary_axis.Length() > std::numeric_limits<double>::epsilon())
	{
		primary_axis.Normal();

		switch(axis_name)
		{
			case XPLUS:
			case XMINUS:
				secondary_axis_a = d_rotation_mat.Z() * primary_axis;
				secondary_axis_b = primary_axis * d_rotation_mat.Y();
				break;

			case YPLUS:
			case YMINUS:
				secondary_axis_a = d_rotation_mat.X() * primary_axis;
				secondary_axis_b = primary_axis * d_rotation_mat.Z();
				break;

			case ZPLUS:
			case ZMINUS:
				secondary_axis_a = d_rotation_mat.Y() * primary_axis;
				secondary_axis_b = primary_axis * d_rotation_mat.X();
				break;
		}

		if(secondary_axis_a.Length() > secondary_axis_b.Length())
		{
			secondary_axis_a.Normal();
			secondary_axis_b = primary_axis * secondary_axis_a;
		}
		else
		{
			secondary_axis_b.Normal();
			secondary_axis_a = secondary_axis_b * primary_axis;
		}

		switch(axis_name)
		{
			case XPLUS:
				d_rotation_mat.X(primary_axis);
				d_rotation_mat.Y(secondary_axis_a);
				d_rotation_mat.Z(secondary_axis_b);
				break;

			case XMINUS:
				d_rotation_mat.X(-1 * primary_axis);
				d_rotation_mat.Y(secondary_axis_a);
				d_rotation_mat.Z(-1 * secondary_axis_b);
				break;

			case YPLUS:
				d_rotation_mat.Y(primary_axis);
				d_rotation_mat.Z(secondary_axis_a);
				d_rotation_mat.X(secondary_axis_b);
				break;

			case YMINUS:
				d_rotation_mat.Y(-1 * primary_axis);
				d_rotation_mat.Z(secondary_axis_a);
				d_rotation_mat.X(-1 * secondary_axis_b);
				break;

			case ZPLUS:
				d_rotation_mat.Z(primary_axis);
				d_rotation_mat.X(secondary_axis_a);
				d_rotation_mat.Y(secondary_axis_b);
				break;

			case ZMINUS:
				d_rotation_mat.Z(-1 * primary_axis);
				d_rotation_mat.X(secondary_axis_a);
				d_rotation_mat.Y(-1 * secondary_axis_b);
				break;
		}

		d_rotation_mat_inv = d_rotation_mat;
		d_rotation_mat_inv.Transpose();
	}
}

void TReferenceSystem::Skew2(
	const TVector3						&axis,
	const TAxis							axis_name,
	const TAxis							fixed_axis_name)
{
	TVector3							fixed_axis;
	TVector3							rotate_axis;
	TVector3							third_axis;

	assert(fixed_axis_name != axis_name);

	if(axis.Length() > std::numeric_limits<double>::epsilon())
	{
		switch(fixed_axis_name)
		{
			case XPLUS:
			case XMINUS:
				fixed_axis = d_rotation_mat.X();
				break;

			case YPLUS:
			case YMINUS:
				fixed_axis = d_rotation_mat.Y();
				break;

			case ZPLUS:
			case ZMINUS:
				fixed_axis = d_rotation_mat.Z();
				break;
		}

		third_axis = fixed_axis * axis;

		if(third_axis.Length() > std::numeric_limits<double>::epsilon())
		{
			third_axis.Normal();

			rotate_axis = third_axis * fixed_axis;

// Note:     The direction of the third axis is dependent on the selection of the major axis.
// Example:  fixed_axis_name=ZPLUS/ZMINUS therefore the major axis is X.  If 'axis_name' axis is Y the
//			 rotation from Y to X is negative and the third axis will always be reversed.

			if(fixed_axis_name == XPLUS || fixed_axis_name == XMINUS)
			{
				if(axis_name == YPLUS)
				{
					d_rotation_mat.Y(rotate_axis);
					d_rotation_mat.Z(third_axis);
				}
				else if(axis_name == YMINUS)
				{
					d_rotation_mat.Y(-1 * rotate_axis);
					d_rotation_mat.Z(-1 * third_axis);
				}
				else if(axis_name == ZPLUS)
				{
					d_rotation_mat.Y(-1 * third_axis);
					d_rotation_mat.Z(rotate_axis);
				}
				else if(axis_name == ZMINUS)
				{
					d_rotation_mat.Y(third_axis);
					d_rotation_mat.Z(-1 * rotate_axis);
				}
			}
			else if(fixed_axis_name == YPLUS || fixed_axis_name == YMINUS)
			{
				if(axis_name == XPLUS)
				{
					d_rotation_mat.X(rotate_axis);
					d_rotation_mat.Z(-1 * third_axis);
				}
				else if(axis_name == XMINUS)
				{
					d_rotation_mat.X(-1 * rotate_axis);
					d_rotation_mat.Z(third_axis);
				}
				else if(axis_name == ZPLUS)
				{
					d_rotation_mat.X(third_axis);
					d_rotation_mat.Z(rotate_axis);
				}
				else if(axis_name == ZMINUS)
				{
					d_rotation_mat.X(-1 * third_axis);
					d_rotation_mat.Z(-1 * rotate_axis);
				}
			}
			else if(fixed_axis_name == ZPLUS || fixed_axis_name == ZMINUS)
			{
				if(axis_name == XPLUS)
				{
					d_rotation_mat.X(rotate_axis);
					d_rotation_mat.Y(third_axis);
				}
				else if(axis_name == XMINUS)
				{
					d_rotation_mat.X(-1 * rotate_axis);
					d_rotation_mat.Y(-1 * third_axis);
				}
				else if(axis_name == YPLUS)
				{
					d_rotation_mat.X(-1 * third_axis);
					d_rotation_mat.Y(rotate_axis);
				}
				else if(axis_name == YMINUS)
				{
					d_rotation_mat.X(third_axis);
					d_rotation_mat.Y(-1 * rotate_axis);
				}
			}

			d_rotation_mat_inv = d_rotation_mat;
			d_rotation_mat_inv.Transpose();
		}
	}
}


void TReferenceSystem::Set_Origin(
	const TVector3						&pnt,
	const TReferenceSystem::TAxis		axis_name)
{
	TVector3							ref_axis;
	double								distance;
	
	switch(axis_name)
	{
		case TReferenceSystem::XPLUS:
		case TReferenceSystem::XMINUS:
			ref_axis = d_rotation_mat.X();
			break;
			
		case TReferenceSystem::YPLUS:
		case TReferenceSystem::YMINUS:
			ref_axis = d_rotation_mat.Y();
			break;
			
		case TReferenceSystem::ZPLUS:
		case TReferenceSystem::ZMINUS:
			ref_axis = d_rotation_mat.Z();
			break;
	}
	
	distance = (d_origin - pnt) ^ ref_axis;
	d_origin -= (distance * ref_axis);
}

void TReferenceSystem::Offset_Origin(
	const double						&offset,
	const TReferenceSystem::TAxis		axis_name)
{
	TVector3							ref_axis;
	
	switch(axis_name)
	{
		case TReferenceSystem::XPLUS:
		case TReferenceSystem::XMINUS:
			ref_axis = d_rotation_mat.X();
			break;
			
		case TReferenceSystem::YPLUS:
		case TReferenceSystem::YMINUS:
			ref_axis = d_rotation_mat.Y();
			break;
			
		case TReferenceSystem::ZPLUS:
		case TReferenceSystem::ZMINUS:
			ref_axis = d_rotation_mat.Z();
			break;
	}
	
	d_origin += (offset * ref_axis);
}

void TReferenceSystem::Rotate(
	const double						&angle,
	const TAxis							fixed_axis_name)
{
	TVector3							fixed_axis;

	switch(fixed_axis_name)
	{
		case XPLUS:
		case XMINUS:
			fixed_axis = d_rotation_mat.X();
			break;

		case YPLUS:
		case YMINUS:
			fixed_axis = d_rotation_mat.Y();
			break;

		case ZPLUS:
		case ZMINUS:
			fixed_axis = d_rotation_mat.Z();
			break;
	}

	fixed_axis = d_rotation_mat * fixed_axis;
	fixed_axis.Normal(); // The length may creep if the axis are not orthogonal

	d_rotation_mat.Rotate(fixed_axis, angle);

	d_rotation_mat_inv = d_rotation_mat;
	d_rotation_mat_inv.Transpose();
}

