/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#ifndef REFERENCESYSTEMHEADERFILE
#define REFERENCESYSTEMHEADERFILE

#include "vector3.h"
#include "mat4.h"

class TReferenceSystem
{
public:

enum TAxis
{
	XPLUS=0,
	YPLUS,
	ZPLUS,
	XMINUS,
	YMINUS,
	ZMINUS
};

// CREATORS
	TReferenceSystem(void);
	TReferenceSystem(const TReferenceSystem&);
	~TReferenceSystem(void);

// ACCESSORS
	TVector3 ToWorld(const TVector3 &pnt) const;			// convert data from alignment to world axis system
	TVector3 FromWorld(const TVector3 &pnt) const;			// convert data from world axis system to alignment
	
	TVector3 AxisToWorld(const TVector3 &axis) const;		// does not translate origin
	TVector3 AxisFromWorld(const TVector3 &axis) const;		// does not translate origin

	TVector3 Origin(void) const;
	TMat4 Rotation_Mat(void) const;
	TMat4 Rotation_Mat_Inv(void) const;
	
// MANIPULATORS
	void Clear_Alignment(void);
	void Set_Rotation_Mat(const TMat4 &mat);

	void Skew1(const TVector3 &axis,const TAxis axis_name);
	void Skew2(const TVector3 &axis,const TAxis axis_name,const TAxis fixed_axis_name);
	
	void Set_Origin(const TVector3 &pnt);
	void Set_Origin(const TVector3 &pnt,const TReferenceSystem::TAxis axis_name);
	void Offset_Origin(const double &offset,const TReferenceSystem::TAxis axis_name);
	
	void Rotate(const double &angle,const TAxis fixed_axis_name);

	TReferenceSystem& operator=(const TReferenceSystem&);

private:
	TVector3							d_origin;
	TMat4								d_rotation_mat;
	TMat4								d_rotation_mat_inv;
};

inline TVector3 TReferenceSystem::ToWorld(
    const TVector3						&pnt) const
{
	return (d_rotation_mat_inv * pnt) + d_origin;
}

inline TVector3 TReferenceSystem::FromWorld(
    const TVector3						&pnt) const
{
	return d_rotation_mat * (pnt - d_origin);
}

inline TVector3 TReferenceSystem::AxisToWorld(
	const TVector3						&axis) const
{
	return d_rotation_mat_inv * axis;
}

inline TVector3 TReferenceSystem::AxisFromWorld(
	const TVector3						&axis) const
{
	return d_rotation_mat * axis;
}

inline TVector3 TReferenceSystem::Origin(void) const
{
	return d_origin;
}

inline TMat4 TReferenceSystem::Rotation_Mat(void) const
{
	return d_rotation_mat;
}

inline TMat4 TReferenceSystem::Rotation_Mat_Inv(void) const
{
	return d_rotation_mat_inv;
}

inline void TReferenceSystem::Set_Rotation_Mat(
	const TMat4							&mat)
{
	d_rotation_mat.X(mat.X());
	d_rotation_mat.Y(mat.Y());
	d_rotation_mat.Z(mat.Z());
	d_rotation_mat_inv = d_rotation_mat;
	d_rotation_mat_inv.Transpose();
}

inline void TReferenceSystem::Set_Origin(
	const TVector3						&pnt)
{
	d_origin = pnt;
}



#endif
