/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QVariant>
#include <QAction>
#include <QApplication>
#include <QGridLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QPushButton>
#include <QSpacerItem>
#include <QTableWidgetSelectionRange>
#include <QTextEdit>
#include <QScrollBar>
#include <QHeaderView>
#include <QPalette>
#include <QKeyEvent>
#include <QFocusEvent>
#include <QFontMetrics>
#include <QMenu>
#include <QContextMenuEvent>
#include <QClipboard>

#include <assert.h>

#include "tableeditor.h"

#if (QT_VERSION >= QT_VERSION_CHECK(5, 14, 0))
// const static Qt::SplitBehavior 			g_split_behavior_skip = Qt::SkipEmptyParts;
const static Qt::SplitBehavior 			g_split_behavior_keep = Qt::KeepEmptyParts;
#else
// const static QString::SplitBehavior 	g_split_behavior_skip = QString::SkipEmptyParts;
const static QString::SplitBehavior 	g_split_behavior_keep = QString::KeepEmptyParts;
#endif

static const int						CUT_COPY_PRECISON(16);

TTableLineEditor::TTableLineEditor(
	QWidget								*Parent)
:QLineEdit(Parent)
{
	QPalette 							Palette(this->palette());
	
	this->setFrame(QFrame::Panel | QFrame::Plain);
	this->setPalette(Palette);
	this->setAlignment(Qt::AlignRight);
}

TTableLineEditor::~TTableLineEditor(void)
{

}

void TTableLineEditor::keyPressEvent(
	QKeyEvent 							*KeyEvent)
{
	if(KeyEvent->key() == Qt::Key_Return ||
		KeyEvent->key() == Qt::Key_Enter ||
		KeyEvent->key() == Qt::Key_Escape)
	{
		this->hide();
	}
	
	
	QLineEdit::keyPressEvent(KeyEvent);
}


void TTableLineEditor::focusOutEvent(
	QFocusEvent 						*FocusEvent)
{
	if(FocusEvent->lostFocus())
	{
		this->hide();
		FocusEvent->ignore();
		
		emit Edit_Complete();
	}
}

TEditTable::TEditTable(
	QWidget								*Parent)
:QTableWidget(Parent)		
{
	d_table_editor = new TTableLineEditor(this);
	d_table_editor->hide();

	d_cell_data = 0;
	d_column_count = 0;
	d_row_count = 0;
	d_table_editor_changed = false;
	d_edit_mode = TEditTable::EDIT_MODE_REPLACE;
	d_selection_mode = TEditTable::SELECTION_MODE_DOWN;
	d_display_precision = 1;
	
	d_enable_invert_action = true;
	d_readonly_mode = false;
	
	d_selection_rect.setCoords(-1,-1,-1,-1);
	
	this->setSelectionMode(QAbstractItemView::ExtendedSelection);
	
	// create actions
	d_cut_action = new QAction(this);
	d_cut_action->setText(QStringLiteral("&Cut"));
	connect(d_cut_action,&QAction::triggered,this,&TEditTable::Edit_Cut);
	
	d_copy_action = new QAction(this);
	d_copy_action->setText(QStringLiteral("C&opy"));
	connect(d_copy_action,&QAction::triggered,this,&TEditTable::Edit_Copy);
	
	d_paste_action = new QAction(this);
	d_paste_action->setText(QStringLiteral("&Paste"));
	d_paste_action->setEnabled(false);
	connect(d_paste_action,&QAction::triggered,this,&TEditTable::Edit_Paste);
	
	d_delete_action = new QAction(this);
	d_delete_action->setText(QStringLiteral("&Delete"));
	connect(d_delete_action,&QAction::triggered,this,&TEditTable::Edit_Delete);
	
	d_invert_action = new QAction(this);
	d_invert_action->setText(QStringLiteral("&Invert Sign"));
	connect(d_invert_action,&QAction::triggered,this,&TEditTable::Edit_Invert);
	
	connect(this,&TEditTable::itemSelectionChanged,this,&TEditTable::Table_Selection_Changed);
	
	connect(d_table_editor,&TTableLineEditor::textChanged,this,&TEditTable::Cell_Editor_Changed);
	connect(d_table_editor,&TTableLineEditor::Edit_Complete,this,&TEditTable::Cell_Editor_Closed);
	
	connect(QApplication::clipboard(),&QClipboard::dataChanged,this,&TEditTable::Clipboard_Data_Changed);
}

TEditTable::~ TEditTable(void)
{
	if(d_cell_data)
	{
		delete[] d_cell_data;
		
		d_cell_data = 0;
		d_column_count = 0;
		d_row_count = 0;
	}
}


QSize TEditTable::sizeHint(void) const
{
	int									min_width(0);
	int									cntr;
	QSize								size;
	
	for(cntr = 0;cntr < d_column_count;cntr++)
	{
		min_width += this->columnWidth(cntr);
	}
	
	min_width += this->verticalHeader()->sizeHint().width();
	min_width += this->verticalScrollBar()->sizeHint().width();
		
	size.setWidth(min_width);
	size.setHeight(min_width);
	
	return size;
}

QSize TEditTable::minimumSizeHint(void) const
{
	QSize								size;
	int									min_width(0);

	min_width += this->verticalHeader()->sizeHint().width();
	min_width += this->verticalScrollBar()->sizeHint().width();

	size.setWidth(min_width);
	size.setHeight(min_width);
	
	return size;
}


double TEditTable::Cell_Value(
	const int							row,
	const int							column) const
{	
	return d_cell_data[this->Row_Column_To_Offset(row,column)];
}


int TEditTable::Row_Column_To_Offset(
	const int 							row,
	const int 							column) const
{
	assert(d_row_count);
	assert(d_column_count);
	
	return row + column * d_row_count;
}

void TEditTable::Offset_To_Row_Column(
	int 								*row,
	int 								*column,
	const int 							offset) const
{
	assert(d_row_count);
	assert(d_column_count);
	assert(offset < (d_row_count * d_column_count));
	
	*column = offset / d_row_count;
	*row = offset - (*column * d_row_count);
}

void TEditTable::Set_Cell_Count(
	const int 							rows,
	const int 							columns)
{
	int									row_cntr;
	int									column_cntr;
	QTableWidgetItem					*table_item;
	QRect								font_rect;
	QFontMetrics						font_metrics(qApp->font());
	
	this->clear();
	this->setRowCount(0);
	this->setColumnCount(0);

	if(d_cell_data)
	{
		delete[] d_cell_data;
		d_cell_data = 0;
	}
	
	if(rows < 1 || columns < 1)
	{
		return;
	}
	
	font_rect = font_metrics.boundingRect("99999j999^");
	font_rect.adjust(-2,-2,2,6);
	
	d_cell_data = new double[rows * columns];
	memset(d_cell_data,0,sizeof(double) * rows * columns);
	
	d_column_count = columns;
	d_row_count = rows;
	
	this->setUpdatesEnabled(false);
	
	this->setRowCount(rows);
	this->setColumnCount(columns);
	
	for(row_cntr = 0;row_cntr < rows;row_cntr++)
	{
		table_item = new QTableWidgetItem();
		table_item->setTextAlignment(Qt::AlignRight);
		
		this->setVerticalHeaderItem(row_cntr,table_item);
		
		for(column_cntr = 0;column_cntr < columns;column_cntr++)
		{
			table_item = new QTableWidgetItem();
			table_item->setTextAlignment(Qt::AlignRight);
			table_item->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
			this->setItem(row_cntr,column_cntr,table_item);
			
		}
		
		this->setRowHeight(row_cntr,font_rect.height());
	}
	
	for(column_cntr = 0;column_cntr < columns;column_cntr++)
	{
		table_item = new QTableWidgetItem();
		this->setHorizontalHeaderItem(column_cntr,table_item);
		
		this->setColumnWidth(column_cntr,font_rect.width());
	}
	
	this->setUpdatesEnabled(true);

}

void TEditTable::Set_Cell_Value(
	const int 							row,
	const int 							column,
	const double 						&value)
{
	QString								text_value;
	
	d_cell_data[this->Row_Column_To_Offset(row,column)] = value;
	
	text_value = QString("%1").arg(value,0,'f',d_display_precision);
	
	this->item(row,column)->setText(text_value);
}

void TEditTable::Set_Cell_Values(
	const double						*values)
{
	int									index;
	int									max_index;
	
	assert(values);
	
	this->clearSelection();
	
	max_index = d_row_count * d_column_count;
	
	for(index=0;index < max_index;index++)
	{
		d_cell_data[index] = values[index];
	}
	
	// update display
	this->Set_Display_Precision(d_display_precision);
}

void TEditTable::Set_Edit_Mode(
	const TEditTable::TEditMode			&edit_mode)
{
	d_edit_mode = edit_mode;
}

void TEditTable::Set_Selection_Mode(
	const TEditTable::TSelectionMode	&selection_mode)
{
	d_selection_mode = selection_mode;
}

void TEditTable::Set_Display_Precision(
	const int							precision)
{
	int									row;
	int									column;
	int									index;
	QString								text;
	
	d_display_precision = precision;
	
	this->setUpdatesEnabled(false);

	this->clearSelection();
	
	for(row = 0;row < d_row_count;row++)
	{
		for(column = 0;column < d_column_count;column++)
		{
			index = this->Row_Column_To_Offset(row,column);
			
			text = QString("%1").arg(d_cell_data[index],0,'f',d_display_precision);
			
			this->item(row,column)->setText(text);
		}
	}
	
	this->setUpdatesEnabled(true);
}

void TEditTable::Resize_Columns_To_Contents(
	const bool							force_min_width)
{
	int									min_width(2);
	int									cntr;
	
	this->resizeColumnsToContents();
	
	if(!force_min_width)
	{
		return;
	}
	
	for(cntr = 0;cntr < d_column_count;cntr++)
	{
		min_width += this->columnWidth(cntr);
	}
	
	min_width += this->verticalHeader()->sizeHint().width();
	
	if(this->verticalScrollBar()->isVisible())
	{
		min_width += this->verticalScrollBar()->sizeHint().width();
	}

	this->setMinimumWidth(min_width);
}

void TEditTable::keyPressEvent(
	QKeyEvent 							*key_event)
{
	int									current_row;
	int									current_column;
	int									max_rows;
											 
	if(key_event->key() == Qt::Key_Return ||
	   key_event->key() == Qt::Key_Enter)
	{
		max_rows = this->rowCount();
		current_row = this->currentRow();
		current_column = this->currentColumn();
		
		if(d_selection_mode == TEditTable::SELECTION_MODE_UP)
		{
			current_row --;
			
			if(current_row < 0)
			{
				current_row = max_rows - 1;
			}
		}
		else
		{
			current_row++;
			
			if(!(current_row < max_rows))
			{
				current_row = 0;
			}
		}
		
		
		this->setCurrentCell(current_row,current_column);
//		this->setItemSelected(this->currentItem(),true);	// depreciated
		this->currentItem()->setSelected(true);
		
		key_event->accept();
		
		return;
	}
	else
	{
		switch(key_event->key())
		{
			case Qt::Key_0:
			case Qt::Key_1:
			case Qt::Key_2:
			case Qt::Key_3:
			case Qt::Key_4:
			case Qt::Key_5:
			case Qt::Key_6:
			case Qt::Key_7:
			case Qt::Key_8:
			case Qt::Key_9:
			case Qt::Key_Minus:
			case Qt::Key_Plus:
			case Qt::Key_Period:
				
				current_row = this->currentRow();
				current_column = this->currentColumn();
				
				if(!d_readonly_mode)
				{
					this->Activate_Cell(current_row,current_column);
					d_table_editor->setText(key_event->text());
				}
				
				return;
				
#ifdef Q_OS_MAC	
			case Qt::Key_Backspace:
#else
			case Qt::Key_Delete:
#endif
				this->Edit_Delete();
				return;				
		}
		
		if(key_event->modifiers() == Qt::CTRL)
		{
			switch(key_event->key())
			{
				case Qt::Key_X:
					if(!d_readonly_mode)
					{
						this->Edit_Cut();
					}
					return;
					
				case Qt::Key_C:
					this->Edit_Copy();
					return;
					
				case Qt::Key_V:
					if(!d_readonly_mode)
					{
						this->Edit_Paste();
					}
					return;
			}
		}
	}
	
	key_event->ignore();
}

void TEditTable::focusOutEvent(
	QFocusEvent 						*focus_event)
{
	if(focus_event->reason() != Qt::PopupFocusReason)
	{
		this->setRangeSelected(
			   QTableWidgetSelectionRange(d_selection_rect.top(),
										  d_selection_rect.left(),
										  d_selection_rect.bottom(),
										  d_selection_rect.right()),
												false);
	}
}

void TEditTable::contextMenuEvent(
	QContextMenuEvent					*menu_event)
{
    QMenu								menu(this);
	
	if(!d_readonly_mode)
	{
		menu.addAction(d_cut_action);
	}
	
    menu.addAction(d_copy_action);
	
	if(!d_readonly_mode)
	{
		menu.addAction(d_paste_action);
		menu.addAction(d_delete_action);
	}
	
	if(d_enable_invert_action)
	{
		menu.addSeparator();
		menu.addAction(d_invert_action);
	}

    menu.exec(menu_event->globalPos());
}

void TEditTable::Table_Selection_Changed(void)
{
	QList<QTableWidgetSelectionRange>	selection_range;
	QList<QTableWidgetSelectionRange>::const_iterator selection_iter;
	int									min_column(-1);
	int									max_column(-1);
	int									min_row(-1);
	int									max_row(-1);
	bool								init_loop(true);
	
	selection_range = this->selectedRanges();

	d_selection_rect.setRect(min_column,min_row,0,0);
	
	for(selection_iter = selection_range.begin();selection_iter != selection_range.end();selection_iter++)
	{
		if(init_loop)
		{
			min_column = (*selection_iter).leftColumn();
			min_row = (*selection_iter).topRow();
			max_column = (*selection_iter).rightColumn();
			max_row = (*selection_iter).bottomRow();
			
			init_loop = false;
		}
		else
		{
			if((*selection_iter).leftColumn() < min_column)
			{
				min_column = (*selection_iter).leftColumn();
			}
			
			if((*selection_iter).rightColumn() > max_column)
			{
				max_column = (*selection_iter).rightColumn();
			}
			
			if((*selection_iter).topRow() < min_row)
			{
				min_row = (*selection_iter).topRow();
			}
			
			if((*selection_iter).bottomRow() > max_row)
			{
				max_row = (*selection_iter).bottomRow();
			}
		}
	}
	
	d_selection_rect.setRect(min_column,
							 min_row,
							 max_column - min_column + 1,
							 max_row - min_row + 1);			// width and height minimum 1
	
	emit Selection_Changed();
}

void TEditTable::Activate_Cell(
	int									row,
	int									column)
{
	QRect								cell_rect;
	const int							frame_width(1);
	const int							frame_height(1);
	
	if(column < 0 || row < 0)
	{
		return;
	}
	
	cell_rect.setRect(this->columnViewportPosition(column) + this->verticalHeader()->width() + frame_width,
					  this->rowViewportPosition(row) + this->horizontalHeader()->height() + frame_height,
					  this->columnWidth(column),
					  this->rowHeight(row));
	
	d_table_editor->resize(cell_rect.size());
	d_table_editor->move(cell_rect.topLeft());
		
	if(d_table_editor->isHidden())
	{
		d_table_editor->show();
		d_table_editor->setFocus(Qt::OtherFocusReason);
		
	}
		
	d_table_editor_changed = false;

	if(d_selection_rect.top() != row)
	{
		d_selection_rect.setTop(row);
	}
	
	if(d_selection_rect.left() != column)
	{
		d_selection_rect.setLeft(column);
	}
}

void TEditTable::Cell_Editor_Changed(void)
{
	d_table_editor_changed = true;
}

void TEditTable::Cell_Editor_Closed(void)
{
	int									row;
	int									column;
	double								dval;
	QString								text;
	bool								state;
		
	this->setFocus();
	
	if(d_table_editor_changed)
	{
		row = d_selection_rect.top();
		column = d_selection_rect.left();
		
		if(!(row < 0 || column < 0))
		{
			text = d_table_editor->text();
			
			// clear text from text editor
			state = d_table_editor->blockSignals(true);
			d_table_editor->setText(QString());
			d_table_editor->blockSignals(state);
			
			dval = text.toDouble(&state);
			
			if(state)
			{
				if(d_edit_mode == TEditTable::EDIT_MODE_APPEND)
				{
					dval += d_cell_data[this->Row_Column_To_Offset(row,column)];
				}

				text = QString("%1").arg(dval,0,'f',d_display_precision);

				d_cell_data[this->Row_Column_To_Offset(row,column)] = dval;
				this->item(row,column)->setText(text);
				
				emit Cell_Changed(row,column,dval);
			}
		}
		
		d_table_editor_changed = false;
	}
}

void TEditTable::Clipboard_Data_Changed(void)
{
	d_paste_action->setEnabled(!QApplication::clipboard()->text(QClipboard::Clipboard).isEmpty());
}

void TEditTable::Edit_Cut(void)
{
	int									select_row_count;
	int									select_column_count;
	int									row_cntr;
	int									column_cntr;
	QString								text;
	
	if(d_selection_rect.left() == -1 || d_selection_rect.top() == -1 || d_selection_rect.right() == -1 || d_selection_rect.bottom() == -1)
	{
		return;
	}
	
	emit Start_Multiple_Cell_Change(QString("Cut values"));
	
	text = QString();

	select_row_count = d_selection_rect.height();
	select_column_count = d_selection_rect.width();
	
	for (row_cntr = 0; row_cntr < select_row_count; row_cntr++)
	{		
		for(column_cntr = 0; column_cntr < select_column_count; column_cntr++)
		{
			if (column_cntr > 0)
			{
				text += "\t";
			}
			
			text += QString("%1")
					.arg(this->Cell_Value(d_selection_rect.top() + row_cntr, d_selection_rect.left() + column_cntr),0,'f',CUT_COPY_PRECISON);
			
			this->Set_Cell_Value(d_selection_rect.top() + row_cntr, d_selection_rect.left() + column_cntr,0.0);
			
			emit Cell_Changed(d_selection_rect.top() + row_cntr, d_selection_rect.left() + column_cntr,0.0);
		}

		text += "\n";

	}
	emit End_Multiple_Cell_Change();
	
	QApplication::clipboard()->setText(text,QClipboard::Clipboard);	
}

void TEditTable::Edit_Copy(void)
{
	QString								text;
	int									select_row_count;
	int									select_column_count;
	int									row_cntr;
	int									column_cntr;
	
	if(d_selection_rect.left() == -1 || d_selection_rect.top() == -1 || d_selection_rect.right() == -1 || d_selection_rect.bottom() == -1)
	{
		return;
	}

	text = QString();
	
	select_row_count = d_selection_rect.height();
	select_column_count = d_selection_rect.width();
	
	for (row_cntr = 0; row_cntr < select_row_count; row_cntr++)
	{		
		for(column_cntr = 0; column_cntr < select_column_count; column_cntr++)
		{
			if (column_cntr > 0)
			{
				text += "\t";
			}
			
			text += QString("%1")
						.arg(this->Cell_Value(d_selection_rect.top() + row_cntr, d_selection_rect.left() + column_cntr),0,'f',CUT_COPY_PRECISON);
		}
		
		text += "\n";
	}
		
	QApplication::clipboard()->setText(text,QClipboard::Clipboard);
}

void TEditTable::Edit_Delete(void)
{
	int									select_row_count;
	int									select_column_count;
	int									row_cntr;
	int									column_cntr;
	
	if(d_selection_rect.left() == -1 || d_selection_rect.top() == -1 || d_selection_rect.right() == -1 || d_selection_rect.bottom() == -1)
	{
		return;
	}
	
	emit Start_Multiple_Cell_Change(QString("Delete values"));
	
	select_row_count = d_selection_rect.height();
	select_column_count = d_selection_rect.width();
	
	for (row_cntr = 0; row_cntr < select_row_count; row_cntr++)
	{		
		for(column_cntr = 0; column_cntr < select_column_count; column_cntr++)
		{			
			this->Set_Cell_Value(d_selection_rect.top() + row_cntr, d_selection_rect.left() + column_cntr,0.0);
			
			emit Cell_Changed(d_selection_rect.top() + row_cntr, d_selection_rect.left() + column_cntr,0.0);
		}
	}
	emit End_Multiple_Cell_Change();
}

void TEditTable::Edit_Invert(void)
{
	int									select_row_count;
	int									select_column_count;
	int									row_cntr;
	int									column_cntr;
	double								cell_value;
	
	if(d_selection_rect.left() == -1 || d_selection_rect.top() == -1 || d_selection_rect.right() == -1 || d_selection_rect.bottom() == -1)
	{
		return;
	}
	
	emit Start_Multiple_Cell_Change(QString("Invert Sign"));
	
	select_row_count = d_selection_rect.height();
	select_column_count = d_selection_rect.width();
	
	for (row_cntr = 0; row_cntr < select_row_count; row_cntr++)
	{		
		for(column_cntr = 0; column_cntr < select_column_count; column_cntr++)
		{			
			cell_value = -1 * this->Cell_Value(d_selection_rect.top() + row_cntr, d_selection_rect.left() + column_cntr);
			
			this->Set_Cell_Value(d_selection_rect.top() + row_cntr, d_selection_rect.left() + column_cntr,cell_value);
			
			emit Cell_Changed(d_selection_rect.top() + row_cntr, d_selection_rect.left() + column_cntr,cell_value);
		}
	}
	emit End_Multiple_Cell_Change();
}


void TEditTable::Edit_Paste(void)
{
	int									select_row_count;
	int									select_column_count;
	int									row_count;
	int									column_count;
	int									max_table_rows;
	int									max_table_columns;
	int									max_selection_rows;
	int									max_selection_columns;
	int									row_cntr;
	int									column_cntr;
	double								cell_value;
	double								new_cell_value;
	QString								text;
	QStringList							text_rows;
	
	if(d_selection_rect.left() == -1 || d_selection_rect.top() == -1 || d_selection_rect.right() == -1 || d_selection_rect.bottom() == -1)
	{
		return;
	}

	text = QApplication::clipboard()->text(QClipboard::Clipboard);
	
	// remove trailing line terminators, if any
	while(text.endsWith("\n") == true && text.length() > 0)
	{
		text.chop(1);
	}	
	
	if(!text.length())
	{
		return;
	}
		
	text_rows = text.split("\n",g_split_behavior_keep);
	
	select_row_count = d_selection_rect.height();
	select_column_count = d_selection_rect.width();
	
	row_count = text_rows.size();
	column_count = text_rows[0].count("\t") + 1;	// text_rows is never empty.  column count minimum is 1
	
	max_table_rows = d_row_count;
	max_table_columns = d_column_count;
		
	if(!(select_row_count == 1 && select_column_count == 1))
	{
		// do not paste outside of selected range
		if((select_row_count != row_count) || (select_column_count != column_count))
		{
			max_table_rows = d_selection_rect.top() + select_row_count;
			max_table_columns = d_selection_rect.left() + select_column_count;
		}
		
		max_selection_rows = select_row_count;
		max_selection_columns = select_column_count;
	}
	else
	{
		max_selection_rows = 1;
		max_selection_columns = 1;
	}
	
	emit Start_Multiple_Cell_Change(QString("Paste values"));
	
	for(row_cntr = 0;row_cntr < max_selection_rows;row_cntr += row_count)
	{
		for (int i = 0; i < row_count; ++i)
		{
			QStringList cols = text_rows[i].split("\t",g_split_behavior_keep);
			
			for(column_cntr = 0;column_cntr < max_selection_columns;column_cntr += column_count)
			{
				for (int j = 0;j < column_count; ++j)
				{
					int row = d_selection_rect.top() + i;
					int col = d_selection_rect.left() + j;
					
					if (row < d_row_count && col < d_column_count)
					{
						if(!cols[j].isEmpty())
						{
							text = cols[j];
							
							if((row + row_cntr) < max_table_rows && (col + column_cntr) < max_table_columns)
							{								
								cell_value = this->Cell_Value(row + row_cntr,col + column_cntr);
								
								new_cell_value = text.toDouble();

								if(d_edit_mode == TEditTable::EDIT_MODE_APPEND)
								{
									new_cell_value += cell_value;
								}
								
								this->Set_Cell_Value(row + row_cntr,col + column_cntr,new_cell_value);
								
								emit Cell_Changed(row + row_cntr,col + column_cntr,new_cell_value);
								
							}
						}
					}
				}
			}
		}
	}
	
	emit End_Multiple_Cell_Change();
}




