/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////



#include <QApplication>
#include <QComboBox>
#include <QGridLayout>
#include <QLabel>
#include <QSpacerItem>
#include <QStringList>
#include <QSizePolicy>

#include "iconwidget.h"
#include "titlewidget.h"
#include "controller.h"
#include "driver.h"
#include "logdialog.h"
#include "exercisedialog.h"
#include "temperaturemonitordialog.h"

#include "cmmwidget.h"

TCMMWidget::TCMMWidget(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QWidget(const_cast<QWidget*>(parent),flags)
{
	QGridLayout							*options_layout;
	QGridLayout							*widget_layout;
	QLabel								*controller_type_label;
	QLabel								*status_label;
	QSpacerItem							*control_hspacer;
	QSpacerItem							*options_vspacer;
	TTitleWidget						*title_widget;

	widget_layout = new QGridLayout(this);
	widget_layout->setContentsMargins(0,0,0,0);

	title_widget = new TTitleWidget(this);
	widget_layout->addWidget(title_widget,0,0,1,8);

	d_connect_option = new TIconWidget(this);
	d_connect_option->setSizePolicy(QSizePolicy(QSizePolicy::Maximum, QSizePolicy::Preferred));
	widget_layout->addWidget(d_connect_option,1,0,1,1);

	d_disconnect_option = new TIconWidget(this);
	d_disconnect_option->setSizePolicy(QSizePolicy(QSizePolicy::Maximum, QSizePolicy::Preferred));
	widget_layout->addWidget(d_disconnect_option,1,1,1,1);
	
	d_home_option = new TIconWidget(this);
	d_home_option->setSizePolicy(QSizePolicy(QSizePolicy::Maximum, QSizePolicy::Preferred));
	widget_layout->addWidget(d_home_option,1,2,1,1);
	
	d_exercise_option = new TIconWidget(this);
	d_exercise_option->setSizePolicy(QSizePolicy(QSizePolicy::Maximum, QSizePolicy::Preferred));
	widget_layout->addWidget(d_exercise_option,1,3,1,1);
	
	d_temperature_monitor_option = new TIconWidget(this);
	d_temperature_monitor_option->setSizePolicy(QSizePolicy(QSizePolicy::Maximum, QSizePolicy::Preferred));
	widget_layout->addWidget(d_temperature_monitor_option,1,4,1,1);

	d_log_option = new TIconWidget(this);
	d_temperature_monitor_option->setSizePolicy(QSizePolicy(QSizePolicy::Maximum, QSizePolicy::Preferred));
	widget_layout->addWidget(d_log_option,1,5,1,1);

	control_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	widget_layout->addItem(control_hspacer,1,6,1,1);

	options_layout = new QGridLayout();
	options_layout->setVerticalSpacing(2);

	controller_type_label = new QLabel(this);
	controller_type_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	options_layout->addWidget(controller_type_label,0,0,1,1);
	
	d_controller_type_combo = new QComboBox(this);
	options_layout->addWidget(d_controller_type_combo,0,1,1,1);

	status_label = new QLabel(this);
	status_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	options_layout->addWidget(status_label,1,0,1,1);
	
	d_status_label = new QLabel(this);
	options_layout->addWidget(d_status_label,1,1,1,1);
	
	options_vspacer = new QSpacerItem(0,0,QSizePolicy::Minimum,QSizePolicy::Expanding);
	options_layout->addItem(options_vspacer,2,0,1,2);

	widget_layout->addLayout(options_layout,1,7,2,1);
	
	d_log_dialog = new TLogDialog(this);
	
	d_connect_option->Set_Icon(QStringLiteral(":/icon64/connect_icon64.png"));
	d_connect_option->Set_Text(QStringLiteral("Connect"));
	
	d_disconnect_option->Set_Icon(QStringLiteral(":/icon64/disconnect_icon64.png"));
	d_disconnect_option->Set_Text(QStringLiteral("Disconnect"));
	
	d_home_option->Set_Icon(QStringLiteral(":/icon64/home_icon64.png"));
	d_home_option->Set_Text(QStringLiteral("Send\nHome"));
	
	d_exercise_option->Set_Icon(QStringLiteral(":/icon64/exercise_icon64.png"));
	d_exercise_option->Set_Text(QStringLiteral("Exercise\nMachine"));

	d_log_option->Set_Icon(QStringLiteral(":/icon64/record_icon64.png"));
	d_log_option->Set_Text(QStringLiteral("Log"));
	
	d_temperature_monitor_option->Set_Icon(QStringLiteral(":/icon64/temperature_icon64.png"));
	d_temperature_monitor_option->Set_Text(QStringLiteral("Temperature"));

	d_controller_type_combo->addItem(QStringLiteral("I++ Client"),TController::CONTROLLER_IPPCLIENT);
	d_controller_type_combo->addItem(QStringLiteral("Leitz"),TController::CONTROLLER_LEITZ);
	d_controller_type_combo->addItem(QStringLiteral("DC"),TController::CONTROLLER_DC);
	d_controller_type_combo->addItem(QStringLiteral("Virtual CMM"),TController::CONTROLLER_VIRTUAL);
	
	title_widget->Set_Text(QStringLiteral("CMM"));
	controller_type_label->setText(QStringLiteral("Driver Type:"));
	status_label->setText(QStringLiteral("Status:"));
	
	// default
	d_temperature_settings.upper_temperature_limit = 22.0;
	d_temperature_settings.lower_temperature_limit = 18.0;
	
	d_temperature_monitor_dialog = new TTemperatureMonitorDialog(this);
	
	d_temperature_monitor_dialog->Set_Lower_Temperature_Limit(d_temperature_settings.lower_temperature_limit);
	d_temperature_monitor_dialog->Set_Upper_Temperature_Limit(d_temperature_settings.upper_temperature_limit);

	this->Set_Driver_State(TDriver::DRIVER_STATE_NOT_CONNECTED);

	d_controller_type_combo->setCurrentIndex(1);	// I++
	d_log_dialog->setVisible(false);
	
	connect(d_connect_option,&TIconWidget::Clicked,this,&TCMMWidget::Connect);
	connect(d_disconnect_option,&TIconWidget::Clicked,this,&TCMMWidget::Disconnect);
	connect(d_home_option,&TIconWidget::Clicked,this,&TCMMWidget::Home);
	connect(d_exercise_option,&TIconWidget::Clicked,this,&TCMMWidget::Exercise_Machine);
	connect(d_log_option,&TIconWidget::Clicked,this,&TCMMWidget::Log_Clicked);
	connect(d_temperature_monitor_option,&TIconWidget::Clicked,this,&TCMMWidget::Toggle_Temperature_Monitor_Dialog);

	connect(d_temperature_monitor_dialog,&TTemperatureMonitorDialog::Request_Sensor_Value,this,&TCMMWidget::Request_Sensor_Value);
	connect(d_temperature_monitor_dialog,&TTemperatureMonitorDialog::Temperature_Limits_Changed,this,&TCMMWidget::Temperature_Limits_Changed);
}

TCMMWidget::~TCMMWidget(void)
{
}

TController::TControllerType TCMMWidget::Controller_Type(void) const
{
	return static_cast<TController::TControllerType>(d_controller_type_combo->currentData(Qt::UserRole).toInt());
}

TCMMWidget::TTemperatureSettings TCMMWidget::Temperature_Settings(void) const
{
	return d_temperature_settings;
}

void TCMMWidget::Set_Controller_Type(
	const TController::TControllerType	type)
{
	int									cntr;
	
	for(cntr = 0;cntr < d_controller_type_combo->count();++cntr)
	{
		if(static_cast<TController::TControllerType>(d_controller_type_combo->itemData(cntr).toInt()) == type)
		{
			d_controller_type_combo->setCurrentIndex(cntr);
			return;
		}
	}
}

void TCMMWidget::Set_Driver_State(
	const TDriver::TDriverState			state)
{
	d_driver_state = state;
	
	switch(d_driver_state)
	{
		case TDriver::DRIVER_STATE_NOT_CONNECTED:
			
			d_connect_option->setEnabled(true);
			d_disconnect_option->setEnabled(false);
			d_home_option->setEnabled(false);
			d_exercise_option->setEnabled(false);
			d_temperature_monitor_option->setEnabled(false);
			d_controller_type_combo->setEnabled(true);
			
			d_status_label->setText(QStringLiteral("Not Connected"));
			
			this->Add_Log_Text(QStringLiteral("Driver Not Connected"));
			
			if(d_temperature_monitor_dialog->isVisible())
			{
				d_temperature_monitor_dialog->hide();
				d_temperature_monitor_dialog->Disable_Log_Option();
			}
			break;
			
		case TDriver::DRIVER_STATE_CONNECTING:
			
			d_connect_option->setEnabled(false);
			d_disconnect_option->setEnabled(false);
			d_home_option->setEnabled(false);
			d_exercise_option->setEnabled(false);
			d_temperature_monitor_option->setEnabled(false);
			d_controller_type_combo->setEnabled(false);
	
			d_status_label->setText(QStringLiteral("Connecting..."));
			
			this->Clear_Log();
			this->Add_Log_Text(QStringLiteral("Driver Connecting..."));

			break;
			
		case TDriver::DRIVER_STATE_CONNECTED:
			
			d_connect_option->setEnabled(false);
			d_disconnect_option->setEnabled(true);
			d_home_option->setEnabled(true);
			d_exercise_option->setEnabled(true);
			d_temperature_monitor_option->setEnabled(true);
			d_controller_type_combo->setEnabled(false);
						
			d_status_label->setText(QStringLiteral("Connected"));
			
			this->Add_Log_Text(QStringLiteral("Driver Connected"));

			break;
	}
}

void TCMMWidget::Set_Temperature_Settings(
	const TCMMWidget::TTemperatureSettings &settings)
{
	d_temperature_settings = settings;
}

void TCMMWidget::Set_Sensor_Value(
	int 								sensor_id,
	const double 						&value,
	const QString						&log_path)
{
	d_temperature_monitor_dialog->Set_Sensor_Value(sensor_id,value,log_path);
}

void TCMMWidget::Set_Temperature_Sensors_X(
	const QString						&sensors)
{
	d_temperature_monitor_dialog->Set_Temperature_Sensors_X(sensors);
}

void TCMMWidget::Set_Temperature_Sensors_Y(
	const QString						&sensors)
{
	d_temperature_monitor_dialog->Set_Temperature_Sensors_Y(sensors);
}

void TCMMWidget::Set_Temperature_Sensors_Z(
	const QString						&sensors)
{
	d_temperature_monitor_dialog->Set_Temperature_Sensors_Z(sensors);
}

void TCMMWidget::Set_Temperature_Sensors_Part(
	const QString						&sensors)
{
	d_temperature_monitor_dialog->Set_Temperature_Sensors_Part(sensors);
}

void TCMMWidget::Enable_Disconnect_Option(
	const bool							state)
{
	d_disconnect_option->setEnabled(state);
	
	if(state && d_driver_state != TDriver::DRIVER_STATE_CONNECTED)
	{
		d_disconnect_option->setEnabled(false);
		d_exercise_option->setEnabled(false);
		d_temperature_monitor_option->setEnabled(false);
	}
}

void TCMMWidget::Enable_Options(
	const bool							home,
	const bool							exercise,
	const bool							temperature)
{
	if(d_driver_state != TDriver::DRIVER_STATE_CONNECTED)
	{
		d_disconnect_option->setEnabled(false);
		d_home_option->setEnabled(false);
		d_exercise_option->setEnabled(false);
		d_temperature_monitor_option->setEnabled(false);
	}
	else
	{
		d_home_option->setEnabled(home);
		d_exercise_option->setEnabled(exercise);
		d_temperature_monitor_option->setEnabled(temperature);
	}
}

void TCMMWidget::Clear_Log(void)
{
	d_log_dialog->Clear_Log();
}

void TCMMWidget::Add_Log_Text(
	const QString						&text)
{
	d_log_dialog->Add_Log_Text(text);
}

void TCMMWidget::Log_Clicked(void)
{
	d_log_dialog->setVisible(!d_log_dialog->isVisible());
}

void TCMMWidget::Toggle_Temperature_Monitor_Dialog(void)
{
	if(d_temperature_monitor_dialog->isVisible())
	{
		d_temperature_monitor_dialog->hide();
	}
	else
	{
		d_temperature_monitor_dialog->Enable_Log_Option();
		d_temperature_monitor_dialog->show();
	}
}

void TCMMWidget::Temperature_Limits_Changed(void)
{
	d_temperature_settings.lower_temperature_limit = d_temperature_monitor_dialog->Lower_Temperature_Limit();
	d_temperature_settings.upper_temperature_limit = d_temperature_monitor_dialog->Upper_Temperature_Limit();
}


