/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <assert.h>

#include "command_dimension.h"

TCommandDimension::TCommandDimension(void)
{
	d_command_type = TCommand::COMMAND_DIMENSION;
	d_memory_id_feature1 = -1;
	d_memory_id_feature2 = -1;
}

TCommandDimension::~TCommandDimension(void)
{
}

TCommandDimension::TDimensionItem TCommandDimension::Dimension_Item(
	const int							index)
{
	assert(static_cast<unsigned int>(index) < d_dimensions.size());
	
	return d_dimensions[index];
}

QString TCommandDimension::Dimension_Item_Text(
	const int							index)
{
	TDimensionItem						item;
	QString								line;
	double								deviation;
	double								ulim,llim;
	double								oot;
	
	item = this->Dimension_Item(index);
	
	//           1         2         3         4         5         6         7         8
	// 012345678901234567890123456789012345678901234567890123456789012345678901234567890
	// xxxxxx +99999.9999 +99999.9999 +99999.9999 +99999.9999 +99999.9999 +99999.9999
	//  axis       |   actual  |  nominal  |    dev    |    utol   |    ltol   |    oot
	
	
	line = QString("%1").arg(item.axis_name,-12,QChar(' '));
	line += QString("%1").arg(item.actual,12,'f',4);
	
	switch(item.dimension_format)
	{
		case TCommandDimension::FORMAT_MEAS:
			break;
			
		case TCommandDimension::FORMAT_ERR:
			
			deviation = item.actual - item.nominal;
			
			line += QString("%1").arg(item.nominal,12,'f',4);
			line += QString("%1").arg(deviation,12,'f',4);
			
			break;
			
		case TCommandDimension::FORMAT_COMPLETE:
			
			deviation = item.actual - item.nominal;
			
			line += QString("%1").arg(item.nominal,12,'f',4);
			line += QString("%1").arg(deviation,12,'f',4);
			
			line += QString("%1").arg(item.utol,12,'f',4);
			line += QString("%1").arg(item.ltol,12,'f',4);
			
			ulim = item.nominal + item.utol;
			llim = item.nominal - item.ltol;
			
			if(item.actual > ulim)
			{
				oot = item.actual - ulim;
				line += QString("%1").arg(oot,12,'f',4);
			}
			else if(item.actual < llim)
			{
				oot = item.actual - llim;
				line += QString("%1").arg(oot,12,'f',4);
			}

			break;
	}
	
	return line;
}

QString TCommandDimension::Dimension_Item_Text_Title_Meas(void)
{
	return QStringLiteral("Name           Actual\n-----------|------------");
}

QString TCommandDimension::Dimension_Item_Text_Title_Err(void)
{
	return QStringLiteral("Name            Actual      Nominal       Dev\n------------|-----------|-----------|------------");
}

QString TCommandDimension::Dimension_Item_Text_Title_Complete(void)
{
	return QStringLiteral("Name          Actual     Nominal       Dev         UTol        LTol        OOT\n-----------|-----------|-----------|-----------|-----------|-----------|-----------");
}

void TCommandDimension::Set_Actual(
	const TDimensionField				field,
	const double						&value)
{
	std::vector<TCommandDimension::TDimensionItem>::iterator iter;
	
	for(iter = d_dimensions.begin();iter != d_dimensions.end();++iter)
	{
		if((*iter).dimension_field == field)
		{
			(*iter).actual = value;
			return;
		}
	}
}

void TCommandDimension::Set_Nominal(
	const TDimensionField				field,
	const double						&value)
{
	std::vector<TCommandDimension::TDimensionItem>::iterator iter;
	
	for(iter = d_dimensions.begin();iter != d_dimensions.end();++iter)
	{
		if((*iter).dimension_field == field)
		{
			(*iter).nominal = value;
			return;
		}
	}
}

void TCommandDimension::Add_Dimension(
	const TCommandDimension::TDimensionItem	&item)
{
	d_dimensions.push_back(item);
}

void TCommandDimension::Add_Dimension(
	const TCommandDimension::TDimensionField field,
	const QString						&axis_name,
	const double						&actual)
{
	TCommandDimension::TDimensionItem	item;
	
	item.axis_name = axis_name;
	item.dimension_field = field;
	item.dimension_format = TCommandDimension::FORMAT_MEAS;
	item.actual = actual;
	item.nominal = 0.0;
	item.utol = 0.0;
	item.ltol = 0.0;
	
	d_dimensions.push_back(item);
}

void TCommandDimension::Add_Dimension(
	const TCommandDimension::TDimensionField field,
	const QString						&axis_name,
	const double						&actual,
	const double						&nominal)
{
	TCommandDimension::TDimensionItem	item;
	
	item.axis_name = axis_name;
	item.dimension_field = field;
	item.dimension_format = TCommandDimension::FORMAT_ERR;
	item.actual = actual;
	item.nominal = nominal;
	item.utol = 0.0;
	item.ltol = 0.0;
	
	d_dimensions.push_back(item);
}

void TCommandDimension::Add_Dimension(
	const TCommandDimension::TDimensionField field,
	const QString						&axis_name,
	const double						&actual,
	const double						&nominal,
	const double						&utol,
	const double						&ltol)
{
	TCommandDimension::TDimensionItem	item;
	
	item.axis_name = axis_name;
	item.dimension_field = field;
	item.dimension_format = TCommandDimension::FORMAT_COMPLETE;
	item.actual = actual;
	item.nominal = nominal;
	item.utol = utol;
	item.ltol = ltol;
	
	d_dimensions.push_back(item);
}

