/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QThread>
#include <assert.h>

#include "controller.h"

TController::TController(
	const TLinkProtocol::TProtocolType	protocol_type)
:d_link_protocol(protocol_type)
{
	d_mode = MODE_MANUAL;
	d_previous_mode = MODE_MANUAL;

	d_is_connected = false;
	d_is_homed = false;
	d_is_machine_estop = true;
	d_is_machine_error_state = true;
	d_position_updates_enabled = false;
	d_position_precision_updates_enabled = false;

	d_tip_radius = 2.0;
	d_tool_offset.Set(0,0,0);
	d_tool_vector.Set(0,0,-1);
	
	d_max_speed = 500.0;
	d_max_acceleration = 1000.0;
	
	d_abort_command = 0;
}

TController::~TController(void)
{

}

QString TController::Get_Next_Event(
	bool								* const valid)
{
	QString								text;
	
	assert(valid);
	
	*valid = false;
	
	if(d_event_text.size())
	{
		*valid = true;
		text = d_event_text[0];
		
		d_event_text.erase(d_event_text.begin());
	}
	
	return text;
}

TController::TErrorData TController::Get_Next_Error(
	bool								* const valid)
{
	TController::TErrorData				error_data;
	
	assert(valid);
	
	*valid = false;
	
	if(d_error_data.size())
	{
		*valid = true;
		error_data = d_error_data[0];
		
		d_error_data.erase(d_error_data.begin());
	}
	
	return error_data;
}

TController::TKeyType TController::Get_Next_Keypress(
	bool								* const valid)
{
	TController::TKeyType				key(TController::KEY_DONE);
	
	assert(valid);
	
	*valid = false;
	
	if(d_key_press.size())
	{
		*valid = true;
		key = d_key_press[0];
		
		d_key_press.erase(d_key_press.begin());
	}
	
	return key;
}

TController::TTouchPoint TController::Get_Next_TouchPoint(
	bool								* const valid)
{
	TController::TTouchPoint			touch_point;
	
	assert(valid);
	
	*valid = false;
	
	if(d_touch_points.size())
	{
		touch_point = d_touch_points[0];
		*valid = true;
		
		d_touch_points.erase(d_touch_points.begin());
	}
	
	return touch_point;
}

TController::TSensorData TController::Get_Next_Sensor(
	bool								* const valid)
{
	TController::TSensorData			sensor_data;
	
	assert(valid);
	
	*valid = false;
	
	if(d_sensor_data.size())
	{
		sensor_data = d_sensor_data[0];
		*valid = true;
		
		d_sensor_data.erase(d_sensor_data.begin());
	}
	
	return sensor_data;
}

std::vector<QString> TController::Command_Response_Queue_Entry(
	const int 							index)
{
	std::vector<QString>::const_iterator iter;
	
	assert(static_cast<unsigned int>(index) < d_command_response_queue.size());
		
	return d_command_response_queue[index];
}

bool TController::Initialize(void)
{
	return true;
}

bool TController::Refresh_Tools(void)
{
	return true;
}

void TController::Add_Error(
	const QString						&error_text,
	const int							severity)
{
	TController::TErrorData				error_data;
		
	error_data.text = error_text;
	error_data.severity = severity;
	
	d_error_data.push_back(error_data);
}

bool TController::Query_Tool(
	const QString						&tool_name)
{
	Q_UNUSED(tool_name);
	
	return true;
}

bool TController::Set_Tool_Name(
	const QString						&tool_name)
{
	Q_UNUSED(tool_name);
	
	return true;
}

bool TController::Set_Tool_Type(
	const QString						&tool_type)
{
	Q_UNUSED(tool_type);
	
	return false;
}

bool TController::Set_Tool_Data(
	const TVector3						&xyz,
	const double						&tip_diameter)
{	
	d_tool_offset = xyz;
	d_tip_radius = tip_diameter / 2.0;
	
	d_tool_vector = d_tool_offset;

	if(d_tool_vector.Length() > 0.000001)
	{
		d_tool_vector.Normal();
	}
	else
	{
		d_tool_vector.Set(0,0,-1);
	}
	
	return false;
}

bool TController::Set_Tool_Angles(
	const double						&angle_a,
	const double						&angle_b)
{
	Q_UNUSED(angle_a);
	Q_UNUSED(angle_b);
	
	return false;
}

bool TController::Enable_Position_Updates(
	const bool 							state,
	const bool 							precision_update)
{	
	d_position_updates_enabled = state;
	d_position_precision_updates_enabled = precision_update;
		
	return true;
}

void TController::Set_Serial_DeviceName(
	const QString						&device_name)
{
	d_link_protocol.Set_Serial_DeviceName(device_name);
}

void TController::Set_Serial_Baud(
	const TLibSerialDeviceEnum::TBaudRate baud)
{
	d_link_protocol.Set_Serial_Baud(baud);
}

void TController::Set_Serial_Data(
	const TLibSerialDeviceEnum::TDataBits data)
{
	d_link_protocol.Set_Serial_Data(data);
}

void TController::Set_Serial_Parity(
	const TLibSerialDeviceEnum::TParity parity)
{
	d_link_protocol.Set_Serial_Parity(parity);
}

void TController::Set_Serial_StopBits(
	const TLibSerialDeviceEnum::TStopBits stop)
{
	d_link_protocol.Set_Serial_StopBits(stop);
}

void TController::Set_Serial_Flow(
	const TLibSerialDeviceEnum::TFlowControl flow)
{
	d_link_protocol.Set_Serial_Flow(flow);
}

void TController::Set_Serial_ReadTimeout(
	const int							timeout)
{
	d_link_protocol.Set_Serial_ReadTimeout(timeout);
}

void TController::Set_Socket_Hostname(
	const QString						&name)
{
	d_link_protocol.Set_Socket_Hostname(name);
}

void TController::Set_Socket_Port(
	const int							port)
{
	d_link_protocol.Set_Socket_Port(port);
}

TController::TSendResult TController::Send_Command(
	const QByteArray					&command,
	const TController::TSendMode		send_mode,
	const std::vector<QString> 			&expected_responses,
	std::vector<QString>				* const actual_responses,
	const int							timeout)
{
	static const int					RX_BUFFER_SIZE(256);
	static const int					WAIT_READ_IDLE_TIME(10);	// time between polling for new data to arrive
	char								line_terminators[] = "\r\n";
	char								rx_buffer[RX_BUFFER_SIZE];
	QString								response;
	QString								text;
	int									cntr;
	int									rx_processed_position;
	int									rx_used_buffer_size;
	int									rx_buffer_position;
	int									expected_line_count;
	int									line_count;
	int									timeout_counter;
	int									timeout_limit;
	int									rx_read_size;
	int									rx_max_read_size;
	int									rx_actual_read;
	bool								check_initial;
	bool								check_final;

	
	assert(actual_responses);
	
	if(timeout < 1)
	{
		timeout_limit = 1000;
	}
	else if(timeout > 120)
	{
		timeout_limit = 120000;
	}
	else
	{
		timeout_limit = 1000 * timeout;
	}
	
	// Clear any data from RX buffer that should not be there
	do
	{
		QCoreApplication::processEvents();

		cntr = d_link_protocol.bytesAvailable();
		
		if(cntr)
		{
			d_rx_data += d_link_protocol.read(cntr);
			QThread::msleep(1);
		}
		
	} while(cntr);
	
	actual_responses->clear();
		
	if(d_link_protocol.write(command) < 0)
	{
		d_last_error = QString("IO error on write.  Error code: %1\r\n").arg(d_link_protocol.Get_Last_Error());
		return TController::SEND_ERROR;
	}
	
	if(send_mode == TController::MODE_QUEUE_RESPONSE)
	{
		d_command_response_queue.push_back(expected_responses);
		return TController::SEND_SUCCESS;
	}

	expected_line_count = static_cast<int>(expected_responses.size());
	
	if(expected_line_count == 0)
	{
		return TController::SEND_SUCCESS;
	}
	
	line_count = 0;
	rx_buffer_position = 0;
	rx_processed_position = 0;
	rx_used_buffer_size = 0;
	timeout_counter = 0;
	
	if(send_mode == TController::MODE_WAIT_RESPONSE)
	{
		while(line_count < expected_line_count)
		{
			QCoreApplication::processEvents();
			
			if(d_link_protocol.bytesAvailable() != 0 &&
			   d_link_protocol.getChar(&rx_buffer[rx_buffer_position]))
			{
				// always mirror rx data for processing.
				d_rx_data.append(rx_buffer[rx_buffer_position]);
				
				rx_buffer_position++;
				timeout_counter = 0;
			}
			else
			{
				QThread::msleep(WAIT_READ_IDLE_TIME);
				timeout_counter += WAIT_READ_IDLE_TIME;
				
				if(timeout_counter > timeout_limit)
				{
					d_last_error = QStringLiteral("Insufficient data received within timeout period.\r\n");
					return TController::SEND_RESPONSE_INVALID;
				}
				
				if(d_abort_command)
				{
					if(*d_abort_command)
					{
						d_last_error = QStringLiteral("Command aborted.\r\n");
						return TController::SEND_RESPONSE_ABORT;
					}
				}
			}
			
			if(!(rx_buffer_position < RX_BUFFER_SIZE))
			{
				d_last_error = QStringLiteral("Read buffer overflow.\r\n");
				return TController::SEND_RESPONSE_INVALID;
			}
			
			while(rx_processed_position < rx_buffer_position)
			{
				if(rx_buffer[rx_processed_position] == line_terminators[0] || rx_buffer[rx_processed_position] == line_terminators[1])
				{
					if(rx_processed_position)	// make sure it isn't an empty string.  <\r><\n> sequence for example
					{
						text = QString::fromLatin1(rx_buffer,rx_processed_position);

						rx_buffer_position -= (rx_processed_position + 1);	// Don't forget the terminator!
						rx_processed_position = 0;
												
						if(expected_responses[line_count].length() == 0)	// accept anything
						{
							actual_responses->push_back(text);
							++line_count;
						}
						else if(text.trimmed().startsWith(expected_responses[line_count],Qt::CaseInsensitive))
						{
							actual_responses->push_back(text);
							++line_count;
						}
					}
					else
					{
						// Shift memory down
						memmove(rx_buffer,rx_buffer + rx_buffer_position,RX_BUFFER_SIZE - rx_buffer_position);
						memset(rx_buffer + RX_BUFFER_SIZE - rx_buffer_position,0,rx_buffer_position);
						--rx_buffer_position;
					}
				}
				else if(rx_buffer[rx_processed_position] < 0x20)	// ignore the control char's
				{
					--rx_processed_position;
					--rx_buffer_position;
				}
				else
				{
					++rx_processed_position;
				}
			}
			
			if(d_abort_command)
			{
				if(*d_abort_command)
				{
					d_last_error = QStringLiteral("Command aborted.\r\n");
					return TController::SEND_RESPONSE_ABORT;
				}
			}
		}
	}
	else if(send_mode == TController::MODE_WAIT_RESPONSE_STREAM)
	{
		check_initial = true;
		check_final = false;
		
		if(expected_line_count != 2)
		{
			d_last_error = QStringLiteral("Expected Response must contain two entries.");
			return TController::SEND_RESPONSE_INVALID;
		}

		do
		{
			QCoreApplication::processEvents();
			
			rx_read_size = d_link_protocol.bytesAvailable();
			
			if(rx_read_size)
			{
				rx_max_read_size = RX_BUFFER_SIZE - rx_buffer_position;
				
				if(rx_max_read_size < 1)
				{
					d_last_error = QStringLiteral("Read buffer overflow.");
					return TController::SEND_RESPONSE_INVALID;
				}
				
				if(rx_read_size > rx_max_read_size)
				{
					rx_read_size = rx_max_read_size;
				}
				
				rx_actual_read = d_link_protocol.read(&rx_buffer[rx_buffer_position],rx_read_size);
				
				// always mirror rx data for processing.  May contain unsolicited hits.
				d_rx_data.append(&rx_buffer[rx_buffer_position],rx_actual_read);
				
				rx_used_buffer_size = rx_actual_read + rx_buffer_position;
				timeout_counter = 0;
			}
			else
			{
				QThread::msleep(WAIT_READ_IDLE_TIME);
				timeout_counter += WAIT_READ_IDLE_TIME;
				
				if(timeout_counter > timeout_limit)
				{
					d_last_error = QStringLiteral("Insufficient data received within timeout period.");
					return TController::SEND_RESPONSE_INVALID;
				}
				
				if(d_abort_command)
				{
					if(*d_abort_command)
					{
						d_last_error = QStringLiteral("Command aborted.\r\n");
						return TController::SEND_RESPONSE_ABORT;
					}
				}
			}

			while(rx_buffer_position < rx_used_buffer_size)
			{
				if(rx_buffer[rx_buffer_position] == line_terminators[0] || rx_buffer[rx_buffer_position] == line_terminators[1])
				{
					if(rx_buffer_position)	// make sure it isn't an empty string.  <\r><\n> sequence for example
					{
						text = QString::fromLatin1(rx_buffer,rx_buffer_position);
												
						if(check_initial == true)
						{
							check_initial = false;
														
							if(0 != QString::compare(text,expected_responses[0],Qt::CaseInsensitive))
							{								
								d_last_error = QString("First line of input stream does not match expected data:\n");
								d_last_error += text + "\n";
								d_last_error += expected_responses[0];
								return TController::SEND_RESPONSE_INVALID;
							}
						}
						else
						{
							if(0 == QString::compare(text,expected_responses[1],Qt::CaseInsensitive))
							{
								check_final = true;
							}
						}
						
						actual_responses->push_back(text);
					}
					
					memmove(rx_buffer,rx_buffer + rx_buffer_position + 1,RX_BUFFER_SIZE - (rx_buffer_position + 1));
					memset(rx_buffer + RX_BUFFER_SIZE - (rx_buffer_position + 1),0,(rx_buffer_position + 1));
					
					rx_used_buffer_size -= (rx_buffer_position + 1);
					rx_buffer_position = 0;
				}
				else
				{
					++rx_buffer_position;
				}
			}

		} while(!check_final);
	}

	return TController::SEND_SUCCESS;
}

QByteArray TController::ReadPort(void)
{
	int									size;
	QByteArray							data;
	QByteArray							processed_data;
	int									index;
	QString								text;
	
	do
	{
		size = d_link_protocol.bytesAvailable();
		
		if(size > 0)
		{
			d_rx_data.append(d_link_protocol.read(size));
		}
		
		QThread::msleep(1);
		
	} while(size);

	data = d_rx_data;
	d_rx_data.clear();
	
	if(d_command_response_queue.size())
	{
		do
		{
			index = data.indexOf('\n');
			
			if(!(index < 0))
			{
				text = QString::fromLatin1(data.mid(0,index));
				
				this->Check_Queued_Responses(text);
								
				processed_data.append(data.mid(0,index+1));
				
				data.remove(0,index + 1);
			}
			else
			{
				if(data.size())
				{
					d_rx_data.append(data);	// not a valid line yet.  put it back
					
					data.clear();
				}
			}
			
		}while(!(index < 0));
		
	}
	else
	{
		processed_data = data;
	}
	
	return processed_data;
}

bool TController::Check_Queued_Responses(
	const QString						&text)
{
	std::vector<std::vector<QString> >::iterator queue_iter;
	std::vector<QString>::iterator		iter;
	
	for(queue_iter = d_command_response_queue.begin();queue_iter != d_command_response_queue.end();++queue_iter)
	{
		for(iter = (*queue_iter).begin();iter != (*queue_iter).end();++iter)
		{
			if(text.startsWith(*iter))
			{
				(*queue_iter).erase(iter);
				
				if((*queue_iter).size() == 0)
				{
					d_command_response_queue.erase(queue_iter);
				}
				
				return true;
			}
		}
	}
	
	return false;
}


