/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////



#ifndef CONTROLLERHEADERFILE
#define CONTROLLERHEADERFILE

#include <QString>
#include <QStringList>
#include <vector>

#include "../../../libserialdevice/src/libserialdevice.h"
#include "../../../core/vector3.h"

#include "linkprotocol.h"

class TController
{
public:
	
enum TMode
{
	MODE_MANUAL = 0,
	MODE_DCC
};
	
enum TControllerType
{
	CONTROLLER_IPPCLIENT=0,
	CONTROLLER_LEITZ,
	CONTROLLER_DC,
	CONTROLLER_VIRTUAL
};
	
enum TKeyType
{
	KEY_DONE=0,
	KEY_ERASE_HIT
};
	
struct TErrorData
{
	QString								text;
	int									severity;	// 0 == information, >0 reset required
};
	
struct TTouchPoint
{
	TVector3							xyz;
	TVector3							ijk;
};
	
struct TSensorData
{
	int									sensor_id;
	double								value;
};

// CREATORS
	TController(const TLinkProtocol::TProtocolType protocol_type);
	virtual ~TController(void) = 0;
	
// ACCESSORS
	QString Last_Error(void) const {return d_last_error;}
	
	QString Get_Next_Event(bool * const valid);
	TErrorData Get_Next_Error(bool * const valid);
	TKeyType Get_Next_Keypress(bool * const valid);
	TTouchPoint Get_Next_TouchPoint(bool * const valid);
	TSensorData Get_Next_Sensor(bool * const valid);
	
	int Command_Response_Queue_Size(void) const {return static_cast<int>(d_command_response_queue.size());}
	std::vector<QString> Command_Response_Queue_Entry(const int index);
	TMode Mode(void) const {return d_mode;}
	TControllerType Controller_Type(void) const {return d_controller_type;}
	
	bool Is_Homed(void) const {return d_is_homed;}
	bool Is_EStop(void) const {return d_is_machine_estop;}
	bool Is_Error(void) const {return d_is_machine_error_state;}
	
	void Set_Homed_State(const bool state) {d_is_homed = state;}
	
	double Tip_Radius(void) const {return d_tip_radius;}
	double Tip_Diameter(void) const {return d_tip_radius * 2.0;}
	TVector3 Tool_Offset(void) const {return d_tool_offset;}
	TVector3 Tool_Vector(void) const {return d_tool_vector;}
	
	double Query_Tip_Radius(void) const {return d_query_tip_radius;}
	double Query_Tip_Diameter(void) const {return d_query_tip_radius * 2.0;}
	TVector3 Query_Tool_Offset(void) const {return d_query_tool_offset;}
	TVector3 Query_Tool_Vector(void) const {return d_query_tool_vector;}

	double Approach_Distance(void) const {return d_current_approach_distance;}
	bool Updates_Enabled(void) const {return d_position_updates_enabled;}
	bool Precision_Updates_Enabled(void) const {return d_position_precision_updates_enabled;}
	
	virtual QStringList Get_Ipp_Tool_List(void) {return QStringList();}
	
// MANIPULATORS
	void Set_Abort_Command(bool * abort_command) {d_abort_command = abort_command;}	// set boolean pointer to true to cancel active command
	
	virtual bool Connect(void) = 0;
	virtual bool Initialize(void) = 0;
	virtual bool Refresh_Tools(void) = 0;
	virtual bool Send_Home(void) = 0;
	virtual void Disconnect(void) = 0;
	virtual void Close(void) = 0;
	
	virtual bool Clear_Error(void) {return true;}
	virtual void Check_Machine_Status(void) = 0;
	virtual void Abort(void) {;}
	
	void Purge_Errors(void) {d_error_data.clear();}
	void Add_Error(const QString &error_text,const int severity);
	
	virtual bool Query_Tool(const QString &tool_name);
	virtual bool Set_Tool_Name(const QString &tool_name);
	virtual bool Set_Tool_Type(const QString &tool_type);							// tool_type must be 'TTP'
	virtual bool Set_Tool_Data(const TVector3 &xyz,const double &tip_diameter);
	virtual bool Set_Tool_Angles(const double &angle_a,const double &angle_b);
		
	virtual void Set_Mode(const TMode mode) {d_mode = mode;}
	void Store_Current_Mode(void) {d_previous_mode = d_mode;}
	void Restore_Previous_Mode(void) {d_mode = d_previous_mode;}
	void Clear_Previous_Mode(void) {d_previous_mode = MODE_MANUAL;}
	
	virtual bool Set_Move_Speed(const double&) {return true;}
	virtual bool Set_Touch_Speed(const double&) {return true;}
	virtual bool Set_Acceleration(const double&) {return true;}
	virtual bool Set_Approach_Distance(const double&) {return true;}
	virtual bool Enable_Blended_Moves(void) {return true;}
	virtual bool Disable_Blended_Moves(void) {return true;}
	virtual bool Enable_Position_Updates(const bool state,const bool precision_update);

	virtual bool Get_Position(TVector3 * const pnt) = 0;
	virtual bool Get_Position_DRO(TVector3 * const pnt) = 0;
	virtual bool Move_To(const TVector3 &pnt) = 0;
	virtual bool Touch(const TVector3 &target, const TVector3 &approach) = 0;
	
	virtual bool Get_X_Temperature(const QString &sensor_list,double * const value,bool * const valid) = 0;
	virtual bool Get_Y_Temperature(const QString &sensor_list,double * const value,bool * const valid) = 0;
	virtual bool Get_Z_Temperature(const QString &sensor_list,double * const value,bool * const valid) = 0;
	virtual bool Get_Part_Temperature(const QString &sensor_list,double * const value,bool * const valid) = 0;
	virtual bool Get_Sensor_Value(int sensor_id) {Q_UNUSED(sensor_id);return false;}
	
	virtual bool Set_Part_Expansion_Coefficient(const double  &expansion_coefficient) = 0;	// expansion in um/m/˚C
	
	virtual QString Get_X_Sensors(void) {return QStringLiteral("-1");}
	virtual QString Get_Y_Sensors(void) {return QStringLiteral("-1");}
	virtual QString Get_Z_Sensors(void) {return QStringLiteral("-1");}
	virtual QString Get_Part_Sensors(void) {return QStringLiteral("-1");}
	
	virtual void Process_Rx_Data(void) = 0;
	
	void Set_Serial_DeviceName(const QString &device_name);
	void Set_Serial_Baud(const TLibSerialDeviceEnum::TBaudRate baud);
	void Set_Serial_Data(const TLibSerialDeviceEnum::TDataBits data);
	void Set_Serial_Parity(const TLibSerialDeviceEnum::TParity parity);
	void Set_Serial_StopBits(const TLibSerialDeviceEnum::TStopBits stop);
	void Set_Serial_Flow(const TLibSerialDeviceEnum::TFlowControl flow);
	void Set_Serial_ReadTimeout(const int timeout);
	
	void Set_Socket_Hostname(const QString &name);
	void Set_Socket_Port(const int port);

	
protected:
	
enum TSendMode
{
	MODE_WAIT_RESPONSE = 0,				// Waits for specified response(s).  Empty response entry will accept anything.  Empty response list will not wait for a response
	MODE_WAIT_RESPONSE_STREAM,			// Response is an unknown length with only the first and last responses required to match
	MODE_QUEUE_RESPONSE					// Command is sent without waiting for a response.  Response is checked eventually.
};
	
enum TSendResult
{
	SEND_SUCCESS = 0,
	SEND_ERROR,
	SEND_RESPONSE_ABORT,				// result when command is aborted while waiting for response
	SEND_RESPONSE_INVALID
};

	TLinkProtocol						d_link_protocol;
	
	TMode								d_mode;
	TMode								d_previous_mode;
	TControllerType						d_controller_type;
	
	bool								d_is_connected;
	bool								d_is_homed;
	bool								d_is_machine_estop;
	bool								d_is_machine_error_state;
	bool								d_position_updates_enabled;
	bool								d_position_precision_updates_enabled;

	TVector3							d_tool_offset;
	TVector3							d_tool_vector;
	TVector3							d_query_tool_offset;
	TVector3							d_query_tool_vector;
	double								d_tip_radius;
	double								d_query_tip_radius;
	double								d_current_approach_distance;
	double								d_max_speed;
	double								d_max_acceleration;

	QByteArray							d_rx_data;
	bool								*d_abort_command;

	std::vector<std::vector<QString> >	d_command_response_queue;
	
	std::vector<QString>				d_event_text;
	std::vector<TErrorData>				d_error_data;
	std::vector<TKeyType>				d_key_press;
	std::vector<TTouchPoint>			d_touch_points;
	std::vector<TSensorData>			d_sensor_data;

	mutable QString						d_last_error;
	
	TSendResult Send_Command(const QByteArray &command,
										const TController::TSendMode send_mode,
										const std::vector<QString> &expected_responses,
										std::vector<QString> * const actual_responses,
										const int timeout = 2);			// timeout in seconds
	
	QByteArray ReadPort(void);

	void Add_Event_Text(const QString &text) {d_event_text.push_back(text);}
	
private:
	
	bool Check_Queued_Responses(const QString &text);		// removes response if found

	
// NOT IMPLEMENTED
	TController(const TController&);
	TController& operator=(const TController&);
};

#endif
