/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QRegularExpression>
#include <QRegularExpressionMatch>
#include <assert.h>

#include "controller_ippclient.h"

static const int						X_TEMPERATURE_SENSOR_ID(1);
static const int						Y_TEMPERATURE_SENSOR_ID(2);
static const int						Z_TEMPERATURE_SENSOR_ID(3);
static const int						PART_TEMPERATURE_SENSOR_ID(4);

TControllerIppClient::TControllerIppClient(
	const TLinkProtocol::TProtocolType	protocol_type)
:TController(protocol_type)
{
	d_controller_type = TController::CONTROLLER_IPPCLIENT;
}

TControllerIppClient::~TControllerIppClient(void)
{
}

QStringList TControllerIppClient::Get_Ipp_Tool_List(void)
{
	if(!this->Refresh_Tools())
	{
		this->Add_Event_Text(QStringLiteral("ERR:  Error getting I++ tool list."));
	}
	
	return d_ipp_tool_list;
}

bool TControllerIppClient::Connect(void)
{
	QString								write_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	TController::TSendResult			write_status;

	if(!d_link_protocol.Connect())
	{
		this->Add_Event_Text(QString("ERR: %1").arg(d_link_protocol.Get_Last_Error()));
		return false;
	}
		
	this->Add_Event_Text(QStringLiteral("Connection opened."));
	this->Add_Event_Text(QStringLiteral("Initializing controller."));
	
	d_command_id = 0;
	d_event_command_id = 0;
	
	// StartSession()
	d_command_id++;
	write_string = QString("%1 StartSession()").arg(d_command_id,5,10,QChar('0'));
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
										MODE_WAIT_RESPONSE,
										expected_responses,
										&actual_responses,
										60);	// Wait up to one minute for connection to complete
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  StartSession() did not return expected response."));
		return false;
	}

	return true;
}

bool TControllerIppClient::Initialize(void)
{
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	QRegularExpression					regular_expression;
	QRegularExpressionMatch				expression_match;
	QStringList							expression_text_list;
	QString								write_string;
	QString								source_string;
	TController::TSendResult			write_status;
	bool								valid;

	d_position_updates_enabled = false;
	d_position_precision_updates_enabled = false;
	d_coordinate_id = QString();
	
	// Is Homed
	d_command_id++;
	write_string = QString("%1 IsHomed()").arg(d_command_id,5,10,QChar('0'));
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 # IsHomed").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_WAIT_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
		
	d_is_homed = true;
		
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  IsHomed() did not return expected response."));
		return false;
	}

	if(actual_responses.size() != 3)
	{
		this->Add_Event_Text(QStringLiteral("ERR:  IsHomed() did not return expected response."));
		return false;
	}
	
	source_string = actual_responses[1];
	
	regular_expression.setPattern(QStringLiteral("[IsHomed(]{8}([{0,1}[0-9.]{1})[)]{1}"));
	expression_match = regular_expression.match(source_string);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() > 0)
		{
			d_is_homed = (0 != expression_text_list[1].toInt(&valid));
			
			if(!valid)
			{
				this->Add_Event_Text(QStringLiteral("ERR:  Invalid data returned from IsHomed() command."));
				return false;
			}
		}
	}
	else
	{
		this->Add_Event_Text(QStringLiteral("ERR:  Invalid data returned from IsHomed() command."));
		return false;
	}
	
//	if(!this->Refresh_Tools())
//	{
//		return false;
//	}

	// SetMachineCSY()
	d_command_id++;
	write_string = QString("%1 SetCoordSystem(MachineCsy)").arg(d_command_id,5,10,QChar('0'));
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
										MODE_WAIT_RESPONSE,
										expected_responses,
										&actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  SetCoordSystem() did not return expected response."));
		return false;
	}

	// OnPtMeasReport()
	d_command_id++;
	write_string = QString("%1 OnPtMeasReport(X(),Y(),Z(),IJK())").arg(d_command_id,5,10,QChar('0'));
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
										MODE_WAIT_RESPONSE,
										expected_responses,
										&actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  OnPtMeasReport() did not return expected response."));
		return false;
	}

	
	// EnableUser()
	d_command_id++;
	write_string = QString("%1 EnableUser()").arg(d_command_id,5,10,QChar('0'));
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
										MODE_WAIT_RESPONSE,
										expected_responses,
										&actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  EnableUser() did not return expected response."));
		return false;
	}

	return true;
}

bool TControllerIppClient::Refresh_Tools(void)
{
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	QStringList							expression_text_list;
	QString								write_string;
	QString								source_string;
	TController::TSendResult			write_status;
	int									cntr;
	
	// EnumTools()
	d_command_id++;
	write_string = QString("%1 EnumTools()").arg(d_command_id,5,10,QChar('0'));
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	d_ipp_tool_list.clear();
	
	write_status = this->Send_Command(write_string.toLatin1(),
										MODE_WAIT_RESPONSE_STREAM,
										expected_responses,
										&actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
		
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  EnumTools() did not return expected response."));
		return false;
	}
	
	if(actual_responses.size() < 2)
	{
		this->Add_Event_Text(QStringLiteral("ERR:  EnumTools() did not return expected response."));
		return false;
	}

	// remove first and last
	actual_responses.erase(actual_responses.begin());
	actual_responses.pop_back();
	
	for(cntr = 0;cntr < static_cast<int>(actual_responses.size());++cntr)
	{
		source_string = actual_responses[cntr];
		
		source_string.remove(0,7);			// command id and # hash
		source_string.remove('\"');
		
		d_ipp_tool_list.push_back(source_string.trimmed());
	}
	
	return true;
}

bool TControllerIppClient::Send_Home(void)
{
	QString								write_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	TController::TSendResult			write_status;

	// Home()
	d_command_id++;
	write_string = QString("%1 Home()").arg(d_command_id,5,10,QChar('0'));
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));	// require command confirmation only
	
		
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
										MODE_WAIT_RESPONSE,
										expected_responses,
										&actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}

	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text("ERR:  Home() did not return expected response.");
		return false;
	}
		
	expected_responses.clear();
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));

	d_command_response_queue.push_back(expected_responses);

	return true;
}

void TControllerIppClient::Disconnect(void)
{
	QString								write_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	TController::TSendResult			write_status;
	
	if(d_coordinate_id.length())	// only stop if acutally started
	{
		// StopDeamon()
		d_command_id++;
		write_string = QString("%1 StopDaemon(%2)").arg(d_command_id,5,10,QChar('0')).arg(d_coordinate_id);
		
		expected_responses.clear();
		expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
		expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
		
		this->Add_Event_Text(write_string);
		write_string += QStringLiteral("\r\n");
		
		write_status = this->Send_Command(write_string.toLatin1(),
										  MODE_WAIT_RESPONSE,
										  expected_responses,
										  &actual_responses);
		
		for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
		{
			this->Add_Event_Text(*str_iter);
		}
		
		if(write_status != TController::SEND_SUCCESS)
		{
			this->Add_Event_Text(d_last_error);
			this->Add_Event_Text(QStringLiteral("ERR:  StopDaemon() did not return expected response."));
		}		
	}
}

void TControllerIppClient::Close(void)
{
	QString								write_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	TController::TSendResult			write_status;

	// EndSession()
	d_command_id++;
	write_string = QString("%1 EndSession()").arg(d_command_id,5,10,QChar('0'));
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_WAIT_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  EndSession() did not return expected response."));
	}
	
	d_link_protocol.Close();
}

bool TControllerIppClient::Clear_Error(void)
{
	QString								write_string;
	QString								source_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	TController::TSendResult			write_status;

	
	// ClearAllErrors()
	d_command_id++;
	write_string = QString("%1 ClearAllErrors()").arg(d_command_id,5,10,QChar('0'));
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									 MODE_WAIT_RESPONSE,
									 expected_responses,
									 &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  ClearAllErrors() did not return expected response."));
		
		return false;
	}
	
	return true;
}

void TControllerIppClient::Check_Machine_Status(void)
{
	QString								write_string;
	QString								source_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	TController::TErrorData				error_data;
	QRegularExpression					regular_expression;
	QRegularExpressionMatch				expression_match;
	QStringList							expression_text_list;
	TController::TSendResult			write_status;
	bool								valid;
	
	d_is_machine_error_state = false;
	d_is_machine_estop = false;

	// GetErrStatusE()
	d_event_command_id++;
	write_string = QString("E%1 GetErrStatusE()").arg(d_event_command_id,4,10,QChar('0'));
	
	expected_responses.clear();
	expected_responses.push_back(QString("E%1 &").arg(d_event_command_id,4,10,QChar('0')));
	expected_responses.push_back(QString("E%1 # ErrStatus").arg(d_event_command_id,4,10,QChar('0')));
	expected_responses.push_back(QString("E%1 %").arg(d_event_command_id,4,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									 MODE_WAIT_RESPONSE,
									 expected_responses,
									 &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  GetErrStatusE() did not return expected response."));
		
		d_is_machine_error_state = true;
		return;
	}
			
	if(actual_responses.size() != 3)
	{
		this->Add_Event_Text(QStringLiteral("ERR:  GetErrStatusE() did not return expected response."));
		
		d_is_machine_error_state = true;
		return;
	}

	source_string = actual_responses[1];
	d_is_machine_estop = false;
	
	regular_expression.setPattern(QStringLiteral("[ErrStatus(]{10}([{0,1}[0-9.]{1})[)]{1}"));
	expression_match = regular_expression.match(source_string);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() > 1)
		{
			d_is_machine_estop = (0 != expression_text_list[1].toInt(&valid));
			
			if(!valid)
			{
				this->Add_Event_Text(QStringLiteral("ERR:  Invalid data returned from GetErrStatusE() command."));

				d_is_machine_error_state = true;
				return;
			}
		}
	}
	else
	{
		this->Add_Event_Text(QStringLiteral("ERR:  Invalid data returned from GetErrStatusE() command."));

		d_is_machine_error_state = true;
		return;
	}
}

void TControllerIppClient::Abort(void)
{
	QString								write_string;
	QString								source_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;

	// AbortE()
	d_event_command_id++;
	write_string = QString("E%1 AbortE()").arg(d_event_command_id,4,10,QChar('0'));
	
	expected_responses.clear();
	expected_responses.push_back(QString("E%1 %").arg(d_event_command_id,4,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
		
	write_string += QStringLiteral("\r\n");
			
	this->Send_Command(write_string.toLatin1(),
									 MODE_WAIT_RESPONSE,
									 expected_responses,
									 &actual_responses);
			
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	d_command_response_queue.clear();
	d_touch_ids.clear();
}

bool TControllerIppClient::Query_Tool(
	const QString						&tool_name)
{
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	QRegularExpression					regular_expression;
	QRegularExpressionMatch				expression_match;
	QStringList							expression_text_list;
	QStringList							list;
	QString								write_string;
	QString								source_string;
	QString								line;
	TVector3							pnt;
	TVector3							vec;
	TController::TSendResult			write_status;
	bool								valid;

	// ChangeTool
	d_command_id++;
	write_string = QString("%1 FindTool(\"%2\")")
										.arg(d_command_id,5,10,QChar('0'))
										.arg(tool_name);
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									 MODE_WAIT_RESPONSE,
									 expected_responses,
									 &actual_responses,
									 30);	// Wait up to 30 seconds for probe change to complete
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  FindTool() did not return expected response."));
		return false;
	}
	
	// Tip Radius
	d_command_id++;
	write_string = QString("%1 GetProp(FoundTool.AvrRadius())").arg(d_command_id,5,10,QChar('0'));
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 # FoundTool.AvrRadius").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									 MODE_WAIT_RESPONSE,
									 expected_responses,
									 &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(actual_responses.size() != 3)
	{
		this->Add_Event_Text("ERR:  GetProp() did not return expected response.");
		return false;
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text("ERR:  GetProp() did not return expected response.");
		return false;
	}
	
	source_string = actual_responses[1];
	
	regular_expression.setPattern(QStringLiteral("[AvrRadius(]{10}([{0,1}[0-9.]{1,12})[)]{1}"));
	expression_match = regular_expression.match(source_string);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() > 1)
		{
			d_query_tip_radius = expression_text_list[1].toDouble(&valid);
			
			if(!valid)
			{
				this->Add_Event_Text(QStringLiteral("ERR:  Invalid data returned from GetProp() command."));
				return false;
			}
		}
	}
	else
	{
		this->Add_Event_Text(QStringLiteral("ERR:  Invalid data returned from GetProp() command."));
		return false;
	}
	
	// Probe offset and vector
	d_command_id++;
	write_string = QString("%1 GetProp(FoundTool.EOffset())").arg(d_command_id,5,10,QChar('0'));
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 # FoundTool.EOffset").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									 MODE_WAIT_RESPONSE,
									 expected_responses,
									 &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(actual_responses.size() != 3)
	{
		this->Add_Event_Text("ERR:  GetProp() did not return expected response.");
		return false;
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text("ERR:  GetProp() did not return expected response.");
		return false;
	}
	
	source_string = actual_responses[1];
	
	regular_expression.setPattern(QStringLiteral("[EOffset(]{8}([ ]{0,1}[-?]{0,2}[0-9.]{1,16},[ ]{0,2}[-?]{0,1}[0-9. ]{1,16},[ ]{0,2}[-?]{0,1}[0-9. ]{1,16})[)]{1}"));
	expression_match = regular_expression.match(source_string);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		list = expression_text_list[1].split(',');
		
		if(list.size() != 3)
		{
			this->Add_Event_Text(QStringLiteral("ERR:  Invalid XYZ data returned from GetProp(FoundTool.EOffset()) command."));
			return false;
		}
		
		pnt.x = list[0].toDouble(&valid);
		if(!valid)
		{
			this->Add_Event_Text(QStringLiteral("ERR:  Invalid X data returned from GetProp(FoundTool.EOffset()) command."));
			return false;
		}
		
		pnt.y = list[1].toDouble(&valid);
		if(!valid)
		{
			this->Add_Event_Text(QStringLiteral("ERR:  Invalid Y data returned from GetProp(FoundTool.EOffset()) command."));
			return false;
		}
		
		pnt.z = list[2].toDouble(&valid);
		if(!valid)
		{
			this->Add_Event_Text(QStringLiteral("ERR:  Invalid Z data returned from GetProp(FoundTool.EOffset()) command."));
			return false;
		}
	}
	else
	{
		this->Add_Event_Text(QStringLiteral("ERR:  Invalid XYZ data returned from GetProp(FoundTool.EOffset()) command."));
		return false;
	}

	vec = pnt;
	
	if(vec.Length() > 0.000001)
	{
		vec.Normal();
	}
	
	d_query_tool_offset = pnt;
	d_query_tool_vector = vec;
	
	return true;
}

bool TControllerIppClient::Set_Tool_Name(
	const QString						&tool_name)
{
	QString								write_string;
	QString								source_string;
	QString								line;
	QStringList							list;
	TVector3							pnt;
	TVector3							vec;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	QRegularExpression					regular_expression;
	QRegularExpressionMatch				expression_match;
	QStringList							expression_text_list;
	TController::TSendResult			write_status;
	bool								valid;
	
	// ChangeTool
	d_command_id++;
	write_string = QString("%1 ChangeTool(\"%2\")")
										.arg(d_command_id,5,10,QChar('0'))
										.arg(tool_name);
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									 MODE_WAIT_RESPONSE,
									 expected_responses,
									 &actual_responses,
									 30);	// Wait up to 30 seconds for probe change to complete
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  ChangeTool() did not return expected response."));
		return false;
	}
	
	// Tip Radius
	d_command_id++;
	write_string = QString("%1 GetProp(Tool.AvrRadius())").arg(d_command_id,5,10,QChar('0'));
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 # Tool.AvrRadius").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									 MODE_WAIT_RESPONSE,
									 expected_responses,
									 &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(actual_responses.size() != 3)
	{
		this->Add_Event_Text(QStringLiteral("ERR:  GetProp() did not return expected response."));
		return false;
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  GetProp() did not return expected response."));
		return false;
	}
	
	source_string = actual_responses[1];
	
	regular_expression.setPattern(QStringLiteral("[AvrRadius(]{10}([{0,1}[0-9.]{1,12})[)]{1}"));
	expression_match = regular_expression.match(source_string);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() > 1)
		{
			d_tip_radius = expression_text_list[1].toDouble(&valid);
			
			if(!valid)
			{
				this->Add_Event_Text(QStringLiteral("ERR:  Invalid data returned from GetProp() command."));
				return false;
			}
		}
	}
	else
	{
		this->Add_Event_Text(QStringLiteral("ERR:  Invalid data returned from GetProp() command."));
		return false;
	}
	
	// Probe offset and vector
	d_command_id++;
	write_string = QString("%1 GetProp(Tool.EOffset())").arg(d_command_id,5,10,QChar('0'));
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 # Tool.EOffset").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									 MODE_WAIT_RESPONSE,
									 expected_responses,
									 &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(actual_responses.size() != 3)
	{
		this->Add_Event_Text(QStringLiteral("ERR:  GetProp() did not return expected response."));
		return false;
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  GetProp() did not return expected response."));
		return false;
	}
	
	source_string = actual_responses[1];
	
	regular_expression.setPattern(QStringLiteral("[EOffset(]{8}([ ]{0,1}[-?]{0,2}[0-9.]{1,16},[ ]{0,2}[-?]{0,1}[0-9. ]{1,16},[ ]{0,2}[-?]{0,1}[0-9. ]{1,16})[)]{1}"));
	expression_match = regular_expression.match(source_string);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		list = expression_text_list[1].split(',');
		
		if(list.size() != 3)
		{
			this->Add_Event_Text(QStringLiteral("ERR:  Invalid XYZ data returned from GetProp(FoundTool.EOffset()) command."));
			return false;
		}
		
		pnt.x = list[0].toDouble(&valid);
		if(!valid)
		{
			this->Add_Event_Text(QStringLiteral("ERR:  Invalid X data returned from GetProp(FoundTool.EOffset()) command."));
			return false;
		}
		
		pnt.y = list[1].toDouble(&valid);
		if(!valid)
		{
			this->Add_Event_Text(QStringLiteral("ERR:  Invalid Y data returned from GetProp(FoundTool.EOffset()) command."));
			return false;
		}
		
		pnt.z = list[2].toDouble(&valid);
		if(!valid)
		{
			this->Add_Event_Text(QStringLiteral("ERR:  Invalid Z data returned from GetProp(FoundTool.EOffset()) command."));
			return false;
		}
	}
	else
	{
		this->Add_Event_Text(QStringLiteral("ERR:  Invalid XYZ data returned from GetProp(FoundTool.EOffset()) command."));
		return false;
	}

	vec = pnt;
	
	if(vec.Length() > 0.000001)
	{
		vec.Normal();
	}
	
	d_tool_offset = pnt;
	d_tool_vector = vec;
	
	// Max Speed
	d_command_id++;
	write_string = QString("%1 GetProp(Tool.GoToPar.Speed.Max())").arg(d_command_id,5,10,QChar('0'));
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 # Tool.GoToPar.Speed.Max").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_WAIT_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(actual_responses.size() != 3)
	{
		this->Add_Event_Text(QStringLiteral("ERR:  GetProp() did not return expected response."));
		return false;
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  GetProp() did not return expected response."));
		return false;
	}
	
	source_string = actual_responses[1];
	
	regular_expression.setPattern(QStringLiteral("[GoToPar.Speed.Max(]{18}([{0,1}[0-9.]{1,12})[)]{1}"));
	expression_match = regular_expression.match(source_string);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() > 1)
		{
			d_max_speed = expression_text_list[1].toDouble(&valid);
			
			if(!valid)
			{
				this->Add_Event_Text(QStringLiteral("ERR:  Invalid data returned from GetProp() command."));
				return false;
			}
		}
	}
	else
	{
		this->Add_Event_Text(QStringLiteral("ERR:  Invalid data returned from GetProp() command."));
		return false;
	}

	// Max Acceleration
	d_command_id++;
	write_string = QString("%1 GetProp(Tool.GoToPar.Accel.Max())").arg(d_command_id,5,10,QChar('0'));
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 # Tool.GoToPar.Accel.Max").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_WAIT_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(actual_responses.size() != 3)
	{
		this->Add_Event_Text(QStringLiteral("ERR:  GetProp() did not return expected response."));
		return false;
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  GetProp() did not return expected response."));
		return false;
	}
	
	source_string = actual_responses[1];
	
	regular_expression.setPattern(QStringLiteral("[GoToPar.Accel.Max(]{18}([{0,1}[0-9.]{1,12})[)]{1}"));
	expression_match = regular_expression.match(source_string);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() > 1)
		{
			d_max_acceleration = expression_text_list[1].toDouble(&valid);
			
			if(!valid)
			{
				this->Add_Event_Text(QStringLiteral("ERR:  Invalid data returned from GetProp() command."));
				return false;
			}
		}
	}
	else
	{
		this->Add_Event_Text(QStringLiteral("ERR:  Invalid data returned from GetProp() command."));
		return false;
	}

	return true;
}

bool TControllerIppClient::Set_Tool_Data(
	const TVector3						&xyz,
	const double						&tip_diameter)
{
	QString								write_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	TController::TSendResult			write_status;

	d_tool_offset = xyz;
	d_tip_radius = tip_diameter / 2.0;
	
	// SetProp(Tool.EOffsetX(),EOffsetY(),EOffsetZ())
	d_command_id++;
	write_string = QString("%1 SetProp(Tool.EOffsetX(%2),Tool.EOffsetY(%3),Tool.EOffsetZ(%4))")
									.arg(d_command_id,5,10,QChar('0'))
									.arg(xyz.x,0,'f',4)
									.arg(xyz.y,0,'f',4)
									.arg(xyz.z,0,'f',4);
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));

	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");

	write_status = this->Send_Command(
									write_string.toLatin1(),
									MODE_WAIT_RESPONSE,
									expected_responses,
									&actual_responses);
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  SetProp Tool_EOffset data not accepted."));
		return false;
	}
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));

	d_command_response_queue.push_back(expected_responses);

	return true;
}


void TControllerIppClient::Set_Mode(
	const TMode							mode)
{
	QString								write_string;
	QString								source_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	TController::TSendResult			write_status;

	d_mode = mode;
	
	if(d_mode == TController::MODE_MANUAL)
	{
		// EnableUser()
		d_command_id++;
		write_string = QString("%1 EnableUser()").arg(d_command_id,5,10,QChar('0'));
		
		expected_responses.clear();
		expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
		expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));

		this->Add_Event_Text(write_string);
		write_string += QStringLiteral("\r\n");
		
		write_status = this->Send_Command(write_string.toLatin1(),
										 MODE_WAIT_RESPONSE,
										 expected_responses,
										 &actual_responses);
		
		for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
		{
			this->Add_Event_Text(*str_iter);
		}
		
		if(write_status != TController::SEND_SUCCESS)
		{
			this->Add_Event_Text(d_last_error);
			this->Add_Event_Text(QStringLiteral("ERR:  EnableUser() did not return expected response."));
		}		
	}
}


bool TControllerIppClient::Set_Move_Speed(
	const double						&speed)
{
	QString								write_string;
	QString								source_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	TController::TSendResult			write_status;
	double								actual_speed(speed);
	
	if(actual_speed > d_max_speed)
	{
		this->Add_Event_Text(QStringLiteral("WAR:  Move speed greater than controllers max move speed."));
		actual_speed = d_max_speed;
	}

	// Move Speed
	d_command_id++;
	write_string = QString("%1 SetProp(Tool.GoToPar.Speed(%2))").arg(d_command_id,5,10,QChar('0')).arg(actual_speed,0,'f',3);
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									 MODE_WAIT_RESPONSE,
									 expected_responses,
									 &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  SetProp() did not return expected response."));
		return false;
	}
	
	return true;
}

bool TControllerIppClient::Set_Touch_Speed(
	const double						&speed)
{
	QString								write_string;
	QString								source_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	TController::TSendResult			write_status;

	// Touch Speed
	d_command_id++;
	write_string = QString("%1 SetProp(Tool.PtMeasPar.Speed(%2))").arg(d_command_id,5,10,QChar('0')).arg(speed,0,'f',3);
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
										 MODE_WAIT_RESPONSE,
										 expected_responses,
										 &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  SetProp() did not return expected response."));
		return false;
	}
			
	return true;
}

bool TControllerIppClient::Set_Acceleration(
	const double						&accel)
{
	QString								write_string;
	QString								source_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	TController::TSendResult			write_status;
	double								actual_acceleration(accel);
	
	if(actual_acceleration > d_max_acceleration)
	{
		this->Add_Event_Text(QStringLiteral("WAR:  Acceleration greater than controllers max acceleration."));
		actual_acceleration = d_max_acceleration;
	}

	// Acceleration
	d_command_id++;
	write_string = QString("%1 SetProp(Tool.GoToPar.Accel(%2))")
										.arg(d_command_id,5,10,QChar('0'))
										.arg(actual_acceleration,0,'f',3);
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									 MODE_WAIT_RESPONSE,
									 expected_responses,
									 &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  SetProp() did not return expected response."));
		return false;
	}
	
	return true;
}

bool TControllerIppClient::Set_Approach_Distance(
	const double						&dist)
{
	QString								write_string;
	QString								source_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	TController::TSendResult			write_status;

	d_current_approach_distance = dist;
	
	// Approach distance
	d_command_id++;
	write_string = QString("%1 SetProp(Tool.PtMeasPar.Approach(%2))").arg(d_command_id,5,10,QChar('0')).arg(dist,0,'f',3);
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									 MODE_WAIT_RESPONSE,
									 expected_responses,
									 &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  SetProp() did not return expected response."));
		return false;
	}
	
	// Retract distance
	d_command_id++;
	write_string = QString("%1 SetProp(Tool.PtMeasPar.Retract(%2))")
										.arg(d_command_id,5,10,QChar('0'))
										.arg(dist,0,'f',3);
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									 MODE_WAIT_RESPONSE,
									 expected_responses,
									 &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  SetProp() did not return expected response."));
		return false;
	}
	
	// search distance
	d_command_id++;
	write_string = QString("%1 SetProp(Tool.PtMeasPar.Search(%2))")
										.arg(d_command_id,5,10,QChar('0'))
										.arg(dist * 2,0,'f',3);
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									 MODE_WAIT_RESPONSE,
									 expected_responses,
									 &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  SetProp() did not return expected response."));
		return false;
	}
	
	return true;
}

bool TControllerIppClient::Enable_Position_Updates(
	const bool 							state,
	const bool 							precision_update)
{
	QString								write_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	TController::TSendResult			write_status;

	if(state == d_position_updates_enabled && precision_update == d_position_precision_updates_enabled)
	{
		// no change;
		return true;
	}
	
	d_position_updates_enabled = state;
	d_position_precision_updates_enabled = precision_update;
	
	if(d_coordinate_id.length())
	{
		// StopDeamon()
		d_command_id++;
		write_string = QString("%1 StopDaemon(%2)").arg(d_command_id,5,10,QChar('0')).arg(d_coordinate_id);
		
		expected_responses.clear();
		expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
		expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
		
		this->Add_Event_Text(write_string);
		write_string += QStringLiteral("\r\n");
		
		write_status = this->Send_Command(write_string.toLatin1(),
										MODE_WAIT_RESPONSE,
										expected_responses,
										&actual_responses);
		
		for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
		{
			this->Add_Event_Text(*str_iter);
		}
		
		d_coordinate_id = QString();
		
		if(write_status != TController::SEND_SUCCESS)
		{
			this->Add_Event_Text(d_last_error);
			this->Add_Event_Text(QStringLiteral("ERR:  StopDaemon() did not return expected response."));
			return false;
		}	
	}

	// I++ 1.7 has 10 pnts/sec limit
	// active and inprecise is 5 seconds, 10 mm.  Assuming top speed is 500 mm/sec
	// active and precise is 0.25 seconds, 200 mm.  Assuming top speed is 500 mm/sec
	if(state && !precision_update)
	{
		// Start coordinate reporting daemon at slow update time, short distance
		d_event_command_id++;
		write_string = QString("E%1 OnMoveReportE(Time(5),Dis(10),X(),Y(),Z())").arg(d_event_command_id,4,10,QChar('0'));
		
		expected_responses.clear();
		expected_responses.push_back(QString("E%1 &").arg(d_event_command_id,4,10,QChar('0')));
		expected_responses.push_back(QString("E%1 %").arg(d_event_command_id,4,10,QChar('0')));

		d_coordinate_id = QString("E%1").arg(d_event_command_id,4,10,QChar('0'));

		this->Add_Event_Text(write_string);
		write_string += QStringLiteral("\r\n");
		
		write_status = this->Send_Command(write_string.toLatin1(),
										MODE_WAIT_RESPONSE,
										expected_responses,
										&actual_responses);
		
		for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
		{
			this->Add_Event_Text(*str_iter);
		}
		
		if(write_status != TController::SEND_SUCCESS)
		{
			this->Add_Event_Text(d_last_error);
			this->Add_Event_Text(QStringLiteral("ERR:  OnMoveReportE() did not return expected response."));
			return false;
		}
		
		return true;
	}
	else if(state && precision_update)
	{
		// Start coordinate reporting daemon at fast update time, long distance
		d_event_command_id++;
		write_string = QString("E%1 OnMoveReportE(Time(0.25),Dis(200),X(),Y(),Z())").arg(d_event_command_id,4,10,QChar('0'));	// 10 pnts/sec limit, 500 mm/sec
		
		expected_responses.clear();
		expected_responses.push_back(QString("E%1 &").arg(d_event_command_id,4,10,QChar('0')));
		expected_responses.push_back(QString("E%1 %").arg(d_event_command_id,4,10,QChar('0')));

		d_coordinate_id = QString("E%1").arg(d_event_command_id,4,10,QChar('0'));

		this->Add_Event_Text(write_string);
		write_string += QStringLiteral("\r\n");
		
		write_status = this->Send_Command(write_string.toLatin1(),
										MODE_WAIT_RESPONSE,
										expected_responses,
										&actual_responses);
		
		for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
		{
			this->Add_Event_Text(*str_iter);
		}
		
		if(write_status != TController::SEND_SUCCESS)
		{
			this->Add_Event_Text(d_last_error);
			this->Add_Event_Text(QStringLiteral("ERR:  OnMoveReportE() did not return expected response."));
			return false;
		}
		
		return true;
	}
	
	return true;
}

bool TControllerIppClient::Get_Position(
	TVector3							* const pnt)
{
	TVector3							pt;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	QRegularExpression					regular_expression;
	QRegularExpressionMatch				expression_match;
	QStringList							expression_text_list;
	QString								write_string;
	QString								source_string;
	TController::TSendResult			write_status;
	bool								valid;

	assert(pnt);
		
	// GETPOS
	d_command_id++;
	write_string = QString("%1 Get(X(),Y(),Z())").arg(d_command_id,5,10,QChar('0'));
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 #").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									 MODE_WAIT_RESPONSE,
									 expected_responses,
									 &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(actual_responses.size() != 3)
	{
		this->Add_Event_Text("ERR:  Get() did not return expected response.");
		return false;
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text("ERR:  Get() did not return expected response.");
		return false;
	}
	
	source_string = actual_responses[1];
	
	regular_expression.setPattern(QStringLiteral("[X(]{2}([-?]{0,1}[0-9.]{1,12})[)]{1}"));
	expression_match = regular_expression.match(source_string);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() > 1)
		{
			pt.x = expression_text_list[1].toDouble(&valid);
			
			if(!valid)
			{
				this->Add_Event_Text(QStringLiteral("ERR:  Invalid X data returned from Get() command."));
				return false;
			}
		}
	}
	else
	{
		this->Add_Event_Text(QStringLiteral("ERR:  Invalid X data returned from Get() command."));
		return false;
	}
	
	regular_expression.setPattern(QStringLiteral("[Y(]{2}([-?]{0,1}[0-9.]{1,12})[)]{1}"));
	expression_match = regular_expression.match(source_string);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() > 1)
		{
			pt.y = expression_text_list[1].toDouble(&valid);
			
			if(!valid)
			{
				this->Add_Event_Text(QStringLiteral("ERR:  Invalid Y data returned from Get() command."));
				return false;
			}
		}
	}
	else
	{
		this->Add_Event_Text(QStringLiteral("ERR:  Invalid Y data returned from Get() command."));
		return false;
	}

	regular_expression.setPattern(QStringLiteral("[Z(]{2}([-?]{0,1}[0-9.]{1,12})[)]{1}"));
	expression_match = regular_expression.match(source_string);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() > 1)
		{
			pt.z = expression_text_list[1].toDouble(&valid);
			
			if(!valid)
			{
				this->Add_Event_Text(QStringLiteral("ERR:  Invalid Z data returned from Get() command."));
				return false;
			}
		}
	}
	else
	{
		this->Add_Event_Text(QStringLiteral("ERR:  Invalid Z data returned from Get() command."));
		return false;
	}
	
	d_current_position = pt;
	*pnt = pt;
	
	return true;
}

bool TControllerIppClient::Get_Position_DRO(
	TVector3							* const pnt)
{	
	*pnt = d_current_position;
	
	return true;
}

bool TControllerIppClient::Move_To(
	const TVector3						&pnt)
{
	QString								write_string;
	QString								source_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	TController::TSendResult			write_status;

	if(d_mode == TController::MODE_DCC)
	{
		// MOVABS
		d_command_id++;
		write_string = QString("%1 GoTo(X(%2), Y(%3), Z(%4))")
									.arg(d_command_id,5,10,QChar('0'))
									.arg(pnt.x,0,'f',5)
									.arg(pnt.y,0,'f',5)
									.arg(pnt.z,0,'f',5);
		
		expected_responses.clear();
		expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));	// require command confirmation only
		
		this->Add_Event_Text(write_string);
		write_string += QStringLiteral("\r\n");
		
		write_status = this->Send_Command(write_string.toLatin1(),
										 MODE_WAIT_RESPONSE,
										 expected_responses,
										 &actual_responses);
		
		for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
		{
			this->Add_Event_Text(*str_iter);
		}

		if(write_status != TController::SEND_SUCCESS)
		{
			this->Add_Event_Text(d_last_error);
			this->Add_Event_Text("ERR:  GoTo() did not return expected response.");
			return false;
		}
		
		expected_responses.clear();
		expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));

		d_command_response_queue.push_back(expected_responses);
	}
	
	return true;
}

bool TControllerIppClient::Touch(
	const TVector3						&target,
	const TVector3						&approach)
{
	QString								write_string;
	QString								source_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	TController::TTouchPoint			touch_point;
	TController::TSendResult			write_status;
	static const double					ZERO_EPSILON(0.00001);

	touch_point.xyz = target;
	touch_point.ijk = approach * -1.0;
	
	if(touch_point.ijk.Length() > ZERO_EPSILON)
	{
		touch_point.ijk.Normal();
	}
	else
	{
		this->Add_Event_Text("ERR:  Touch point IJK not defined.");
		return false;
	}

	if(d_mode == TController::MODE_DCC)
	{
		// PtMeas
		d_command_id++;
		write_string = QString("%1 PtMeas(X(%2), Y(%3), Z(%4), IJK(%5, %6, %7))")
											.arg(d_command_id,5,10,QChar('0'))
											.arg(touch_point.xyz.x,0,'f',5)
											.arg(touch_point.xyz.y,0,'f',5)
											.arg(touch_point.xyz.z,0,'f',5)
											.arg(touch_point.ijk.i,0,'f',5)
											.arg(touch_point.ijk.j,0,'f',5)
											.arg(touch_point.ijk.k,0,'f',5);
		
		expected_responses.clear();
		expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));	// require command confirmation only
		
		this->Add_Event_Text(write_string);
		write_string += QStringLiteral("\r\n");
		
		d_touch_ids.push_back(QString("%1").arg(d_command_id,5,10,QChar('0')));
		
		write_status = this->Send_Command(write_string.toLatin1(),
										 MODE_WAIT_RESPONSE,
										 expected_responses,
										 &actual_responses);				

		if(write_status != TController::SEND_SUCCESS)
		{
			this->Add_Event_Text(d_last_error);
			this->Add_Event_Text("ERR:  PtMeas() did not return expected response.");
			return false;
		}
		
		expected_responses.clear();
		expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));

		d_command_response_queue.push_back(expected_responses);
	}
	
	return true;
}

bool TControllerIppClient::Get_X_Temperature(
	const QString						&sensor_list,
	double								* const value,
	bool								* const valid)
{
	QString								write_string;
	QString								source_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	std::vector<int>					sensors;
	QStringList							list;
	QStringList::const_iterator			iter;
	QRegularExpression					regular_expression;
	QRegularExpressionMatch				expression_match;
	QStringList							expression_text_list;
	double								dval(20.0);
	int									sensor_id;
	bool								state;
	TController::TSendResult			write_status;

	assert(value);
	assert(valid);
	
	*valid = false;
	
	list = sensor_list.split(',');
	
	for(iter = list.begin();iter != list.end();++iter)
	{
		sensor_id = (*iter).toInt(&state);
		
		if(valid && !(sensor_id < 0))
		{
			sensors.push_back(sensor_id);
		}
	}
	
	if(sensors.size() == 0 || sensor_list.length() == 0)
	{
		return true;
	}
	
	// get sensor data
	// cannot query by specific sensor numbers
	
	d_command_id++;
	write_string = QString("%1 GetProp(Machine.Temperature.X())").arg(d_command_id,5,10,QChar('0'));
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 # Machine.Temperature.X").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_WAIT_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(actual_responses.size() != 3)
	{
		this->Add_Event_Text("ERR:  GetProp() did not return expected response.");
		return false;
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text("ERR:  GetProp() did not return expected response.");
		return false;
	}
	
	source_string = actual_responses[1];
	
	regular_expression.setPattern(QStringLiteral("[Machine.Temperature.X(]{22}([-?]{0,1}[0-9.]{1,12})[)]{1}"));
	expression_match = regular_expression.match(source_string);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() > 1)
		{
			dval = expression_text_list[1].toDouble(&state);
			
			if(!state)
			{
				this->Add_Event_Text(QStringLiteral("ERR:  Invalid data returned from GetProp() command."));
				return false;
			}
		}
	}
	else
	{
		this->Add_Event_Text(QStringLiteral("ERR:  Invalid data returned from GetProp() command."));
		return false;
	}

	*valid = true;
	*value = dval;

	return true;
}

bool TControllerIppClient::Get_Y_Temperature(
	const QString						&sensor_list,
	double								* const value,
	bool								* const valid)
{
	QString								write_string;
	QString								source_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	std::vector<int>					sensors;
	QStringList							list;
	QStringList::const_iterator			iter;
	QRegularExpression					regular_expression;
	QRegularExpressionMatch				expression_match;
	QStringList							expression_text_list;
	double								dval(20.0);
	int									sensor_id;
	bool								state;
	TController::TSendResult			write_status;

	assert(value);
	assert(valid);
	
	*valid = false;
	
	list = sensor_list.split(',');
	
	for(iter = list.begin();iter != list.end();++iter)
	{
		sensor_id = (*iter).toInt(&state);
		
		if(valid && !(sensor_id < 0))
		{
			sensors.push_back(sensor_id);
		}
	}
	
	if(sensors.size() == 0 || sensor_list.length() == 0)
	{
		return true;
	}
	
	// get sensor data
	// cannot query by specific sensor numbers
	
	d_command_id++;
	write_string = QString("%1 GetProp(Machine.Temperature.Y())").arg(d_command_id,5,10,QChar('0'));
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 # Machine.Temperature.Y").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_WAIT_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(actual_responses.size() != 3)
	{
		this->Add_Event_Text("ERR:  GetProp() did not return expected response.");
		return false;
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text("ERR:  GetProp() did not return expected response.");
		return false;
	}
	
	source_string = actual_responses[1];
	
	regular_expression.setPattern(QStringLiteral("[Machine.Temperature.Y(]{22}([-?]{0,1}[0-9.]{1,12})[)]{1}"));
	expression_match = regular_expression.match(source_string);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() > 1)
		{
			dval = expression_text_list[1].toDouble(&state);
			
			if(!state)
			{
				this->Add_Event_Text(QStringLiteral("ERR:  Invalid data returned from GetProp() command."));
				return false;
			}
		}
	}
	else
	{
		this->Add_Event_Text(QStringLiteral("ERR:  Invalid data returned from GetProp() command."));
		return false;
	}
	
	*valid = true;
	*value = dval;
	
	return true;
}

bool TControllerIppClient::Get_Z_Temperature(
	const QString						&sensor_list,
	double								* const value,
	bool								* const valid)
{
	QString								write_string;
	QString								source_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	std::vector<int>					sensors;
	QStringList							list;
	QStringList::const_iterator			iter;
	QRegularExpression					regular_expression;
	QRegularExpressionMatch				expression_match;
	QStringList							expression_text_list;
	double								dval(20.0);
	int									sensor_id;
	bool								state;
	TController::TSendResult			write_status;

	assert(value);
	assert(valid);
	
	*valid = false;
	
	list = sensor_list.split(',');
	
	for(iter = list.begin();iter != list.end();++iter)
	{
		sensor_id = (*iter).toInt(&state);
		
		if(valid && !(sensor_id < 0))
		{
			sensors.push_back(sensor_id);
		}
	}
	
	if(sensors.size() == 0 || sensor_list.length() == 0)
	{
		return true;
	}
	
	// get sensor data
	// cannot query by specific sensor numbers
	
	d_command_id++;
	write_string = QString("%1 GetProp(Machine.Temperature.Z())").arg(d_command_id,5,10,QChar('0'));
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 # Machine.Temperature.Z").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_WAIT_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(actual_responses.size() != 3)
	{
		this->Add_Event_Text("ERR:  GetProp() did not return expected response.");
		return false;
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text("ERR:  GetProp() did not return expected response.");
		return false;
	}
	
	source_string = actual_responses[1];
	
	regular_expression.setPattern(QStringLiteral("[Machine.Temperature.Z(]{22}([-?]{0,1}[0-9.]{1,12})[)]{1}"));
	expression_match = regular_expression.match(source_string);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() > 1)
		{
			dval = expression_text_list[1].toDouble(&state);
			
			if(!state)
			{
				this->Add_Event_Text(QStringLiteral("ERR:  Invalid data returned from GetProp() command."));
				return false;
			}
		}
	}
	else
	{
		this->Add_Event_Text(QStringLiteral("ERR:  Invalid data returned from GetProp() command."));
		return false;
	}

	*valid = true;
	*value = dval;
	
	return true;
}

bool TControllerIppClient::Get_Part_Temperature(
	const QString						&sensor_list,
	double								* const value,
	bool								* const valid)
{
	QString								write_string;
	QString								source_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	std::vector<int>					sensors;
	QStringList							list;
	QStringList::const_iterator			iter;
	QRegularExpression					regular_expression;
	QRegularExpressionMatch				expression_match;
	QStringList							expression_text_list;
	double								dval(20.0);
	int									sensor_id;
	bool								state;
	TController::TSendResult			write_status;

	assert(value);
	assert(valid);
	
	*valid = false;
	
	list = sensor_list.split(',');
	
	for(iter = list.begin();iter != list.end();++iter)
	{
		sensor_id = (*iter).toInt(&state);
		
		if(valid && !(sensor_id < 0))
		{
			sensors.push_back(sensor_id);
		}
	}
	
	if(sensors.size() == 0 || sensor_list.length() == 0)
	{
		return true;
	}
	
	// get sensor data
	// cannot query by specific sensor numbers
	
	d_command_id++;
	write_string = QString("%1 GetProp(Part.Temperature())").arg(d_command_id,5,10,QChar('0'));
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 # Part.Temperature").arg(d_command_id,5,10,QChar('0')));
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_WAIT_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(actual_responses.size() != 3)
	{
		this->Add_Event_Text("ERR:  GetProp() did not return expected response.");
		return false;
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text("ERR:  GetProp() did not return expected response.");
		return false;
	}
	
	source_string = actual_responses[1];
	
	regular_expression.setPattern(QStringLiteral("[Part.Temperature(]{17}([-?]{0,1}[0-9.]{1,12})[)]{1}"));
	expression_match = regular_expression.match(source_string);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() > 1)
		{
			dval = expression_text_list[1].toDouble(&state);
			
			if(!state)
			{
				this->Add_Event_Text(QStringLiteral("ERR:  Invalid data returned from GetProp() command."));
				return false;
			}
		}
	}
	else
	{
		this->Add_Event_Text(QStringLiteral("ERR:  Invalid data returned from GetProp() command."));
		return false;
	}

	*valid = true;
	*value = dval;
	
	return true;
}

bool TControllerIppClient::Get_Sensor_Value(
	int									sensor_id)
{
	QString								write_string;
	QString								source_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	
	if(sensor_id == X_TEMPERATURE_SENSOR_ID)
	{
		d_command_id++;
		write_string = QString("%1 GetProp(Machine.Temperature.X())").arg(d_command_id,5,10,QChar('0'));
		
		expected_responses.clear();
		expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));	// require command confirmation only
		
		this->Add_Event_Text(write_string);
		write_string += QStringLiteral("\r\n");
		
		this->Send_Command(write_string.toLatin1(),
										  MODE_WAIT_RESPONSE,
										  expected_responses,
										  &actual_responses);
		
		expected_responses.clear();
		expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));

		d_command_response_queue.push_back(expected_responses);
		
		return true;
	}
	else if(sensor_id == Y_TEMPERATURE_SENSOR_ID)
	{
		d_command_id++;
		write_string = QString("%1 GetProp(Machine.Temperature.Y())").arg(d_command_id,5,10,QChar('0'));
		
		expected_responses.clear();
		expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));	// require command confirmation only
		
		this->Add_Event_Text(write_string);
		write_string += QStringLiteral("\r\n");
		
		this->Send_Command(write_string.toLatin1(),
										  MODE_WAIT_RESPONSE,
										  expected_responses,
										  &actual_responses);
		expected_responses.clear();
		expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));

		d_command_response_queue.push_back(expected_responses);

		return true;
	}
	else if(sensor_id == Z_TEMPERATURE_SENSOR_ID)
	{
		d_command_id++;
		write_string = QString("%1 GetProp(Machine.Temperature.Z())").arg(d_command_id,5,10,QChar('0'));
		
		expected_responses.clear();
		expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));	// require command confirmation only
		
		this->Add_Event_Text(write_string);
		write_string += QStringLiteral("\r\n");
		
		this->Send_Command(write_string.toLatin1(),
										  MODE_WAIT_RESPONSE,
										  expected_responses,
										  &actual_responses);
		
		expected_responses.clear();
		expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));

		d_command_response_queue.push_back(expected_responses);

		return true;
	}
	else if(sensor_id == PART_TEMPERATURE_SENSOR_ID)
	{
		d_command_id++;
		write_string = QString("%1 GetProp(Part.Temperature())").arg(d_command_id,5,10,QChar('0'));
		
		expected_responses.clear();
		expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));	// require command confirmation only

		this->Add_Event_Text(write_string);
		write_string += QStringLiteral("\r\n");
		
		this->Send_Command(write_string.toLatin1(),
										  MODE_WAIT_RESPONSE,
										  expected_responses,
										  &actual_responses);
		
		expected_responses.clear();
		expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));

		d_command_response_queue.push_back(expected_responses);
		return true;
	}
	
	return false;
}

bool TControllerIppClient::Set_Part_Expansion_Coefficient(
	const double						&expansion_coefficient)
{
	QString								write_string;
	QString								source_string;
	std::vector<QString>				expected_responses;
	std::vector<QString>				actual_responses;
	std::vector<QString>::iterator		str_iter;
	TController::TSendResult			write_status;

	// Approach distance
	d_command_id++;
	write_string = QString("%1 SetProp(Part.XpanCoefficient(%2))")
						.arg(d_command_id,5,10,QChar('0'))
						.arg(expansion_coefficient,0,'f',1);
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 &").arg(d_command_id,5,10,QChar('0')));	// require command confirmation only
	
	this->Add_Event_Text(write_string);
	write_string += QStringLiteral("\r\n");
	
	write_status = this->Send_Command(write_string.toLatin1(),
									  MODE_WAIT_RESPONSE,
									  expected_responses,
									  &actual_responses);
	
	for(str_iter = actual_responses.begin();str_iter != actual_responses.end();++str_iter)
	{
		this->Add_Event_Text(*str_iter);
	}
	
	if(write_status != TController::SEND_SUCCESS)
	{
		this->Add_Event_Text(d_last_error);
		this->Add_Event_Text(QStringLiteral("ERR:  SetProp() did not return expected response."));
		return false;
	}
	
	expected_responses.clear();
	expected_responses.push_back(QString("%1 %").arg(d_command_id,5,10,QChar('0')));

	d_command_response_queue.push_back(expected_responses);

	return true;
}

QString TControllerIppClient::Get_X_Sensors(void)
{
	double								temperature;
	bool								valid;
	
	this->Get_X_Temperature(QStringLiteral("1"),&temperature,&valid);
	
	if(!valid)
	{
		return QStringLiteral("-1");
	}
	
	if(temperature < 0.0)		// renishaw returns -273 for invalid.  Anything below zero is probably invalid
	{
		return QStringLiteral("-1");
	}

	return QString("%1").arg(X_TEMPERATURE_SENSOR_ID);
}

QString TControllerIppClient::Get_Y_Sensors(void)
{
	double								temperature;
	bool								valid;
	
	this->Get_Y_Temperature(QStringLiteral("2"),&temperature,&valid);
	
	if(!valid)
	{
		return QStringLiteral("-1");
	}
	
	if(temperature < 0.0)		// renishaw returns -273 for invalid.  Anything below zero is probably invalid
	{
		return QStringLiteral("-1");
	}
	
	return QString("%1").arg(Y_TEMPERATURE_SENSOR_ID);
}

QString TControllerIppClient::Get_Z_Sensors(void)
{
	double								temperature;
	bool								valid;
	
	this->Get_Z_Temperature(QStringLiteral("3"),&temperature,&valid);
	
	if(!valid)
	{
		return QStringLiteral("-1");
	}
	
	if(temperature < 0.0)		// renishaw returns -273 for invalid.  Anything below zero is probably invalid
	{
		return QStringLiteral("-1");
	}
	
	return QString("%1").arg(Z_TEMPERATURE_SENSOR_ID);
}

QString TControllerIppClient::Get_Part_Sensors(void)
{
	double								temperature;
	bool								valid;
	
	this->Get_Part_Temperature(QStringLiteral("4"),&temperature,&valid);
	
	if(!valid)
	{
		return QStringLiteral("-1");
	}
	
	if(temperature < 0.0)		// renishaw returns -273 for invalid.  Anything below zero is probably invalid
	{
		return QStringLiteral("-1");
	}
	
	return QString("%1").arg(PART_TEMPERATURE_SENSOR_ID);
}

void TControllerIppClient::Process_Rx_Data(void)
{
	TController::TErrorData				error_data;
	int									index;
	int									ival;
	QString								text;
	QRegularExpression					regular_expression;
	QRegularExpressionMatch				expression_match;
	QStringList							expression_text_list;
	
	d_rx_data += this->ReadPort();
	
	do
	{
		index = d_rx_data.indexOf('\n');
		
		if(!(index < 0))
		{
			text = QString::fromLatin1(d_rx_data.mid(0,index)).simplified();
			d_rx_data.remove(0,index + 1);
			
#ifdef ENABLE_DEBUG_MODE
			qWarning(QString("RECV: %1").arg(text).toLatin1());
#endif
			
			if(text.contains(QStringLiteral("! Error(")))
			{
				regular_expression.setPattern(QStringLiteral("[! Error(]{8}([0-9]{1})"));
				expression_match = regular_expression.match(text);
				
				if(expression_match.hasMatch())
				{
					expression_text_list = expression_match.capturedTexts();
					
					if(expression_text_list.size() > 1)
					{
						ival = expression_text_list[1].toInt();
						
						if(ival > 1)
						{
							error_data.text = text;
							
							if(text.contains(QStringLiteral("Transaction aborted"),Qt::CaseInsensitive))
							{
								error_data.severity = 0;
							}
							else
							{
								error_data.severity = 1;
							}
							
							d_error_data.push_back(error_data);
						}
					}
				}
			}

			else if(text.contains(QStringLiteral("PtMeas")))
			{
				this->Process_Manual_Touch(text);
			}
			else if(text.contains(QStringLiteral("Machine.Temperature.X")))
			{
				this->Process_Machine_Temeprature_X(text);
			}
			else if(text.contains(QStringLiteral("Machine.Temperature.Y")))
			{
				this->Process_Machine_Temeprature_Y(text);
			}
			else if(text.contains(QStringLiteral("Machine.Temperature.Z")))
			{
				this->Process_Machine_Temeprature_Z(text);
			}
			else if(text.contains(QStringLiteral("Part.Temperature")))
			{
				this->Process_Part_Temeprature(text);
			}
			else if(d_coordinate_id.length() > 0 && text.startsWith(d_coordinate_id))
			{
				this->Process_Coordinate_Update(text);
			}
			else if(this->Is_Touch(text))
			{
				this->Process_Touch(text);
			}
			else if(text.contains(QStringLiteral("KeyPress")))
			{
				if(text.contains(QStringLiteral("Done")) ||
				   text.contains(QStringLiteral("F1")))
				{
					d_key_press.push_back(TController::KEY_DONE);
				}
				else if(text.contains(QStringLiteral("Del")) ||
						text.contains(QStringLiteral("F2")))
				{
					d_key_press.push_back(TController::KEY_ERASE_HIT);
				}
			}
		}
		
	}while(!(index < 0));
}

bool TControllerIppClient::Is_Touch(
	const QString						&text)
{
	std::vector<QString>::iterator		iter;
	QString								init_text;
	
	for(iter = d_touch_ids.begin();iter != d_touch_ids.end();++iter)
	{
		init_text = (*iter) + QStringLiteral(" # ");
		
		if(text.startsWith(init_text))
		{
			d_touch_ids.erase(iter);
			return true;
		}
	}
	
	return false;
}

void TControllerIppClient::Process_Manual_Touch(
	const QString						&text)
{
	QRegularExpression					regular_expression;
	QRegularExpressionMatch				expression_match;
	QStringList							expression_text_list;
	QStringList							list;
	QString								line;
	TController::TTouchPoint			touch_point;
	bool								valid;
	static const double					ZERO_EPSILON(0.00001);

	regular_expression.setPattern(QStringLiteral("[X(]{2}([-?]{0,1}[0-9.]{1,12})[)]{1}"));
	expression_match = regular_expression.match(text);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() > 1)
		{
			touch_point.xyz.x = expression_text_list[1].toDouble(&valid);
			
			if(!valid)
			{
				this->Add_Event_Text(QStringLiteral("ERR:  Process_Manual_Touch X data returned from PtMeas() command."));
				return;
			}
		}
	}
	else
	{
		this->Add_Event_Text(QStringLiteral("ERR:  Process_Manual_Touch X data returned from PtMeas() command."));
		return;
	}
	
	regular_expression.setPattern(QStringLiteral("[Y(]{2}([-?]{0,1}[0-9.]{1,12})[)]{1}"));
	expression_match = regular_expression.match(text);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() > 1)
		{
			touch_point.xyz.y = expression_text_list[1].toDouble(&valid);
			
			if(!valid)
			{
				this->Add_Event_Text(QStringLiteral("ERR:  Process_Manual_Touch Y data returned from PtMeas() command."));
				return;
			}
		}
	}
	else
	{
		this->Add_Event_Text(QStringLiteral("ERR:  Process_Manual_Touch Y data returned from PtMeas() command."));
		return;
	}
	
	regular_expression.setPattern(QStringLiteral("[Z(]{2}([-?]{0,1}[0-9.]{1,12})[)]{1}"));
	expression_match = regular_expression.match(text);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() > 1)
		{
			touch_point.xyz.z = expression_text_list[1].toDouble(&valid);
			
			if(!valid)
			{
				this->Add_Event_Text(QStringLiteral("ERR:  Process_Manual_Touch Z data returned from PtMeas() command."));
				return;
			}
		}
	}
	else
	{
		this->Add_Event_Text(QStringLiteral("ERR:  Process_Manual_Touch Z data returned from PtMeas() command."));
		return;
	}
	
	regular_expression.setPattern(QStringLiteral("[IJK(]{4}([ ]{0,1}[-?]{0,2}[0-9.]{1,16},[ ]{0,2}[-?]{0,1}[0-9. ]{1,16},[ ]{0,2}[-?]{0,1}[0-9. ]{1,16})[)]{1}"));
	expression_match = regular_expression.match(text);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		list = expression_text_list[1].split(',');
		
		if(list.size() != 3)
		{
			this->Add_Event_Text(QStringLiteral("ERR:  Process_Manual_Touch IJK data returned from PtMeas(IJK()) command."));
			return;
		}
		
		touch_point.ijk.i = list[0].toDouble(&valid);
		if(!valid)
		{
			this->Add_Event_Text(QStringLiteral("ERR:  Process_Manual_Touch I data returned from PtMeas(IJK()) command."));
			return;
		}
		
		touch_point.ijk.j = list[1].toDouble(&valid);
		if(!valid)
		{
			this->Add_Event_Text(QStringLiteral("ERR:  Process_Manual_Touch J data returned from PtMeas(IJK()) command."));
			return;
		}
		
		touch_point.ijk.k = list[2].toDouble(&valid);
		if(!valid)
		{
			this->Add_Event_Text(QStringLiteral("ERR:  Process_Manual_Touch K data returned from PtMeas(IJK()) command."));
			return;
		}
	}
	else
	{
		this->Add_Event_Text(QStringLiteral("ERR:  Process_Manual_Touch IJK data returned from PtMeas() command."));
		return;
	}
	
	if(touch_point.ijk.Length() > ZERO_EPSILON)
	{
		touch_point.ijk.Normal();
	}
	else
	{
		touch_point.ijk.Set(0,0,1);
	}
	
	touch_point.ijk *= (-1.0);	// returned point is surface normal.  must be approach direction
	
	d_touch_points.push_back(touch_point);
	d_current_position = touch_point.xyz;
}

void TControllerIppClient::Process_Touch(
	const QString						&text)
{
	QRegularExpression					regular_expression;
	QRegularExpressionMatch				expression_match;
	QStringList							expression_text_list;
	QStringList							list;
	QString								line;
	TController::TTouchPoint			touch_point;
	bool								valid;
	static const double					ZERO_EPSILON(0.00001);
	
	regular_expression.setPattern(QStringLiteral("[X(]{2}([-?]{0,1}[0-9.]{1,12})[)]{1}"));
	expression_match = regular_expression.match(text);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() > 1)
		{
			touch_point.xyz.x = expression_text_list[1].toDouble(&valid);
			
			if(!valid)
			{
				this->Add_Event_Text(QStringLiteral("ERR:  Process_Touch X data returned from PtMeas() command."));
				return;
			}
		}
	}
	else
	{
		this->Add_Event_Text(QStringLiteral("ERR:  Process_Touch X data returned from PtMeas() command."));
		return;
	}

	regular_expression.setPattern(QStringLiteral("[Y(]{2}([-?]{0,1}[0-9.]{1,12})[)]{1}"));
	expression_match = regular_expression.match(text);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() > 1)
		{
			touch_point.xyz.y = expression_text_list[1].toDouble(&valid);
			
			if(!valid)
			{
				this->Add_Event_Text(QStringLiteral("ERR:  Process_Touch Y data returned from PtMeas() command."));
				return;
			}
		}
	}
	else
	{
		this->Add_Event_Text(QStringLiteral("ERR:  Process_Touch Y data returned from PtMeas() command."));
		return;
	}
	
	regular_expression.setPattern(QStringLiteral("[Z(]{2}([-?]{0,1}[0-9.]{1,12})[)]{1}"));
	expression_match = regular_expression.match(text);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() > 1)
		{
			touch_point.xyz.z = expression_text_list[1].toDouble(&valid);
			
			if(!valid)
			{
				this->Add_Event_Text(QStringLiteral("ERR:  Process_Touch Z data returned from PtMeas() command."));
				return;
			}
		}
	}
	else
	{
		this->Add_Event_Text(QStringLiteral("ERR:  Process_Touch Z data returned from PtMeas() command."));
		return;
	}
	
	regular_expression.setPattern(QStringLiteral("[IJK(]{4}([ ]{0,1}[-?]{0,2}[0-9.]{1,16},[ ]{0,2}[-?]{0,1}[0-9. ]{1,16},[ ]{0,2}[-?]{0,1}[0-9. ]{1,16})[)]{1}"));
	expression_match = regular_expression.match(text);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		list = expression_text_list[1].split(',');
		
		if(list.size() != 3)
		{
			this->Add_Event_Text(QStringLiteral("ERR:  Process_Touch IJK data returned from PtMeas(IJK()) command."));
			return;
		}
		
		touch_point.ijk.i = list[0].toDouble(&valid);
		if(!valid)
		{
			this->Add_Event_Text(QStringLiteral("ERR:  Process_Touch I data returned from PtMeas(IJK()) command."));
			return;
		}
		
		touch_point.ijk.j = list[1].toDouble(&valid);
		if(!valid)
		{
			this->Add_Event_Text(QStringLiteral("ERR:  Process_Touch J data returned from PtMeas(IJK()) command."));
			return;
		}
		
		touch_point.ijk.k = list[2].toDouble(&valid);
		if(!valid)
		{
			this->Add_Event_Text(QStringLiteral("ERR:  Process_Touch K data returned from PtMeas(IJK()) command."));
			return;
		}
	}
	else
	{
		this->Add_Event_Text(QStringLiteral("ERR:  Process_Touch IJK data returned from PtMeas(IJK()) command."));
		return;
	}
	
	if(touch_point.ijk.Length() > ZERO_EPSILON)
	{
		touch_point.ijk.Normal();
	}
	
	touch_point.ijk *= (-1.0);	// returned point is surface normal.  must be approach direction
	
	d_touch_points.push_back(touch_point);
	d_current_position = touch_point.xyz;
}

void TControllerIppClient::Process_Coordinate_Update(
	const QString						&text)
{
	QRegularExpression					regular_expression;
	QRegularExpressionMatch				expression_match;
	QStringList							expression_text_list;
	QStringList							list;
	QString								line;
	TVector3							pt;
	TController::TTouchPoint			touch_point;
	bool								valid;
	
	
	regular_expression.setPattern(QStringLiteral("[X(]{2}([-?]{0,1}[0-9.]{1,12})[)]{1}"));
	expression_match = regular_expression.match(text);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() > 1)
		{
			pt.x = expression_text_list[1].toDouble(&valid);
			
			if(!valid)
			{
				this->Add_Event_Text(QStringLiteral("ERR:  Invalid X data returned from Coordinate_Update."));
				return;
			}
		}
	}
	
	regular_expression.setPattern(QStringLiteral("[Y(]{2}([-?]{0,1}[0-9.]{1,12})[)]{1}"));
	expression_match = regular_expression.match(text);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() > 1)
		{
			pt.y = expression_text_list[1].toDouble(&valid);
			
			if(!valid)
			{
				this->Add_Event_Text(QStringLiteral("ERR:  Invalid Y data returned from Coordinate_Update."));
				return;
			}
		}
	}

	regular_expression.setPattern(QStringLiteral("[Z(]{2}([-?]{0,1}[0-9.]{1,12})[)]{1}"));
	expression_match = regular_expression.match(text);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() > 1)
		{
			pt.z = expression_text_list[1].toDouble(&valid);
			
			if(!valid)
			{
				this->Add_Event_Text(QStringLiteral("ERR:  Invalid Z data returned from Coordinate_Update."));
				return;
			}
		}
	}

	d_current_position = pt;
}

void TControllerIppClient::Process_Machine_Temeprature_X(
	const QString						&text)
{
	QRegularExpression					regular_expression;
	QRegularExpressionMatch				expression_match;
	QStringList							expression_text_list;
	TController::TSensorData			sensor_data;
	bool								state;
	
	regular_expression.setPattern(QStringLiteral("[Machine.Temperature.X(]{22}([-?]{0,1}[0-9.]{1,12})[)]{1}"));
	expression_match = regular_expression.match(text);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() > 1)
		{
			sensor_data.value = expression_text_list[1].toDouble(&state);
			
			if(!state)
			{
				this->Add_Event_Text(QStringLiteral("ERR:  Invalid data returned from GetProp() command."));
				return;
			}
			else
			{
				sensor_data.sensor_id = X_TEMPERATURE_SENSOR_ID;
				d_sensor_data.push_back(sensor_data);
			}
		}
	}
}

void TControllerIppClient::Process_Machine_Temeprature_Y(
	const QString						&text)
{
	QRegularExpression					regular_expression;
	QRegularExpressionMatch				expression_match;
	QStringList							expression_text_list;
	TController::TSensorData			sensor_data;
	bool								state;
	
	regular_expression.setPattern(QStringLiteral("[Machine.Temperature.Y(]{22}([-?]{0,1}[0-9.]{1,12})[)]{1}"));
	expression_match = regular_expression.match(text);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() > 1)
		{
			sensor_data.value = expression_text_list[1].toDouble(&state);
			
			if(!state)
			{
				this->Add_Event_Text(QStringLiteral("ERR:  Invalid data returned from GetProp() command."));
				return;
			}
			else
			{
				sensor_data.sensor_id = Y_TEMPERATURE_SENSOR_ID;
				d_sensor_data.push_back(sensor_data);
			}
		}
	}
}

void TControllerIppClient::Process_Machine_Temeprature_Z(
	const QString						&text)
{
	QRegularExpression					regular_expression;
	QRegularExpressionMatch				expression_match;
	QStringList							expression_text_list;
	TController::TSensorData			sensor_data;
	bool								state;
	
	regular_expression.setPattern(QStringLiteral("[Machine.Temperature.Z(]{22}([-?]{0,1}[0-9.]{1,12})[)]{1}"));
	expression_match = regular_expression.match(text);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() > 1)
		{
			sensor_data.value = expression_text_list[1].toDouble(&state);
			
			if(!state)
			{
				this->Add_Event_Text(QStringLiteral("ERR:  Invalid data returned from GetProp() command."));
				return;
			}
			else
			{
				sensor_data.sensor_id = Z_TEMPERATURE_SENSOR_ID;
				d_sensor_data.push_back(sensor_data);
			}
		}
	}
}

void TControllerIppClient::Process_Part_Temeprature(
	const QString						&text)
{
	QRegularExpression					regular_expression;
	QRegularExpressionMatch				expression_match;
	QStringList							expression_text_list;
	TController::TSensorData			sensor_data;
	bool								state;
	
	regular_expression.setPattern(QStringLiteral("[Part.Temperature(]{17}([-?]{0,1}[0-9.]{1,12})[)]{1}"));
	expression_match = regular_expression.match(text);
	
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() > 1)
		{
			sensor_data.value = expression_text_list[1].toDouble(&state);
			
			if(!state)
			{
				this->Add_Event_Text(QStringLiteral("ERR:  Invalid data returned from GetProp() command."));
				return;
			}
			else
			{
				sensor_data.sensor_id = PART_TEMPERATURE_SENSOR_ID;
				d_sensor_data.push_back(sensor_data);
			}
		}
	}
	
}



