/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <assert.h>

#include "controller_ippclient.h"
#include "controller_leitz.h"
#include "controller_dc.h"

#include "driver.h"

#ifdef ENABLE_DEBUG_MODE

#include <QWidget>
#include "executionqueuewidget.h"

TExecutionQueueWidget 					*g_execution_queue_widget;

#endif

static const int						CONTROLLER_QUEUE_SIZE(8);


TDriver::TDriver(QObject * const parent)
:QThread(const_cast<QObject*>(parent))
{
	d_controller_type = TController::CONTROLLER_VIRTUAL;
	d_protocol_type = TLinkProtocol::PROTOCOL_NONE;
	d_controller_queue_size = CONTROLLER_QUEUE_SIZE;
	
#ifdef ENABLE_DEBUG_MODE
	
	g_execution_queue_widget = new TExecutionQueueWidget;
	g_execution_queue_widget->show();
	
#endif
}

TDriver::~TDriver(void)
{
#ifdef ENABLE_DEBUG_MODE
	
	delete g_execution_queue_widget;
	
#endif
}

void TDriver::Reset(
	const TController::TControllerType		&controller_type)
{
	d_controller_type = controller_type;
}

void TDriver::Add_Command(
	const TDriver::TCommand				&command,
	const bool 							insert_front)
{
	d_mutex.lock();
	
	if(insert_front && d_commands.size())
	{
		d_commands.insert(d_commands.begin(),command);
	}
	else
	{
		d_commands.push_back(command);
	}
	
	d_mutex.unlock();
}

void TDriver::Run(void)
{
	this->Set_State_Running();
}

void TDriver::Stop(void)
{
	this->Set_State_Stopped();
	this->Purge_Commands();
}

void TDriver::Pause(void)
{
	this->Set_State_Paused();
}

void TDriver::Resume(void)
{
	this->Set_State_Resume();
}

void TDriver::Enable_Position_Updates(
	const bool							state,
	const TDriver::TCoordinateUpdateSpeed update_speed)
{
	d_position_updates_enabled = state;
	d_position_update_speed = update_speed;
}

void TDriver::Set_Command_Queue_Size(
	const int							size)
{
	if(size > 0 && !(size > CONTROLLER_QUEUE_SIZE))
	{
		d_controller_queue_size = size;
	}
	else
	{
		d_controller_queue_size = CONTROLLER_QUEUE_SIZE;
	}
}

void TDriver::run(void)
{
	TController							*controller(0);
	TController::TTouchPoint			controller_touch_point;
	TController::TSensorData			controller_sensor_data;
	TController::TErrorData				error_data;
	TController::TKeyType				key_type;
	TDriver::TCommand					command;
	QStringList							list;
	QStringList::const_iterator			list_iter;
	QString								text;
	TVector3							pnt;
	TVector3							vec;
	double								dval;
	int									dro_update_cntr;
	int									status_update_cntr;
	int									ival;
	bool								init;
	bool								error_on_connect;
	bool								data_valid;
	static const int					IDLE_SLEEP_TIME(250);		// update timing critical to I++ precision DRO daemon
	static const int					CLOSE_DELAY_TIME(500);
	static const int					DRO_UPDATE_INTERVAL(4);		// update DRO rate ~ 1 second @ IDLE_SLEEP_TIME == 250ms.
	static const int					STATUS_UPDATE_INTERVAL(8);	// 2 seconds
	
#ifdef ENABLE_DEBUG_MODE
	bool								updated_execution_widget_data;
#endif

	init = true;
	d_abort_command_signal = false;
	d_position_updates_enabled = false;
	status_update_cntr = STATUS_UPDATE_INTERVAL;
	d_position_update_speed = TDriver::UPDATE_SPEED_DEFAULT;
	d_driver_state = TDriver::STATE_NULL;
	d_machine_state = TDriver::MACHINE_NORMAL;
	dro_update_cntr = 0;
	
	this->Set_Machine_Manual_Mode();
	this->Set_Machine_Homed();
	
	emit Connecting();
	
	switch(d_controller_type)
	{
		case TController::CONTROLLER_IPPCLIENT:
			controller = new TControllerIppClient(d_protocol_type);
			break;
			
		case TController::CONTROLLER_LEITZ:
			controller = new TControllerLeitz(d_protocol_type);
			break;
			
		case TController::CONTROLLER_DC:
			controller = new TControllerDC(d_protocol_type);
			break;
			
		case TController::CONTROLLER_VIRTUAL:	// identical to leitz
			controller = new TControllerLeitz(d_protocol_type);
			break;
	}
	
	if(controller)
	{
		
#ifdef ENABLE_DEBUG_MODE
		connect(this,&TDriver::Clear_Queued_Commands,g_execution_queue_widget,&TExecutionQueueWidget::Clear_Queued_Commands);
		connect(this,&TDriver::Add_Queued_Command,g_execution_queue_widget,&TExecutionQueueWidget::Add_Queued_Command);
		connect(this,&TDriver::Clear_Response_Commands,g_execution_queue_widget,&TExecutionQueueWidget::Clear_Response_Commands);
		connect(this,&TDriver::Add_Response_Command,g_execution_queue_widget,&TExecutionQueueWidget::Add_Response_Command);
#endif
		
		controller->Set_Abort_Command(&d_abort_command_signal);
		
		if(d_protocol_type == TLinkProtocol::PROTOCOL_SERIAL)
		{
			controller->Set_Serial_DeviceName(d_serial_device_name);
			controller->Set_Serial_Baud(d_serial_baud_rate);
			controller->Set_Serial_Data(d_serial_data_bits);
			controller->Set_Serial_Parity(d_serial_parity);
			controller->Set_Serial_StopBits(d_serial_stop_bits);
			controller->Set_Serial_Flow(d_serial_flow);
			controller->Set_Serial_ReadTimeout(d_serial_read_timeout);
		}
		else if(d_protocol_type == TLinkProtocol::PROTOCOL_ETHERNET)
		{
			controller->Set_Socket_Hostname(d_socket_hostname);
			controller->Set_Socket_Port(d_socket_port);
		}
		
		do
		{
			if(controller->Mode() == TController::MODE_MANUAL && this->Is_Manual_Mode() == false)
			{
				this->Set_Machine_Manual_Mode();
			}
			else if(controller->Mode() == TController::MODE_DCC && this->Is_Manual_Mode() == true)
			{
				this->Clear_Machine_Manual_Mode();
			}
			
			if(init)
			{
				this->Set_State_Running();
				this->Clear_Machine_EStop();

				controller->Purge_Errors();
				error_on_connect = false;
				
				if(controller->Connect())
				{
					emit Connected();
				}
				else
				{
					this->Set_State_Exit();
				}
				
				this->msleep(IDLE_SLEEP_TIME);
			
				controller->Process_Rx_Data();

				// check for initial connection errors.  probably not good if connection errors exist
				do
				{
					error_data = controller->Get_Next_Error(&data_valid);
					
					if(data_valid)
					{
						if(error_data.severity > 0)
						{
							error_on_connect = true;
						}
						
						emit Error(error_data.text,error_data.severity);
					}
					
				}while(data_valid);

				if(error_on_connect)
				{
					this->Set_State_Exit();
				}

				if(d_controller_type == TController::CONTROLLER_IPPCLIENT)
				{
					emit Clear_Ipp_Tools();
				}
			
				if(!this->Is_State_Exit())
				{
					if(!controller->Initialize())
					{
						this->Set_State_Exit();
						emit Event(QString("ERR:  Machine: %1").arg(controller->Last_Error()));
					}
					
					if(d_controller_type == TController::CONTROLLER_IPPCLIENT)
					{
						list = controller->Get_Ipp_Tool_List();
						
						for(list_iter = list.begin();list_iter != list.end();++list_iter)
						{
							emit Add_Ipp_Tool((*list_iter));
						}
						
						emit Add_Ipp_Tools_Complete();
					}
				}
				
				controller->Check_Machine_Status();

				if(controller->Is_EStop())
				{
					if(!this->Is_EStop())
					{
						controller->Add_Error(QStringLiteral("Machine in emergency stop"),0);
					}
					
					this->Set_Machine_EStop();
				}
				else if(this->Is_EStop())
				{
					this->Clear_Machine_EStop();
				}
				
				if(controller->Is_Homed())
				{
					if(!this->Is_Homed())
					{
						this->Set_Machine_Homed();
					}
				}
				else if(this->Is_Homed())
				{
					this->Clear_Machine_Homed();
				}
				
				emit Update_Machine_State(true);
				
				init = false;
			}
			
			// check for completed queued commands.
			while(controller->Command_Response_Queue_Size() < static_cast<int>(d_queued_commands.size()) && d_queued_commands.size() > 0)
			{
				command = d_queued_commands[0];
				d_queued_commands.erase(d_queued_commands.begin());
				
				switch(command.command_type)
				{
					case DRIVER_SEND_HOME:
						controller->Set_Homed_State(true);
						this->Set_Machine_Homed();

						emit Update_Machine_State(false);
						break;
						
					default:
						break;
				}
			};
			
#ifdef ENABLE_DEBUG_MODE
			updated_execution_widget_data = false;
#endif
			
			while((controller->Command_Response_Queue_Size() < d_controller_queue_size) &&
				  this->Is_State_Running() &&
				  this->Get_Next_Command(&command))
			{
				
#ifdef ENABLE_DEBUG_MODE
	
				this->Debug_Show_Execution_Queues(controller);
				updated_execution_widget_data = true;
#endif
				controller->Process_Rx_Data();	// periodic check
				
				switch(command.command_type)
				{
					case DRIVER_COMMAND_NULL:
						break;
						
					case DRIVER_SEND_HOME:
						
						if(!controller->Is_EStop())
						{
							this->Clear_Machine_Homed();

							emit Update_Machine_State(false);

							if(controller->Send_Home())
							{
								d_queued_commands.push_back(command);
							}
						}
						else if(controller->Is_Error())
						{
							this->Stop();
						}

						break;

						
					case DRIVER_SET_MOVE_SPEED:
						
						if(!controller->Set_Move_Speed(command.dvalue1))
						{
							this->Stop();
							controller->Add_Error(QString("Command Set_Move_Speed failed"),2);
						}
						
						break;
						
					case DRIVER_SET_TOUCH_SPEED:
						
						if(!controller->Set_Touch_Speed(command.dvalue1))
						{
							this->Stop();
							controller->Add_Error(QString("Command Set_Touch_Speed failed"),2);
						}

						break;
						
					case DRIVER_SET_ACCELERATION:
						
						if(!controller->Set_Acceleration(command.dvalue1))
						{
							this->Stop();
							controller->Add_Error(QString("Command Set_Acceleration failed"),2);
						}

						break;
						
					case DRIVER_SET_APPROACH_DISTANCE:
						
						if(!controller->Set_Approach_Distance(command.dvalue1))
						{
							this->Stop();
							controller->Add_Error(QString("Command Set_Approach_Distance failed"),2);
						}
						else
						{
							emit Approach_Distance(command.dvalue1);
						}

						break;
						
					case DRIVER_ENABLE_BLENDED_MOVES:
						
						if(!controller->Enable_Blended_Moves())
						{
							this->Stop();
							controller->Add_Error(QString("Command Enable_Blended_Moves failed"),2);
						}
						
						break;
						
					case DRIVER_DISABLE_BLENDED_MOVES:
						
						if(!controller->Disable_Blended_Moves())
						{
							this->Stop();
							controller->Add_Error(QString("Command Disable_Blended_Moves failed"),2);
						}

						break;
						
					case DRIVER_QUERY_TOOL_DATA:
						
						if(controller->Query_Tool(command.text))
						{
							if(controller->Controller_Type() == TController::CONTROLLER_IPPCLIENT)
							{
								pnt = controller->Query_Tool_Offset();
								vec = controller->Query_Tool_Vector();
								dval = controller->Query_Tip_Diameter();
							}
							else // unable to query by name.  returning current values instead
							{
								pnt = controller->Tool_Offset();
								vec = controller->Tool_Vector();
								dval = controller->Tip_Diameter();
							}

							emit Query_Tool_Data(command.text,pnt.x,pnt.y,pnt.z,vec.i,vec.j,vec.k,dval);
						}
						else
						{
							this->Stop();
							controller->Add_Error(QString("Command Query_Tool failed"),2);
						}
		
						break;


					case DRIVER_SET_TOOL_NAME:
						
						if(controller->Controller_Type() == TController::CONTROLLER_IPPCLIENT)	// I++ retrieves information from new tool
						{
							if(!controller->Is_EStop())
							{
								ival = 0;
								emit Tool_Index_Confirmation_Name(command.text,&ival);
								
								if(ival == 2)
								{
									if(!controller->Set_Tool_Name(command.text))
									{
										this->Stop();
										controller->Add_Error(QString("Command Set_Tool_Name failed"),2);
									}
								}
								else if(ival == 1)
								{
									emit Event(QString("Manual Head Index: %1").arg(command.text));
								}
								else
								{
									this->Stop();
									emit Tool_Update_Error();
								}
							}
							else if(controller->Is_Error())
							{
								this->Stop();
								emit Tool_Update_Error();
							}
						}
						
						emit Tool_Name_Changed(command.text);

						break;
						
					case DRIVER_SET_TOOL_TYPE:
												
						if(controller->Controller_Type() != TController::CONTROLLER_IPPCLIENT)
						{
							if(!controller->Is_EStop())
							{
								if(controller->Set_Tool_Type(command.text))
								{
									d_queued_commands.push_back(command);
								}
							}
							else if(controller->Is_Error())
							{
								this->Stop();
							}
						}
						
						break;
	
					case DRIVER_SET_TOOL_DATA:						
						
						if(!controller->Is_EStop())
						{
							controller->Set_Tool_Data(command.xyz,command.dvalue1);	// xyzd
						}
						else if(controller->Is_Error())
						{
							this->Stop();
							emit Tool_Update_Error();
						}
						
						break;
											
					case DRIVER_SET_TOOL_AB_ANGLES:
												
						if(controller->Controller_Type() != TController::CONTROLLER_IPPCLIENT)
						{
							if(!controller->Is_EStop())
							{
								ival = 0;
								emit Tool_Index_Confirmation_AB(command.dvalue1,command.dvalue2,&ival);
								
								if(ival == 2)
								{
									if(controller->Set_Tool_Angles(command.dvalue1,command.dvalue2))	// angle_a,angle_b
									{
										d_queued_commands.push_back(command);
									}
									else
									{
										this->Stop();
										controller->Add_Error(QString("Command Set_Tool_AB_Angles failed"),2);
									}
								}
								else if(ival == 1)
								{
									emit Event(QString("Manual Head Index AB: %1 %2").arg(command.dvalue1,0,'f',1).arg(command.dvalue2,0,'f',1));
								}
								else
								{
									this->Stop();
									emit Tool_Update_Error();
								}
							}
							else if(controller->Is_Error())
							{
								this->Stop();
								emit Tool_Update_Error();
							}
						}
						
						break;
						
					case DRIVER_REFRESH_TOOLS:
						
						if(d_controller_type == TController::CONTROLLER_IPPCLIENT)
						{
							emit Clear_Ipp_Tools();
							
							list = controller->Get_Ipp_Tool_List();
							
							for(list_iter = list.begin();list_iter != list.end();++list_iter)
							{
								emit Add_Ipp_Tool((*list_iter));
							}
							
							emit Add_Ipp_Tools_Complete();
						}
						
						break;

					case DRIVER_SET_MODE_MANUAL:
						
						controller->Set_Mode(TController::MODE_MANUAL);
						
						this->Set_Machine_Manual_Mode();
						
						break;
						
					case DRIVER_SET_MODE_DCC:
						
						controller->Set_Mode(TController::MODE_DCC);
						
						this->Clear_Machine_Manual_Mode();

						break;
						
					case DRIVER_GET_POSITION:
						
						if(controller->Get_Position(&pnt))
						{
							emit Position(pnt.x,pnt.y,pnt.z);
						}
						else
						{
							this->Stop();
						}
				
						break;
						
					case DRIVER_MOVE_TO:
												
						if(!controller->Is_EStop())
						{
							if(controller->Move_To(command.xyz))
							{
								d_queued_commands.push_back(command);
							}
						}
						else if(controller->Is_Error())
						{
							this->Stop();
						}
						
						break;
						
					case DRIVER_TOUCH_POINT:
						
						if(!controller->Is_EStop())
						{
							if(controller->Touch(command.xyz,command.ijk))
							{
								d_queued_commands.push_back(command);
							}
						}
						else if(controller->Is_Error())
						{
							this->Stop();
						}
						
						break;
						
					case DRIVER_GET_X_TEMPERATURE:
						
						if(controller->Get_X_Temperature(command.text,&dval,&data_valid))
						{
							if(data_valid)
							{
								emit Temperature_X(dval);
							}
						}
						else
						{
							this->Stop();
							controller->Add_Error(QString("Command Get_X_Temperature failed"),2);
						}
						
						break;
						
					case DRIVER_GET_Y_TEMPERATURE:
						
						if(controller->Get_Y_Temperature(command.text,&dval,&data_valid))
						{
							if(data_valid)
							{
								emit Temperature_Y(dval);
							}
						}
						else
						{
							this->Stop();
							controller->Add_Error(QString("Command Get_Y_Temperature failed"),2);
						}
						
						break;
						
					case DRIVER_GET_Z_TEMPERATURE:
						
						if(controller->Get_Z_Temperature(command.text,&dval,&data_valid))
						{
							if(data_valid)
							{
								emit Temperature_Z(dval);
							}
						}
						else
						{
							this->Stop();
							controller->Add_Error(QString("Command Get_Z_Temperature failed"),2);
						}
						
						break;
						
					case DRIVER_GET_PART_TEMPERATURE:
						
						if(controller->Get_Part_Temperature(command.text,&dval,&data_valid))
						{
							if(data_valid)
							{
								emit Temperature_Part(dval);
							}
						}
						else
						{
							this->Stop();
							controller->Add_Error(QString("Command Get_Part_Temperature failed"),2);
						}
						
						break;
						
					case DRIVER_SET_PART_CTE:
						
						if(!controller->Set_Part_Expansion_Coefficient(command.dvalue1))
						{
							this->Stop();
							controller->Add_Error(QString("Command Set_Part_Expansion_Coefficient failed"),2);
						}
						break;
						
					case DRIVER_GET_SENSOR_IDS:
						
						text = controller->Get_X_Sensors();
						emit Sensors_X(text);
						
						text = controller->Get_Y_Sensors();
						emit Sensors_Y(text);
						
						text = controller->Get_Z_Sensors();
						emit Sensors_Z(text);

						text = controller->Get_Part_Sensors();
						emit Sensors_Part(text);

						break;
						
					case DRIVER_GET_SENSOR_VALUE:
						
						if(controller->Get_Sensor_Value(command.ivalue))
						{
							d_queued_commands.push_back(command);
						}
						else
						{
							this->Stop();
							controller->Add_Error(QString("Command Get_Sensor_Value failed"),2);
						}
						break;
						
					case DRIVER_SYNC:
						
						emit Driver_Sync(command.ivalue);

						break;
				}
				
				if(this->Is_State_Stopped())
				{
					this->msleep(IDLE_SLEEP_TIME);

					controller->Process_Rx_Data();	// periodic check

					// check for completed queued commands.
					while(controller->Command_Response_Queue_Size() < static_cast<int>(d_queued_commands.size()) && d_queued_commands.size() > 0)
					{
						d_queued_commands.erase(d_queued_commands.begin());
					};

					this->Put_Back_Commands(d_queued_commands);
					d_queued_commands.clear();

					this->Put_Back_Command(command);
				}
			}
			
#ifdef ENABLE_DEBUG_MODE
			if(!updated_execution_widget_data)
			{
				this->Debug_Show_Execution_Queues(controller);
			}
#endif

			if(this->Is_State_Stopped())
			{
				if(d_queued_commands.size())
				{
					this->Put_Back_Commands(d_queued_commands);
					d_queued_commands.clear();
				}

				if(this->Is_State_Abort())
				{
					controller->Abort();
					this->Clear_State_Abort();

					controller->Clear_Error();
					controller->Check_Machine_Status();
				}
				
				if(this->Is_State_Paused())
				{
					controller->Store_Current_Mode();
					this->Clear_State_Paused();
				}
				else
				{
					controller->Clear_Previous_Mode();
					this->Purge_Commands();				// removes previously queued commands
					this->Set_State_Running();
				}
				
				controller->Set_Mode(TController::MODE_MANUAL);
				
				this->Set_Machine_Manual_Mode();
				
				this->Clear_State_Stopped();
			}
			else if(this->Is_State_Resume())
			{
				controller->Restore_Previous_Mode();
				this->Set_State_Running();
			}
			
			// send accumulated errors.  severity > 0 triggers a stop command
			do
			{
				error_data = controller->Get_Next_Error(&data_valid);
				
				if(data_valid && this->Is_State_Running())
				{
					emit Error(error_data.text,error_data.severity);
				}
				
			}while(data_valid);

			// update events (various text used for logging and general messages)
			do
			{
				text = controller->Get_Next_Event(&data_valid);
				
				if(data_valid)
				{
					emit Event(text);
				}
				
			}while(data_valid);
									
			// update jogbox key press
			do
			{
				key_type = controller->Get_Next_Keypress(&data_valid);
				
				if(data_valid)
				{
					switch(key_type)
					{
						case TController::KEY_DONE:
							emit Key_Done();
							break;
							
						case TController::KEY_ERASE_HIT:							
							if(controller->Mode() == TController::MODE_MANUAL)
							{
								emit Key_Erase_Hit();
							}
							break;
					}
				}
				
			}while(data_valid);

			this->msleep(IDLE_SLEEP_TIME);
			
			controller->Process_Rx_Data();	// periodic check
			
			// check for touch points
			do
			{
				controller_touch_point = controller->Get_Next_TouchPoint(&data_valid);
				
				if(data_valid)
				{
					emit Touch_Point(controller_touch_point.xyz.x,
									 controller_touch_point.xyz.y,
									 controller_touch_point.xyz.z,
									 controller_touch_point.ijk.i,
									 controller_touch_point.ijk.j,
									 controller_touch_point.ijk.k);					
				}
				
			}while(data_valid);
			
			// check for temperature sensor data
			do
			{
				controller_sensor_data = controller->Get_Next_Sensor(&data_valid);
				
				if(data_valid)
				{
					emit Sensor_Value(controller_sensor_data.sensor_id,controller_sensor_data.value);
				}
				
			}while(data_valid);

			// update DRO
			if(this->Is_State_Running())
			{			
				// only send if changed state
				// disable if position updates cannot be turned on
				if((d_position_updates_enabled != controller->Updates_Enabled()))
				{
					if(!controller->Enable_Position_Updates(d_position_updates_enabled,d_position_update_speed == TDriver::UPDATE_SPEED_HIGH))
					{
						d_position_updates_enabled = false;
						d_position_update_speed = TDriver::UPDATE_SPEED_DEFAULT;
					}
				}

				if(d_position_updates_enabled)
				{
					if(!(dro_update_cntr < DRO_UPDATE_INTERVAL) || (d_position_update_speed == TDriver::UPDATE_SPEED_HIGH))
					{
						if(controller->Get_Position_DRO(&pnt))
						{
							emit Position(pnt.x,pnt.y,pnt.z);
						}
						
						dro_update_cntr = 0;
					}
					
					++dro_update_cntr;
				}
			}
			
			// update controller status
			if(controller->Is_EStop())
			{
				if(!(status_update_cntr < STATUS_UPDATE_INTERVAL))
				{
					controller->Check_Machine_Status();
					status_update_cntr = 0;
				}
				
				++status_update_cntr;
				
				if(!this->Is_EStop())
				{
					controller->Add_Error(QStringLiteral("Machine in emergency stop"),0);
					
					this->Set_Machine_EStop();
				}
			}
			else
			{
				status_update_cntr = STATUS_UPDATE_INTERVAL;

				this->Clear_Machine_EStop();
			}
			
			if(controller->Is_Error())
			{
				controller->Abort();
				this->Clear_State_Abort();

				controller->Clear_Error();
				controller->Check_Machine_Status();
			}
			
		}while(!this->Is_State_Exit());
		
		this->Clear_State_Exit();
			
		controller->Disconnect();
		
		this->msleep(CLOSE_DELAY_TIME);

		controller->Close();
		
#ifdef ENABLE_DEBUG_MODE
	disconnect(this,&TDriver::Clear_Queued_Commands,g_execution_queue_widget,&TExecutionQueueWidget::Clear_Queued_Commands);
	disconnect(this,&TDriver::Add_Queued_Command,g_execution_queue_widget,&TExecutionQueueWidget::Add_Queued_Command);
	disconnect(this,&TDriver::Clear_Response_Commands,g_execution_queue_widget,&TExecutionQueueWidget::Clear_Response_Commands);
	disconnect(this,&TDriver::Add_Response_Command,g_execution_queue_widget,&TExecutionQueueWidget::Add_Response_Command);
#endif

		delete controller;
	}
}

int TDriver::Command_Count(void) const
{
	int									size;
	
	d_mutex.lock();
	size = static_cast<int>(d_commands.size());
	d_mutex.unlock();

	return size;
}

bool TDriver::Get_Next_Command(
	TDriver::TCommand					* const command)
{
	bool								state(false);
	
	assert(command);

	d_mutex.lock();
	if(d_commands.size())
	{
		state = true;
		
		*command = d_commands[0];
		d_commands.erase(d_commands.begin());
	}
	d_mutex.unlock();
	
	return state;
}

void TDriver::Put_Back_Command(
	const TDriver::TCommand				&command)
{
	d_mutex.lock();
	d_commands.insert(d_commands.begin(),command);
	d_mutex.unlock();
}

void TDriver::Put_Back_Commands(
	const std::vector<TDriver::TCommand> &commands)
{
	std::vector<TDriver::TCommand>::const_reverse_iterator iter;
	
	d_mutex.lock();
	for(iter = commands.rbegin();iter != commands.rend();++iter)
	{
		d_commands.insert(d_commands.begin(),(*iter));
	}
	d_mutex.unlock();

}

void TDriver::Purge_Commands(void)
{
	d_mutex.lock();
	d_commands.clear();
	d_mutex.unlock();
}

#ifdef ENABLE_DEBUG_MODE

void TDriver::Debug_Show_Execution_Queues(
	TController 						* const controller)
{
	std::vector<TDriver::TCommand>::const_iterator iter;
	QString								text;
	int									response_size;
	int									cntr;
	std::vector<QString>				responses;
	std::vector<QString>::const_iterator response_iter;

	emit Clear_Queued_Commands();
			
	for(iter = d_queued_commands.begin();iter != d_queued_commands.end();++iter)
	{
		switch((*iter).command_type)
		{
			case TDriver::DRIVER_COMMAND_NULL:
				text = QStringLiteral("NULL                 ");
				break;
			case TDriver::DRIVER_SEND_HOME:
				text = QStringLiteral("SEND_HOME            ");
				break;
			case TDriver::DRIVER_SET_MODE_MANUAL:
				text = QStringLiteral("SET_MANUAL_MODE      ");
				break;
			case TDriver::DRIVER_SET_MODE_DCC:
				text = QStringLiteral("SET_DCC_MODE         ");
				break;
			case TDriver::DRIVER_GET_POSITION:
				text = QStringLiteral("GET_POSITION         ");
				break;
			case TDriver::DRIVER_MOVE_TO:
				text = QStringLiteral("MOVE_TO              ");
				text += QString(" [XYZ %1 %2 %3]").arg((*iter).xyz.x,0,'f',3).arg((*iter).xyz.y,0,'f',3).arg((*iter).xyz.z,0,'f',3);
				break;
			case TDriver::DRIVER_TOUCH_POINT:
				text = QStringLiteral("TOUCH_POINT          ");
				text += QString(" [XYZ %1 %2 %3]").arg((*iter).xyz.x,0,'f',3).arg((*iter).xyz.y,0,'f',3).arg((*iter).xyz.z,0,'f',3);
				text += QString(" [IJK %1 %2 %3]").arg((*iter).ijk.x,0,'f',5).arg((*iter).ijk.y,0,'f',5).arg((*iter).ijk.z,0,'f',5);
				break;
			case TDriver::DRIVER_SET_TOOL_NAME:
				text = QStringLiteral("SET_TOOL_NAME        ");
				text += QString(" [%1]").arg((*iter).text);
				break;
			case TDriver::DRIVER_SET_TOOL_TYPE:
				text = QStringLiteral("SET_TOOL_TYPE        ");
				break;
			case TDriver::DRIVER_QUERY_TOOL_DATA:
				text = QStringLiteral("QUERY_TOOL_DATA      ");
				text += QString(" [%1]").arg((*iter).text);
				break;
			case TDriver::DRIVER_SET_TOOL_DATA:
				text = QStringLiteral("SET_TOOL_DATA        ");
				text += QString(" XYZ %1 %2 %3]").arg((*iter).xyz.x,0,'f',3).arg((*iter).xyz.y,0,'f',3).arg((*iter).xyz.z,0,'f',3);
				break;
			case TDriver::DRIVER_SET_TOOL_AB_ANGLES:
				text = QStringLiteral("SET_TOOL_AB_ANGLES   ");
				text += QString(" AB %1 %2]").arg((*iter).dvalue1,0,'f',1).arg((*iter).dvalue2,0,'f',2);
				break;
			case TDriver::DRIVER_REFRESH_TOOLS:
				text = QStringLiteral("REFRESH_TOOLS        ");
				break;
			case TDriver::DRIVER_SET_MOVE_SPEED:
				text = QStringLiteral("SET_MOVE_SPEED       ");
				break;
			case TDriver::DRIVER_SET_TOUCH_SPEED:
				text = QStringLiteral("SET_TOUCH_SPEED      ");
				break;
			case TDriver::DRIVER_SET_ACCELERATION:
				text = QStringLiteral("SET_ACCELERATION     ");
				break;
			case TDriver::DRIVER_SET_APPROACH_DISTANCE:
				text = QStringLiteral("SET_APPROACH_DIST    ");
				break;
			case TDriver::DRIVER_ENABLE_BLENDED_MOVES:
				text = QStringLiteral("ENABLE_BLENDED_MOVES ");
				break;
			case TDriver::DRIVER_DISABLE_BLENDED_MOVES:
				text = QStringLiteral("DISABLE_BLENDED_MOVES");
				break;
			case TDriver::DRIVER_GET_X_TEMPERATURE:
				text = QStringLiteral("GET_X_TEMPERATURE    ");
				break;
			case TDriver::DRIVER_GET_Y_TEMPERATURE:
				text = QStringLiteral("GET_Y_TEMPERATURE    ");
				break;
			case TDriver::DRIVER_GET_Z_TEMPERATURE:
				text = QStringLiteral("GET_Z_TEMPERATURE    ");
				break;
			case TDriver::DRIVER_GET_PART_TEMPERATURE:
				text = QStringLiteral("GET_PART_TEMPERATURE ");
				break;
			case TDriver::DRIVER_SET_PART_CTE:
				text = QStringLiteral("SET_PART_EXP_COEFF   ");
				break;
			case TDriver::DRIVER_GET_SENSOR_IDS:
				text = QStringLiteral("GET_SENSOR_IDS       ");
				break;
			case TDriver::DRIVER_GET_SENSOR_VALUE:
				text = QStringLiteral("GET_SENSOR_VALUE     ");
				break;
			case TDriver::DRIVER_SYNC:
				text = QStringLiteral("DRIVER_SYNC          ");
				break;
		}
		
		emit Add_Queued_Command(text);
	}
		
			
	response_size = controller->Command_Response_Queue_Size();
	
	emit Clear_Response_Commands();

	for(cntr = 0;cntr < response_size;++cntr)
	{
		responses = controller->Command_Response_Queue_Entry(cntr);
		
		text.clear();
		
		for(response_iter = responses.begin();response_iter != responses.end();++response_iter)
		{
			text += QString("[%1]").arg(*response_iter);
		}
		
		emit Add_Response_Command(text);
	}
}

#endif


