/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QCheckBox>
#include <QComboBox>
#include <QDoubleSpinBox>
#include <QSpinBox>
#include <QFrame>
#include <QGridLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QLineEdit>
#include <QSizePolicy>
#include <QSpacerItem>
#include <QStringList>
#include <QToolButton>
#include <QPushButton>
#include <QPlainTextEdit>
#include <QVBoxLayout>
#include <QCloseEvent>
#include <QDateTime>
#include <QFont>
#include <QFile>
#include <QColor>
#include <assert.h>

#include "../../core/graphwidget.h"
#include "../../core/messagebox.h"

#include "coordinatedisplaywidget.h"
#include "laseroffsetdialog.h"
#include "laserinputedit.h"

#include "laserdialog.h"

static const unsigned int				MAX_FILTER_LEVEL = 10;

TLaserDialog::TLaserDialog(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QDialog(const_cast<QWidget*>(parent),flags)
{
	QFrame								*dialog_hseparator1;
	QFrame								*dialog_hseparator2;
	QFrame								*dialog_hseparator3;
	QFrame								*exectionbar_hseparator;
	QGridLayout							*executionbar_layout;
	QGridLayout							*laser_layout;
	QGridLayout							*options_layout;
	QGridLayout							*start_end_points_layout;
	QHBoxLayout							*previous_next_layout;
	QVBoxLayout							*dialog_vlayout;
	QLabel								*active_tool_label;
	QLabel								*actual_position_label;
	QLabel								*description_label;
	QLabel								*status_label;
	QLabel								*start_point_label;
	QLabel								*start_x_label;
	QLabel								*start_y_label;
	QLabel								*start_z_label;
	QLabel								*start_map_label;
	QLabel								*start_map_x_label;
	QLabel								*start_map_y_label;
	QLabel								*start_map_z_label;
	QLabel								*end_point_label;
	QLabel								*end_map_label;
	QLabel								*end_x_label;
	QLabel								*end_y_label;
	QLabel								*end_z_label;
	QLabel								*end_map_x_label;
	QLabel								*end_map_y_label;
	QLabel								*end_map_z_label;
	QLabel								*measurement_length_label;
	QLabel								*increment_label;
	QLabel								*filter_level_label;
	QLabel								*premove_units_label;
	QLabel								*increment_units_label;
	QLabel								*laser_value_label;
	QLabel								*measurement_pattern_label;
	QLabel								*data_collection_mode_label;
	QLabel								*laser_measurement_type_label;
	QLabel								*tool_offset_x_label;
	QLabel								*tool_offset_y_label;
	QLabel								*tool_offset_z_label;
	QLabel								*premove_offset_label;
	QLabel								*target_position_label;
	QSpacerItem							*executionbar_hspacer_left;
	QSpacerItem							*executionbar_hspacer_right;
	QSpacerItem							*prev_next_hspacer;
	QSpacerItem							*statusbar_hspacer;
	QSizePolicy							size_policy_fixed_fixed(QSizePolicy::Fixed,QSizePolicy::Fixed);
	QSizePolicy							size_policy_expanding_preferred(QSizePolicy::Expanding, QSizePolicy::Preferred);
	QFont								laser_input_font(QApplication::font());
	QFont								graph_font(QApplication::font());
	QFont								logtext_font("Courier New");

	laser_input_font.setPointSize(laser_input_font.pointSize() + 2);
	laser_input_font.setWeight(QFont::DemiBold);	// demi-bold
	laser_input_font.setStretch(110);	// 1.10%
	
	graph_font.setPointSize(graph_font.pointSize() - 2);

	size_policy_fixed_fixed.setHorizontalStretch(0);
	size_policy_fixed_fixed.setVerticalStretch(0);
	
	size_policy_expanding_preferred.setHorizontalStretch(0);
	size_policy_expanding_preferred.setVerticalStretch(0);
	
	this->resize(745,831);
	
	dialog_vlayout = new QVBoxLayout(this);
	
	executionbar_layout = new QGridLayout();
	
	d_play_button = new QToolButton(this);
	d_play_button->setMinimumSize(32,32);
	d_play_button->setMaximumSize(32,32);
	d_play_button->setIconSize(QSize(32,32));
	executionbar_layout->addWidget(d_play_button,0,0,1,1);
	
	d_stop_button = new QToolButton(this);
	d_stop_button->setMinimumSize(32,32);
	d_stop_button->setMaximumSize(32,32);
	d_stop_button->setIconSize(QSize(32,32));
	executionbar_layout->addWidget(d_stop_button,0,1,1,1);
	
	executionbar_hspacer_left = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	executionbar_layout->addItem(executionbar_hspacer_left,0,2,1,2);

	active_tool_label = new QLabel(this);
	active_tool_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	executionbar_layout->addWidget(active_tool_label,0,4,1,1);
	
	d_active_tool_combo = new QComboBox(this);
	executionbar_layout->addWidget(d_active_tool_combo,0,5,1,1);

	d_close_button = new QToolButton(this);
	d_close_button->setMinimumSize(32,32);
	d_close_button->setMaximumSize(32,32);
	d_close_button->setIconSize(QSize(32,32));
	executionbar_layout->addWidget(d_close_button,0,8,1,1);
	
	exectionbar_hseparator = new QFrame(this);
	exectionbar_hseparator->setFrameShape(QFrame::HLine);
	exectionbar_hseparator->setFrameShadow(QFrame::Sunken);
	executionbar_layout->addWidget(exectionbar_hseparator,1,0,1,9);
	
	statusbar_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	executionbar_layout->addItem(statusbar_hspacer,2,6,4,1);

	d_icon_label = new QLabel(this);
	d_icon_label->setSizePolicy(size_policy_fixed_fixed);
	d_icon_label->setMinimumSize(128,128);
	d_icon_label->setMaximumSize(128,128);
	d_icon_label->setFrameShape(QFrame::StyledPanel);
	executionbar_layout->addWidget(d_icon_label,2,0,4,4);

	tool_offset_x_label = new QLabel(this);
	executionbar_layout->addWidget(tool_offset_x_label,2,4,1,1);
	
	d_tool_x_edit = new QLineEdit(this);
	d_tool_x_edit->setAlignment(Qt::AlignCenter);
	executionbar_layout->addWidget(d_tool_x_edit,2,5,1,1);
	
	tool_offset_y_label = new QLabel(this);
	executionbar_layout->addWidget(tool_offset_y_label,3,4,1,1);

	d_tool_y_edit = new QLineEdit(this);
	d_tool_y_edit->setAlignment(Qt::AlignCenter);
	executionbar_layout->addWidget(d_tool_y_edit,3,5,1,1);
	
	tool_offset_z_label = new QLabel(this);
	executionbar_layout->addWidget(tool_offset_z_label,4,4,1,1);

	d_tool_z_edit = new QLineEdit(this);
	d_tool_z_edit->setAlignment(Qt::AlignCenter);
	executionbar_layout->addWidget(d_tool_z_edit,4,5,1,1);
		
	d_set_offsets_button = new QPushButton(this);
	d_set_offsets_button->setSizePolicy(size_policy_expanding_preferred);
	d_set_offsets_button->setAutoDefault(false);
	executionbar_layout->addWidget(d_set_offsets_button,5,5,1,1);

	executionbar_hspacer_right = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	executionbar_layout->addItem(executionbar_hspacer_right,0,6,1,2);
	
	d_coordinate_display_widget = new TCoordinateDisplayWidget(this);
	executionbar_layout->addWidget(d_coordinate_display_widget,2,7,4,2);

	dialog_vlayout->addLayout(executionbar_layout);
	
	dialog_hseparator1 = new QFrame(this);
	dialog_hseparator1->setFrameShape(QFrame::HLine);
	dialog_hseparator1->setFrameShadow(QFrame::Sunken);
	dialog_vlayout->addWidget(dialog_hseparator1);
	
	d_start_end_points_widget = new QWidget(this);
	
	start_end_points_layout = new QGridLayout(d_start_end_points_widget);
	start_end_points_layout->setContentsMargins(0,0,0,0);
	
	start_point_label = new QLabel(d_start_end_points_widget);
	start_point_label->setAlignment(Qt::AlignLeading|Qt::AlignLeft|Qt::AlignVCenter);
	start_end_points_layout->addWidget(start_point_label,0,0,1,1);
	
	start_x_label = new QLabel(d_start_end_points_widget);
	start_end_points_layout->addWidget(start_x_label,0,1,1,1);
	
	d_start_x_edit = new QLineEdit(d_start_end_points_widget);
	d_start_x_edit->setAlignment(Qt::AlignCenter);
	start_end_points_layout->addWidget(d_start_x_edit,0,2,1,1);
	
	start_y_label = new QLabel(d_start_end_points_widget);
	start_end_points_layout->addWidget(start_y_label,0,3,1,1);
	
	d_start_y_edit = new QLineEdit(d_start_end_points_widget);
	d_start_y_edit->setAlignment(Qt::AlignCenter);
	start_end_points_layout->addWidget(d_start_y_edit,0,4,1,1);
	
	start_z_label = new QLabel(d_start_end_points_widget);
	start_end_points_layout->addWidget(start_z_label,0,5,1,1);
	
	d_start_z_edit = new QLineEdit(d_start_end_points_widget);
	d_start_z_edit->setAlignment(Qt::AlignCenter);
	start_end_points_layout->addWidget(d_start_z_edit,0,6,1,1);

	start_map_label = new QLabel(d_start_end_points_widget);
	start_map_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	start_end_points_layout->addWidget(start_map_label,1,0,1,1);
	
	start_map_x_label = new QLabel(d_start_end_points_widget);
	start_end_points_layout->addWidget(start_map_x_label,1,1,1,1);
	
	d_start_map_x_label = new QLabel(d_start_end_points_widget);
	d_start_map_x_label->setAlignment(Qt::AlignCenter);
	d_start_map_x_label->setFrameShape(QFrame::StyledPanel);
	start_end_points_layout->addWidget(d_start_map_x_label,1,2,1,1);
	
	start_map_y_label = new QLabel(d_start_end_points_widget);
	start_end_points_layout->addWidget(start_map_y_label,1,3,1,1);
	
	d_start_map_y_label = new QLabel(d_start_end_points_widget);
	d_start_map_y_label->setAlignment(Qt::AlignCenter);
	d_start_map_y_label->setFrameShape(QFrame::StyledPanel);
	start_end_points_layout->addWidget(d_start_map_y_label,1,4,1,1);
	
	start_map_z_label = new QLabel(d_start_end_points_widget);
	start_end_points_layout->addWidget(start_map_z_label,1,5,1,1);
	
	d_start_map_z_label = new QLabel(d_start_end_points_widget);
	d_start_map_z_label->setAlignment(Qt::AlignCenter);
	d_start_map_z_label->setFrameShape(QFrame::StyledPanel);
	start_end_points_layout->addWidget(d_start_map_z_label,1,6,1,1);
	
	d_start_getpos_button = new QPushButton(d_start_end_points_widget);
	d_start_getpos_button->setSizePolicy(size_policy_expanding_preferred);
	d_start_getpos_button->setAutoDefault(false);
	start_end_points_layout->addWidget(d_start_getpos_button,0,7,2,1);
	
	d_start_offset_button = new QPushButton(d_start_end_points_widget);
	d_start_offset_button->setSizePolicy(size_policy_expanding_preferred);
	d_start_offset_button->setAutoDefault(false);
	start_end_points_layout->addWidget(d_start_offset_button,0,8,2,1);
	
	d_start_move_button = new QPushButton(d_start_end_points_widget);
	d_start_move_button->setSizePolicy(size_policy_expanding_preferred);
	d_start_move_button->setAutoDefault(false);
	start_end_points_layout->addWidget(d_start_move_button,0,9,2,1);

	end_point_label = new QLabel(d_start_end_points_widget);
	end_point_label->setAlignment(Qt::AlignLeading|Qt::AlignLeft|Qt::AlignVCenter);
	start_end_points_layout->addWidget(end_point_label,2,0,1,1);
	
	end_x_label = new QLabel(d_start_end_points_widget);
	start_end_points_layout->addWidget(end_x_label,2,1,1,1);
	
	d_end_x_edit = new QLineEdit(d_start_end_points_widget);
	d_end_x_edit->setAlignment(Qt::AlignCenter);
	start_end_points_layout->addWidget(d_end_x_edit,2,2,1,1);
	
	end_y_label = new QLabel(d_start_end_points_widget);
	start_end_points_layout->addWidget(end_y_label,2,3,1,1);
	
	d_end_y_edit = new QLineEdit(d_start_end_points_widget);
	d_end_y_edit->setAlignment(Qt::AlignCenter);
	start_end_points_layout->addWidget(d_end_y_edit,2,4,1,1);
	
	end_z_label = new QLabel(d_start_end_points_widget);
	start_end_points_layout->addWidget(end_z_label,2,5,1,1);
	
	d_end_z_edit = new QLineEdit(d_start_end_points_widget);
	d_end_z_edit->setAlignment(Qt::AlignCenter);
	start_end_points_layout->addWidget(d_end_z_edit,2,6,1,1);
	
	end_map_label = new QLabel(d_start_end_points_widget);
	end_map_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	start_end_points_layout->addWidget(end_map_label,3,0,1,1);
	
	end_map_x_label = new QLabel(d_start_end_points_widget);
	start_end_points_layout->addWidget(end_map_x_label,3,1,1,1);
	
	d_end_map_x_label = new QLabel(d_start_end_points_widget);
	d_end_map_x_label->setAlignment(Qt::AlignCenter);
	d_end_map_x_label->setFrameShape(QFrame::StyledPanel);
	start_end_points_layout->addWidget(d_end_map_x_label,3,2,1,1);
	
	end_map_y_label = new QLabel(d_start_end_points_widget);
	start_end_points_layout->addWidget(end_map_y_label,3,3,1,1);
	
	d_end_map_y_label = new QLabel(d_start_end_points_widget);
	d_end_map_y_label->setAlignment(Qt::AlignCenter);
	d_end_map_y_label->setFrameShape(QFrame::StyledPanel);
	start_end_points_layout->addWidget(d_end_map_y_label,3,4,1,1);
	
	end_map_z_label = new QLabel(d_start_end_points_widget);
	start_end_points_layout->addWidget(end_map_z_label,3,5,1,1);
	
	d_end_map_z_label = new QLabel(d_start_end_points_widget);
	d_end_map_z_label->setAlignment(Qt::AlignCenter);
	d_end_map_z_label->setFrameShape(QFrame::StyledPanel);
	start_end_points_layout->addWidget(d_end_map_z_label,3,6,1,1);
	
	d_end_getpos_button = new QPushButton(d_start_end_points_widget);
	d_end_getpos_button->setSizePolicy(size_policy_expanding_preferred);
	d_end_getpos_button->setAutoDefault(false);
	start_end_points_layout->addWidget(d_end_getpos_button,2,7,2,1);
	
	d_end_offset_button = new QPushButton(d_start_end_points_widget);
	d_end_offset_button->setSizePolicy(size_policy_expanding_preferred);
	d_end_offset_button->setAutoDefault(false);
	start_end_points_layout->addWidget(d_end_offset_button,2,8,2,1);
	
	d_end_move_button = new QPushButton(d_start_end_points_widget);
	d_end_move_button->setSizePolicy(size_policy_expanding_preferred);
	d_end_move_button->setAutoDefault(false);
	start_end_points_layout->addWidget(d_end_move_button,2,9,2,1);

	dialog_vlayout->addWidget(d_start_end_points_widget);
	
	dialog_hseparator2 = new QFrame(this);
	dialog_hseparator2->setFrameShape(QFrame::HLine);
	dialog_hseparator2->setFrameShadow(QFrame::Sunken);
	dialog_vlayout->addWidget(dialog_hseparator2);
	
	d_options_widget = new QWidget(this);
	
	options_layout = new QGridLayout(d_options_widget);
	options_layout->setContentsMargins(0,0,0,0);
	
	description_label = new QLabel(d_options_widget);
	options_layout->addWidget(description_label,0,0,1,1);
	
	d_description_edit = new QLineEdit(d_options_widget);
	d_description_edit->setAlignment(Qt::AlignLeft|Qt::AlignVCenter);
	options_layout->addWidget(d_description_edit,0,1,1,2);
	
	premove_offset_label = new QLabel(d_options_widget);
	options_layout->addWidget(premove_offset_label,1,0,1,1);
	
	d_premove_offset_spin = new QDoubleSpinBox(d_options_widget);
	d_premove_offset_spin->setAlignment(Qt::AlignCenter);
	d_premove_offset_spin->setMinimum(0.5);
	d_premove_offset_spin->setMaximum(5);
	d_premove_offset_spin->setSingleStep(0.5);
	options_layout->addWidget(d_premove_offset_spin,1,1,1,1);
	
	premove_units_label = new QLabel(d_options_widget);
	options_layout->addWidget(premove_units_label,1,2,1,1);
	
	increment_label = new QLabel(d_options_widget);
	options_layout->addWidget(increment_label,2,0,1,1);
	
	d_increment_spin = new QDoubleSpinBox(d_options_widget);
	d_increment_spin->setAlignment(Qt::AlignCenter);
	d_increment_spin->setMinimum(10);
	d_increment_spin->setMaximum(1000);
	d_increment_spin->setSingleStep(10);
	options_layout->addWidget(d_increment_spin,2,1,1,1);
	
	increment_units_label = new QLabel(d_options_widget);
	options_layout->addWidget(increment_units_label,2,2,1,1);
	
	filter_level_label = new QLabel(d_options_widget);
	options_layout->addWidget(filter_level_label,3,0,1,1);
	
	d_filter_level_spin = new QSpinBox(d_options_widget);
	d_filter_level_spin->setAlignment(Qt::AlignCenter);
	options_layout->addWidget(d_filter_level_spin,3,1,1,1);
	
	measurement_length_label = new QLabel(d_options_widget);
	measurement_length_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	options_layout->addWidget(measurement_length_label,0,3,1,1);
	
	d_measurement_length_label = new QLabel(d_options_widget);
	d_measurement_length_label->setAlignment(Qt::AlignLeft|Qt::AlignVCenter);
	options_layout->addWidget(d_measurement_length_label,0,4,1,1);
	
	measurement_pattern_label = new QLabel(d_options_widget);
	measurement_pattern_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	options_layout->addWidget(measurement_pattern_label,1,3,1,1);
	
	d_measurement_pattern_combo = new QComboBox(d_options_widget);
	options_layout->addWidget(d_measurement_pattern_combo,1,4,1,1);
	dialog_vlayout->addWidget(d_options_widget);
	
	data_collection_mode_label = new QLabel(d_options_widget);
	data_collection_mode_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	options_layout->addWidget(data_collection_mode_label,2,3,1,1);

	d_data_collection_mode_combo = new QComboBox(d_options_widget);
	options_layout->addWidget(d_data_collection_mode_combo,2,4,1,1);

	laser_measurement_type_label = new QLabel(d_options_widget);
	laser_measurement_type_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	options_layout->addWidget(laser_measurement_type_label,3,3,1,1);

	d_laser_measurement_type_combo = new QComboBox(d_options_widget);
	options_layout->addWidget(d_laser_measurement_type_combo,3,4,1,1);
	
	dialog_hseparator3 = new QFrame(this);
	dialog_hseparator3->setFrameShape(QFrame::HLine);
	dialog_hseparator3->setFrameShadow(QFrame::Sunken);
	dialog_vlayout->addWidget(dialog_hseparator3);
	
	d_laser_widget = new QWidget(this);
	
	laser_layout = new QGridLayout(d_laser_widget);
	laser_layout->setContentsMargins(0,0,0,0);
	
	target_position_label = new QLabel(d_laser_widget);
	laser_layout->addWidget(target_position_label,0,0,1,1);
	
	d_target_position_edit = new QLineEdit(d_laser_widget);
	d_target_position_edit->setAlignment(Qt::AlignCenter);
	d_target_position_edit->setReadOnly(true);
	laser_layout->addWidget(d_target_position_edit,0,1,1,1);
	
	actual_position_label = new QLabel(d_laser_widget);
	laser_layout->addWidget(actual_position_label,1,0,1,1);
	
	d_actual_position_edit = new QLineEdit(d_laser_widget);
	d_actual_position_edit->setAlignment(Qt::AlignCenter);
	d_actual_position_edit->setReadOnly(true);
	laser_layout->addWidget(d_actual_position_edit,1,1,1,1);
	
	status_label = new QLabel(d_laser_widget);
	laser_layout->addWidget(status_label,2,0,1,1);
	
	d_status_edit = new QLineEdit(d_laser_widget);
	d_status_edit->setAlignment(Qt::AlignCenter);
	d_status_edit->setReadOnly(true);
	laser_layout->addWidget(d_status_edit,2,1,1,1);
	
	previous_next_layout = new QHBoxLayout();
	
	d_previous_button = new QToolButton(d_laser_widget);
	d_previous_button->setMinimumSize(32,32);
	d_previous_button->setMaximumSize(32,32);
	d_previous_button->setIconSize(QSize(32,32));
	previous_next_layout->addWidget(d_previous_button);
	
	prev_next_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	previous_next_layout->addItem(prev_next_hspacer);
	
	d_next_button = new QToolButton(d_laser_widget);
	d_next_button->setMinimumSize(32,32);
	d_next_button->setMaximumSize(32,32);
	d_next_button->setIconSize(QSize(32,32));
	previous_next_layout->addWidget(d_next_button);
	laser_layout->addLayout(previous_next_layout,3,1,1,1);

	laser_value_label = new QLabel(d_laser_widget);
	laser_value_label->setAlignment(Qt::AlignLeading|Qt::AlignLeft|Qt::AlignVCenter);
	laser_value_label->setFont(laser_input_font);
	laser_layout->addWidget(laser_value_label,4,0,1,1);
	
	d_laser_value_edit = new TLaserInputEdit(d_laser_widget);
	d_laser_value_edit->setSizePolicy(size_policy_expanding_preferred);
	d_laser_value_edit->setAlignment(Qt::AlignCenter);
	d_laser_value_edit->setFont(laser_input_font);
	laser_layout->addWidget(d_laser_value_edit,4,1,1,1);

	d_graph_widget = new TGraphWidget(d_laser_widget);
	d_graph_widget->setMinimumSize(450,200);
	d_graph_widget->setFont(graph_font);
	laser_layout->addWidget(d_graph_widget,0,2,5,1);
	
	dialog_vlayout->addWidget(d_laser_widget);
	
	d_information_text = new QPlainTextEdit(this);
	d_information_text->setFont(logtext_font);
	d_information_text->setLineWrapMode(QPlainTextEdit::NoWrap);
	d_information_text->setReadOnly(true);
	dialog_vlayout->addWidget(d_information_text);
	
	d_message_box = new TMessageBox(this);
	d_laser_offset_Dialog = 0;
	
	QWidget::setTabOrder(d_play_button,d_stop_button);
	QWidget::setTabOrder(d_stop_button,d_active_tool_combo);
	QWidget::setTabOrder(d_active_tool_combo,d_tool_x_edit);
	QWidget::setTabOrder(d_tool_x_edit,d_tool_y_edit);
	QWidget::setTabOrder(d_tool_y_edit,d_tool_z_edit);
	QWidget::setTabOrder(d_tool_z_edit,d_set_offsets_button);
	QWidget::setTabOrder(d_set_offsets_button,d_start_x_edit);
	QWidget::setTabOrder(d_start_x_edit,d_start_y_edit);
	QWidget::setTabOrder(d_start_y_edit,d_start_z_edit);
	QWidget::setTabOrder(d_start_z_edit,d_start_getpos_button);
	QWidget::setTabOrder(d_start_getpos_button,d_start_offset_button);
	QWidget::setTabOrder(d_start_offset_button,d_start_move_button);
	QWidget::setTabOrder(d_start_move_button,d_start_map_x_label);
	QWidget::setTabOrder(d_start_map_x_label,d_start_map_y_label);
	QWidget::setTabOrder(d_start_map_y_label,d_start_map_z_label);
	QWidget::setTabOrder(d_start_map_z_label,d_end_x_edit);
	QWidget::setTabOrder(d_end_x_edit,d_end_y_edit);
	QWidget::setTabOrder(d_end_y_edit,d_end_z_edit);
	QWidget::setTabOrder(d_end_z_edit,d_end_getpos_button);
	QWidget::setTabOrder(d_end_getpos_button,d_end_offset_button);
	QWidget::setTabOrder(d_end_offset_button,d_end_move_button);
	QWidget::setTabOrder(d_end_move_button,d_end_map_x_label);
	QWidget::setTabOrder(d_end_map_x_label,d_end_map_y_label);
	QWidget::setTabOrder(d_end_map_y_label,d_end_map_z_label);
	QWidget::setTabOrder(d_end_map_z_label,d_description_edit);
	QWidget::setTabOrder(d_description_edit,d_premove_offset_spin);
	QWidget::setTabOrder(d_premove_offset_spin,d_increment_spin);
	QWidget::setTabOrder(d_increment_spin,d_filter_level_spin);
	QWidget::setTabOrder(d_filter_level_spin,d_measurement_pattern_combo);
	QWidget::setTabOrder(d_measurement_pattern_combo,d_data_collection_mode_combo);
	QWidget::setTabOrder(d_data_collection_mode_combo,d_laser_measurement_type_combo);
	QWidget::setTabOrder(d_laser_measurement_type_combo,d_target_position_edit);
	QWidget::setTabOrder(d_target_position_edit,d_actual_position_edit);
	QWidget::setTabOrder(d_actual_position_edit,d_status_edit);
	QWidget::setTabOrder(d_status_edit,d_laser_value_edit);
	QWidget::setTabOrder(d_laser_value_edit,d_previous_button);
	QWidget::setTabOrder(d_previous_button,d_next_button);
	QWidget::setTabOrder(d_next_button,d_information_text);
	QWidget::setTabOrder(d_information_text,d_close_button);
	
	d_measurement_pattern_combo->addItem(QStringLiteral("Data Collection"),TLaserDialog::MEASUREMENT_DATA_COLLECTION);
	d_measurement_pattern_combo->addItem(QStringLiteral("Testing 10360-2"),TLaserDialog::MEASUREMENT_10360_2);
	
	d_data_collection_mode_combo->addItem(QStringLiteral("Bidirectional"),TLaserDialog::MODE_BIDIRECTIONAL);
	d_data_collection_mode_combo->addItem(QStringLiteral("Unidirectional"),TLaserDialog::MODE_UNIDIRECTIONAL);
	
	this->setWindowTitle(QStringLiteral("Laser Measurement"));
	
	d_play_button->setIcon(QIcon(":/icon32/play_icon32.png"));
	d_stop_button->setIcon(QIcon(":/icon32/stop_icon32.png"));
	d_close_button->setIcon(QIcon(":/icon32/close_icon32.png"));
	d_next_button->setIcon(QIcon(":/icon32/next_icon32.png"));
	d_previous_button->setIcon(QIcon(":/icon32/previous_icon32.png"));
	
	d_play_button->setToolTip("Play Sequence");
	d_stop_button->setToolTip("Stop Sequence");
	d_close_button->setToolTip("Close Dialog");
	d_next_button->setToolTip("Next Step");
	d_previous_button->setToolTip("Previous Step");

	active_tool_label->setText(QStringLiteral("Active Tool:"));
	tool_offset_x_label->setText(QStringLiteral("Tool Offset X:"));
	tool_offset_y_label->setText(QStringLiteral("Tool Offset Y:"));
	tool_offset_z_label->setText(QStringLiteral("Tool Offset Z:"));
	start_point_label->setText(QStringLiteral("Start Point:"));
	start_x_label->setText(QStringLiteral("X:"));
	start_y_label->setText(QStringLiteral("Y:"));
	start_z_label->setText(QStringLiteral("Z:"));
	start_map_label->setText(QStringLiteral("Map:"));
	start_map_x_label->setText(QStringLiteral("X:"));
	start_map_y_label->setText(QStringLiteral("Y:"));
	start_map_z_label->setText(QStringLiteral("Z:"));
	end_point_label->setText(QStringLiteral("End Point:"));
	end_x_label->setText(QStringLiteral("X:"));
	end_y_label->setText(QStringLiteral("Y:"));
	end_z_label->setText(QStringLiteral("Z:"));
	end_map_label->setText(QStringLiteral("Map:"));
	end_map_x_label->setText(QStringLiteral("X:"));
	end_map_y_label->setText(QStringLiteral("Y:"));
	end_map_z_label->setText(QStringLiteral("Z:"));
	measurement_length_label->setText(QStringLiteral("Max Length:"));
	description_label->setText(QStringLiteral("Description:"));
	premove_offset_label->setText(QStringLiteral("Premove Offset:"));
	premove_units_label->setText(QStringLiteral("mm"));
	increment_label->setText(QStringLiteral("Increment:"));
	increment_units_label->setText(QStringLiteral("mm"));
	filter_level_label->setText(QStringLiteral("Filter Level:"));
	laser_measurement_type_label->setText(QStringLiteral("Measurement Type:"));
	data_collection_mode_label->setText(QStringLiteral("Data Collection Mode:"));
	measurement_pattern_label->setText(QStringLiteral("Measurement Pattern:"));
	target_position_label->setText(QStringLiteral("Target Position:"));
	actual_position_label->setText(QStringLiteral("Actual Position:"));
	status_label->setText(QStringLiteral("Status:"));
	laser_value_label->setText(QStringLiteral("Laser Value:"));
	d_description_edit->setText("X Axis");
	
	d_graph_widget->Title(QString());
	d_graph_widget->VerticalLegend("Deviation");
	d_graph_widget->HorizontalLegend("Length");
	
	d_tool_x_edit->setText("0.0000");
	d_tool_y_edit->setText("0.0000");
	d_tool_z_edit->setText("0.0000");
	d_start_x_edit->setText("100.0000");
	d_start_y_edit->setText("100.0000");
	d_start_z_edit->setText("-250.0000");
	d_end_x_edit->setText("100.0000");
	d_end_y_edit->setText("400.0000");
	d_end_z_edit->setText("-250.0000");

	d_previous_button->setText(QStringLiteral("Prev"));
	d_next_button->setText(QStringLiteral("Next"));
	d_start_getpos_button->setText(QStringLiteral("Get Pos"));
	d_start_offset_button->setText(QStringLiteral("Offset"));
	d_start_move_button->setText(QStringLiteral("Move"));
	d_end_getpos_button->setText(QStringLiteral("Get Pos"));
	d_end_offset_button->setText(QStringLiteral("Offset"));
	d_end_move_button->setText(QStringLiteral("Move"));
	d_set_offsets_button->setText(QStringLiteral("Set Tool Offsets"));

	// defaults
	d_controller_type = TController::CONTROLLER_VIRTUAL;
	
	this->Set_State(TLaserDialog::STATE_STOPPED);
	
	d_input_data_error = false;
	d_filter_level_spin->setValue(MAX_FILTER_LEVEL / 3);

	d_increment_spin->setValue(50.0);
	d_premove_offset_spin->setValue(2.0);
	
	d_graph_widget->SetScaleMinimumY(0.010,true);
	d_graph_widget->EnableMouseCoordinateDisplay(false);
	d_graph_widget->Enable_Direct_Render(true);
	d_graph_widget->SetGraphType(TGraphWidget::POINT_GRAPH);
	
	this->Update_Map_Points();
	this->Update_Measurement_Axis();
	this->Update_Measurement_Length();
	
	d_start_getpos_button->setText(QStringLiteral("Get Pos"));
	d_start_offset_button->setText(QStringLiteral("Offset"));
	d_start_move_button->setText(QStringLiteral("Move"));
	d_end_getpos_button->setText(QStringLiteral("Get Pos"));
	d_end_offset_button->setText(QStringLiteral("Offset"));
	d_end_move_button->setText(QStringLiteral("Move"));
	
	connect(d_close_button,&QToolButton::clicked,this,&TLaserDialog::Close);
	connect(d_set_offsets_button,&QPushButton::clicked,this,&TLaserDialog::Set_Tool_Offsets);
	
	connect(d_start_getpos_button,&QPushButton::clicked,this,&TLaserDialog::Get_Start_Position);
	connect(d_end_getpos_button,&QPushButton::clicked,this,&TLaserDialog::Get_End_Position);
	connect(d_start_offset_button,&QPushButton::clicked,this,&TLaserDialog::Offset_Start_Position);
	connect(d_end_offset_button,&QPushButton::clicked,this,&TLaserDialog::Offset_End_Position);
	connect(d_start_move_button,&QPushButton::clicked,this,&TLaserDialog::Move_Start_Position);
	connect(d_end_move_button,&QPushButton::clicked,this,&TLaserDialog::Move_End_Position);
	connect(d_next_button,&QToolButton::clicked,this,&TLaserDialog::Next);
	connect(d_previous_button,&QToolButton::clicked,this,&TLaserDialog::Previous);
	connect(d_play_button,&QToolButton::clicked,this,&TLaserDialog::Start);
	connect(d_stop_button,&QToolButton::clicked,this,&TLaserDialog::Stop);
	
	connect(d_start_x_edit,&QLineEdit::editingFinished,this,&TLaserDialog::Start_Position_Edited);
	connect(d_start_y_edit,&QLineEdit::editingFinished,this,&TLaserDialog::Start_Position_Edited);
	connect(d_start_z_edit,&QLineEdit::editingFinished,this,&TLaserDialog::Start_Position_Edited);
	connect(d_end_x_edit,&QLineEdit::editingFinished,this,&TLaserDialog::End_Position_Edited);
	connect(d_end_y_edit,&QLineEdit::editingFinished,this,&TLaserDialog::End_Position_Edited);
	connect(d_end_z_edit,&QLineEdit::editingFinished,this,&TLaserDialog::End_Position_Edited);

	connect(d_active_tool_combo,static_cast<void (QComboBox::*)(const int)>(&QComboBox::activated),this,&TLaserDialog::Tool_Selection_Changed);
	connect(d_measurement_pattern_combo,static_cast<void (QComboBox::*)(const int)>(&QComboBox::currentIndexChanged),this,&TLaserDialog::Measurement_Pattern_Changed);
}

TLaserDialog::~TLaserDialog(void)
{
}

TLaserDialog::TParameters TLaserDialog::Parameters(void)
{
	TLaserDialog::TParameters			parameters;
	
	parameters.description = d_description_edit->text();
	
	parameters.increment = d_increment_spin->value();
	parameters.premove_offset = d_premove_offset_spin->value();
	parameters.start_x = d_start_x_edit->text().toDouble();
	parameters.start_y = d_start_y_edit->text().toDouble();
	parameters.start_z = d_start_z_edit->text().toDouble();
	parameters.end_x = d_end_x_edit->text().toDouble();
	parameters.end_y = d_end_y_edit->text().toDouble();
	parameters.end_z = d_end_z_edit->text().toDouble();
	parameters.tool_offset_x = d_tool_x_edit->text().toDouble();
	parameters.tool_offset_y = d_tool_y_edit->text().toDouble();
	parameters.tool_offset_z = d_tool_z_edit->text().toDouble();

	parameters.measurement_pattern = d_measurement_pattern_combo->currentIndex();
	parameters.data_collection_mode = d_data_collection_mode_combo->currentIndex();
	parameters.measurement_type = d_laser_measurement_type_combo->currentIndex();
	parameters.filter_level = d_filter_level_spin->value();
		
	return parameters;
}

std::vector<TDriver::TCommand> TLaserDialog::Get_Next_Driver_Commands(void)
{
	std::vector<TDriver::TCommand>		commands;
	
	commands = d_commands;
	d_commands.clear();
	
	return commands;
}

void TLaserDialog::Sequence_Canceled(void)
{
	this->Stop();
}

void TLaserDialog::Set_Parameters(
	TLaserDialog::TParameters			&parameters)
{
	d_description_edit->setText(parameters.description);

	d_increment_spin->setValue(parameters.increment);
	d_premove_offset_spin->setValue(parameters.premove_offset);
	d_start_x_edit->setText(QString("%1").arg(parameters.start_x,0,'f',4));
	d_start_y_edit->setText(QString("%1").arg(parameters.start_y,0,'f',4));
	d_start_z_edit->setText(QString("%1").arg(parameters.start_z,0,'f',4));
	d_end_x_edit->setText(QString("%1").arg(parameters.end_x,0,'f',4));
	d_end_y_edit->setText(QString("%1").arg(parameters.end_y,0,'f',4));
	d_end_z_edit->setText(QString("%1").arg(parameters.end_z,0,'f',4));
	d_tool_x_edit->setText(QString("%1").arg(parameters.tool_offset_x,0,'f',4));
	d_tool_y_edit->setText(QString("%1").arg(parameters.tool_offset_y,0,'f',4));
	d_tool_z_edit->setText(QString("%1").arg(parameters.tool_offset_z,0,'f',4));

	d_measurement_pattern_combo->setCurrentIndex(parameters.measurement_pattern);
	d_data_collection_mode_combo->setCurrentIndex(parameters.data_collection_mode);
	d_laser_measurement_type_combo->setCurrentIndex(parameters.measurement_type);
	d_filter_level_spin->setValue(parameters.filter_level);
		
	this->Update_Map_Points();
	this->Update_Measurement_Axis();
	this->Update_Measurement_Length();
}

void TLaserDialog::Set_Tool_Data(
	const TToolWidget::TToolItem 		&tool_item)
{	
	d_tool_x_edit->setText(QString::number(tool_item.xyz.x,'f',4));
	d_tool_y_edit->setText(QString::number(tool_item.xyz.y,'f',4));
	d_tool_z_edit->setText(QString::number(tool_item.xyz.z,'f',4));
}

void TLaserDialog::Set_Controller_Type(
	TController::TControllerType 		type)
{
	d_controller_type = type;
}

void TLaserDialog::Reset(
	const QString						&output_path,
	const QStringList					&tool_list,
	const QString						&active_tool_name)
{
	bool								prev_state;
	int									index;
	
	d_output_path = output_path;
	
	prev_state = d_active_tool_combo->blockSignals(true);
	
	d_active_tool_combo->clear();
	d_active_tool_combo->addItems(tool_list);
	
	// use active tool name fist index (no name)
	if(active_tool_name.length())
	{
		index = d_active_tool_combo->findText(active_tool_name);
		
		if(!(index < 0))
		{
			d_active_tool_combo->setCurrentIndex(index);
		}
		else
		{
			d_active_tool_combo->setCurrentIndex(0);	// default to '-- select tool --'
		}
	}
	else
	{
		d_active_tool_combo->setCurrentIndex(0);	// default to '-- select tool --'
	}
	
	d_active_tool_combo->blockSignals(prev_state);
	d_active_tool_combo->setEnabled(true);
}

void TLaserDialog::Reset_Tool_Selection(void)
{
	bool								prev_state;
	
	if(d_active_tool_combo->count())
	{
		prev_state = d_active_tool_combo->blockSignals(true);
		d_active_tool_combo->setCurrentIndex(0);	// default to '-- select tool --'
		d_active_tool_combo->blockSignals(prev_state);
	}
}

void TLaserDialog::Update_Position(
	const TVector3						&xyz)
{
	d_coordinate_display_widget->Set_XYZ(xyz);
	
	d_filter_data.insert(d_filter_data.begin(),xyz);
	
	while(d_filter_data.size() > MAX_FILTER_LEVEL)
	{
		d_filter_data.erase(d_filter_data.end() - 1);
	}
	
	if(d_current_state == TLaserDialog::STATE_RUNNING)
	{
		this->Update_Actual_Position();
	}
}

void TLaserDialog::Driver_Error(
	const QString						&error,
	const int							severity)
{
	Q_UNUSED(error);
	
	this->Add_Log_Text(QString("ERR:  Severity %1 '%2'").arg(severity).arg(error));
	
	if(d_current_state != TLaserDialog::STATE_STOPPED)
	{
		// send stop and reset for sever errors
		if(severity > 0)
		{
			this->Add_Log_Text(QStringLiteral("INF:  Move Sequence Cancelled"));
			this->Stop();
		}
	}
}

void TLaserDialog::Close(void)
{
	if(d_current_state != TLaserDialog::STATE_STOPPED)
	{
		emit Stop_Driver();
	}
	
	this->Set_State(TLaserDialog::STATE_STOPPED);
	this->close();
}

void TLaserDialog::Add_Log_Text(
	const QString						&text)
{
	d_information_text->moveCursor(QTextCursor::End);
	d_information_text->appendPlainText(text);
}

void TLaserDialog::Set_Tool_Offsets(void)
{
	TVector3							tool_offset;
	int									index;
	bool								prev_state;
	bool								valid;

	if(d_controller_type == TController::CONTROLLER_IPPCLIENT)
	{
		index = d_active_tool_combo->currentIndex();
		
		if(index < 1)
		{
			d_message_box->setText("Tool Error");
			d_message_box->setInformativeText("Tool must be selected prior to setting tool offset.");
			d_message_box->setDetailedText(QString());
			d_message_box->setStandardButtons(QMessageBox::Ok);
			d_message_box->setDefaultButton(QMessageBox::Ok);
			d_message_box->setIcon(QMessageBox::Critical);

			d_message_box->exec();
			return;
		}
	}

	tool_offset = this->Get_Tool_Offsets(&valid);

	if(!valid)
	{
		d_message_box->setText("Tool Offset Error");
		d_message_box->setInformativeText("Tool offset data is not valid.");
		d_message_box->setDetailedText(QString());
		d_message_box->setStandardButtons(QMessageBox::Ok);
		d_message_box->setDefaultButton(QMessageBox::Ok);
		d_message_box->setIcon(QMessageBox::Critical);

		d_message_box->exec();
		return;
	}

	if(d_controller_type == TController::CONTROLLER_IPPCLIENT)
	{
		d_message_box->setText("Tool Offset Change");
		d_message_box->setInformativeText("The tool offset of the selected probe will be changed on the server.  Are you sure?");
		d_message_box->setDetailedText(QString());
		d_message_box->setStandardButtons(QMessageBox::Yes|QMessageBox::No);
		d_message_box->setDefaultButton(QMessageBox::No);
		d_message_box->setIcon(QMessageBox::Warning);

		if(d_message_box->exec() == QMessageBox::No)
		{
			return;
		}
	}
	else
	{
		if(d_active_tool_combo->count())
		{
			prev_state = d_active_tool_combo->blockSignals(true);
			d_active_tool_combo->setCurrentIndex(0);
			d_active_tool_combo->blockSignals(prev_state);
		}
	}

	emit Set_Active_Tool_Offsets(tool_offset.x,tool_offset.y,tool_offset.z,0.0);
}

void TLaserDialog::Tool_Selection_Changed(
	int									index)
{
	QString								text;
	
	text = d_active_tool_combo->itemText(index);
	
	emit Change_Tool(text);
}

void TLaserDialog::Measurement_Pattern_Changed(
	int									index)
{
	TLaserDialog::TMeasurementPattern	measurement_pattern;
	
	measurement_pattern = static_cast<TLaserDialog::TMeasurementPattern>(d_measurement_pattern_combo->itemData(index).toInt());
	
	switch(measurement_pattern)
	{
		case TLaserDialog::MEASUREMENT_DATA_COLLECTION:
			d_data_collection_mode_combo->setEnabled(true);
			d_laser_measurement_type_combo->setEnabled(true);
			d_increment_spin->setEnabled(true);
			break;
			
		case TLaserDialog::MEASUREMENT_10360_2:
			d_data_collection_mode_combo->setEnabled(false);
			d_laser_measurement_type_combo->setEnabled(false);
			d_increment_spin->setEnabled(false);
			break;
	}
}

void TLaserDialog::Start_Position_Edited(void)
{
	this->Update_Map_Points();
	this->Update_Measurement_Axis();
	this->Update_Measurement_Length();
	
	d_play_button->setEnabled(!d_input_data_error);
}

void TLaserDialog::End_Position_Edited(void)
{
	this->Update_Map_Points();
	this->Update_Measurement_Axis();
	this->Update_Measurement_Length();
	
	d_play_button->setEnabled(!d_input_data_error);
}

void TLaserDialog::Get_Start_Position(void)
{
	TVector3							position;
	int									index;

	if(d_controller_type == TController::CONTROLLER_IPPCLIENT)
	{
		index = d_active_tool_combo->currentIndex();
		
		if(index < 1)
		{
			d_message_box->setText("Tool Error");
			d_message_box->setInformativeText("Tool must be selected prior to reading position.");
			d_message_box->setDetailedText(QString());
			d_message_box->setStandardButtons(QMessageBox::Ok);
			d_message_box->setDefaultButton(QMessageBox::Ok);
			d_message_box->setIcon(QMessageBox::Critical);

			d_message_box->exec();
			return;
		}
	}
		
	position = d_coordinate_display_widget->Get_XYZ();
	
	this->Set_Start_Point(position);
	this->Start_Position_Edited();
}

void TLaserDialog::Get_End_Position(void)
{
	TVector3							position;
	int									index;
	
	if(d_controller_type == TController::CONTROLLER_IPPCLIENT)
	{
		index = d_active_tool_combo->currentIndex();

		if(index < 1)
		{
			d_message_box->setText("Tool Error");
			d_message_box->setInformativeText("Tool must be selected prior to reading position.");
			d_message_box->setDetailedText(QString());
			d_message_box->setStandardButtons(QMessageBox::Ok);
			d_message_box->setDefaultButton(QMessageBox::Ok);
			d_message_box->setIcon(QMessageBox::Critical);

			d_message_box->exec();
			return;
		}
	}

	position = d_coordinate_display_widget->Get_XYZ();

	this->Set_End_Point(position);
	this->End_Position_Edited();
}

void TLaserDialog::Offset_Start_Position(void)
{
	TVector3							start;
	TVector3							end;
	TVector3							offset;
	
	if(!d_laser_offset_Dialog)
	{
		d_laser_offset_Dialog = new TLaserOffsetDialog(this);
	}
	
	d_laser_offset_Dialog->Reset(TLaserOffsetDialog::OFFSET_XYZ,0.0);
	
	if(QDialog::Accepted == d_laser_offset_Dialog->exec())
	{
		if(d_laser_offset_Dialog->Is_Offset_Position())
		{
			start = this->Get_Start_Point();
			end = this->Get_End_Point();
			
			offset.x = d_laser_offset_Dialog->X();
			offset.y = d_laser_offset_Dialog->Y();
			offset.z = d_laser_offset_Dialog->Z();
			
			start += offset;
			end += offset;
			
			this->Set_Start_Point(start);
			this->Set_End_Point(end);
			
			this->Update_Map_Points();
			this->Update_Measurement_Length();
		}
		else
		{
			assert(false);	// should never be here
		}
		
		if(d_laser_offset_Dialog->Is_Automove() && d_laser_offset_Dialog->Is_Offset_Position())
		{
			this->Move_Start_Position();
		}
	}
}

void TLaserDialog::Offset_End_Position(void)
{
	TVector3							start;
	TVector3							end;
	TVector3							offset;
	TVector3							middle;
	TVector3							vec;
	double								length;
	static const double					MINIMUM_LENGTH(1.0);;
	
	if(!d_laser_offset_Dialog)
	{
		d_laser_offset_Dialog = new TLaserOffsetDialog(this);
	}
	
	start = this->Get_Start_Point();
	end = this->Get_End_Point();

	vec = end - start;
	length = vec.Length();

	if(length < MINIMUM_LENGTH)
	{
		d_laser_offset_Dialog->Reset(TLaserOffsetDialog::OFFSET_XYZ,length);
	}
	else
	{
		d_laser_offset_Dialog->Reset(TLaserOffsetDialog::OFFSET_XYZ_OR_LENGTH,length);
	}
	
	if(QDialog::Accepted == d_laser_offset_Dialog->exec())
	{
		if(d_laser_offset_Dialog->Is_Offset_Position())
		{
			offset.x = d_laser_offset_Dialog->X();
			offset.y = d_laser_offset_Dialog->Y();
			offset.z = d_laser_offset_Dialog->Z();
			
			start += offset;
			end += offset;
			
			this->Set_Start_Point(start);
			this->Set_End_Point(end);
			
			this->Update_Map_Points();
			this->Update_Measurement_Length();
		}
		else if(d_laser_offset_Dialog->Is_Offset_Length())
		{
			if(length > MINIMUM_LENGTH)	// don't bother if points are too close
			{
				vec.Normal();
				length = d_laser_offset_Dialog->Length();
				
				if(d_laser_offset_Dialog->Offset_From_Start())
				{
					end = start + (vec * length);
					this->Set_End_Point(end);
				}
				else if(d_laser_offset_Dialog->Offset_From_Center())
				{
					middle = (start + end) / 2.0;
					
					start = middle - (vec * length / 2.0);
					end = middle + (vec * length / 2.0);
					
					this->Set_Start_Point(start);
					this->Set_End_Point(end);
				}
				
				this->Update_Map_Points();
				this->Update_Measurement_Length();
			}
		}
		else
		{
			assert(false);
		}
		
		if(d_laser_offset_Dialog->Is_Automove() && d_laser_offset_Dialog->Is_Offset_Position())
		{
			this->Move_End_Position();
		}
	}
}

void TLaserDialog::Move_Start_Position(void)
{
	TVector3							position;

	bool								state(false);
	int									index;
	
	if(d_controller_type == TController::CONTROLLER_IPPCLIENT)
	{
		index = d_active_tool_combo->currentIndex();

		if(index < 1)
		{
			d_message_box->setText("Tool Error");
			d_message_box->setInformativeText("Tool must be selected prior to move.");
			d_message_box->setDetailedText(QString());
			d_message_box->setStandardButtons(QMessageBox::Ok);
			d_message_box->setDefaultButton(QMessageBox::Ok);
			d_message_box->setIcon(QMessageBox::Critical);

			d_message_box->exec();
			return;
		}
	}

	position = this->Get_Start_Point(&state);
	if(!state) return;
	
	emit Move_Position(position.x,position.y,position.z);
}

void TLaserDialog::Move_End_Position(void)
{
	TVector3							position;
	bool								state(false);
	int									index;
	
	if(d_controller_type == TController::CONTROLLER_IPPCLIENT)
	{
		index = d_active_tool_combo->currentIndex();

		if(index < 1)
		{
			d_message_box->setText("Tool Error");
			d_message_box->setInformativeText("Tool must be selected prior to move.");
			d_message_box->setDetailedText(QString());
			d_message_box->setStandardButtons(QMessageBox::Ok);
			d_message_box->setDefaultButton(QMessageBox::Ok);
			d_message_box->setIcon(QMessageBox::Critical);

			d_message_box->exec();
			return;
		}
	}

	position = this->Get_End_Point(&state);
	if(!state) return;

	emit Move_Position(position.x,position.y,position.z);
}

void TLaserDialog::Laser_Value_Changed(void)
{
	TLaserDialog::TMeasurementPattern	measurement_pattern;
	TLaserDialog::TMeasuredData			measured_data;
	
	if(!d_laser_value_edit->hasFocus())
	{
		return;
	}
	
	measurement_pattern = static_cast<TLaserDialog::TMeasurementPattern>(d_measurement_pattern_combo->currentData().toInt());
	
	measured_data.nominal_target = d_target_position_edit->text().toDouble();
	measured_data.actual_target = d_actual_position_edit->text().toDouble();
	measured_data.position = d_filter_position;
	measured_data.laser_value = d_laser_value_edit->Value();
	measured_data.deviation = 0.0;
	
	switch(measurement_pattern)
	{
		case TLaserDialog::MEASUREMENT_DATA_COLLECTION:
			measured_data.index = static_cast<int>(d_direction);
			break;
			
		case TLaserDialog::MEASUREMENT_10360_2:
			measured_data.index = static_cast<int>(d_10360_target_index);
			break;
	}
	
	this->Add_Measurement(measured_data);
	
	switch(measurement_pattern)
	{
		case TLaserDialog::MEASUREMENT_DATA_COLLECTION:

			switch(d_direction)
			{
				case TLaserDialog::DIRECTION_FORWARD:
					this->Data_Collection_Next(true);
					break;
					
				case TLaserDialog::DIRECTION_REVERSE:
					this->Data_Collection_Previous(true);
					break;
			}

			break;
			
		case TLaserDialog::MEASUREMENT_10360_2:
			this->Data_10360_Next(true);
			break;
	}
}

void TLaserDialog::Next(void)
{
	TLaserDialog::TMeasurementPattern	measurement_pattern;
	
	measurement_pattern = static_cast<TLaserDialog::TMeasurementPattern>(d_measurement_pattern_combo->currentData().toInt());
	
	switch(measurement_pattern)
	{
		case TLaserDialog::MEASUREMENT_DATA_COLLECTION:
			
			if(d_direction == TLaserDialog::DIRECTION_FORWARD)
			{
				this->Data_Collection_Next(false);
			}
			else
			{
				this->Data_Collection_Previous(false);
			}
			
			break;
			
		case TLaserDialog::MEASUREMENT_10360_2:
			this->Data_10360_Next(false);
			break;
	}
}

void TLaserDialog::Previous(void)
{
	TLaserDialog::TMeasurementPattern	measurement_pattern;
	
	measurement_pattern = static_cast<TLaserDialog::TMeasurementPattern>(d_measurement_pattern_combo->currentData().toInt());
	
	switch(measurement_pattern)
	{
		case TLaserDialog::MEASUREMENT_DATA_COLLECTION:
			
			if(d_direction == TLaserDialog::DIRECTION_FORWARD)
			{
				this->Data_Collection_Previous(false);
			}
			else
			{
				this->Data_Collection_Next(false);
			}
			
			break;
			
		case TLaserDialog::MEASUREMENT_10360_2:
			this->Data_10360_Previous(false);
			break;
	}
}

void TLaserDialog::Start(void)
{
	TLaserDialog::TMeasurementPattern	measurement_pattern;
	TDriver::TCommand					command;
	TVector3							pnt;
	double								length;
	double								dval;
	int									index;
	int									total_count;
	int									fractional_count;
	static const double					MINIMUM_10360_LENGTH(100.0);
	
	if(d_controller_type == TController::CONTROLLER_IPPCLIENT)
	{
		index = d_active_tool_combo->currentIndex();
		
		if(index < 1)
		{
			d_message_box->setText("Tool Offset Error");
			d_message_box->setInformativeText("Tool must be selected prior to measurement.");
			d_message_box->setDetailedText(QString());
			d_message_box->setStandardButtons(QMessageBox::Ok);
			d_message_box->setDefaultButton(QMessageBox::Ok);
			d_message_box->setIcon(QMessageBox::Critical);

			d_message_box->exec();
			return;
		}
	}
	
	if(d_input_data_error)
	{
		d_message_box->setText("Start End Point Error");
		d_message_box->setInformativeText("The start and end points are too close together or does not contain values.");
		d_message_box->setDetailedText(QString());
		d_message_box->setStandardButtons(QMessageBox::Ok);
		d_message_box->setDefaultButton(QMessageBox::Ok);
		d_message_box->setIcon(QMessageBox::Critical);

		d_message_box->exec();
		return;
	}
	
	d_message_box->setText("Start Sequence");
	d_message_box->setInformativeText("Do you want to start the measurement sequence?.");
	d_message_box->setDetailedText(QString());
	d_message_box->setStandardButtons(QMessageBox::Yes | QMessageBox::No);
	d_message_box->setDefaultButton(QMessageBox::Yes);
	d_message_box->setIcon(QMessageBox::Question);

	if(QMessageBox::No == d_message_box->exec())
	{
		return;
	}

	this->Clear_Log();
	
	d_measured_data.clear();
	
	command.command_type = TDriver::DRIVER_SET_MODE_DCC;
	d_commands.push_back(command);
	
	measurement_pattern = static_cast<TLaserDialog::TMeasurementPattern>(d_measurement_pattern_combo->currentData().toInt());
	
	switch(measurement_pattern)
	{
		case TLaserDialog::MEASUREMENT_DATA_COLLECTION:
			
			d_start_point = this->Get_Start_Point();
			d_end_point = this->Get_End_Point();
			
			d_data_collection_target = -1.0 * d_premove_offset_spin->value();
			pnt = d_start_point + (d_measurement_axis * d_data_collection_target);
			
			command.command_type = TDriver::DRIVER_MOVE_TO;
			command.xyz = pnt;
			d_commands.push_back(command);
			this->Add_Log_Text(QString("INF  Moving to  %1, %2, %3  Position: %4")
							   .arg(command.xyz.x,0,'f',4)
							   .arg(command.xyz.y,0,'f',4)
							   .arg(command.xyz.z,0,'f',4)
							   .arg(d_data_collection_target,0,'f',4));
			
			d_data_collection_target = 0.0;
			
			command.command_type = TDriver::DRIVER_MOVE_TO;
			command.xyz = d_start_point;
			d_commands.push_back(command);
			this->Add_Log_Text(QString("INF  Moving to  %1, %2, %3  Position: %4   Waiting for laser data...")
							   .arg(command.xyz.x,0,'f',4)
							   .arg(command.xyz.y,0,'f',4)
							   .arg(command.xyz.z,0,'f',4)
							   .arg(d_data_collection_target,0,'f',4));
			
			d_target_position_edit->setText(QString("%1").arg(d_data_collection_target,0,'f',4));
			d_status_edit->setText(QStringLiteral("Forward"));
			d_direction = TLaserDialog::DIRECTION_FORWARD;
			
			break;
			
		case TLaserDialog::MEASUREMENT_10360_2:
			
			memset(d_10360_targets,0.0,sizeof(d_10360_targets));

			d_start_point = this->Get_Start_Point();
			d_end_point = this->Get_End_Point();
			
			pnt = d_end_point - d_start_point;
			length = pnt.Length();
			
			if(length < MINIMUM_10360_LENGTH)
			{
				d_message_box->setText("Start End Point Error");
				d_message_box->setInformativeText("The start and end points are too close together.");
				d_message_box->setDetailedText(QString());
				d_message_box->setStandardButtons(QMessageBox::Ok);
				d_message_box->setDefaultButton(QMessageBox::Ok);
				d_message_box->setIcon(QMessageBox::Critical);
				
				d_message_box->exec();
				return;
			}
			
			d_10360_target_index = 0;
			
			total_count = static_cast<int>( 0.5 + length / 10.0);
			fractional_count = total_count / 5;
			
			// define targets
			d_10360_targets[0] = 0.0;
			d_10360_targets[2] = 0.0;
			d_10360_targets[4] = 0.0;
			
			d_10360_targets[6] = 0.0;
			d_10360_targets[8] = 0.0;
			d_10360_targets[10] = 0.0;
			
			d_10360_targets[12] = 0.0;
			d_10360_targets[14] = 0.0;
			d_10360_targets[16] = 0.0;
			
			d_10360_targets[18] = 0.0;
			d_10360_targets[20] = 0.0;
			d_10360_targets[22] = 0.0;
			
			d_10360_targets[24] = 0.0;
			d_10360_targets[26] = 0.0;
			d_10360_targets[28] = 0.0;
			
			d_10360_targets[1] = static_cast<double>(fractional_count) * 10.0;
			d_10360_targets[3] = static_cast<double>(fractional_count) * 10.0;
			d_10360_targets[5] = static_cast<double>(fractional_count) * 10.0;
			
			d_10360_targets[7] = static_cast<double>(fractional_count * 2) * 10.0;
			d_10360_targets[9] = static_cast<double>(fractional_count * 2) * 10.0;
			d_10360_targets[11] = static_cast<double>(fractional_count * 2) * 10.0;
			
			d_10360_targets[13] = static_cast<double>(fractional_count * 3) * 10.0;
			d_10360_targets[15] = static_cast<double>(fractional_count * 3) * 10.0;
			d_10360_targets[17] = static_cast<double>(fractional_count * 3) * 10.0;
			
			d_10360_targets[19] = static_cast<double>(fractional_count * 4) * 10.0;
			d_10360_targets[21] = static_cast<double>(fractional_count * 4) * 10.0;
			d_10360_targets[23] = static_cast<double>(fractional_count * 4) * 10.0;
			
			d_10360_targets[25] = static_cast<double>(total_count) * 10.0;
			d_10360_targets[27] = static_cast<double>(total_count) * 10.0;
			d_10360_targets[29] = static_cast<double>(total_count) * 10.0;
			
			dval = -1.0 * d_premove_offset_spin->value();
			pnt = d_start_point + (d_measurement_axis * dval);
			
			command.command_type = TDriver::DRIVER_MOVE_TO;
			command.xyz = pnt;
			d_commands.push_back(command);
			this->Add_Log_Text(QString("INF  Moving to  %1, %2, %3  Position: %4")
							   .arg(command.xyz.x,0,'f',4)
							   .arg(command.xyz.y,0,'f',4)
							   .arg(command.xyz.z,0,'f',4)
							   .arg(dval,0,'f',4));
						
			command.command_type = TDriver::DRIVER_MOVE_TO;
			command.xyz = d_start_point;
			d_commands.push_back(command);
			this->Add_Log_Text(QString("INF  Moving to  %1, %2, %3  Position: %4   Waiting for laser data...")
							   .arg(command.xyz.x,0,'f',4)
							   .arg(command.xyz.y,0,'f',4)
							   .arg(command.xyz.z,0,'f',4)
							   .arg(d_10360_targets[d_10360_target_index],0,'f',4));
			
			d_target_position_edit->setText(QString("%1").arg(d_10360_targets[d_10360_target_index],0,'f',4));			
			d_status_edit->setText(QString("Target %1 of 30").arg(d_10360_target_index + 1));

			break;
	}

	connect(d_laser_value_edit,&TLaserInputEdit::editingFinished,this,&TLaserDialog::Laser_Value_Changed);

	this->Set_State(TLaserDialog::STATE_RUNNING);
	
	emit Sequence_Start();
}

void TLaserDialog::Stop(void)
{
	QString								file_name;
	
	disconnect(d_laser_value_edit,&TLaserInputEdit::editingFinished,this,&TLaserDialog::Laser_Value_Changed);

	emit Stop_Driver();
	emit Sequence_End();

	d_actual_position_edit->setText(QString());

	d_target_position_edit->setText(QString());
	d_status_edit->setText(QString());

	this->Set_State(TLaserDialog::STATE_STOPPED);
	
	this->Write_Output_File();
}

void TLaserDialog::closeEvent(
	QCloseEvent							*event)
{
	if(d_current_state == TLaserDialog::STATE_RUNNING)
	{
		emit Stop_Driver();
	}
	
	event->accept();
}

TVector3 TLaserDialog::Get_Start_Point(
	bool * const						state) const
{
	TVector3							pnt;
	bool								conversion_state;
	
	if(state)
	{
		*state = false;
	}
	
	pnt.x = d_start_x_edit->text().toDouble(&conversion_state);
	if(!conversion_state) return pnt;
	
	pnt.y = d_start_y_edit->text().toDouble(&conversion_state);
	if(!conversion_state) return pnt;
	
	pnt.z = d_start_z_edit->text().toDouble(&conversion_state);
	if(!conversion_state) return pnt;
	
	if(state)
	{
		*state = true;
	}
	
	return pnt;
}

TVector3 TLaserDialog::Get_End_Point(
	bool * const						state) const
{
	TVector3							pnt;
	bool								conversion_state;
	
	if(state)
	{
		*state = false;
	}
	
	pnt.x = d_end_x_edit->text().toDouble(&conversion_state);
	if(!conversion_state) return pnt;
	
	pnt.y = d_end_y_edit->text().toDouble(&conversion_state);
	if(!conversion_state) return pnt;
	
	pnt.z = d_end_z_edit->text().toDouble(&conversion_state);
	if(!conversion_state) return pnt;
	
	if(state)
	{
		*state = true;
	}
	
	return pnt;
}

TVector3 TLaserDialog::Get_Tool_Offsets(
	bool * const						state) const
{
	TVector3							pnt;
	bool								conversion_state;
	
	if(state)
	{
		*state = false;
	}
	
	pnt.x = d_tool_x_edit->text().toDouble(&conversion_state);
	if(!conversion_state) return pnt;
	
	pnt.y = d_tool_y_edit->text().toDouble(&conversion_state);
	if(!conversion_state) return pnt;
	
	pnt.z = d_tool_z_edit->text().toDouble(&conversion_state);
	if(!conversion_state) return pnt;
	
	if(state)
	{
		*state = true;
	}
	
	return pnt;
}

void TLaserDialog::Write_Output_File(void)
{
	std::vector<TMeasuredData>::const_iterator iter;
	QFile								file;
	QString								file_name;
	QString								text;
	TVector3							vec;
	TLaserDialog::TMeasurementPattern	measurement_pattern;
	TLaserDialog::TLaserMeasurementType	measurement_type(TLaserDialog::LASER_SCALE);
	
	if(d_measured_data.size())
	{
		file_name = d_output_path;
		
		measurement_pattern = static_cast<TLaserDialog::TMeasurementPattern>(d_measurement_pattern_combo->currentData().toInt());
		
		switch(measurement_pattern)
		{
			case TLaserDialog::MEASUREMENT_DATA_COLLECTION:
				file_name.append(QString("/PMove Laser - "));
				break;
				
			case TLaserDialog::MEASUREMENT_10360_2:
				file_name.append(QString("/B89.4.10360 Laser - "));
				break;
		}
		
		file_name.append(d_description_edit->text());
		file_name.append(" - ");
		file_name.append(QDateTime::currentDateTime().toString(" [ddd MMMM d yyyy hh-mm-ss]"));
		file_name.append(QStringLiteral(".dat"));
		
		this->Add_Log_Text(QString("INF  Writing output file '%1'").arg(file_name));
				
		file.setFileName(file_name);
		
		if(file.open(QIODevice::WriteOnly | QIODevice::Text))
		{
			switch(measurement_pattern)
			{
				case TLaserDialog::MEASUREMENT_DATA_COLLECTION:
					file.write("PMove_Raw_Measurement:Version=1:Type=Machine_And_Laser\n\n");
					break;
					
				case TLaserDialog::MEASUREMENT_10360_2:
					file.write("B89.4.10360_Raw_Measurement:Version=1:Type=Machine_And_Laser\n\n");
					break;
			}

			vec = this->Get_Tool_Offsets();
			text = QString("Probe_Offset:%1,%2,%3\n").arg(vec.x,0,'f',4).arg(vec.y,0,'f',4).arg(vec.z,0,'f',4);
			file.write(text.toLatin1());
			
			vec = this->Get_Start_Point();
			text = QString("Start_Point:%1,%2,%3\n").arg(vec.x,0,'f',4).arg(vec.y,0,'f',4).arg(vec.z,0,'f',4);
			file.write(text.toLatin1());
			
			text = QString("Measurement_Axis:%1,%2,%3\n\n").arg(d_measurement_axis.i,0,'f',9).arg(d_measurement_axis.j,0,'f',9).arg(d_measurement_axis.k,0,'f',9);
			file.write(text.toLatin1());
			
			measurement_type = static_cast<TLaserDialog::TLaserMeasurementType>(d_laser_measurement_type_combo->currentData().toInt());
			
			if(measurement_pattern == TLaserDialog::MEASUREMENT_DATA_COLLECTION)
			{
				switch(measurement_type)
				{
					case TLaserDialog::LASER_SCALE:
						file.write("Measurement_Type:Scale\n\n");
						break;
						
					case TLaserDialog::LASER_STRAIGHTNESS_X:
						file.write("Measurement_Type:Straightness_In_X\n\n");
						break;
						
					case TLaserDialog::LASER_STRAIGHTNESS_Y:
						file.write("Measurement_Type:Straightness_In_Y\n\n");
						break;
						
					case TLaserDialog::LASER_STRAIGHTNESS_Z:
						file.write("Measurement_Type:Straightness_In_Z\n\n");
						break;
						
					case TLaserDialog::LASER_ANGULAR:
						file.write("Measurement_Type:Angular\n\n");
						break;
				}
			}
			else
			{
				file.write("Measurement_Type:Scale\n\n");
			}
			
			text = QStringLiteral("# Point:<direction/index>,<nominal_distance>,<actual_distance>,<laser_value>,<X,Y,Z>,<deviation>\n");
			text += QStringLiteral("# Value of <actual_distance> is deviation parallel to measurement axis\n");
			text += QStringLiteral("# Value of <deviation> is zero offset and slope compensated if measurement type is straightness\n\n");
			file.write(text.toLatin1());

			file.write("Measurement_Begin:\n\n");
			
			text = QString("Description:%1\n\n").arg(d_description_edit->text().trimmed());
			file.write(text.toLatin1());

			for(iter = d_measured_data.begin();iter != d_measured_data.end();++iter)
			{
				file.write(QString("Point:%1,").arg((*iter).index).toLatin1());
				file.write(QString("%1,").arg((*iter).nominal_target,0,'f',4).toLatin1());
				file.write(QString("%1,").arg((*iter).actual_target,0,'f',4).toLatin1());
				file.write(QString("%1,").arg((*iter).laser_value,0,'f',4).toLatin1());
				file.write(QString("%1,").arg((*iter).position.x,0,'f',4).toLatin1());
				file.write(QString("%1,").arg((*iter).position.y,0,'f',4).toLatin1());
				file.write(QString("%1,").arg((*iter).position.z,0,'f',4).toLatin1());
				file.write(QString("%1\n").arg((*iter).deviation,0,'f',4).toLatin1());
			}
			
			file.write("\nMeasurement_End:\n");

			file.close();
		}
	}
}

void TLaserDialog::Set_State(
	const TState						state)
{
	TLaserDialog::TMeasurementPattern	measurement_pattern;
	
	d_current_state = state;
	
	switch(state)
	{
		case TLaserDialog::STATE_STOPPED:
			
			d_start_end_points_widget->setEnabled(true);
			d_options_widget->setEnabled(true);
			d_laser_widget->setEnabled(false);
			
			d_tool_x_edit->setEnabled(true);
			d_tool_y_edit->setEnabled(true);
			d_tool_z_edit->setEnabled(true);
			d_set_offsets_button->setEnabled(true);

			d_play_button->setEnabled(true);
			d_stop_button->setEnabled(false);
			d_close_button->setEnabled(true);
			
			measurement_pattern = static_cast<TLaserDialog::TMeasurementPattern>(d_measurement_pattern_combo->currentData().toInt());
			
			switch(measurement_pattern)
			{
				case TLaserDialog::MEASUREMENT_DATA_COLLECTION:
					d_data_collection_mode_combo->setEnabled(true);
					d_increment_spin->setEnabled(true);
					break;
					
				case TLaserDialog::MEASUREMENT_10360_2:
					d_data_collection_mode_combo->setEnabled(false);
					d_increment_spin->setEnabled(false);
					break;
			}

			d_icon_label->setPixmap(QPixmap(":/icon128/laser_off_icon128.png"));

			break;
			
		case TLaserDialog::STATE_RUNNING:
			
			d_graph_widget->DeleteDataSets();
			
			d_start_end_points_widget->setEnabled(false);
			d_options_widget->setEnabled(false);
			d_laser_widget->setEnabled(true);
			
			d_tool_x_edit->setEnabled(false);
			d_tool_y_edit->setEnabled(false);
			d_tool_z_edit->setEnabled(false);
			d_set_offsets_button->setEnabled(false);

			d_play_button->setEnabled(false);
			d_stop_button->setEnabled(true);
			d_close_button->setEnabled(false);
			
			d_icon_label->setPixmap(QPixmap(":/icon128/laser_icon128.png"));

			break;
	}
}

void TLaserDialog::Update_Actual_Position(void)
{
	TVector3							position_sum;
	double								weight;
	double								weight_increment;
	double								weight_divisor;
	double								actual_length;
	int									count;
	int									cntr;
	
	count = d_filter_level_spin->value();
	
	if(d_filter_data.size() < static_cast<unsigned int>(count))
	{
		d_actual_position_edit->setText(QStringLiteral("-----"));
		return;
	}
	
	if(count < 1)
	{
		d_filter_position = d_filter_data[0];
	}
	else
	{
		position_sum.Set(0,0,0);
		weight = 0.0;
		weight_increment = 1.0;
		weight_divisor = static_cast<double>(MAX_FILTER_LEVEL + 1)/ static_cast<double>(MAX_FILTER_LEVEL);
		
		for(cntr = 0;cntr < count;++cntr)
		{
			position_sum += d_filter_data[cntr] * weight_increment;
			
			weight += weight_increment;
			weight_increment /= weight_divisor;
		}
		
		d_filter_position = position_sum / weight;
	}
	
	actual_length = (d_filter_position - d_start_point) ^ d_measurement_axis;

	d_actual_position_edit->setText(QString("%1").arg(actual_length,0,'f',4));
}

void TLaserDialog::Update_Measurement_Axis(void)
{
	TVector3							start;
	TVector3							end;
	TVector3							axis;
	double								dot_axis;
	bool								state(false);
	bool								prev_state;
	TLaserDialog::TLaserMeasurementType	type(TLaserDialog::LASER_SCALE);
	int									index;
	static const double					MINIMUM_LENGTH(10.0);
	static const double					STRAIGHTNESS_ALIGN_LIMIT(0.93979);	// must be within 20 deg of machine axis for straightness
	
	if(d_laser_measurement_type_combo->count())
	{
		type = static_cast<TLaserDialog::TLaserMeasurementType>(d_laser_measurement_type_combo->currentData().toInt());
	}
	
	d_laser_measurement_type_combo->clear();
	
	start = this->Get_Start_Point(&state);
	if(!state)
	{
		d_input_data_error = true;
		return;
	}
	
	end = this->Get_End_Point(&state);
	if(!state)
	{
		d_input_data_error = true;
		return;
	}
	
	axis = end - start;
	
	if(axis.Length() < MINIMUM_LENGTH)
	{
		d_input_data_error = true;
		return;
	}
	
	axis.Normal();
	d_measurement_axis = axis;
	d_input_data_error = false;
	
	d_laser_measurement_type_combo->addItem(QStringLiteral("Scale"),TLaserDialog::LASER_SCALE);
	
	dot_axis = fabs(d_measurement_axis ^ TVector3(1,0,0));
	
	if(dot_axis > STRAIGHTNESS_ALIGN_LIMIT)
	{
		d_laser_measurement_type_combo->addItem(QStringLiteral("Straightness in Y"),TLaserDialog::LASER_STRAIGHTNESS_Y);
		d_laser_measurement_type_combo->addItem(QStringLiteral("Straightness in Z"),TLaserDialog::LASER_STRAIGHTNESS_Z);
	}
	
	dot_axis = fabs(d_measurement_axis ^ TVector3(0,1,0));
	
	if(dot_axis > STRAIGHTNESS_ALIGN_LIMIT)
	{
		d_laser_measurement_type_combo->addItem(QStringLiteral("Straightness in X"),TLaserDialog::LASER_STRAIGHTNESS_X);
		d_laser_measurement_type_combo->addItem(QStringLiteral("Straightness in Z"),TLaserDialog::LASER_STRAIGHTNESS_Z);
	}
	
	dot_axis = fabs(d_measurement_axis ^ TVector3(0,0,1));
	
	if(dot_axis > STRAIGHTNESS_ALIGN_LIMIT)
	{
		d_laser_measurement_type_combo->addItem(QStringLiteral("Straightness in X"),TLaserDialog::LASER_STRAIGHTNESS_X);
		d_laser_measurement_type_combo->addItem(QStringLiteral("Straightness in Y"),TLaserDialog::LASER_STRAIGHTNESS_Y);
	}
	
	d_laser_measurement_type_combo->addItem(QStringLiteral("Angular"),TLaserDialog::LASER_ANGULAR);
	
	index = d_laser_measurement_type_combo->findData(type);
	
	if(!(index < 0))
	{
		prev_state = d_laser_measurement_type_combo->blockSignals(true);
		d_laser_measurement_type_combo->setCurrentIndex(index);
		d_laser_measurement_type_combo->blockSignals(prev_state);
	}
}

void TLaserDialog::Update_Map_Points(void)
{
	TVector3							start;
	TVector3							end;
	TVector3							tool_offset;
	bool								state;
	
	start = this->Get_Start_Point(&state);
	if(!state)
	{
		d_input_data_error = true;
		return;
	}
	
	end = this->Get_End_Point(&state);
	if(!state)
	{
		d_input_data_error = true;
		return;
	}

	tool_offset = this->Get_Tool_Offsets(&state);
	
	this->Set_Start_Map_Point(start - tool_offset);
	this->Set_End_Map_Point(end - tool_offset);
}

void TLaserDialog::Update_Measurement_Length(void)
{
	TVector3							start;
	TVector3							end;
	TVector3							vec;
	
	start = this->Get_Start_Point();
	end = this->Get_End_Point();
	
	vec = end - start;
	
	d_measurement_length_label->setText(QString("%1 mm").arg(vec.Length(),0,'f',3));
}

void TLaserDialog::Set_Start_Point(
	const TVector3						&xyz)
{
	bool								prev_state;
	
	prev_state = d_start_x_edit->blockSignals(true);
	d_start_x_edit->setText(QString("%1").arg(xyz.x,0,'f',4));
	d_start_x_edit->blockSignals(prev_state);
	
	prev_state = d_start_y_edit->blockSignals(true);
	d_start_y_edit->setText(QString("%1").arg(xyz.y,0,'f',4));
	d_start_y_edit->blockSignals(prev_state);

	prev_state = d_start_z_edit->blockSignals(true);
	d_start_z_edit->setText(QString("%1").arg(xyz.z,0,'f',4));
	d_start_z_edit->blockSignals(prev_state);
}

void TLaserDialog::Set_Start_Map_Point(
	const TVector3						&xyz)
{
	d_start_map_x_label->setText(QString("%1").arg(xyz.x,0,'f',4));
	d_start_map_y_label->setText(QString("%1").arg(xyz.y,0,'f',4));
	d_start_map_z_label->setText(QString("%1").arg(xyz.z,0,'f',4));
}

void TLaserDialog::Set_End_Point(
	const TVector3						&xyz)
{
	bool								prev_state;
	
	prev_state = d_end_x_edit->blockSignals(true);
	d_end_x_edit->setText(QString("%1").arg(xyz.x,0,'f',4));
	d_end_x_edit->blockSignals(prev_state);
	
	prev_state = d_end_y_edit->blockSignals(true);
	d_end_y_edit->setText(QString("%1").arg(xyz.y,0,'f',4));
	d_end_y_edit->blockSignals(prev_state);
	
	prev_state = d_end_z_edit->blockSignals(true);
	d_end_z_edit->setText(QString("%1").arg(xyz.z,0,'f',4));
	d_end_z_edit->blockSignals(prev_state);
}

void TLaserDialog::Set_End_Map_Point(
	const TVector3						&xyz)
{
	d_end_map_x_label->setText(QString("%1").arg(xyz.x,0,'f',4));
	d_end_map_y_label->setText(QString("%1").arg(xyz.y,0,'f',4));
	d_end_map_z_label->setText(QString("%1").arg(xyz.z,0,'f',4));
}

void TLaserDialog::Data_Collection_Next(
	const bool							stop_on_end)
{
	TDriver::TCommand					command;
	double								offset;
	TVector3							pnt;
	double								length;
	TLaserDialog::TMode					data_collection_mode;
	static const double					ZERO_EPSILON(0.1);
	
	offset = d_premove_offset_spin->value();
	
	pnt = d_end_point - d_start_point;
	length = pnt.Length() + ZERO_EPSILON;
	
	data_collection_mode = static_cast<TLaserDialog::TMode>(d_data_collection_mode_combo->currentData().toInt());
	
	command.command_type = TDriver::DRIVER_MOVE_TO;
	
	d_data_collection_target += d_increment_spin->value();
	
	if(d_data_collection_target > length)
	{
		if((stop_on_end == true) &&
		   (d_current_state == TLaserDialog::STATE_RUNNING) &&
		   (data_collection_mode == TLaserDialog::MODE_UNIDIRECTIONAL))
		{
			this->Add_Log_Text(QStringLiteral("Sequence Complete"));
			
			this->Stop();
			return;
		}
		
		d_data_collection_target -= d_increment_spin->value();
		
		pnt = d_start_point + (d_measurement_axis * (d_data_collection_target + offset));
		
		command.xyz = pnt;
		d_commands.push_back(command);
		this->Add_Log_Text(QString("INF  Moving to  %1, %2, %3  Position: %4")
						   .arg(command.xyz.x,0,'f',4)
						   .arg(command.xyz.y,0,'f',4)
						   .arg(command.xyz.z,0,'f',4)
						   .arg(d_data_collection_target,0,'f',4));
		
		this->Add_Log_Text(QStringLiteral("INF  Switching Direction"));

		if(d_direction == TLaserDialog::DIRECTION_FORWARD)
		{
			d_direction = TLaserDialog::DIRECTION_REVERSE;
			d_status_edit->setText(QStringLiteral("Reverse"));
		}
		else
		{
			d_direction = TLaserDialog::DIRECTION_FORWARD;
			d_status_edit->setText(QStringLiteral("Forward"));
		}
		
		
		command.xyz = d_start_point + (d_measurement_axis * d_data_collection_target);
		d_commands.push_back(command);
		this->Add_Log_Text(QString("INF  Moving to  %1, %2, %3  Position: %4   Waiting for laser data...")
						   .arg(command.xyz.x,0,'f',4)
						   .arg(command.xyz.y,0,'f',4)
						   .arg(command.xyz.z,0,'f',4)
						   .arg(d_data_collection_target,0,'f',4));
	}
	else
	{
		command.xyz = d_start_point + (d_measurement_axis * d_data_collection_target);
		d_commands.push_back(command);
		this->Add_Log_Text(QString("INF  Moving to  %1, %2, %3  Position: %4   Waiting for laser data...")
						   .arg(command.xyz.x,0,'f',4)
						   .arg(command.xyz.y,0,'f',4)
						   .arg(command.xyz.z,0,'f',4)
						   .arg(d_data_collection_target,0,'f',4));
	}
	
	d_target_position_edit->setText(QString("%1").arg(d_data_collection_target,0,'f',4));
	
	d_laser_value_edit->selectAll();
}

void TLaserDialog::Data_Collection_Previous(
	const bool							stop_on_end)
{
	TDriver::TCommand					command;
	double								offset;
	TVector3							pnt;
	TLaserDialog::TMode					data_collection_mode;
	static const double					ZERO_EPSILON(0.1);
	
	offset = d_premove_offset_spin->value();
	
	data_collection_mode = static_cast<TLaserDialog::TMode>(d_data_collection_mode_combo->currentData().toInt());
	
	command.command_type = TDriver::DRIVER_MOVE_TO;
	
	d_data_collection_target -= d_increment_spin->value();
	
	if(d_data_collection_target < (-ZERO_EPSILON))
	{
		if((stop_on_end == true) &&
		   (d_current_state == TLaserDialog::STATE_RUNNING) &&
		   (data_collection_mode == TLaserDialog::MODE_BIDIRECTIONAL))
		{
			this->Add_Log_Text(QStringLiteral("Sequence Complete"));
			
			this->Stop();
			return;
		}
		
		d_data_collection_target = -1.0 * offset;
		
		pnt = d_start_point + (d_measurement_axis * d_data_collection_target);
		
		command.xyz = pnt;
		d_commands.push_back(command);
		this->Add_Log_Text(QString("INF  Moving to  %1, %2, %3  Position: %4")
						   .arg(command.xyz.x,0,'f',4)
						   .arg(command.xyz.y,0,'f',4)
						   .arg(command.xyz.z,0,'f',4)
						   .arg(d_data_collection_target,0,'f',4));
		
		this->Add_Log_Text(QStringLiteral("INF  Switching Direction"));
		
		if(d_direction == TLaserDialog::DIRECTION_FORWARD)
		{
			d_direction = TLaserDialog::DIRECTION_REVERSE;
			d_status_edit->setText(QStringLiteral("Reverse"));
		}
		else
		{
			d_direction = TLaserDialog::DIRECTION_FORWARD;
			d_status_edit->setText(QStringLiteral("Forward"));
		}
		
		this->Add_Log_Text(QStringLiteral("INF  Switching Direction"));
		
		pnt = d_start_point;
		d_data_collection_target = 0.0;
		
		command.xyz = d_start_point;
		d_commands.push_back(command);
		this->Add_Log_Text(QString("INF  Moving to  %1, %2, %3  Position: %4   Waiting for laser data...")
						   .arg(command.xyz.x,0,'f',4)
						   .arg(command.xyz.y,0,'f',4)
						   .arg(command.xyz.z,0,'f',4)
						   .arg(d_data_collection_target,0,'f',4));
		
	}
	else
	{
		command.xyz = d_start_point + (d_measurement_axis * d_data_collection_target);
		d_commands.push_back(command);
		this->Add_Log_Text(QString("INF  Moving to  %1, %2, %3  Position: %4   Waiting for laser data...")
						   .arg(command.xyz.x,0,'f',4)
						   .arg(command.xyz.y,0,'f',4)
						   .arg(command.xyz.z,0,'f',4)
						   .arg(d_data_collection_target,0,'f',4));
	}
	
	d_target_position_edit->setText(QString("%1").arg(d_data_collection_target,0,'f',4));
	
	d_laser_value_edit->selectAll();
}

void TLaserDialog::Data_10360_Next(
	const bool							stop_on_end)
{
	TDriver::TCommand					command;
	double								offset;
	double								target;
	static const int					MAX_TARGET_INDEX(29);
	
	++d_10360_target_index;
	
	if(d_10360_target_index > MAX_TARGET_INDEX)
	{
		d_10360_target_index = MAX_TARGET_INDEX;
		
		if((stop_on_end == true) &&
		   (d_current_state == TLaserDialog::STATE_RUNNING))
		{
			this->Add_Log_Text(QStringLiteral("Sequence Complete"));
			
			this->Stop();
			return;
		}
	}

	
	offset = d_premove_offset_spin->value();
	target = d_10360_targets[d_10360_target_index];
	
	if((d_10360_target_index % 2) == 0 || d_10360_target_index == 0)
	{
		command.command_type = TDriver::DRIVER_MOVE_TO;
		command.xyz = d_start_point - (d_measurement_axis * offset);
		d_commands.push_back(command);
		this->Add_Log_Text(QString("INF  Moving to  %1, %2, %3  Position: %4")
						   .arg(command.xyz.x,0,'f',4)
						   .arg(command.xyz.y,0,'f',4)
						   .arg(command.xyz.z,0,'f',4)
						   .arg(-1.0 * offset,0,'f',4));

	}
	
	command.command_type = TDriver::DRIVER_MOVE_TO;
	command.xyz = d_start_point + (d_measurement_axis * target);
	d_commands.push_back(command);
	this->Add_Log_Text(QString("INF  Moving to  %1, %2, %3  Position: %4   Waiting for laser data...")
					   .arg(command.xyz.x,0,'f',4)
					   .arg(command.xyz.y,0,'f',4)
					   .arg(command.xyz.z,0,'f',4)
					   .arg(target,0,'f',4));
	
	d_status_edit->setText(QString("Target %1 of 30").arg(d_10360_target_index + 1));
	d_target_position_edit->setText(QString("%1").arg(target,0,'f',4));
	
	d_laser_value_edit->selectAll();
}

void TLaserDialog::Data_10360_Previous(
	const bool							stop_on_end)
{
	TDriver::TCommand					command;
	double								offset;
	double								target;
	
	--d_10360_target_index;

	if(d_10360_target_index < 0)
	{
		d_10360_target_index = 0;
		
		if((stop_on_end == true) &&
		   (d_current_state == TLaserDialog::STATE_RUNNING))
		{
			this->Add_Log_Text(QStringLiteral("Sequence Complete"));
			
			this->Stop();
			return;
		}
	}
	
	offset = d_premove_offset_spin->value();
	target = d_10360_targets[d_10360_target_index];
	
	if((d_10360_target_index % 2) == 0 || d_10360_target_index == 0)
	{
		command.command_type = TDriver::DRIVER_MOVE_TO;
		command.xyz = d_start_point - (d_measurement_axis * offset);
		d_commands.push_back(command);
		this->Add_Log_Text(QString("INF  Moving to  %1, %2, %3  Position: %4")
						   .arg(command.xyz.x,0,'f',4)
						   .arg(command.xyz.y,0,'f',4)
						   .arg(command.xyz.z,0,'f',4)
						   .arg(-1.0 * offset,0,'f',4));
		
	}
	
	command.command_type = TDriver::DRIVER_MOVE_TO;
	command.xyz = d_start_point + (d_measurement_axis * target);
	d_commands.push_back(command);
	this->Add_Log_Text(QString("INF  Moving to  %1, %2, %3  Position: %4   Waiting for laser data...")
					   .arg(command.xyz.x,0,'f',4)
					   .arg(command.xyz.y,0,'f',4)
					   .arg(command.xyz.z,0,'f',4)
					   .arg(target,0,'f',4));
	
	d_status_edit->setText(QString("Target %1 of 30").arg(d_10360_target_index + 1));
	d_target_position_edit->setText(QString("%1").arg(target,0,'f',4));
	
	d_laser_value_edit->selectAll();
}

void TLaserDialog::Add_Measurement(
	const TLaserDialog::TMeasuredData	&measured_data)
{
	std::vector<TLaserDialog::TMeasuredData>::iterator iter;
	static const double					TARGET_WINDOW(1.0);
	
	for(iter = d_measured_data.begin();iter != d_measured_data.end();++iter)
	{
		if(((*iter).index == measured_data.index) && (fabs((*iter).nominal_target - measured_data.nominal_target) < TARGET_WINDOW ))
		{
			(*iter) = measured_data;
			this->Update_Graph_Display();
			return;
		}
	}
	
	d_measured_data.push_back(measured_data);
	
	this->Update_Graph_Display();
}

void TLaserDialog::Update_Graph_Display(void)
{
	std::vector<TLaserDialog::TMeasuredData>::iterator iter;
	std::vector<TVector2>::iterator		pnt_iter;
	std::vector<TVector3>::iterator		position_iter;
	std::vector<TVector2>				graph_pnts;
	std::vector<TVector3>				position_pnts;
	TVector3							zero_position;
	TVector2							zero_pnt;
	TVector2							pnt;
	TLaserDialog::TMeasurementPattern	measurement_pattern;
	TLaserDialog::TLaserMeasurementType	laser_measurement_type(TLaserDialog::LASER_SCALE);
	double								slope;
	int									set;
	bool								init(true);
	
	if(!d_measured_data.size())
	{
		return;
	}
	
	measurement_pattern = static_cast<TLaserDialog::TMeasurementPattern>(d_measurement_pattern_combo->currentData().toInt());

	switch(measurement_pattern)
	{
		case TLaserDialog::MEASUREMENT_DATA_COLLECTION:
			laser_measurement_type = static_cast<TLaserDialog::TLaserMeasurementType>(d_laser_measurement_type_combo->currentData().toInt());
			break;
			
		case TLaserDialog::MEASUREMENT_10360_2:
			laser_measurement_type = TLaserDialog::LASER_SCALE;
			break;
	}
	
	for(iter = d_measured_data.begin();iter != d_measured_data.end();++iter)
	{
		if(init)
		{
			zero_position = (*iter).position;
			init = false;
		}
		
		pnt.x = (*iter).actual_target;
		pnt.y = (*iter).laser_value;
		
		graph_pnts.push_back(pnt);
		position_pnts.push_back((*iter).position - zero_position);
	}
	
	assert(graph_pnts.size());
	assert(position_pnts.size() == graph_pnts.size());
	assert(position_pnts.size() == d_measured_data.size());
	
	switch(laser_measurement_type)
	{
		case TLaserDialog::LASER_SCALE:
			
			// p.x is actual machine position along measurement line
			// p.y is data from laser
			// deviation is machine - laser
			
			for(pnt_iter = graph_pnts.begin();pnt_iter != graph_pnts.end();++pnt_iter)
			{
				(*pnt_iter).y = (*pnt_iter).x - (*pnt_iter).y;	
			}
			
			break;
			
		case TLaserDialog::LASER_STRAIGHTNESS_X:
			
			// p.y is laser straightness data
			// pos.x is actual machine position in X after zero offset.
			// deviation is machine - laser

			for(pnt_iter = graph_pnts.begin(),position_iter = position_pnts.begin();
				pnt_iter != graph_pnts.end();
				++pnt_iter,++position_iter)
			{
				(*pnt_iter).y = (*position_iter).x - (*pnt_iter).y;
			}
			
			// remove gradient
			if(this->Fit_Line(graph_pnts,&slope))
			{
				for(pnt_iter = graph_pnts.begin();pnt_iter != graph_pnts.end();++pnt_iter)
				{
					(*pnt_iter).y -= (slope * (*pnt_iter).x);
				}
			}

			break;
			
		case TLaserDialog::LASER_STRAIGHTNESS_Y:
			
			// p.y is laser straightness data
			// pos.y is actual machine position in Y after zero offset.
			// deviation is machine - laser

			for(pnt_iter = graph_pnts.begin(),position_iter = position_pnts.begin();
				pnt_iter != graph_pnts.end();
				++pnt_iter,++position_iter)
			{
				(*pnt_iter).y = (*position_iter).y - (*pnt_iter).y;
			}
			
			// remove gradient
			if(this->Fit_Line(graph_pnts,&slope))
			{
				for(pnt_iter = graph_pnts.begin();pnt_iter != graph_pnts.end();++pnt_iter)
				{
					(*pnt_iter).y -= (slope * (*pnt_iter).x);
				}
			}
			
			break;
			
		case TLaserDialog::LASER_STRAIGHTNESS_Z:
			
			// p.y is laser straightness data
			// pos.z is actual machine position in Z after zero offset.
			// deviation is machine - laser

			for(pnt_iter = graph_pnts.begin(),position_iter = position_pnts.begin();
				pnt_iter != graph_pnts.end();
				++pnt_iter,++position_iter)
			{
				(*pnt_iter).y = (*position_iter).z - (*pnt_iter).y;
			}
			
			// remove gradient
			if(this->Fit_Line(graph_pnts,&slope))
			{
				for(pnt_iter = graph_pnts.begin();pnt_iter != graph_pnts.end();++pnt_iter)
				{
					(*pnt_iter).y -= (slope * (*pnt_iter).x);
				}
			}
			
			break;
			
		case TLaserDialog::LASER_ANGULAR:
			break;
	}
	
	// zero offset points and write deviations
	zero_pnt = graph_pnts[0];
	for(pnt_iter = graph_pnts.begin(),iter = d_measured_data.begin();
		pnt_iter != graph_pnts.end();
		++pnt_iter,++iter)
	{
		(*pnt_iter).y -= zero_pnt.y;
		(*iter).deviation = (*pnt_iter).y;
	}
	
	d_graph_widget->DeleteDataSets();
	set = d_graph_widget->CreateDataSet(QColor(0,0,0));
	d_graph_widget->SetPoints(set,graph_pnts);
	
	d_graph_widget->UpdateGraph();
}

bool TLaserDialog::Fit_Line(
	const std::vector<TVector2>			&pnts,
	double 								*slope,
	double 								*yintercept)
{
	double								XY,X2,Y2,Xs,Ys;
	int									point_count;
	std::vector<TVector2>::const_iterator 	pnt_iter;
	
	point_count = static_cast<int>(pnts.size());
	
	if(point_count < 2)
	{
		return false;
	}
	
	assert(slope);
	
	XY = X2 = Y2 = Xs = Ys =0;
	
	for(pnt_iter = pnts.begin();pnt_iter != pnts.end();++pnt_iter)
	{
		XY += ((*pnt_iter).x * (*pnt_iter).y);
		X2 += ((*pnt_iter).x * (*pnt_iter).x);
		Y2 += ((*pnt_iter).y * (*pnt_iter).y);
		Xs += (*pnt_iter).x;
		Ys += (*pnt_iter).y;
	}
	
	*slope = static_cast<double>(point_count) * XY - Xs * Ys;
	*slope /= (static_cast<double>(point_count) * (X2) - Xs * Xs);
	
	if(yintercept)
	{
		*yintercept = Ys - (*slope) * Xs;
		*yintercept /= static_cast<double>(point_count);
	}
	
	return true;
}

void TLaserDialog::Clear_Log(void)
{
	d_information_text->clear();
}


