/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QFrame>
#include <QGridLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QLineEdit>
#include <QCheckBox>
#include <QRadioButton>
#include <QPushButton>
#include <QSpacerItem>
#include <QComboBox>
#include <assert.h>

#include "laseroffsetdialog.h"

static const double						MAX_OFFSET = 10.0;

TLaserOffsetDialog::TLaserOffsetDialog(
	const QWidget						*parent,
	Qt::WindowFlags						flags)
:QDialog(const_cast<QWidget*>(parent),flags)
{
	QGridLayout							*grid_layout;
	QLabel								*offset_x_label;
	QLabel								*offset_z_label;
	QLabel								*offset_y_label;
	QHBoxLayout							*button_hlayout;
	QSpacerItem							*button_hspacer;
	QPushButton							*cancel_button;
	QPushButton							*accept_button;
	QFrame								*separator_line;
	QSpacerItem							*control_vspacer;
	QLabel								*length_label;

	this->resize(400, 150);

	grid_layout = new QGridLayout(this);

	offset_x_label = new QLabel(this);
	offset_x_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	grid_layout->addWidget(offset_x_label, 0, 0, 1, 1);

	d_offset_x_edit = new QLineEdit(this);
	d_offset_x_edit->setAlignment(Qt::AlignCenter);
	grid_layout->addWidget(d_offset_x_edit, 0, 1, 1, 1);
	
	d_offset_position_radio = new QRadioButton(this);
	d_offset_position_radio->setChecked(true);
	grid_layout->addWidget(d_offset_position_radio, 0, 2, 1, 1);

	offset_y_label = new QLabel(this);
	offset_y_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	grid_layout->addWidget(offset_y_label, 1, 0, 1, 1);
	
	d_offset_y_edit = new QLineEdit(this);
	d_offset_y_edit->setAlignment(Qt::AlignCenter);
	grid_layout->addWidget(d_offset_y_edit, 1, 1, 1, 1);
	
	d_update_length_radio = new QRadioButton(this);
	grid_layout->addWidget(d_update_length_radio, 1, 2, 1, 1);

	offset_z_label = new QLabel(this);
	offset_z_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	grid_layout->addWidget(offset_z_label, 2, 0, 1, 1);

	d_offset_z_edit = new QLineEdit(this);
	d_offset_z_edit->setAlignment(Qt::AlignCenter);
	grid_layout->addWidget(d_offset_z_edit, 2, 1, 1, 1);
	
	d_update_option_combo = new QComboBox(this);
	grid_layout->addWidget(d_update_option_combo, 2, 2, 1, 1);
	
	length_label = new QLabel(this);
	length_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	grid_layout->addWidget(length_label, 3, 0, 1, 1);
	
	d_length_edit = new QLineEdit(this);
	d_length_edit->setAlignment(Qt::AlignCenter);
	grid_layout->addWidget(d_length_edit, 3, 1, 1, 1);
	
	d_automove_check = new QCheckBox(this);
	grid_layout->addWidget(d_automove_check, 3, 2, 1, 1);
	
	separator_line = new QFrame(this);
	separator_line->setFrameShape(QFrame::HLine);
	separator_line->setFrameShadow(QFrame::Sunken);
	grid_layout->addWidget(separator_line, 4, 0, 1, 3);
	
	control_vspacer = new QSpacerItem(20, 20, QSizePolicy::Minimum, QSizePolicy::Expanding);
	grid_layout->addItem(control_vspacer, 5, 0, 1, 3);

	button_hlayout = new QHBoxLayout();

	button_hspacer = new QSpacerItem(20, 20, QSizePolicy::Expanding, QSizePolicy::Minimum);
	button_hlayout->addItem(button_hspacer);

	cancel_button = new QPushButton(this);
	cancel_button->setAutoDefault(false);
	button_hlayout->addWidget(cancel_button);

	accept_button = new QPushButton(this);
	accept_button->setAutoDefault(false);
	button_hlayout->addWidget(accept_button);

	grid_layout->addLayout(button_hlayout, 6, 0, 1, 3);

	QWidget::setTabOrder(d_offset_x_edit, d_offset_y_edit);
	QWidget::setTabOrder(d_offset_y_edit, d_offset_z_edit);
	QWidget::setTabOrder(d_offset_z_edit, d_length_edit);
	QWidget::setTabOrder(d_length_edit, d_offset_position_radio);
	QWidget::setTabOrder(d_offset_position_radio, d_update_length_radio);
	QWidget::setTabOrder(d_update_length_radio, d_update_option_combo);
	QWidget::setTabOrder(d_update_option_combo, d_automove_check);
	QWidget::setTabOrder(d_automove_check, cancel_button);
	QWidget::setTabOrder(cancel_button, accept_button);

	this->setWindowTitle(QObject::tr("Offset Position"));
	
	offset_x_label->setText(QObject::tr("X Offset:"));
	offset_y_label->setText(QObject::tr("Y Offset:"));
	offset_z_label->setText(QObject::tr("Z Offset:"));
	length_label->setText(QObject::tr("Length:"));
	d_offset_position_radio->setText(QObject::tr("Offset XYZ"));
	d_update_length_radio->setText(QObject::tr("Adjust Length"));
	d_automove_check->setText(QObject::tr("Auto Move"));
	cancel_button->setText(QObject::tr("Cancel"));
	accept_button->setText(QObject::tr("Accept"));
	
	d_update_option_combo->addItem(QStringLiteral("From Start"));	// index 0
	d_update_option_combo->addItem(QStringLiteral("From Center"));	// index 1
	
	connect(cancel_button,SIGNAL(clicked(void)),this,SLOT(reject(void)));
	connect(accept_button,SIGNAL(clicked(void)),this,SLOT(accept(void)));
	connect(d_offset_position_radio,SIGNAL(toggled(bool)),this,SLOT(Offset_Position_Toggled(bool)));
}

TLaserOffsetDialog::~TLaserOffsetDialog(void)
{
}

bool TLaserOffsetDialog::Is_Offset_Position(void) const
{
	return d_offset_position_radio->isChecked();
}

bool TLaserOffsetDialog::Is_Offset_Length(void) const
{
	return d_update_length_radio->isChecked();
}

bool TLaserOffsetDialog::Is_Automove(void) const
{
	return d_automove_check->isChecked();
}

double TLaserOffsetDialog::X(void) const
{
	double								dval;
	
	dval = d_offset_x_edit->text().toDouble();
	
	// don't allow excessive offsets
	if(dval > MAX_OFFSET)
	{
		dval = MAX_OFFSET;
	}
	else if(dval < (-MAX_OFFSET))
	{
		dval = (-MAX_OFFSET);
	}
	
	return dval;
}

double TLaserOffsetDialog::Y(void) const
{
	double								dval;
	
	dval = d_offset_y_edit->text().toDouble();
	
	// don't allow excessive offsets
	if(dval > MAX_OFFSET)
	{
		dval = MAX_OFFSET;
	}
	else if(dval < (-MAX_OFFSET))
	{
		dval = (-MAX_OFFSET);
	}

	return dval;
}

double TLaserOffsetDialog::Z(void) const
{
	double								dval;
	
	dval = d_offset_z_edit->text().toDouble();
	
	// don't allow excessive offsets
	if(dval > MAX_OFFSET)
	{
		dval = MAX_OFFSET;
	}
	else if(dval < (-MAX_OFFSET))
	{
		dval = (-MAX_OFFSET);
	}

	return dval;
}

double TLaserOffsetDialog::Length(void) const
{
	double								dval;
	
	dval = d_length_edit->text().toDouble();
	
	return dval;
}

bool TLaserOffsetDialog::Offset_From_Start(void) const
{
	if(d_update_option_combo->currentIndex() == 0)
	{
		return true;
	}
	
	return false;
}

bool TLaserOffsetDialog::Offset_From_Center(void) const
{
	if(d_update_option_combo->currentIndex() == 1)
	{
		return true;
	}
	
	return false;
}

void TLaserOffsetDialog::Reset(
	const TLaserOffsetDialog::TOption		option,
	const double						&length)
{
	if(option == TLaserOffsetDialog::OFFSET_XYZ)
	{
		d_offset_position_radio->setChecked(true);
		d_update_length_radio->setChecked(false);
		
		d_offset_position_radio->setEnabled(false);
		d_update_length_radio->setEnabled(false);
		
		d_length_edit->setEnabled(false);
		d_update_option_combo->setEnabled(false);
		
		d_offset_x_edit->setEnabled(true);
		d_offset_y_edit->setEnabled(true);
		d_offset_z_edit->setEnabled(true);
	}
	else if(option == TLaserOffsetDialog::OFFSET_XYZ_OR_LENGTH)
	{
		d_offset_position_radio->setEnabled(true);
		d_update_length_radio->setEnabled(true);
		
		d_length_edit->setEnabled(!d_offset_position_radio->isChecked());
		d_update_option_combo->setEnabled(!d_offset_position_radio->isChecked());

		d_offset_x_edit->setEnabled(d_offset_position_radio->isChecked());
		d_offset_y_edit->setEnabled(d_offset_position_radio->isChecked());
		d_offset_z_edit->setEnabled(d_offset_position_radio->isChecked());

	}
	else
	{
		assert(false);
	}
	
	d_length_edit->setText(QString("%1").arg(length,0,'f',4));
	d_offset_x_edit->setText(QStringLiteral("0.0000"));
	d_offset_y_edit->setText(QStringLiteral("0.0000"));
	d_offset_z_edit->setText(QStringLiteral("0.0000"));
}

void TLaserOffsetDialog::Offset_Position_Toggled(
	bool								state)
{
	d_length_edit->setEnabled(!state);
	d_update_option_combo->setEnabled(!state);
	
	d_offset_x_edit->setEnabled(state);
	d_offset_y_edit->setEnabled(state);
	d_offset_z_edit->setEnabled(state);
}





