/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QAction>
#include <QApplication>
#include <QHBoxLayout>
#include <QLabel>
#include <QPushButton>
#include <QSpacerItem>
#include <QWhatsThis>
#include <QWidget>
#include <QComboBox>
#include <QProgressBar>
#include <QTimer>
#include <QScrollBar>
#include <QStringList>
#include <QMessageBox>
#include <QCloseEvent>
#include <assert.h>

#include "../../../core/optioneditor.h"

#include "linkdialog.h"

TLinkDialog::TLinkDialog(
	const QWidget						*parent,
	Qt::WindowFlags 					flags)
: QDialog(const_cast<QWidget*>(parent),flags)
{
	QVBoxLayout 						*dialog_vspacer;
	QHBoxLayout 						*bottom_button_hlayout;
	QHBoxLayout 						*logtitle_hlayout;
	QSpacerItem 						*bottom_button_hspacer;
	QSpacerItem 						*logtitle_hspacer;

	this->resize(QSize(600,300));
	
	dialog_vspacer = new QVBoxLayout(this);
	
	d_option_editor = new TOptionEditorWidget(this);
	d_option_editor->headerItem()->setText(0, QStringLiteral("Option"));
	d_option_editor->headerItem()->setText(1, QStringLiteral("Value"));
	dialog_vspacer->addWidget(d_option_editor);

	logtitle_hlayout = new QHBoxLayout();
	
	logtitle_hspacer = new QSpacerItem(20, 20, QSizePolicy::Expanding, QSizePolicy::Minimum);
	logtitle_hlayout->addItem(logtitle_hspacer);
	
	dialog_vspacer->addLayout(logtitle_hlayout);

	bottom_button_hlayout = new QHBoxLayout();
	
	bottom_button_hspacer = new QSpacerItem(20, 20, QSizePolicy::Expanding, QSizePolicy::Minimum);
	bottom_button_hlayout->addItem(bottom_button_hspacer);
	
	d_cancel_button = new QPushButton(this);
	d_cancel_button->setAutoDefault(false);
	bottom_button_hlayout->addWidget(d_cancel_button);
	
	d_accept_button = new QPushButton(this);
	d_accept_button->setAutoDefault(false);
	bottom_button_hlayout->addWidget(d_accept_button);
	
	dialog_vspacer->addLayout(bottom_button_hlayout);
	
	QWidget::setTabOrder(d_option_editor, d_cancel_button);
	QWidget::setTabOrder(d_cancel_button, d_accept_button);
	
	d_cancel_button->setText(QStringLiteral("Cancel"));
	d_accept_button->setText(QStringLiteral("Accept"));
	
	// default values
	d_controller_type = TController::CONTROLLER_VIRTUAL;

	d_connection_method = 0;	// serial
	d_serial_baudrate = TLibSerialDeviceEnum::SD_BAUD_9600;

	d_connection_defaults.default_leitz_address = QStringLiteral("80.128.0.2");
	d_connection_defaults.default_leitz_port = 2001;
	d_connection_defaults.default_leitz_commport = QStringLiteral("/dev/ttyUSB0");
	
#ifdef Q_OS_MAC
	d_connection_defaults.default_leitz_commport = QStringLiteral("/dev/ttys001");
#endif
#ifdef Q_OS_WIN
	d_connection_defaults.default_leitz_commport = QStringLiteral("COM1");
#endif
	
	d_serial_comport = d_connection_defaults.default_leitz_commport;

	d_connection_defaults.default_connection_method = 0;	// serial

	d_connection_defaults.default_dc_address = QStringLiteral("100.0.0.1");
	d_connection_defaults.default_dc_port = 1234;
	
	d_connection_defaults.default_ipp_address = QStringLiteral("127.0.0.1");
	d_connection_defaults.default_ipp_port = 1294;

	this->setWindowTitle(QStringLiteral("Communication Options"));
	
	connect(d_cancel_button,&QPushButton::clicked,this,&TLinkDialog::reject);
	connect(d_accept_button,&QPushButton::clicked,this,&TLinkDialog::Accept);
	connect(d_option_editor,&TOptionEditorWidget::Option_Combo_Item_Changed,this,&TLinkDialog::Option_Changed);
}

TLinkDialog::~TLinkDialog()
{
	
}

TLinkProtocol::TProtocolType TLinkDialog::Connection_Protocol(void) const
{
	QString								text;
	TLinkProtocol::TProtocolType		type(TLinkProtocol::PROTOCOL_NONE);
	
	text = d_option_editor->Option_Value(d_protocol_method_index);
	
	if(text.contains(QStringLiteral("Serial")))
	{
		type = TLinkProtocol::PROTOCOL_SERIAL;
	}
	else if(text.contains(QStringLiteral("Ethernet")))
	{
		type = TLinkProtocol::PROTOCOL_ETHERNET;
	}
	
	return type;
}

QString TLinkDialog::Comm_Port(void) const
{
	return d_option_editor->Option_Value(d_serial_comport_index);
}

TLibSerialDeviceEnum::TBaudRate TLinkDialog::Baud_Rate(void)
{
	return this->Convert_Baudrate_Option(d_option_editor->Option_Value(d_serial_baud_index));
}

QString TLinkDialog::Ethernet_Address(void) const
{
	return d_option_editor->Option_Value(d_ethernet_index);
}

int TLinkDialog::Ethernet_Port(void) const
{
	return d_option_editor->Option_Value(d_ethernet_portnumber_index).toInt();
}

void TLinkDialog::Set_Connection_Defaults(
	const TConnectionDefaults 			&connection_defaults)
{
	d_connection_defaults = connection_defaults;
}

void TLinkDialog::Reset(
	TController::TControllerType		controller_type)
{
	d_controller_type = controller_type;
	
	switch(d_controller_type)
	{
		case TController::CONTROLLER_DC:
			d_connection_method = 1;	// ethernet
			d_ethernet_address = d_connection_defaults.default_dc_address;
			d_ethernet_port = d_connection_defaults.default_dc_port;
			break;

		case TController::CONTROLLER_IPPCLIENT:
			d_connection_method = 1;	// ethernet
			d_ethernet_address = d_connection_defaults.default_ipp_address;
			d_ethernet_port = d_connection_defaults.default_ipp_port;
			break;
			
		case TController::CONTROLLER_LEITZ:
			d_connection_method = d_connection_defaults.default_connection_method;
			d_serial_comport = d_connection_defaults.default_leitz_commport;
			d_ethernet_address = d_connection_defaults.default_leitz_address;
			d_ethernet_port = d_connection_defaults.default_leitz_port;
			break;
			
		case TController::CONTROLLER_VIRTUAL:
			d_connection_method = 1;	// ethernet
			d_ethernet_address = d_connection_defaults.default_virtual_address;
			d_ethernet_port = d_connection_defaults.default_virtual_port;
			break;
	}
	
	this->Create_Options();
}

void TLinkDialog::Accept(void)
{
	TLinkProtocol::TProtocolType		type;
	
	this->setFocus();
	
	type = this->Connection_Protocol();
	
	switch(type)
	{
		case TLinkProtocol::PROTOCOL_SERIAL:
			d_connection_method = 0;
			break;
			
		case TLinkProtocol::PROTOCOL_ETHERNET:
			d_connection_method = 1;
			break;
			
		case TLinkProtocol::PROTOCOL_NONE:
			d_connection_method = 2;
			break;
	}
	
	d_serial_comport = d_option_editor->Option_Value(d_serial_comport_index);
	d_ethernet_address = d_option_editor->Option_Value(d_ethernet_index);
	d_ethernet_port = d_option_editor->Option_Value(d_ethernet_portnumber_index).toInt();
	
	switch(d_controller_type)
	{
		case TController::CONTROLLER_DC:
			d_connection_defaults.default_dc_address = d_ethernet_address;
			d_connection_defaults.default_dc_port = d_ethernet_port;
			break;
			
		case TController::CONTROLLER_IPPCLIENT:
			d_connection_defaults.default_ipp_address = d_ethernet_address;
			d_connection_defaults.default_ipp_port = d_ethernet_port;
			break;
			
		case TController::CONTROLLER_LEITZ:
			d_connection_defaults.default_connection_method = d_connection_method;
			d_connection_defaults.default_leitz_commport = d_serial_comport;
			d_connection_defaults.default_leitz_address = d_ethernet_address;
			d_connection_defaults.default_leitz_port = d_ethernet_port;
			break;
			
		case TController::CONTROLLER_VIRTUAL:
			d_connection_defaults.default_virtual_address = d_ethernet_address;
			d_connection_defaults.default_virtual_port = d_ethernet_port;
			break;

	}

	this->accept();
}

void TLinkDialog::Option_Changed(
	int									index,
	const QString						&text)
{
	int									protocol;
		
	if(index == d_protocol_method_index)
	{
		if(text.contains("Serial"))
		{
			protocol = 0;	// serial
		}
		else
		{
			protocol = 1;	// ethernet
		}

		if(protocol == 0)
		{
			if(d_serial_comport_index < 0 || d_serial_baud_index < 0)
			{
				d_option_editor->Enable_Option(d_ethernet_index,true);
				d_option_editor->Enable_Option(d_ethernet_portnumber_index,true);
			}
			else
			{
				d_option_editor->Enable_Option(d_serial_comport_index,true);
				d_option_editor->Enable_Option(d_serial_baud_index,true);
				
				d_option_editor->Enable_Option(d_ethernet_index,false);
				d_option_editor->Enable_Option(d_ethernet_portnumber_index,false);
			}
		}
		else
		{
			d_option_editor->Enable_Option(d_serial_comport_index,false);
			d_option_editor->Enable_Option(d_serial_baud_index,false);
			
			d_option_editor->Enable_Option(d_ethernet_index,true);
			d_option_editor->Enable_Option(d_ethernet_portnumber_index,true);
		}
	}
}


void TLinkDialog::Create_Options(void)
{
	QStringList							connection_methods;
	QStringList							serial_baud_rates;
	int									baud_index(0);
	
// clear and invalidate all options
	d_option_editor->clear();
	
	d_protocol_method_index = -1;
	d_serial_comport_index = -1;
	d_serial_baud_index = -1;
	d_ethernet_index = -1;
	d_ethernet_portnumber_index = -1;
	
	connection_methods.push_back(QStringLiteral("Serial"));
	connection_methods.push_back(QStringLiteral("Ethernet"));
	
	serial_baud_rates.push_back(QStringLiteral("1200 Baud"));
	serial_baud_rates.push_back(QStringLiteral("2400 Baud"));
	serial_baud_rates.push_back(QStringLiteral("4800 Baud"));
	serial_baud_rates.push_back(QStringLiteral("9600 Baud"));
	serial_baud_rates.push_back(QStringLiteral("19200 Baud"));
	serial_baud_rates.push_back(QStringLiteral("38400 Baud"));
	serial_baud_rates.push_back(QStringLiteral("57600 Baud"));
	serial_baud_rates.push_back(QStringLiteral("115200 Baud"));
	
	switch(d_serial_baudrate)
	{
		case TLibSerialDeviceEnum::SD_BAUD_1200:
			baud_index = 0;
			break;
		case TLibSerialDeviceEnum::SD_BAUD_2400:
			baud_index = 1;
			break;
		case TLibSerialDeviceEnum::SD_BAUD_4800:
			baud_index = 2;
			break;
		case TLibSerialDeviceEnum::SD_BAUD_9600:
			baud_index = 3;
			break;
		case TLibSerialDeviceEnum::SD_BAUD_19200:
			baud_index = 4;
			break;
		case TLibSerialDeviceEnum::SD_BAUD_38400:
			baud_index = 5;
			break;
		case TLibSerialDeviceEnum::SD_BAUD_57600:
			baud_index = 6;
			break;
		case TLibSerialDeviceEnum::SD_BAUD_115200:
			baud_index = 7;
			break;
			
		default:
			break;
	}
	
	if(!(d_connection_method < connection_methods.size()) ||
	   d_connection_method < 0)
	{
		d_connection_method = 0;
	}
	
	d_protocol_method_index = d_option_editor->Add_Combobox_Option(QStringLiteral("Communication Method"),connection_methods,d_connection_method);
	d_serial_comport_index = d_option_editor->Add_Line_Editor_Option(QStringLiteral("Serial Comm Port"),d_serial_comport);
	d_serial_baud_index = d_option_editor->Add_Combobox_Option(QStringLiteral("Baud Rate"),serial_baud_rates,baud_index);
	d_ethernet_index = d_option_editor->Add_Line_Editor_Option(QStringLiteral("Controller IP"),d_ethernet_address);
	d_ethernet_portnumber_index = d_option_editor->Add_Line_Editor_Option(QStringLiteral("Port Number"),QString("%1").arg(d_ethernet_port));
	
	d_option_editor->resizeColumnToContents(0);
	d_option_editor->resizeColumnToContents(1);
	
	if(d_connection_method == 0)
	{
		if(d_serial_comport_index < 0 || d_serial_baud_index < 0)
		{
			d_option_editor->Enable_Option(d_ethernet_index,true);
			d_option_editor->Enable_Option(d_ethernet_portnumber_index,true);
		}
		else
		{
			d_option_editor->Enable_Option(d_serial_comport_index,true);
			d_option_editor->Enable_Option(d_serial_baud_index,true);
			
			d_option_editor->Enable_Option(d_ethernet_index,false);
			d_option_editor->Enable_Option(d_ethernet_portnumber_index,false);
		}
	}
	else
	{
		d_option_editor->Enable_Option(d_serial_comport_index,false);
		d_option_editor->Enable_Option(d_serial_baud_index,false);
		
		d_option_editor->Enable_Option(d_ethernet_index,true);
		d_option_editor->Enable_Option(d_ethernet_portnumber_index,true);
	}

}

TLibSerialDeviceEnum::TBaudRate TLinkDialog::Convert_Baudrate_Option(
	const QString						&option_text)
{	
	if(option_text.contains(QStringLiteral("115200")))
	{
		return TLibSerialDeviceEnum::SD_BAUD_115200;
	}
	else if(option_text.contains(QStringLiteral("57600")))
	{
		return TLibSerialDeviceEnum::SD_BAUD_57600;
	}
	else if(option_text.contains(QStringLiteral("38400")))
	{
		return TLibSerialDeviceEnum::SD_BAUD_38400;
	}
	else if(option_text.contains(QStringLiteral("19200")))
	{
		return TLibSerialDeviceEnum::SD_BAUD_19200;
	}
	else if(option_text.contains(QStringLiteral("9600")))
	{
		return TLibSerialDeviceEnum::SD_BAUD_9600;
	}
	else if(option_text.contains(QStringLiteral("4800")))
	{
		return TLibSerialDeviceEnum::SD_BAUD_4800;
	}
	else if(option_text.contains(QStringLiteral("2400")))
	{
		return TLibSerialDeviceEnum::SD_BAUD_2400;
	}
	else if(option_text.contains(QStringLiteral("1200")))
	{
		return TLibSerialDeviceEnum::SD_BAUD_1200;
	}
	
	return TLibSerialDeviceEnum::SD_BAUD_9600;
}
