/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QFrame>
#include <QGridLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QLineEdit>
#include <QPushButton>
#include <QRadioButton>
#include <QSizePolicy>
#include <QSpacerItem>
#include <QToolButton>
#include <QFileDialog>
#include <QDir>
#include <QFile>

#include "measure_outputdialog.h"

TMeasureOutputDialog::TMeasureOutputDialog(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QDialog(const_cast<QWidget*>(parent),flags)
{
	QFrame								*lower_separator_hline;
	QFrame								*upper_separator_hline;
	QGridLayout							*dialog_layout;
	QHBoxLayout							*button_hlayout;
	QLabel								*application_path_label;
	QPushButton							*cancel_button;
	QSizePolicy							size_policy(QSizePolicy::Maximum,QSizePolicy::Preferred);
	QSpacerItem							*button_hspacer;
	QSpacerItem							*dialog_vspacer;

	size_policy.setHorizontalStretch(0);
	size_policy.setVerticalStretch(0);

	this->resize(701,229);

	dialog_layout = new QGridLayout(this);

	application_path_label = new QLabel(this);
	application_path_label->setSizePolicy(size_policy);
	dialog_layout->addWidget(application_path_label,0,0,1,1);

	d_application_path_label = new QLabel(this);
	dialog_layout->addWidget(d_application_path_label,0,1,1,3);

	upper_separator_hline = new QFrame(this);
	upper_separator_hline->setFrameShape(QFrame::HLine);
	upper_separator_hline->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(upper_separator_hline,1,0,1,4);

	d_relative_path_browse = new QToolButton(this);
	d_relative_path_browse->setToolTip(QStringLiteral("Relative path browse"));
	dialog_layout->addWidget(d_relative_path_browse,2,3,1,1);

	d_absolute_path_browse = new QToolButton(this);
	d_absolute_path_browse->setToolTip(QStringLiteral("Absolute path browse"));
	dialog_layout->addWidget(d_absolute_path_browse,3,3,1,1);

	lower_separator_hline = new QFrame(this);
	lower_separator_hline->setFrameShape(QFrame::HLine);
	lower_separator_hline->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(lower_separator_hline,4,0,1,4);

	dialog_vspacer = new QSpacerItem(0,0,QSizePolicy::Minimum,QSizePolicy::Expanding);
	dialog_layout->addItem(dialog_vspacer,5,0,1,4);

	button_hlayout = new QHBoxLayout();

	button_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	button_hlayout->addItem(button_hspacer);

	cancel_button = new QPushButton(this);
	cancel_button->setAutoDefault(false);
	button_hlayout->addWidget(cancel_button);

	d_apply_button = new QPushButton(this);
	d_apply_button->setAutoDefault(false);
	button_hlayout->addWidget(d_apply_button);
	dialog_layout->addLayout(button_hlayout,6,0,1,4);

	d_relative_path_check = new QRadioButton(this);
	d_relative_path_check->setToolTip(QStringLiteral("Use path relative to application executable"));
	d_relative_path_check->setChecked(true);
	dialog_layout->addWidget(d_relative_path_check,2,0,1,1);

	d_absolute_path_check = new QRadioButton(this);
	d_absolute_path_check->setToolTip(QStringLiteral("Use absolute path"));
	dialog_layout->addWidget(d_absolute_path_check,3,0,1,1);

	d_relative_path_edit = new QLineEdit(this);
	d_relative_path_edit->setToolTip(QStringLiteral("Relative path location"));
	dialog_layout->addWidget(d_relative_path_edit,2,1,1,2);

	d_absolute_path_edit = new QLineEdit(this);
	d_absolute_path_edit->setToolTip(QStringLiteral("Absolute path location"));
	dialog_layout->addWidget(d_absolute_path_edit,3,1,1,2);

	this->setWindowTitle(QStringLiteral("Output Settings"));
	
	application_path_label->setText(QStringLiteral("Application Path:"));
	d_relative_path_browse->setText(QStringLiteral("..."));
	d_absolute_path_browse->setText(QStringLiteral("..."));
	cancel_button->setText(QStringLiteral("Cancel"));
	d_apply_button->setText(QStringLiteral("Apply"));
	d_relative_path_check->setText(QStringLiteral("Relative Path:"));
	d_absolute_path_check->setText(QStringLiteral("Absolute Path:"));
	
	connect(d_absolute_path_check,&QRadioButton::toggled,this,&TMeasureOutputDialog::Path_Option_Changed);
	connect(d_relative_path_browse,&QToolButton::clicked,this,&TMeasureOutputDialog::Browse_Relative_Path);
	connect(d_absolute_path_browse,&QToolButton::clicked,this,&TMeasureOutputDialog::Browse_Absolute_Path);
	
	connect(cancel_button,&QPushButton::clicked,this,&TMeasureOutputDialog::reject);
	connect(d_apply_button,&QPushButton::clicked,this,&TMeasureOutputDialog::accept);

}

TMeasureOutputDialog::~TMeasureOutputDialog(void)
{
}

TReportWidget::TOutputSettings TMeasureOutputDialog::Settings(void) const
{
	TReportWidget::TOutputSettings	settings;
	
	settings.output_relative_path = QDir::fromNativeSeparators(d_relative_path_edit->text());
	settings.output_absolute_path = QDir::fromNativeSeparators(d_absolute_path_edit->text());
	settings.use_relative_path = d_relative_path_check->isChecked();
	
	return settings;
}

void TMeasureOutputDialog::Reset(
	const QString						&application_path,
	const TReportWidget::TOutputSettings &settings)
{
	d_application_path = application_path;
	
	d_application_path_label->setText(QDir::toNativeSeparators(d_application_path));
	d_relative_path_edit->setText(QDir::toNativeSeparators(settings.output_relative_path));
	d_absolute_path_edit->setText(QDir::toNativeSeparators(settings.output_absolute_path));
	
	if(settings.use_relative_path)
	{
		d_relative_path_check->setChecked(true);
	}
	else
	{
		d_absolute_path_check->setChecked(true);
	}
	
	d_relative_path_edit->setEnabled(d_relative_path_check->isChecked());
	d_relative_path_browse->setEnabled(d_relative_path_check->isChecked());
	
	d_absolute_path_edit->setEnabled(d_absolute_path_check->isChecked());
	d_absolute_path_browse->setEnabled(d_absolute_path_check->isChecked());
}

void TMeasureOutputDialog::Path_Option_Changed(void)
{
	
	d_relative_path_edit->setEnabled(d_relative_path_check->isChecked());
	d_relative_path_browse->setEnabled(d_relative_path_check->isChecked());
	
	d_absolute_path_edit->setEnabled(d_absolute_path_check->isChecked());
	d_absolute_path_browse->setEnabled(d_absolute_path_check->isChecked());
}

void TMeasureOutputDialog::Browse_Relative_Path(void)
{
	QString								file_name;
	QString								default_path;
	QString								text;
	QDir								dir;
	
	default_path = d_application_path;
	dir.setPath(default_path);
	
	text = d_relative_path_edit->text();
	
	if(text.startsWith('\\') || text.startsWith('/'))
	{
		text.remove(0,1);
	}
	
	default_path += QString("/%1").arg(text);
	
	file_name = QFileDialog::getExistingDirectory(this,QStringLiteral("Select measurement output folder"),default_path);
	
	if(file_name.length())
	{
		text = dir.relativeFilePath(file_name);
		d_relative_path_edit->setText(text);
	}
}

void TMeasureOutputDialog::Browse_Absolute_Path(void)
{
	QString								file_name;
	QString								default_path;
	
	default_path = d_absolute_path_edit->text();
	
	file_name = QFileDialog::getExistingDirectory(this,QStringLiteral("Select measurement output folder"),default_path);
	
	if(file_name.length())
	{
		d_absolute_path_edit->setText(file_name);
	}
}


