/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#ifndef MEASURESEQUENCEHEADERFILE
#define MEASURESEQUENCEHEADERFILE

#include <QObject>
#include <QFile>
#include <QString>
#include <vector>

#include "../../../core/referencesystem.h"

#include "toolwidget.h"
#include "settingswidget.h"

#include "command_align.h"
#include "command_feature_line.h"
#include "command_feature_plane.h"
#include "command_feature_point.h"
#include "command_feature_sphere.h"
#include "command_feature_circle.h"
#include "command_modal_mode.h"
#include "command_modal_prompt.h"
#include "command_modal_sync.h"
#include "command_move.h"
#include "command_move_cube.h"
#include "command_dimension.h"
#include "command_querytool.h"
#include "command_settool.h"
#include "command_sequence.h"
#include "toolwidget.h"

#include "measure_types.h"

class TMeasureRoutines : public QObject
{
Q_OBJECT
	
public:

struct TParameters
{
	QString								stepgauge_10360_position_name;
	QString								stepgauge_b89_position_name;
	QString								stepgauge_square_position_name;
	QString								gaugeblock_position_name;
	QString								stepgauge_serial_number;
	QString								gaugeblock_serial_number;
	QString								ballbar_position_name;
	QString								ballbar_10360_position_name;
	QString								ballbar_10360_serial_number;
	QString								point_repeat_description;
	QString								roll_tool1_name;
	QString								roll_tool2_name;
	
	double								axis_x_temperature;
	double								axis_y_temperature;
	double								axis_z_temperature;
	double								part_temperature;
	double								stepgauge_measurement_zero_position;
	double								stepgauge_measurement_b89_start;
	double								stepgauge_zero_step_spacing;
	double								stepgauge_nominal_block_spacing;
	double								stepgauge_b89_nominal_block_spacing;
	double								stepgauge_expansion_coeff;
	double								ballbar_expansion_coeff;
	double								ballbar_sphere_diameter;
	double								ballbar_10360_expansion_coeff;
	double								ballbar_10360_sphere_diameter;
	double								ballbar_10360_sphere_spacing;
	double								gaugeblock_expansion_coeff;
	double								gaugeblock_nominal_length;
	double								point_repeat_move_distance;
	double								sphere_nominal_diameter;
	double								sphere_stem_vector_i;
	double								sphere_stem_vector_j;
	double								sphere_stem_vector_k;
	double								ringgauge_nominal_diameter;
	double								pingauge_nominal_diameter;

	int									stepgauge_type;
	int									stepgauge_b89_measurement_count;
	int									stepgauge_square_measurement_count;
	int									point_repeat_measurement_count;
	int									sphere_repeat_measurement_count;
	int									ballbar_measurement_count;
	int									ballbar_10360_sphere0_offset;
	int									ringgauge_point_count;
	int									pingauge_point_count;
};
	
struct TFeatureMemory
{
	TVector3							xyz;
	TVector3							ijk;
	double								diameter;
	double								form;
	int									memory_id;
};
	
// CREATORS
	TMeasureRoutines(void);
	~TMeasureRoutines(void);
	
// ACCESSORS
	TParameters Measurement_Parameters(void) const {return d_parameters;}
	std::vector<TToolWidget::TToolItem> Calibration_Tool_Items(void) const {return d_calibration_tool_items;}
	QString Calibration_Tool_Name(void) const {return d_calibration_tool_name;}
	
	MeasureTypes::TSequenceFunction Current_Sequence_Function(void) const {return d_current_sequence_function;}
	
	double Tip_Radius(void) const {return d_active_tool.tip_diameter / 2.0;}
	TVector3 Tool_Vector(void) const {return d_active_tool.ijk;}
	
	double Approach_Distance(void) const {return d_approach_distance;}
	
	TFeatureMemory Feature_Memory(const int memory_id,bool * const valid) const;
	
	bool Manual_Mode(void) const {return d_manual_mode;}
	
	bool Is_Sequence_Complete(void) const {return d_command_sequence.Is_Sequence_Complete();}
	TCommand * Get_Next_Command(bool *command_valid) const {return d_command_sequence.Get_Next_Command(command_valid);}
	
	TVector3 Refsys_ToWorld(const TVector3 &pt) const {return d_refsys.ToWorld(pt);}
	TVector3 Refsys_FromWorld(const TVector3 &pt) const {return d_refsys.FromWorld(pt);}
	TVector3 Refsys_AxisToWorld(const TVector3 &vec) const {return d_refsys.AxisToWorld(vec);}
	TVector3 Refsys_AxisFromWorld(const TVector3 &vec) const {return d_refsys.AxisFromWorld(vec);}
	TVector3 Refsys_Origin(void) const {return d_refsys.Origin();}
	
	TVector3 Refsys_Axis_X(void) const {return d_refsys.Rotation_Mat().X();}
	TVector3 Refsys_Axis_Y(void) const {return d_refsys.Rotation_Mat().Y();}
	TVector3 Refsys_Axis_Z(void) const {return d_refsys.Rotation_Mat().Z();}
	
// MANIPULATORS
	
	void Clear_Tool_Data(void) {d_tool_items.clear();}
	void Delete_Sequence_Commands(void) {d_command_sequence.Delete_Commands();}
	
	void Set_Measurement_Parameters(const TParameters &data) {d_parameters = data;}
	void Set_Tool_Data(const TToolWidget::TToolItem &tool_data);
	bool Set_Active_Tool_Name(const QString &name);
	void Set_Calibration_Tool_Data(const QString &name,const std::vector<TToolWidget::TToolItem> &data);
	
	void Set_Feature_Memory(const TFeatureMemory &fm);
	void Remove_Feature_Memory(const int memory_id);

	void Set_Current_Sequence_Function(const MeasureTypes::TSequenceFunction &function) {d_current_sequence_function = function;}
	void Set_Configuration(const MeasureTypes::TConfiguration configuration) {d_configuration = configuration;}

	void Set_Output_Path(const QString &path) {d_output_path = path;}
	void Set_Axis_Expansion_Coefficients(const double &x,const double &y,const double &z);
	void Set_Artifact_Settings(const TSettingsWidget::TArtifactSettings &settings) {d_artifact_settings = settings;}
	
	void Reset_Current_Sequence_Function(void) {d_current_sequence_function = MeasureTypes::SEQUENCE_NULL;}
	
	void Generate_Sequence(void);
	void Start_Sequence(void);
	void Increment_Current_Sequence_Step(const MeasureTypes::TAlignmentOptions alignment_option);		// update's d_current_sequence_function to next step

	void Close_Output_File(void);	
	void Set_Approach_Distance(const double &distance) {d_approach_distance = distance;}
	
	void Set_Sphere_Stem_Vector(const TVector3 &stem) {d_stem_vector_sphere = stem;}
	
	void Set_Manual_Mode(const bool state){d_manual_mode = state;}
	
	void Refsys_Skew1(const TVector3 &axis,const TReferenceSystem::TAxis axis_name) {d_refsys.Skew1(axis,axis_name);}
	void Refsys_Skew2(const TVector3 &axis,const TReferenceSystem::TAxis axis_name,const TReferenceSystem::TAxis fixed_axis_name) {d_refsys.Skew2(axis,axis_name,fixed_axis_name);}
	void Refsys_Set_Origin(const TVector3 &pnt) {d_refsys.Set_Origin(pnt);}
	void Refsys_Set_Origin(const TVector3 &pnt,const TReferenceSystem::TAxis axis_name) {d_refsys.Set_Origin(pnt,axis_name);}
	void Refsys_Offset_Origin(const double &offset,const TReferenceSystem::TAxis axis_name) {d_refsys.Offset_Origin(offset,axis_name);}
	void Refsys_Rotate(const double &angle,const TReferenceSystem::TAxis fixed_axis_name) {d_refsys.Rotate(angle,fixed_axis_name);}
	void Refsys_Clear_Alignment(void) {d_refsys.Clear_Alignment();}
	
	void Process_Command(const MeasureTypes::TDisplayCoordinateSystem display_coordinate_system,TCommand * const command);
	
signals:
	void Add_Log_Text(const QString&);
	void Update_Last_Configuration(void);
	
private:
	
	TCommandSequence					d_command_sequence;
	std::vector<TFeatureMemory>			d_feature_memory;
	TParameters							d_parameters;
	TSettingsWidget::TArtifactSettings	d_artifact_settings;

	TReferenceSystem					d_refsys;
	TVector3							d_stepgauge_line_approach_vector;
	TVector3							d_stepgauge_manual_points[6];
	
	TVector3							d_stem_vector_sphere;

	QFile								d_output_file;
	QString								d_output_path;
	QString								d_calibration_tool_name;
	TToolWidget::TToolItem				d_active_tool;

	MeasureTypes::TSequenceFunction		d_current_sequence_function;
	MeasureTypes::TConfiguration		d_configuration;
	
	bool								d_manual_mode;
	int									d_active_roll_sphere;
	double								d_ringgauge_diameter_error;
	double								d_pingauge_diameter_error;
	double								d_approach_distance;
	
	double								d_expansion_coeff_x;
	double								d_expansion_coeff_y;
	double								d_expansion_coeff_z;
	
	std::vector<TToolWidget::TToolItem> d_calibration_tool_items;
	std::vector<TToolWidget::TToolItem>	d_tool_items;
	
	bool Tool_Item_Exists(const QString &name) const;
	TToolWidget::TToolItem Get_Tool_Item(const QString &name,bool * const valid) const;
	
	bool Initialize_Roll(void);
	bool Change_Tool_Roll(void);
	bool Move_Clear_Roll(void);

	bool Align_Stepgauge_Manual(void);
	bool Align_Gaugeblock_Manual(void);
	bool Align_Point_Manual(void);
	bool Align_Sphere_Manual(void);
	bool Align_Sphere_Manual_Calibration(void);
	bool Align_Ballbar_B89_Manual(void);
	bool Align_Ballbar_10360_Manual(void);
	bool Align_Ringgauge_Manual(void);
	bool Align_Pingauge_Manual(void);
	
	bool Align_Stepgauge_DCC(void);
	bool Align_Gaugeblock_DCC(void);
	bool Align_Point_DCC(void);
	bool Align_Sphere_DCC(void);
	bool Align_Ballbar_B89_DCC(void);
	bool Align_Ballbar_10360_DCC(void);
	bool Align_Ringgauge_DCC(void);
	bool Align_Pingauge_DCC(void);
	
	bool Measure_Stepgauge_10360(void);
	bool Measure_Stepgauge_B89(void);
	bool Measure_Stepgauge_Square(void);
	bool Measure_Gaugeblock(void);
	bool Measure_Point_Repeat(void);
	bool Measure_Sphere_Rpt(void);
	bool Measure_Sphere_Pftu(void);
	bool Measure_Ballbar_B89(void);
	bool Measure_Ballbar_10360(void);
	bool Measure_Ringgauge(void);
	bool Measure_Pingauge(void);
	bool Measure_Roll_Sphere(void);

	bool Calibrate_Tools(void);
	
	bool Report_Stepgauge_10360(void);
	bool Report_Stepgauge_B89(void);
	bool Report_Stepgauge_Square(void);
	bool Report_Gaugeblock(void);
	bool Report_Point_Repeat(void);
	bool Report_Sphere_Rpt(void);
	bool Report_Sphere_Pftu(void);
	bool Report_Ballbar(void);
	bool Report_Ballbar_10360(void);
	bool Report_Ringgauge(void);
	bool Report_Pingauge(void);
	bool Report_Roll(void);
	bool Report_Tool(void);

	QString Generate_Output_File_Name(void) const;
	void Initialize_Output_File(void);

	static QString Axis_Name(const TReferenceSystem::TAxis axis);
	static bool Vector_Different(const TVector3 &v1,const TVector3 &v2);
	
// NOT IMPLEMENTED
	TMeasureRoutines(const TMeasureRoutines&);
	TMeasureRoutines& operator=(const TMeasureRoutines&);
};


#endif
