/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#ifndef MEASUREDIALOGHEADERFILE
#define MEASUREDIALOGHEADERFILE

#include <QDialog>
#include <QFile>
#include <QPixmap>
#include <vector>

#include "../../../core/referencesystem.h"

#include "driver.h"
#include "controller.h"
#include "settingswidget.h"
#include "toolwidget.h"
#include "measurewidget.h"
#include "measure_routines.h"
#include "command_sequence.h"
#include "measure_types.h"

class QCheckBox;
class QLineEdit;
class QPlainTextEdit;
class QToolButton;
class QPushButton;
class QLabel;
class QComboBox;
class QCloseEvent;

class TInputTextWidget;
class TInputOptionWidget;
class TInputWidget;
class TCommandFeature;
class TPointCountWidget;
class TCoordinateDisplayWidget;
class TTempcompInformationDialog;
class TMessageBox;

class TMeasureDialog : public QDialog
{
Q_OBJECT

public:
	
enum TState
{
	STATE_STOPPED = 0,
	STATE_PAUSED,
	STATE_RUNNING
};
	
enum TThermalCompensationMode
{
	SOFTWARE_COMPENSATION_AXIS_AND_PART = 0,
	CONTROLLER_AXIS_AND_PART,
	CONTROLLER_AXIS_SOFTWARE_PART
};

// CREATORS
	TMeasureDialog(const QWidget *parent=0, Qt::WindowFlags flags=Qt::WindowFlags());
	virtual ~TMeasureDialog(void);

// ACCESSORS
	TMeasureRoutines::TParameters Parameters(void) const {return d_measure_routines.Measurement_Parameters();}
	bool Temperature_Compensation_State(void) const {return d_enable_temperature_compensation;}
	
	double Approach_Distance(void) const {return d_measure_routines.Approach_Distance();}
	
	std::vector<TDriver::TCommand> Get_Next_Driver_Commands(void);

// MANIPULATORS
	void Driver_Error(const QString &error,const int severity);
	void Sequence_Canceled(void);
	
	void Done_Key_Pressed(void);
	void Erase_Key_Pressed(void);
	
	void Enable_Temperature_Compensation(const bool state);
	void Driver_Sync(void) {d_waiting_driver_sync = false;}
	
	void Set_Measurement_Parameters(const TMeasureRoutines::TParameters &data) {d_measure_routines.Set_Measurement_Parameters(data);}
	void Set_Tool_Data(TToolWidget::TToolItem &tool_data) {d_measure_routines.Set_Tool_Data(tool_data);}
	void Set_Calibration_Tool_Data(const QString &name,const std::vector<TToolWidget::TToolItem> &data);
	void Clear_Tool_Data(void) {d_measure_routines.Clear_Tool_Data();}

	void Reset(const TController::TControllerType controller_type,const MeasureTypes::TConfiguration configuration,const QString &output_path,const QStringList &tool_list,const QString &active_tool);
	void Reset_Tool_Selection(void);
	
	void Set_Artifact_Settings(const TSettingsWidget::TArtifactSettings &artifact_settings);
	void Set_Machine_Settings(const TSettingsWidget::TMachineSettings &machine_settings);
	void Set_Thermal_Settings(const TThermalCompensationMode mode);
	
	bool Add_Measured_Point(const TVector3 &xyz,const TVector3 &ijk);
	void Update_Position(const TVector3 &xyz);
	
	void Set_Approach_Distance(const double &distance) {d_measure_routines.Set_Approach_Distance(distance);}
	
	void Set_Temperature_X(const double &value);
	void Set_Temperature_Y(const double &value);
	void Set_Temperature_Z(const double &value);
	void Set_Temperature_Part(const double &value);

signals:
	void Refresh_Temperatures(void);
	void Update_Part_Expansion_Coefficient(const double&);
	void Sequence_Start(void);
	void Sequence_End(void);
	void Pause_Driver(void);
	void Resume_Driver(void);
	void Stop_Driver(void);
	void Update_Tip_Diameter(const QString&,const double&, const double&, const double&);	// tool_name, angle_a, angle_b, tip_diameter
	void Change_Tool(const QString &name);

private slots:
	void Start(void);
	void Pause(bool);
	void Stop(void);
	void Erase(void);
	void Next(void);
	void Close(void);	
	void Temperature_Compensation_Toggled(bool);
	void Input_Left_Option_Changed(int,int);	// option_index,option data
	void Input_Right_Option_Changed(int,int);	// option_index,option data
	void Tool_Selection_Changed(int);
	
	void Add_Log_Text(const QString&);
	void Update_Last_Configuration(void);
	
protected:
	virtual void closeEvent(QCloseEvent *event);

private:

	TMeasureRoutines					d_measure_routines;
	TCommandFeature						*d_active_feature;

	QLabel								*d_alignment_mode_label;
	QComboBox							*d_active_tool_combo;
	QComboBox							*d_alignment_options_combo;
	QCheckBox							*d_temperature_compensation_check;
	QLabel								*d_icon_label;
	QLineEdit							*d_path_edit;
	QPlainTextEdit						*d_information_text;
	QToolButton							*d_stop_button;
	QToolButton							*d_close_button;
	QToolButton							*d_pause_button;
	QToolButton							*d_play_button;
	QPushButton							*d_erase_button;
	QPushButton							*d_next_button;
	TInputWidget						*d_input_widget_left;
	TInputWidget						*d_input_widget_right;
	TPointCountWidget					*d_point_count_widget;
	TCoordinateDisplayWidget			*d_coordinate_display_widget;
	TTempcompInformationDialog			*d_tempcomp_information_dialog;
	TMessageBox							*d_message_box;
	
	QPixmap								d_configuration_pixmap;
	QPixmap								d_saved_pixmap;
	
	MeasureTypes::TConfiguration		d_configuration;
	MeasureTypes::TConfiguration		d_last_configuration;
	MeasureTypes::TStepgaugeType		d_last_stepgauge_type;
	
	TState								d_current_state;
	TController::TControllerType 		d_controller_type;
	TSettingsWidget::TMachineSettings	d_machine_settings;
	TThermalCompensationMode			d_thermal_compensation_mode;
	
	bool								d_enable_temperature_compensation;
	bool								d_waiting_driver_sync;
	
	MeasureTypes::TDisplayCoordinateSystem d_display_coordinate_system;
	
	void Set_State(const TState);
	void Set_Pixmap(const QPixmap &pixmap,const bool save_current);
	void Restore_Pixmap(void);
	
	TVector3 Add_Temperature_Compensation(const TVector3 &pnt) const;
	TVector3 Remove_Temperature_Compensation(const TVector3 &pnt) const;
	TVector3 Temperature_Compensation_Correction(const TVector3 &pnt) const;
	void Set_Alignment_Option(const MeasureTypes::TAlignmentOptions option);
	void Update_Active_Feature(void);
	
	void Write_User_Inputs(TMeasureRoutines::TParameters * const data);
	void Clear_Log(void);


// NOT IMPLEMENTED
	TMeasureDialog(const TMeasureDialog&);
	TMeasureDialog& operator=(const TMeasureDialog&);
};

#endif
