/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QGridLayout>
#include <QScrollArea>
#include <QWidget>
#include <QVBoxLayout>
#include <QSpacerItem>
#include <QCloseEvent>
#include <QScreen>
#include <QRect>
#include <QSize>
#include <QPoint>
#include <QTimer>
#include <QSettings>
#include <QDir>

#include "../../../core/messagebox.h"

#include "logowidget.h"
#include "cmmwidget.h"
#include "settingswidget.h"
#include "toolwidget.h"
#include "measurewidget.h"
#include "reportwidget.h"
#include "linkdialog.h"
#include "driver.h"
#include "measuredialog.h"
#include "laserdialog.h"
#include "exercisedialog.h"
#include "measure_routines.h"

#include "measuredirect.h"

TMeasureDirect::TMeasureDirect(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QMainWindow(const_cast<QWidget*>(parent),flags)
{
	QGridLayout							*main_central_layout;
	QScrollArea							*scroll_area;
	QWidget								*main_central_widget;
	QWidget								*scroll_central_widget;
	QVBoxLayout							*scroll_central_layout;
	QSpacerItem							*control_vspacer;
	QDir								data_path;
	QRect								desktop_rect;
	QRect								window_rect;
	QPoint								position;
	QSize								size;
	QString								text;
	bool								bval;
	TSettingsWidget::TMachineSettings	machine_settings;
	TSettingsWidget::TArtifactSettings	artifact_settings;
	TCMMWidget::TTemperatureSettings 	temperature_settings;
	TReportWidget::TOutputSettings		output_settings;
	TMeasureRoutines::TParameters		measure_parameters;
	TLaserDialog::TParameters			laser_parameters;
	TLinkDialog::TConnectionDefaults	connection_defaults;
	
	d_application_path = qApp->applicationDirPath();
	
#ifdef Q_OS_MAC
	int index = d_application_path.lastIndexOf(QStringLiteral(".app/"));
	d_application_path.truncate(index);
	
	index = d_application_path.lastIndexOf('/');
	d_application_path.truncate(index);
#endif
	
#ifdef Q_OS_WIN
	d_data_path = d_application_path;
#else
	d_data_path = QDir::homePath();
#endif
		
	data_path.setPath(d_data_path);

	if(!data_path.exists(QStringLiteral(".measuredirect")))
	{
		data_path.mkdir(QStringLiteral(".measuredirect"));
	}
	
	data_path.cd(QStringLiteral(".measuredirect"));
	d_data_path = data_path.absolutePath();
	
	d_settings = new QSettings(d_data_path + QStringLiteral("/measuredirect_settings.ini"),QSettings::IniFormat,this);

	position = d_settings->value(QStringLiteral("Mainwindow_Position"), QPoint(30,30)).toPoint();
	size = d_settings->value(QStringLiteral("Mainwindow_Size"), QSize(951, 901)).toSize();

	desktop_rect = QGuiApplication::primaryScreen()->availableGeometry();
	
	window_rect.moveTopLeft(position);
	window_rect.setSize(size);
	
	window_rect = window_rect & desktop_rect;
	
	if(window_rect.isValid() && window_rect.width() > 500 && window_rect.height() > 500)
	{
		this->resize(window_rect.size());
		this->move(window_rect.topLeft());
	}
	else
	{
		this->resize(951, 901);
		this->move(QPoint(30,30));
	}

	main_central_widget = new QWidget(this);
	this->setCentralWidget(main_central_widget);

	main_central_layout = new QGridLayout(main_central_widget);

	scroll_area = new QScrollArea(main_central_widget);
	scroll_area->setWidgetResizable(true);
	main_central_layout->addWidget(scroll_area,0,0,1,1);

	scroll_central_widget = new QWidget();
	scroll_area->setWidget(scroll_central_widget);
	
	scroll_central_layout = new QVBoxLayout(scroll_central_widget);
	scroll_central_layout->setSpacing(4);

	d_logo_widget = new TLogoWidget(scroll_central_widget);
	scroll_central_layout->addWidget(d_logo_widget);

	d_cmm_widget = new TCMMWidget(scroll_central_widget);
	scroll_central_layout->addWidget(d_cmm_widget);
	
	d_settings_widget = new TSettingsWidget(scroll_central_widget);
	scroll_central_layout->addWidget(d_settings_widget);
	
	d_tool_widget = new TToolWidget(scroll_central_widget);
	scroll_central_layout->addWidget(d_tool_widget);
	
	d_measure_widget = new TMeasureWidget(scroll_central_widget);
	scroll_central_layout->addWidget(d_measure_widget);
	
	d_report_widget = new TReportWidget(scroll_central_widget);
	scroll_central_layout->addWidget(d_report_widget);

	control_vspacer = new QSpacerItem(0,0,QSizePolicy::Minimum,QSizePolicy::Expanding);
	scroll_central_layout->addItem(control_vspacer);

	d_measure_dialog = new TMeasureDialog(this);
	d_laser_dialog = new TLaserDialog(this);
	d_exercise_dialog = new TExerciseDialog(this);
	d_link_dialog = new TLinkDialog(this);
	d_message_box = new TMessageBox(this);
	
	d_process_timer = new QTimer(this);
	d_process_timer->setInterval(250);
	d_process_timer->setSingleShot(false);
	
	// init
	d_driver = 0;
	d_close_on_thread_finish = false;

	d_settings_widget->setEnabled(false);
	d_tool_widget->setEnabled(false);
	d_measure_widget->setEnabled(false);
	d_report_widget->Enable_Options(true);
	
	// defaults
	d_cmm_widget->Set_Controller_Type(static_cast<TController::TControllerType>(d_settings->value(QStringLiteral("Connect_Controller_Type"),0).toInt()));
	
	connection_defaults.default_leitz_address = d_settings->value(QStringLiteral("Connection_Default_Leitz_Address"),QStringLiteral("80.128.0.2")).toString();
	connection_defaults.default_dc_address = d_settings->value(QStringLiteral("Connection_Default_DC_Address"),QStringLiteral("100.0.0.1")).toString();
	connection_defaults.default_virtual_address = d_settings->value(QStringLiteral("Connection_Default_Virtual_Address"),QStringLiteral("127.0.0.1")).toString();
	connection_defaults.default_ipp_address = d_settings->value(QStringLiteral("Connection_Default_IPP_Address"),QStringLiteral("127.0.0.1")).toString();

	text = QStringLiteral("/dev/ttyUSB0");

#ifdef Q_OS_MAC
	text = QStringLiteral("/dev/ttys001");
#endif
#ifdef Q_OS_WIN
	text = QStringLiteral("COM1");
#endif

	connection_defaults.default_connection_method = d_settings->value(QStringLiteral("Connection_Default_Method"),0).toInt();

	connection_defaults.default_leitz_commport = d_settings->value(QStringLiteral("Connection_Default_Leitz_Serial"),text).toString();
	connection_defaults.default_leitz_port = d_settings->value(QStringLiteral("Connection_Default_Leitz_Port"),2001).toInt();
	connection_defaults.default_dc_port = d_settings->value(QStringLiteral("Connection_Default_DC_Port"),1234).toInt();
	connection_defaults.default_virtual_port = d_settings->value(QStringLiteral("Connection_Default_Virtual_Port"),2001).toInt();
	connection_defaults.default_ipp_port = d_settings->value(QStringLiteral("Connection_Default_IPP_Port"),1294).toInt();

	d_link_dialog->Set_Connection_Defaults(connection_defaults);

	machine_settings.move_speed = d_settings->value(QStringLiteral("Machine_Move_Speed"),250.0).toDouble();
	machine_settings.touch_speed = d_settings->value(QStringLiteral("Machine_Touch_Speed"),3.0).toDouble();
	machine_settings.acceleration = d_settings->value(QStringLiteral("Machine_Acceleration"),250.0).toDouble();
	machine_settings.approach_distance = d_settings->value(QStringLiteral("Machine_Approach_Distance"),4.0).toDouble();
	machine_settings.expansion_coeff_x = d_settings->value(QStringLiteral("Machine_Expansion_Coefficient_X"),10.0).toDouble();
	machine_settings.expansion_coeff_y = d_settings->value(QStringLiteral("Machine_Expansion_Coefficient_Y"),10.0).toDouble();
	machine_settings.expansion_coeff_z = d_settings->value(QStringLiteral("Machine_Expansion_Coefficient_Z"),10.0).toDouble();
	machine_settings.sensors_x = d_settings->value(QStringLiteral("Machine_Temperature_Sensors_X"),QStringLiteral("-1")).toString();
	machine_settings.sensors_y = d_settings->value(QStringLiteral("Machine_Temperature_Sensors_Y"),QStringLiteral("-1")).toString();
	machine_settings.sensors_z = d_settings->value(QStringLiteral("Machine_Temperature_Sensors_Z"),QStringLiteral("-1")).toString();
	machine_settings.sensors_part = d_settings->value(QStringLiteral("Machine_Temperature_Sensors_Part"),QStringLiteral("-1")).toString();
	machine_settings.thermal_compensation_mode = static_cast<TSettingsWidget::TThermalCompensationMode>(d_settings->value(QStringLiteral("Machine_Thermal_Compensation_Mode"),0).toInt());
	machine_settings.blended_move_mode = static_cast<TSettingsWidget::TBlendedMoveMode>(d_settings->value(QStringLiteral("Machine_Blended_Move_Mode"),0).toInt());

	d_settings_widget->Set_Machine_Settings(machine_settings);
	d_cmm_widget->Set_Temperature_Sensors_X(machine_settings.sensors_x);
	d_cmm_widget->Set_Temperature_Sensors_Y(machine_settings.sensors_y);
	d_cmm_widget->Set_Temperature_Sensors_Z(machine_settings.sensors_z);
	d_cmm_widget->Set_Temperature_Sensors_Part(machine_settings.sensors_part);
	d_measure_dialog->Set_Machine_Settings(machine_settings);

	artifact_settings.sg_checkmaster_pln_x_start = d_settings->value(QStringLiteral("Artifact_SG_Checkmaster_Plane_X_Start"),25.0).toDouble();
	artifact_settings.sg_checkmaster_pln_x_end = d_settings->value(QStringLiteral("Artifact_SG_Checkmaster_Plane_X_End"),-5.0).toDouble();
	artifact_settings.sg_checkmaster_pln_y1 = d_settings->value(QStringLiteral("Artifact_SG_Checkmaster_Plane_Y1"),2.0).toDouble();
	artifact_settings.sg_checkmaster_pln_y2 = d_settings->value(QStringLiteral("Artifact_SG_Checkmaster_Plane_Y2"),6.0).toDouble();
	artifact_settings.sg_checkmaster_ln_x_start = d_settings->value(QStringLiteral("Artifact_SG_Checkmaster_Line_X_Start"),25.0).toDouble();
	artifact_settings.sg_checkmaster_ln_x_end = d_settings->value(QStringLiteral("Artifact_SG_Checkmaster_Line_X_End"),-5.0).toDouble();
	artifact_settings.sg_checkmaster_ln_z = d_settings->value(QStringLiteral("Artifact_SG_Checkmaster_Line_Z"),-4.0).toDouble();
	artifact_settings.sg_checkmaster_pnt_y = d_settings->value(QStringLiteral("Artifact_SG_Checkmaster_Point_Y"),4.0).toDouble();
	artifact_settings.sg_checkmaster_pnt_z = d_settings->value(QStringLiteral("Artifact_SG_Checkmaster_Point_Z"),-4.0).toDouble();
	artifact_settings.sg_checkmaster_clear_z = d_settings->value(QStringLiteral("Artifact_SG_Checkmaster_Clear_Z"),15.0).toDouble();
	
	artifact_settings.sg_webber_clear_z = d_settings->value(QStringLiteral("Artifact_SG_Webber_Clear_Z"),25.0).toDouble();
	
	artifact_settings.sg_koba_pln_x_start = d_settings->value(QStringLiteral("Artifact_SG_Koba_Plane_X_Start"),10.0).toDouble();
	artifact_settings.sg_koba_pln_x_end = d_settings->value(QStringLiteral("Artifact_SG_Koba_Plane_X_End"),-10.0).toDouble();
	artifact_settings.sg_koba_pln_y1 = d_settings->value(QStringLiteral("Artifact_SG_Koba_Plane_Y1"),9.0).toDouble();
	artifact_settings.sg_koba_pln_y2 = d_settings->value(QStringLiteral("Artifact_SG_Koba_Plane_Y2"),46.0).toDouble();
	artifact_settings.sg_koba_ln_x_start = d_settings->value(QStringLiteral("Artifact_SG_Koba_Line_X_Start"),10.0).toDouble();
	artifact_settings.sg_koba_ln_x_end = d_settings->value(QStringLiteral("Artifact_SG_Koba_Line_X_End"),-10.0).toDouble();
	artifact_settings.sg_koba_ln_z = d_settings->value(QStringLiteral("Artifact_SG_Koba_Line_Z"),-11.0).toDouble();
	artifact_settings.sg_koba_pnt_y = d_settings->value(QStringLiteral("Artifact_SG_Koba_Point_Y"),27.5).toDouble();
	artifact_settings.sg_koba_pnt_z = d_settings->value(QStringLiteral("Artifact_SG_Koba_Point_Z"),-30.0).toDouble();
	artifact_settings.sg_koba_clear_probe = d_settings->value(QStringLiteral("Artifact_SG_Koba_Clear_Probe"),45.0).toDouble();
	
	artifact_settings.gaugeblock_pln_y = d_settings->value(QStringLiteral("Artifact_Gauge_BLock_Plane_Y"),0.0).toDouble();
	artifact_settings.gaugeblock_pln_z = d_settings->value(QStringLiteral("Artifact_Gauge_BLock_Plane_Z"),0.0).toDouble();
	artifact_settings.gaugeblock_pln_sp = d_settings->value(QStringLiteral("Artifact_Gauge_BLock_Spacing"),2.0).toDouble();
	artifact_settings.gaugeblock_clear_z = d_settings->value(QStringLiteral("Artifact_Gauge_BLock_Clear_Z"),25.0).toDouble();
	artifact_settings.ballbar_clear_distance = d_settings->value(QStringLiteral("Artifact_Ballbar_Clear_Distance"),50.0).toDouble();
	artifact_settings.ballbar_avoidance_angle = d_settings->value(QStringLiteral("Artifact_Ballbar_Avoidance_Angle"),60.0).toDouble();
	artifact_settings.ballbar_10360_clear_distance = d_settings->value(QStringLiteral("Artifact_Ballbar_10360_Clear_Distance"),50.0).toDouble();
	artifact_settings.ballbar_10360_sphere_points = d_settings->value(QStringLiteral("Artifact_Ballbar_10360_Sphere_Points"),13).toInt();
	artifact_settings.ringgauge_clear_z = d_settings->value(QStringLiteral("Artifact_Ring_Gauge_Clear_Z"),10.0).toDouble();
	artifact_settings.ringgauge_pln_offset = d_settings->value(QStringLiteral("Artifact_Ring_Gauge_Plane_Offset"),4.0).toDouble();
	artifact_settings.ringgauge_cir_z = d_settings->value(QStringLiteral("Artifact_Ring_Gauge_Circle_Z"),-4.0).toDouble();
	artifact_settings.pingauge_clear_z = d_settings->value(QStringLiteral("Artifact_Pin_Gauge_Clear_Z"),10.0).toDouble();
	artifact_settings.pingauge_pln_offset = d_settings->value(QStringLiteral("Artifact_Pin_Gauge_Plane_Offset"),-2.0).toDouble();
	artifact_settings.pingauge_cir_z = d_settings->value(QStringLiteral("Artifact_Pin_Gauge_Circle_Z"),-4.0).toDouble();
	artifact_settings.plane_sample_radius = d_settings->value(QStringLiteral("Plane_Sample_Radius"),3.0).toDouble();
	artifact_settings.roll_clear_z = d_settings->value(QStringLiteral("Artifact_Roll_Clear_Z"),100.0).toDouble();
	artifact_settings.roll_sphere_points = d_settings->value(QStringLiteral("Artifact_Roll_Sphere_Points"),13).toInt();
	artifact_settings.calibration_sphere_clear_z = d_settings->value(QStringLiteral("Artifact_Calibration_Sphere_Clear_Z"),100.0).toDouble();
	artifact_settings.calibration_sphere_points = d_settings->value(QStringLiteral("Artifact_Calibration_Sphere_Points"),7).toInt();

	d_settings_widget->Set_Artifact_Settings(artifact_settings);
	d_measure_dialog->Set_Artifact_Settings(artifact_settings);
	
	temperature_settings.upper_temperature_limit = d_settings->value(QStringLiteral("Temperature_Upper_Limit"),22.0).toDouble();
	temperature_settings.lower_temperature_limit = d_settings->value(QStringLiteral("Temperature_Lower_Limit"),18.0).toDouble();

	d_cmm_widget->Set_Temperature_Settings(temperature_settings);
	
	output_settings.output_relative_path = d_settings->value(QStringLiteral("Output_Relative_Path"),QStringLiteral("Measurement_Data")).toString();
	output_settings.output_absolute_path = d_settings->value(QStringLiteral("Output_Absolute_Path"),d_data_path + QStringLiteral("/Measurement_Data")).toString();
	
	// default to relative path option on windows
	bval = true;
	
#ifndef Q_OS_WIN
	bval = false;
#endif
	
	output_settings.use_relative_path = d_settings->value(QStringLiteral("Output_Use_Relative_Path"),bval).toBool();
	
	d_report_widget->Set_Output_Settings(d_data_path,output_settings);

	measure_parameters.axis_x_temperature = d_settings->value(QStringLiteral("Measurement_Axis_X_Temperature"),20.0).toDouble();
	measure_parameters.axis_y_temperature = d_settings->value(QStringLiteral("Measurement_Axis_Y_Temperature"),20.0).toDouble();
	measure_parameters.axis_z_temperature = d_settings->value(QStringLiteral("Measurement_Axis_Z_Temperature"),20.0).toDouble();
	measure_parameters.part_temperature = d_settings->value(QStringLiteral("Measurement_Part_Temperature"),20.0).toDouble();
	measure_parameters.sphere_stem_vector_i = d_settings->value(QStringLiteral("Measurement_Sphere_Stem_I"),0.0).toDouble();
	measure_parameters.sphere_stem_vector_j = d_settings->value(QStringLiteral("Measurement_Sphere_Stem_J"),0.0).toDouble();
	measure_parameters.sphere_stem_vector_k = d_settings->value(QStringLiteral("Measurement_Sphere_Stem_K"),1.0).toDouble();
	measure_parameters.stepgauge_b89_measurement_count = d_settings->value(QStringLiteral("Measurement_Stepgauge_B89_Measurement_Count"),3).toInt();
	measure_parameters.stepgauge_square_measurement_count = d_settings->value(QStringLiteral("Measurement_Stepgauge_Square_Measurement_Count"),3).toInt();
	measure_parameters.stepgauge_type = d_settings->value(QStringLiteral("Measurement_Stepgauge_Type"),0).toInt();
	measure_parameters.stepgauge_10360_position_name = d_settings->value(QStringLiteral("Measurement_Stepgauge_10360_Position_Name"),QStringLiteral("Position 1")).toString();
	measure_parameters.stepgauge_b89_position_name = d_settings->value(QStringLiteral("Measurement_Stepgauge_B89_Position_Name"),QStringLiteral("Axis X")).toString();
	measure_parameters.stepgauge_square_position_name = d_settings->value(QStringLiteral("Measurement_Stepgauge_Square_Position_Name"),QStringLiteral("SQ1")).toString();
	measure_parameters.stepgauge_zero_step_spacing = d_settings->value(QStringLiteral("Measurement_Stepgauge_Zero_Step_Spacing"),10.0).toDouble();
	measure_parameters.stepgauge_serial_number = d_settings->value(QStringLiteral("Measurement_Stepgauge_Serial_Number"),QStringLiteral("1520007")).toString();
	measure_parameters.stepgauge_measurement_zero_position = d_settings->value(QStringLiteral("Measurement_Stepgauge_Zero_Position"),0.0).toDouble();
	measure_parameters.stepgauge_measurement_b89_start = d_settings->value(QStringLiteral("Measurement_Stepgauge_B89_Start"),10.0).toDouble();
	measure_parameters.stepgauge_expansion_coeff = d_settings->value(QStringLiteral("Measurement_Stepgauge_Expansion_Coefficient"),10.0).toDouble();
	measure_parameters.stepgauge_nominal_block_spacing = d_settings->value(QStringLiteral("Measurement_Stepgauge_Nominal_Block_Spacing"),20.0).toDouble();
	measure_parameters.stepgauge_b89_nominal_block_spacing = d_settings->value(QStringLiteral("Measurement_Stepgauge_B89_Nominal_Block_Spacing"),100.0).toDouble();
	measure_parameters.gaugeblock_nominal_length = d_settings->value(QStringLiteral("Measurement_Gaugeblock_Nominal_Length"),12.7).toDouble();
	measure_parameters.gaugeblock_expansion_coeff = d_settings->value(QStringLiteral("Measurement_Gaugeblock_Expansion_Coefficient"),10.0).toDouble();
	measure_parameters.gaugeblock_position_name = d_settings->value(QStringLiteral("Measurement_Gaugeblock_Position_Name"),QStringLiteral("Position 1")).toString();
	measure_parameters.gaugeblock_serial_number = d_settings->value(QStringLiteral("Measurement_Gaugeblock_Serial_Number"),QStringLiteral("131417")).toString();
	measure_parameters.ballbar_position_name = d_settings->value(QStringLiteral("Measurement_Ballbar_Position_Name"),QStringLiteral("P1")).toString();
	measure_parameters.ballbar_sphere_diameter = d_settings->value(QStringLiteral("Measurement_Ballbar_Sphere_Diameter"),25.0).toDouble();
	measure_parameters.ballbar_expansion_coeff = d_settings->value(QStringLiteral("Measurement_Ballbar_Expansion_Coefficient"),10.0).toDouble();
	measure_parameters.ballbar_measurement_count = d_settings->value(QStringLiteral("Measurement_Ballbar_Measurement_Count"),2).toInt();
	measure_parameters.ringgauge_nominal_diameter = d_settings->value(QStringLiteral("Measurement_Ringgauge_Diameter"),100.0).toDouble();
	measure_parameters.ringgauge_point_count = d_settings->value(QStringLiteral("Measurement_Ringgauge_Point_Count"),10).toInt();
	measure_parameters.point_repeat_move_distance = d_settings->value(QStringLiteral("Measurement_Point_Repeat_Move_Distance"),25.0).toDouble();
	measure_parameters.sphere_nominal_diameter = d_settings->value(QStringLiteral("Measurement_Sphere_Nominal_Diameter"),25.0).toDouble();
	measure_parameters.point_repeat_measurement_count = d_settings->value(QStringLiteral("Measurement_Point_Repeat_Measurement_Count"),10).toInt();
	measure_parameters.sphere_repeat_measurement_count = d_settings->value(QStringLiteral("Measurement_Sphere_Repeat_Measurement_Count"),10).toInt();
	measure_parameters.pingauge_nominal_diameter = d_settings->value(QStringLiteral("Measurement_Pingauge_Diameter"),25.0).toDouble();
	measure_parameters.pingauge_point_count = d_settings->value(QStringLiteral("Measurement_Pingauge_Point_Count"),10).toInt();
	measure_parameters.point_repeat_description = d_settings->value(QStringLiteral("Measurement_Point_Repeat_Description"),QStringLiteral("X Axis")).toString();
	measure_parameters.roll_tool1_name = d_settings->value(QStringLiteral("Measurement_Roll_Tool1_Name"),QString()).toString();
	measure_parameters.roll_tool2_name = d_settings->value(QStringLiteral("Measurement_Roll_Tool2_Name"),QString()).toString();
	measure_parameters.ballbar_10360_position_name = d_settings->value(QStringLiteral("Measurement_Ballbar_10360_Position_Name"),QStringLiteral("Position 1")).toString();
	measure_parameters.ballbar_10360_serial_number = d_settings->value(QStringLiteral("Measurement_Ballbar_10360_Serial_Number"),QStringLiteral("90210")).toString();
	measure_parameters.ballbar_10360_sphere_diameter = d_settings->value(QStringLiteral("Measurement_Ballbar_10360_Sphere_Diameter"),25.0).toDouble();
	measure_parameters.ballbar_10360_sphere_spacing = d_settings->value(QStringLiteral("Measurement_Ballbar_10360_Sphere_Spacing"),50.0).toDouble();
	measure_parameters.ballbar_10360_expansion_coeff = d_settings->value(QStringLiteral("Measurement_Ballbar_10360_Expansion_Coefficient"),12.0).toDouble();
	measure_parameters.ballbar_10360_sphere0_offset = d_settings->value(QStringLiteral("Measurement_Ballbar_10360_Sphere0_Offset"),0).toInt();

	d_measure_dialog->Set_Measurement_Parameters(measure_parameters);
	d_measure_dialog->Enable_Temperature_Compensation(d_settings->value(QStringLiteral("Measurement_Enable_Temperature_Compensation"),false).toBool());

	laser_parameters.description = d_settings->value(QStringLiteral("Laser_Description"),QStringLiteral("X Axis")).toString();
	laser_parameters.increment = d_settings->value(QStringLiteral("Laser_Increment"),50.0).toDouble();
	laser_parameters.premove_offset = d_settings->value(QStringLiteral("Laser_Premove_Offset"),2.0).toDouble();
	laser_parameters.start_x = d_settings->value(QStringLiteral("Laser_Start_X"),50.0).toDouble();
	laser_parameters.start_y = d_settings->value(QStringLiteral("Laser_Start_Y"),50.0).toDouble();
	laser_parameters.start_z = d_settings->value(QStringLiteral("Laser_Start_Z"),-250.0).toDouble();
	laser_parameters.end_x = d_settings->value(QStringLiteral("Laser_End_X"),50.0).toDouble();
	laser_parameters.end_y = d_settings->value(QStringLiteral("Laser_End_Y"),500.0).toDouble();
	laser_parameters.end_z = d_settings->value(QStringLiteral("Laser_End_Z"),-250.0).toDouble();
	laser_parameters.tool_offset_x = d_settings->value(QStringLiteral("Laser_Tool_Offset_X"),0.0).toDouble();
	laser_parameters.tool_offset_y = d_settings->value(QStringLiteral("Laser_Tool_Offset_Y"),0.0).toDouble();
	laser_parameters.tool_offset_z = d_settings->value(QStringLiteral("Laser_Tool_Offset_Z"),-130.0).toDouble();
	laser_parameters.measurement_pattern = d_settings->value(QStringLiteral("Laser_Measurement_Pattern"),0).toInt();
	laser_parameters.data_collection_mode = d_settings->value(QStringLiteral("Laser_Collection_Mode"),0).toInt();
	laser_parameters.measurement_type = d_settings->value(QStringLiteral("Laser_Measurement_Type"),0).toInt();
	laser_parameters.filter_level = d_settings->value(QStringLiteral("Laser_Filter_Level8"),0).toInt();

	d_laser_dialog->Set_Parameters(laser_parameters);
	
	d_tool_widget->Load_Tool_Data(d_data_path);

	this->setWindowTitle(QStringLiteral("MeasureDirect - Version 9.0"));	// version in measureview should match along with copyright date in logowidget.cpp
	
	connect(d_cmm_widget,&TCMMWidget::Connect,this,&TMeasureDirect::CMM_Connect);
	connect(d_cmm_widget,&TCMMWidget::Disconnect,this,&TMeasureDirect::CMM_Disconnect);
	connect(d_cmm_widget,&TCMMWidget::Home,this,&TMeasureDirect::CMM_Home);
	connect(d_cmm_widget,&TCMMWidget::Exercise_Machine,this,&TMeasureDirect::CMM_Exercise_Machine);
	connect(d_cmm_widget,&TCMMWidget::Request_Sensor_Value,this,&TMeasureDirect::Request_Sensor_Value);

	connect(d_tool_widget,&TToolWidget::Calibrate_Tools,this,&TMeasureDirect::Calibrate_Tools);
	connect(d_tool_widget,&TToolWidget::Refresh_Tools,this,&TMeasureDirect::Refresh_Tools);

	connect(d_settings_widget,&TSettingsWidget::Machine_Settings_Changed,this,&TMeasureDirect::Machine_Settings_Changed);
	connect(d_settings_widget,&TSettingsWidget::Artifact_Settings_Changed,this,&TMeasureDirect::Artifact_Settings_Changed);
	connect(d_settings_widget,&TSettingsWidget::Request_Sensor_IDs,this,&TMeasureDirect::Request_Sensor_IDs);
	
	connect(d_measure_dialog,&TMeasureDialog::Sequence_Start,this,&TMeasureDirect::Measure_Sequence_Start);
	connect(d_measure_dialog,&TMeasureDialog::Sequence_End,this,&TMeasureDirect::Measure_Sequence_End);
	connect(d_measure_dialog,&TMeasureDialog::Refresh_Temperatures,this,&TMeasureDirect::Measure_Refresh_Temperatures);
	connect(d_measure_dialog,&TMeasureDialog::Update_Part_Expansion_Coefficient,this,&TMeasureDirect::Measure_Update_Part_Expansion_Coefficient);
	connect(d_measure_dialog,&TMeasureDialog::Pause_Driver,this,&TMeasureDirect::Pause_Driver);
	connect(d_measure_dialog,&TMeasureDialog::Resume_Driver,this,&TMeasureDirect::Resume_Driver);
	connect(d_measure_dialog,&TMeasureDialog::Stop_Driver,this,&TMeasureDirect::Stop_Driver);
	connect(d_measure_dialog,&TMeasureDialog::Update_Tip_Diameter,this,&TMeasureDirect::Measure_Update_Tip_Diameter);
	connect(d_measure_dialog,&TMeasureDialog::Change_Tool,this,&TMeasureDirect::Change_Tool);
	connect(d_measure_widget,&TMeasureWidget::Measure_Stepgauge_10360,this,&TMeasureDirect::Measure_Stepgauge_10360);
	connect(d_measure_widget,&TMeasureWidget::Measure_Stepgauge_B89,this,&TMeasureDirect::Measure_Stepgauge_B89);
	connect(d_measure_widget,&TMeasureWidget::Measure_Stepgauge_Square,this,&TMeasureDirect::Measure_Stepgauge_Square);
	connect(d_measure_widget,&TMeasureWidget::Measure_Ballbar,this,&TMeasureDirect::Measure_Ballbar);
	connect(d_measure_widget,&TMeasureWidget::Measure_Ballbar_10360,this,&TMeasureDirect::Measure_Ballbar_10360);
	connect(d_measure_widget,&TMeasureWidget::Measure_Gaugeblock,this,&TMeasureDirect::Measure_Gaugeblock);
	connect(d_measure_widget,&TMeasureWidget::Measure_Sphere_Pftu,this,&TMeasureDirect::Measure_Sphere_Pftu);
	connect(d_measure_widget,&TMeasureWidget::Measure_Ringgauge,this,&TMeasureDirect::Measure_Ringgauge);
	connect(d_measure_widget,&TMeasureWidget::Measure_Pingauge,this,&TMeasureDirect::Measure_Pingauge);
	connect(d_measure_widget,&TMeasureWidget::Measure_Sphere_Repeat,this,&TMeasureDirect::Measure_Sphere_Repeat);
	connect(d_measure_widget,&TMeasureWidget::Measure_Point_Repeat,this,&TMeasureDirect::Measure_Point_Repeat);
	connect(d_measure_widget,&TMeasureWidget::Measure_Roll,this,&TMeasureDirect::Measure_Roll);
	connect(d_measure_widget,&TMeasureWidget::Measure_Laser,this,&TMeasureDirect::Measure_Laser);

	connect(d_exercise_dialog,&TExerciseDialog::Sequence_Start,this,&TMeasureDirect::Exercise_Sequence_Start);
	connect(d_exercise_dialog,&TExerciseDialog::Sequence_Cancel,this,&TMeasureDirect::Exercise_Sequence_Cancel);
	connect(d_exercise_dialog,&TExerciseDialog::Sequence_End,this,&TMeasureDirect::Exercise_Sequence_End);
	connect(d_exercise_dialog,&TExerciseDialog::Get_Position,this,&TMeasureDirect::Exercise_Get_Position);
	connect(d_exercise_dialog,&TExerciseDialog::Move_Position,this,&TMeasureDirect::Move_Position);
	connect(d_exercise_dialog,&TExerciseDialog::Change_Tool,this,&TMeasureDirect::Change_Tool);
	
	connect(d_laser_dialog,&TLaserDialog::Sequence_Start,this,&TMeasureDirect::Laser_Sequence_Start);
	connect(d_laser_dialog,&TLaserDialog::Sequence_End,this,&TMeasureDirect::Laser_Sequence_End);
	connect(d_laser_dialog,&TLaserDialog::Change_Tool,this,&TMeasureDirect::Change_Tool);
	connect(d_laser_dialog,&TLaserDialog::Set_Active_Tool_Offsets,this,&TMeasureDirect::Laser_Set_Tool_Data);
	connect(d_laser_dialog,&TLaserDialog::Stop_Driver,this,&TMeasureDirect::Stop_Driver);
	connect(d_laser_dialog,&TLaserDialog::Move_Position,this,&TMeasureDirect::Move_Position);

	connect(d_process_timer,&QTimer::timeout,this,&TMeasureDirect::Process_Queued_Commands);
}

TMeasureDirect::~TMeasureDirect(void)
{
}

void TMeasureDirect::Calibrate_Tools(void)
{
	d_measure_dialog->Set_Calibration_Tool_Data(d_tool_widget->Calibration_Tool_Name(),d_tool_widget->Calibration_Tool_Items());
	d_measure_dialog->Reset(d_cmm_widget->Controller_Type(),MeasureTypes::CALIBRATE_TOOLS,d_report_widget->Output_Path(),QStringList(),QString());
	
	if(d_measure_dialog->isVisible())
	{
		d_measure_dialog->raise();
		d_measure_dialog->activateWindow();
	}
	else
	{
		d_measure_dialog->show();
	}

	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(true,TDriver::UPDATE_SPEED_DEFAULT);
	}
}

void TMeasureDirect::Change_Tool(
	const QString						&name)
{
	TToolWidget::TToolItem				tool_item;
	TController::TControllerType		controller_type;
	TDriver::TCommand					command;
	bool								valid;

	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		if(d_driver->Is_EStop())
		{
			d_message_box->setText(QStringLiteral("Error - Severity 0"));
			d_message_box->setInformativeText(QStringLiteral("Machine is in E-Stop."));
			d_message_box->setDetailedText(QString());
			d_message_box->setStandardButtons(QMessageBox::Ok);
			d_message_box->setDefaultButton(QMessageBox::Ok);
			d_message_box->setIcon(QMessageBox::Warning);
			
			d_message_box->exec();
			
			d_tool_widget->Invalidate_Active_Tool();
			
			d_measure_dialog->Reset_Tool_Selection();
			d_exercise_dialog->Reset_Tool_Selection();
			d_laser_dialog->Reset_Tool_Selection();

			return;
		}

		tool_item = d_tool_widget->Tool_Item(name,&valid);
		
		if(!valid)
		{
			d_cmm_widget->Add_Log_Text(QString("ERR  Cannot set tool '%1'").arg(name));
			
			d_message_box->setText(QStringLiteral("Tool Error"));
			d_message_box->setInformativeText(QString("Cannot set tool '%1'").arg(name));
			d_message_box->setDetailedText(QString());
			d_message_box->setStandardButtons(QMessageBox::Ok);
			d_message_box->setDefaultButton(QMessageBox::Ok);
			d_message_box->setIcon(QMessageBox::Critical);
			
			d_message_box->exec();
			
			d_tool_widget->Invalidate_Active_Tool();
			
			d_measure_dialog->Reset_Tool_Selection();
			d_exercise_dialog->Reset_Tool_Selection();
			d_laser_dialog->Reset_Tool_Selection();

			return;
		}


		command.command_type = TDriver::DRIVER_SET_TOOL_NAME;
		command.text = tool_item.name;
		d_driver->Add_Command(command);

		controller_type = d_cmm_widget->Controller_Type();
		
		if(controller_type != TController::CONTROLLER_IPPCLIENT)
		{
			command.command_type = TDriver::DRIVER_SET_TOOL_TYPE;
			command.text = QStringLiteral("TTP");
			d_driver->Add_Command(command);
			
			command.command_type = TDriver::DRIVER_SET_TOOL_AB_ANGLES;
			command.dvalue1 = tool_item.angle_a;
			command.dvalue2 = tool_item.angle_b;
			d_driver->Add_Command(command);
			
			command.command_type = TDriver::DRIVER_SET_TOOL_DATA;
			command.xyz = tool_item.xyz;
			command.dvalue1 = tool_item.tip_diameter;
			d_driver->Add_Command(command);
		}
		
		command.command_type = TDriver::DRIVER_QUERY_TOOL_DATA;
		command.text = tool_item.name;
		d_driver->Add_Command(command);
	}
}

void TMeasureDirect::Refresh_Tools(void)
{
	TDriver::TCommand					command;
	TController::TControllerType		controller_type;

	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		controller_type = d_cmm_widget->Controller_Type();

		if(controller_type == TController::CONTROLLER_IPPCLIENT)
		{
			d_tool_widget->Invalidate_Active_Tool();
			d_measure_dialog->Clear_Tool_Data();
			
			command.command_type = TDriver::DRIVER_REFRESH_TOOLS;
			d_driver->Add_Command(command);
		}
	}
}

void TMeasureDirect::Move_Position(
	const double						&x,
	const double						&y,
	const double						&z)
{
	TDriver::TCommand					command;
		
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_message_box->setText(QStringLiteral("DCC Mode"));
		d_message_box->setInformativeText(QStringLiteral("Clear Probe For DCC Moves"));
		d_message_box->setDetailedText(QString());
		d_message_box->setStandardButtons(QMessageBox::Ok | QMessageBox::Cancel);
		d_message_box->setDefaultButton(QMessageBox::Ok);
		d_message_box->setIcon(QMessageBox::Warning);
		
		if(QMessageBox::Ok == d_message_box->exec())
		{
			command.command_type = TDriver::DRIVER_SET_MODE_DCC;
			d_driver->Add_Command(command);
			
			command.command_type = TDriver::DRIVER_MOVE_TO;
			command.xyz.Set(x,y,z);
			d_driver->Add_Command(command);
			
			command.command_type = TDriver::DRIVER_SET_MODE_MANUAL;
			d_driver->Add_Command(command);
		}
	}
}

void TMeasureDirect::Pause_Driver(void)
{
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Pause();
	}
}

void TMeasureDirect::Resume_Driver(void)
{
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Resume();
	}
}

void TMeasureDirect::Stop_Driver(void)
{
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Stop();
	}
}

void TMeasureDirect::Driver_Ended(void)
{
	d_cmm_widget->Set_Driver_State(TDriver::DRIVER_STATE_NOT_CONNECTED);
	
	d_settings_widget->setEnabled(false);
	d_tool_widget->setEnabled(false);
	d_measure_widget->setEnabled(false);

	if(d_driver)
	{
		disconnect(d_driver,0,0,0);
		
		d_driver->deleteLater();
		d_driver = 0;
	}
	
	if(d_close_on_thread_finish)
	{
		this->close();
	}
}

void TMeasureDirect::Machine_Settings_Changed(void)
{
	TSettingsWidget::TMachineSettings	machine_settings;
	TDriver::TCommand					command;

	machine_settings = d_settings_widget->Machine_Settings();
	
	if(d_driver &&
	   d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED &&
	   (d_tool_widget->Active_Tool_Valid() == true || d_cmm_widget->Controller_Type() != TController::CONTROLLER_IPPCLIENT))	// I++ requires active probe first
	{
		command.command_type = TDriver::DRIVER_SET_MOVE_SPEED;
		command.dvalue1 = machine_settings.move_speed;
		d_driver->Add_Command(command);
		
		command.command_type = TDriver::DRIVER_SET_TOUCH_SPEED;
		command.dvalue1 = machine_settings.touch_speed;
		d_driver->Add_Command(command);
		
		command.command_type = TDriver::DRIVER_SET_ACCELERATION;
		command.dvalue1 = machine_settings.acceleration;
		d_driver->Add_Command(command);
		
		command.command_type = TDriver::DRIVER_SET_APPROACH_DISTANCE;
		command.dvalue1 = machine_settings.approach_distance;
		d_driver->Add_Command(command);
		
		switch(machine_settings.blended_move_mode)
		{
			case TSettingsWidget::BLENDED_MOVE_IGNORED:
				break;
				
			case TSettingsWidget::BLENDED_MOVE_ENABLE:
				
				command.command_type = TDriver::DRIVER_ENABLE_BLENDED_MOVES;
				d_driver->Add_Command(command);
				
				break;
				
			case TSettingsWidget::BLENDED_MOVE_DISABLE:
				
				command.command_type = TDriver::DRIVER_DISABLE_BLENDED_MOVES;
				d_driver->Add_Command(command);
				
				break;
		}
	}
	
	d_cmm_widget->Set_Temperature_Sensors_X(machine_settings.sensors_x);
	d_cmm_widget->Set_Temperature_Sensors_Y(machine_settings.sensors_y);
	d_cmm_widget->Set_Temperature_Sensors_Z(machine_settings.sensors_z);
	d_cmm_widget->Set_Temperature_Sensors_Part(machine_settings.sensors_part);
	d_measure_dialog->Set_Machine_Settings(machine_settings);
	
	this->Thermal_Settings_Changed();
}

void TMeasureDirect::Thermal_Settings_Changed(void)
{
	TSettingsWidget::TMachineSettings	machine_settings;
	TMeasureDialog::TThermalCompensationMode mode;
	TController::TControllerType		controller_type;
	
	machine_settings = d_settings_widget->Machine_Settings();
	
	mode = TMeasureDialog::SOFTWARE_COMPENSATION_AXIS_AND_PART;
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		if(machine_settings.thermal_compensation_mode == TSettingsWidget::CONTROLLER_THERMAL_COMPENSATION_ACTIVE)
		{
			controller_type = d_cmm_widget->Controller_Type();
			
			switch(controller_type)
			{
				case TController::CONTROLLER_IPPCLIENT:
					mode = TMeasureDialog::CONTROLLER_AXIS_AND_PART;
					break;
					
				case TController::CONTROLLER_VIRTUAL:
				case TController::CONTROLLER_DC:
				case TController::CONTROLLER_LEITZ:
					mode = TMeasureDialog::CONTROLLER_AXIS_SOFTWARE_PART;
					break;
			}
		}
	}
	
	d_measure_dialog->Set_Thermal_Settings(mode);
}

void TMeasureDirect::Artifact_Settings_Changed(void)
{
	TSettingsWidget::TArtifactSettings	settings;
	
	settings = d_settings_widget->Artifact_Settings();

	d_measure_dialog->Set_Artifact_Settings(settings);
}

void TMeasureDirect::Request_Sensor_IDs(void)
{
	TDriver::TCommand					command;
		
	command.command_type = TDriver::DRIVER_GET_SENSOR_IDS;

	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Add_Command(command);
	}
}

void TMeasureDirect::Request_Sensor_Value(
	int									sensor_id)
{
	TDriver::TCommand					command;
		
	command.command_type = TDriver::DRIVER_GET_SENSOR_VALUE;
	command.ivalue = sensor_id;

	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Add_Command(command,true);	// immediate execution required
	}
}

void TMeasureDirect::CMM_Connect(void)
{
	TController::TControllerType		controller_type;
	TLinkProtocol::TProtocolType		connection_protocol;
	
	if(d_driver)
	{
		d_driver->Close_Driver();
		return;
	}
	
	controller_type = d_cmm_widget->Controller_Type();
	
	d_link_dialog->Reset(controller_type);
	
	if(d_link_dialog->exec() != QDialog::Accepted)
	{
		return;
	}
	
	d_tool_widget->Invalidate_Active_Tool();

	d_driver = new TDriver(this);
	
	connection_protocol = d_link_dialog->Connection_Protocol();
	
	d_driver->Set_Protocol_Type(connection_protocol);

	if(connection_protocol == TLinkProtocol::PROTOCOL_SERIAL)
	{
		d_driver->Set_Serial_DeviceName(d_link_dialog->Comm_Port());
		d_driver->Set_Serial_Baud(d_link_dialog->Baud_Rate());
		d_driver->Set_Serial_Data(TLibSerialDeviceEnum::SD_DATA_BITS_8);
		d_driver->Set_Serial_Parity(TLibSerialDeviceEnum::SD_PARITY_NONE);
		d_driver->Set_Serial_StopBits(TLibSerialDeviceEnum::SD_STOP_BITS_1);
		d_driver->Set_Serial_Flow(TLibSerialDeviceEnum::SD_FLOW_NONE);
		d_driver->Set_Serial_ReadTimeout(4);
		
	}
	else if(connection_protocol == TLinkProtocol::PROTOCOL_ETHERNET)
	{
		d_driver->Set_Socket_Hostname(d_link_dialog->Ethernet_Address());
		d_driver->Set_Socket_Port(d_link_dialog->Ethernet_Port());
	}
	
	d_driver->Reset(controller_type);
	
	connect(d_driver,&TDriver::Connecting,this,&TMeasureDirect::Driver_Connecting);
	connect(d_driver,&TDriver::Connected,this,&TMeasureDirect::Driver_Connected);
	connect(d_driver,&TDriver::Update_Machine_State,this,&TMeasureDirect::Driver_Update_Machine_State);

	connect(d_driver,&TDriver::finished,this,&TMeasureDirect::Driver_Ended);
	connect(d_driver,&TDriver::Touch_Point, this, &TMeasureDirect::Driver_Touch_Point);
	connect(d_driver,&TDriver::Position,this,&TMeasureDirect::Driver_Position);
	connect(d_driver,&TDriver::Event,this,&TMeasureDirect::Driver_Event);
	connect(d_driver,&TDriver::Error,this,&TMeasureDirect::Driver_Error);
	connect(d_driver,&TDriver::Key_Done,this,&TMeasureDirect::Driver_Key_Done);
	connect(d_driver,&TDriver::Key_Erase_Hit,this,&TMeasureDirect::Driver_Key_Erase_Hit);
	connect(d_driver,&TDriver::Query_Tool_Data,this,&TMeasureDirect::Driver_Query_Tool_Data);
	connect(d_driver,&TDriver::Tool_Index_Confirmation_AB,this,&TMeasureDirect::Driver_Tool_Index_Confirmation_AB,Qt::BlockingQueuedConnection);
	connect(d_driver,&TDriver::Tool_Index_Confirmation_Name,this,&TMeasureDirect::Driver_Tool_Index_Confirmation_Name,Qt::BlockingQueuedConnection);
	connect(d_driver,&TDriver::Tool_Name_Changed,this,&TMeasureDirect::Driver_Tool_Name_Changed);
	connect(d_driver,&TDriver::Tool_Update_Error,this,&TMeasureDirect::Driver_Tool_Update_Error);
	connect(d_driver,&TDriver::Approach_Distance,this,&TMeasureDirect::Driver_Approach_Distance);
	connect(d_driver,&TDriver::Driver_Sync,this,&TMeasureDirect::Driver_Sync);

	connect(d_driver,&TDriver::Temperature_X,this,&TMeasureDirect::Driver_Temperature_X);
	connect(d_driver,&TDriver::Temperature_Y,this,&TMeasureDirect::Driver_Temperature_Y);
	connect(d_driver,&TDriver::Temperature_Z,this,&TMeasureDirect::Driver_Temperature_Z);
	connect(d_driver,&TDriver::Temperature_Part,this,&TMeasureDirect::Driver_Temperature_Part);
	connect(d_driver,&TDriver::Sensor_Value,this,&TMeasureDirect::Driver_Sensor_Value);
	
	connect(d_driver,&TDriver::Sensors_X,this,&TMeasureDirect::Driver_Sensors_X);
	connect(d_driver,&TDriver::Sensors_Y,this,&TMeasureDirect::Driver_Sensors_Y);
	connect(d_driver,&TDriver::Sensors_Z,this,&TMeasureDirect::Driver_Sensors_Z);
	connect(d_driver,&TDriver::Sensors_Part,this,&TMeasureDirect::Driver_Sensors_Part);

	connect(d_driver,&TDriver::Clear_Ipp_Tools,this,&TMeasureDirect::Driver_Clear_Ipp_Tools);
	connect(d_driver,&TDriver::Add_Ipp_Tool,this,&TMeasureDirect::Driver_Add_Ipp_Tool);
	connect(d_driver,&TDriver::Add_Ipp_Tools_Complete,this,&TMeasureDirect::Driver_Add_Ipp_Tools_Complete);
	
	d_driver->start(QThread::NormalPriority);
}

void TMeasureDirect::CMM_Disconnect(void)
{
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Close_Driver();
	}
}

void TMeasureDirect::CMM_Home(void)
{
	TDriver::TCommand					command;

	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		if(d_driver->Is_EStop())
		{
			d_message_box->setText(QStringLiteral("Error - Severity 0"));
			d_message_box->setInformativeText(QStringLiteral("Machine is in E-Stop."));
			d_message_box->setDetailedText(QString());
			d_message_box->setStandardButtons(QMessageBox::Ok);
			d_message_box->setDefaultButton(QMessageBox::Ok);
			d_message_box->setIcon(QMessageBox::Warning);
			
			d_message_box->exec();
			return;
		}

		d_message_box->setText(QStringLiteral("Home CMM"));
		d_message_box->setInformativeText(QStringLiteral("Do you want to home the CMM?"));
		d_message_box->setDetailedText(QString());
		d_message_box->setStandardButtons(QMessageBox::Yes | QMessageBox::No);
		d_message_box->setDefaultButton(QMessageBox::Yes);
		d_message_box->setIcon(QMessageBox::Warning);
		
		if(QMessageBox::No == d_message_box->exec())
		{
			return;
		}
		
		command.command_type = TDriver::DRIVER_SEND_HOME;
		d_driver->Add_Command(command);
	}
}

void TMeasureDirect::CMM_Exercise_Machine(void)
{
	QStringList							list;
	QString								active_tool_name;

	list = d_tool_widget->Get_Tool_List();
	list.push_front(QStringLiteral("-- Select Tool --"));
	
	active_tool_name = d_tool_widget->Active_Tool_Name();

	d_exercise_dialog->Reset(list,active_tool_name);
	d_exercise_dialog->exec();
}

void TMeasureDirect::Measure_Sequence_Start(void)
{
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		if(d_driver->Is_EStop())
		{
			d_message_box->setText(QStringLiteral("Error - Severity 0"));
			d_message_box->setInformativeText(QStringLiteral("Machine is in E-Stop."));
			d_message_box->setDetailedText(QString());
			d_message_box->setStandardButtons(QMessageBox::Ok);
			d_message_box->setDefaultButton(QMessageBox::Ok);
			d_message_box->setIcon(QMessageBox::Warning);
			
			d_message_box->exec();

			d_measure_dialog->Sequence_Canceled();
			return;
		}
		else if(!d_driver->Is_Homed())
		{
			d_message_box->setText(QStringLiteral("Home Not Set"));
			d_message_box->setInformativeText(QStringLiteral("Machine must be homed before it can be used."));
			d_message_box->setDetailedText(QString());
			d_message_box->setStandardButtons(QMessageBox::Ok);
			d_message_box->setDefaultButton(QMessageBox::Ok);
			d_message_box->setIcon(QMessageBox::Information);
			
			d_message_box->exec();

			d_measure_dialog->Sequence_Canceled();
			return;
		}

		d_driver->Run();
		d_driver->Set_Command_Queue_Size(-1);

		this->Machine_Settings_Changed();	// force settings

		d_process_timer->start();
	}
}

void TMeasureDirect::Measure_Sequence_End(void)
{
	TDriver::TCommand					command;
	
	d_process_timer->stop();
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		command.command_type = TDriver::DRIVER_SET_MODE_MANUAL;
		d_driver->Add_Command(command);
	}
}

void TMeasureDirect::Measure_Refresh_Temperatures(void)
{
	TDriver::TCommand					command;
	TSettingsWidget::TMachineSettings	machine_settings;
	
	machine_settings = d_settings_widget->Machine_Settings();
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		command.command_type = TDriver::DRIVER_GET_X_TEMPERATURE;
		command.text = machine_settings.sensors_x;
		d_driver->Add_Command(command);
		
		command.command_type = TDriver::DRIVER_GET_Y_TEMPERATURE;
		command.text = machine_settings.sensors_y;
		d_driver->Add_Command(command);
		
		command.command_type = TDriver::DRIVER_GET_Z_TEMPERATURE;
		command.text = machine_settings.sensors_z;
		d_driver->Add_Command(command);
		
		command.command_type = TDriver::DRIVER_GET_PART_TEMPERATURE;
		command.text = machine_settings.sensors_part;
		d_driver->Add_Command(command);
	}
}

void TMeasureDirect::Measure_Update_Part_Expansion_Coefficient(
	const double						&expansion_coefficient)
{
	TDriver::TCommand					command;

	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		command.command_type = TDriver::DRIVER_SET_PART_CTE;
		command.dvalue1 = expansion_coefficient;
		d_driver->Add_Command(command);
	}
}

void TMeasureDirect::Measure_Update_Tip_Diameter(
	const QString						&tool_base_name,
	const double						&angle_a,
	const double						&angle_b,
	const double						&tip_diameter)
{
	TDriver::TCommand					command;
	TToolWidget::TToolItem				tool_item;
	TController::TControllerType		controller_type;
	QString								active_tool_name;
	QString								update_tool_name;
	bool								valid;
	
	d_tool_widget->Update_Tool_Tip_Diameter(tool_base_name,angle_a,angle_b,tip_diameter);

	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		valid = d_tool_widget->Active_Tool_Valid();
		if(valid)
		{
			active_tool_name = d_tool_widget->Active_Tool_Name();
			update_tool_name = TToolWidget::Format_Tool_Name(tool_base_name,angle_a,angle_b);

			if(update_tool_name == active_tool_name)
			{
				controller_type = d_cmm_widget->Controller_Type();
				
				if(controller_type != TController::CONTROLLER_IPPCLIENT)
				{
					tool_item = d_tool_widget->Tool_Item(active_tool_name,&valid);

					if(valid)
					{
						command.command_type = TDriver::DRIVER_SET_TOOL_DATA;
						command.xyz = tool_item.xyz;
						command.dvalue1 = tool_item.tip_diameter;
						d_driver->Add_Command(command);
					}
				}
			 
				command.command_type = TDriver::DRIVER_QUERY_TOOL_DATA;
				command.text = active_tool_name;
				d_driver->Add_Command(command);
			}
		}
	}
}

void TMeasureDirect::Measure_Stepgauge_10360(void)
{
	QStringList							list;
	QString								active_tool_name;

	list = d_tool_widget->Get_Tool_List();
	list.push_front(QStringLiteral("-- Select Tool --"));
	
	active_tool_name = d_tool_widget->Active_Tool_Name();

	d_measure_dialog->Reset(d_cmm_widget->Controller_Type(),MeasureTypes::MEASURE_STEPGAUGE_10360,d_report_widget->Output_Path(),list,active_tool_name);
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(true,TDriver::UPDATE_SPEED_DEFAULT);
	}
	
	d_measure_dialog->exec();
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(false,TDriver::UPDATE_SPEED_DEFAULT);
	}
}

void TMeasureDirect::Measure_Stepgauge_B89(void)
{
	QStringList							list;
	QString								active_tool_name;
	
	list = d_tool_widget->Get_Tool_List();
	list.push_front(QStringLiteral("-- Select Tool --"));
	
	active_tool_name = d_tool_widget->Active_Tool_Name();
	
	d_measure_dialog->Reset(d_cmm_widget->Controller_Type(),MeasureTypes::MEASURE_STEPGAUGE_B89,d_report_widget->Output_Path(),list,active_tool_name);
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(true,TDriver::UPDATE_SPEED_DEFAULT);
	}
	
	d_measure_dialog->exec();
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(false,TDriver::UPDATE_SPEED_DEFAULT);
	}
}

void TMeasureDirect::Measure_Stepgauge_Square(void)
{
	QStringList							list;
	QString								active_tool_name;

	list = d_tool_widget->Get_Tool_List();
	list.push_front(QStringLiteral("-- Select Tool --"));
	
	active_tool_name = d_tool_widget->Active_Tool_Name();

	d_measure_dialog->Reset(d_cmm_widget->Controller_Type(),MeasureTypes::MEASURE_STEPGAUGE_SQUARE,d_report_widget->Output_Path(),list,active_tool_name);

	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(true,TDriver::UPDATE_SPEED_DEFAULT);
	}
	
	d_measure_dialog->exec();
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(false,TDriver::UPDATE_SPEED_DEFAULT);
	}
}

void TMeasureDirect::Measure_Ballbar(void)
{
	QStringList							list;
	QString								active_tool_name;
	
	list = d_tool_widget->Get_Tool_List();
	list.push_front(QStringLiteral("-- Select Tool --"));
		
	active_tool_name = d_tool_widget->Active_Tool_Name();
	
	d_measure_dialog->Reset(d_cmm_widget->Controller_Type(),MeasureTypes::MEASURE_BALLBAR_B89,d_report_widget->Output_Path(),list,active_tool_name);

	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(true,TDriver::UPDATE_SPEED_DEFAULT);
	}
	
	
	d_measure_dialog->exec();
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(false,TDriver::UPDATE_SPEED_DEFAULT);
	}
}

void TMeasureDirect::Measure_Ballbar_10360(void)
{
	QStringList							list;
	QString								active_tool_name;
	
	list = d_tool_widget->Get_Tool_List();
	list.push_front(QStringLiteral("-- Select Tool --"));
		
	active_tool_name = d_tool_widget->Active_Tool_Name();
	
	d_measure_dialog->Reset(d_cmm_widget->Controller_Type(),MeasureTypes::MEASURE_BALLBAR_10360,d_report_widget->Output_Path(),list,active_tool_name);

	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(true,TDriver::UPDATE_SPEED_DEFAULT);
	}
	
	d_measure_dialog->exec();
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(false,TDriver::UPDATE_SPEED_DEFAULT);
	}
}

void TMeasureDirect::Measure_Gaugeblock(void)
{
	QStringList							list;
	QString								active_tool_name;

	list = d_tool_widget->Get_Tool_List();
	list.push_front(QStringLiteral("-- Select Tool --"));
	
	active_tool_name = d_tool_widget->Active_Tool_Name();
	
	d_measure_dialog->Reset(d_cmm_widget->Controller_Type(),MeasureTypes::MEASURE_GAUGEBLOCK,d_report_widget->Output_Path(),list,active_tool_name);

	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(true,TDriver::UPDATE_SPEED_DEFAULT);
	}
	
	d_measure_dialog->exec();
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(false,TDriver::UPDATE_SPEED_DEFAULT);
	}
}

void TMeasureDirect::Measure_Sphere_Pftu(void)
{
	QStringList							list;
	QString								active_tool_name;
	
	list = d_tool_widget->Get_Tool_List();
	list.push_front(QStringLiteral("-- Select Tool --"));
	
	active_tool_name = d_tool_widget->Active_Tool_Name();
		
	d_measure_dialog->Reset(d_cmm_widget->Controller_Type(),MeasureTypes::MEASURE_SPHERE_PFTU,d_report_widget->Output_Path(),list,active_tool_name);
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(true,TDriver::UPDATE_SPEED_DEFAULT);
	}
	
	d_measure_dialog->exec();
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(false,TDriver::UPDATE_SPEED_DEFAULT);
	}
}

void TMeasureDirect::Measure_Ringgauge(void)
{
	QStringList							list;
	QString								active_tool_name;
	
	list = d_tool_widget->Get_Tool_List();
	list.push_front(QStringLiteral("-- Select Tool --"));
	
	active_tool_name = d_tool_widget->Active_Tool_Name();
		
	d_measure_dialog->Reset(d_cmm_widget->Controller_Type(),MeasureTypes::MEASURE_RINGGAUGE,d_report_widget->Output_Path(),list,active_tool_name);
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(true,TDriver::UPDATE_SPEED_DEFAULT);
	}
	
	d_measure_dialog->exec();
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(false,TDriver::UPDATE_SPEED_DEFAULT);
	}
}

void TMeasureDirect::Measure_Pingauge(void)
{
	QStringList							list;
	QString								active_tool_name;
	
	active_tool_name = d_tool_widget->Active_Tool_Name();
	
	list = d_tool_widget->Get_Tool_List();
	list.push_front(QStringLiteral("-- Select Tool --"));
		
	d_measure_dialog->Reset(d_cmm_widget->Controller_Type(),MeasureTypes::MEASURE_PINGAUGE,d_report_widget->Output_Path(),list,active_tool_name);
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(true,TDriver::UPDATE_SPEED_DEFAULT);
	}
	
	d_measure_dialog->exec();
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(false,TDriver::UPDATE_SPEED_DEFAULT);
	}
}

void TMeasureDirect::Measure_Sphere_Repeat(void)
{
	QStringList							list;
	QString								active_tool_name;
	
	list = d_tool_widget->Get_Tool_List();
	list.push_front(QStringLiteral("-- Select Tool --"));
	
	active_tool_name = d_tool_widget->Active_Tool_Name();
		
	d_measure_dialog->Reset(d_cmm_widget->Controller_Type(),MeasureTypes::MEASURE_SPHERE_REPEAT,d_report_widget->Output_Path(),list,active_tool_name);
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(true,TDriver::UPDATE_SPEED_DEFAULT);
	}
	
	d_measure_dialog->exec();
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(false,TDriver::UPDATE_SPEED_DEFAULT);
	}
}

void TMeasureDirect::Measure_Point_Repeat(void)
{
	QStringList							list;
	QString								active_tool_name;
	
	list = d_tool_widget->Get_Tool_List();
	list.push_front(QStringLiteral("-- Select Tool --"));
	
	active_tool_name = d_tool_widget->Active_Tool_Name();
	
	d_measure_dialog->Reset(d_cmm_widget->Controller_Type(),MeasureTypes::MEASURE_POINT_REPEAT,d_report_widget->Output_Path(),list,active_tool_name);
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(true,TDriver::UPDATE_SPEED_DEFAULT);
	}
	
	d_measure_dialog->exec();
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(false,TDriver::UPDATE_SPEED_DEFAULT);
	}
}

void TMeasureDirect::Measure_Roll(void)
{
	QStringList							list;
	QString								active_tool_name;

	list = d_tool_widget->Get_Tool_List();
	list.push_front(QStringLiteral("-- Select Tool --"));
	
	active_tool_name = d_tool_widget->Active_Tool_Name();

	d_measure_dialog->Reset(d_cmm_widget->Controller_Type(),MeasureTypes::MEASURE_ROLL_OFFSET_TOOL,d_report_widget->Output_Path(),list,active_tool_name);
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(true,TDriver::UPDATE_SPEED_DEFAULT);
	}
	
	d_measure_dialog->exec();
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(false,TDriver::UPDATE_SPEED_DEFAULT);
	}
}

void TMeasureDirect::Measure_Laser(void)
{
	QStringList							list;
	QString								active_tool_name;
	
	list = d_tool_widget->Get_Tool_List();
	list.push_front(QStringLiteral("-- Select Tool --"));
	
	active_tool_name = d_tool_widget->Active_Tool_Name();

	d_laser_dialog->Reset(d_report_widget->Output_Path(),list,active_tool_name);
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(true,TDriver::UPDATE_SPEED_HIGH);
	}
	
	d_laser_dialog->exec();
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Enable_Position_Updates(false,TDriver::UPDATE_SPEED_DEFAULT);
	}
}

void TMeasureDirect::Laser_Set_Tool_Data(
	const double						&x,
	const double						&y,
	const double						&z,
	const double						&tip_diameter)
{
	TDriver::TCommand					command;

	command.command_type = TDriver::DRIVER_SET_TOOL_DATA;
	command.xyz.Set(x,y,z);
	command.dvalue1 = tip_diameter;
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Add_Command(command);
		
		if(d_cmm_widget->Controller_Type() != TController::CONTROLLER_IPPCLIENT)
		{
			d_tool_widget->Invalidate_Active_Tool();
		}
		
		d_tool_widget->Update_Tool_Information_Offsets(x,y,z);
	}
}

void TMeasureDirect::Laser_Sequence_Start(void)
{
	std::vector<TDriver::TCommand>		commands;
	std::vector<TDriver::TCommand>::iterator iter;
	TDriver::TCommand					command;
		
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		if(d_driver->Is_EStop())
		{
			d_message_box->setText(QStringLiteral("Error - Severity 0"));
			d_message_box->setInformativeText(QStringLiteral("Machine is in E-Stop."));
			d_message_box->setDetailedText(QString());
			d_message_box->setStandardButtons(QMessageBox::Ok);
			d_message_box->setDefaultButton(QMessageBox::Ok);
			d_message_box->setIcon(QMessageBox::Warning);
			
			d_message_box->exec();

			d_laser_dialog->Sequence_Canceled();
			return;
		}
		else if(!d_driver->Is_Homed())
		{
			d_message_box->setText(QStringLiteral("Home Not Set"));
			d_message_box->setInformativeText(QStringLiteral("Machine must be homed before it can be used."));
			d_message_box->setDetailedText(QString());
			d_message_box->setStandardButtons(QMessageBox::Ok);
			d_message_box->setDefaultButton(QMessageBox::Ok);
			d_message_box->setIcon(QMessageBox::Information);
			
			d_message_box->exec();

			d_laser_dialog->Sequence_Canceled();
			return;
		}

		d_driver->Run();
		d_driver->Set_Command_Queue_Size(1);
	
		this->Machine_Settings_Changed();	// force settings
		
		d_measure_widget->setEnabled(false);
		d_process_timer->start();
	}
}

void TMeasureDirect::Laser_Sequence_End(void)
{
	TDriver::TCommand					command;
	
	d_process_timer->stop();
	
	d_measure_widget->setEnabled(true);
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		command.command_type = TDriver::DRIVER_SET_MODE_MANUAL;
		d_driver->Add_Command(command);
	}
}

void TMeasureDirect::Driver_Connecting(void)
{
	d_cmm_widget->Set_Driver_State(TDriver::DRIVER_STATE_CONNECTING);
}

void TMeasureDirect::Driver_Connected(void)
{
	d_cmm_widget->Set_Driver_State(TDriver::DRIVER_STATE_CONNECTED);
	
	d_tool_widget->Reset(d_cmm_widget->Controller_Type());
	d_settings_widget->Reset(d_cmm_widget->Controller_Type());
	
	d_laser_dialog->Set_Controller_Type(d_cmm_widget->Controller_Type());
	
	d_settings_widget->setEnabled(true);
	d_tool_widget->setEnabled(false);
	d_measure_widget->setEnabled(false);

	// force settings
	this->Thermal_Settings_Changed();
}

void TMeasureDirect::Driver_Update_Machine_State(
	const bool							show_warning)
{
	bool								is_homed;
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		is_homed = d_driver->Is_Homed();
		
		d_tool_widget->setEnabled(is_homed);
		d_measure_widget->setEnabled(is_homed);
		d_cmm_widget->Enable_Options(true,is_homed,true);
		
		if(show_warning)
		{
			if(d_driver->Is_EStop())
			{
				d_message_box->setText(QStringLiteral("Error - Severity 0"));
				d_message_box->setInformativeText(QStringLiteral("Machine is in E-Stop."));
				d_message_box->setDetailedText(QString());
				d_message_box->setStandardButtons(QMessageBox::Ok);
				d_message_box->setDefaultButton(QMessageBox::Ok);
				d_message_box->setIcon(QMessageBox::Warning);
				
				d_message_box->exec();
			}
			else if(!is_homed)
			{
				d_message_box->setText(QStringLiteral("Home Not Set"));
				d_message_box->setInformativeText(QStringLiteral("Machine must be homed before it can be used."));
				d_message_box->setDetailedText(QString());
				d_message_box->setStandardButtons(QMessageBox::Ok);
				d_message_box->setDefaultButton(QMessageBox::Ok);
				d_message_box->setIcon(QMessageBox::Information);
				
				d_message_box->exec();
			}
		}
	}
}

void TMeasureDirect::Driver_Position(
	const double						&x,
	const double						&y,
	const double						&z)
{
	d_measure_dialog->Update_Position(TVector3(x,y,z));
	d_exercise_dialog->Set_Position(TVector3(x,y,z));
	d_laser_dialog->Update_Position(TVector3(x,y,z));
}

void TMeasureDirect::Driver_Touch_Point(
	const double						&x,
	const double						&y,
	const double						&z,
	const double						&i,
	const double						&j,
	const double						&k)
{
	QString								text;
	
	text = QString("TOUCH %1,%2,%3,%4,%5,%6  (approach)")
										.arg(x,12,'f',5)
										.arg(y,12,'f',5)
										.arg(z,12,'f',5)
										.arg(i,9,'f',6)
										.arg(j,9,'f',6)
										.arg(k,9,'f',6);
	
	d_cmm_widget->Add_Log_Text(text);

	// touch may return false if unexpected 
	// used to trigger the next feature.  point resent so it is not lost.
	if(!d_measure_dialog->Add_Measured_Point(TVector3(x,y,z),TVector3(i,j,k)))
	{
		this->Process_Queued_Commands();
		this->d_measure_dialog->Add_Measured_Point(TVector3(x,y,z),TVector3(i,j,k));
	}
}

void TMeasureDirect::Driver_Event(
	const QString						&text)
{
	d_cmm_widget->Add_Log_Text(text);
}

void TMeasureDirect::Driver_Error(
	const QString						&error,
	const int							severity)
{
	d_cmm_widget->Add_Log_Text(QString("ERR:  Severity %1 '%2'").arg(severity).arg(error));
	
	d_measure_dialog->Driver_Error(error,severity);
	d_exercise_dialog->Driver_Error(error,severity);
	d_laser_dialog->Driver_Error(error,severity);
	
	if(d_message_box->isVisible() == false)	// only show one error box.
	{
		d_message_box->setText(QString("Error - Severity %1").arg(severity));
		d_message_box->setInformativeText(error);
		d_message_box->setDetailedText(QString());
		d_message_box->setStandardButtons(QMessageBox::Ok);
		d_message_box->setDefaultButton(QMessageBox::Ok);
		d_message_box->setIcon(QMessageBox::Warning);
		
		d_message_box->exec();
	}
}

void TMeasureDirect::Driver_Key_Done(void)
{
	d_measure_dialog->Done_Key_Pressed();
}

void TMeasureDirect::Driver_Key_Erase_Hit(void)
{
	d_measure_dialog->Erase_Key_Pressed();
}

void TMeasureDirect::Driver_Query_Tool_Data(
	const QString						&name,
	const double						&x,
	const double						&y,
	const double						&z,
	const double 						&i,
	const double 						&j,
	const double 						&k,
	const double						&tip_diameter)
{
	TToolWidget::TToolItem				tool_item;
	bool								valid;
		
	if(d_cmm_widget->Controller_Type() == TController::CONTROLLER_IPPCLIENT)
	{
		tool_item.name = name;
		tool_item.angle_a = 0.0;	// AB probe angles not reliable from I++
		tool_item.angle_b = 0.0;
		tool_item.xyz.Set(x,y,z);
		tool_item.ijk.Set(i,j,k);
		tool_item.tip_diameter = tip_diameter;
		
		d_tool_widget->Update_Ipp_Tool(tool_item);
		valid = true;
	}
	else
	{
		tool_item = d_tool_widget->Tool_Item(name,&valid);
	}
		
	if(valid)
	{
		d_tool_widget->Update_Tool_Information(tool_item);
		
		d_measure_dialog->Set_Tool_Data(tool_item);
		
		if(d_laser_dialog->isVisible())
		{
			d_laser_dialog->Set_Tool_Data(tool_item);
		}
	}
}

void TMeasureDirect::Driver_Tool_Index_Confirmation_AB(
	const double						&angle_a,
	const double						&angle_b,
	int									* const state)
{
	assert(state);
	
	*state = 0;
	
	if(d_tool_widget->Is_Motorized_Probe_Head())
	{
		*state = 2;

		if(d_driver->Is_Manual_Mode())
		{
			d_message_box->setText(QStringLiteral("Tool Index"));
			d_message_box->setInformativeText(QString("Do you want to index head to A%1 B%2 ?").arg(angle_a,0,'f',1).arg(angle_b,0,'f',1));
			d_message_box->setDetailedText(QString());
			d_message_box->setStandardButtons(QMessageBox::Yes | QMessageBox::No);
			d_message_box->setDefaultButton(QMessageBox::Yes);
			d_message_box->setIcon(QMessageBox::Warning);
			
			if(QMessageBox::Yes != d_message_box->exec())
			{
				*state = 0;
				d_measure_dialog->Sequence_Canceled();
				d_exercise_dialog->Sequence_Canceled();
				d_laser_dialog->Sequence_Canceled();

				d_tool_widget->Invalidate_Active_Tool();
				
				d_measure_dialog->Reset_Tool_Selection();
				d_exercise_dialog->Reset_Tool_Selection();
				d_laser_dialog->Reset_Tool_Selection();
			}
		}
	}
	else
	{
		d_message_box->setText(QStringLiteral("Tool Index"));
		d_message_box->setInformativeText(QString("Index head to A%1 B%2 then press Okay to continue.").arg(angle_a,0,'f',1).arg(angle_b,0,'f',1));
		d_message_box->setDetailedText(QString());
		d_message_box->setStandardButtons(QMessageBox::Ok);
		d_message_box->setDefaultButton(QMessageBox::Ok);
		d_message_box->setIcon(QMessageBox::Warning);
		
		d_message_box->exec();
		
		*state = 1;
	}
}

void TMeasureDirect::Driver_Tool_Index_Confirmation_Name(
	const QString						&name,
	int									* const state)
{
	assert(state);
	
	*state = 0;
	
	if(d_tool_widget->Is_Motorized_Probe_Head() || d_cmm_widget->Controller_Type() == TController::CONTROLLER_IPPCLIENT)
	{
		*state = 2;

		if(d_driver->Is_Manual_Mode())
		{
			d_message_box->setText(QStringLiteral("Tool Index"));
			d_message_box->setInformativeText(QString("Do you want to index head to %1 ?").arg(name));
			d_message_box->setDetailedText(QString());
			d_message_box->setStandardButtons(QMessageBox::Yes | QMessageBox::No);
			d_message_box->setDefaultButton(QMessageBox::Yes);
			d_message_box->setIcon(QMessageBox::Warning);
			
			if(QMessageBox::Yes != d_message_box->exec())
			{
				*state = 0;
				d_measure_dialog->Sequence_Canceled();
				d_exercise_dialog->Sequence_Canceled();
				d_laser_dialog->Sequence_Canceled();
			}
		}
	}
	else
	{
		d_message_box->setText(QStringLiteral("Tool Index"));
		d_message_box->setInformativeText(QString("Index head to %1 then press Okay to continue.").arg(name));
		d_message_box->setDetailedText(QString());
		d_message_box->setStandardButtons(QMessageBox::Ok);
		d_message_box->setDefaultButton(QMessageBox::Ok);
		d_message_box->setIcon(QMessageBox::Warning);
		
		d_message_box->exec();
		
		*state = 1;
	}
}

void TMeasureDirect::Driver_Tool_Name_Changed(
	const QString 						&name)
{
	d_tool_widget->Set_Active_Tool_Name(name);
}

void TMeasureDirect::Driver_Tool_Update_Error(void)
{
	d_tool_widget->Invalidate_Active_Tool();
}

void TMeasureDirect::Driver_Approach_Distance(
	const double						&distance)
{
	d_measure_dialog->Set_Approach_Distance(distance);
}

void TMeasureDirect::Driver_Sync(
	const int							value)
{
	if(value < 0)
	{
		d_measure_dialog->Driver_Sync();
	}
	else
	{
		d_exercise_dialog->Driver_Sync(value);
	}
}

void TMeasureDirect::Driver_Temperature_X(
	const double						&value)
{
	QString								text;
	
	text = QString("X TEMP: %1").arg(value,0,'f',2);
	d_cmm_widget->Add_Log_Text(text);
	
	d_measure_dialog->Set_Temperature_X(value);
}

void TMeasureDirect::Driver_Temperature_Y(
	const double						&value)
{
	QString								text;
	
	text = QString("Y TEMP: %1").arg(value,0,'f',2);
	d_cmm_widget->Add_Log_Text(text);
	
	d_measure_dialog->Set_Temperature_Y(value);
}

void TMeasureDirect::Driver_Temperature_Z(
	const double						&value)
{
	QString								text;
	
	text = QString("Z TEMP: %1").arg(value,0,'f',2);
	d_cmm_widget->Add_Log_Text(text);
	
	d_measure_dialog->Set_Temperature_Z(value);
}

void TMeasureDirect::Driver_Temperature_Part(
	const double						&value)
{
	QString								text;
	
	text = QString("PART TEMP: %1").arg(value,0,'f',2);
	d_cmm_widget->Add_Log_Text(text);

	d_measure_dialog->Set_Temperature_Part(value);
}

void TMeasureDirect::Driver_Sensor_Value(
	int									sensor_id,
	const double						&value)
{
	QString								text;
	
	text = QString("SENSOR %1 value: %2").arg(sensor_id).arg(value,0,'f',2);
	d_cmm_widget->Add_Log_Text(text);

	d_cmm_widget->Set_Sensor_Value(sensor_id,value,d_report_widget->Output_Path());
}

void TMeasureDirect::Driver_Sensors_X(
	const QString						&sensors)
{
	d_settings_widget->Set_Temperature_Sensors_X(sensors);
	d_cmm_widget->Set_Temperature_Sensors_X(sensors);
}

void TMeasureDirect::Driver_Sensors_Y(
	const QString						&sensors)
{
	d_settings_widget->Set_Temperature_Sensors_Y(sensors);
	d_cmm_widget->Set_Temperature_Sensors_Y(sensors);
}

void TMeasureDirect::Driver_Sensors_Z(
	const QString						&sensors)
{
	d_settings_widget->Set_Temperature_Sensors_Z(sensors);
	d_cmm_widget->Set_Temperature_Sensors_Z(sensors);
}

void TMeasureDirect::Driver_Sensors_Part(
	const QString						&sensors)
{
	d_settings_widget->Set_Temperature_Sensors_Part(sensors);
	d_cmm_widget->Set_Temperature_Sensors_Part(sensors);
}

void TMeasureDirect::Driver_Clear_Ipp_Tools(void)
{
	d_tool_widget->Clear_Ipp_Tools();
	
	d_tool_widget->setEnabled(false);
	d_measure_widget->setEnabled(false);
	d_cmm_widget->Enable_Options(false,false,true);
}

void TMeasureDirect::Driver_Add_Ipp_Tool(
	const QString						&tool)
{
	d_tool_widget->Add_Ipp_Tool(tool);
}

void TMeasureDirect::Driver_Add_Ipp_Tools_Complete(void)
{
	bool								is_homed;
	
	if(d_driver)
	{
		is_homed = d_driver->Is_Homed();

		d_tool_widget->setEnabled(is_homed);
		d_measure_widget->setEnabled(is_homed);
		d_cmm_widget->Enable_Options(true,is_homed,true);
	}
}

void TMeasureDirect::Exercise_Sequence_Start(void)
{
	std::vector<TDriver::TCommand>		commands;
	std::vector<TDriver::TCommand>::iterator iter;
	TDriver::TCommand					command;
		
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		if(d_driver->Is_EStop())
		{
			d_message_box->setText(QStringLiteral("Error - Severity 0"));
			d_message_box->setInformativeText(QStringLiteral("Machine is in E-Stop."));
			d_message_box->setDetailedText(QString());
			d_message_box->setStandardButtons(QMessageBox::Ok);
			d_message_box->setDefaultButton(QMessageBox::Ok);
			d_message_box->setIcon(QMessageBox::Warning);
			
			d_message_box->exec();

			d_exercise_dialog->Sequence_Canceled();
			return;
		}
		else if(!d_driver->Is_Homed())
		{
			d_message_box->setText(QStringLiteral("Home Not Set"));
			d_message_box->setInformativeText(QStringLiteral("Machine must be homed before it can be used."));
			d_message_box->setDetailedText(QString());
			d_message_box->setStandardButtons(QMessageBox::Ok);
			d_message_box->setDefaultButton(QMessageBox::Ok);
			d_message_box->setIcon(QMessageBox::Information);
			
			d_message_box->exec();

			d_exercise_dialog->Sequence_Canceled();
			return;
		}

		d_driver->Run();
		d_driver->Set_Command_Queue_Size(1);

		this->Machine_Settings_Changed();	// force settings
		
		d_message_box->setText(QStringLiteral("DCC Mode"));
		d_message_box->setInformativeText(QStringLiteral("Clear Probe For DCC Moves"));
		d_message_box->setDetailedText(QString());
		d_message_box->setStandardButtons(QMessageBox::Ok | QMessageBox::Cancel);
		d_message_box->setDefaultButton(QMessageBox::Ok);
		d_message_box->setIcon(QMessageBox::Warning);
		
		if(QMessageBox::Ok == d_message_box->exec())
		{
			command.command_type = TDriver::DRIVER_SET_MODE_DCC;
			d_driver->Add_Command(command);
			
			commands = d_exercise_dialog->Get_Next_Driver_Commands();
				
			for(iter = commands.begin();iter != commands.end();++iter)
			{
				d_driver->Add_Command(*iter);
			}
			
			command.command_type = TDriver::DRIVER_SET_MODE_MANUAL;
			d_driver->Add_Command(command);
		}
		else
		{
			d_exercise_dialog->Sequence_Canceled();
			d_driver->Set_Command_Queue_Size(-1);
		}
	}
}

void TMeasureDirect::Exercise_Sequence_Cancel(void)
{
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Stop();
		d_driver->Set_Command_Queue_Size(-1);
	}
}

void TMeasureDirect::Exercise_Sequence_End(void)
{
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		d_driver->Set_Command_Queue_Size(-1);
	}
}

void TMeasureDirect::Exercise_Get_Position(void)
{
	TDriver::TCommand					command;
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		command.command_type = TDriver::DRIVER_GET_POSITION;
		d_driver->Add_Command(command);
	}
}

void TMeasureDirect::Process_Queued_Commands(void)
{
	std::vector<TDriver::TCommand>		commands;
	std::vector<TDriver::TCommand>		commands_alt;
	std::vector<TDriver::TCommand>::iterator iter;
	bool								prev_state;
	
	if(d_driver && d_cmm_widget->Driver_State() == TDriver::DRIVER_STATE_CONNECTED)
	{
		prev_state = d_process_timer->blockSignals(true);
		
		commands = d_measure_dialog->Get_Next_Driver_Commands();
		commands_alt = d_laser_dialog->Get_Next_Driver_Commands();
		
		commands.insert(commands.end(),commands_alt.begin(),commands_alt.end());

		for(iter = commands.begin();iter != commands.end();++iter)
		{
			// not sending tool data for i++
			if(!((*iter).command_type == TDriver::DRIVER_SET_TOOL_TYPE && 
				d_cmm_widget->Controller_Type() == TController::CONTROLLER_IPPCLIENT))
			{
				d_driver->Add_Command(*iter);
			}
		}
		
		d_process_timer->blockSignals(prev_state);
	}
}

void TMeasureDirect::closeEvent(
	QCloseEvent							*event)
{
	TSettingsWidget::TMachineSettings	machine_settings;
	TSettingsWidget::TArtifactSettings	artifact_settings;
	TCMMWidget::TTemperatureSettings 	temperature_settings;
	TReportWidget::TOutputSettings		output_settings;
	TMeasureRoutines::TParameters		measure_parameters;
	TLaserDialog::TParameters			laser_parameters;
	TLinkDialog::TConnectionDefaults	connection_defaults;
	
	if(d_driver)
	{
		d_driver->Close_Driver();
		d_close_on_thread_finish = true;
		event->ignore();
		return;
	}
	
	d_settings->setValue(QStringLiteral("Mainwindow_Position"), this->pos());
	d_settings->setValue(QStringLiteral("Mainwindow_Size"), this->size());
	
	d_tool_widget->Save_Tool_Data(d_data_path);
	
	d_settings->setValue(QStringLiteral("Connect_Controller_Type"), d_cmm_widget->Controller_Type());
	
	connection_defaults = d_link_dialog->Connection_Defaults();
	
	d_settings->setValue(QStringLiteral("Connection_Default_Method"), connection_defaults.default_connection_method);
	d_settings->setValue(QStringLiteral("Connection_Default_Leitz_Serial"), connection_defaults.default_leitz_commport);
	d_settings->setValue(QStringLiteral("Connection_Default_Leitz_Port"), connection_defaults.default_leitz_port);
	d_settings->setValue(QStringLiteral("Connection_Default_Leitz_Address"), connection_defaults.default_leitz_address);
	d_settings->setValue(QStringLiteral("Connection_Default_DC_Address"), connection_defaults.default_dc_address);
	d_settings->setValue(QStringLiteral("Connection_Default_DC_Port"), connection_defaults.default_dc_port);
	d_settings->setValue(QStringLiteral("Connection_Default_Virtual_Address"), connection_defaults.default_virtual_address);
	d_settings->setValue(QStringLiteral("Connection_Default_Virtual_Port"), connection_defaults.default_virtual_port);
	d_settings->setValue(QStringLiteral("Connection_Default_IPP_Address"), connection_defaults.default_ipp_address);
	d_settings->setValue(QStringLiteral("Connection_Default_IPP_Port"), connection_defaults.default_ipp_port);

	machine_settings = d_settings_widget->Machine_Settings();

	d_settings->setValue(QStringLiteral("Machine_Move_Speed"), machine_settings.move_speed);
	d_settings->setValue(QStringLiteral("Machine_Touch_Speed"), machine_settings.touch_speed);
	d_settings->setValue(QStringLiteral("Machine_Acceleration"), machine_settings.acceleration);
	d_settings->setValue(QStringLiteral("Machine_Approach_Distance"), machine_settings.approach_distance);
	d_settings->setValue(QStringLiteral("Machine_Expansion_Coefficient_X"), machine_settings.expansion_coeff_x);
	d_settings->setValue(QStringLiteral("Machine_Expansion_Coefficient_Y"), machine_settings.expansion_coeff_y);
	d_settings->setValue(QStringLiteral("Machine_Expansion_Coefficient_Z"), machine_settings.expansion_coeff_z);
	d_settings->setValue(QStringLiteral("Machine_Temperature_Sensors_X"), machine_settings.sensors_x);
	d_settings->setValue(QStringLiteral("Machine_Temperature_Sensors_Y"), machine_settings.sensors_y);
	d_settings->setValue(QStringLiteral("Machine_Temperature_Sensors_Z"), machine_settings.sensors_z);
	d_settings->setValue(QStringLiteral("Machine_Temperature_Sensors_Part"), machine_settings.sensors_part);
	d_settings->setValue(QStringLiteral("Machine_Thermal_Compensation_Mode"), machine_settings.thermal_compensation_mode);
	d_settings->setValue(QStringLiteral("Machine_Blended_Move_Mode"), machine_settings.blended_move_mode);

	artifact_settings = d_settings_widget->Artifact_Settings();

	d_settings->setValue(QStringLiteral("Artifact_SG_Checkmaster_Plane_X_Start"), artifact_settings.sg_checkmaster_pln_x_start);
	d_settings->setValue(QStringLiteral("Artifact_SG_Checkmaster_Plane_X_End"), artifact_settings.sg_checkmaster_pln_x_end);
	d_settings->setValue(QStringLiteral("Artifact_SG_Checkmaster_Plane_Y1"), artifact_settings.sg_checkmaster_pln_y1);
	d_settings->setValue(QStringLiteral("Artifact_SG_Checkmaster_Plane_Y2"), artifact_settings.sg_checkmaster_pln_y2);
	d_settings->setValue(QStringLiteral("Artifact_SG_Checkmaster_Line_X_Start"), artifact_settings.sg_checkmaster_ln_x_start);
	d_settings->setValue(QStringLiteral("Artifact_SG_Checkmaster_Line_X_End"), artifact_settings.sg_checkmaster_ln_x_end);
	d_settings->setValue(QStringLiteral("Artifact_SG_Checkmaster_Line_Z"), artifact_settings.sg_checkmaster_ln_z);
	d_settings->setValue(QStringLiteral("Artifact_SG_Checkmaster_Point_Y"), artifact_settings.sg_checkmaster_pnt_y);
	d_settings->setValue(QStringLiteral("Artifact_SG_Checkmaster_Point_Z"), artifact_settings.sg_checkmaster_pnt_z);
	d_settings->setValue(QStringLiteral("Artifact_SG_Checkmaster_Clear_Z"), artifact_settings.sg_checkmaster_clear_z);
	
	d_settings->setValue(QStringLiteral("Artifact_SG_Webber_Clear_Z"), artifact_settings.sg_webber_clear_z);

	d_settings->setValue(QStringLiteral("Artifact_SG_Koba_Plane_X_Start"), artifact_settings.sg_koba_pln_x_start);
	d_settings->setValue(QStringLiteral("Artifact_SG_Koba_Plane_X_End"), artifact_settings.sg_koba_pln_x_end);
	d_settings->setValue(QStringLiteral("Artifact_SG_Koba_Plane_Y1"), artifact_settings.sg_koba_pln_y1);
	d_settings->setValue(QStringLiteral("Artifact_SG_Koba_Plane_Y2"), artifact_settings.sg_koba_pln_y2);
	d_settings->setValue(QStringLiteral("Artifact_SG_Koba_Line_X_Start"), artifact_settings.sg_koba_ln_x_start);
	d_settings->setValue(QStringLiteral("Artifact_SG_Koba_Line_X_End"), artifact_settings.sg_koba_ln_x_end);
	d_settings->setValue(QStringLiteral("Artifact_SG_Koba_Line_Z"), artifact_settings.sg_koba_ln_z);
	d_settings->setValue(QStringLiteral("Artifact_SG_Koba_Point_Y"), artifact_settings.sg_koba_pnt_y);
	d_settings->setValue(QStringLiteral("Artifact_SG_Koba_Point_Z"), artifact_settings.sg_koba_pnt_z);
	d_settings->setValue(QStringLiteral("Artifact_SG_Koba_Clear_Probe"), artifact_settings.sg_koba_clear_probe);
	
	d_settings->setValue(QStringLiteral("Artifact_Gauge_BLock_Plane_Y"), artifact_settings.gaugeblock_pln_y);
	d_settings->setValue(QStringLiteral("Artifact_Gauge_BLock_Plane_Z"), artifact_settings.gaugeblock_pln_z);
	d_settings->setValue(QStringLiteral("Artifact_Gauge_BLock_Spacing"), artifact_settings.gaugeblock_pln_sp);
	d_settings->setValue(QStringLiteral("Artifact_Gauge_BLock_Clear_Z"), artifact_settings.gaugeblock_clear_z);
	d_settings->setValue(QStringLiteral("Artifact_Ballbar_Clear_Distance"), artifact_settings.ballbar_clear_distance);
	d_settings->setValue(QStringLiteral("Artifact_Ballbar_Avoidance_Angle"), artifact_settings.ballbar_avoidance_angle);
	d_settings->setValue(QStringLiteral("Artifact_Ballbar_10360_Clear_Distance"), artifact_settings.ballbar_10360_clear_distance);
	d_settings->setValue(QStringLiteral("Artifact_Ballbar_10360_Sphere_Points"), artifact_settings.ballbar_10360_sphere_points);
	d_settings->setValue(QStringLiteral("Artifact_Ring_Gauge_Plane_Offset"), artifact_settings.ringgauge_pln_offset);
	d_settings->setValue(QStringLiteral("Artifact_Ring_Gauge_Circle_Z"), artifact_settings.ringgauge_cir_z);
	d_settings->setValue(QStringLiteral("Artifact_Ring_Gauge_Clear_Z"), artifact_settings.ringgauge_clear_z);
	d_settings->setValue(QStringLiteral("Artifact_Pin_Gauge_Clear_Z"), artifact_settings.pingauge_clear_z);
	d_settings->setValue(QStringLiteral("Artifact_Pin_Gauge_Plane_Offset"), artifact_settings.pingauge_pln_offset);
	d_settings->setValue(QStringLiteral("Artifact_Pin_Gauge_Circle_Z"), artifact_settings.pingauge_cir_z);
	d_settings->setValue(QStringLiteral("Plane_Sample_Radius"), artifact_settings.plane_sample_radius);
	d_settings->setValue(QStringLiteral("Artifact_Roll_Clear_Z"), artifact_settings.roll_clear_z);
	d_settings->setValue(QStringLiteral("Artifact_Roll_Sphere_Points"), artifact_settings.roll_sphere_points);
	d_settings->setValue(QStringLiteral("Artifact_Calibration_Sphere_Clear_Z"), artifact_settings.calibration_sphere_clear_z);
	d_settings->setValue(QStringLiteral("Artifact_Calibration_Sphere_Points"), artifact_settings.calibration_sphere_points);
	
	temperature_settings = d_cmm_widget->Temperature_Settings();
	
	d_settings->setValue(QStringLiteral("Temperature_Upper_Limit"), temperature_settings.upper_temperature_limit);
	d_settings->setValue(QStringLiteral("Temperature_Lower_Limit"), temperature_settings.lower_temperature_limit);

	output_settings = d_report_widget->Output_Settings();
	
	d_settings->setValue(QStringLiteral("Output_Relative_Path"), output_settings.output_relative_path);
	d_settings->setValue(QStringLiteral("Output_Absolute_Path"), output_settings.output_absolute_path);
	d_settings->setValue(QStringLiteral("Output_Use_Relative_Path"), output_settings.use_relative_path);

	measure_parameters = d_measure_dialog->Parameters();
	
	d_settings->setValue(QStringLiteral("Measurement_Axis_X_Temperature"), measure_parameters.axis_x_temperature);
	d_settings->setValue(QStringLiteral("Measurement_Axis_Y_Temperature"), measure_parameters.axis_y_temperature);
	d_settings->setValue(QStringLiteral("Measurement_Axis_Z_Temperature"), measure_parameters.axis_z_temperature);
	d_settings->setValue(QStringLiteral("Measurement_Part_Temperature"), measure_parameters.part_temperature);
	d_settings->setValue(QStringLiteral("Measurement_Sphere_Stem_I"), measure_parameters.sphere_stem_vector_i);
	d_settings->setValue(QStringLiteral("Measurement_Sphere_Stem_J"), measure_parameters.sphere_stem_vector_j);
	d_settings->setValue(QStringLiteral("Measurement_Sphere_Stem_K"), measure_parameters.sphere_stem_vector_k);
	d_settings->setValue(QStringLiteral("Measurement_Stepgauge_10360_Position_Name"), measure_parameters.stepgauge_10360_position_name);
	d_settings->setValue(QStringLiteral("Measurement_Stepgauge_B89_Position_Name"), measure_parameters.stepgauge_b89_position_name);
	d_settings->setValue(QStringLiteral("Measurement_Stepgauge_Square_Position_Name"), measure_parameters.stepgauge_square_position_name);
	d_settings->setValue(QStringLiteral("Measurement_Stepgauge_Serial_Number"), measure_parameters.stepgauge_serial_number);
	d_settings->setValue(QStringLiteral("Measurement_Stepgauge_Zero_Position"), measure_parameters.stepgauge_measurement_zero_position);
	d_settings->setValue(QStringLiteral("Measurement_Stepgauge_B89_Start"), measure_parameters.stepgauge_measurement_b89_start);
	d_settings->setValue(QStringLiteral("Measurement_Stepgauge_Zero_Step_Spacing"), measure_parameters.stepgauge_zero_step_spacing);
	d_settings->setValue(QStringLiteral("Measurement_Stepgauge_Nominal_Block_Spacing"), measure_parameters.stepgauge_nominal_block_spacing);
	d_settings->setValue(QStringLiteral("Measurement_Stepgauge_B89_Nominal_Block_Spacing"), measure_parameters.stepgauge_b89_nominal_block_spacing);
	d_settings->setValue(QStringLiteral("Measurement_Stepgauge_Expansion_Coefficient"), measure_parameters.stepgauge_expansion_coeff);
	d_settings->setValue(QStringLiteral("Measurement_Stepgauge_Type"),measure_parameters.stepgauge_type);
	d_settings->setValue(QStringLiteral("Measurement_Stepgauge_B89_Measurement_Count"), measure_parameters.stepgauge_b89_measurement_count);
	d_settings->setValue(QStringLiteral("Measurement_Stepgauge_Square_Measurement_Count"), measure_parameters.stepgauge_square_measurement_count);
	d_settings->setValue(QStringLiteral("Measurement_Gaugeblock_Position_Name"), measure_parameters.gaugeblock_position_name);
	d_settings->setValue(QStringLiteral("Measurement_Gaugeblock_Serial_Number"), measure_parameters.gaugeblock_serial_number);
	d_settings->setValue(QStringLiteral("Measurement_Gaugeblock_Expansion_Coefficient"), measure_parameters.gaugeblock_expansion_coeff);
	d_settings->setValue(QStringLiteral("Measurement_Gaugeblock_Nominal_Length"), measure_parameters.gaugeblock_nominal_length);
	d_settings->setValue(QStringLiteral("Measurement_Ballbar_Position_Name"), measure_parameters.ballbar_position_name);
	d_settings->setValue(QStringLiteral("Measurement_Ballbar_Expansion_Coefficient"), measure_parameters.ballbar_expansion_coeff);
	d_settings->setValue(QStringLiteral("Measurement_Ballbar_Sphere_Diameter"), measure_parameters.ballbar_sphere_diameter);
	d_settings->setValue(QStringLiteral("Measurement_Ballbar_Measurement_Count"), measure_parameters.ballbar_measurement_count);
	d_settings->setValue(QStringLiteral("Measurement_Ringgauge_Diameter"), measure_parameters.ringgauge_nominal_diameter);
	d_settings->setValue(QStringLiteral("Measurement_Ringgauge_Point_Count"), measure_parameters.ringgauge_point_count);
	d_settings->setValue(QStringLiteral("Measurement_Point_Repeat_Measurement_Count"), measure_parameters.point_repeat_measurement_count);
	d_settings->setValue(QStringLiteral("Measurement_Sphere_Repeat_Measurement_Count"), measure_parameters.sphere_repeat_measurement_count);
	d_settings->setValue(QStringLiteral("Measurement_Point_Repeat_Move_Distance"), measure_parameters.point_repeat_move_distance);
	d_settings->setValue(QStringLiteral("Measurement_Sphere_Nominal_Diameter"), measure_parameters.sphere_nominal_diameter);
	d_settings->setValue(QStringLiteral("Measurement_Pingauge_Diameter"), measure_parameters.pingauge_nominal_diameter);
	d_settings->setValue(QStringLiteral("Measurement_Pingauge_Point_Count"), measure_parameters.pingauge_point_count);
	d_settings->setValue(QStringLiteral("Measurement_Point_Repeat_Description"), measure_parameters.point_repeat_description);
	d_settings->setValue(QStringLiteral("Measurement_Roll_Tool1_Name"), measure_parameters.roll_tool1_name);
	d_settings->setValue(QStringLiteral("Measurement_Roll_Tool2_Name"), measure_parameters.roll_tool2_name);
	d_settings->setValue(QStringLiteral("Measurement_Ballbar_10360_Position_Name"), measure_parameters.ballbar_10360_position_name);
	d_settings->setValue(QStringLiteral("Measurement_Ballbar_10360_Serial_Number"), measure_parameters.ballbar_10360_serial_number);
	d_settings->setValue(QStringLiteral("Measurement_Ballbar_10360_Sphere_Diameter"), measure_parameters.ballbar_10360_sphere_diameter);
	d_settings->setValue(QStringLiteral("Measurement_Ballbar_10360_Sphere_Spacing"), measure_parameters.ballbar_10360_sphere_spacing);
	d_settings->setValue(QStringLiteral("Measurement_Ballbar_10360_Expansion_Coefficient"), measure_parameters.ballbar_10360_expansion_coeff);
	d_settings->setValue(QStringLiteral("Measurement_Ballbar_10360_Sphere0_Offset"), measure_parameters.ballbar_10360_sphere0_offset);

	d_settings->setValue(QStringLiteral("Measurement_Enable_Temperature_Compensation"), d_measure_dialog->Temperature_Compensation_State());
	
	laser_parameters = d_laser_dialog->Parameters();
	
	d_settings->setValue(QStringLiteral("Laser_Description"), laser_parameters.description);
	d_settings->setValue(QStringLiteral("Laser_Increment"), laser_parameters.increment);
	d_settings->setValue(QStringLiteral("Laser_Premove_Offset"), laser_parameters.premove_offset);
	d_settings->setValue(QStringLiteral("Laser_Start_X"), laser_parameters.start_x);
	d_settings->setValue(QStringLiteral("Laser_Start_Y"), laser_parameters.start_y);
	d_settings->setValue(QStringLiteral("Laser_Start_Z"), laser_parameters.start_z);
	d_settings->setValue(QStringLiteral("Laser_End_X"), laser_parameters.end_x);
	d_settings->setValue(QStringLiteral("Laser_End_Y"), laser_parameters.end_y);
	d_settings->setValue(QStringLiteral("Laser_End_Z"), laser_parameters.end_z);
	d_settings->setValue(QStringLiteral("Laser_Tool_Offset_X"), laser_parameters.tool_offset_x);
	d_settings->setValue(QStringLiteral("Laser_Tool_Offset_Y"), laser_parameters.tool_offset_y);
	d_settings->setValue(QStringLiteral("Laser_Tool_Offset_Z"), laser_parameters.tool_offset_z);
	d_settings->setValue(QStringLiteral("Laser_Measurement_Pattern"), laser_parameters.measurement_pattern);
	d_settings->setValue(QStringLiteral("Laser_Collection_Mode"), laser_parameters.data_collection_mode);
	d_settings->setValue(QStringLiteral("Laser_Measurement_Type"), laser_parameters.measurement_type);
	d_settings->setValue(QStringLiteral("Laser_Filter_Level"), laser_parameters.filter_level);

	event->accept();
}


