/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QProcess>
#include <QGridLayout>
#include <QLabel>
#include <QFrame>
#include <QSpacerItem>
#include <QDir>

#include "titlewidget.h"
#include "iconwidget.h"
#include "measure_outputdialog.h"
#include "validationdialog.h"

#include "reportwidget.h"

TReportWidget::TReportWidget(
	 const QWidget						*parent,
	 const Qt::WindowFlags				flags)
:QWidget(const_cast<QWidget*>(parent),flags)
{
	QGridLayout							*widget_layout;
	QSpacerItem							*control_hspacer;
	TTitleWidget						*title_widget;
		
	widget_layout = new QGridLayout(this);
	widget_layout->setContentsMargins(0,0,0,0);

	title_widget = new TTitleWidget(this);
	widget_layout->addWidget(title_widget,0,0,1,4);

	d_output_settings_option = new TIconWidget(this);
	d_output_settings_option->setSizePolicy(QSizePolicy(QSizePolicy::Maximum, QSizePolicy::Preferred));
	widget_layout->addWidget(d_output_settings_option,1,0,1,1);
	
	d_report_option = new TIconWidget(this);
	d_report_option->setSizePolicy(QSizePolicy(QSizePolicy::Maximum, QSizePolicy::Preferred));
	widget_layout->addWidget(d_report_option,1,1,1,1);
	
	d_validation_option = new TIconWidget(this);
	d_validation_option->setSizePolicy(QSizePolicy(QSizePolicy::Maximum, QSizePolicy::Preferred));
	widget_layout->addWidget(d_validation_option,1,2,1,1);
	
	control_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	widget_layout->addItem(control_hspacer,1,3,1,1);
	
	// defaults
	d_settings_output_dialog = 0;
	d_validation_dialog = 0;
	
	d_application_path = qApp->applicationDirPath();
	
#ifdef Q_OS_MAC
	int index = d_application_path.lastIndexOf(QStringLiteral(".app/"));
	d_application_path.truncate(index);
	
	index = d_application_path.lastIndexOf('/');
	d_application_path.truncate(index);
#endif
	
	d_output_settings.output_relative_path = QStringLiteral("/Measurement_Data");
	d_output_settings.output_absolute_path = d_application_path + QStringLiteral("/Measurement_Data");
	d_output_settings.use_relative_path = false;
	
	title_widget->Set_Text(QStringLiteral("Output, Reporting, and Validation"));
	
	d_output_settings_option->Set_Icon(QStringLiteral(":/icon64/output_icon64.png"));
	d_output_settings_option->Set_Text(QStringLiteral("Output\nPath"));
	
	d_report_option->Set_Icon(QStringLiteral(":/icon64/report_icon64.png"));
	d_report_option->Set_Text(QStringLiteral("Measurement\nReport"));
	
	d_validation_option->Set_Icon(QStringLiteral(":/icon64/validation_icon64.png"));
	d_validation_option->Set_Text(QStringLiteral("Best Fit\nValidation"));
	
	connect(d_output_settings_option,&TIconWidget::Clicked,this,&TReportWidget::Change_Output_Settings);
	connect(d_report_option,&TIconWidget::Clicked,this,&TReportWidget::Open_Report);
	connect(d_validation_option,&TIconWidget::Clicked,this,&TReportWidget::Validation_Test);
}

TReportWidget::~TReportWidget(void)
{
}

TReportWidget::TOutputSettings TReportWidget::Output_Settings(void) const
{
	return d_output_settings;
}

QString TReportWidget::Output_Path(void) const
{
	QDir								dir;

	if(d_output_settings.use_relative_path)
	{
		dir.setPath(d_application_path);
		
		if(dir.cd(d_output_settings.output_relative_path))
		{
			return dir.absolutePath();
		}
	}
	else
	{
		dir.setPath(d_output_settings.output_absolute_path);
		
		if(dir.exists())
		{
			return dir.absolutePath();
		}
	}
		
	// use generic path
	if(d_output_settings.use_relative_path)
	{
		dir.setPath(d_application_path);

		if(!dir.exists(QStringLiteral("Measurement_Data")))
		{
			dir.mkdir(QStringLiteral("Measurement_Data"));
		}

		dir.cd(QStringLiteral("Measurement_Data"));
	}
	else
	{
		dir.setPath(d_data_path);

		if(!dir.exists(QStringLiteral("Measurement_Data")))
		{
			dir.mkdir(QStringLiteral("Measurement_Data"));
		}

		dir.cd(QStringLiteral("Measurement_Data"));
	}
	
	return dir.absolutePath();
}


void TReportWidget::Set_Output_Settings(
	const QString						&data_path,
	const TReportWidget::TOutputSettings &settings)
{
	d_data_path = data_path;
	d_output_settings = settings;
}

void TReportWidget::Enable_Options(
	const bool							state)
{
	d_output_settings_option->setEnabled(state);
}

void TReportWidget::Change_Output_Settings(void)
{
	if(!d_settings_output_dialog)
	{
		d_settings_output_dialog = new TMeasureOutputDialog(this);
	}
	
	d_settings_output_dialog->Reset(d_application_path,d_output_settings);
	
	if(QDialog::Accepted == d_settings_output_dialog->exec())
	{
		d_output_settings = d_settings_output_dialog->Settings();
	}
}

void TReportWidget::Open_Report(void)
{
#ifdef Q_OS_MAC
	QProcess::startDetached(d_application_path + QStringLiteral("/measureview.app/Contents/MacOS/measureview"),QStringList());
#endif
	
#ifdef Q_OS_WIN
	QProcess::startDetached(QStringLiteral("measureview.exe"),QStringList());
#endif

#ifdef Q_OS_LINUX
	QProcess::startDetached(d_application_path + QStringLiteral("/measureview"),QStringList());
#endif
}

void TReportWidget::Validation_Test(void)
{	
	if(!d_validation_dialog)
	{
		d_validation_dialog = new TValidationDialog(this);
	}
	
	d_validation_dialog->Reset(this->Output_Path());
	d_validation_dialog->exec();
}

