/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QFrame>
#include <QGridLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QLineEdit>
#include <QPushButton>
#include <QComboBox>
#include <QSpacerItem>
#include <QList>
#include <set>

#include "settings_machinedialog.h"

TSettingsMachineDialog::TSettingsMachineDialog(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QDialog(const_cast<QWidget*>(parent),flags)
{
	QFrame								*separator_lower_hline;
	QFrame								*separator_upper_hline;
	QGridLayout							*dialog_layout;
	QHBoxLayout							*button_hlayout;
	QLabel								*acceleration_label;
	QLabel								*acceleration_units_label;
	QLabel								*approach_dist_label;
	QLabel								*approach_dist_units_label;
	QLabel								*expansion_coeff_x_label;
	QLabel								*expansion_coeff_x_units_label;
	QLabel								*expansion_coeff_y_label;
	QLabel								*expansion_coeff_y_units_label;
	QLabel								*expansion_coeff_z_label;
	QLabel								*expansion_coeff_z_units_label;
	QLabel								*move_speed_label;
	QLabel								*move_speed_units_label;
	QLabel								*sensors_part_label;
	QLabel								*sensors_x_label;
	QLabel								*sensors_y_label;
	QLabel								*sensors_z_label;
	QLabel								*touch_speed_label;
	QLabel								*touch_speed_units_label;
	QLabel								*blended_move_mode_label;
	QLabel								*compensation_mode_label;
	QPushButton							*cancel_button;
	QSpacerItem							*button_hspacer;
	QSpacerItem							*dailog_vspacer;
	QSizePolicy							sizepolicy_minimum_preferred(QSizePolicy::Minimum, QSizePolicy::Preferred);
	
	sizepolicy_minimum_preferred.setHorizontalStretch(0);
	sizepolicy_minimum_preferred.setVerticalStretch(0);

	this->resize(431,501);
	
	dialog_layout = new QGridLayout(this);
	
	move_speed_label = new QLabel(this);
	dialog_layout->addWidget(move_speed_label,0,0,1,1);
	
	d_move_speed_edit = new QLineEdit(this);
	d_move_speed_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_move_speed_edit,0,1,1,1);
	
	move_speed_units_label = new QLabel(this);
	dialog_layout->addWidget(move_speed_units_label,0,2,1,1);
	
	acceleration_label = new QLabel(this);
	dialog_layout->addWidget(acceleration_label,1,0,1,1);
	
	d_acceleration_edit = new QLineEdit(this);
	d_acceleration_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_acceleration_edit,1,1,1,1);
	
	acceleration_units_label = new QLabel(this);
	dialog_layout->addWidget(acceleration_units_label,1,2,1,1);
	
	touch_speed_label = new QLabel(this);
	dialog_layout->addWidget(touch_speed_label,2,0,1,1);
	
	d_touch_speed_edit = new QLineEdit(this);
	d_touch_speed_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_touch_speed_edit,2,1,1,1);
	
	touch_speed_units_label = new QLabel(this);
	dialog_layout->addWidget(touch_speed_units_label,2,2,1,1);
	
	approach_dist_label = new QLabel(this);
	dialog_layout->addWidget(approach_dist_label,3,0,1,1);
	
	d_approach_dist_edit = new QLineEdit(this);
	d_approach_dist_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_approach_dist_edit,3,1,1,1);
	
	approach_dist_units_label = new QLabel(this);
	dialog_layout->addWidget(approach_dist_units_label,3,2,1,1);
	
	blended_move_mode_label = new QLabel(this);
	dialog_layout->addWidget(blended_move_mode_label,4,0,1,1);
	
	d_blended_move_mode_combo = new QComboBox(this);
	dialog_layout->addWidget(d_blended_move_mode_combo,4,1,1,1);

	separator_upper_hline = new QFrame(this);
	separator_upper_hline->setFrameShape(QFrame::HLine);
	separator_upper_hline->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(separator_upper_hline,5,0,1,4);
	
	expansion_coeff_x_label = new QLabel(this);
	dialog_layout->addWidget(expansion_coeff_x_label,6,0,1,1);
	
	d_expansion_coeff_x_edit = new QLineEdit(this);
	d_expansion_coeff_x_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_expansion_coeff_x_edit,6,1,1,1);
	
	expansion_coeff_x_units_label = new QLabel(this);
	dialog_layout->addWidget(expansion_coeff_x_units_label,6,2,1,1);
	
	expansion_coeff_y_label = new QLabel(this);
	dialog_layout->addWidget(expansion_coeff_y_label,7,0,1,1);
	
	d_expansion_coeff_y_edit = new QLineEdit(this);
	d_expansion_coeff_y_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_expansion_coeff_y_edit,7,1,1,1);
	
	expansion_coeff_y_units_label = new QLabel(this);
	dialog_layout->addWidget(expansion_coeff_y_units_label,7,2,1,1);
	
	expansion_coeff_z_label = new QLabel(this);
	dialog_layout->addWidget(expansion_coeff_z_label,8,0,1,1);
	
	d_expansion_coeff_z_edit = new QLineEdit(this);
	d_expansion_coeff_z_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_expansion_coeff_z_edit,8,1,1,1);
	
	expansion_coeff_z_units_label = new QLabel(this);
	dialog_layout->addWidget(expansion_coeff_z_units_label,8,2,1,1);
	
	sensors_x_label = new QLabel(this);
	dialog_layout->addWidget(sensors_x_label,9,0,1,1);
	
	d_sensors_x_edit = new QLineEdit(this);
	d_sensors_x_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_sensors_x_edit,9,1,1,1);
	
	sensors_y_label = new QLabel(this);
	dialog_layout->addWidget(sensors_y_label,10,0,1,1);
	
	d_sensors_y_edit = new QLineEdit(this);
	d_sensors_y_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_sensors_y_edit,10,1,1,1);
	
	sensors_z_label = new QLabel(this);
	dialog_layout->addWidget(sensors_z_label,11,0,1,1);
	
	d_sensors_z_edit = new QLineEdit(this);
	d_sensors_z_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_sensors_z_edit,11,1,1,1);
	
	sensors_part_label = new QLabel(this);
	dialog_layout->addWidget(sensors_part_label,12,0,1,1);
	
	d_sensors_part_edit = new QLineEdit(this);
	d_sensors_part_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_sensors_part_edit,12,1,1,1);
	
	d_get_sensor_ids_button = new QPushButton(this);
	d_get_sensor_ids_button->setAutoDefault(false);
	sizepolicy_minimum_preferred.setHeightForWidth(d_get_sensor_ids_button->sizePolicy().hasHeightForWidth());
	d_get_sensor_ids_button->setSizePolicy(sizepolicy_minimum_preferred);
	dialog_layout->addWidget(d_get_sensor_ids_button,9,2,4,1);

	compensation_mode_label = new QLabel(this);
	dialog_layout->addWidget(compensation_mode_label,13,0,1,1);
	
	d_thermal_compensation_mode_combo = new QComboBox(this);
	dialog_layout->addWidget(d_thermal_compensation_mode_combo,13,1,1,1);

	separator_lower_hline = new QFrame(this);
	separator_lower_hline->setFrameShape(QFrame::HLine);
	separator_lower_hline->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(separator_lower_hline,14,0,1,4);
	
	dailog_vspacer = new QSpacerItem(0,0,QSizePolicy::Minimum,QSizePolicy::Expanding);
	dialog_layout->addItem(dailog_vspacer,15,0,1,4);
	
	button_hlayout = new QHBoxLayout();

	button_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	button_hlayout->addItem(button_hspacer);
	
	cancel_button = new QPushButton(this);
	cancel_button->setAutoDefault(false);
	button_hlayout->addWidget(cancel_button);
	
	d_apply_button = new QPushButton(this);
	d_apply_button->setAutoDefault(false);
	button_hlayout->addWidget(d_apply_button);
	dialog_layout->addLayout(button_hlayout,16,0,1,3);
	
	d_thermal_compensation_mode_combo->addItem(QStringLiteral("Compensation Not Active"),TSettingsWidget::CONTROLLER_THERMAL_COMPENSATION_OFF);
	d_thermal_compensation_mode_combo->addItem(QStringLiteral("Compensation Always Active"),TSettingsWidget::CONTROLLER_THERMAL_COMPENSATION_ACTIVE);
	
	d_thermal_compensation_mode_combo->setCurrentIndex(0);	// controller compensation inactive
	
	d_blended_move_mode_combo->addItem(QStringLiteral("Ignore Blended Moves"),TSettingsWidget::BLENDED_MOVE_IGNORED);
	d_blended_move_mode_combo->addItem(QStringLiteral("Enable Blended Moves"),TSettingsWidget::BLENDED_MOVE_ENABLE);
	d_blended_move_mode_combo->addItem(QStringLiteral("Disable Blended Moves"),TSettingsWidget::BLENDED_MOVE_DISABLE);
	
	d_blended_move_mode_combo->setCurrentIndex(1);	// blended moves enabled

	this->setWindowTitle(QStringLiteral("Machine Settings"));
	
	move_speed_label->setText(QStringLiteral("Move Speed:"));
	move_speed_units_label->setText(QStringLiteral("mm/sec"));
	acceleration_label->setText(QStringLiteral("Acceleration:"));
	acceleration_units_label->setText(QStringLiteral("mm/sec^2"));
	touch_speed_label->setText(QStringLiteral("Touch Speed:"));
	touch_speed_units_label->setText(QStringLiteral("mm/sec"));
	approach_dist_label->setText(QStringLiteral("Approach Distance:"));
	approach_dist_units_label->setText(QStringLiteral("mm"));
	expansion_coeff_x_label->setText(QStringLiteral("X Expansion Coefficient:"));
	expansion_coeff_x_units_label->setText(QStringLiteral("um/m/˚C"));
	expansion_coeff_y_label->setText(QStringLiteral("Y Expansion Coefficient:"));
	expansion_coeff_y_units_label->setText(QStringLiteral("um/m/˚C"));
	expansion_coeff_z_label->setText(QStringLiteral("Z Expansion Coefficient:"));
	expansion_coeff_z_units_label->setText(QStringLiteral("um/m/˚C"));
	sensors_x_label->setText(QStringLiteral("X Temperature Sensor(s):"));
	sensors_y_label->setText(QStringLiteral("Y Temperature Sensor(s):"));
	sensors_z_label->setText(QStringLiteral("Z Temperature Sensor(s):"));
	sensors_part_label->setText(QStringLiteral("Part Temperature Sensor(s):"));
	blended_move_mode_label->setText(QStringLiteral("Blended Move Mode (Fly):"));
	compensation_mode_label->setText(QStringLiteral("Controller Thermal Mode:"));
	d_get_sensor_ids_button->setText(QStringLiteral("Get Sensor ID's"));
	cancel_button->setText(QStringLiteral("Cancel"));
	d_apply_button->setText(QStringLiteral("Apply"));

	connect(cancel_button,&QPushButton::clicked,this,&TSettingsMachineDialog::reject);
	connect(d_apply_button,&QPushButton::clicked,this,&TSettingsMachineDialog::accept);
	
	connect(d_move_speed_edit,&QLineEdit::textEdited,this,&TSettingsMachineDialog::Options_Changed);
	connect(d_touch_speed_edit,&QLineEdit::textEdited,this,&TSettingsMachineDialog::Options_Changed);
	connect(d_acceleration_edit,&QLineEdit::textEdited,this,&TSettingsMachineDialog::Options_Changed);
	connect(d_approach_dist_edit,&QLineEdit::textEdited,this,&TSettingsMachineDialog::Options_Changed);
	
	connect(d_expansion_coeff_x_edit,&QLineEdit::textEdited,this,&TSettingsMachineDialog::Options_Changed);
	connect(d_expansion_coeff_y_edit,&QLineEdit::textEdited,this,&TSettingsMachineDialog::Options_Changed);
	connect(d_expansion_coeff_z_edit,&QLineEdit::textEdited,this,&TSettingsMachineDialog::Options_Changed);
	connect(d_sensors_x_edit,&QLineEdit::textEdited,this,&TSettingsMachineDialog::Options_Changed);
	connect(d_sensors_y_edit,&QLineEdit::textEdited,this,&TSettingsMachineDialog::Options_Changed);
	connect(d_sensors_z_edit,&QLineEdit::textEdited,this,&TSettingsMachineDialog::Options_Changed);
	connect(d_sensors_part_edit,&QLineEdit::textEdited,this,&TSettingsMachineDialog::Options_Changed);
	
	connect(d_thermal_compensation_mode_combo,static_cast<void (QComboBox::*)(int)>(&QComboBox::currentIndexChanged),this,&TSettingsMachineDialog::Thermal_Mode_Changed);
	connect(d_blended_move_mode_combo,static_cast<void (QComboBox::*)(int)>(&QComboBox::currentIndexChanged),this,&TSettingsMachineDialog::Move_Mode_Changed);
	connect(d_get_sensor_ids_button,&QPushButton::clicked,this,&TSettingsMachineDialog::Request_Sensor_IDs);
}

TSettingsMachineDialog::~TSettingsMachineDialog(void)
{
}

TSettingsWidget::TMachineSettings TSettingsMachineDialog::Settings(void) const
{
	return d_settings;
}

void TSettingsMachineDialog::Reset(
	const TSettingsWidget::TMachineSettings	&settings,
	const TController::TControllerType	controller_type)
{
	d_settings = settings;
	
	d_move_speed_edit->setText(QString("%1").arg(d_settings.move_speed,0,'f',0));
	d_touch_speed_edit->setText(QString("%1").arg(d_settings.touch_speed,0,'f',0));
	d_acceleration_edit->setText(QString("%1").arg(d_settings.acceleration,0,'f',0));
	d_approach_dist_edit->setText(QString("%1").arg(d_settings.approach_distance,0,'f',0));

	d_expansion_coeff_x_edit->setText(QString("%1").arg(d_settings.expansion_coeff_x,0,'f',1));
	d_expansion_coeff_y_edit->setText(QString("%1").arg(d_settings.expansion_coeff_y,0,'f',1));
	d_expansion_coeff_z_edit->setText(QString("%1").arg(d_settings.expansion_coeff_z,0,'f',1));

	d_sensors_x_edit->setText(TSettingsMachineDialog::Sensor_Data(d_settings.sensors_x));
	d_sensors_y_edit->setText(TSettingsMachineDialog::Sensor_Data(d_settings.sensors_y));
	d_sensors_z_edit->setText(TSettingsMachineDialog::Sensor_Data(d_settings.sensors_z));
	d_sensors_part_edit->setText(TSettingsMachineDialog::Sensor_Data(d_settings.sensors_part));
	
	if(d_settings.thermal_compensation_mode == TSettingsWidget::CONTROLLER_THERMAL_COMPENSATION_ACTIVE)
	{
		d_thermal_compensation_mode_combo->setCurrentIndex(1);	// controller compensation active
	}
	else
	{
		d_thermal_compensation_mode_combo->setCurrentIndex(0);	// controller compensation inactive
	}
	
	switch(d_settings.blended_move_mode)
	{
		case 0:
		default:
			d_blended_move_mode_combo->setCurrentIndex(0);
			break;
			
		case 1:
			d_blended_move_mode_combo->setCurrentIndex(1);
			break;
			
		case 2:
			d_blended_move_mode_combo->setCurrentIndex(2);
			break;
	}
	
	switch(controller_type)
	{			
		case TController::CONTROLLER_IPPCLIENT:
			d_thermal_compensation_mode_combo->setEnabled(true);
			d_blended_move_mode_combo->setEnabled(false);
			d_get_sensor_ids_button->setEnabled(true);
			break;
			
		case TController::CONTROLLER_DC:
		case TController::CONTROLLER_LEITZ:
		case TController::CONTROLLER_VIRTUAL:
			d_thermal_compensation_mode_combo->setEnabled(true);
			d_blended_move_mode_combo->setEnabled(true);
			d_get_sensor_ids_button->setEnabled(true);
			break;
	}

}

void TSettingsMachineDialog::Set_Temperature_Sensors_X(
	const QString						&sensors)
{
	d_sensors_x_edit->setText(sensors);
	d_settings.sensors_x = sensors;
}

void TSettingsMachineDialog::Set_Temperature_Sensors_Y(
	const QString						&sensors)
{
	d_sensors_y_edit->setText(sensors);
	d_settings.sensors_y = sensors;
}

void TSettingsMachineDialog::Set_Temperature_Sensors_Z(
	const QString						&sensors)
{
	d_sensors_z_edit->setText(sensors);
	d_settings.sensors_z = sensors;
}

void TSettingsMachineDialog::Set_Temperature_Sensors_Part(
	const QString						&sensors)
{
	d_sensors_part_edit->setText(sensors);
	d_settings.sensors_part = sensors;
}

void TSettingsMachineDialog::Options_Changed(void)
{
	QString								sensors;
	
	// validation checks
	d_settings.move_speed = d_move_speed_edit->text().toDouble();
	d_settings.touch_speed = d_touch_speed_edit->text().toDouble();
	d_settings.acceleration = d_acceleration_edit->text().toDouble();
	d_settings.approach_distance = d_approach_dist_edit->text().toDouble();
	
	d_settings.expansion_coeff_x = d_expansion_coeff_x_edit->text().toDouble();
	d_settings.expansion_coeff_y = d_expansion_coeff_y_edit->text().toDouble();
	d_settings.expansion_coeff_z = d_expansion_coeff_z_edit->text().toDouble();
	d_settings.sensors_x = TSettingsMachineDialog::Sensor_Data(d_sensors_x_edit->text());
	d_settings.sensors_y = TSettingsMachineDialog::Sensor_Data(d_sensors_y_edit->text());
	d_settings.sensors_z = TSettingsMachineDialog::Sensor_Data(d_sensors_z_edit->text());
	d_settings.sensors_part = TSettingsMachineDialog::Sensor_Data(d_sensors_part_edit->text());

	d_apply_button->setEnabled(true);

	if(d_settings.move_speed < 1 || d_settings.move_speed > 500)
	{
		d_move_speed_edit->setStyleSheet(QStringLiteral("background-color: rgb(255, 102, 102);"));
		d_apply_button->setEnabled(false);
	}
	else
	{
		d_move_speed_edit->setStyleSheet(QString());
	}
	
	if(d_settings.touch_speed < 0.1 || d_settings.touch_speed > 20)
	{
		d_touch_speed_edit->setStyleSheet(QStringLiteral("background-color: rgb(255, 102, 102);"));
		d_apply_button->setEnabled(false);
	}
	else
	{
		d_touch_speed_edit->setStyleSheet(QString());
	}

	if(d_settings.acceleration < 10 || d_settings.acceleration > 1000)
	{
		d_acceleration_edit->setStyleSheet(QStringLiteral("background-color: rgb(255, 102, 102);"));
		d_apply_button->setEnabled(false);
	}
	else
	{
		d_acceleration_edit->setStyleSheet(QString());
	}

	if(d_settings.approach_distance < 0.5 || d_settings.approach_distance > 10)
	{
		d_approach_dist_edit->setStyleSheet(QStringLiteral("background-color: rgb(255, 102, 102);"));
		d_apply_button->setEnabled(false);
	}
	else
	{
		d_approach_dist_edit->setStyleSheet(QString());
	}
	
	if(d_settings.expansion_coeff_x < 0 || d_settings.expansion_coeff_x > 30)
	{
		d_expansion_coeff_x_edit->setStyleSheet(QStringLiteral("background-color: rgb(255, 102, 102);"));
		d_apply_button->setEnabled(false);
	}
	else
	{
		d_expansion_coeff_x_edit->setStyleSheet(QString());
	}
	
	if(d_settings.expansion_coeff_y < 0 || d_settings.expansion_coeff_y > 30)
	{
		d_expansion_coeff_y_edit->setStyleSheet(QStringLiteral("background-color: rgb(255, 102, 102);"));
		d_apply_button->setEnabled(false);
	}
	else
	{
		d_expansion_coeff_y_edit->setStyleSheet(QString());
	}

	if(d_settings.expansion_coeff_z < 0 || d_settings.expansion_coeff_z > 30)
	{
		d_expansion_coeff_z_edit->setStyleSheet(QStringLiteral("background-color: rgb(255, 102, 102);"));
		d_apply_button->setEnabled(false);
	}
	else
	{
		d_expansion_coeff_z_edit->setStyleSheet(QString());
	}
	
	sensors = d_settings.sensors_x;
	sensors.append(',');
	sensors += d_settings.sensors_y;
	sensors.append(',');
	sensors += d_settings.sensors_z;
	sensors.append(',');
	sensors += d_settings.sensors_part;
	
	if(!this->Validate_Sensor_List(sensors))
	{
		d_sensors_x_edit->setStyleSheet(QStringLiteral("background-color: rgb(255, 102, 102);"));
		d_sensors_y_edit->setStyleSheet(QStringLiteral("background-color: rgb(255, 102, 102);"));
		d_sensors_z_edit->setStyleSheet(QStringLiteral("background-color: rgb(255, 102, 102);"));
		d_sensors_part_edit->setStyleSheet(QStringLiteral("background-color: rgb(255, 102, 102);"));
		d_apply_button->setEnabled(false);
	}
	else
	{
		d_sensors_x_edit->setStyleSheet(QString());
		d_sensors_y_edit->setStyleSheet(QString());
		d_sensors_z_edit->setStyleSheet(QString());
		d_sensors_part_edit->setStyleSheet(QString());
	}
}

void TSettingsMachineDialog::Thermal_Mode_Changed(void)
{
	if(d_thermal_compensation_mode_combo->currentIndex() == 0)
	{
		d_settings.thermal_compensation_mode = TSettingsWidget::CONTROLLER_THERMAL_COMPENSATION_OFF;
	}
	else
	{
		d_settings.thermal_compensation_mode = TSettingsWidget::CONTROLLER_THERMAL_COMPENSATION_ACTIVE;
	}
}

void TSettingsMachineDialog::Move_Mode_Changed(void)
{
	d_settings.blended_move_mode = TSettingsWidget::BLENDED_MOVE_IGNORED;

	switch(d_blended_move_mode_combo->currentIndex())
	{
		case 0:
			break;
			
		case 1:
			d_settings.blended_move_mode = TSettingsWidget::BLENDED_MOVE_ENABLE;
			break;
			
		case 2:
			d_settings.blended_move_mode = TSettingsWidget::BLENDED_MOVE_DISABLE;
			break;
	}
}

bool TSettingsMachineDialog::Validate_Sensor_List(
	QString								&sensors)
{
	std::set<int>						sensor_ids;
	QList<QString>::const_iterator		iter;
	QStringList							list;
	int									sensor_id;
	bool								valid;
	
	list = sensors.split(',');
	
	for(iter = list.begin();iter != list.end();++iter)
	{
		sensor_id = (*iter).toInt(&valid);
		
		if(!valid)
		{
			return false;
		}
		
		if(sensor_id > 0)
		{
			if(sensor_ids.find(sensor_id) != sensor_ids.end())
			{
				return false;
			}
			
			sensor_ids.insert(sensor_id);
		}
	}
	
	return true;
}

QString TSettingsMachineDialog::Sensor_Data(
	const QString						&text)
{
	QString								sensor_text;
	QStringList							list;
	int									sensor_id;
	int									cntr;
	bool								init(true);
	bool								valid;
	
	list = text.split(',');
	
	for(cntr = 0;cntr < list.size();++cntr)
	{
		sensor_id = list[cntr].toInt(&valid);
		
		if(valid)
		{
			if(init)
			{
				sensor_text = QString("%1").arg(sensor_id);
				init = false;
			}
			else
			{
				sensor_text += QString(",%1").arg(sensor_id);
			}
		}
	}
	
	if(init)
	{
		sensor_text = QStringLiteral("-1");
	}
	
	return sensor_text;
}


