/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QFrame>
#include <QGridLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QLineEdit>
#include <QPlainTextEdit>
#include <QPushButton>
#include <QSizePolicy>
#include <QSpacerItem>
#include <cmath>
#include <assert.h>

#include "tooladdanglesdialog.h"

TToolAddAnglesDialog::TToolAddAnglesDialog(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QDialog(const_cast<QWidget*>(parent),flags)
{
	QFrame								*lower_hline;
	QFrame								*upper_hline;
	QFrame								*right_vline;
	QGridLayout							*dialog_layout;
	QHBoxLayout							*button_hlayout;
	QLabel								*angle_list_label;
	QLabel								*angle_a_label;
	QLabel								*angle_b_label;
	QLabel								*end_a_label;
	QLabel								*end_b_label;
	QLabel								*increment_a_label;
	QLabel								*increment_b_label;
	QLabel								*range_title_label;
	QLabel								*single_title_label;
	QLabel								*start_a_label;
	QLabel								*start_b_label;
	QPushButton							*cancel_button;
	QPushButton							*add_angle_button;
	QPushButton							*add_angles_button;
	QSizePolicy							size_policy_minimum_minimum(QSizePolicy::Minimum,QSizePolicy::Minimum);
	QSpacerItem							*button_hspacer;
	QSpacerItem							*dialog_vspacer;

	size_policy_minimum_minimum.setHorizontalStretch(0);
	size_policy_minimum_minimum.setVerticalStretch(0);

	this->resize(591,341);

	dialog_layout = new QGridLayout(this);

	single_title_label = new QLabel(this);
	single_title_label->setAlignment(Qt::AlignLeading|Qt::AlignLeft|Qt::AlignTop);
	dialog_layout->addWidget(single_title_label,0,0,2,1);

	angle_a_label = new QLabel(this);
	angle_a_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	dialog_layout->addWidget(angle_a_label,0,1,1,1);

	d_angle_a_edit = new QLineEdit(this);
	d_angle_a_edit->setAlignment(Qt::AlignCenter);
	d_angle_a_edit->setMinimumSize(QSize(80, 0));
	dialog_layout->addWidget(d_angle_a_edit,0,2,1,1);

	angle_b_label = new QLabel(this);
	angle_b_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	dialog_layout->addWidget(angle_b_label,1,1,1,1);

	d_angle_b_edit = new QLineEdit(this);
	d_angle_b_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_angle_b_edit,1,2,1,1);
	
	add_angle_button = new QPushButton(this);
	add_angle_button->setSizePolicy(size_policy_minimum_minimum);
	add_angle_button->setAutoDefault(false);
	dialog_layout->addWidget(add_angle_button,0,3,2,1);

	upper_hline = new QFrame(this);
	upper_hline->setFrameShape(QFrame::HLine);
	upper_hline->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(upper_hline,2,0,2,4);

	range_title_label = new QLabel(this);
	range_title_label->setAlignment(Qt::AlignLeading|Qt::AlignLeft|Qt::AlignTop);
	dialog_layout->addWidget(range_title_label,4,0,6,1);

	start_a_label = new QLabel(this);
	start_a_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	dialog_layout->addWidget(start_a_label,4,1,1,1);

	d_start_a_edit = new QLineEdit(this);
	d_start_a_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_start_a_edit,4,2,1,1);

	end_a_label = new QLabel(this);
	end_a_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	dialog_layout->addWidget(end_a_label,5,1,1,1);

	d_end_a_edit = new QLineEdit(this);
	d_end_a_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_end_a_edit,5,2,1,1);

	increment_a_label = new QLabel(this);
	increment_a_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	dialog_layout->addWidget(increment_a_label,6,1,1,1);

	d_increment_a_edit = new QLineEdit(this);
	d_increment_a_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_increment_a_edit,6,2,1,1);

	start_b_label = new QLabel(this);
	start_b_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	dialog_layout->addWidget(start_b_label,7,1,1,1);

	d_start_b_edit = new QLineEdit(this);
	d_start_b_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_start_b_edit,7,2,1,1);

	end_b_label = new QLabel(this);
	end_b_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	dialog_layout->addWidget(end_b_label,8,1,1,1);

	d_end_b_edit = new QLineEdit(this);
	d_end_b_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_end_b_edit,8,2,1,1);

	increment_b_label = new QLabel(this);
	increment_b_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	dialog_layout->addWidget(increment_b_label,9,1,1,1);

	d_increment_b_edit = new QLineEdit(this);
	d_increment_b_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_increment_b_edit,9,2,1,1);
	
	add_angles_button = new QPushButton(this);
	add_angles_button->setSizePolicy(size_policy_minimum_minimum);
	add_angles_button->setAutoDefault(false);
	dialog_layout->addWidget(add_angles_button,3,3,7,1);
	
	right_vline = new QFrame(this);
	right_vline->setFrameShape(QFrame::VLine);
	right_vline->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(right_vline,0,4,10,1);
	
	angle_list_label = new QLabel(this);
	dialog_layout->addWidget(angle_list_label,0,5,1,1);

	d_angle_list_edit = new QPlainTextEdit(this);
	d_angle_list_edit->setUndoRedoEnabled(false);
	d_angle_list_edit->setLineWrapMode(QPlainTextEdit::NoWrap);
//	d_angle_list_edit->setFont(text_font);
	dialog_layout->addWidget(d_angle_list_edit,1,5,9,1);
	
	lower_hline = new QFrame(this);
	lower_hline->setFrameShape(QFrame::HLine);
	lower_hline->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(lower_hline,10,0,1,6);

	dialog_vspacer = new QSpacerItem(0,0,QSizePolicy::Minimum,QSizePolicy::Expanding);
	dialog_layout->addItem(dialog_vspacer,11,0,1,6);

	button_hlayout = new QHBoxLayout();

	button_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	button_hlayout->addItem(button_hspacer);

	cancel_button = new QPushButton(this);
	cancel_button->setAutoDefault(false);
	button_hlayout->addWidget(cancel_button);
	
	d_accept_button =new QPushButton(this);
	d_accept_button->setAutoDefault(true);
	button_hlayout->addWidget(d_accept_button);

	dialog_layout->addLayout(button_hlayout,12,0,1,6);

	this->setWindowTitle(QStringLiteral("Add Angles"));
	
	single_title_label->setText(QStringLiteral("Single:"));
	angle_a_label->setText(QStringLiteral("Angle A:"));
	angle_b_label->setText(QStringLiteral("Angle B:"));
	add_angles_button->setText(QStringLiteral("Add Angles"));
	range_title_label->setText(QStringLiteral("Range:"));
	start_a_label->setText(QStringLiteral("Start A:"));
	end_a_label->setText(QStringLiteral("End A:"));
	increment_a_label->setText(QStringLiteral("Increment A:"));
	start_b_label->setText(QStringLiteral("Start B:"));
	end_b_label->setText(QStringLiteral("End B:"));
	increment_b_label->setText(QStringLiteral("Increment B:"));
	d_accept_button->setText(QStringLiteral("Accept"));
	cancel_button->setText(QStringLiteral("Cancel"));
	add_angle_button->setText(QStringLiteral("Add Angle"));
	angle_a_label->setText(QStringLiteral("Angle A:"));
	angle_a_label->setText(QStringLiteral("Angle A:"));
	angle_list_label->setText(QStringLiteral("New Angles:"));
	
	d_angle_a_edit->setText(QStringLiteral("0.0"));
	d_angle_b_edit->setText(QStringLiteral("0.0"));
	d_start_a_edit->setText(QStringLiteral("45.0"));
	d_end_a_edit->setText(QStringLiteral("45.0"));
	d_increment_a_edit->setText(QStringLiteral("45.0"));
	d_start_b_edit->setText(QStringLiteral("-135.0"));
	d_end_b_edit->setText(QStringLiteral("135.0"));
	d_increment_b_edit->setText(QStringLiteral("90.0"));
	
	connect(cancel_button,&QPushButton::clicked,this,&TToolAddAnglesDialog::reject);
	connect(d_accept_button,&QPushButton::clicked,this,&TToolAddAnglesDialog::accept);
	
	connect(add_angle_button,&QPushButton::clicked,this,&TToolAddAnglesDialog::Add_Angle);
	connect(add_angles_button,&QPushButton::clicked,this,&TToolAddAnglesDialog::Add_Angles);
}

TToolAddAnglesDialog::~TToolAddAnglesDialog(void)
{
}

void TToolAddAnglesDialog::Reset(void)
{
	d_accept_button->setEnabled(false);
	
	d_angles.clear();
	d_angle_signatures.clear();
	d_angle_list_edit->clear();
}

void TToolAddAnglesDialog::Add_Angle(void)
{
	TAngle								angle;
	int									angle_signature;
	
	angle.angle_a = d_angle_a_edit->text().toDouble();
	angle.angle_b = d_angle_b_edit->text().toDouble();
	
	// validate
	if(angle.angle_a < 0.0) return;
	if(angle.angle_b < -180.0) return;
	if(angle.angle_a > 105.0) return;
	if(angle.angle_b > 180.0) return;
	
	angle_signature = this->Angle_Signature(angle.angle_a,angle.angle_b);
	
	if(d_angle_signatures.find(angle_signature) == d_angle_signatures.end())
	{
		d_angles.push_back(angle);
		d_angle_signatures.insert(angle_signature);
	}
	
	d_accept_button->setEnabled(d_angles.size() > 0);
	this->Update_Angle_List();
}

void TToolAddAnglesDialog::Add_Angles(void)
{
	TAngle								angle;
	double								start_a;
	double								end_a;
	double								incr_a;
	double								start_b;
	double								end_b;
	double								incr_b;
	double								range;
	double								incr;
	double								angle_a;
	double								angle_b;
	int									count_a;
	int									cntr_a;
	int									count_b;
	int									cntr_b;
	int									angle_signature;
	static const double					MIN_INCREMENT(2.5);
	
	start_a = d_start_a_edit->text().toDouble();
	end_a = d_end_a_edit->text().toDouble();
	incr_a = d_increment_a_edit->text().toDouble();

	start_b = d_start_b_edit->text().toDouble();
	end_b = d_end_b_edit->text().toDouble();
	incr_b = d_increment_b_edit->text().toDouble();

	// validate
	if(start_a < 0.0) return;
	if(end_a < 0.0) return;
	if(start_a > 105.0) return;
	if(end_a > 105.0) return;

	if(start_b < -180.0) return;
	if(end_b < -180.0) return;
	if(start_b > 180.0) return;
	if(end_b > 180.0) return;
	
	// A angles
	range = fabs(end_a - start_a);
	incr = fabs(incr_a);
	
	if(incr < MIN_INCREMENT)
	{
		incr = MIN_INCREMENT;
	}
	
	count_a = 1 + static_cast<int>(0.5 + range / incr);
	
	// verify increment sign
	if(end_a > start_a)
	{
		incr_a = incr;
	}
	else
	{
		incr_a = -1.0 * incr;
	}
	
	// B angles
	range = fabs(end_b - start_b);
	incr = fabs(incr_b);
	
	if(incr < MIN_INCREMENT)
	{
		incr = MIN_INCREMENT;
	}

	count_b = 1 + static_cast<int>(0.5 + range / incr);
	
	// verify increment sign
	if(end_b > start_b)
	{
		incr_b = incr;
	}
	else
	{
		incr_b = -1.0 * incr;
	}
	
	angle_b = start_b;
	
	for(cntr_b = 0;cntr_b < count_b;++cntr_b)
	{
		angle_a = start_a;
		
		for(cntr_a = 0;cntr_a < count_a;++cntr_a)
		{
			angle.angle_a = angle_a;
			angle.angle_b = angle_b;

			angle_signature = this->Angle_Signature(angle.angle_a,angle.angle_b);
			
			if(d_angle_signatures.find(angle_signature) == d_angle_signatures.end())
			{
				d_angles.push_back(angle);
				d_angle_signatures.insert(angle_signature);
			}
			
			angle_a += incr_a;
		}
		
		angle_b += incr_b;
	}

	this->Update_Angle_List();
	d_accept_button->setEnabled(d_angles.size() > 0);
}


int TToolAddAnglesDialog::Angle_Signature(
	const double						&angle_a,
	const double						&angle_b)
{
	int									result(0);
	short								*angle_ab(reinterpret_cast<short*>(&result));
	
	// convert two floating point values to a single integer
	// simplier to search or sort angles
	// resolution is 0.5 degrees.
	// largest range is -180 to 180
	
	// angles made all positive by adding 180.0 to A and B
	
	angle_ab[0] = static_cast<short>((angle_a + 180.0) * 2.0 + 0.5);
	angle_ab[1] = static_cast<short>((angle_b + 180.0) * 2.0 + 0.5);
	
	return result;
}

void TToolAddAnglesDialog::Update_Angle_List(void)
{
	std::vector<TAngle>::const_iterator iter;
	TToolAddAnglesDialog::TAngle		angles;
	QString								text;
	bool								prev_state;
	
	d_angle_list_edit->clear();
	
	prev_state = d_angle_list_edit->blockSignals(true);

	for(iter = d_angles.begin();iter != d_angles.end();++iter)
	{
		angles = (*iter);
		
		text = QString("A%1_B%2").arg(angles.angle_a,0,'f',1).arg(angles.angle_b,0,'f',1);
		
		d_angle_list_edit->appendPlainText(text);
	}
	
	d_angle_list_edit->blockSignals(prev_state);
}


