/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QFrame>
#include <QGridLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QListWidget>
#include <QListWidgetItem>
#include <QPushButton>
#include <QSpacerItem>
#include <QTreeWidget>
#include <QHeaderView>
#include <QList>
#include <QSize>
#include <QIcon>

#include "toolbuilderdialog.h"

static const int						USERROLE_CONNECTION_REQUIRED(Qt::UserRole);
static const int						USERROLE_CONNECTION_PROVIDES(Qt::UserRole+1);
static const int						USERROLE_COMPONENT_TYPE(Qt::UserRole+2);

TToolBuilderDialog::TToolBuilderDialog(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QDialog(const_cast<QWidget*>(parent),flags)
{
	QFrame								*separator_hline;
	QGridLayout							*dialog_layout;
	QHBoxLayout							*button_hlayout;
	QLabel								*tool_build_label;
	QLabel								*tool_components_label;
	QPushButton							*cancel_button;
	QSpacerItem							*button_hspacer;

	this->resize(563,463);

	dialog_layout = new QGridLayout(this);

	tool_build_label = new QLabel(this);
	dialog_layout->addWidget(tool_build_label,0,0,1,1);

	tool_components_label = new QLabel(this);
	dialog_layout->addWidget(tool_components_label,0,1,1,1);

	d_tool_build_tree = new QTreeWidget(this);
	d_tool_build_tree->header()->setVisible(false);
	d_tool_build_tree->setIndentation(0);
	d_tool_build_tree->setColumnCount(2);
	d_tool_build_tree->setIconSize(QSize(32,32));
	dialog_layout->addWidget(d_tool_build_tree,1,0,1,1);

	d_tool_components_list = new QListWidget(this);
	dialog_layout->addWidget(d_tool_components_list,1,1,1,1);

	separator_hline = new QFrame(this);
	separator_hline->setFrameShape(QFrame::HLine);
	separator_hline->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(separator_hline,2,0,1,2);

	button_hlayout = new QHBoxLayout();

	button_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	button_hlayout->addItem(button_hspacer);

	cancel_button = new QPushButton(this);
	cancel_button->setAutoDefault(false);
	button_hlayout->addWidget(cancel_button);

	d_accept_button = new QPushButton(this);
	d_accept_button->setAutoDefault(false);
	button_hlayout->addWidget(d_accept_button);
	dialog_layout->addLayout(button_hlayout,3,0,1,2);

	this->setWindowTitle(QStringLiteral("Tool Configuration Builder"));
	
	tool_build_label->setText(QStringLiteral("Tool Build"));
	tool_components_label->setText(QStringLiteral("Tool Components"));

	cancel_button->setText(QStringLiteral("Cancel"));
	d_accept_button->setText(QStringLiteral("Accept"));
	
	
	// defaults
	d_tool_build_top_item = new QTreeWidgetItem(d_tool_build_tree);
	d_tool_build_top_item->setData(0,USERROLE_CONNECTION_REQUIRED,TToolBuilderData::CONNECT_NOTHING);
	d_tool_build_top_item->setData(0,USERROLE_CONNECTION_PROVIDES,TToolBuilderData::CONNECT_CMM);
	d_tool_build_top_item->setData(0,USERROLE_COMPONENT_TYPE,TToolBuilderData::COMPONENT_CMM);
	d_tool_build_top_item->setText(0,QStringLiteral("CMM"));
	d_tool_build_top_item->setIcon(1,QIcon(":/builder/cmm_icon32.png"));

	d_tool_build_tree->setCurrentItem(d_tool_build_top_item);

	this->Build_Selection_Changed();
	d_accept_button->setEnabled(false);
	
	connect(d_tool_build_tree,&QTreeWidget::itemSelectionChanged,this,&TToolBuilderDialog::Build_Selection_Changed);
	connect(d_tool_components_list,&QListWidget::itemDoubleClicked,this,&TToolBuilderDialog::Component_Selection_Changed);
	
	connect(cancel_button,&QPushButton::clicked,this,&TToolBuilderDialog::reject);
	connect(d_accept_button,&QPushButton::clicked,this,&TToolBuilderDialog::accept);
}

TToolBuilderDialog::~TToolBuilderDialog(void)
{
}

TVector3 TToolBuilderDialog::Fixed_Offset(void) const
{
	TVector3							offset;
	QString								name;
	TToolBuilderData::TConnectionType	connection_required;
	TToolBuilderData::TConnectionType	connection_provides;

	QTreeWidgetItem						*item;
	
	item = d_tool_build_top_item;
	
	while(item->childCount())
	{
		item = item->child(0);
		name = item->text(0);
		
		connection_required = static_cast<TToolBuilderData::TConnectionType>(item->data(0,USERROLE_CONNECTION_REQUIRED).toInt());
		connection_provides = static_cast<TToolBuilderData::TConnectionType>(item->data(0,USERROLE_CONNECTION_PROVIDES).toInt());

		offset += d_tool_builder_data.Get_Fixed_Offset(name,connection_required,connection_provides);
	}
	
	return offset;
}

TVector3 TToolBuilderDialog::Rotational_Offset(void) const
{
	TVector3							offset;
	QString								name;
	TToolBuilderData::TConnectionType	connection_required;
	TToolBuilderData::TConnectionType	connection_provides;
	
	QTreeWidgetItem						*item;
	
	item = d_tool_build_top_item;
	
	while(item->childCount())
	{
		item = item->child(0);
		name = item->text(0);
		
		connection_required = static_cast<TToolBuilderData::TConnectionType>(item->data(0,USERROLE_CONNECTION_REQUIRED).toInt());
		connection_provides = static_cast<TToolBuilderData::TConnectionType>(item->data(0,USERROLE_CONNECTION_PROVIDES).toInt());
		
		offset += d_tool_builder_data.Get_Rotational_Offset(name,connection_required,connection_provides);
	}
	
	return offset;
}

QString TToolBuilderDialog::Configuration_Name(void) const
{
	QString								name;
	QTreeWidgetItem						*item;
	
	item = d_tool_build_top_item;
	
	while(item->childCount())
	{
		item = item->child(0);
		
		if(name.length())
		{
			name += QString(" + %1").arg(item->text(0));
		}
		else
		{
			name = QString("%1").arg(item->text(0));
		}
	}
	
	return name;
}

void TToolBuilderDialog::Reset(
	const QStringList					&existing_names)
{
	QTreeWidgetItem						*child;

	d_existing_configuration_names = existing_names;
	d_accept_button->setEnabled(false);
	
	// remove any children from top level item
	while(d_tool_build_top_item->childCount())
	{
		child = d_tool_build_top_item->takeChild(0);
		delete child;
	}
	
	d_tool_build_tree->setCurrentItem(d_tool_build_top_item);
}

void TToolBuilderDialog::Build_Selection_Changed(void)
{
	QList<QTreeWidgetItem*>				selected_items;
	TToolBuilderData::TConnectionType	connection_required;
	TToolBuilderData::TConnectionType	connection_provides;
	std::vector<TToolBuilderData::TComponent> components;
	std::vector<TToolBuilderData::TComponent>::const_iterator iter;
	QTreeWidgetItem						*builder_item;
	QListWidgetItem						*component_item;
	
	selected_items = d_tool_build_tree->selectedItems();
	
	d_tool_components_list->clear();
	
	if(selected_items.size() == 1)
	{
		builder_item = selected_items[0];
		
		connection_required = static_cast<TToolBuilderData::TConnectionType>(builder_item->data(0,USERROLE_CONNECTION_REQUIRED).toInt());
		connection_provides = static_cast<TToolBuilderData::TConnectionType>(builder_item->data(0,USERROLE_CONNECTION_PROVIDES).toInt());
		
		components = d_tool_builder_data.Connection_Components(connection_provides);
		
		for(iter = components.begin();iter != components.end();++iter)
		{
			component_item = new QListWidgetItem(d_tool_components_list);
			component_item->setText((*iter).name);
			component_item->setData(USERROLE_CONNECTION_REQUIRED,(*iter).connection_required);
			component_item->setData(USERROLE_CONNECTION_PROVIDES,(*iter).connection_provides);
			component_item->setData(USERROLE_COMPONENT_TYPE,(*iter).component_type);
		}
	}
}

void TToolBuilderDialog::Component_Selection_Changed(void)
{
	QList<QListWidgetItem*>				selected_items;
	QListWidgetItem						*item;
	QTreeWidgetItem						*current_item;
	QTreeWidgetItem						*builder_item;
	QTreeWidgetItem						*child;
	TToolBuilderData::TConnectionType	connection_required;
	TToolBuilderData::TConnectionType	connection_provides;
	TToolBuilderData::TComponentType	component_type;

	selected_items = d_tool_components_list->selectedItems();
	
	if(selected_items.size() == 1)
	{
		item = selected_items[0];
		
		connection_required = static_cast<TToolBuilderData::TConnectionType>(item->data(USERROLE_CONNECTION_REQUIRED).toInt());
		connection_provides = static_cast<TToolBuilderData::TConnectionType>(item->data(USERROLE_CONNECTION_PROVIDES).toInt());
		component_type = static_cast<TToolBuilderData::TComponentType>(item->data(USERROLE_COMPONENT_TYPE).toInt());
		
		current_item = d_tool_build_tree->currentItem();
		
		if(current_item)
		{
			// remove any children of current item
			while(current_item->childCount())
			{
				child = current_item->takeChild(0);
				delete child;
			}
			
			builder_item = new QTreeWidgetItem(current_item);
			builder_item->setData(0,USERROLE_CONNECTION_REQUIRED,connection_required);
			builder_item->setData(0,USERROLE_CONNECTION_PROVIDES,connection_provides);
			builder_item->setData(0,USERROLE_COMPONENT_TYPE,component_type);
			builder_item->setText(0,item->text());
			
			switch(component_type)
			{
				case TToolBuilderData::COMPONENT_CMM:
					builder_item->setIcon(1,QIcon(":/builder/cmm_icon32.png"));
					break;
					
				case TToolBuilderData::COMPONENT_PROBE_HEAD:
					builder_item->setIcon(1,QIcon(":/builder/probehead_icon32.png"));
					break;
					
				case TToolBuilderData::COMPONENT_QUICK_CONNECT:
					builder_item->setIcon(1,QIcon(":/builder/quickconnect_icon32.png"));
					break;
					
				case TToolBuilderData::COMPONENT_EXTENSION:
					builder_item->setIcon(1,QIcon(":/builder/extension_icon32.png"));
					break;
					
				case TToolBuilderData::COMPONENT_PROBE:
					builder_item->setIcon(1,QIcon(":/builder/probe_icon32.png"));
					break;
					
				case TToolBuilderData::COMPONENT_STYLUS:
					builder_item->setIcon(1,QIcon(":/builder/stylus_icon32.png"));
					break;
					
			}
			
			builder_item->setExpanded(true);
			d_tool_build_tree->setCurrentItem(builder_item);
			
			if(connection_provides == TToolBuilderData::CONNECT_TIP)
			{
				if(d_existing_configuration_names.contains(this->Configuration_Name()))
				{
					d_accept_button->setEnabled(false);
				}
				else
				{
					d_accept_button->setEnabled(true);
				}
			}
		}
	}
}

