/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QFile>
#include <algorithm>
#include <assert.h>

#include "../../../core/xmlfile.h"

#include "tooldata.h"

// sort by A then B angles
struct TToolData_Sort
{
	bool operator() (TToolData::TToolEntry a,TToolData::TToolEntry b)
	{
		double							t1_angle_a,t2_angle_a;
		double							t1_angle_b,t2_angle_b;

		TToolData::Decode_AB_Angles(&t1_angle_a,&t1_angle_b,a.angle_ab_signature);
		TToolData::Decode_AB_Angles(&t2_angle_a,&t2_angle_b,b.angle_ab_signature);
		
		if(t1_angle_a == t2_angle_a)
		{
			return (t1_angle_b < t2_angle_b);
		}
		
		return (t1_angle_a < t2_angle_a);
	}
};

TToolData::TToolData(void)
{
	d_a0b0_orientation = TToolData::AXIS_ZMINUS;
	d_a90b0_orientation = TToolData::AXIS_YMINUS;
	d_orientation_valid = true;
	d_motorized_probe_head = true;
}

TToolData::~TToolData(void)
{
}

bool TToolData::Save_Tool_Data(
	const QString						&path) const
{
	QString								file_name;
	QString								text;
	TXmlFile							xml_file;
	QDomElement							e0;
	QDomElement							e1;
	QDomElement							e2;
	QDomElement							e3;
	QDomElement							e4;
	std::vector<TToolData::TConfiguration>::const_iterator iter_config;
	std::vector<TToolData::TTool>::const_iterator iter_tool;
	std::vector<TToolData::TToolEntry>::const_iterator iter_tool_entry;
	
	file_name = path;
	file_name.replace('\\','/');
	
	if(file_name.endsWith('/'))
	{
		file_name.append("tool_data.xml");
	}
	else
	{
		file_name.append("/tool_data.xml");
	}
	
	try
	{
		xml_file.Set_File_Name(file_name);
		xml_file.Reset(QStringLiteral("Probe_Data"),1);
		
		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Orientation"));
		
		e1 = xml_file.Create_Node(&e0,QStringLiteral("A0B0"));
		xml_file.Write_Text_Node(&e1,this->Axis_To_String(d_a0b0_orientation));
		
		e1 = xml_file.Create_Node(&e0,QStringLiteral("A90B0"));
		xml_file.Write_Text_Node(&e1,this->Axis_To_String(d_a90b0_orientation));

		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Options"));
		e1 = xml_file.Create_Node(&e0,QStringLiteral("Motorized_Probe_Head"));
		xml_file.Write_Text_Node(&e1,d_motorized_probe_head ? QStringLiteral("Yes"):QStringLiteral("No"));
		
		// configuraitons
		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Configurations"));

		for(iter_config = d_configurations.begin();iter_config != d_configurations.end();++iter_config)
		{
			e1 = xml_file.Create_Node(&e0,QStringLiteral("Configuration"));
			
			e2 = xml_file.Create_Node(&e1,QStringLiteral("Name"));
			xml_file.Write_Text_Node(&e2,(*iter_config).configuration_name);
			
			e2 = xml_file.Create_Node(&e1,QStringLiteral("Offset_Fixed"));
			xml_file.Write_Text_Node(&e2,QString("%1,%2,%3").arg((*iter_config).offset_fixed.x,0,'f',3).arg((*iter_config).offset_fixed.y,0,'f',3).arg((*iter_config).offset_fixed.z,0,'f',3));
			
			e2 = xml_file.Create_Node(&e1,QStringLiteral("Offset_Rotational"));
			xml_file.Write_Text_Node(&e2,QString("%1,%2,%3").arg((*iter_config).offset_rotational.x,0,'f',3).arg((*iter_config).offset_rotational.y,0,'f',3).arg((*iter_config).offset_rotational.z,0,'f',3));
		}
		
		// tools
		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Tools"));
		
		for(iter_tool = d_tools.begin();iter_tool != d_tools.end();++iter_tool)
		{
			e1 = xml_file.Create_Node(&e0,QStringLiteral("Tool"));
			
			e2 = xml_file.Create_Node(&e1,QStringLiteral("Name"));
			xml_file.Write_Text_Node(&e2,(*iter_tool).tool_base_name);
			
			e2 = xml_file.Create_Node(&e1,QStringLiteral("Configuration"));
			xml_file.Write_Text_Node(&e2,(*iter_tool).configuration_name);

			e2 = xml_file.Create_Node(&e1,QStringLiteral("ToolData"));

			for(iter_tool_entry = (*iter_tool).tool_entries.begin();iter_tool_entry != (*iter_tool).tool_entries.end();++iter_tool_entry)
			{
				e3 = xml_file.Create_Node(&e2,QStringLiteral("Position"));

				e4 = xml_file.Create_Node(&e3,QStringLiteral("Angle_AB"));
				xml_file.Write_Text_Node(&e4,QString("%1").arg((*iter_tool_entry).angle_ab_signature));
				
				e4 = xml_file.Create_Node(&e3,QStringLiteral("Tip_Diameter"));
				xml_file.Write_Text_Node(&e4,QString("%1").arg((*iter_tool_entry).tip_diameter,0,'f',5));
			}

		}
		
		xml_file.Write_File();
	}
	catch(TXmlFileException exception)
	{
		
		d_last_error = QString("ERR Save_Tool_Data: %1").arg(exception.ErrorText()).toLatin1();
		return false;
	}
	
	return true;
}

QStringList TToolData::Tool_Base_Names(void) const
{
	QStringList							list;
	std::vector<TToolData::TTool>::const_iterator iter;
	
	for(iter = d_tools.begin();iter != d_tools.end();++iter)
	{
		list.push_back((*iter).tool_base_name);
	}
	
	return list;
}

TToolData::TTool TToolData::Tool(
	const QString						&tool_base_name,
	bool								*valid) const
{
	TToolData::TTool					tool;
	std::vector<TToolData::TTool>::const_iterator iter;
	
	if(valid)
	{
		*valid = false;
	}
	
	for(iter = d_tools.begin();iter != d_tools.end();++iter)
	{
		if((*iter).tool_base_name == tool_base_name)
		{
			tool = (*iter);
			
			if(valid)
			{
				*valid = true;
			}
			
			break;
		}
	}
	
	return tool;
}

TToolData::TTool TToolData::Tool(
	const int							index) const
{
	assert(static_cast<unsigned int>(index) < d_tools.size());
	
	return d_tools[index];
}

QStringList TToolData::Offset_Configuration_Names(void) const
{
	QStringList							list;
	std::vector<TConfiguration>::const_iterator iter;
	
	for(iter = d_configurations.begin();iter != d_configurations.end();++iter)
	{
		list.push_back((*iter).configuration_name);
	}
	
	return list;
}

TToolData::TConfiguration TToolData::Offset_Configuration(
	const QString						&configuration_name,
	bool								*valid) const
{
	TToolData::TConfiguration			offset_configuration;
	std::vector<TToolData::TConfiguration>::const_iterator iter;

	if(valid)
	{
		*valid = false;
	}
	
	for(iter = d_configurations.begin();iter != d_configurations.end();++iter)
	{
		if((*iter).configuration_name == configuration_name)
		{
			offset_configuration = (*iter);
			
			if(valid)
			{
				*valid = true;
			}
			
			break;
		}
	}
	
	offset_configuration.offset_fixed = this->To_Orientation(offset_configuration.offset_fixed);
	offset_configuration.offset_rotational = this->To_Orientation(offset_configuration.offset_rotational);
	
	return offset_configuration;
}

TToolData::TToolEntry TToolData::Tool_Entry(
	const QString						&tool_base_name,
	const double						&angle_a,
	const double						&angle_b,
	TVector3							* const offset,
	TVector3							* const vector,
	bool								*valid) const
{
	std::vector<TToolData::TTool>::const_iterator iter;
	std::vector<TToolData::TToolEntry>::const_iterator iter_tool_entry;
	TToolData::TToolEntry				tool_entry;
	int									angle_ab_signature;
	bool								offset_valid(false);
	bool								vector_valid(false);
	
	this->Encode_AB_Angles(angle_a,angle_b,&angle_ab_signature);

	if(valid)
	{
		*valid = false;
	}
	
	for(iter = d_tools.begin();iter != d_tools.end();++iter)
	{
		if((*iter).tool_base_name == tool_base_name)
		{
			for(iter_tool_entry = (*iter).tool_entries.begin();iter_tool_entry != (*iter).tool_entries.end();++iter_tool_entry)
			{				
				if((*iter_tool_entry).angle_ab_signature == angle_ab_signature)
				{
					tool_entry = (*iter_tool_entry);
					
					if(offset)
					{
						*offset = this->Calculate_Offset((*iter).configuration_name,angle_a,angle_b,&offset_valid);
					}
					
					if(vector)
					{
						*vector = this->Calculate_Vector((*iter).configuration_name,angle_a,angle_b,&vector_valid);
					}

					if(valid)
					{
						*valid = offset_valid && vector_valid;
					}
					
					break;

				}
			}
		}
	}
	
	return tool_entry;
}

bool TToolData::Load_Tool_Data(
	const QString						&path)
{
	QString								file_name;
	TXmlFile							xml_file;
	QDomElement							element;
	int									version;
	QString								text;
	
	file_name = path;
	file_name.replace('\\','/');
	
	if(file_name.endsWith('/'))
	{
		file_name.append(QStringLiteral("tool_data.xml"));
	}
	else
	{
		file_name.append(QStringLiteral("/tool_data.xml"));
	}
	
	d_tools.clear();
	d_configurations.clear();
	
	if(!QFile::exists(file_name))
	{
		this->Generate_Default_Offset_Configurations();
		this->Generate_Default_Tools();
		
		return true;
	}
	
	try
	{
		xml_file.Set_File_Name(file_name);
		xml_file.Open(QStringLiteral("Probe_Data"),&version);
		
		switch(version)
		{
			case 1:
				this->Load_Data_V1(&xml_file);
				break;
				
			default:
				throw TXmlFileException(QStringLiteral("ERR:  File version not recognized"));
				break;
		}
	}
	catch(TXmlFileException exception)
	{
		d_last_error = QString("ERR Load_Tool_Data: %1").arg(exception.ErrorText()).toLatin1();
		return false;
	}
	
	this->Generate_Default_Offset_Configurations();
	this->Generate_Default_Tools();
		
	return true;
}

bool TToolData::Set_Probe_Head_Orientation(
	const TToolData::TAxis				a0b0,
	const TToolData::TAxis				a90b0)
{
	d_a0b0_orientation = a0b0;
	d_a90b0_orientation = a90b0;
	
	d_orientation_valid = this->Update_Orientation_Mat();
	
	return d_orientation_valid;
}

void TToolData::Add_Tool(
	const QString						&tool_base_name,
	const QString						&configuration_name)
{
	TToolData::TTool					tool;
	
	tool.tool_base_name = tool_base_name;
	tool.configuration_name = configuration_name;
	
	// not checking for duplicates.
	d_tools.push_back(tool);
}

void TToolData::Add_Tool_Angle(
	const QString						&tool_base_name,
	const double						&angle_a,
	const double						&angle_b,
	const double						&tip_diameter)
{
	std::vector<TToolData::TTool>::iterator iter;
	TToolData_Sort						tool_data_sort;
	TToolData::TToolEntry				tool_entry;
	int									angle_ab_signature;
	
	this->Encode_AB_Angles(angle_a,angle_b,&angle_ab_signature);
	
	for(iter = d_tools.begin();iter != d_tools.end();++iter)
	{
		if((*iter).tool_base_name == tool_base_name)
		{
			if(!this->ToolData_Contains_Angle((*iter).tool_entries,angle_ab_signature))
			{
				tool_entry.angle_ab_signature = angle_ab_signature;
				tool_entry.tip_diameter = tip_diameter;
			
				(*iter).tool_entries.push_back(tool_entry);
				
				std::sort((*iter).tool_entries.begin(), (*iter).tool_entries.end(), tool_data_sort);
			}
			return;
		}
	}
}

bool TToolData::Remove_Tool_Angle(
	const QString						&tool_base_name,
	const double						&angle_a,
	const double						&angle_b)
{
	std::vector<TToolData::TTool>::iterator iter;
	std::vector<TToolData::TToolEntry>::iterator iter_tool_entry;
	int									angle_ab_signature;
	
	this->Encode_AB_Angles(angle_a,angle_b,&angle_ab_signature);

	for(iter = d_tools.begin();iter != d_tools.end();++iter)
	{
		if((*iter).tool_base_name == tool_base_name)
		{
			for(iter_tool_entry = (*iter).tool_entries.begin();iter_tool_entry != (*iter).tool_entries.end();++iter_tool_entry)
			{
				if((*iter_tool_entry).angle_ab_signature == angle_ab_signature)
				{
					(*iter).tool_entries.erase(iter_tool_entry);
					return true;
				}
			}
			
			return false;
		}
	}
	
	return false;
}

void TToolData::Update_Tool_Tip_Diameter(
	const QString						&tool_base_name,
	const double						&angle_a,
	const double						&angle_b,
	const double						&tip_diameter)
{
	std::vector<TTool>::iterator		iter;
	std::vector<TToolData::TToolEntry>::iterator iter_tool_entry;
	int									angle_ab_signature;

	this->Encode_AB_Angles(angle_a,angle_b,&angle_ab_signature);

	for(iter = d_tools.begin();iter != d_tools.end();++iter)
	{
		if((*iter).tool_base_name == tool_base_name)
		{
			for(iter_tool_entry = (*iter).tool_entries.begin();iter_tool_entry != (*iter).tool_entries.end();++iter_tool_entry)
			{
				if((*iter_tool_entry).angle_ab_signature == angle_ab_signature)
				{
					(*iter_tool_entry).tip_diameter = tip_diameter;
					
					break;
				}
			}
			return;
		}
	}
}

void TToolData::Remove_Tool(
	const QString						&tool_base_name)
{
	std::vector<TTool>::iterator 		iter;
	
	for(iter = d_tools.begin();iter != d_tools.end();++iter)
	{
		if((*iter).tool_base_name == tool_base_name)
		{
			d_tools.erase(iter);
			return;
		}
	}
}

void TToolData::Update_Tool_Configuration(
	const QString						&tool_base_name,
	const QString						&configuration_name)
{
	std::vector<TTool>::iterator 		iter;
		
	for(iter = d_tools.begin();iter != d_tools.end();++iter)
	{
		if((*iter).tool_base_name == tool_base_name)
		{
			(*iter).configuration_name = configuration_name;
		}
	}
}

void TToolData::Update_Configuration(
	const QString						&configuration_name,
	const TVector3						&offset_fixed,
	const TVector3						&offset_rotational)
{
	TToolData::TConfiguration			offset_configuration;
	std::vector<TToolData::TConfiguration>::iterator iter;
	
	if(!configuration_name.length())
	{
		return;
	}

	for(iter = d_configurations.begin();iter != d_configurations.end();++iter)
	{
		if((*iter).configuration_name == configuration_name)
		{
			(*iter).offset_fixed = this->From_Orientation(offset_fixed);
			(*iter).offset_rotational = this->From_Orientation(offset_rotational);
			
			return;
		}
	}
	
	// create new entry
	offset_configuration.configuration_name = configuration_name;
	
	offset_configuration.offset_fixed = this->From_Orientation(offset_fixed);
	offset_configuration.offset_rotational = this->From_Orientation(offset_rotational);
	
	d_configurations.push_back(offset_configuration);
}

void TToolData::Remove_Configuration(
	const QString						&configuration_name)
{
	std::vector<TConfiguration>::iterator iter;

	for(iter = d_configurations.begin();iter != d_configurations.end();++iter)
	{
		if((*iter).configuration_name == configuration_name)
		{
			d_configurations.erase(iter);
			return;
		}
	}
}

void TToolData::Encode_AB_Angles(
	const double						&angle_a,
	const double						&angle_b,
	int									* const encoded_value)
{
	int									result;
	short								*angle_ab(reinterpret_cast<short*>(&result));
	
	assert(encoded_value);
	
	// convert two floating point values to a single integer
	// simplier to search or sort angles
	// resolution is 0.5 degrees.
	// largest range is -180 to 180

	// angles made all positive by adding 180.0 to A and B
	
	angle_ab[0] = static_cast<short>((angle_a + 180.0) * 2.0 + 0.5);
	angle_ab[1] = static_cast<short>((angle_b + 180.0) * 2.0 + 0.5);
	
	*encoded_value = result;
}

void TToolData::Decode_AB_Angles(
	double								* const angle_a,
	double								* const angle_b,
	const int							encoded_value)
{
	int									value;
	short								*angle_ab(reinterpret_cast<short*>(&value));

	assert(angle_a);
	assert(angle_b);
	
	value = encoded_value;
	
	*angle_a = static_cast<double>(angle_ab[0]) / 2.0 - 180.0;
	*angle_b = static_cast<double>(angle_ab[1]) / 2.0 - 180.0;
}

TVector3 TToolData::Get_Axis(
	const TToolData::TAxis				axis) const
{
	TVector3							vec;
	
	switch(axis)
	{
		case TToolData::AXIS_XMINUS:
			vec.Set(-1,0,0);
			break;
			
		case TToolData::AXIS_XPLUS:
			vec.Set(1,0,0);
			break;
			
		case TToolData::AXIS_YMINUS:
			vec.Set(0,-1,0);
			break;
			
		case TToolData::AXIS_YPLUS:
			vec.Set(0,1,0);
			break;
			
		case TToolData::AXIS_ZMINUS:
			vec.Set(0,0,-1);
			break;
			
		case TToolData::AXIS_ZPLUS:
			vec.Set(0,0,1);
			break;
	}
	
	return vec;
}

QString TToolData::Axis_To_String(
	const TToolData::TAxis				axis) const
{
	QString								text;
	
	switch(axis)
	{
		case TToolData::AXIS_XMINUS:
			text = QStringLiteral("XMINUS");
			break;
			
		case TToolData::AXIS_XPLUS:
			text = QStringLiteral("XPLUS");
			break;
			
		case TToolData::AXIS_YMINUS:
			text = QStringLiteral("YMINUS");
			break;
			
		case TToolData::AXIS_YPLUS:
			text = QStringLiteral("YPLUS");
			break;
			
		case TToolData::AXIS_ZMINUS:
			text = QStringLiteral("ZMINUS");
			break;
			
		case TToolData::AXIS_ZPLUS:
		default:
			text = QStringLiteral("ZPLUS");
			break;
	}
	
	return text;
}

TToolData::TAxis TToolData::String_To_Axis(
	const QString						&text) const
{
	TToolData::TAxis					axis;
	
	if(text.compare(QStringLiteral("XMINUS")) == 0)
	{
		axis = TToolData::AXIS_XMINUS;
	}
	else if(text.compare(QStringLiteral("XPLUS")) == 0)
	{
		axis = TToolData::AXIS_XPLUS;
	}
	else if(text.compare(QStringLiteral("YMINUS")) == 0)
	{
		axis = TToolData::AXIS_YMINUS;
	}
	else if(text.compare(QStringLiteral("YPLUS")) == 0)
	{
		axis = TToolData::AXIS_YPLUS;
	}
	else if(text.compare(QStringLiteral("ZMINUS")) == 0)
	{
		axis = TToolData::AXIS_ZMINUS;
	}
	else
	{
		axis = TToolData::AXIS_ZPLUS;
	}

	return axis;
}

bool TToolData::ToolData_Contains_Angle(
	const std::vector<TToolData::TToolEntry> &tool_entry,
	const int							angle_ab_signature) const
{
	std::vector<TToolData::TToolEntry>::const_iterator iter;
	
	for(iter = tool_entry.begin();iter != tool_entry.end();++iter)
	{
		if((*iter).angle_ab_signature == angle_ab_signature)
		{
			return true;
		}
	}
	
	return false;
}

TVector3 TToolData::To_Orientation(
	const TVector3						&offset) const
{
	TMat4								mat(d_orientation_mat);

	mat.Transpose();
	
	return mat * offset;
}

TVector3 TToolData::From_Orientation(
	const TVector3						&offset) const
{
	return d_orientation_mat * offset;
}

TVector3 TToolData::Calculate_Offset(
	const QString						&configuration_name,
	const double						&angle_a,
	const double						&angle_b,
	bool 								* const valid) const
{
	std::vector<TConfiguration>::const_iterator iter;
	TVector3							offset;
	TVector3							offset_rotational;
	TMat4								rot_mat;
	TMat4								inv_mat;
	static const double					DEG_TO_RAD(-0.01745329251994);
	
	assert(valid);
	*valid = false;
	
	for(iter = d_configurations.begin();iter != d_configurations.end();++iter)
	{
		if((*iter).configuration_name == configuration_name)
		{
			rot_mat.Rotate(rot_mat.Y(),angle_a * DEG_TO_RAD);
			rot_mat.Rotate(TVector3(0,0,1),angle_b * DEG_TO_RAD);
			
			inv_mat = d_orientation_mat;
			inv_mat.Transpose();
			
			offset = inv_mat * (*iter).offset_fixed;
			
			offset_rotational = rot_mat * (*iter).offset_rotational;
			offset_rotational = inv_mat * offset_rotational;
			
			offset += offset_rotational;
			
			*valid = true;
		}
	}

	return offset;
}

TVector3 TToolData::Calculate_Vector(
	const QString						&configuration_name,
	const double						&angle_a,
	const double						&angle_b,
	bool 								* const valid) const
{
	std::vector<TConfiguration>::const_iterator iter;
	TVector3							vector(0,0,-1);
	TMat4								rot_mat;
	TMat4								inv_mat;
	static const double					DEG_TO_RAD(-0.01745329251994);
		
	assert(valid);
	*valid = false;

	for(iter = d_configurations.begin();iter != d_configurations.end();++iter)
	{
		if((*iter).configuration_name == configuration_name)
		{
			rot_mat.Rotate(rot_mat.Y(),angle_a * DEG_TO_RAD);
			rot_mat.Rotate(TVector3(0,0,1),angle_b * DEG_TO_RAD);
			
			inv_mat = d_orientation_mat;
			inv_mat.Transpose();
			
			vector = rot_mat * TVector3(0,0,1);
			vector = inv_mat * vector;
			
			*valid = true;
		}
	}

	return vector;
}

bool TToolData::Update_Orientation_Mat(void)
{
	TVector3							v1;
	TVector3							v2;
	TVector3							v3;
	static const double					ZERO_EPSILON(0.001);
	
	v1 = this->Get_Axis(d_a0b0_orientation);
	v2 = this->Get_Axis(d_a90b0_orientation);
	
	v3 = v2 * v1;
	
	if(!(v3.Length() > ZERO_EPSILON))
	{
		return false;
	}
	
	d_orientation_mat.Z(v1);
	d_orientation_mat.X(v2);
	d_orientation_mat.Y(v3);
	
	return true;
}

void TToolData::Generate_Default_Offset_Configurations(void)
{
	TConfiguration				offset_configuration;
	
	if(d_configurations.size())
	{
		return;
	}
	
	// offset in direction of A90B0 is X
	// offset in direction of A90B90 is Y
	// offset in direction of A0B0 is Z

	// PH10T
	offset_configuration.configuration_name = QStringLiteral("Ren PH10T + TP20 + TIPX10");
	offset_configuration.offset_fixed.Set(0.0,0.0,77.0);
	offset_configuration.offset_rotational.Set(0.0,0.0,73.0);
	d_configurations.push_back(offset_configuration);
	
	// PH10T + Ext 50
	offset_configuration.configuration_name = QStringLiteral("Ren PH10T + EXT50 + TP20 + TIPX10");
	offset_configuration.offset_fixed.Set(0.0,0.0,77.0);
	offset_configuration.offset_rotational.Set(0.0,0.0,123.0);
	d_configurations.push_back(offset_configuration);
	
	// PH10M
	offset_configuration.configuration_name = QStringLiteral("Ren PH10M + PAA1 + TP20 + TIPX10");
	offset_configuration.offset_fixed.Set(0.0,0.0,78.55);
	offset_configuration.offset_rotational.Set(0.0,0.0,116.52);
	d_configurations.push_back(offset_configuration);

	offset_configuration.configuration_name = QStringLiteral("Ren PH10M + PAA1 + TP200 + TIPX10");
	offset_configuration.offset_fixed.Set(0.0,0.0,78.55);
	offset_configuration.offset_rotational.Set(0.0,0.0,121.52);
	d_configurations.push_back(offset_configuration);
	
	// PH10M + Ext 50
	offset_configuration.configuration_name = QStringLiteral("Ren PH10M + PAA1 + EXT50 + TP20 + TIPX10");
	offset_configuration.offset_fixed.Set(0.0,0.0,78.55);
	offset_configuration.offset_rotational.Set(0.0,0.0,166.52);
	d_configurations.push_back(offset_configuration);
	
	offset_configuration.configuration_name = QStringLiteral("Ren PH10M + PAA1 + EXT50 + TP200 + TIPX10");
	offset_configuration.offset_fixed.Set(0.0,0.0,78.55);
	offset_configuration.offset_rotational.Set(0.0,0.0,171.52);
	d_configurations.push_back(offset_configuration);

	// PH10MQ
	offset_configuration.configuration_name = QStringLiteral("Ren PH10MQ + PAA1 + TP20 + TIPX10");
	offset_configuration.offset_fixed.Set(0.0,0.0,35);
	offset_configuration.offset_rotational.Set(0.0,0.0,116.52);
	d_configurations.push_back(offset_configuration);
	
	offset_configuration.configuration_name = QStringLiteral("Ren PH10MQ + PAA1 + TP200 + TIPX10");
	offset_configuration.offset_fixed.Set(0.0,0.0,35);
	offset_configuration.offset_rotational.Set(0.0,0.0,121.52);
	d_configurations.push_back(offset_configuration);

	// PH10MQ + Ext 50
	offset_configuration.configuration_name = QStringLiteral("Ren PH10MQ + PAA1 + EXT50 + TP20 + TIPX10");
	offset_configuration.offset_fixed.Set(0.0,0.0,35);
	offset_configuration.offset_rotational.Set(0.0,0.0,166.52);
	d_configurations.push_back(offset_configuration);

	offset_configuration.configuration_name = QStringLiteral("Ren PH10MQ + PAA1 + EXT50 + TP200 + TIPX10");
	offset_configuration.offset_fixed.Set(0.0,0.0,35);
	offset_configuration.offset_rotational.Set(0.0,0.0,171.52);
	d_configurations.push_back(offset_configuration);

	// Hex Head
	offset_configuration.configuration_name = QStringLiteral("Hex HH-A-T5 + TKJ_31 + HP-TM + TIPX10");
	offset_configuration.offset_fixed.Set(0.0,0.0,67.5);
	offset_configuration.offset_rotational.Set(12.0,0.0,124.075);
	d_configurations.push_back(offset_configuration);
	
	// Hex Head + 50
	offset_configuration.configuration_name = QStringLiteral("Hex HH-A-T5 + TKJ_31 + EXT50 + HP-TM + TIPX10");
	offset_configuration.offset_fixed.Set(0.0,0.0,67.5);
	offset_configuration.offset_rotational.Set(12.0,0.0,174.075);
	d_configurations.push_back(offset_configuration);
	
	// Hex MQ Head
	offset_configuration.configuration_name = QStringLiteral("Hex HH-AS8-T5 + TKJ_31 + HP-TM + TIPX10");
	offset_configuration.offset_fixed.Set(0.0,0.0,44.4);
	offset_configuration.offset_rotational.Set(12.0,0.0,124.075);
	d_configurations.push_back(offset_configuration);
	
	// Hex MQ Head + 50
	offset_configuration.configuration_name = QStringLiteral("Hex HH-AS8-T5 + TKJ_31 + EXT50 + HP-TM + TIPX10");
	offset_configuration.offset_fixed.Set(0.0,0.0,44.4);
	offset_configuration.offset_rotational.Set(12.0,0.0,174.075);
	d_configurations.push_back(offset_configuration);
}

void TToolData::Generate_Default_Tools(void)
{
	TToolData::TTool					tool;
	
	if(d_tools.size())
	{
		return;
	}
	
	tool.tool_base_name = QStringLiteral("E0_Tool");
	tool.configuration_name = QStringLiteral("PH10MQ + PAA1 + TP20 + 4x10");
	d_tools.push_back(tool);
	
	tool.tool_base_name = QStringLiteral("E150_Tool");
	tool.configuration_name = QStringLiteral("PH10MQ + PAA1 + EXT50 + TP20 + 4x10");
	d_tools.push_back(tool);
}

void TToolData::Load_Data_V1(
	TXmlFile							* const xml_file)
{
	QDomElement							element;
	QDomElement							e0;
	QDomElement							e1;
	QDomElement							e2;
	QDomElement							e3;
	QDomElement							e4;
	QDomElement							e5;
	QString								text;
	QStringList							list;
	TToolData::TConfiguration			configuration;
	TToolData::TTool					tool;
	TToolData::TToolEntry				tool_entry;
	TToolData_Sort						tool_data_sort;
	
	assert(xml_file);
	
	
	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Orientation"));
	
	e1 = xml_file->Get_Node(&e0,QStringLiteral("A0B0"));
	text = xml_file->Read_Text_Node(&e1);
	d_a0b0_orientation = this->String_To_Axis(text);
	
	e1 = xml_file->Get_Node(&e0,QStringLiteral("A90B0"));
	text = xml_file->Read_Text_Node(&e1);
	d_a90b0_orientation = this->String_To_Axis(text);

	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Options"));
	e1 = xml_file->Get_Node(&e0,QStringLiteral("Motorized_Probe_Head"));
	text = xml_file->Read_Text_Node(&e1);
	d_motorized_probe_head = (text.compare(QStringLiteral("Yes")) == 0);

	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Configurations"));

	e1 = xml_file->Get_Node(&e0,QStringLiteral("Configuration"),true);

	while(!e1.isNull())
	{
		e2 = xml_file->Get_Node(&e1,QStringLiteral("Name"));
		configuration.configuration_name = xml_file->Read_Text_Node(&e2);
		
		e2 = xml_file->Get_Node(&e1,QStringLiteral("Offset_Fixed"));
		text = xml_file->Read_Text_Node(&e2);
		list = text.split(',');
		
		if(list.size() == 3)
		{
			configuration.offset_fixed.x = list[0].toDouble();
			configuration.offset_fixed.y = list[1].toDouble();
			configuration.offset_fixed.z = list[2].toDouble();
		}
		else
		{
			throw TXmlFileException(QStringLiteral("ERR:  Offset_Fixed must have three values"));
		}
		
		e2 = xml_file->Get_Node(&e1,QStringLiteral("Offset_Rotational"));
		text = xml_file->Read_Text_Node(&e2);
		list = text.split(',');
		
		if(list.size() == 3)
		{
			configuration.offset_rotational.x = list[0].toDouble();
			configuration.offset_rotational.y = list[1].toDouble();
			configuration.offset_rotational.z = list[2].toDouble();
		}
		else
		{
			throw TXmlFileException(QStringLiteral("ERR:  Offset_Rotational must have three values"));
		}
		
		d_configurations.push_back(configuration);

		e1 = xml_file->Get_Sibling_Node(&e1,QStringLiteral("Configuration"),true);
	}
 
	
	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Tools"));
	
	e1 = xml_file->Get_Node(&e0,QStringLiteral("Tool"),true);
	
	while(!e1.isNull())
	{
		e2 = xml_file->Get_Node(&e1,QStringLiteral("Name"));
		tool.tool_base_name = xml_file->Read_Text_Node(&e2);
		
		e2 = xml_file->Get_Node(&e1,QStringLiteral("Configuration"));
		tool.configuration_name = xml_file->Read_Text_Node(&e2);
		
		tool.tool_entries.clear();
		
		e3 = xml_file->Get_Node(&e1,QStringLiteral("ToolData"));
		
		e4 = xml_file->Get_Node(&e3,QStringLiteral("Position"),true);
		
		while(!e4.isNull())
		{
			e5 = xml_file->Get_Node(&e4,QStringLiteral("Angle_AB"));
			text = xml_file->Read_Text_Node(&e5);
			tool_entry.angle_ab_signature = text.toInt();
			
			e5 = xml_file->Get_Node(&e4,QStringLiteral("Tip_Diameter"));
			text = xml_file->Read_Text_Node(&e5);
			tool_entry.tip_diameter = text.toDouble();
			
			tool.tool_entries.push_back(tool_entry);
			
			e4 = xml_file->Get_Sibling_Node(&e4,QStringLiteral("Position"),true);
		}
		
		std::sort(tool.tool_entries.begin(), tool.tool_entries.end(), tool_data_sort);

		d_tools.push_back(tool);
		
		e1 = xml_file->Get_Sibling_Node(&e1,QStringLiteral("Tool"),true);
	}
}
