/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#ifndef TOOLDATAHEADERFILE
#define TOOLDATAHEADERFILE

#include <QString>
#include <QStringList>
#include <vector>

#include "../../../core/vector3.h"
#include "../../../core/mat4.h"

class TXmlFile;

class TToolData
{
public:
	
enum TAxis
{
	AXIS_XPLUS=0,
	AXIS_XMINUS,
	AXIS_YPLUS,
	AXIS_YMINUS,
	AXIS_ZPLUS,
	AXIS_ZMINUS
};
	
struct TToolEntry
{
	int									angle_ab_signature;
	double								tip_diameter;
};

struct TConfiguration
{
	QString								configuration_name;
	TVector3							offset_fixed;
	TVector3							offset_rotational;
};
	
struct TTool
{
	QString								tool_base_name;
	QString								configuration_name;
	std::vector<TToolEntry>				tool_entries;
};

// CREATORS
	TToolData(void);
	~TToolData(void);
	
// ACCESSORS
	QString Last_Error(void) const {return d_last_error;}
	
	bool Save_Tool_Data(const QString &path) const;

	TAxis Probe_Head_Orientation_A0B0(void) const {return d_a0b0_orientation;}
	TAxis Probe_Head_Orientation_A90B0(void) const {return d_a90b0_orientation;}
	bool Probe_Head_Orientation_Valid(void) const {return d_orientation_valid;}
	TVector3 Offset_In_Head_Orientation(const TVector3 &offset) const {return this->To_Orientation(offset);}

	bool Is_Motorized_Probe_Head(void) const {return d_motorized_probe_head;}
	
	int Tool_Count(void) const {return static_cast<int>(d_tools.size());}
	QStringList Tool_Base_Names(void) const;
	TTool Tool(const QString &tool_base_name,bool *valid=0) const;
	TTool Tool(const int index) const;

	QStringList Offset_Configuration_Names(void) const;
	TConfiguration Offset_Configuration(const QString &configuration_name,bool *valid=0) const;
	
	TToolEntry Tool_Entry(const QString &tool_base_name,const double &angle_a,const double &angle_b,TVector3 * const offset = 0, TVector3 * const vector = 0,bool *valid=0) const;
	
// MANIPULATORS
	bool Load_Tool_Data(const QString &path);

	bool Set_Probe_Head_Orientation(const TAxis a0b0, const TAxis a90b0);
	void Set_Motorized_Probe_Head(const bool state) {d_motorized_probe_head = state;}
	
	void Add_Tool(const QString &tool_base_name,const QString &configuration_name);
	void Add_Tool_Angle(const QString &tool_base_name,const double &angle_a,const double &angle_b,const double &tip_diameter);
	bool Remove_Tool_Angle(const QString &tool_base_name,const double &angle_a,const double &angle_b);
	void Update_Tool_Tip_Diameter(const QString &tool_base_name,const double &angle_a,const double &angle_b,const double &tip_diameter);
	
	void Remove_Tool(const QString &tool_base_name);
	
	void Update_Tool_Configuration(const QString &tool_base_name,const QString &configuration_name);
	
	void Update_Configuration(const QString &configuration_name,const TVector3 &offset_fixed, const TVector3 &offset_rotational);
	void Remove_Configuration(const QString &configuration_name);

	static void Encode_AB_Angles(const double &angle_a,const double &angle_b,int * const encoded_value);
	static void Decode_AB_Angles(double * const angle_a, double * const angle_b,const int encoded_value);
	
private:
	
friend struct TToolData_Sort;

	TAxis								d_a0b0_orientation;
	TAxis								d_a90b0_orientation;
	TMat4								d_orientation_mat;
	bool								d_orientation_valid;
	bool								d_motorized_probe_head;
	
	std::vector<TConfiguration>			d_configurations;
	std::vector<TTool> 					d_tools;
	
	mutable QString						d_last_error;

	TVector3 Get_Axis(const TAxis axis) const;
	QString Axis_To_String(const TAxis axis) const;
	TAxis String_To_Axis(const QString &text) const;
	bool ToolData_Contains_Angle(const std::vector<TToolEntry> &dtool_data,const int angle_ab_signature) const;
	
	TVector3 To_Orientation(const TVector3 &offset) const;
	TVector3 From_Orientation(const TVector3 &offset) const;
	TVector3 Calculate_Offset(const QString &configuration_name,const double &angle_a,const double &angle_b,bool * const valid) const;
	TVector3 Calculate_Vector(const QString &configuration_name,const double &angle_a,const double &angle_b,bool * const valid) const;

	bool Update_Orientation_Mat(void);
	
	void Generate_Default_Offset_Configurations(void);
	void Generate_Default_Tools(void);
	
	void Load_Data_V1(TXmlFile * const xml_file);
	
// NOT IMPLEMENTED
	TToolData(const TToolData&);
	TToolData& operator=(const TToolData&);
};

#endif
