/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QStringList>
#include <QByteArray>
#include <QTextStream>

#include "../../../core/vector2.h"

#include "measuredata_ballbar.h"

TMeasureBallbar::TMeasureBallbar(void)
{
	d_file_type = TMeasureData::TYPE_BALLBAR;
}

TMeasureBallbar::~TMeasureBallbar(void)
{
}

QStringList TMeasureBallbar::Text_Report(void) const
{
	std::vector<TBallbarData>::const_iterator iter;
	QStringList							list;
	QString								text;
	double								deviation;
	double								max_length(0.0);
	double								min_length(0.0);
	bool								init;
	
	list.push_back(QStringLiteral("ASME B89.4.1 Ballbar Measurement"));
	list.push_back(QStringLiteral("================================================================"));
	list.push_back(QString());
	
	list.push_back(QStringLiteral("Ballbar Measurements"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	list.push_back(QStringLiteral("    Position        Actual    Deviation"));
	list.push_back(QString());
	
	init = true;
	for(iter = d_ballbar_data.begin();iter != d_ballbar_data.end();++iter)
	{
		deviation = (*iter).length - d_ballbar_average_length;
		
		text = QString("%1  %2 %3").arg((*iter).name,12).arg((*iter).length,12,'f',4).arg(deviation,12,'f',4);
		list.push_back(text);
		
		if(init)
		{
			max_length = (*iter).length;
			min_length = (*iter).length;
			
			init = false;
		}
		else
		{
			if((*iter).length > max_length) max_length = (*iter).length;
			if((*iter).length < min_length) min_length = (*iter).length;
		}
	}
	
	list.push_back(QString());
	list.push_back(QString("  Minimum Length: %1 mm").arg(min_length,0,'f',4));
	list.push_back(QString("  Maximum Length: %1 mm").arg(max_length,0,'f',4));
	list.push_back(QString("  Mean Length:    %1 mm").arg(d_ballbar_average_length,0,'f',4));
	list.push_back(QString("  Length Range:   %1 mm").arg(max_length - min_length,0,'f',4));

	return list;
}

std::vector<std::vector<TVector2> > TMeasureBallbar::Graph_Data(void) const
{
	std::vector<TBallbarData>::const_iterator iter;
	std::vector<TVector2>				graph_data;
	std::vector<std::vector<TVector2> > graph_data_sets;
	TVector2							pnt;
	double								deviation;
	int									cntr;

	for(iter = d_ballbar_data.begin(),cntr = 0;iter != d_ballbar_data.end();++iter,++cntr)
	{
		deviation = (*iter).length - d_ballbar_average_length;
		
		pnt.x = (*iter).length;
		pnt.y = deviation;
		
		graph_data.push_back(pnt);
	}
	
	graph_data_sets.push_back(graph_data);

	return graph_data_sets;
}

bool TMeasureBallbar::Load_Data(
	const QByteArray					&file_data)
{
	QTextStream							stream(file_data,QIODevice::ReadOnly);
	std::vector<TBallbarData>::const_iterator iter;
	TBallbarData						ballbar_data;
	TVector3							vec;
	QString								line;
	QStringList							list;
	bool								init(true);
	
	d_ballbar_data.clear();
	
	while(!stream.atEnd())
	{
		line = stream.readLine().trimmed();
		
		if(line.startsWith('#') ||
		   line.length() == 0)
		{
			continue;
		}
		
		if(init)
		{
			if(!line.startsWith(QStringLiteral("B89.4.1_Data:Version=1:Type=BallBar")))
			{
				d_last_error = QStringLiteral("Artifact identification line not recognized");
				return false;
			}
			
			init = false;
		}
		else
		{
			list = line.split(',');
			
			if(list.size() == 13)
			{
				ballbar_data.name = list[0];
				
				ballbar_data.sph1.x = list[2].toDouble();
				ballbar_data.sph1.y = list[3].toDouble();
				ballbar_data.sph1.z = list[4].toDouble();
				
				ballbar_data.sph2.x = list[5].toDouble();
				ballbar_data.sph2.y = list[6].toDouble();
				ballbar_data.sph2.z = list[7].toDouble();
				
				ballbar_data.sph1_form = list[8].toDouble();
				ballbar_data.sph2_form = list[9].toDouble();
				
				ballbar_data.tool_offset.x = list[10].toDouble();
				ballbar_data.tool_offset.y = list[11].toDouble();
				ballbar_data.tool_offset.z = list[12].toDouble();
				
				vec = ballbar_data.sph2 - ballbar_data.sph1;
				ballbar_data.length = vec.Length();
				
				d_ballbar_data.push_back(ballbar_data);
			}
		}
	}
	
	if(d_ballbar_data.size())
	{
		d_ballbar_average_length = 0.0;

		for(iter = d_ballbar_data.begin();iter != d_ballbar_data.end();++iter)
		{
			d_ballbar_average_length += (*iter).length;
		}
		
		d_ballbar_average_length /= static_cast<double>(d_ballbar_data.size());
	}
	
	
	return true;
}
