/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QStringList>
#include <QByteArray>
#include <QTextStream>

#include "../../../core/vector2.h"

#include "measuredata_ballbar10360.h"

TMeasureBallbar10360::TMeasureBallbar10360(void)
{
	d_file_type = TMeasureData::TYPE_STEPGAUGE_10360;
}

TMeasureBallbar10360::~TMeasureBallbar10360(void)
{
}

QStringList TMeasureBallbar10360::Text_Report(void) const
{
	std::vector<TMeasureBallbar10360::TLengthData>::const_iterator iter;
	QStringList							list;
	QString								text;
	double								deviation;
	TVector3							vec;
	
	list.push_back(QStringLiteral("Ballbar 10360 Measurement"));
	list.push_back(QStringLiteral("================================================================"));
	list.push_back(QString());
	
	list.push_back(QStringLiteral("Description"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	text = QString("   Position: %1").arg(d_measurement_description);
	list.push_back(text);
	
	text = QString("  Equipment: %1").arg(d_equipment_identification);
	list.push_back(text);
	
	text = QString("Zero Offset: %1").arg(d_sphere_zero_offset);
	list.push_back(text);

	list.push_back(QString());
	list.push_back(QStringLiteral("Alignment"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	vec = d_alignment_data.X();
	text = QString("  X Axis: %1, %2, %3").arg(vec.x,12,'f',9).arg(vec.y,12,'f',9).arg(vec.z,12,'f',9);
	list.push_back(text);
	
	vec = d_alignment_data.Y();
	text = QString("  Y Axis: %1, %2, %3").arg(vec.x,12,'f',9).arg(vec.y,12,'f',9).arg(vec.z,12,'f',9);
	list.push_back(text);
	
	vec = d_alignment_data.Z();
	text = QString("  Z Axis: %1, %2, %3").arg(vec.x,12,'f',9).arg(vec.y,12,'f',9).arg(vec.z,12,'f',9);
	list.push_back(text);
	
	vec = d_alignment_data.Origin();
	text = QString("  Origin: %1, %2, %3").arg(vec.x,12,'f',4).arg(vec.y,12,'f',4).arg(vec.z,12,'f',4);
	list.push_back(text);
	
	list.push_back(QString());
	list.push_back(QStringLiteral("Tool"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	text = QString("  Offset: %1, %2, %3").arg(tool_offset.x,12,'f',4).arg(tool_offset.y,12,'f',4).arg(tool_offset.z,12,'f',4);
	list.push_back(text);
	
	text = QString("  Vector: %1, %2, %3").arg(d_probe_data_vector.x,12,'f',9).arg(d_probe_data_vector.y,12,'f',9).arg(d_probe_data_vector.z,12,'f',9);
	list.push_back(text);
	
	list.push_back(QString());
	list.push_back(QStringLiteral("Scale Expansion Coefficients"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	text = QString("  X Axis: %1 um/m/˚C").arg(d_scale_expansion_coefficient.x,0,'f',1);
	list.push_back(text);
	
	text = QString("  Y Axis: %1 um/m/˚C").arg(d_scale_expansion_coefficient.y,0,'f',1);
	list.push_back(text);
	
	text = QString("  Z Axis: %1 um/m/˚C").arg(d_scale_expansion_coefficient.z,0,'f',1);
	list.push_back(text);
	
	list.push_back(QString());
	list.push_back(QStringLiteral("Axis and Part Temperature"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	text = QString("  X Axis: %1 ˚C").arg(d_temperature_x,0,'f',2);
	list.push_back(text);
	
	text = QString("  Y Axis: %1 ˚C").arg(d_temperature_y,0,'f',2);
	list.push_back(text);
	
	text = QString("  Z Axis: %1 ˚C").arg(d_temperature_z,0,'f',2);
	list.push_back(text);
	
	text = QString("  Part:   %1 ˚C").arg(d_temperature_gauge,0,'f',2);
	list.push_back(text);

	list.push_back(QString());
	list.push_back(QStringLiteral("Measurement Lengths mm"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	list.push_back(QStringLiteral("     Nominal        Actual      Deviation"));
	list.push_back(QString());

	for(iter = d_length_data.begin();iter != d_length_data.end();++iter)
	{
		deviation = (*iter).actual - (*iter).nominal;
		
		text = QString("%1  %2  %3").arg((*iter).nominal,12,'f',4).arg((*iter).actual,12,'f',4).arg(deviation,12,'f',4);
		list.push_back(text);
	}
	
	return list;
}

std::vector<std::vector<TVector2> > TMeasureBallbar10360::Graph_Data(void) const
{
	std::vector<std::vector<TVector2> > graph_data;
	std::vector<TVector2>				graph_item_points_1;
	std::vector<TVector2>				graph_item_points_2;
	std::vector<TVector2>				graph_item_points_3;
	TVector2							pnt;
	TVector2							vec;
	double								deviation;
	std::vector<TLengthData>::const_iterator iter;
	
	if(d_length_data.size() == 15)
	{
		for(iter = d_length_data.begin();iter != d_length_data.end();iter += 3)
		{
			deviation = (*iter).actual - (*iter).nominal;
			pnt.Set((*iter).actual,deviation);
			
			graph_item_points_1.push_back(pnt);
			
			deviation = (*(iter+1)).actual - (*(iter+1)).nominal;
			pnt.Set((*iter).actual,deviation);
			
			graph_item_points_2.push_back(pnt);
			
			deviation = (*(iter+2)).actual - (*(iter+2)).nominal;
			pnt.Set((*iter).actual,deviation);
			
			graph_item_points_3.push_back(pnt);
		}
		
		graph_data.push_back(graph_item_points_1);
		graph_data.push_back(graph_item_points_2);
		graph_data.push_back(graph_item_points_3);
	}
	
	return graph_data;
}

bool TMeasureBallbar10360::Load_Data(
	const QByteArray					&file_data)
{
	QTextStream							stream(file_data,QIODevice::ReadOnly);
	TMeasureBallbar10360::TSphereData	sphere_data;
	QString								line;
	QString								text;
	QString								text_section;
	QStringList							list;
	TVector3							sph0,sphx;
	TVector3							vec;
	bool								found_id(false);
	bool								found_description(false);
	bool								found_scale(false);
	bool								inside_scale(false);
	bool								found_temperature(false);
	bool								inside_temperature(false);
	bool								found_alignment(false);
	bool								inside_alignment(false);
	bool								found_probe_offset(false);
	bool								found_probe_vector(false);
	bool								found_reference_position(false);
	bool								found_data(false);
	bool								inside_data(false);
	bool								init(true);
	
	d_sphere_data.clear();
	d_length_data.clear();
	
	while(!stream.atEnd())
	{
		line = stream.readLine().trimmed();
		
		if(line.startsWith('#') ||
		   line.length() == 0)
		{
			continue;
		}
		
		if(init)
		{
			if(!line.startsWith(QStringLiteral("B89.4.10360_Raw_Measurement:Version=1:Type=BallBar_10360")))
			{
				d_last_error = QStringLiteral("Stepgauge identification line not recognized");
				return false;
			}
			
			init = false;
		}
		else
		{
			if(line.compare(QStringLiteral("Scales_Begin:"),Qt::CaseInsensitive) == 0)
			{
				found_scale = true;
				inside_scale = true;
			}
			else if(line.compare(QStringLiteral("Scales_End:"),Qt::CaseInsensitive) == 0)
			{
				inside_scale = false;
			}
			else if(line.startsWith(QStringLiteral("Equipment_Identification:"),Qt::CaseInsensitive))
			{
				found_id = true;
				
				d_equipment_identification = line.mid(25);	// remove 'Equipment_Identification:'
			}
			else if(line.compare(QStringLiteral("Alignment_Begin:"),Qt::CaseInsensitive) == 0)
			{
				found_alignment = true;
				inside_alignment = true;
			}
			else if(line.compare(QStringLiteral("Alignment_End:"),Qt::CaseInsensitive) == 0)
			{
				inside_alignment = false;
			}
			else if(line.compare(QStringLiteral("Temperature_Begin:"),Qt::CaseInsensitive) == 0)
			{
				found_temperature = true;
				inside_temperature = true;
			}
			else if(line.compare(QStringLiteral("Temperature_End:"),Qt::CaseInsensitive) == 0)
			{
				inside_temperature = false;
			}
			else if(line.compare(QStringLiteral("Measurement_Begin:"),Qt::CaseInsensitive) == 0)
			{
				found_data = true;
				inside_data = true;
			}
			else if(line.compare(QStringLiteral("Measurement_End::"),Qt::CaseInsensitive) == 0)
			{
				inside_data = false;
			}
			else if(line.startsWith(QStringLiteral("Probe_Offset:"),Qt::CaseInsensitive))
			{
				found_probe_offset = true;
				
				text = line.mid(13);	// remove 'Probe_Offset:'
				list = text.split(',');
				
				if(list.size() == 3)
				{
					tool_offset.x = list[0].toDouble();
					tool_offset.y = list[1].toDouble();
					tool_offset.z = list[2].toDouble();
				}
				else
				{
					d_last_error = QStringLiteral("Probe offset data not recognized.");
					return false;
				}
				
			}
			else if(line.startsWith(QStringLiteral("Probe_Vector:"),Qt::CaseInsensitive))
			{
				found_probe_vector = true;
				
				text = line.mid(13);	// remove 'Probe_Vector:'
				list = text.split(',');
				
				if(list.size() == 3)
				{
					d_probe_data_vector.x = list[0].toDouble();
					d_probe_data_vector.y = list[1].toDouble();
					d_probe_data_vector.z = list[2].toDouble();
				}
				else
				{
					d_last_error = QStringLiteral("Probe vector data not recognized.");
					return false;
				}
			}
			else
			{
				if(inside_alignment == true &&
						inside_data == false &&
						inside_temperature == false &&
						inside_scale == false)
				{
					if(line.startsWith(QStringLiteral("X_Axis:"),Qt::CaseInsensitive))
					{
						text = line.mid(7);	// remove 'X_Axis:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.X(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("X Alignment data not recognized.");
							return false;
						}
					}
					else if(line.startsWith(QStringLiteral("Y_Axis:"),Qt::CaseInsensitive))
					{
						text = line.mid(7);	// remove 'Y_Axis:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.Y(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("Y Alignment data not recognized.");
							return false;
						}
					}
					else if(line.startsWith(QStringLiteral("Z_Axis:"),Qt::CaseInsensitive))
					{
						text = line.mid(7);	// remove 'Z_Axis:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.Z(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("Z Alignment data not recognized.");
							return false;
						}
					}
					else if(line.startsWith(QStringLiteral("Translation:"),Qt::CaseInsensitive))
					{
						text = line.mid(12);	// remove 'Translation:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.Origin(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("Alignment translation data not recognized.");
							return false;
						}
					}
				}
				else if(inside_alignment == false &&
				   inside_data == false &&
				   inside_temperature == true &&
				   inside_scale == false)
				{
					if(line.startsWith(QStringLiteral("X_Scale:"),Qt::CaseInsensitive))
					{
						text = line.mid(8);	// remove 'X_Scale:'
						d_temperature_x = text.toDouble();
					}
					else if(line.startsWith(QStringLiteral("Y_Scale:"),Qt::CaseInsensitive))
					{
						text = line.mid(8);	// remove 'Y_Scale:'
						d_temperature_y = text.toDouble();
					}
					else if(line.startsWith(QStringLiteral("Z_Scale:"),Qt::CaseInsensitive))
					{
						text = line.mid(8);	// remove 'Z_Scale:'
						d_temperature_z = text.toDouble();
					}
					else if(line.startsWith(QStringLiteral("Gauge:"),Qt::CaseInsensitive))
					{
						text = line.mid(6);	// remove 'Gauge:'
						d_temperature_gauge = text.toDouble();
					}
				}
				else if(inside_alignment == false &&
						inside_data == false &&
						inside_temperature == false &&
						inside_scale == true)
				{
					if(line.startsWith(QStringLiteral("X_Expansion_Coefficient:"),Qt::CaseInsensitive))
					{
						text = line.mid(24);	// remove 'X_Expansion_Coefficient:'
						d_scale_expansion_coefficient.x = text.toDouble();
					}
					else if(line.startsWith(QStringLiteral("Y_Expansion_Coefficient:"),Qt::CaseInsensitive))
					{
						text = line.mid(24);	// remove 'Y_Expansion_Coefficient:'
						d_scale_expansion_coefficient.y = text.toDouble();
					}
					else if(line.startsWith(QStringLiteral("Z_Expansion_Coefficient:"),Qt::CaseInsensitive))
					{
						text = line.mid(24);	// remove 'Z_Expansion_Coefficient:'
						d_scale_expansion_coefficient.z = text.toDouble();
					}
				}
				else if(inside_alignment == false &&
						inside_data == true &&
						inside_temperature == false &&
						inside_scale == false)
				{
					if(line.startsWith(QStringLiteral("Description:"),Qt::CaseInsensitive))
					{
						found_description = true;
						d_measurement_description = line.mid(12);	// remove 'Description:'
					}
					else if(line.startsWith(QStringLiteral("Sphere_Zero_Offset:"),Qt::CaseInsensitive))
					{
						found_reference_position = true;
						text = line.mid(19);	// remove 'Sphere_Zero_Offset:'
						
						d_sphere_zero_offset = text.toInt();
					}
					else if(line.startsWith(QStringLiteral("Sphere:"),Qt::CaseInsensitive))
					{
						text = line.mid(7);	// remove 'Sphere:'

						list = text.split(',');
												
						if(list.size() == 8)
						{
							sphere_data.actual.x = list[3].toDouble();
							sphere_data.actual.y = list[4].toDouble();
							sphere_data.actual.z = list[5].toDouble();
		
							sphere_data.nominal = sphere_data.actual;
						
							d_sphere_data.push_back(sphere_data);

						}
					}
				}
			}
		}
	}
	
	if(found_alignment == false ||
	   found_id == false ||
	   found_description == false ||
	   found_scale == false ||
	   found_temperature == false ||
	   found_probe_offset == false ||
	   found_probe_vector == false ||
	   found_reference_position == false ||
	   found_data == false)
	{
		d_last_error = QStringLiteral("Data missing from measurement file.");
		return false;
	}
		
	if(d_sphere_data.size() != 30)
	{
		d_last_error = QStringLiteral("Data missing from measurement file.");
		return false;
	}
	
	this->Update_Length_Data();
	
	return true;
}

bool TMeasureBallbar10360::Set_Nominal_Data(
	const std::vector<TVector3>			&data)
{
	std::vector<TSphereData>::iterator	iter;
	TVector3							sphere_zero;
	TVector3							vec;
	TVector3							nominal;
	bool								valid;
	
	if(d_sphere_zero_offset < 0)
	{
		d_last_error = QStringLiteral("Sphere zero offset invalid.  Cannot be less than zero.");
		return false;
	}
	
	if(!(static_cast<unsigned int>(d_sphere_zero_offset) < data.size()))
	{
		d_last_error = QStringLiteral("Sphere zero offset invalid.  Offset greater than size of the nominal data.");
		return false;
	}
	
	sphere_zero = data[d_sphere_zero_offset];

	for(iter = d_sphere_data.begin();iter != d_sphere_data.end();++iter)
	{
		vec = (*iter).actual;
		vec += sphere_zero;
		
		nominal = this->Get_Nominal(vec,data,&valid);
		
		if(valid)
		{
			(*iter).nominal = nominal - sphere_zero;
		}
		else
		{
			d_last_error = QStringLiteral("Nominal data not found.  Possible wrong reference step position or measurement error exceeds maximum allowed.");
			return false;
		}
	}
	
	this->Update_Length_Data();
	
	return true;
}

void TMeasureBallbar10360::Update_Length_Data(void)
{
	std::vector<TSphereData>::const_iterator iter;
	TLengthData							length_data;
	TVector3							vec;
	
	d_length_data.clear();
	
	if(d_sphere_data.size() == 30)	// pattern of points is assumed;
	{
		for(iter = d_sphere_data.begin();iter != d_sphere_data.end();iter += 2)
		{
			vec = (*(iter+1)).nominal - (*iter).nominal;
			length_data.nominal = vec.Length();
			
			vec = (*(iter+1)).actual - (*iter).actual;
			length_data.actual = vec.Length();
			
			d_length_data.push_back(length_data);
		}
	}
}

TVector3 TMeasureBallbar10360::Get_Nominal(
	const TVector3						&position,
	const std::vector<TVector3>			&data,
	bool								* const valid) const
{
	std::vector<TVector3>::const_iterator iter;
	TVector3							nominal;
	double								deviation;
	static const double					MAXIMUM_DEVIATION(5.0);	// nominal must be within +/- of this value.
	
	assert(valid);
	*valid = false;
	
	for(iter = data.begin();iter != data.end();++iter)
	{
		deviation = fabs((*iter).x - position.x);
		
		if(!(deviation > MAXIMUM_DEVIATION))
		{
			*valid = true;
			
			nominal = (*iter);
			break;
		}
	}

	return nominal;
}
