/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QStringList>
#include <QByteArray>
#include <QTextStream>

#include "../../../core/vector2.h"

#include "measuredata_gaugeblock.h"

TMeasureGaugeblock::TMeasureGaugeblock(void)
{
	d_file_type = TMeasureData::TYPE_GAUGEBLOCK;
}

TMeasureGaugeblock::~TMeasureGaugeblock(void)
{
}

QStringList TMeasureGaugeblock::Text_Report(void) const
{
	std::vector<double>::const_iterator iter;
	QStringList							list;
	QString								text;
	double								mean_length;
	TVector3							vec;
	
	list.push_back(QStringLiteral("ASME B89.4.10360 Gauge Block Measurement"));
	list.push_back(QStringLiteral("================================================================"));
	list.push_back(QString());
	
	list.push_back(QStringLiteral("Description"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	text = QString("  Position:  %1").arg(d_measurement_description);
	list.push_back(text);
	
	text = QString("  Equipment: %1").arg(d_equipment_identification);
	list.push_back(text);
	
	list.push_back(QString());
	list.push_back(QStringLiteral("Alignment"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	vec = d_alignment_data.X();
	text = QString("  X Axis: %1, %2, %3").arg(vec.x,12,'f',9).arg(vec.y,12,'f',9).arg(vec.z,12,'f',9);
	list.push_back(text);
	
	vec = d_alignment_data.Y();
	text = QString("  Y Axis: %1, %2, %3").arg(vec.x,12,'f',9).arg(vec.y,12,'f',9).arg(vec.z,12,'f',9);
	list.push_back(text);
	
	vec = d_alignment_data.Z();
	text = QString("  Z Axis: %1, %2, %3").arg(vec.x,12,'f',9).arg(vec.y,12,'f',9).arg(vec.z,12,'f',9);
	list.push_back(text);
	
	vec = d_alignment_data.Origin();
	text = QString("  Origin: %1, %2, %3").arg(vec.x,12,'f',4).arg(vec.y,12,'f',4).arg(vec.z,12,'f',4);
	list.push_back(text);
	
	list.push_back(QString());
	text = QString("  Datum Flatness: %1 mm").arg(d_datum_flatness,0,'f',4);
	list.push_back(text);

	
	list.push_back(QString());
	list.push_back(QStringLiteral("Tool"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	text = QString("  Offset: %1, %2, %3").arg(tool_offset.x,12,'f',4).arg(tool_offset.y,12,'f',4).arg(tool_offset.z,12,'f',4);
	list.push_back(text);
	
	text = QString("  Vector: %1, %2, %3").arg(d_probe_data_vector.x,12,'f',9).arg(d_probe_data_vector.y,12,'f',9).arg(d_probe_data_vector.z,12,'f',9);
	list.push_back(text);
	
	list.push_back(QString());
	list.push_back(QStringLiteral("Scale Expansion Coefficients"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	text = QString("  X Axis: %1 um/m/˚C").arg(d_scale_expansion_coefficient.x,0,'f',1);
	list.push_back(text);
	
	text = QString("  Y Axis: %1 um/m/˚C").arg(d_scale_expansion_coefficient.y,0,'f',1);
	list.push_back(text);
	
	text = QString("  Z Axis: %1 um/m/˚C").arg(d_scale_expansion_coefficient.z,0,'f',1);
	list.push_back(text);
	
	list.push_back(QString());
	list.push_back(QStringLiteral("Axis and Part Temperature"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	text = QString("  X Axis: %1 ˚C").arg(d_temperature_x,0,'f',2);
	list.push_back(text);
	
	text = QString("  Y Axis: %1 ˚C").arg(d_temperature_y,0,'f',2);
	list.push_back(text);
	
	text = QString("  Z Axis: %1 ˚C").arg(d_temperature_z,0,'f',2);
	list.push_back(text);
	
	text = QString("  Part:   %1 ˚C").arg(d_temperature_gauge,0,'f',2);
	list.push_back(text);
	
	list.push_back(QString());
	list.push_back(QStringLiteral("Measurement Lengths"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	list.push_back(QStringLiteral("     Actual"));
	list.push_back(QString());
	
	mean_length = 0.0;
	for(iter = d_length_data.begin();iter != d_length_data.end();++iter)
	{
		text = QString("%1").arg((*iter),12,'f',4);
		list.push_back(text);
		
		mean_length += (*iter);
	}
	
	if(d_length_data.size())
	{
		mean_length /= static_cast<double>(d_length_data.size());
		
		list.push_back(QString());
		list.push_back(QStringLiteral("Mean:"));
		
		text = QString("%1 mm").arg(mean_length,12,'f',4);
		list.push_back(text);
	}
	
	return list;
}

std::vector<std::vector<TVector2> > TMeasureGaugeblock::Graph_Data(void) const
{
	std::vector<std::vector<TVector2> > graph_data;
	
	return graph_data;
}

bool TMeasureGaugeblock::Load_Data(
	const QByteArray					&file_data)
{
	QTextStream							stream(file_data,QIODevice::ReadOnly);
	QString								line;
	QString								text;
	QString								text_section;
	QStringList							list;
	double								point_data;
	double								length_data;
	bool								found_id(false);
	bool								found_description(false);
	bool								found_scale(false);
	bool								inside_scale(false);
	bool								found_temperature(false);
	bool								inside_temperature(false);
	bool								found_alignment(false);
	bool								inside_alignment(false);
	bool								found_probe_offset(false);
	bool								found_probe_vector(false);
	bool								found_datum_flatness(false);
	bool								found_data(false);
	bool								inside_data(false);
	bool								init(true);
	
	d_point_data.clear();
	d_length_data.clear();

	while(!stream.atEnd())
	{
		line = stream.readLine().trimmed();
		
		if(line.startsWith('#') ||
		   line.length() == 0)
		{
			continue;
		}
		
		if(init)
		{
			if(!line.startsWith(QStringLiteral("B89.4.10360_Raw_Measurement:Version=1:Type=Single_Gauge")))
			{
				d_last_error = QStringLiteral("Gauge block identification line not recognized");
				return false;
			}
			
			init = false;
		}
		else
		{
			if(line.compare(QStringLiteral("Scales_Begin:"),Qt::CaseInsensitive) == 0)
			{
				found_scale = true;
				inside_scale = true;
			}
			else if(line.compare(QStringLiteral("Scales_End:"),Qt::CaseInsensitive) == 0)
			{
				inside_scale = false;
			}
			else if(line.startsWith(QStringLiteral("Equipment_Identification:"),Qt::CaseInsensitive))
			{
				found_id = true;
				
				d_equipment_identification = line.mid(25);	// remove 'Equipment_Identification:'
			}
			else if(line.compare(QStringLiteral("Alignment_Begin:"),Qt::CaseInsensitive) == 0)
			{
				found_alignment = true;
				inside_alignment = true;
			}
			else if(line.compare(QStringLiteral("Alignment_End:"),Qt::CaseInsensitive) == 0)
			{
				inside_alignment = false;
			}
			else if(line.compare(QStringLiteral("Temperature_Begin:"),Qt::CaseInsensitive) == 0)
			{
				found_temperature = true;
				inside_temperature = true;
			}
			else if(line.compare(QStringLiteral("Temperature_End:"),Qt::CaseInsensitive) == 0)
			{
				inside_temperature = false;
			}
			else if(line.compare(QStringLiteral("Measurement_Begin:"),Qt::CaseInsensitive) == 0)
			{
				found_data = true;
				inside_data = true;
			}
			else if(line.compare(QStringLiteral("Measurement_End::"),Qt::CaseInsensitive) == 0)
			{
				inside_data = false;
			}
			else if(line.startsWith(QStringLiteral("Probe_Offset:"),Qt::CaseInsensitive))
			{
				found_probe_offset = true;
				
				text = line.mid(13);	// remove 'Probe_Offset:'
				list = text.split(',');
				
				if(list.size() == 3)
				{
					tool_offset.x = list[0].toDouble();
					tool_offset.y = list[1].toDouble();
					tool_offset.z = list[2].toDouble();
				}
				else
				{
					d_last_error = QStringLiteral("Probe offset data not recognized.");
					return false;
				}
				
			}
			else if(line.startsWith(QStringLiteral("Probe_Vector:"),Qt::CaseInsensitive))
			{
				found_probe_vector = true;
				
				text = line.mid(13);	// remove 'Probe_Vector:'
				list = text.split(',');
				
				if(list.size() == 3)
				{
					d_probe_data_vector.x = list[0].toDouble();
					d_probe_data_vector.y = list[1].toDouble();
					d_probe_data_vector.z = list[2].toDouble();
				}
				else
				{
					d_last_error = QStringLiteral("Probe vector data not recognized.");
					return false;
				}
			}
			else
			{
				if(inside_alignment == true &&
				   inside_data == false &&
				   inside_temperature == false &&
				   inside_scale == false)
				{
					if(line.startsWith(QStringLiteral("X_Axis:"),Qt::CaseInsensitive))
					{
						text = line.mid(7);	// remove 'X_Axis:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.X(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("X Alignment data not recognized.");
							return false;
						}
					}
					else if(line.startsWith(QStringLiteral("Y_Axis:"),Qt::CaseInsensitive))
					{
						text = line.mid(7);	// remove 'Y_Axis:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.Y(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("Y Alignment data not recognized.");
							return false;
						}
					}
					else if(line.startsWith(QStringLiteral("Z_Axis:"),Qt::CaseInsensitive))
					{
						text = line.mid(7);	// remove 'Z_Axis:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.Z(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("Z Alignment data not recognized.");
							return false;
						}
					}
					else if(line.startsWith(QStringLiteral("Translation:"),Qt::CaseInsensitive))
					{
						text = line.mid(12);	// remove 'Translation:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.Origin(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("Alignment translation data not recognized.");
							return false;
						}
					}
				}
				else if(inside_alignment == false &&
						inside_data == false &&
						inside_temperature == true &&
						inside_scale == false)
				{
					if(line.startsWith(QStringLiteral("X_Scale:"),Qt::CaseInsensitive))
					{
						text = line.mid(8);	// remove 'X_Scale:'
						d_temperature_x = text.toDouble();
					}
					else if(line.startsWith(QStringLiteral("Y_Scale:"),Qt::CaseInsensitive))
					{
						text = line.mid(8);	// remove 'Y_Scale:'
						d_temperature_y = text.toDouble();
					}
					else if(line.startsWith(QStringLiteral("Z_Scale:"),Qt::CaseInsensitive))
					{
						text = line.mid(8);	// remove 'Z_Scale:'
						d_temperature_z = text.toDouble();
					}
					else if(line.startsWith(QStringLiteral("Gauge:"),Qt::CaseInsensitive))
					{
						text = line.mid(6);	// remove 'Gauge:'
						d_temperature_gauge = text.toDouble();
					}
				}
				else if(inside_alignment == false &&
						inside_data == false &&
						inside_temperature == false &&
						inside_scale == true)
				{
					if(line.startsWith(QStringLiteral("X_Expansion_Coefficient:"),Qt::CaseInsensitive))
					{
						text = line.mid(24);	// remove 'X_Expansion_Coefficient:'
						d_scale_expansion_coefficient.x = text.toDouble();
					}
					else if(line.startsWith(QStringLiteral("Y_Expansion_Coefficient:"),Qt::CaseInsensitive))
					{
						text = line.mid(24);	// remove 'Y_Expansion_Coefficient:'
						d_scale_expansion_coefficient.y = text.toDouble();
					}
					else if(line.startsWith(QStringLiteral("Z_Expansion_Coefficient:"),Qt::CaseInsensitive))
					{
						text = line.mid(24);	// remove 'Z_Expansion_Coefficient:'
						d_scale_expansion_coefficient.z = text.toDouble();
					}
				}
				else if(inside_alignment == false &&
						inside_data == true &&
						inside_temperature == false &&
						inside_scale == false)
				{
					if(line.startsWith(QStringLiteral("Description:"),Qt::CaseInsensitive))
					{
						found_description = true;
						d_measurement_description = line.mid(12);	// remove 'Description:'
					}
					else if(line.startsWith(QStringLiteral("Datum_Flatness:"),Qt::CaseInsensitive))
					{
						found_datum_flatness = true;
						text = line.mid(15);	// remove 'Datum_Flatness:'
						
						d_datum_flatness = text.toDouble();
					}
					else if(line.startsWith(QStringLiteral("Point:"),Qt::CaseInsensitive))
					{
						text = line.mid(6);	// remove 'Point:'
						
						list = text.split(',');
						
						if(list.size() == 5)
						{
							point_data = list[2].toDouble();
							
							d_point_data.push_back(point_data);
						}
					}
				}
			}
		}
	}
	
	if(found_alignment == false ||
	   found_id == false ||
	   found_description == false ||
	   found_scale == false ||
	   found_temperature == false ||
	   found_probe_offset == false ||
	   found_probe_vector == false ||
	   found_datum_flatness == false ||
	   found_data == false)
	{
		d_last_error = QStringLiteral("Data missing from measurement file.");
		return false;
	}
	
	if(d_point_data.size() != 6)
	{
		d_last_error = QStringLiteral("Data missing from measurement file.");
		return false;
	}
	
	length_data = d_point_data[1] - d_point_data[0];
	d_length_data.push_back(length_data);
	
	length_data = d_point_data[3] - d_point_data[2];
	d_length_data.push_back(length_data);
	
	length_data = d_point_data[5] - d_point_data[4];
	d_length_data.push_back(length_data);
	
	return true;
}
