/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QStringList>
#include <QByteArray>
#include <QTextStream>

#include "../../../core/vector2.h"

#include "measuredata_machinelaser10360.h"

struct Sort_Functor_Index
{
	inline bool operator() (const TMeasureMachineLaser10360::TMeasuredData &m1,
							const TMeasureMachineLaser10360::TMeasuredData &m2)
	{
		return (m1.index < m2.index);
	}
};

TMeasureMachineLaser10360::TMeasureMachineLaser10360(void)
{
	d_file_type = TMeasureData::TYPE_MACHINE_LASER_10360;
}

TMeasureMachineLaser10360::~TMeasureMachineLaser10360(void)
{
}

QStringList TMeasureMachineLaser10360::Text_Report(void) const
{
	std::vector<TMeasureMachineLaser10360::TLengthData>::const_iterator iter;
	QStringList							list;
	QString								text;
	double								deviation;
	TVector3							vec;
	
	list.push_back(QStringLiteral("ASME B89.4.10360 Laser Measurement"));
	list.push_back(QStringLiteral("================================================================"));
	list.push_back(QString());
	
	list.push_back(QStringLiteral("Description"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	text = QString("  Position:         %1").arg(d_measurement_description);
	list.push_back(text);
	
	switch(d_measurement_type)
	{
		case TMeasureMachineLaser10360::MEASUREMENT_SCALE:
			text = QStringLiteral("Scale");
			break;
		case TMeasureMachineLaser10360::MEASUREMENT_STRAIGHTNESS_X:
			text = QStringLiteral("Straightness in X");
			break;
		case TMeasureMachineLaser10360::MEASUREMENT_STRAIGHTNESS_Y:
			text = QStringLiteral("Straightness in Y");
			break;
		case TMeasureMachineLaser10360::MEASUREMENT_STRAIGHTNESS_Z:
			text = QStringLiteral("Straightness in Z");
			break;
		case TMeasureMachineLaser10360::MEASUREMENT_ANGULAR:
			text = QStringLiteral("Angular");
			break;
	}
	
	text = QString("  Measurement_Type: %1").arg(text);
	list.push_back(text);
	
	list.push_back(QString());
	list.push_back(QStringLiteral("Alignment"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	text = QString("  Start Point: %1, %2, %3").arg(d_start_point.x,12,'f',4).arg(d_start_point.y,12,'f',4).arg(d_start_point.z,12,'f',4);
	list.push_back(text);
	
	text = QString("  Direction:   %1, %2, %3").arg(d_measurement_axis.i,12,'f',9).arg(d_measurement_axis.j,12,'f',9).arg(d_measurement_axis.k,12,'f',9);
	list.push_back(text);
	
	list.push_back(QString());
	list.push_back(QStringLiteral("Tool"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());

	text = QString("  Offset:      %1, %2, %3").arg(tool_offset.x,12,'f',4).arg(tool_offset.y,12,'f',4).arg(tool_offset.z,12,'f',4);
	list.push_back(text);
	
	list.push_back(QString());
	list.push_back(QStringLiteral("Measurement Lengths mm"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	list.push_back(QStringLiteral("     Nominal        Actual      Deviation"));
	list.push_back(QString());
	
	for(iter = d_length_data.begin();iter != d_length_data.end();++iter)
	{
		deviation = (*iter).actual - (*iter).nominal;
		
		text = QString("%1  %2  %3").arg((*iter).nominal,12,'f',4).arg((*iter).actual,12,'f',4).arg(deviation,12,'f',4);
		list.push_back(text);
	}
	
	return list;
}

std::vector<std::vector<TVector2> > TMeasureMachineLaser10360::Graph_Data(void) const
{
	std::vector<std::vector<TVector2> > graph_data;
	std::vector<TVector2>				graph_item_points_1;
	std::vector<TVector2>				graph_item_points_2;
	std::vector<TVector2>				graph_item_points_3;
	TVector2							pnt;
	TVector2							vec;
	double								deviation;
	std::vector<TLengthData>::const_iterator iter;
	
	if(d_length_data.size() == 15)
	{
		for(iter = d_length_data.begin();iter != d_length_data.end();iter += 3)
		{
			deviation = (*iter).actual - (*iter).nominal;
			pnt.Set((*iter).actual,deviation);
			
			graph_item_points_1.push_back(pnt);
			
			deviation = (*(iter+1)).actual - (*(iter+1)).nominal;
			pnt.Set((*iter).actual,deviation);
			
			graph_item_points_2.push_back(pnt);
			
			deviation = (*(iter+2)).actual - (*(iter+2)).nominal;
			pnt.Set((*iter).actual,deviation);
			
			graph_item_points_3.push_back(pnt);
		}
		
		graph_data.push_back(graph_item_points_1);
		graph_data.push_back(graph_item_points_2);
		graph_data.push_back(graph_item_points_3);
	}
	
	return graph_data;
}

bool TMeasureMachineLaser10360::Load_Data(
	const QByteArray					&file_data)
{
	QTextStream							stream(file_data,QIODevice::ReadOnly);
	TMeasureMachineLaser10360::TMeasuredData measured_data;
	QString								line;
	QString								text;
	QString								text_section;
	QStringList							list;
	bool								found_start_point(false);
	bool								found_measurement_axis(false);
	bool								found_measurement_type(false);
	bool								found_description(false);
	bool								found_probe_offset(false);
	bool								found_data(false);
	bool								inside_data(false);
	bool								init(true);
	
	d_measured_data.clear();
	
	while(!stream.atEnd())
	{
		line = stream.readLine().trimmed();
		
		if(line.startsWith('#') ||
		   line.length() == 0)
		{
			continue;
		}
		
		if(init)
		{
			if(!line.startsWith(QStringLiteral("B89.4.10360_Raw_Measurement:Version=1:Type=Machine_And_Laser")))
			{
				d_last_error = QStringLiteral("Machine_Laser identification line not recognized");
				return false;
			}
			
			init = false;
		}
		else
		{
			if(line.startsWith(QStringLiteral("Probe_Offset:"),Qt::CaseInsensitive))
			{
				found_probe_offset = true;
				
				text = line.mid(13);	// remove 'Probe_Offset:'
				list = text.split(',');
				
				if(list.size() == 3)
				{
					tool_offset.x = list[0].toDouble();
					tool_offset.y = list[1].toDouble();
					tool_offset.z = list[2].toDouble();
				}
				else
				{
					d_last_error = QStringLiteral("Probe offset data not recognized.");
					return false;
				}
			}
			else if(line.startsWith(QStringLiteral("Start_Point:"),Qt::CaseInsensitive))
			{
				found_start_point = true;
				
				text = line.mid(12);	// remove 'Start_Point:'
				list = text.split(',');
				
				if(list.size() == 3)
				{
					d_start_point.x = list[0].toDouble();
					d_start_point.y = list[1].toDouble();
					d_start_point.z = list[2].toDouble();
				}
				else
				{
					d_last_error = QStringLiteral("Start point data not recognized.");
					return false;
				}
			}
			else if(line.startsWith(QStringLiteral("Measurement_Axis:"),Qt::CaseInsensitive))
			{
				found_measurement_axis = true;
				
				text = line.mid(17);	// remove 'Measurement_Axis:'
				list = text.split(',');
				
				if(list.size() == 3)
				{
					d_measurement_axis.x = list[0].toDouble();
					d_measurement_axis.y = list[1].toDouble();
					d_measurement_axis.z = list[2].toDouble();
				}
				else
				{
					d_last_error = QStringLiteral("Measurement Axis data not recognized.");
					return false;
				}
			}
			else if(line.startsWith(QStringLiteral("Measurement_Type:"),Qt::CaseInsensitive))
			{
				found_measurement_type = true;
				
				text = line.mid(17);	// remove 'Measurement_Type:'
				
				if(text.contains(QStringLiteral("Scale"),Qt::CaseInsensitive))
				{
					d_measurement_type = TMeasureMachineLaser10360::MEASUREMENT_SCALE;
				}
				else if(text.contains(QStringLiteral("Straightness_X"),Qt::CaseInsensitive))
				{
					d_measurement_type = TMeasureMachineLaser10360::MEASUREMENT_STRAIGHTNESS_X;
				}
				else if(text.contains(QStringLiteral("Straightness_Y"),Qt::CaseInsensitive))
				{
					d_measurement_type = TMeasureMachineLaser10360::MEASUREMENT_STRAIGHTNESS_Y;
				}
				else if(text.contains(QStringLiteral("Straightness_Z"),Qt::CaseInsensitive))
				{
					d_measurement_type = TMeasureMachineLaser10360::MEASUREMENT_STRAIGHTNESS_Z;
				}
				else if(text.contains(QStringLiteral("Angular"),Qt::CaseInsensitive))
				{
					d_measurement_type = TMeasureMachineLaser10360::MEASUREMENT_ANGULAR;
				}
				else
				{
					d_last_error = QStringLiteral("Measurement Type not recognized.");
					return false;
				}
			}
			else if(line.compare(QStringLiteral("Measurement_Begin:"),Qt::CaseInsensitive) == 0)
			{
				found_data = true;
				inside_data = true;
			}
			else if(line.compare(QStringLiteral("Measurement_End::"),Qt::CaseInsensitive) == 0)
			{
				inside_data = false;
			}
			else
			{
				if(inside_data == true)
				{
					if(line.startsWith(QStringLiteral("Description:"),Qt::CaseInsensitive))
					{
						found_description = true;
						d_measurement_description = line.mid(12);	// remove 'Description:'
					}
					else if(line.startsWith(QStringLiteral("Point:"),Qt::CaseInsensitive))
					{
						text = line.mid(6);	// remove 'Point:'
						
						list = text.split(',');
						
						if(list.size() > 7)
						{
							measured_data.index = list[0].toInt();
							measured_data.nominal_target = list[1].toDouble();
							measured_data.actual_target = list[2].toDouble();
							measured_data.laser_value = list[3].toDouble();
							measured_data.position.x = list[4].toDouble();
							measured_data.position.y = list[5].toDouble();
							measured_data.position.z = list[6].toDouble();
							measured_data.deviation = list[7].toDouble();
							
							d_measured_data.push_back(measured_data);
						}
						
						
					}
				}
			}
		}
	}
	
	if(found_probe_offset == false ||
	   found_start_point == false ||
	   found_measurement_axis == false ||
	   found_measurement_type == false ||
	   found_description == false ||
	   found_data == false)
	{
		d_last_error = QStringLiteral("Data missing from measurement file.");
		return false;
	}
	
	if(d_measured_data.size() != 30)
	{
		d_last_error = QStringLiteral("Data missing from measurement file.");
		return false;
	}
	
	this->Update_Length_Data();
	
	return true;
}


void TMeasureMachineLaser10360::Update_Length_Data(void)
{
	TLengthData							length_data;
	std::vector<TMeasuredData>::const_iterator iter;
	
	d_length_data.clear();
	
	if(d_measured_data.size() == 30)	// pattern of points is assumed;
	{
		// make sure everything is in the proper order
		std::sort(d_measured_data.begin(), d_measured_data.end(), Sort_Functor_Index());
		
		for(iter = d_measured_data.begin();iter != d_measured_data.end();iter += 2)
		{
			length_data.nominal = (*(iter+1)).laser_value - (*iter).laser_value;
			length_data.actual = (*(iter+1)).actual_target - (*iter).actual_target;
			d_length_data.push_back(length_data);
		}
	}
}

