/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QStringList>
#include <QByteArray>
#include <QTextStream>

#include "../../../core/vector2.h"

#include "measuredata_pingauge.h"

TMeasurePin::TMeasurePin(void)
{
	d_file_type = TMeasureData::TYPE_PINGAUGE;
}

TMeasurePin::~TMeasurePin(void)
{
}

QStringList TMeasurePin::Text_Report(void) const
{
	std::vector<TMeasurePin::TPointData>::const_iterator iter;
	QStringList							list;
	QString								text;
	TVector3							vec;
	
	list.push_back(QStringLiteral("Pin Gauge Measurement"));
	list.push_back(QStringLiteral("================================================================"));
	list.push_back(QString());
	
	list.push_back(QStringLiteral("Alignment"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	vec = d_alignment_data.X();
	text = QString("  X Axis: %1, %2, %3").arg(vec.x,12,'f',9).arg(vec.y,12,'f',9).arg(vec.z,12,'f',9);
	list.push_back(text);
	
	vec = d_alignment_data.Y();
	text = QString("  Y Axis: %1, %2, %3").arg(vec.x,12,'f',9).arg(vec.y,12,'f',9).arg(vec.z,12,'f',9);
	list.push_back(text);

	vec = d_alignment_data.Z();
	text = QString("  Z Axis: %1, %2, %3").arg(vec.x,12,'f',9).arg(vec.y,12,'f',9).arg(vec.z,12,'f',9);
	list.push_back(text);

	vec = d_alignment_data.Origin();
	text = QString("  Origin: %1, %2, %3").arg(vec.x,12,'f',4).arg(vec.y,12,'f',4).arg(vec.z,12,'f',4);
	list.push_back(text);
	
	list.push_back(QString());
	list.push_back(QStringLiteral("Tool"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	text = QString("  Offset: %1, %2, %3").arg(tool_offset.x,12,'f',4).arg(tool_offset.y,12,'f',4).arg(tool_offset.z,12,'f',4);
	list.push_back(text);
	
	text = QString("  Vector: %1, %2, %3").arg(d_probe_data_vector.x,12,'f',9).arg(d_probe_data_vector.y,12,'f',9).arg(d_probe_data_vector.z,12,'f',9);
	list.push_back(text);

	list.push_back(QString());
	list.push_back(QStringLiteral("Feature"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());

	text = QString("  Description: %1").arg(d_circle_id);
	list.push_back(text);

	text = QString("          XYZ: %1, %2, %3").arg(d_circle_position.x,0,'f',4).arg(d_circle_position.y,0,'f',4).arg(d_circle_position.z,0,'f',4);
	list.push_back(text);

	text = QString("         Diam: %1 mm").arg(d_circle_diameter,0,'f',4);
	list.push_back(text);

	text = QString("         Form: %1 mm").arg(d_circle_form,0,'f',4);
	list.push_back(text);

	list.push_back(QString());
	list.push_back(QStringLiteral("Data Points"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	list.push_back(QStringLiteral("        X             Y             Z             I             J             K           Dev"));

	for(iter = d_point_data.begin();iter != d_point_data.end();++iter)
	{
		text = QString("%1, %2, %3,").arg((*iter).xyz.x,12,'f',4).arg((*iter).xyz.y,12,'f',4).arg((*iter).xyz.z,12,'f',4);
		text += QString(" %1, %2, %3,").arg((*iter).ijk.x,12,'f',9).arg((*iter).ijk.y,12,'f',9).arg((*iter).ijk.z,12,'f',9);
		text += QString(" %1").arg((*iter).deviation,12,'f',4);
		list.push_back(text);
	}
	
	return list;
}

std::vector<std::vector<TVector2> > TMeasurePin::Graph_Data(void) const
{
	std::vector<std::vector<TVector2> > graph_data;
	std::vector<TVector2>				graph_item_nominal;
	std::vector<TVector2>				graph_item_points;
	TVector2							pnt;
	TVector2							vec;
	std::vector<TPointData>::const_iterator iter;
	double								dev_multiplier(1.0);
	static const double					MIN_FORM(0.0001);
	static const double					MAX_MULTIPLIER(10000);
	
	if(d_circle_form > MIN_FORM)
	{
		dev_multiplier = 0.25 * d_circle_diameter / d_circle_form;
	}
	
	if(dev_multiplier > MAX_MULTIPLIER)
	{
		dev_multiplier = MAX_MULTIPLIER;
	}
	
	if(dev_multiplier < 1.0)
	{
		dev_multiplier = 1.0;
	}
	
	for(iter = d_point_data.begin();iter != d_point_data.end();++iter)
	{
		pnt.Set((*iter).xyz.x,(*iter).xyz.y);
		
		graph_item_nominal.push_back(pnt);
		
		vec.Set((*iter).ijk.x,(*iter).ijk.y);
		
		pnt += (vec * (*iter).deviation * dev_multiplier);
		graph_item_points.push_back(pnt);
	}
	
	// wrap last point to start
	if(d_point_data.size())
	{
		graph_item_nominal.push_back(graph_item_nominal[0]);
		graph_item_points.push_back(graph_item_points[0]);
	}
	
	graph_data.push_back(graph_item_nominal);
	graph_data.push_back(graph_item_points);
	graph_data.push_back(graph_item_points);
	
	return graph_data;
}

bool TMeasurePin::Load_Data(
	const QByteArray					&file_data)
{
	QTextStream							stream(file_data,QIODevice::ReadOnly);
	TMeasurePin::TPointData				point_data;
	QString								line;
	QString								text;
	QString								text_section;
	QStringList							list;
	int									index;
	bool								found_alignment(false);
	bool								inside_alignment(false);
	bool								found_probe_offset(false);
	bool								found_probe_vector(false);
	bool								found_results(false);
	bool								found_data(false);
	bool								inside_data(false);
	bool								init(true);
	
	d_point_data.clear();
	
	while(!stream.atEnd())
	{
		line = stream.readLine().trimmed();
		
		if(line.startsWith('#') ||
		   line.length() == 0)
		{
			continue;
		}
		
		if(init)
		{
			if(!line.startsWith(QStringLiteral("Artifact_Measurement:Version=1:Type=Pin_Gauge")))
			{
				d_last_error = QStringLiteral("Artifact identification line not recognized");
				return false;
			}
			
			init = false;
		}
		else
		{
			if(line.compare(QStringLiteral("Alignment_Begin:"),Qt::CaseInsensitive) == 0)
			{
				found_alignment = true;
				inside_alignment = true;
			}
			else if(line.compare(QStringLiteral("Alignment_End:"),Qt::CaseInsensitive) == 0)
			{
				inside_alignment = false;
			}
			else if(line.compare(QStringLiteral("Point_Data_Begin:"),Qt::CaseInsensitive) == 0)
			{
				found_data = true;
				inside_data = true;
			}
			else if(line.compare(QStringLiteral("Point_Data_End::"),Qt::CaseInsensitive) == 0)
			{
				inside_data = false;
			}
			else if(line.startsWith(QStringLiteral("Probe_Offset:"),Qt::CaseInsensitive))
			{
				found_probe_offset = true;
				
				text = line.mid(13);	// remove 'Probe_Offset:'
				list = text.split(',');
				
				if(list.size() == 3)
				{
					tool_offset.x = list[0].toDouble();
					tool_offset.y = list[1].toDouble();
					tool_offset.z = list[2].toDouble();
				}
				else
				{
					d_last_error = QStringLiteral("Probe offset data not recognized.");
					return false;
				}
				
			}
			else if(line.startsWith(QStringLiteral("Probe_Vector:"),Qt::CaseInsensitive))
			{
				found_probe_vector = true;
				
				text = line.mid(13);	// remove 'Probe_Vector:'
				list = text.split(',');
				
				if(list.size() == 3)
				{
					d_probe_data_vector.x = list[0].toDouble();
					d_probe_data_vector.y = list[1].toDouble();
					d_probe_data_vector.z = list[2].toDouble();
				}
				else
				{
					d_last_error = QStringLiteral("Probe vector data not recognized.");
					return false;
				}
			}
			else
			{
				if(inside_alignment == false &&
				   inside_data == false)
				{
					if(line.startsWith(QStringLiteral("Circle"),Qt::CaseInsensitive))
					{
						found_results = true;
						
						text = line;
						index = text.indexOf(QStringLiteral("Form:"),Qt::CaseInsensitive);
						
						if(index < 0)
						{
							d_last_error = QStringLiteral("Feature form error not found.");
							return false;
						}
						
						text_section = text.mid(index + 5);
						d_circle_form = text_section.toDouble();
						
						text.truncate(index);
						
						index = text.indexOf(QStringLiteral("Diam:"),Qt::CaseInsensitive);
						
						if(index < 0)
						{
							d_last_error = QStringLiteral("Feature diameter not found.");
							return false;
						}
						
						text_section = text.mid(index + 5);
						d_circle_diameter = text_section.toDouble();
						
						text.truncate(index);
						
						index = text.indexOf(QStringLiteral("XYZ:"),Qt::CaseInsensitive);
						
						if(index < 0)
						{
							d_last_error = QStringLiteral("Feature position not found.");
							return false;
						}
						
						text_section = text.mid(index + 4);
						list = text_section.split(',');

						if(list.size() == 3)
						{
							d_circle_position.x = list[0].toDouble();
							d_circle_position.y = list[1].toDouble();
							d_circle_position.z = list[2].toDouble();
						}
						else
						{
							d_last_error = QStringLiteral("Feature position not recognized.");
							return false;
						}
						
						text.truncate(index);

						d_circle_id = text.trimmed();
					}
				}
				else if(inside_alignment == true &&
						inside_data == false)
				{
					if(line.startsWith(QStringLiteral("X_Axis:"),Qt::CaseInsensitive))
					{
						text = line.mid(7);	// remove 'X_Axis:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.X(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("X Alignment data not recognized.");
							return false;
						}
					}
					else if(line.startsWith(QStringLiteral("Y_Axis:"),Qt::CaseInsensitive))
					{
						text = line.mid(7);	// remove 'Y_Axis:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.Y(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("Y Alignment data not recognized.");
							return false;
						}
					}
					else if(line.startsWith(QStringLiteral("Z_Axis:"),Qt::CaseInsensitive))
					{
						text = line.mid(7);	// remove 'Z_Axis:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.Z(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("Z Alignment data not recognized.");
							return false;
						}
					}
					else if(line.startsWith(QStringLiteral("Translation:"),Qt::CaseInsensitive))
					{
						text = line.mid(12);	// remove 'Translation:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.Origin(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("Alignment translation data not recognized.");
							return false;
						}
					}
				}
				else if(inside_alignment == false &&
						inside_data == true)
				{
					list = line.split(',');
					
					if(list.size() == 7)
					{
						point_data.xyz.x = list[0].toDouble();
						point_data.xyz.y = list[1].toDouble();
						point_data.xyz.z = list[2].toDouble();
					
						point_data.ijk.x = list[3].toDouble();
						point_data.ijk.y = list[4].toDouble();
						point_data.ijk.z = list[5].toDouble();

						point_data.deviation = list[6].toDouble();
						
						d_point_data.push_back(point_data);
					}
				}
			}
		}
	}
	
	if(found_alignment == false ||
	   found_probe_offset == false ||
	   found_probe_vector == false ||
	   found_results == false ||
	   found_data == false)
	{
		d_last_error = QStringLiteral("Data missing from measurement file.");
		return false;
	}
	
	return true;
}
