/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QStringList>
#include <QByteArray>
#include <QTextStream>

#include "../../../core/vector2.h"

#include "measuredata_rolloffsettool.h"

TMeasureRollOffsetTool::TMeasureRollOffsetTool(void)
{
	d_file_type = TMeasureData::TYPE_ROLL_OFFSET_TOOL;
}

TMeasureRollOffsetTool::~TMeasureRollOffsetTool(void)
{
}

QStringList TMeasureRollOffsetTool::Text_Report(void) const
{
	std::vector<TVector3>::const_iterator iter;
	QStringList							list;
	QString								text;
	QString								report_title;
	TVector3							vec;
	TVector3							sph1;
	TVector3							sph2;
	TVector3							tool_difference;
	TVector3							sphere_difference;
	TVector3							meas_vector;
	double								tool_offset_difference;
	double								sphere_height_difference;
	double								diff1,diff2;
	double								difference;
	int									axis(0);
	static const double					ZERO_EPSILON(0.000001);
	static const double					MINIMUM_DISTANCE(100.0);
	
	list.push_back(QStringLiteral("Roll from Offset Tools"));
	list.push_back(QStringLiteral("================================================================"));
	list.push_back(QString());
	
	list.push_back(QString());
	list.push_back(QStringLiteral("Tools"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	text = QString("  Tool 1 Offset: %1, %2, %3").arg(d_tool1_offset.x,12,'f',4).arg(d_tool1_offset.y,12,'f',4).arg(d_tool1_offset.z,12,'f',4);
	list.push_back(text);
	
	text = QString("  Tool 1 Vector: %1, %2, %3").arg(d_tool1_vector.x,12,'f',9).arg(d_tool1_vector.y,12,'f',9).arg(d_tool1_vector.z,12,'f',9);
	list.push_back(text);
	
	text = QString("  Tool 2 Offset: %1, %2, %3").arg(d_tool2_offset.x,12,'f',4).arg(d_tool2_offset.y,12,'f',4).arg(d_tool2_offset.z,12,'f',4);
	list.push_back(text);
	
	text = QString("  Tool 2 Vector: %1, %2, %3").arg(d_tool2_vector.x,12,'f',9).arg(d_tool2_vector.y,12,'f',9).arg(d_tool2_vector.z,12,'f',9);
	list.push_back(text);
	list.push_back(QString());
	
	list.push_back(QString());
	list.push_back(QStringLiteral("Spheres"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	text = QString("  Sphere 1 (Tool 1): %1, %2, %3").arg(d_sphere1_tool1.x,12,'f',4).arg(d_sphere1_tool1.y,12,'f',4).arg(d_sphere1_tool1.z,12,'f',4);
	text += QString("  Diam: %1  Form:%2").arg(d_sphere1_tool1_diameter,9,'f',4).arg(d_sphere1_tool1_form,7,'f',4);
	list.push_back(text);
	
	text = QString("  Sphere 1 (Tool 2): %1, %2, %3").arg(d_sphere1_tool2.x,12,'f',4).arg(d_sphere1_tool2.y,12,'f',4).arg(d_sphere1_tool2.z,12,'f',4);
	text += QString("  Diam: %1  Form:%2").arg(d_sphere1_tool2_diameter,9,'f',4).arg(d_sphere1_tool2_form,7,'f',4);
	list.push_back(text);
	
	text = QString("  Sphere 2 (Tool 2): %1, %2, %3").arg(d_sphere2_tool2.x,12,'f',4).arg(d_sphere2_tool2.y,12,'f',4).arg(d_sphere2_tool2.z,12,'f',4);
	text += QString("  Diam: %1  Form:%2").arg(d_sphere2_tool2_diameter,9,'f',4).arg(d_sphere2_tool2_form,7,'f',4);
	list.push_back(text);
	
	text = QString("  Sphere 2 (Tool 1): %1, %2, %3").arg(d_sphere2_tool1.x,12,'f',4).arg(d_sphere2_tool1.y,12,'f',4).arg(d_sphere2_tool1.z,12,'f',4);
	text += QString("  Diam: %1  Form:%2").arg(d_sphere2_tool1_diameter,9,'f',4).arg(d_sphere2_tool1_form,7,'f',4);
	list.push_back(text);
	
	tool_difference = d_tool1_offset - d_tool2_offset;
	
	sph1 = (d_sphere1_tool1 + d_sphere1_tool2) / 2;
	sph2 = (d_sphere2_tool1 + d_sphere2_tool2) / 2;
	
	sphere_difference = sph2 - sph1;
	
	if(fabs(sphere_difference.z) > MINIMUM_DISTANCE)
	{
		axis = 3;	// always assume Z axis before other axis
	}
	else if(fabs(sphere_difference.x) > MINIMUM_DISTANCE &&
			fabs(sphere_difference.x) > fabs(sphere_difference.y))
	{
		axis = 1;
	}
	else if(fabs(sphere_difference.y) > MINIMUM_DISTANCE)
	{
		axis = 2;
	}
	else
	{
		list.push_back(QStringLiteral("ERR:  Unable to calculate roll.  Cannot determine correction axis."));
	}

	if(axis)
	{
		switch(axis)
		{
			case 1:
				sphere_difference.y = 0.0;
				sphere_difference.z = 0.0;
				
				report_title = QStringLiteral("Result - X Axis Roll");
				break;
				
			case 2:
				sphere_difference.x = 0.0;
				sphere_difference.z = 0.0;
				
				report_title = QStringLiteral("Result - Y Axis Roll");
				break;
				
			case 3:
				sphere_difference.x = 0.0;
				sphere_difference.y = 0.0;
				
				report_title = QStringLiteral("Result - Z Axis Roll");
				break;
		}
		
		meas_vector = sphere_difference * tool_difference;
		
		if(meas_vector.Length() > ZERO_EPSILON)
		{
			meas_vector.Normal();
			
			tool_offset_difference = tool_difference.Length();
			sphere_height_difference = sphere_difference.Length();
			
			if(tool_offset_difference > MINIMUM_DISTANCE &&
				sphere_height_difference > MINIMUM_DISTANCE)
			{
				list.push_back(QString());
				list.push_back(report_title);
				list.push_back(QStringLiteral("--------------------------------------------"));
				list.push_back(QString());

				diff1 = (d_sphere1_tool1 - d_sphere1_tool2) ^ meas_vector;
				diff2 = (d_sphere2_tool1 - d_sphere2_tool2) ^ meas_vector;
				
				difference = diff2 - diff1;
				
				text = QString("  Measurement error:       %1").arg(difference,12,'f',4);
				list.push_back(text);
				
				text = QString("  Offset between tools:    %1").arg(tool_offset_difference,12,'f',4);
				list.push_back(text);
				
				text = QString("  Offset between spheres:  %1").arg(sphere_height_difference,12,'f',4);
				list.push_back(text);

				difference /= tool_offset_difference;
				difference *= 1000.0;
				
				difference /= sphere_height_difference;
				difference *= 1000.0;
				
				list.push_back(QString());

				text = QString("  Correction gradient:     %1 mm/m (CW = Positive)").arg(difference,12,'f',4);
				list.push_back(text);
				
				difference *= (-1.0);
				text = QString("  Correction gradient:     %1 mm/m (CCW = Positive)").arg(difference,12,'f',4);
				list.push_back(text);
				
				list.push_back(QString());
				list.push_back(QStringLiteral("  Note:  CW/CCW sign refers to the nominal sign of the compensation map data for the axis roll."));
			}
			else
			{
				list.push_back(QStringLiteral("ERR:  Unable to calculate roll.  Tool offset too small or insufficient sphere separation."));
			}
		}
		else
		{
			list.push_back(QStringLiteral("ERR:  Unable to calculate roll.  Tool and correction axis may be aligned."));
		}
	}

	return list;
}

std::vector<std::vector<TVector2> > TMeasureRollOffsetTool::Graph_Data(void) const
{
	std::vector<std::vector<TVector2> > graph_data;
	
	return graph_data;
}

bool TMeasureRollOffsetTool::Load_Data(
	const QByteArray					&file_data)
{
	QTextStream							stream(file_data,QIODevice::ReadOnly);
	std::vector<TVector3>::iterator		iter;
	QString								line;
	QString								text;
	QString								text_section;
	QStringList							list;
	TVector3							pnt;
	int									index;
	bool								found_tool1_offset(false);
	bool								found_tool1_vector(false);
	bool								found_tool2_offset(false);
	bool								found_tool2_vector(false);
	bool								found_sphere1_probe1(false);
	bool								found_sphere1_probe2(false);
	bool								found_sphere2_probe1(false);
	bool								found_sphere2_probe2(false);
	bool								init(true);
	
	while(!stream.atEnd())
	{
		line = stream.readLine().trimmed();
		
		if(line.startsWith('#') ||
		   line.length() == 0)
		{
			continue;
		}
		
		if(init)
		{
			if(!line.startsWith(QStringLiteral("Artifact_Measurement:Version=1:Type=Roll_Offset_Tool")))
			{
				d_last_error = QStringLiteral("Artifact identification line not recognized");
				return false;
			}
			
			init = false;
		}
		else
		{
			if(line.startsWith(QStringLiteral("Tool_Offset_1:"),Qt::CaseInsensitive))
			{
				found_tool1_offset = true;
				
				text = line.mid(14);	// remove 'Tool_Offset_x:'
				list = text.split(',');
				
				if(list.size() == 3)
				{
					d_tool1_offset.x = list[0].toDouble();
					d_tool1_offset.y = list[1].toDouble();
					d_tool1_offset.z = list[2].toDouble();
				}
				else
				{
					d_last_error = QStringLiteral("Tool1 offset data not recognized.");
					return false;
				}
				
			}
			else if(line.startsWith(QStringLiteral("Tool_Vector_1:"),Qt::CaseInsensitive))
			{
				found_tool1_vector = true;
				
				text = line.mid(14);	// remove 'Tool_Vector_x:'
				list = text.split(',');
				
				if(list.size() == 3)
				{
					d_tool1_vector.x = list[0].toDouble();
					d_tool1_vector.y = list[1].toDouble();
					d_tool1_vector.z = list[2].toDouble();
				}
				else
				{
					d_last_error = QStringLiteral("Tool1 vector data not recognized.");
					return false;
				}
			}
			else if(line.startsWith(QStringLiteral("Tool_Offset_2:"),Qt::CaseInsensitive))
			{
				found_tool2_offset = true;
				
				text = line.mid(14);	// remove 'Tool_Offset_x:'
				list = text.split(',');
				
				if(list.size() == 3)
				{
					d_tool2_offset.x = list[0].toDouble();
					d_tool2_offset.y = list[1].toDouble();
					d_tool2_offset.z = list[2].toDouble();
				}
				else
				{
					d_last_error = QStringLiteral("Tool2 offset data not recognized.");
					return false;
				}
				
			}
			else if(line.startsWith(QStringLiteral("Tool_Vector_2:"),Qt::CaseInsensitive))
			{
				found_tool2_vector = true;
				
				text = line.mid(14);	// remove 'Tool_Vector_x:'
				list = text.split(',');
				
				if(list.size() == 3)
				{
					d_tool2_vector.x = list[0].toDouble();
					d_tool2_vector.y = list[1].toDouble();
					d_tool2_vector.z = list[2].toDouble();
				}
				else
				{
					d_last_error = QStringLiteral("Tool2 vector data not recognized.");
					return false;
				}
			}
			else if(line.startsWith(QStringLiteral("Sphere_1_Tool_1"),Qt::CaseInsensitive))
			{
				found_sphere1_probe1 = true;
				
				text = line.mid(15);	// remove 'Sphere_x_Tool_x'
				
				index = text.indexOf(QStringLiteral("XYZ:"),Qt::CaseInsensitive);
				
				if(index < 0)
				{
					d_last_error = QStringLiteral("Sphere1, Tool1, entry not recognized.");
					return false;
				}
				
				text_section = text.mid(index + 4);
				index = text_section.indexOf(QStringLiteral("Diam:"),Qt::CaseInsensitive);
				
				if(index < 0)
				{
					d_last_error = QStringLiteral("Sphere1, Tool1, entry not recognized.");
					return false;
				}

				text_section.truncate(index);
				
				list = text_section.split(',');
				
				if(list.size() == 3)
				{
					d_sphere1_tool1.x = list[0].toDouble();
					d_sphere1_tool1.y = list[1].toDouble();
					d_sphere1_tool1.z = list[2].toDouble();
				}
				else
				{
					d_last_error = QStringLiteral("Sphere1, Tool1, entry not recognized.");
					return false;
				}
				
				index = text.indexOf(QStringLiteral("Diam:"),Qt::CaseInsensitive);
				
				if(index < 0)
				{
					d_last_error = QStringLiteral("Sphere1, Tool1, entry not recognized.");
					return false;
				}
				
				text_section = text.mid(index + 5);
				index = text_section.indexOf(QStringLiteral("Form:"),Qt::CaseInsensitive);
				
				if(index < 0)
				{
					d_last_error = QStringLiteral("Sphere1, Tool1, entry not recognized.");
					return false;
				}

				text_section.truncate(index);

				d_sphere1_tool1_diameter = text_section.toDouble();
				
				index = text.indexOf(QStringLiteral("Form:"),Qt::CaseInsensitive);
				
				if(index < 0)
				{
					d_last_error = QStringLiteral("Sphere1, Tool1, entry not recognized.");
					return false;
				}
				
				text_section = text.mid(index + 5);
				d_sphere1_tool1_form = text_section.toDouble();
			}
			else if(line.startsWith(QStringLiteral("Sphere_1_Tool_2"),Qt::CaseInsensitive))
			{
				found_sphere1_probe2 = true;
				
				text = line.mid(15);	// remove 'Sphere_x_Tool_x'
				
				index = text.indexOf(QStringLiteral("XYZ:"),Qt::CaseInsensitive);
				
				if(index < 0)
				{
					d_last_error = QStringLiteral("Sphere1, Tool2, entry not recognized.");
					return false;
				}
				
				text_section = text.mid(index + 4);
				index = text_section.indexOf(QStringLiteral("Diam:"),Qt::CaseInsensitive);
				
				if(index < 0)
				{
					d_last_error = QStringLiteral("Sphere1, Tool2, entry not recognized.");
					return false;
				}

				text_section.truncate(index);
				
				list = text_section.split(',');
				
				if(list.size() == 3)
				{
					d_sphere1_tool2.x = list[0].toDouble();
					d_sphere1_tool2.y = list[1].toDouble();
					d_sphere1_tool2.z = list[2].toDouble();
				}
				else
				{
					d_last_error = QStringLiteral("Sphere1, Tool2, entry not recognized.");
					return false;
				}
				
				index = text.indexOf(QStringLiteral("Diam:"),Qt::CaseInsensitive);
				
				if(index < 0)
				{
					d_last_error = QStringLiteral("Sphere1, Tool2, entry not recognized.");
					return false;
				}
				
				text_section = text.mid(index + 5);
				index = text_section.indexOf(QStringLiteral("Form:"),Qt::CaseInsensitive);
				
				if(index < 0)
				{
					d_last_error = QStringLiteral("Sphere1, Tool2, entry not recognized.");
					return false;
				}

				text_section.truncate(index);

				d_sphere1_tool2_diameter = text_section.toDouble();
				
				index = text.indexOf(QStringLiteral("Form:"),Qt::CaseInsensitive);
				
				if(index < 0)
				{
					d_last_error = QStringLiteral("Sphere1, Tool2, entry not recognized.");
					return false;
				}
				
				text_section = text.mid(index + 5);
				d_sphere1_tool2_form = text_section.toDouble();
			}
			else if(line.startsWith(QStringLiteral("Sphere_2_Tool_1"),Qt::CaseInsensitive))
			{
				found_sphere2_probe1 = true;
				
				text = line.mid(15);	// remove 'Sphere_x_Tool_x'
				
				index = text.indexOf(QStringLiteral("XYZ:"),Qt::CaseInsensitive);
				
				if(index < 0)
				{
					d_last_error = QStringLiteral("Sphere2, Tool1, entry not recognized.");
					return false;
				}
				
				text_section = text.mid(index + 4);
				index = text_section.indexOf(QStringLiteral("Diam:"),Qt::CaseInsensitive);
				
				if(index < 0)
				{
					d_last_error = QStringLiteral("Sphere2, Tool1, entry not recognized.");
					return false;
				}

				text_section.truncate(index);
				
				list = text_section.split(',');
				
				if(list.size() == 3)
				{
					d_sphere2_tool1.x = list[0].toDouble();
					d_sphere2_tool1.y = list[1].toDouble();
					d_sphere2_tool1.z = list[2].toDouble();
				}
				else
				{
					d_last_error = QStringLiteral("Sphere2, Tool1, entry not recognized.");
					return false;
				}
				
				index = text.indexOf(QStringLiteral("Diam:"),Qt::CaseInsensitive);
				
				if(index < 0)
				{
					d_last_error = QStringLiteral("Sphere2, Tool1, entry not recognized.");
					return false;
				}
				
				text_section = text.mid(index + 5);
				index = text_section.indexOf(QStringLiteral("Form:"),Qt::CaseInsensitive);
				
				if(index < 0)
				{
					d_last_error = QStringLiteral("Sphere2, Tool1, entry not recognized.");
					return false;
				}

				text_section.truncate(index);

				d_sphere2_tool1_diameter = text_section.toDouble();
				
				index = text.indexOf(QStringLiteral("Form:"),Qt::CaseInsensitive);
				
				if(index < 0)
				{
					d_last_error = QStringLiteral("Sphere2, Tool1, entry not recognized.");
					return false;
				}
				
				text_section = text.mid(index + 5);
				d_sphere2_tool1_form = text_section.toDouble();
			}
			else if(line.startsWith(QStringLiteral("Sphere_2_Tool_2"),Qt::CaseInsensitive))
			{
				found_sphere2_probe2 = true;
				
				text = line.mid(15);	// remove 'Sphere_x_Tool_x'
				
				index = text.indexOf(QStringLiteral("XYZ:"),Qt::CaseInsensitive);
				
				if(index < 0)
				{
					d_last_error = QStringLiteral("Sphere2, Tool2, entry not recognized.");
					return false;
				}
				
				text_section = text.mid(index + 4);
				index = text_section.indexOf(QStringLiteral("Diam:"),Qt::CaseInsensitive);
				
				if(index < 0)
				{
					d_last_error = QStringLiteral("Sphere2, Tool2, entry not recognized.");
					return false;
				}

				text_section.truncate(index);
				
				list = text_section.split(',');
				
				if(list.size() == 3)
				{
					d_sphere2_tool2.x = list[0].toDouble();
					d_sphere2_tool2.y = list[1].toDouble();
					d_sphere2_tool2.z = list[2].toDouble();
				}
				else
				{
					d_last_error = QStringLiteral("Sphere2, Tool2, entry not recognized.");
					return false;
				}
				
				index = text.indexOf(QStringLiteral("Diam:"),Qt::CaseInsensitive);
				
				if(index < 0)
				{
					d_last_error = QStringLiteral("Sphere2, Tool2, entry not recognized.");
					return false;
				}
				
				text_section = text.mid(index + 5);
				index = text_section.indexOf(QStringLiteral("Form:"),Qt::CaseInsensitive);
				
				if(index < 0)
				{
					d_last_error = QStringLiteral("Sphere2, Tool2, entry not recognized.");
					return false;
				}

				text_section.truncate(index);

				d_sphere2_tool2_diameter = text_section.toDouble();
				
				index = text.indexOf(QStringLiteral("Form:"),Qt::CaseInsensitive);
				
				if(index < 0)
				{
					d_last_error = QStringLiteral("Sphere2, Tool2, entry not recognized.");
					return false;
				}
				
				text_section = text.mid(index + 5);
				d_sphere2_tool2_form = text_section.toDouble();
			}
		}
	}
	
	if(found_tool1_offset == false ||
	   found_tool2_offset == false ||
	   found_tool1_vector == false ||
	   found_tool2_vector == false ||
	   found_sphere1_probe1 == false ||
	   found_sphere1_probe2 == false ||
	   found_sphere2_probe1 == false ||
	   found_sphere2_probe2 == false)
	{
		d_last_error = QStringLiteral("Data missing from measurement file.");
		return false;
	}

	return true;
}
