/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QStringList>
#include <QByteArray>
#include <QTextStream>
#include <algorithm>
#include <cmath>

#include "../../../core/vector2.h"
#include "../../../core/interpolation.h"


#include "measuredata_stepgaugesquare.h"

// sort by X position
struct TPointData_Sort
{
	bool operator() (TMeasureStepgaugeSquare::TPointData a,TMeasureStepgaugeSquare::TPointData b)
	{
		return (a.actual < b.actual);
	}
};

TMeasureStepgaugeSquare::TMeasureStepgaugeSquare(void)
{
	d_file_type = TMeasureData::TYPE_STEPGAUGE_SQUARE;
	
	d_error_gradient = 0.0;
}

TMeasureStepgaugeSquare::~TMeasureStepgaugeSquare(void)
{
}

QStringList TMeasureStepgaugeSquare::Text_Report(void) const
{
	std::vector<TMeasureStepgaugeSquare::TLengthData>::const_iterator iter;
	QStringList							list;
	QString								text;
	double								deviation;
	TVector3							vec;
	
	list.push_back(QStringLiteral("Stepgauge Squareness Measurement"));
	list.push_back(QStringLiteral("================================================================"));
	list.push_back(QString());
	
	list.push_back(QStringLiteral("Description"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	text = QString("  Position:  %1").arg(d_measurement_description);
	list.push_back(text);
	
	text = QString("  Equipment: %1").arg(d_equipment_identification);
	list.push_back(text);
	
	text = QString("  Ref. Step: %1 mm").arg(d_reference_step_position,0,'f',3);
	list.push_back(text);
	
	list.push_back(QString());
	list.push_back(QStringLiteral("Alignment"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	vec = d_alignment_data.X();
	text = QString("  X Axis: %1, %2, %3").arg(vec.x,12,'f',9).arg(vec.y,12,'f',9).arg(vec.z,12,'f',9);
	list.push_back(text);
	
	vec = d_alignment_data.Y();
	text = QString("  Y Axis: %1, %2, %3").arg(vec.x,12,'f',9).arg(vec.y,12,'f',9).arg(vec.z,12,'f',9);
	list.push_back(text);
	
	vec = d_alignment_data.Z();
	text = QString("  Z Axis: %1, %2, %3").arg(vec.x,12,'f',9).arg(vec.y,12,'f',9).arg(vec.z,12,'f',9);
	list.push_back(text);
	
	vec = d_alignment_data.Origin();
	text = QString("  Origin: %1, %2, %3").arg(vec.x,12,'f',4).arg(vec.y,12,'f',4).arg(vec.z,12,'f',4);
	list.push_back(text);
	
	list.push_back(QString());
	list.push_back(QStringLiteral("Tool"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	text = QString("  Offset: %1, %2, %3").arg(tool_offset.x,12,'f',4).arg(tool_offset.y,12,'f',4).arg(tool_offset.z,12,'f',4);
	list.push_back(text);
	
	text = QString("  Vector: %1, %2, %3").arg(d_probe_data_vector.x,12,'f',9).arg(d_probe_data_vector.y,12,'f',9).arg(d_probe_data_vector.z,12,'f',9);
	list.push_back(text);
	
	list.push_back(QString());
	list.push_back(QStringLiteral("Interpretation"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	
	text = QString("  Slope:     %1 mm/m").arg(d_error_gradient * 1000.0,0,'f',4);
	list.push_back(text);
	
	text = QString("  Intercept: %1 mm").arg(d_error_y_intercept,0,'f',4);
	list.push_back(text);


	list.push_back(QString());
	list.push_back(QStringLiteral("Measurement Lengths"));
	list.push_back(QStringLiteral("--------------------------------------------"));
	list.push_back(QString());
	list.push_back(QStringLiteral("     Nominal        Actual      Deviation"));
	list.push_back(QString());
	
	for(iter = d_length_data.begin();iter != d_length_data.end();++iter)
	{
		deviation = (*iter).actual - (*iter).nominal;
		
		text = QString("%1  %2  %3").arg((*iter).nominal,12,'f',4).arg((*iter).actual,12,'f',4).arg(deviation,12,'f',4);
		list.push_back(text);
	}
	
	return list;
}

std::vector<std::vector<TVector2> > TMeasureStepgaugeSquare::Graph_Data(void) const
{
	std::vector<std::vector<TVector2> > graph_data;
	std::vector<TVector2>				graph_item_length;
	TVector2							pnt;
	TVector2							vec;
	double								deviation;
	std::vector<TLengthData>::const_iterator iter;
	
	for(iter = d_length_data.begin();iter != d_length_data.end();++iter)
	{
		deviation = (*iter).actual - (*iter).nominal;
		pnt.Set((*iter).actual,deviation);
		
		graph_item_length.push_back(pnt);
	}
		
	graph_data.push_back(graph_item_length);
	
	return graph_data;
}

bool TMeasureStepgaugeSquare::Load_Data(
	const QByteArray					&file_data)
{
	QTextStream							stream(file_data,QIODevice::ReadOnly);
	TMeasureStepgaugeSquare::TPointData	point_data;
	QString								line;
	QString								text;
	QString								text_section;
	QStringList							list;
	bool								found_id(false);
	bool								found_description(false);
	bool								found_alignment(false);
	bool								inside_alignment(false);
	bool								found_probe_offset(false);
	bool								found_probe_vector(false);
	bool								found_reference_position(false);
	bool								found_data(false);
	bool								inside_data(false);
	bool								init(true);
	
	d_point_data.clear();
	d_length_data.clear();
	
	while(!stream.atEnd())
	{
		line = stream.readLine().trimmed();
		
		if(line.startsWith('#') ||
		   line.length() == 0)
		{
			continue;
		}
		
		if(init)
		{
			if(!line.startsWith(QStringLiteral("Stepgauge_Raw_Measurement:Version=1:Type=Squareness")))
			{
				d_last_error = QStringLiteral("Stepgauge identification line not recognized");
				return false;
			}
			
			init = false;
		}
		else
		{
			if(line.startsWith(QStringLiteral("Equipment_Identification:"),Qt::CaseInsensitive))
			{
				found_id = true;
				
				d_equipment_identification = line.mid(25);	// remove 'Equipment_Identification:'
			}
			else if(line.compare(QStringLiteral("Alignment_Begin:"),Qt::CaseInsensitive) == 0)
			{
				found_alignment = true;
				inside_alignment = true;
			}
			else if(line.compare(QStringLiteral("Alignment_End:"),Qt::CaseInsensitive) == 0)
			{
				inside_alignment = false;
			}
			else if(line.compare(QStringLiteral("Measurement_Begin:"),Qt::CaseInsensitive) == 0)
			{
				found_data = true;
				inside_data = true;
			}
			else if(line.compare(QStringLiteral("Measurement_End::"),Qt::CaseInsensitive) == 0)
			{
				inside_data = false;
			}
			else if(line.startsWith(QStringLiteral("Probe_Offset:"),Qt::CaseInsensitive))
			{
				found_probe_offset = true;
				
				text = line.mid(13);	// remove 'Probe_Offset:'
				list = text.split(',');
				
				if(list.size() == 3)
				{
					tool_offset.x = list[0].toDouble();
					tool_offset.y = list[1].toDouble();
					tool_offset.z = list[2].toDouble();
				}
				else
				{
					d_last_error = QStringLiteral("Probe offset data not recognized.");
					return false;
				}
				
			}
			else if(line.startsWith(QStringLiteral("Probe_Vector:"),Qt::CaseInsensitive))
			{
				found_probe_vector = true;
				
				text = line.mid(13);	// remove 'Probe_Vector:'
				list = text.split(',');
				
				if(list.size() == 3)
				{
					d_probe_data_vector.x = list[0].toDouble();
					d_probe_data_vector.y = list[1].toDouble();
					d_probe_data_vector.z = list[2].toDouble();
				}
				else
				{
					d_last_error = QStringLiteral("Probe vector data not recognized.");
					return false;
				}
			}
			else
			{
				if(inside_alignment == true &&
				   inside_data == false)
				{
					if(line.startsWith(QStringLiteral("X_Axis:"),Qt::CaseInsensitive))
					{
						text = line.mid(7);	// remove 'X_Axis:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.X(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("X Alignment data not recognized.");
							return false;
						}
					}
					else if(line.startsWith(QStringLiteral("Y_Axis:"),Qt::CaseInsensitive))
					{
						text = line.mid(7);	// remove 'Y_Axis:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.Y(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("Y Alignment data not recognized.");
							return false;
						}
					}
					else if(line.startsWith(QStringLiteral("Z_Axis:"),Qt::CaseInsensitive))
					{
						text = line.mid(7);	// remove 'Z_Axis:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.Z(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("Z Alignment data not recognized.");
							return false;
						}
					}
					else if(line.startsWith(QStringLiteral("Translation:"),Qt::CaseInsensitive))
					{
						text = line.mid(12);	// remove 'Translation:'
						list = text.split(',');
						
						if(list.size() == 3)
						{
							d_alignment_data.Origin(list[0].toDouble(),list[1].toDouble(),list[2].toDouble());
						}
						else
						{
							d_last_error = QStringLiteral("Alignment translation data not recognized.");
							return false;
						}
					}
				}
				else if(inside_alignment == false &&
						inside_data == true)
				{
					if(line.startsWith(QStringLiteral("Description:"),Qt::CaseInsensitive))
					{
						found_description = true;
						d_measurement_description = line.mid(12);	// remove 'Description:'
					}
					else if(line.startsWith(QStringLiteral("Reference_Step_Position:"),Qt::CaseInsensitive))
					{
						found_reference_position = true;
						text = line.mid(24);	// remove 'Reference_Step_Position:'
						
						d_reference_step_position = text.toDouble();
					}
					else if(line.startsWith(QStringLiteral("Point:"),Qt::CaseInsensitive))
					{
						text = line.mid(6);	// remove 'Point:'
						
						list = text.split(',');
						
						if(list.size() == 5)
						{
							point_data.actual = list[2].toDouble();
							point_data.nominal = point_data.actual;
							
							d_point_data.push_back(point_data);
						}
					}
				}
			}
		}
	}
	
	if(found_alignment == false ||
	   found_id == false ||
	   found_description == false ||
	   found_probe_offset == false ||
	   found_probe_vector == false ||
	   found_reference_position == false ||
	   found_data == false)
	{
		d_last_error = QStringLiteral("Data missing from measurement file.");
		return false;
	}
	
	this->Consolidate_Points();
	this->Update_Length_Data();
	
	return true;
}

bool TMeasureStepgaugeSquare::Set_Nominal_Data(
	const std::vector<TVector3>			&data)
{
	std::vector<TPointData>::iterator	iter;
	double								dval;
	double								nominal;
	bool								valid;
	
	for(iter = d_point_data.begin();iter != d_point_data.end();++iter)
	{
		dval = (*iter).actual;
		dval += d_reference_step_position;
		
		nominal = this->Get_Nominal(dval,data,&valid);
		
		if(valid)
		{
			(*iter).nominal = nominal - d_reference_step_position;
		}
		else
		{
			d_last_error = QStringLiteral("Nominal data not found.  Possible wrong reference step position or measurement error exceeds maximum allowed.");
			
			return false;
		}
	}
	
	this->Update_Length_Data();
	
	return true;
}

void TMeasureStepgaugeSquare::Consolidate_Points(void)
{
	std::vector<TPointData>				new_point_data;
	std::vector<TPointData>::const_iterator iter;
	TPointData_Sort						sort_actual;
	TPointData							point;
	double								actual;
	double								actual_sum;
	int									cntr;
	bool								init;
	static const double					MIN_DISTANCE(1.0);
	
	std::sort(d_point_data.begin(), d_point_data.end(), sort_actual);

	init = true;
	cntr = 0;
	for(iter = d_point_data.begin();iter != d_point_data.end();++iter)
	{
		if(init)
		{
			actual = (*iter).actual;
			cntr = 1;
			actual_sum = actual;
			
			init = false;
		}
		else
		{
			if(fabs(actual - (*iter).actual) > MIN_DISTANCE)
			{
				actual = (*iter).actual;

				point.actual = actual_sum / static_cast<double>(cntr);
				point.nominal = point.actual;
				
				new_point_data.push_back(point);
				
				actual_sum = actual;
				cntr = 1;
			}
			else
			{
				actual = (*iter).actual;
				actual_sum += actual;
				++cntr;
			}
		}
	}
	
	if(cntr)
	{
		point.actual = actual_sum / static_cast<double>(cntr);
		point.nominal = point.actual;
		
		new_point_data.push_back(point);
	}
	
	d_point_data = new_point_data;
}


void TMeasureStepgaugeSquare::Update_Length_Data(void)
{
	std::vector<TPointData>::const_iterator iter;
	TLengthData							length_data;
	TPointData							zero_step;
	std::vector<TVector2>				gradient_data;
	TVector2							gradient_point;
	bool								init;
	
	d_length_data.clear();
	d_error_gradient = 0.0;
	
	init = true;
	for(iter = d_point_data.begin();iter != d_point_data.end();++iter)
	{
		if(init)
		{
			zero_step = (*iter);
			init = false;
		}
		else
		{
			length_data.nominal = (*iter).nominal - zero_step.nominal;
			length_data.actual = (*iter).actual - zero_step.actual;
			
			d_length_data.push_back(length_data);
			
			gradient_point.x = length_data.actual;
			gradient_point.y = length_data.actual - length_data.nominal;
			
			gradient_data.push_back(gradient_point);
		}
	}
	
	Regression::Fit_Line(gradient_data,&d_error_gradient,&d_error_y_intercept);
}

double TMeasureStepgaugeSquare::Get_Nominal(
	const double						&position,
	const std::vector<TVector3>			&data,
	bool								* const valid) const
{
	std::vector<TVector3>::const_iterator iter;
	double								nominal(0.0);
	double								deviation;
	static const double					MAXIMUM_DEVIATION(2.0);	// nominal must be within +/- of this value.
	
	assert(valid);
	*valid = false;
	
	for(iter = data.begin();iter != data.end();++iter)
	{
		deviation = fabs((*iter).x - position);
		
		if(!(deviation > MAXIMUM_DEVIATION))
		{
			*valid = true;
			
			nominal = (*iter).x;
			break;
		}
	}
	
	return nominal;
}
