/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QStringList>
#include <QByteArray>
#include <QTextStream>

#include "../../../core/vector2.h"

#include "measuredata_temperature.h"

TMeasureTemperature::TMeasureTemperature(void)
{
	d_file_type = TMeasureData::TYPE_TEMPERATURE_LOG;
}

TMeasureTemperature::~TMeasureTemperature(void)
{
}

QStringList TMeasureTemperature::Text_Report(void) const
{
	std::set<int>::const_iterator		set_iter;
	std::vector<double>					temp_data;
	QStringList							list;
	double								min,max,mean,stdev;
	
	
	list.push_back(QStringLiteral("Temperature Log"));
	list.push_back(QStringLiteral("================================================================"));
	list.push_back(QString());
	
	for(set_iter = d_sensor_ids.begin();set_iter != d_sensor_ids.end();++set_iter)
	{
		list.push_back(QString("Sensor: %1").arg(*set_iter));
		list.push_back(QStringLiteral("--------------------------------------------"));

		temp_data = this->Get_Sensor_Temperatures(*set_iter);
		
		if(this->Get_Statistics(temp_data,&min,&max,&mean,&stdev))
		{
			list.push_back(QString("  Min:   %1 C").arg(min,6,'f',2));
			list.push_back(QString("  Max:   %1 C").arg(max,6,'f',2));
			list.push_back(QString("  Mean:  %1 C").arg(mean,6,'f',2));
			list.push_back(QString("  Stdev: %1 C").arg(stdev,6,'f',2));
		}
		else
		{
			list.push_back(QStringLiteral("<Unable to calculate statistics>"));
		}
		
		list.push_back(QString());
	}

	return list;
}

std::vector<std::vector<TVector2> > TMeasureTemperature::Graph_Data(void) const
{
	std::vector<std::vector<TVector2> > graph_data;
	std::vector<TVector2>				graph_set;
	TVector2							graph_pnt;
	std::set<int>::const_iterator		set_iter;
	std::vector<TMeasureTemperature::TSensorData> temp_data;
	std::vector<TMeasureTemperature::TSensorData>::const_iterator temp_iter;
	
	for(set_iter = d_sensor_ids.begin();set_iter != d_sensor_ids.end();++set_iter)
	{
		temp_data = this->Get_Sensor_Data(*set_iter);
		
		graph_set.clear();
		
		for(temp_iter = temp_data.begin();temp_iter != temp_data.end();++temp_iter)
		{
			graph_pnt.x = static_cast<double>((*temp_iter).hour);
			graph_pnt.x += (static_cast<double>((*temp_iter).minute) / 60.0);
			graph_pnt.x += (static_cast<double>((*temp_iter).second) / 3600.0);
			
			graph_pnt.y = (*temp_iter).temperature;
			
			graph_set.push_back(graph_pnt);
		}
		
		if(graph_set.size())
		{
			graph_data.push_back(graph_set);
		}
	}
	
	return graph_data;
}

bool TMeasureTemperature::Load_Data(
	const QByteArray					&file_data)
{
	QTextStream							stream(file_data,QIODevice::ReadOnly);
	QString								line;
	QString								text;
	QString								text_time;
	QStringList							list;
	bool								init(true);
	TMeasureTemperature::TSensorData	sensor_data;
	
	d_points.clear();
	
	d_sensor_data.clear();
	d_sensor_ids.clear();
	
	while(!stream.atEnd())
	{
		line = stream.readLine().trimmed();
		
		if(line.startsWith('#') ||
		   line.length() == 0)
		{
			continue;
		}
		
		if(init)
		{
			if(!line.startsWith(QStringLiteral("Temperature_Log:Version=1:Type=Machine_Sensors")))
			{
				d_last_error = QStringLiteral("File identification line not recognized");
				return false;
			}
			
			init = false;
		}
		else
		{
			
			list = line.split(',');
			
			if(list.size() == 3)
			{
				text_time = list[0];
				
				if(text_time.length() == 8)		// [hhmmss]
				{
					text = text_time.mid(1,2);
					sensor_data.hour = text.toInt();
					
					text = text_time.mid(3,2);
					sensor_data.minute = text.toInt();
					
					text = text_time.mid(5,2);
					sensor_data.second = text.toInt();
					
					sensor_data.sensor_id = list[1].toInt();
					sensor_data.temperature = list[2].toDouble();
					
					if(sensor_data.sensor_id != 0 && sensor_data.temperature > 0.0)
					{
						d_sensor_data.push_back(sensor_data);
						d_sensor_ids.insert(sensor_data.sensor_id);
					}
				}
			}
		}
	}
	
	if(d_sensor_data.size() == 0 || d_sensor_ids.size() == 0)
	{
		d_last_error = QStringLiteral("Data missing from measurement file.");
		return false;
	}
	
	return true;
}

std::vector<TMeasureTemperature::TSensorData> TMeasureTemperature::Get_Sensor_Data(
	const int							sensor_id) const
{
	std::vector<TMeasureTemperature::TSensorData> sensor_data;
	std::vector<TMeasureTemperature::TSensorData>::const_iterator iter;
	
	for(iter = d_sensor_data.begin();iter != d_sensor_data.end();++iter)
	{
		if((*iter).sensor_id == sensor_id)
		{
			sensor_data.push_back(*iter);
		}
	}

	return sensor_data;
}

std::vector<double> TMeasureTemperature::Get_Sensor_Temperatures(
	const int 							sensor_id) const
{
	std::vector<double> 				sensor_temperatures;
	std::vector<TMeasureTemperature::TSensorData>::const_iterator iter;
	
	for(iter = d_sensor_data.begin();iter != d_sensor_data.end();++iter)
	{
		if((*iter).sensor_id == sensor_id)
		{
			sensor_temperatures.push_back((*iter).temperature);
		}
	}

	return sensor_temperatures;
}

bool TMeasureTemperature::Get_Statistics(
	const std::vector<double>			&data,
	double 								* const min,
	double 								* const max,
	double 								* const mean,
	double 								* const stdev) const
{
	std::vector<double>::const_iterator	iter;
	double								sum;
	double								dval;
	bool								init(true);
	
	assert(min);
	assert(max);
	assert(mean);
	assert(stdev);
	
	*min = 0.0;
	*max = 0.0;
	*mean = 0.0;
	*stdev = 0.0;
		
	if(!data.size())
	{
		return false;
	}
	
	sum = 0.0;
		
	for(iter = data.begin();iter != data.end();++iter)
	{
		if(init)
		{
			*min = (*iter);
			*max = (*iter);
			
			init = false;
		}
		else
		{
			if((*iter) > (*max)) (*max) = (*iter);
			if((*iter) < (*min)) (*min) = (*iter);
		}
		
		sum += (*iter);
	}
	
	*mean = sum / static_cast<double>(data.size());
	sum = 0.0;
	
	for(iter = data.begin();iter != data.end();++iter)
	{
		dval = (*iter) - (*mean);
		
		dval = dval * dval;
		
		sum += dval;
	}
	
	if(data.size() > 1)
	{
		sum /= static_cast<double>(data.size() - 1);	// stdev of sample
		
		*stdev = sqrt(sum);
	}
	
	return true;
}
