/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QDockWidget>
#include <QMenuBar>
#include <QSize>
#include <QStatusBar>
#include <QToolBar>
#include <QWidget>
#include <QLabel>
#include <QAction>
#include <QCloseEvent>
#include <QScreen>
#include <QPlainTextEdit>
#include <QSplitter>
#include <QToolButton>
#include <QVBoxLayout>
#include <QGridLayout>
#include <QSettings>
#include <QFileSystemWatcher>
#include <QFile>
#include <QFileInfo>
#include <QByteArray>
#include <QFont>
#include <QDir>
#include <QRect>
#include <QSize>
#include <QPoint>
#include <QColor>
#include <assert.h>

#include "../../../core/vector2.h"
#include "../../../core/vector3.h"
#include "../../../core/graphwidget.h"
#include "../../../core/messagebox.h"

#include "measuredata.h"
#include "measuredata_pingauge.h"
#include "measuredata_ringgauge.h"
#include "measuredata_stepgauge10360.h"
#include "measuredata_stepgaugeb89.h"
#include "measuredata_stepgaugesquare.h"
#include "measuredata_tool.h"
#include "measuredata_pftu.h"
#include "measuredata_gaugeblock.h"
#include "measuredata_ballbar.h"
#include "measuredata_ballbar10360.h"
#include "measuredata_pointrepeat.h"
#include "measuredata_rpt.h"
#include "measuredata_rolloffsettool.h"
#include "measuredata_temperature.h"
#include "measuredata_machinelaser10360.h"
#include "measuredata_machinelaserpmove.h"
#include "measuredata_validationcircle.h"
#include "measuredata_validationsphere.h"
#include "measuredata_validationline.h"
#include "measuredata_validationplane.h"

#include "nominalwidget.h"
#include "measuredwidget.h"

#include "measureview.h"

TMeasureView::TMeasureView(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QMainWindow(const_cast<QWidget*>(parent),flags)
{
	QWidget								*central_widget;
	QWidget								*logo_widget;
	QLabel								*logo_label;
	QLabel								*copyright_label;
	QSplitter							*vertical_splitter;
	QVBoxLayout 						*central_widget_layout;
	QGridLayout							*logo_layout;
	QDockWidget							*measured_doc;
	QDockWidget							*nominal_dock;
	QMenuBar							*menu_bar;
	QMenu								*file_menu;
	QMenu								*view_menu;
	QStatusBar							*status_bar;
	QToolBar							*tool_bar;
	QDir								data_path;
	QRect								desktop_rect;
	QRect								window_rect;
	QPoint								position;
	QSize								size;
	bool								bval;
	QFont								text_font(QStringLiteral("courier new"));
	QFont								copyright_font;
	TMeasuredWidget::TInputSettings		input_settings;
	
	copyright_font.setFamily(QStringLiteral("Helvetica"));
	copyright_font.setPointSize(8);
	copyright_font.setItalic(true);

	text_font.setFixedPitch(true);

	d_application_path = qApp->applicationDirPath();
	
#ifdef Q_OS_MAC
	int index = d_application_path.lastIndexOf(QStringLiteral(".app/"));
	d_application_path.truncate(index);
	
	index = d_application_path.lastIndexOf('/');
	d_application_path.truncate(index);
#endif
	
#ifdef Q_OS_WIN
	d_data_path = d_application_path;
#else
	d_data_path = QDir::homePath();
#endif
	
	data_path.setPath(d_data_path);
	
	if(!data_path.exists(QStringLiteral(".measuredirect")))
	{
		data_path.mkdir(QStringLiteral(".measuredirect"));
	}
	
	data_path.cd(QStringLiteral(".measuredirect"));
	d_data_path = data_path.absolutePath();
	
	d_settings = new QSettings(d_data_path + QStringLiteral("/measureview_settings.ini"),QSettings::IniFormat,this);
	
	d_file_system_watcher = new QFileSystemWatcher(this);
	
	position = d_settings->value("Mainwindow_Position", QPoint(30,30)).toPoint();
	size = d_settings->value("Mainwindow_Size", QSize(801, 601)).toSize();
	
	desktop_rect = QGuiApplication::primaryScreen()->availableGeometry();
	
	window_rect.moveTopLeft(position);
	window_rect.setSize(size);
	
	window_rect = window_rect & desktop_rect;
	
	if(window_rect.isValid() && window_rect.width() > 100 && window_rect.height() > 100)
	{
		this->resize(window_rect.size());
		this->move(window_rect.topLeft());
	}
	else
	{
		this->resize(801,601);
		this->move(QPoint(30,30));
	}
	
	central_widget = new QWidget(this);
	this->setCentralWidget(central_widget);
	
	central_widget_layout = new QVBoxLayout(central_widget);

	vertical_splitter = new QSplitter(Qt::Vertical,central_widget);
	
	d_text_display = new QPlainTextEdit;
	d_text_display->setFont(text_font);
	d_text_display->setUndoRedoEnabled(false);
	d_text_display->setLineWrapMode(QPlainTextEdit::NoWrap);
	d_text_display->setReadOnly(true);
	
	vertical_splitter->addWidget(d_text_display);
	
	d_graph_display = new TGraphWidget(central_widget);
	d_graph_display->setMinimumHeight(150);
	vertical_splitter->addWidget(d_graph_display);
	
	central_widget_layout->addWidget(vertical_splitter);
	
	logo_widget = new QWidget(this);
	
	logo_layout = new QGridLayout(logo_widget);
	logo_layout->setContentsMargins(0,0,0,0);
	
	logo_label = new QLabel(logo_widget);
	logo_label->setMinimumSize(100,40);
	logo_label->setMaximumSize(100,40);
	logo_layout->addWidget(logo_label,0,0,1,1);
	
	copyright_label = new QLabel(logo_widget);
	copyright_label->setFont(copyright_font);
	copyright_label->setAlignment(Qt::AlignBottom|Qt::AlignRight|Qt::AlignTrailing);
	logo_layout->addWidget(copyright_label,0,1,1,1);
	
	logo_label->setPixmap(QPixmap(QStringLiteral(":/images/logo.png")));

	central_widget_layout->addWidget(logo_widget);
	
	menu_bar = new QMenuBar(this);
	this->setMenuBar(menu_bar);
	menu_bar->resize(800,22);

	tool_bar = new QToolBar(this);
	this->addToolBar(Qt::TopToolBarArea,tool_bar);
	tool_bar->setIconSize(QSize(32,32));
	
	d_edit_tolerance_button = new QToolButton;
	d_edit_tolerance_button->setFixedSize(32,32);
	d_edit_tolerance_button->setIconSize(QSize(32,32));
	d_edit_tolerance_button->setIcon(QIcon(QStringLiteral(":/icon32/tolerance_icon32.png")));
	d_edit_tolerance_button->setToolTip(QStringLiteral("Set Stepgauge Measurement Tolerance"));
	tool_bar->addWidget(d_edit_tolerance_button);
	
	d_point_graph_button = new QToolButton;
	d_point_graph_button->setFixedSize(32,32);
	d_point_graph_button->setIconSize(QSize(32,32));
	d_point_graph_button->setIcon(QIcon(QStringLiteral(":/icon32/pointgraph_icon32.png")));
	d_point_graph_button->setToolTip(QStringLiteral("Show Stepgauge Point Graph"));
	d_point_graph_button->setCheckable(true);
	tool_bar->addWidget(d_point_graph_button);
	
	d_line_graph_button = new QToolButton;
	d_line_graph_button->setFixedSize(32,32);
	d_line_graph_button->setIconSize(QSize(32,32));
	d_line_graph_button->setIcon(QIcon(QStringLiteral(":/icon32/linegraph_icon32.png")));
	d_line_graph_button->setToolTip(QStringLiteral("Show Stepgauge Line Graph"));
	d_line_graph_button->setCheckable(true);
	tool_bar->addWidget(d_line_graph_button);

	status_bar = new QStatusBar(this);
	this->setStatusBar(status_bar);
	
	d_action_quit = new QAction(this);
	d_action_quit->setShortcut(QKeySequence::Quit);
	
	file_menu = new QMenu();
	file_menu->addAction(d_action_quit);
	menu_bar->addMenu(file_menu);

	view_menu = new QMenu();
	menu_bar->addMenu(view_menu);
	
	menu_bar->addAction(file_menu->menuAction());
	menu_bar->addAction(view_menu->menuAction());

	nominal_dock = new QDockWidget(this);
	nominal_dock->setMinimumWidth(450);
	view_menu->addAction(nominal_dock->toggleViewAction());
	this->addDockWidget(Qt::LeftDockWidgetArea,nominal_dock);

	d_stepgauge_nominal_widget = new TNominalWidget();
	nominal_dock->setWidget(d_stepgauge_nominal_widget);
	nominal_dock->setAllowedAreas(Qt::LeftDockWidgetArea|Qt::RightDockWidgetArea);

	measured_doc = new QDockWidget(this);
	measured_doc->setMinimumWidth(450);
	view_menu->addAction(measured_doc->toggleViewAction());
	this->addDockWidget(Qt::LeftDockWidgetArea,measured_doc);

	d_measured_widget = new TMeasuredWidget();
	measured_doc->setWidget(d_measured_widget);
	measured_doc->setAllowedAreas(Qt::LeftDockWidgetArea|Qt::RightDockWidgetArea);
	
	d_tolerance_dialog = new TToleranceDialog(this);
	
	input_settings.input_relative_path = d_settings->value(QStringLiteral("Input_Relative_Path"),QStringLiteral("Measurement_Data")).toString();
	input_settings.input_absolute_path = d_settings->value(QStringLiteral("Input_Absolute_Path"),d_data_path + QStringLiteral("/Measurement_Data")).toString();
	
	// default to relative path option on windows
	bval = true;
	
#ifndef Q_OS_WIN
	bval = false;
#endif

	input_settings.use_relative_path = d_settings->value(QStringLiteral("Input_Use_Relative_Path"),bval).toBool();
	
	d_measured_widget->Set_Input_Settings(input_settings);
	
	d_watched_path = d_measured_widget->File_Path();
	d_file_system_watcher->addPath(d_watched_path);
	
	d_tolerance_dialog->Set_Tolerance_Type(static_cast<TToleranceDialog::TToleranceType>(d_settings->value(QStringLiteral("Tolerance_Type"),0).toInt()));
	d_tolerance_dialog->Set_Tolerance_A(d_settings->value(QStringLiteral("Tolerance_A"),0.012).toDouble());
	d_tolerance_dialog->Set_Tolerance_B(d_settings->value(QStringLiteral("Tolerance_B"),0.006).toDouble());
	d_tolerance_dialog->Set_Tolerance_C(d_settings->value(QStringLiteral("Tolerance_C"),1000.0).toDouble());
	d_tolerance_dialog->Set_Tolerance_D(d_settings->value(QStringLiteral("Tolerance_D"),0.015).toDouble());
	d_tolerance_dialog->Set_Upper_Temperature_Limit(d_settings->value(QStringLiteral("Tolerance_Upper_Temperature_Limit"),22).toDouble());
	d_tolerance_dialog->Set_Lower_Temperature_Limit(d_settings->value(QStringLiteral("Tolerance_Lower_Temperature_Limit"),18).toDouble());
	
	d_tolerance_dialog->Update();
	
	d_message_box = new TMessageBox(this);
	
	// defaults
	d_active_data = 0;
	d_line_graph_button->setChecked(false);
	d_point_graph_button->setChecked(true);
	
	d_point_graph_button->setEnabled(false);
	d_line_graph_button->setEnabled(false);

	d_stepgauge_nominal_widget->Initialize(d_data_path);

	this->setWindowTitle(QStringLiteral("MeasureView - Version 9.0")); 	// version in measuredirect should match
	
	tool_bar->setWindowTitle(QStringLiteral("toolBar"));
	d_edit_tolerance_button->setText(QStringLiteral("test"));
	nominal_dock->setWindowTitle(QStringLiteral("Stepgauge Nominal"));
	measured_doc->setWindowTitle(QStringLiteral("Measurement Files"));
	d_action_quit->setText(QStringLiteral("Quit"));
	file_menu->setTitle(QStringLiteral("File"));
	view_menu->setTitle(QStringLiteral("View"));
	copyright_label->setText(QStringLiteral("Copyright (C) 2025 Select Calibration Incorporated.  www.selectcalibration.ca"));

	d_graph_display->Title(QString());
	d_graph_display->VerticalLegend(QStringLiteral("Deviation"));
	d_graph_display->HorizontalLegend(QStringLiteral("Length"));

	connect(d_action_quit,&QAction::triggered, this, &TMeasureView::close);
	connect(d_measured_widget,&TMeasuredWidget::Load_Data,this,&TMeasureView::Load_Data);
	connect(d_measured_widget,&TMeasuredWidget::Remove_Data,this,&TMeasureView::Remove_Data);
	connect(d_edit_tolerance_button,&QToolButton::clicked,this,&TMeasureView::Edit_Tolerance);
	connect(d_point_graph_button,&QToolButton::toggled,this,&TMeasureView::Toggle_Point_Graph_Type);
	connect(d_line_graph_button,&QToolButton::toggled,this,&TMeasureView::Toggle_Line_Graph_Type);
	
	connect(d_file_system_watcher,&QFileSystemWatcher::directoryChanged,this,&TMeasureView::Update_File_List);
	connect(d_measured_widget,&TMeasuredWidget::File_Path_Changed,this,&TMeasureView::Measure_File_Path_Changed);
}

TMeasureView::~TMeasureView(void)
{
}

void TMeasureView::closeEvent(
	QCloseEvent							*event)
{
	TMeasuredWidget::TInputSettings		input_settings;
	
	if(d_active_data)
	{
		delete d_active_data;
		d_active_data = 0;
	}
	
	d_file_system_watcher->removePath(d_watched_path);

	d_settings->setValue(QStringLiteral("Mainwindow_Position"), this->pos());
	d_settings->setValue(QStringLiteral("Mainwindow_Size"), this->size());
	
	input_settings = d_measured_widget->Input_Settings();
	
	d_settings->setValue(QStringLiteral("Input_Relative_Path"), input_settings.input_relative_path);
	d_settings->setValue(QStringLiteral("Input_Absolute_Path"), input_settings.input_absolute_path);
	d_settings->setValue(QStringLiteral("Input_Use_Relative_Path"), input_settings.use_relative_path);

	d_settings->setValue(QStringLiteral("Tolerance_Type"), d_tolerance_dialog->Tolerance_Type());
	d_settings->setValue(QStringLiteral("Tolerance_A"), d_tolerance_dialog->Tolerance_A());
	d_settings->setValue(QStringLiteral("Tolerance_B"), d_tolerance_dialog->Tolerance_B());
	d_settings->setValue(QStringLiteral("Tolerance_C"), d_tolerance_dialog->Tolerance_C());
	d_settings->setValue(QStringLiteral("Tolerance_D"), d_tolerance_dialog->Tolerance_D());
	d_settings->setValue(QStringLiteral("Tolerance_Upper_Temperature_Limit"), d_tolerance_dialog->Upper_Temperature_Limit());
	d_settings->setValue(QStringLiteral("Tolerance_Lower_Temperature_Limit"), d_tolerance_dialog->Lower_Temperature_Limit());
	
	event->accept();
}

void TMeasureView::Load_Data(
	const QString						&file_name)
{
	QFile								file;
	QFileInfo							file_info;
	QByteArray							file_data;
	TMeasureData::TFileType				file_type;
	QStringList							text_data;
	QStringList::const_iterator			iter;
	std::vector<std::vector<TVector2> >	graph_data;
	std::vector<std::vector<TVector2> >::iterator	graph_iter;
	std::vector<TVector2>::const_iterator pnt_iter;
	std::vector<TVector3>				nominal_data;
	TVector2							pnt;
	QString								text;
	int									set_id;
	int									cntr;
	int									ival;
	int									red,green,blue;
	bool								valid;
	double								tol_a;
	double								tol_b;
	double								tol_c;
	double								tol_d;
	double								min_position(0.0);
	double								max_position(0.0);
	double								limit_position;
	double								tolerance;
	double								dval;
	double								length;
	double								range;
	double								window;
	bool								init;
	static const int					NUMBER_FREQUENCY_ITEMS(11);
	int									frequency_array[NUMBER_FREQUENCY_ITEMS];
	static const qint64					MAX_FILE_SIZE(0x40000);	// 262 kb
	
	this->Clear_Log();
	d_graph_display->DeleteDataSets();
	
	memset(frequency_array,0,sizeof(int) * NUMBER_FREQUENCY_ITEMS);
	
	if(d_active_data)
	{
		delete d_active_data;
		d_active_data = 0;
	}
	
	if(!file_name.length())
	{
		d_graph_display->hide();
		return;
	}
	
	file_info.setFile(file_name);
	
	if(file_info.size() > MAX_FILE_SIZE)
	{
		this->Add_Log_Text(QStringLiteral("ERR  Input file too large to be a measurement data file."));
		d_graph_display->hide();
		return;
	}
	else if(file_info.size() < 64)
	{
		this->Add_Log_Text(QStringLiteral("ERR  Input file too small to be a measurement data file."));
		d_graph_display->hide();
		return;
	}
	
	file.setFileName(file_name);
	if(!file.open(QIODevice::ReadOnly))
	{
		this->Add_Log_Text(QStringLiteral("ERR  Cannot open measurement data file."));
		d_graph_display->hide();
		return;
	}
	
	file_data = file.readAll();
	
	file.close();
	
	file_type = TMeasureData::Determine_File_Contents(file_data);
	
	d_point_graph_button->setEnabled(false);
	d_line_graph_button->setEnabled(false);

	switch(file_type)
	{
		case TMeasureData::TYPE_UNKNOWN:
			this->Add_Log_Text(QStringLiteral("ERR  File contents not recognized."));
			d_graph_display->hide();

			return;
			
		case TMeasureData::TYPE_STEPGAUGE_10360:
			d_active_data = new TMeasureStepgauge10360;
			
			d_graph_display->show();
			d_graph_display->Title(QStringLiteral("Step Gauge 10360"));
			d_graph_display->VerticalLegend(QStringLiteral("Deviation"));
			d_graph_display->HorizontalLegend(QStringLiteral("Length"));
			d_graph_display->LockAspectRatio(false);
			d_graph_display->SetVLegendMode(TGraphWidget::DEFAULT_LEGEND_PRECISION);
			d_graph_display->EnableYIncludesZero(true);

			d_point_graph_button->setEnabled(true);
			d_line_graph_button->setEnabled(true);
			
			break;
			
		case TMeasureData::TYPE_STEPGAUGE_B89:
			d_active_data = new TMeasureStepgaugeB89;
			
			d_graph_display->show();
			d_graph_display->Title(QStringLiteral("Step Gauge B89.4.1"));
			d_graph_display->VerticalLegend(QStringLiteral("Deviation"));
			d_graph_display->HorizontalLegend(QStringLiteral("Length"));
			d_graph_display->LockAspectRatio(false);
			d_graph_display->SetVLegendMode(TGraphWidget::DEFAULT_LEGEND_PRECISION);
			d_graph_display->EnableYIncludesZero(true);
			
			d_point_graph_button->setEnabled(true);
			d_line_graph_button->setEnabled(true);
			
			break;
			
		case TMeasureData::TYPE_STEPGAUGE_SQUARE:
			d_active_data = new TMeasureStepgaugeSquare;
			
			d_graph_display->show();
			d_graph_display->Title(QStringLiteral("Step Gauge Square"));
			d_graph_display->VerticalLegend(QStringLiteral("Deviation"));
			d_graph_display->HorizontalLegend(QStringLiteral("Length"));
			d_graph_display->LockAspectRatio(false);
			d_graph_display->SetVLegendMode(TGraphWidget::DEFAULT_LEGEND_PRECISION);
			d_graph_display->EnableYIncludesZero(true);

			d_point_graph_button->setEnabled(true);
			d_line_graph_button->setEnabled(true);

			break;
			
		case TMeasureData::TYPE_PROBE_CALIBRATION:
			d_active_data = new TMeasureTool;
			
			d_graph_display->hide();
			
			break;
			
		case TMeasureData::TYPE_POINT_REPEAT:
			d_active_data = new TMeasurePointRepeat;
			
			d_graph_display->hide();
			
			break;
			
		case TMeasureData::TYPE_SPHERE_RPT:
			d_active_data = new TMeasureRPT;
			
			d_graph_display->hide();
			
			break;
			
		case TMeasureData::TYPE_SPHERE_PFTU:
			d_active_data = new TMeasurePFTU;
			
			d_graph_display->hide();
			
			break;

		case TMeasureData::TYPE_GAUGEBLOCK:
			d_active_data = new TMeasureGaugeblock;
			
			d_graph_display->hide();
			
			break;

		case TMeasureData::TYPE_BALLBAR:
			d_active_data = new TMeasureBallbar;
			
			d_graph_display->show();
			d_graph_display->Title(QStringLiteral("Ballbar"));
			d_graph_display->VerticalLegend(QStringLiteral("Measurement Count"));
			d_graph_display->HorizontalLegend(QStringLiteral("Length"));
			d_graph_display->LockAspectRatio(false);
			d_graph_display->SetVLegendMode(TGraphWidget::INTEGER_LEGEND_PRECISION);
			d_graph_display->EnableYIncludesZero(true);

			break;

		case TMeasureData::TYPE_BALLBAR_10360:
			d_active_data = new TMeasureBallbar10360;
			
			d_graph_display->show();
			d_graph_display->Title(QStringLiteral("Ballbar 10360"));
			d_graph_display->VerticalLegend(QStringLiteral("Deviation"));
			d_graph_display->HorizontalLegend(QStringLiteral("Length"));
			d_graph_display->LockAspectRatio(false);
			d_graph_display->SetVLegendMode(TGraphWidget::DEFAULT_LEGEND_PRECISION);
			d_graph_display->EnableYIncludesZero(true);

			break;

		case TMeasureData::TYPE_PINGAUGE:
			d_active_data = new TMeasurePin;
			
			d_graph_display->show();
			d_graph_display->Title(QStringLiteral("Pin Gauge"));
			d_graph_display->VerticalLegend(QStringLiteral("Y Axis"));
			d_graph_display->HorizontalLegend(QStringLiteral("X Axis"));
			d_graph_display->LockAspectRatio(true);
			d_graph_display->SetVLegendMode(TGraphWidget::DEFAULT_LEGEND_PRECISION);
			d_graph_display->EnableYIncludesZero(true);

			break;
			
		case TMeasureData::TYPE_RINGGAUGE:
			d_active_data = new TMeasureRing;
			
			d_graph_display->show();
			d_graph_display->Title(QStringLiteral("Ring Gauge"));
			d_graph_display->VerticalLegend(QStringLiteral("Y Axis"));
			d_graph_display->HorizontalLegend(QStringLiteral("X Axis"));
			d_graph_display->LockAspectRatio(true);
			d_graph_display->SetVLegendMode(TGraphWidget::DEFAULT_LEGEND_PRECISION);
			d_graph_display->EnableYIncludesZero(true);

			break;

		case TMeasureData::TYPE_ROLL_OFFSET_TOOL:
			d_active_data = new TMeasureRollOffsetTool;
			
			d_graph_display->hide();
			
			break;
			
		case TMeasureData::TYPE_TEMPERATURE_LOG:
			d_active_data = new TMeasureTemperature;
			
			d_graph_display->show();
			d_graph_display->Title(QStringLiteral("Temperature Log"));
			d_graph_display->VerticalLegend(QStringLiteral("Celsius"));
			d_graph_display->HorizontalLegend(QStringLiteral("Hour"));
			d_graph_display->LockAspectRatio(false);
			d_graph_display->SetVLegendMode(TGraphWidget::DEFAULT_LEGEND_PRECISION);
			d_graph_display->EnableYIncludesZero(false);

			break;
			
		case TMeasureData::TYPE_MACHINE_LASER_10360:
			d_active_data = new TMeasureMachineLaser10360;
			
			d_graph_display->show();
			d_graph_display->Title(QStringLiteral("Laser 10360"));
			d_graph_display->VerticalLegend(QStringLiteral("Deviation"));
			d_graph_display->HorizontalLegend(QStringLiteral("Length"));
			d_graph_display->LockAspectRatio(false);
			d_graph_display->SetVLegendMode(TGraphWidget::DEFAULT_LEGEND_PRECISION);
			d_graph_display->EnableYIncludesZero(false);
			
			d_point_graph_button->setEnabled(true);
			d_line_graph_button->setEnabled(true);

			break;
			
		case TMeasureData::TYPE_MACHINE_LASER_PMOVE:
			d_active_data = new TMeasureMachineLaserPMove;
			
			d_graph_display->show();
			d_graph_display->Title(QStringLiteral("Laser Data Collection"));
			d_graph_display->VerticalLegend(QStringLiteral("Deviation"));
			d_graph_display->HorizontalLegend(QStringLiteral("Length"));
			d_graph_display->LockAspectRatio(false);
			d_graph_display->SetVLegendMode(TGraphWidget::DEFAULT_LEGEND_PRECISION);
			d_graph_display->EnableYIncludesZero(false);
			
			d_point_graph_button->setEnabled(true);
			d_line_graph_button->setEnabled(true);

			break;
			
		case TMeasureData::TYPE_VALIDATION_PLANE:
			d_active_data = new TValidationPlane;
			
			d_graph_display->hide();
			
			break;
			
		case TMeasureData::TYPE_VALIDATION_LINE:
			d_active_data = new TValidationLine;
			
			d_graph_display->hide();
			
			break;
			
		case TMeasureData::TYPE_VALIDATION_SPHERE:
			d_active_data = new TValidationSphere;
			
			d_graph_display->hide();
			
			break;
			
		case TMeasureData::TYPE_VALIDATION_CIRCLE:
			d_active_data = new TValidationCircle;
			
			d_graph_display->hide();
			
			break;

	}
	
	if(d_active_data)
	{
		if(!d_active_data->Load_Data(file_data))
		{
			this->Add_Log_Text(QString("ERR  %1").arg(d_active_data->Get_Last_Error()));
			d_graph_display->hide();
			return;
		}
		
		d_active_file_name = file_name;
		
		tol_a = d_tolerance_dialog->Tolerance_A();
		tol_b = d_tolerance_dialog->Tolerance_B();
		tol_c = d_tolerance_dialog->Tolerance_C();
		tol_d = d_tolerance_dialog->Tolerance_D();
		
		if(d_active_data->File_Type() == TMeasureData::TYPE_STEPGAUGE_10360 ||
		   d_active_data->File_Type() == TMeasureData::TYPE_STEPGAUGE_SQUARE ||
		   d_active_data->File_Type() == TMeasureData::TYPE_STEPGAUGE_B89)
		{
			text = d_active_data->Equipment_Identification();
						
			nominal_data = d_stepgauge_nominal_widget->Nominal_Data(text,&valid);
			
			if(!valid)
			{
				this->Add_Log_Text(QString("ERR  Stepgauge nominal data for gauge '%1' not found.").arg(text));
				d_graph_display->hide();
				return;
			}
			
			if(!d_active_data->Set_Nominal_Data(nominal_data))
			{
				this->Add_Log_Text(QString("ERR  %1").arg(d_active_data->Get_Last_Error()));
				d_graph_display->hide();
				return;
			}

		}
		
		text_data = d_active_data->Text_Report();
		
		for(iter = text_data.begin();iter != text_data.end();++iter)
		{
			this->Add_Log_Text(*iter);
		}
		
		graph_data = d_active_data->Graph_Data();
				
		if(d_active_data->File_Type() == TMeasureData::TYPE_STEPGAUGE_10360 ||
		   d_active_data->File_Type() == TMeasureData::TYPE_STEPGAUGE_SQUARE ||
		   d_active_data->File_Type() == TMeasureData::TYPE_STEPGAUGE_B89 ||
		   d_active_data->File_Type() == TMeasureData::TYPE_MACHINE_LASER_10360 ||
		   d_active_data->File_Type() == TMeasureData::TYPE_MACHINE_LASER_PMOVE)
		{
			init = true;
			
			for(graph_iter = graph_data.begin(),cntr = 0;graph_iter != graph_data.end();++graph_iter,++cntr)
			{
				if(d_point_graph_button->isChecked())
				{
					switch(cntr)
					{
						case 0:
							set_id = d_graph_display->CreateDataSet(QColor(64,0,0),TGraphWidget::POINT_GRAPH);
							break;
							
						case 1:
							set_id = d_graph_display->CreateDataSet(QColor(0,64,0),TGraphWidget::POINT_GRAPH);
							break;
							
						case 2:
							set_id = d_graph_display->CreateDataSet(QColor(0,0,64),TGraphWidget::POINT_GRAPH);
							break;
							
						default:
							set_id = d_graph_display->CreateDataSet(QColor(32,32,32),TGraphWidget::POINT_GRAPH);
							break;
					}
				}
				else
				{
					switch(cntr)
					{
						case 0:
							set_id = d_graph_display->CreateDataSet(QColor(128,0,0),TGraphWidget::LINE_GRAPH);
							break;
							
						case 1:
							set_id = d_graph_display->CreateDataSet(QColor(0,128,0),TGraphWidget::LINE_GRAPH);
							break;
							
						case 2:
							set_id = d_graph_display->CreateDataSet(QColor(0,0,128),TGraphWidget::LINE_GRAPH);
							break;
							
						default:
							set_id = d_graph_display->CreateDataSet(QColor(64,64,64),TGraphWidget::LINE_GRAPH);
							break;
					}
				}
				
				d_graph_display->AddPoints(set_id,(*graph_iter));
				
				for(pnt_iter = (*graph_iter).begin();pnt_iter != (*graph_iter).end();++pnt_iter)
				{
					if(init)
					{
						min_position = max_position = (*pnt_iter).x;
						init = false;
					}
					else
					{
						if((*pnt_iter).x < min_position) min_position = (*pnt_iter).x;
						else if((*pnt_iter).x > max_position) max_position = (*pnt_iter).x;
					}
				}
			}
			
			switch(d_tolerance_dialog->Tolerance_Type())
			{
				case TToleranceDialog::TOLERANCE_BAND:
					set_id = d_graph_display->CreateDataSet(QColor(164,164,255));
					
					pnt.x = min_position;
					pnt.y = tol_a;
					d_graph_display->AddPoint(set_id,pnt);
					
					pnt.x = max_position;
					d_graph_display->AddPoint(set_id,pnt);
					
					set_id = d_graph_display->CreateDataSet(QColor(164,164,255));
					
					pnt.x = min_position;
					pnt.y = -tol_a;
					d_graph_display->AddPoint(set_id,pnt);
					
					pnt.x = max_position;
					d_graph_display->AddPoint(set_id,pnt);
					
					break;
					
				case TToleranceDialog::TOLERANCE_FORMULA:
					set_id = d_graph_display->CreateDataSet(QColor(164,164,255));
					
					pnt.x = min_position;
					pnt.y = tol_a + tol_b * min_position / tol_c;
					
					d_graph_display->AddPoint(set_id,pnt);
					
					pnt.x = max_position;
					pnt.y = tol_a + tol_b * max_position / tol_c;
					
					d_graph_display->AddPoint(set_id,pnt);
					
					set_id = d_graph_display->CreateDataSet(QColor(164,164,255));
					
					pnt.x = min_position;
					pnt.y = -tol_a - tol_b * min_position / tol_c;
					
					d_graph_display->AddPoint(set_id,pnt);
					
					pnt.x = max_position;
					pnt.y = -tol_a - tol_b * max_position / tol_c;
					
					d_graph_display->AddPoint(set_id,pnt);
					break;
					
				case TToleranceDialog::TOLERANCE_FORMULA_WITH_LIMIT:
					limit_position = d_tolerance_dialog->Limit_Position();
					
					set_id = d_graph_display->CreateDataSet(QColor(164,164,255));
					
					pnt.x = min_position;
					pnt.y = tol_a + tol_b * min_position / tol_c;
					
					d_graph_display->AddPoint(set_id,pnt);
					
					if(max_position < limit_position)
					{
						pnt.x = max_position;
						pnt.y = tol_a + tol_b * max_position / tol_c;
						
						d_graph_display->AddPoint(set_id,pnt);
					}
					else
					{
						pnt.x = limit_position;
						pnt.y = tol_d;
						
						d_graph_display->AddPoint(set_id,pnt);
						
						pnt.x = max_position;
						d_graph_display->AddPoint(set_id,pnt);
					}
					
					set_id = d_graph_display->CreateDataSet(QColor(164,164,255));
					
					pnt.x = min_position;
					pnt.y = -tol_a - tol_b * min_position / tol_c;
					
					d_graph_display->AddPoint(set_id,pnt);
					
					if(max_position < limit_position)
					{
						pnt.x = max_position;
						pnt.y = -tol_a - tol_b * max_position / tol_c;
						
						d_graph_display->AddPoint(set_id,pnt);
					}
					else
					{
						pnt.x = limit_position;
						pnt.y = -tol_d;
						
						d_graph_display->AddPoint(set_id,pnt);
						
						pnt.x = max_position;
						d_graph_display->AddPoint(set_id,pnt);
					}
					
					break;
			}
		}
 		else if(d_active_data->File_Type() == TMeasureData::TYPE_PINGAUGE ||
				d_active_data->File_Type() == TMeasureData::TYPE_RINGGAUGE)
		{
			for(graph_iter = graph_data.begin(),cntr = 0;graph_iter != graph_data.end();++graph_iter,++cntr)
			{
				switch(cntr)
				{
					case 0:
						set_id = d_graph_display->CreateDataSet(QColor(0,0,0));
						break;
						
					case 1:
						set_id = d_graph_display->CreateDataSet(QColor(160,0,0),TGraphWidget::DOT_GRAPH);
						break;
						
					case 2:
					default:
						set_id = d_graph_display->CreateDataSet(QColor(192,192,192));
						break;
				}
				
				d_graph_display->AddPoints(set_id,(*graph_iter));
			}
		}
		else if(d_active_data->File_Type() == TMeasureData::TYPE_BALLBAR)
		{
			init = true;
			for(graph_iter = graph_data.begin(),cntr = 0;graph_iter != graph_data.end();++graph_iter,++cntr)
			{
				for(pnt_iter = (*graph_iter).begin();pnt_iter != (*graph_iter).end();++pnt_iter)
				{
					if(init)
					{
						min_position = (*pnt_iter).x;
						max_position = min_position;
						
						init = false;
					}
					else
					{
						if((*pnt_iter).x < min_position) min_position = (*pnt_iter).x;
						if((*pnt_iter).x > max_position) max_position = (*pnt_iter).x;
					}
				}
			}
			
			if(!init)
			{
				length = (max_position + min_position)/2.0;
				range = max_position - min_position;

				tolerance = d_tolerance_dialog->Tolerance_Value(length);
				
				this->Add_Log_Text(QString());
				this->Add_Log_Text(QString("  BallBar Tolerance: %1").arg(tolerance,0,'f',4));

				
				if(tolerance > range)
				{
					max_position = length + tolerance/2.0;
					min_position = length - tolerance/2.0;
					
					range = tolerance;
				}
				
				// increase range slightly for better appearance
				range = range * 1.2;
				
				max_position = length + range / 2.0;
				min_position = length - range / 2.0;

				window = (max_position - min_position) / static_cast<double>(NUMBER_FREQUENCY_ITEMS - 1);
				
				for(graph_iter = graph_data.begin(),cntr = 0;graph_iter != graph_data.end();++graph_iter,++cntr)
				{
					for(pnt_iter = (*graph_iter).begin();pnt_iter != (*graph_iter).end();++pnt_iter)
					{
						dval = min_position;
						
						for(cntr = 0;cntr < NUMBER_FREQUENCY_ITEMS;++cntr)
						{
							if((*pnt_iter).x > (dval - (window/2.0)) && (*pnt_iter).x < (dval + (window/2.0)))
							{
								++frequency_array[cntr];
								break;
							}
							
							dval += window;
						}
					}
				}
				
				if(range > 0.000001)
				{

					set_id = d_graph_display->CreateDataSet(QColor(0,0,255),TGraphWidget::BAR_GRAPH);
					
					// Force one value to zero
					pnt.x = length;
					pnt.y = 0;
					
					d_graph_display->AddPoint(set_id,pnt);
					
					ival = 0;
					for(cntr=0;cntr < NUMBER_FREQUENCY_ITEMS;++cntr)
					{
						pnt.x = min_position + static_cast<double>(cntr) * window;
						pnt.y = static_cast<double>(frequency_array[cntr]);
						
						d_graph_display->AddPoint(set_id,pnt);
						
						if(frequency_array[cntr] > ival)
						{
							ival = frequency_array[cntr];
						}
					}
					
					// draw tolerance lines
					
					set_id = d_graph_display->CreateDataSet(QColor(255,0,0),TGraphWidget::LINE_GRAPH);

					pnt.x = length - tolerance / 2.0;
					pnt.y = 0;
					d_graph_display->AddPoint(set_id,pnt);
					
					pnt.y = static_cast<double>(ival + 1);
					d_graph_display->AddPoint(set_id,pnt);
					
					pnt.x = length + tolerance / 2.0;
					pnt.y = static_cast<double>(ival + 1);
					d_graph_display->AddPoint(set_id,pnt);

					pnt.y = 0;
					d_graph_display->AddPoint(set_id,pnt);

				}
			}
		}
		else if(d_active_data->File_Type() == TMeasureData::TYPE_TEMPERATURE_LOG)
		{
			for(graph_iter = graph_data.begin(),cntr = 0;graph_iter != graph_data.end();++graph_iter,++cntr)
			{
				d_graph_display->GenerateRGBColor(cntr,&red,&green,&blue);
				
				set_id = d_graph_display->CreateDataSet(QColor(red,green,blue));
				
				d_graph_display->AddPoints(set_id,(*graph_iter));
			}
		}
	}
	
	d_graph_display->UpdateGraph();
	
	this->Scroll_Log_Top();
}

void TMeasureView::Update_File_List(void)
{
	d_measured_widget->Refresh_File_List();
}

void TMeasureView::Measure_File_Path_Changed(void)
{
	d_file_system_watcher->removePath(d_watched_path);

	d_watched_path = d_measured_widget->File_Path();

	d_file_system_watcher->addPath(d_watched_path);
}

void TMeasureView::Remove_Data(
	const QString						&data_path,
	const QStringList					&filenames)
{
	QStringList::const_iterator			iter;
	QDir								dir;
	QFile								file;
	QString								old_file_name;
	QString								new_file_name;
	QString								temp_file_name;
	int									index_extension;
	int									cntr;
	
	d_message_box->setText("Remove Measured Files");
	d_message_box->setInformativeText(QStringLiteral("Do you want to remove selected measurement files ?"));
	d_message_box->setDetailedText(QStringLiteral("Measurement files are not deleted but moved to the 'old' folder"));
	d_message_box->setStandardButtons(QMessageBox::Yes | QMessageBox::No);
	d_message_box->setDefaultButton(QMessageBox::Yes);
	d_message_box->setIcon(QMessageBox::Warning);
	
	if(QMessageBox::Yes != d_message_box->exec())
	{
		return;
	}
	
	dir.setPath(data_path);
	
	if(!dir.exists(QStringLiteral("old")))
	{
		dir.mkdir(QStringLiteral("old"));
	}

	for(iter = filenames.begin();iter != filenames.end();++iter)
	{
		old_file_name = data_path;
		old_file_name.append('/');
		old_file_name.append((*iter));
		
		file.setFileName(old_file_name);
		
		if(file.exists())
		{
			new_file_name = data_path;
			new_file_name.append("/old/");
			new_file_name.append((*iter));
			
			temp_file_name = new_file_name;
			index_extension = temp_file_name.lastIndexOf('.');

			cntr = 0;
			while(QFile::exists(new_file_name))
			{
				++cntr;
				
				if(!(index_extension < 0))
				{
					new_file_name = temp_file_name;
					new_file_name.insert(index_extension,QString(" - %1").arg(cntr));
				}
				else
				{
					new_file_name = temp_file_name + QString(" - %1").arg(cntr);
				}
			};

			if(!file.rename(new_file_name))
			{
				d_message_box->setText(QStringLiteral("Remove Error"));
				d_message_box->setInformativeText(QStringLiteral("Unable to remove the selected file."));
				d_message_box->setDetailedText(old_file_name);
				d_message_box->setStandardButtons(QMessageBox::Ok);
				d_message_box->setDefaultButton(QMessageBox::Ok);
				d_message_box->setIcon(QMessageBox::Warning);

				d_message_box->exec();
			}
		}
	}
}

void TMeasureView::Edit_Tolerance(void)
{
	if(QDialog::Accepted == d_tolerance_dialog->exec())
	{
		this->Load_Data(d_active_file_name);
	}
}

void TMeasureView::Toggle_Line_Graph_Type(
	bool								state)
{
	bool								prev_state;
	
	prev_state = d_point_graph_button->blockSignals(true);
	d_point_graph_button->setChecked(!state);
	d_point_graph_button->blockSignals(prev_state);
	
	this->Load_Data(d_active_file_name);
}

void TMeasureView::Toggle_Point_Graph_Type(
	bool								state)
{
	bool								prev_state;
	
	prev_state = d_line_graph_button->blockSignals(true);
	d_line_graph_button->setChecked(!state);
	d_line_graph_button->blockSignals(prev_state);
	
	this->Load_Data(d_active_file_name);
}

void TMeasureView::Clear_Log(void)
{
	d_text_display->clear();
}

void TMeasureView::Add_Log_Text(
	const QString						&text)
{
	d_text_display->moveCursor(QTextCursor::End);
	d_text_display->appendPlainText(text);
}

void TMeasureView::Scroll_Log_Top(void)
{
	d_text_display->moveCursor(QTextCursor::Start);
}



