/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QFrame>
#include <QGridLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QLineEdit>
#include <QPushButton>
#include <QRadioButton>
#include <QComboBox>
#include <QSpacerItem>
#include <assert.h>

#include "../../core/tableeditor.h"

#include "nominaleditordialog.h"

TNominalEditorDialog::TNominalEditorDialog(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QDialog(const_cast<QWidget*>(parent),flags)
{
	QFrame								*control_vline_separator;
	QFrame								*lower_hline;
	QFrame								*upper_hline;
	QGridLayout							*dialog_layout;
	QHBoxLayout							*button_hlayout;
	QLabel								*data_type_label;
	QLabel								*edit_mode_label;
	QLabel								*end_label;
	QLabel								*end_units_label;
	QLabel								*increment_label;
	QLabel								*increment_units_label;
	QLabel								*serial_label;
	QLabel								*start_label;
	QLabel								*start_units_label;
	QSpacerItem							*button_hspacer;

	this->resize(500,580);

	dialog_layout = new QGridLayout(this);

	serial_label = new QLabel(this);
	dialog_layout->addWidget(serial_label,0,0,1,1);

	d_serial_edit = new QLineEdit(this);
	d_serial_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_serial_edit,0,1,1,1);

	start_label = new QLabel(this);
	start_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	dialog_layout->addWidget(start_label,1,0,1,1);

	d_start_edit = new QLineEdit(this);
	d_start_edit->setAlignment(Qt::AlignCenter);
	d_start_edit->setReadOnly(true);
	dialog_layout->addWidget(d_start_edit,1,1,1,1);

	start_units_label = new QLabel(this);
	dialog_layout->addWidget(start_units_label,1,2,1,1);

	increment_label = new QLabel(this);
	increment_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	dialog_layout->addWidget(increment_label,2,0,1,1);

	d_increment_edit = new QLineEdit(this);
	d_increment_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_increment_edit,2,1,1,1);

	increment_units_label = new QLabel(this);
	dialog_layout->addWidget(increment_units_label,2,2,1,1);

	end_label = new QLabel(this);
	end_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	dialog_layout->addWidget(end_label,3,0,1,1);

	d_end_edit = new QLineEdit(this);
	d_end_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_end_edit,3,1,1,1);

	end_units_label = new QLabel(this);
	dialog_layout->addWidget(end_units_label,3,2,1,1);

	control_vline_separator = new QFrame(this);
	control_vline_separator->setFrameShape(QFrame::VLine);
	control_vline_separator->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(control_vline_separator,0,3,4,2);

	data_type_label = new QLabel(this);
	dialog_layout->addWidget(data_type_label,0,4,1,1);
	
	d_data_type_combo = new QComboBox(this);
	dialog_layout->addWidget(d_data_type_combo,0,5,1,1);

	edit_mode_label = new QLabel(this);
	dialog_layout->addWidget(edit_mode_label,2,4,1,1);

	d_mode_actual_radio = new QRadioButton(this);
	dialog_layout->addWidget(d_mode_actual_radio,2,5,1,1);

	d_mode_deviation_radio = new QRadioButton(this);
	d_mode_deviation_radio->setChecked(true);
	dialog_layout->addWidget(d_mode_deviation_radio,3,5,1,1);
	
	upper_hline = new QFrame(this);
	upper_hline->setFrameShape(QFrame::HLine);
	upper_hline->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(upper_hline,4,0,1,6);

	d_edit_table = new TEditTable(this);
	dialog_layout->addWidget(d_edit_table,5,0,1,6);

	lower_hline = new QFrame(this);
	lower_hline->setFrameShape(QFrame::HLine);
	lower_hline->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(lower_hline,6,0,1,6);

	button_hlayout = new QHBoxLayout();

	button_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	button_hlayout->addItem(button_hspacer);

	d_cancel_button = new QPushButton(this);
	d_cancel_button->setAutoDefault(false);
	button_hlayout->addWidget(d_cancel_button);

	d_save_button = new QPushButton(this);
	d_save_button->setAutoDefault(false);
	button_hlayout->addWidget(d_save_button);
	dialog_layout->addLayout(button_hlayout,7,0,1,6);
	
	// defaults
	d_edit_table->Set_Edit_Mode(TEditTable::EDIT_MODE_REPLACE);
	d_edit_table->Set_Selection_Mode(TEditTable::SELECTION_MODE_DOWN);
	d_edit_table->Enable_Invert_Action(false);
	
	d_data_type_combo->addItem(QStringLiteral("Type 1D"),TNominalEditorDialog::TYPE_1D);
	d_data_type_combo->addItem(QStringLiteral("Type 3D"),TNominalEditorDialog::TYPE_3D);
	d_nominal_type = TNominalEditorDialog::TYPE_1D;
	d_data_type_combo->setCurrentIndex(0);

	this->setWindowTitle(QStringLiteral("Stepgauge Nominal Editor Dialog"));
	
	serial_label->setText(QStringLiteral("Serial Number:"));
	d_serial_edit->setText(QStringLiteral("90210"));
	start_label->setText(QStringLiteral("Start:"));
	d_start_edit->setText(QStringLiteral("0.000"));
	start_units_label->setText(QStringLiteral("mm"));
	increment_label->setText(QStringLiteral("Increment:"));
	d_increment_edit->setText(QStringLiteral("10.000"));
	increment_units_label->setText(QStringLiteral("mm"));
	edit_mode_label->setText(QStringLiteral("Edit Mode:"));
	data_type_label->setText(QStringLiteral("Data Type:"));
	d_mode_actual_radio->setText(QStringLiteral("Actual (mm)"));
	end_label->setText(QStringLiteral("End:"));
	d_end_edit->setText(QStringLiteral("1010.000"));
	end_units_label->setText(QStringLiteral("mm"));
	d_mode_deviation_radio->setText(QStringLiteral("Deviation (um)"));
	d_cancel_button->setText(QStringLiteral("Cancel"));
	d_save_button->setText(QStringLiteral("Save"));
	
	connect(d_increment_edit,&QLineEdit::editingFinished,this,&TNominalEditorDialog::Update_Editor);
	connect(d_end_edit,&QLineEdit::editingFinished,this,&TNominalEditorDialog::Update_Editor);
	connect(d_mode_actual_radio,&QRadioButton::toggled,this,&TNominalEditorDialog::Toggle_Mode);
	connect(d_data_type_combo,static_cast<void (QComboBox::*)(int)>(&QComboBox::activated),this,&TNominalEditorDialog::Toggle_Type);
	connect(d_edit_table,&TEditTable::Cell_Changed,this,&TNominalEditorDialog::Cell_Changed);
	connect(d_serial_edit,&QLineEdit::textChanged,this,&TNominalEditorDialog::Serial_Changed);
	
	connect(d_cancel_button,&QPushButton::clicked,this,&TNominalEditorDialog::reject);
	connect(d_save_button,&QPushButton::clicked,this,&TNominalEditorDialog::accept);
}

TNominalEditorDialog::~TNominalEditorDialog(void)
{
}

QString TNominalEditorDialog::Serial_Number(void) const
{
	return d_serial_edit->text();
}

void TNominalEditorDialog::Reset(
	const QString						&serial,
	const std::vector<TVector3>			&nominals,
	const TNominalEditorDialog::TNominalType type)
{
	int									index;
	
	d_serial_edit->setText(serial);
	d_nominal_type = type;
	
	index = d_data_type_combo->findData(type);
	
	if(!(index < 0))
	{
		d_data_type_combo->setCurrentIndex(index);
		this->Toggle_Type();
	}
		
	this->Initialize_Editor(nominals);
}

void TNominalEditorDialog::Update_Editor(void)
{
	QString								text;
	double								start;
	double								increment;
	double								end;
	double								position;
	int									row_count;
	int									cntr;
	int									width;
	bool								mode_actual;
	static const double					MINIMUM_INCREMENT(1.0);
	static const double					MINIMUM_LENGTH(2.0);
	
	start = d_start_edit->text().toDouble();
	increment = d_increment_edit->text().toDouble();
	end = d_end_edit->text().toDouble();
	
	if(increment < MINIMUM_INCREMENT ||
	   (end - start) < MINIMUM_LENGTH)
	{
		if(d_nominal_type == TNominalEditorDialog::TYPE_1D)
		{
			d_edit_table->Set_Cell_Count(0,1);
		}
		else
		{
			d_edit_table->Set_Cell_Count(0,3);
		}
		
		return;
	}

	row_count = 1 + static_cast<int>(0.5 + (end - start) / increment);
	d_nominals.resize(row_count);

	d_nominals.resize(row_count);

	if(d_nominal_type == TNominalEditorDialog::TYPE_1D)
	{
		d_edit_table->Set_Cell_Count(row_count,1);

		mode_actual = d_mode_actual_radio->isChecked();
		
		if(mode_actual)
		{
			d_edit_table->horizontalHeaderItem(0)->setText(QStringLiteral("Actual (mm)"));
			d_edit_table->Set_Display_Precision(4);
		}
		else
		{
			d_edit_table->horizontalHeaderItem(0)->setText(QStringLiteral("Deviation (um)"));
			d_edit_table->Set_Display_Precision(1);
		}
		
		position = 0.0;
		for(cntr = 0;cntr < row_count;++cntr)
		{
			d_edit_table->verticalHeaderItem(cntr)->setText(QString("%1").arg(position,0,'f',3));
			
			if(mode_actual)
			{
				d_edit_table->Set_Cell_Value(cntr,0,position);
			}
			else
			{
				d_edit_table->Set_Cell_Value(cntr,0,0.0);
			}
			
			d_nominals[cntr].x = position;
			
			position += increment;
		}
		
		d_edit_table->resizeColumnToContents(0);
	}
	else
	{
		d_edit_table->Set_Cell_Count(row_count,3);
		d_edit_table->Set_Display_Precision(4);

		d_edit_table->horizontalHeaderItem(0)->setText(QStringLiteral("X (mm)"));
		d_edit_table->horizontalHeaderItem(1)->setText(QStringLiteral("Y (mm)"));
		d_edit_table->horizontalHeaderItem(2)->setText(QStringLiteral("Z (mm)"));

		position = 0.0;
		for(cntr = 0;cntr < row_count;++cntr)
		{
			d_edit_table->verticalHeaderItem(cntr)->setText(QString("%1").arg(position,0,'f',3));
			
			d_edit_table->Set_Cell_Value(cntr,0,position);
			d_edit_table->Set_Cell_Value(cntr,1,0.0);
			d_edit_table->Set_Cell_Value(cntr,2,0.0);

			d_nominals[cntr].x = position;
			
			position += increment;
		}
		
		d_edit_table->resizeColumnToContents(0);
		
		width = d_edit_table->columnWidth(0);
		
		d_edit_table->setColumnWidth(1,width);
		d_edit_table->setColumnWidth(2,width);
	}
}

void TNominalEditorDialog::Toggle_Mode(
	bool								state)
{
	Q_UNUSED(state);
	
	std::vector<TVector3>				nominals;
	
	nominals = d_nominals;
	Initialize_Editor(nominals);
}

void TNominalEditorDialog::Toggle_Type(void)
{
	std::vector<TVector3>				nominals;
		
	d_nominal_type = static_cast<TNominalEditorDialog::TNominalType>(d_data_type_combo->currentData(Qt::UserRole).toInt());
	
	if(d_nominal_type == TNominalEditorDialog::TYPE_1D)
	{
		d_mode_actual_radio->setEnabled(true);
		d_mode_deviation_radio->setEnabled(true);
	}
	else
	{
		d_mode_actual_radio->setEnabled(false);
		d_mode_deviation_radio->setEnabled(false);
	}
	
	nominals = d_nominals;
	Initialize_Editor(nominals);
}

void TNominalEditorDialog::Cell_Changed(
	const int							row,
	const int							column,
	const double						&value)
{
	bool								mode_actual;
	double								dval;
	int									ival;
	
	assert(static_cast<unsigned int>(row) < d_nominals.size());

	if(d_nominal_type == TNominalEditorDialog::TYPE_1D)
	{
		mode_actual = d_mode_actual_radio->isChecked();

		if(column == 0)
		{
			if(mode_actual)
			{
				d_nominals[row].x = value;
			}
			else
			{
				dval = d_nominals[row].x;
				ival = static_cast<int>(0.5 + dval * 10);
				
				d_nominals[row].x = static_cast<double>(ival) / 10 + value / 1000.0;
			}
		}
	}
	else
	{
		switch(column)
		{
			case 0:
				d_nominals[row].x = value;
				break;
				
			case 1:
				d_nominals[row].y = value;
				break;
				
			case 2:
				d_nominals[row].z = value;
				break;
		}
	}
}

void TNominalEditorDialog::Serial_Changed(
	const QString						&text)
{
	d_save_button->setEnabled(text.length() > 0);
}

void TNominalEditorDialog::Initialize_Editor(
	const std::vector<TVector3>			&nominals)
{
	std::vector<double>::const_iterator iter;
	QString								text;
	double								start;
	double								increment;
	double								end;
	double								position;
	double								actual;
	int									ival;
	int									data_size;
	int									cntr;
	int									width;
	bool								mode_actual;
	static const double					MINIMUM_INCREMENT(1.0);
	static const double					MINIMUM_LENGTH(2.0);

	d_nominals = nominals;
	
	data_size = static_cast<int>(d_nominals.size());
	mode_actual = d_mode_actual_radio->isChecked();

	if(data_size < 1)
	{
		d_increment_edit->setText(QStringLiteral("10.000"));
		d_end_edit->setText(QStringLiteral("1010.000"));

		this->Update_Editor();
		return;
	}
	
	start = d_nominals[0].x;
	
	ival = static_cast<int>(0.5 + (d_nominals[1].x - d_nominals[0].x) * 10.0);
	increment = static_cast<double>(ival) / 10.0;
	
	end = increment * static_cast<double>(data_size - 1);
	
	if(increment < MINIMUM_INCREMENT ||
	   end < MINIMUM_LENGTH)
	{
		d_increment_edit->setText(QStringLiteral("10.000"));
		d_end_edit->setText(QStringLiteral("1010.000"));
		
		this->Update_Editor();
		return;
	}
	
	d_increment_edit->setText(QString("%1").arg(increment,0,'f',3));
	d_end_edit->setText(QString("%1").arg(end,0,'f',3));
	
	if(d_nominal_type == TNominalEditorDialog::TYPE_1D)
	{
		d_edit_table->Set_Cell_Count(data_size,1);
		
		if(mode_actual)
		{
			d_edit_table->horizontalHeaderItem(0)->setText(QStringLiteral("Actual (mm)"));
			d_edit_table->Set_Display_Precision(4);
		}
		else
		{
			d_edit_table->horizontalHeaderItem(0)->setText(QStringLiteral("Deviation (um)"));
			d_edit_table->Set_Display_Precision(1);
		}

		position = 0.0;
		for(cntr = 0;cntr < data_size;++cntr)
		{
			actual = d_nominals[cntr].x - start;
			
			d_edit_table->verticalHeaderItem(cntr)->setText(QString("%1").arg(position,0,'f',3));

			if(mode_actual)
			{
				d_edit_table->Set_Cell_Value(cntr,0,actual);
			}
			else
			{
				d_edit_table->Set_Cell_Value(cntr,0,(actual - position) * 1000);
			}
			
			position += increment;
		}
		
		d_edit_table->resizeColumnToContents(0);
	}
	else
	{
		d_edit_table->Set_Cell_Count(data_size,3);
		d_edit_table->Set_Display_Precision(4);
		
		d_edit_table->horizontalHeaderItem(0)->setText(QStringLiteral("X (mm)"));
		d_edit_table->horizontalHeaderItem(1)->setText(QStringLiteral("Y (mm)"));
		d_edit_table->horizontalHeaderItem(2)->setText(QStringLiteral("Z (mm)"));

		position = 0.0;
		for(cntr = 0;cntr < data_size;++cntr)
		{
			actual = d_nominals[cntr].x - start;
			
			d_edit_table->verticalHeaderItem(cntr)->setText(QString("%1").arg(position,0,'f',3));
			d_edit_table->Set_Cell_Value(cntr,0,actual);
			d_edit_table->Set_Cell_Value(cntr,1,d_nominals[cntr].y);
			d_edit_table->Set_Cell_Value(cntr,2,d_nominals[cntr].z);

			position += increment;
		}
		
		d_edit_table->resizeColumnToContents(0);
		
		width = d_edit_table->columnWidth(0);
		d_edit_table->setColumnWidth(1,width);
		d_edit_table->setColumnWidth(2,width);
	}
}

