/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QVariant>
#include <QAction>
#include <QApplication>
#include <QButtonGroup>
#include <QFrame>
#include <QGridLayout>
#include <QGroupBox>
#include <QHBoxLayout>
#include <QHeaderView>
#include <QLabel>
#include <QLineEdit>
#include <QPushButton>
#include <QSpacerItem>
#include <QTextEdit>
#include <QVBoxLayout>
#include <QCloseEvent>
#include <QRegularExpression>
#include <QRegularExpressionMatch>

#include "tolerancedialog.h"

TToleranceDialog::TToleranceDialog(
	const QWidget 						*parent,
	Qt::WindowFlags 					flags)
:QDialog(const_cast<QWidget*>(parent),flags)
{
	QGridLayout 						*dialog_layout;
	QGroupBox 							*sample_group_box;
	QGridLayout 						*sample_layout;
	QLabel 								*tolerance1_label;
	QLabel 								*tolerance1_text_label;
	QLabel 								*tolerance2_label;
	QLabel 								*tolerance2_text_label;
	QLabel 								*tolerance3_label;
	QLabel 								*tolerance3_text_label;
	QGroupBox 							*definition_group;
	QVBoxLayout 						*vertical_layout;
	QTextEdit 							*help_text;
	QHBoxLayout 						*tolerance_hlayout;
	QHBoxLayout							*button_hlayout;
	QLabel 								*tolerance_label;
	QLabel 								*temperature_max_label;
	QLabel 								*temperature_min_label;
	QFrame								*separator_line;
	QSpacerItem							*button_hspacer;
	QPushButton							*cancel_button;
	QPalette							palette;
	QBrush								brush(QColor(255, 255, 255, 0));
	QBrush								brush1(QColor(255, 255, 255, 255));

	this->resize(630,400);

	brush.setStyle(Qt::SolidPattern);
	brush1.setStyle(Qt::SolidPattern);
	
	palette.setBrush(QPalette::Active, QPalette::Base, brush);
	palette.setBrush(QPalette::Inactive, QPalette::Base, brush);
	
	palette.setBrush(QPalette::Disabled, QPalette::Base, brush1);

	dialog_layout = new QGridLayout(this);

	sample_group_box = new QGroupBox(this);

	sample_layout = new QGridLayout(sample_group_box);

	tolerance1_label = new QLabel(sample_group_box);
	tolerance1_label->setMinimumSize(QSize(64, 64));
	tolerance1_label->setMaximumSize(QSize(64, 64));
	tolerance1_label->setPixmap(QPixmap(QString::fromUtf8(":/icon64/tolerance1_icon64.png")));
	sample_layout->addWidget(tolerance1_label, 0, 0, 1, 1);

	tolerance1_text_label = new QLabel(sample_group_box);
	sample_layout->addWidget(tolerance1_text_label, 0, 1, 1, 1);

	tolerance2_label = new QLabel(sample_group_box);
	tolerance2_label->setMinimumSize(QSize(64, 64));
	tolerance2_label->setMaximumSize(QSize(64, 64));
	tolerance2_label->setPixmap(QPixmap(QString::fromUtf8(":/icon64/tolerance2_icon64.png")));
	sample_layout->addWidget(tolerance2_label, 1, 0, 1, 1);

	tolerance2_text_label = new QLabel(sample_group_box);
	sample_layout->addWidget(tolerance2_text_label, 1, 1, 1, 1);

	tolerance3_label = new QLabel(sample_group_box);
	tolerance3_label->setMinimumSize(QSize(64, 64));
	tolerance3_label->setMaximumSize(QSize(64, 64));
	tolerance3_label->setPixmap(QPixmap(QString::fromUtf8(":/icon64/tolerance3_icon64.png")));
	sample_layout->addWidget(tolerance3_label, 2, 0, 1, 1);

	tolerance3_text_label = new QLabel(sample_group_box);
	sample_layout->addWidget(tolerance3_text_label, 2, 1, 1, 1);

	dialog_layout->addWidget(sample_group_box, 0, 0, 1, 1);

	definition_group = new QGroupBox(this);

	vertical_layout = new QVBoxLayout(definition_group);
	vertical_layout->setContentsMargins(3, -1, 3, 3);

	help_text = new QTextEdit(definition_group);
	help_text->setPalette(palette);
	help_text->setFrameShape(QFrame::NoFrame);
	help_text->setReadOnly(true);
	vertical_layout->addWidget(help_text);

	dialog_layout->addWidget(definition_group, 0, 1, 1, 1);

	tolerance_hlayout = new QHBoxLayout();

	tolerance_label = new QLabel(this);
	tolerance_label->setAlignment(Qt::AlignRight | Qt::AlignTrailing | Qt::AlignVCenter);
	tolerance_hlayout->addWidget(tolerance_label);

	d_tolerance_edit = new QLineEdit(this);
	d_tolerance_edit->setAlignment(Qt::AlignCenter);
	tolerance_hlayout->addWidget(d_tolerance_edit);

	temperature_min_label = new QLabel(this);
	temperature_min_label->setAlignment(Qt::AlignRight | Qt::AlignTrailing | Qt::AlignVCenter);
	tolerance_hlayout->addWidget(temperature_min_label);

	d_min_temperature_edit = new QLineEdit(this);
	d_min_temperature_edit->setAlignment(Qt::AlignCenter);
	d_min_temperature_edit->setMaximumSize(QSize(51, 16777215));
	tolerance_hlayout->addWidget(d_min_temperature_edit);

	temperature_max_label = new QLabel(this);
	temperature_max_label->setAlignment(Qt::AlignRight | Qt::AlignTrailing | Qt::AlignVCenter);
	tolerance_hlayout->addWidget(temperature_max_label);

	d_max_temperature_edit = new QLineEdit(this);
	d_max_temperature_edit->setAlignment(Qt::AlignCenter);
	d_max_temperature_edit->setMaximumSize(QSize(51, 16777215));
	tolerance_hlayout->addWidget(d_max_temperature_edit);

	dialog_layout->addLayout(tolerance_hlayout, 1, 0, 1, 2);
	
	separator_line = new QFrame(this);
	separator_line->setFrameShape(QFrame::HLine);
	separator_line->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(separator_line,2,0,1,2);
	
	button_hlayout = new QHBoxLayout();
	
	button_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	button_hlayout->addItem(button_hspacer);
	
	cancel_button = new QPushButton(this);
	cancel_button->setAutoDefault(false);
	button_hlayout->addWidget(cancel_button);
	
	d_accept_button = new QPushButton(this);
	d_accept_button->setAutoDefault(false);
	button_hlayout->addWidget(d_accept_button);
	
	dialog_layout->addLayout(button_hlayout,3,0,1,2);

	this->setWindowTitle(QStringLiteral("Tolerance Editor"));

	sample_group_box->setTitle(QStringLiteral("Tolerance Formats"));
	tolerance1_label->setText(QString());
	tolerance1_text_label->setText(QStringLiteral("El Mpe = A"));
	tolerance2_label->setText(QString());
	tolerance2_text_label->setText(QStringLiteral("El Mpe = A + BL/C"));
	tolerance3_label->setText(QString());
	tolerance3_text_label->setText(QStringLiteral("El Mpe = A + BL/C | D"));
	definition_group->setTitle(QStringLiteral("Definitions"));
	
	help_text->setHtml(QStringLiteral("<html><head>\n</head><body style=\" font-weight:400; font-style:normal;\">\n" \
		   "<p style=\" margin-top:0px; margin-bottom:0px; margin-left:0px; margin-right:0px; -qt-block-indent:0; text-indent:0px;\"><span style=\" font-weight:600;\">El Mpe: </span>Maximum Permissible Error in one of the accepted formats (um).</p><br>" \
		   "<p style=\" margin-top:0px; margin-bottom:0px; margin-left:0px; margin-right:0px; -qt-block-indent:0; text-indent:0px;\"><span style=\" font-weight:600;\">Ts_Max: </span>Maximum Permissible Temperature (C)</p><br>" \
		   "<p style=\" margin-top:0px; margin-bottom:0px; margin-left:0px; margin-right:0px; -qt-block-indent:0; text-indent:0px;\"><span style=\" font-weight:600;\">Ts_Min: </span>Minimum Permissible Temperature (C)</p></body></html>"));

	tolerance_label->setText(QStringLiteral("El Mpe:"));
	temperature_max_label->setText(QStringLiteral("Ts_Max:"));
	temperature_min_label->setText(QStringLiteral("Ts_Min:"));
	cancel_button->setText(QStringLiteral("Cancel"));
	d_accept_button->setText(QStringLiteral("Accept"));

	// Set Default Tolerance
	d_tolerance_edit->setText(QStringLiteral("3.0 + 4.0L/1000"));
	d_min_temperature_edit->setText(QStringLiteral("18.00"));
	d_max_temperature_edit->setText(QStringLiteral("22.00"));

	this->Tolerance_Changed();

	connect(d_tolerance_edit,&QLineEdit::textChanged,this,&TToleranceDialog::Tolerance_Changed);
	connect(d_max_temperature_edit,&QLineEdit::textChanged,this,&TToleranceDialog::Tolerance_Changed);
	connect(d_min_temperature_edit,&QLineEdit::textChanged,this,&TToleranceDialog::Tolerance_Changed);
	
	connect(cancel_button,&QPushButton::clicked,this,&TToleranceDialog::reject);
	connect(d_accept_button,&QPushButton::clicked,this,&TToleranceDialog::accept);
}

TToleranceDialog::~TToleranceDialog(void)
{

}

double TToleranceDialog::Tolerance_Value(
	const double						&length) const
{
	double								tolerance(0.0);
	
	switch(d_tolerance_type)
	{
		case TOLERANCE_BAND:
			tolerance = d_tolerance_a;
			break;
			
		case TOLERANCE_FORMULA:
			tolerance = d_tolerance_a;
			
			if(d_tolerance_c > 0.1)
			{
				tolerance += d_tolerance_b * length / d_tolerance_c;
			}			
			break;
			
		case TOLERANCE_FORMULA_WITH_LIMIT:
			tolerance = d_tolerance_a;
			
			if(d_tolerance_c > 0.1)
			{
				tolerance += d_tolerance_b * length / d_tolerance_c;
			}
			
			if(tolerance > d_tolerance_d)
			{
				tolerance = d_tolerance_d;
			}
			break;
	}
	
	return tolerance;
}

double TToleranceDialog::Limit_Position(
	const double 						&TolA,
	const double 						&TolB,
	const double 						&TolC,
	const double 						&TolD) const
{
	double								Slope;
	double								Position(0);

	if(TolD < TolA)
	{
		return Position;
	}

	Slope = TolB / TolC;	// mm / m

	Position = (TolD - TolA) / Slope;

	return Position;
}

void TToleranceDialog::Set_Tolerance_Type(
	const TToleranceType				&type)
{
	d_tolerance_type = type;
}

void TToleranceDialog::Set_Tolerance_A(
	const double						&a)
{
	d_tolerance_a = a;
}

void TToleranceDialog::Set_Tolerance_B(
	const double						&b)
{
	d_tolerance_b = b;
}

void TToleranceDialog::Set_Tolerance_C(
	const double						&c)
{
	d_tolerance_c = c;
}

void TToleranceDialog::Set_Tolerance_D(
	const double						&d)
{
	d_tolerance_d = d;
}

void TToleranceDialog::Set_Lower_Temperature_Limit(
	const double						&tolerance)
{
	d_lower_temperature_limit = tolerance;

	d_min_temperature_edit->setText(QString("%1").arg(d_lower_temperature_limit,0,'f',2));
}

void TToleranceDialog::Set_Upper_Temperature_Limit(
	const double						&tolerance)
{
	d_upper_temperature_limit = tolerance;

	d_max_temperature_edit->setText(QString("%1").arg(d_upper_temperature_limit,0,'f',2));
}

void TToleranceDialog::Update(void)
{
	// determine type of tolerance to use based on current tolerance values
	if(d_tolerance_d > 0)
	{
		d_tolerance_type = TOLERANCE_FORMULA_WITH_LIMIT;
	}
	else if(d_tolerance_c > 0.1)
	{
		d_tolerance_type = TOLERANCE_FORMULA;
	}
	else
	{
		d_tolerance_type = TOLERANCE_BAND;
	}

	this->Update_Display();
	this->Tolerance_Changed();
}

void TToleranceDialog::Tolerance_Changed(void)
{
	QRegularExpression					single_regexp("^([0-9.]{1,8})$");
	QRegularExpression					expression_regexp("^([0-9.]{1,8})[ ]{0,1}[+][ ]{0,1}([0-9.]{1,8})[ ]{0,1}[L,l]{1}[ ]{0,1}[/][ ]{0,1}([0-9.]{1,9})$");
	QRegularExpression					expression_limit_regexp("^([0-9.]{1,8})[ ]{0,1}[+][ ]{0,1}([0-9.]{1,8})[ ]{0,1}[L,l]{1}[ ]{0,1}[/][ ]{0,1}([0-9.]{1,9})[ ]{0,1}[|][ ]{0,1}([0-9.]{1,8})$");
	QRegularExpressionMatch				expression_match;
	QStringList							expression_text_list;
	QString								text;
	bool								state;

	d_upper_temperature_limit = d_max_temperature_edit->text().toDouble(&state);

	if(state == false || d_upper_temperature_limit > 30)
	{
		d_max_temperature_edit->setStyleSheet(QStringLiteral("background-color: rgb(255, 151, 160);"));
	}
	else
	{
		d_max_temperature_edit->setStyleSheet(QString());
	}

	d_lower_temperature_limit = d_min_temperature_edit->text().toDouble(&state);

	if(state == false || d_lower_temperature_limit < 10)
	{
		d_min_temperature_edit->setStyleSheet(QStringLiteral("background-color: rgb(255, 151, 160);"));
	}
	else
	{
		d_min_temperature_edit->setStyleSheet(QString());
	}

	if(!(d_upper_temperature_limit > d_lower_temperature_limit))
	{
		d_max_temperature_edit->setStyleSheet(QStringLiteral("background-color: rgb(255, 151, 160);"));
		d_min_temperature_edit->setStyleSheet(QStringLiteral("background-color: rgb(255, 151, 160);"));
	}

	text = d_tolerance_edit->text().simplified().toUpper();
	
	if(!text.length())
	{
		d_tolerance_edit->setStyleSheet(QStringLiteral("background-color: rgb(255, 151, 160);"));
		return;
	}
	else
	{
		d_tolerance_edit->setStyleSheet(QString());
	}
	
	expression_match = single_regexp.match(text);
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() == 2)
		{
			d_tolerance_type = TToleranceDialog::TOLERANCE_BAND;
			
			d_tolerance_a = expression_text_list[1].toDouble() / 1000.0;
			d_tolerance_b = 0;
			d_tolerance_c = 0;
			d_tolerance_d = 0;
			
			if(d_tolerance_a > 1)
			{
				d_tolerance_edit->setStyleSheet(QStringLiteral("background-color: rgb(255, 151, 160);"));
				return;
			}
			
			return;
		}
	}
	
	expression_match = expression_regexp.match(text);
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() == 4)
		{
			d_tolerance_type = TToleranceDialog::TOLERANCE_FORMULA;
			
			d_tolerance_a = expression_text_list[1].toDouble() / 1000.0;
			d_tolerance_b = expression_text_list[2].toDouble() / 1000.0;
			d_tolerance_c = expression_text_list[3].toDouble();
			d_tolerance_d = 0;
			
			if(d_tolerance_a > 1 || d_tolerance_b > 1)
			{
				d_tolerance_edit->setStyleSheet(QStringLiteral("background-color: rgb(255, 151, 160);"));
				return;
			}
			
			return;
		}
	}
	
	expression_match = expression_limit_regexp.match(text);
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() == 5)
		{
			d_tolerance_type = TToleranceDialog::TOLERANCE_FORMULA_WITH_LIMIT;
			
			d_tolerance_a = expression_text_list[1].toDouble() / 1000.0;
			d_tolerance_b = expression_text_list[2].toDouble() / 1000.0;
			d_tolerance_c = expression_text_list[3].toDouble();
			d_tolerance_d = expression_text_list[4].toDouble() / 1000.0;
			
			if(d_tolerance_a > 1 || d_tolerance_b > 1 || d_tolerance_d > 1)
			{
				d_tolerance_edit->setStyleSheet(QStringLiteral("background-color: rgb(255, 151, 160);"));
				return;
			}
			
			return;
		}
	}

	d_tolerance_edit->setStyleSheet(QStringLiteral("background-color: rgb(255, 151, 160);"));
}

void TToleranceDialog::Update_Display(void)
{
	bool								prev_state;
	
	prev_state = d_tolerance_edit->blockSignals(true);

	switch(d_tolerance_type)
	{
		case TOLERANCE_BAND:
			d_tolerance_edit->setText(QString("%1").arg(d_tolerance_a * 1000.0,0,'f',1));
			break;
			
		case TOLERANCE_FORMULA:
			d_tolerance_edit->setText(QString("%1+%2L/%3")
									.arg(d_tolerance_a * 1000.0,0,'f',1)
									.arg(d_tolerance_b * 1000.0,0,'f',1)
									.arg(d_tolerance_c,0,'f',0));
			break;
			
		case TOLERANCE_FORMULA_WITH_LIMIT:
			d_tolerance_edit->setText(QString("%1+%2L/%3 | %4")
									.arg(d_tolerance_a * 1000.0,0,'f',1)
									.arg(d_tolerance_b * 1000.0,0,'f',1)
									.arg(d_tolerance_c,0,'f',0)
									.arg(d_tolerance_d * 1000.0,0,'f',1));
			break;
	}
	
	d_tolerance_edit->blockSignals(prev_state);
}

