/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QLabel>
#include <QTableWidget>
#include <QTreeWidget>
#include <QTreeWidgetItem>
#include <QRegularExpression>
#include <QRegularExpressionMatch>
#include <QVBoxLayout>
#include <QToolButton>
#include <QFont>

#include "../../core/xmlfile.h"
#include "../../core/messagebox.h"
#include "libraryeditdialog.h"

#include "librarieswidget.h"

TLibrariesWidget::TLibrariesWidget(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QWidget(const_cast<QWidget*>(parent),flags)
{
	QLabel								*source_label;
	QLabel								*target_label;
	QTreeWidgetItem						*libraries_tree_header_item;
	QTreeWidgetItem						*targets_tree_header_item;
	QVBoxLayout							*widget_vlayout;
	QHBoxLayout							*target_button_hlayout;
	QHBoxLayout							*library_button_hlayout;
	QSpacerItem							*target_widget_hspacer;
	QSpacerItem							*library_widget_hspacer;
	QFont								font;

	font.setPointSize(15);
	font.setWeight(50);
	font.setItalic(true);
	font.setBold(false);
	font.setUnderline(true);

	widget_vlayout = new QVBoxLayout(this);

	target_label = new QLabel(this);
	target_label->setFont(font);
	widget_vlayout->addWidget(target_label);

	d_target_tree = new QTreeWidget(this);
	d_target_tree->setColumnCount(1);
	targets_tree_header_item = d_target_tree->headerItem();
	widget_vlayout->addWidget(d_target_tree);
	
	target_button_hlayout = new QHBoxLayout();

	target_widget_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	target_button_hlayout->addItem(target_widget_hspacer);

	d_add_target_button = new QToolButton(this);
	d_add_target_button->setFixedSize(QSize(16,16));
	target_button_hlayout->addWidget(d_add_target_button);

	d_remove_target_button = new QToolButton(this);
	d_remove_target_button->setFixedSize(QSize(16,16));
	target_button_hlayout->addWidget(d_remove_target_button);
	
	widget_vlayout->addLayout(target_button_hlayout);

	source_label = new QLabel(this);
	source_label->setFont(font);
	widget_vlayout->addWidget(source_label);

	d_libraries_tree = new QTreeWidget(this);
	d_libraries_tree->setColumnCount(3);
	d_libraries_tree->setSelectionMode(QAbstractItemView::ExtendedSelection);
	libraries_tree_header_item = d_libraries_tree->headerItem();
	widget_vlayout->addWidget(d_libraries_tree);
	
	library_button_hlayout = new QHBoxLayout();

	library_widget_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	library_button_hlayout->addItem(library_widget_hspacer);

	d_add_library_button = new QToolButton(this);
	d_add_library_button->setFixedSize(QSize(16,16));
	library_button_hlayout->addWidget(d_add_library_button);

	d_remove_library_button = new QToolButton(this);
	d_remove_library_button->setFixedSize(QSize(16,16));
	library_button_hlayout->addWidget(d_remove_library_button);

	d_edit_library_button = new QToolButton(this);
	d_edit_library_button->setFixedSize(QSize(16,16));
	library_button_hlayout->addWidget(d_edit_library_button);

	widget_vlayout->addLayout(library_button_hlayout);
	
	d_add_target_button->setIcon(QIcon(QStringLiteral(":/gui/edit_add_x16.png")));
	d_remove_target_button->setIcon(QIcon(QStringLiteral(":/gui/edit_remove_x16.png")));

	d_add_library_button->setIcon(QIcon(QStringLiteral(":/gui/edit_add_x16.png")));
	d_remove_library_button->setIcon(QIcon(QStringLiteral(":/gui/edit_remove_x16.png")));
	d_edit_library_button->setIcon(QIcon(QStringLiteral(":/gui/edit_edit_x16.png")));

	// defaults
	d_library_edit_dialog = 0;
	d_msg_box = 0;

	d_remove_target_button->setEnabled(false);
	d_remove_library_button->setEnabled(false);
	d_edit_library_button->setEnabled(false);
	
	d_add_target_button->setToolTip(QStringLiteral("Add new target"));
	d_remove_target_button->setToolTip(QStringLiteral("Remove selected target"));
	
	d_add_library_button->setToolTip(QStringLiteral("Add new library"));
	d_remove_library_button->setToolTip(QStringLiteral("Remove selected libraries"));
	d_edit_library_button->setToolTip(QStringLiteral("Edit selected library"));

	target_label->setText(QStringLiteral("Target Library Directories:"));
	source_label->setText(QStringLiteral("Source Libraries:"));
	targets_tree_header_item->setText(0,QStringLiteral("Library and Plugin Directories"));
	libraries_tree_header_item->setText(0,QStringLiteral("Directory"));
	libraries_tree_header_item->setText(1,QStringLiteral("Type"));
	libraries_tree_header_item->setText(2,QStringLiteral("Name"));

	connect(d_target_tree,&QTreeWidget::itemClicked,this,&TLibrariesWidget::Target_Item_Clicked);
	connect(d_target_tree,&QTreeWidget::itemSelectionChanged,this,&TLibrariesWidget::Target_Selection_Changed);
	connect(d_libraries_tree,&QTreeWidget::itemClicked,this,&TLibrariesWidget::Library_Item_Clicked);
	connect(d_libraries_tree,&QTreeWidget::itemSelectionChanged,this,&TLibrariesWidget::Library_Selection_Changed);
	connect(d_add_target_button,&QToolButton::clicked,this,&TLibrariesWidget::Add_Target);
	connect(d_remove_target_button,&QToolButton::clicked,this,&TLibrariesWidget::Remove_Target);
	connect(d_add_library_button,&QToolButton::clicked,this,&TLibrariesWidget::Add_Library);
	connect(d_remove_library_button,&QToolButton::clicked,this,&TLibrariesWidget::Remove_Library);
	connect(d_edit_library_button,&QToolButton::clicked,this,&TLibrariesWidget::Edit_Library);
}

TLibrariesWidget::~TLibrariesWidget(void)
{
}

bool TLibrariesWidget::Save_Defaults(
	const QString 						&file_path)
{
	std::vector<TLibrariesWidget::TTarget>::const_iterator target_iter;
	std::vector<TLibrariesWidget::TLibrary>::const_iterator library_iter;
	QString								file_name;
	TXmlFile							xml_file;
	QDomElement							e0;
	QDomElement							e1;
	QDomElement							e2;
	QString								text;
	
	file_name = file_path + QStringLiteral("/libraries.xml");
	
	this->Update_Target_Data();
	this->Update_Library_Data();

	try
	{
		xml_file.Set_File_Name(file_name);
		xml_file.Reset(QStringLiteral("Library_Data"),1);

		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Targets"));

		for(target_iter = d_targets.begin();target_iter != d_targets.end();++target_iter)
		{
			e1 = xml_file.Create_Node(&e0,QStringLiteral("Target"));
			e2 = xml_file.Create_Node(&e1,QStringLiteral("Name"));
			xml_file.Write_Text_Node(&e2,(*target_iter).target_name);
			
			e2 = xml_file.Create_Node(&e1,QStringLiteral("Checked"));
			(*target_iter).checked ? xml_file.Write_Text_Node(&e2,QStringLiteral("Yes")):xml_file.Write_Text_Node(&e2,QStringLiteral("No"));
		}
		
		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Libraries"));

		for(library_iter = d_libraries.begin();library_iter != d_libraries.end();++library_iter)
		{
			e1 = xml_file.Create_Node(&e0,QStringLiteral("Library"));
			e2 = xml_file.Create_Node(&e1,QStringLiteral("Target_Name"));
			xml_file.Write_Text_Node(&e2,(*library_iter).target_name);
			
			e2 = xml_file.Create_Node(&e1,QStringLiteral("Name"));
			xml_file.Write_Text_Node(&e2,(*library_iter).library_name);

			e2 = xml_file.Create_Node(&e1,QStringLiteral("Type"));
			
			switch((*library_iter).type)
			{
				case TTypes::TYPE_DYLIB:
					xml_file.Write_Text_Node(&e2,QStringLiteral("DYLIB"));
					break;
					
				case TTypes::TYPE_PLUGIN:
					xml_file.Write_Text_Node(&e2,QStringLiteral("PLUGIN"));
					break;
			}
			
			e2 = xml_file.Create_Node(&e1,QStringLiteral("Checked"));
			(*library_iter).checked ? xml_file.Write_Text_Node(&e2,QStringLiteral("Yes")):xml_file.Write_Text_Node(&e2,QStringLiteral("No"));
		}

		xml_file.Write_File();
	}
	catch(TXmlFileException xml_file_exception)
	{
		d_last_error = xml_file_exception.ErrorText();
		return false;
	}

	return true;
}

QStringList TLibrariesWidget::Active_Targets(
	const bool 							with_active_libraries) const
{
	QTreeWidgetItem						*item;
	QStringList							list;
	int									count;
	int									cntr;
	
	count = d_target_tree->topLevelItemCount();
	
	for(cntr = 0;cntr < count;++cntr)
	{
		item = d_target_tree->topLevelItem(cntr);
		
		if(item->checkState(0) == Qt::Checked)
		{
			if(with_active_libraries)
			{
				if(this->Has_Libraries(item->text(0)))
				{
					list.push_back(item->text(0));
				}
			}
			else
			{
				list.push_back(item->text(0));
			}
		}
	}
	
	return list;
}

QStringList TLibrariesWidget::Active_Libraries(
	const QString 						&target_name) const
{
	QTreeWidgetItem						*item;
	QStringList							list;
	int									count;
	int									cntr;
	
	count = d_libraries_tree->topLevelItemCount();
	
	for(cntr = 0;cntr < count;++cntr)
	{
		item = d_libraries_tree->topLevelItem(cntr);
		
		if(item->text(0) == target_name)
		{
			if(item->checkState(0) == Qt::Checked)
			{
				list.push_back(item->text(2));
			}
		}
	}
	
	return list;
}

QStringList TLibrariesWidget::Enabled_Libraries(
	const TTypes::TLibraryType 			library_type) const
{
	std::vector<TLibrariesWidget::TLibrary>::const_iterator library_iter;
	QStringList							list;
	
	for(library_iter = d_libraries.begin();library_iter != d_libraries.end();++library_iter)
	{
		if((*library_iter).type == library_type &&
		   (*library_iter).checked == true)
		{
			list.push_back((*library_iter).library_name);
		}
	}
	
	return list;
}

TLibrariesWidget::TLibrary TLibrariesWidget::Library(
	const QString 						&target_name,
	const QString 						&library_name) const
{
	std::vector<TLibrariesWidget::TLibrary>::const_iterator iter;
	TLibrariesWidget::TLibrary			library;
	
	library.checked = false;
	library.type = TTypes::TYPE_DYLIB;
	
	for(iter = d_libraries.begin();iter != d_libraries.end();++iter)
	{
		if((*iter).target_name == target_name &&
		   (*iter).library_name == library_name)
		{
			library = (*iter);
			break;
		}
	}
	
	return library;
}

bool TLibrariesWidget::Load_Defaults(
	const QString 						&file_path)
{
	TXmlFile							xml_file;
	QString								file_name;
	int									version;
	bool								result(true);
	
	file_name = file_path + QStringLiteral("/libraries.xml");
	
	d_targets.clear();
	d_libraries.clear();

	try
	{
		xml_file.Set_File_Name(file_name);
		xml_file.Open(QStringLiteral("Library_Data"),&version);

		switch(version)
		{
			case 1:
				this->Load_V1(&xml_file);
				break;
				
			default:
				throw TXmlFileException(QStringLiteral("Library datafile version not recognized"));
		}
	}
	catch(TXmlFileException xml_file_exception)
	{
		d_last_error = xml_file_exception.ErrorText();
		
		this->Initialize();
		result = false;
	}
	
	this->Update_Target_Tree();
	this->Update_Library_Tree();
	
	this->Target_Item_Clicked();

	return result;
}

void TLibrariesWidget::Unselect_Targets(void)
{
	std::vector<TLibrariesWidget::TTarget>::iterator iter;
	
	for(iter = d_targets.begin();iter != d_targets.end();++iter)
	{
		(*iter).checked = false;
	}
}

void TLibrariesWidget::Unselect_Libaries(void)
{
	std::vector<TLibrariesWidget::TLibrary>::iterator iter;
	
	for(iter = d_libraries.begin();iter != d_libraries.end();++iter)
	{
		(*iter).checked = false;
	}
}

	
void TLibrariesWidget::Select_Target(
	const QString 						&name)
{
	std::vector<TLibrariesWidget::TTarget>::iterator iter;
	
	for(iter = d_targets.begin();iter != d_targets.end();++iter)
	{
		if((*iter).target_name == name)
		{
			(*iter).checked = true;
			return;
		}
	}
}

void TLibrariesWidget::Select_Library(
	const QString 						&target,
	const QString 						&library)
{
	std::vector<TLibrariesWidget::TLibrary>::iterator iter;
	
	
	for(iter = d_libraries.begin();iter != d_libraries.end();++iter)
	{
		if((*iter).target_name == target)
		{
			if((*iter).library_name.startsWith(library))
			{
				(*iter).checked = true;
				
				return;
			}
		}
	}
}

void TLibrariesWidget::Update(void)
{
	this->Update_Target_Tree();
	this->Update_Library_Tree();
	this->Target_Item_Clicked();
}

void TLibrariesWidget::Target_Item_Clicked(void)
{	
	QStringList							list;
		
	list = this->Active_Targets();
	this->Update_Library_Tree_Status(list);
}

void TLibrariesWidget::Target_Selection_Changed(void)
{
	QList<QTreeWidgetItem*>				items;
	
	items = d_target_tree->selectedItems();
	d_remove_target_button->setEnabled(false);
	
	if(items.size() == 1)
	{
		d_remove_target_button->setEnabled(true);
	}
}

void TLibrariesWidget::Library_Item_Clicked(void)
{
	this->Update_Library_Data();
}

void TLibrariesWidget::Library_Selection_Changed(void)
{
	QList<QTreeWidgetItem*>				items;
	
	items = d_libraries_tree->selectedItems();
	
	d_remove_library_button->setEnabled(false);
	d_edit_library_button->setEnabled(false);

	if(items.size())
	{
		d_remove_library_button->setEnabled(true);
		d_edit_library_button->setEnabled(true);
	}
}

void TLibrariesWidget::Add_Target(void)
{
	TLibraryEditDialog::TEditData		data;
	TLibrariesWidget::TTarget			target;
	QStringList							list;
	
	if(!d_library_edit_dialog)
	{
		d_library_edit_dialog = new TLibraryEditDialog(this);
	}
	
	d_library_edit_dialog->Reset(TTypes::EDITOR_ADD_TARGET,this->Target_Names());
	
	if(QDialog::Accepted == d_library_edit_dialog->exec())
	{
		data = d_library_edit_dialog->Data();
		
		target.target_name = data.target_name;
		target.checked = true;
		
		if(!this->Target_Exists(target.target_name))
		{
			d_targets.push_back(target);
			this->Update_Target_Tree();
			
			list = this->Active_Targets();
			this->Update_Library_Tree_Status(list);
		}
	}
}

void TLibrariesWidget::Remove_Target(void)
{
	QList<QTreeWidgetItem*>				items;
	QList<QTreeWidgetItem*>::const_iterator iter;
	
	items = d_target_tree->selectedItems();
	
	if(!d_msg_box)
	{
		d_msg_box = new TMessageBox(this);
	}
	
	d_msg_box->setText(QStringLiteral("Delete Targets"));
	d_msg_box->setInformativeText(QStringLiteral("Do you want to delete the selected targets?"));
	d_msg_box->setDetailedText(QString());
	d_msg_box->setStandardButtons(QMessageBox::Yes | QMessageBox::No);
	d_msg_box->setDefaultButton(QMessageBox::Yes);
	d_msg_box->setIcon(QMessageBox::Warning);
	
	if(QMessageBox::No == d_msg_box->exec())
	{
		return;
	}

	for(iter = items.begin();iter != items.end();++iter)
	{
		this->Target_Remove((*iter)->text(0));
	}
	
	this->Update_Target_Tree();
}

void TLibrariesWidget::Add_Library(void)
{
	TLibraryEditDialog::TEditData		data;
	TLibrariesWidget::TLibrary			library;

	if(!d_library_edit_dialog)
	{
		d_library_edit_dialog = new TLibraryEditDialog(this);
	}
	
	d_library_edit_dialog->Reset(TTypes::EDITOR_ADD_LIBRARY,this->Target_Names());
	
	if(QDialog::Accepted == d_library_edit_dialog->exec())
	{
		data = d_library_edit_dialog->Data();
		
		library.target_name = data.target_name;
		library.library_name = data.library_name;
		library.type = data.type;
		
		library.checked = true;
		
		if(!this->Library_Exists(library.target_name,library.library_name))
		{
			d_libraries.push_back(library);
			this->Update_Library_Tree();
		}
	}
}

void TLibrariesWidget::Remove_Library(void)
{
	QList<QTreeWidgetItem*>				items;
	QList<QTreeWidgetItem*>::iterator	iter;
	QString								target_name;
	QString								library_name;
	
	if(!d_msg_box)
	{
		d_msg_box = new TMessageBox(this);
	}
	
	d_msg_box->setText(QStringLiteral("Delete Libraries"));
	d_msg_box->setInformativeText(QStringLiteral("Do you want to delete the selected libraries?"));
	d_msg_box->setDetailedText(QString());
	d_msg_box->setStandardButtons(QMessageBox::Yes | QMessageBox::No);
	d_msg_box->setDefaultButton(QMessageBox::Yes);
	d_msg_box->setIcon(QMessageBox::Warning);
	
	if(QMessageBox::No == d_msg_box->exec())
	{
		return;
	}

	items = d_libraries_tree->selectedItems();

	for(iter = items.begin();iter != items.end();++iter)
	{
		target_name = (*iter)->text(0);
		library_name = (*iter)->text(2);

		this->Library_Remove(target_name,library_name);
	}
	
	this->Update_Library_Tree();
}

void TLibrariesWidget::Edit_Library(void)
{
	TLibraryEditDialog::TEditData		data;
	TLibrariesWidget::TLibrary			library;
	QList<QTreeWidgetItem*>				items;
	QList<QTreeWidgetItem*>::iterator	iter;
	QRegularExpression					library_version_expression(QStringLiteral("[.so]{4}([0-9]{1,3})[.]([0-9]{1,3})[.]([0-9]{1,3})$"));
	QRegularExpression					library_name_expression(QStringLiteral("^([a-z,A-Z,0-9]{1,64}[.so]{3})[.]([0-9]{1,3})[.]([0-9]{1,3})[.]([0-9]{1,3})$"));
	QRegularExpressionMatch				library_match;
	QStringList							matched_libraries_list;
	QString								target_name;
	QString								library_name;
	QString								plugin_names;
	QString								text;
	int									library_versions[3];
	TTypes::TLibraryType				type;

	if(!d_library_edit_dialog)
	{
		d_library_edit_dialog = new TLibraryEditDialog(this);
	}
	
	items = d_libraries_tree->selectedItems();

	if(items.size() == 1)
	{
		d_library_edit_dialog->Reset(TTypes::EDITOR_EDIT_LIBRARY,this->Target_Names());
		
		target_name = items[0]->text(0);
		library_name = items[0]->text(2);
		
		text = items[0]->text(1);
		
		if(text == QStringLiteral("Dylib"))
		{
			type = TTypes::TYPE_DYLIB;
		}
		else // PLUGIN
		{
			type = TTypes::TYPE_PLUGIN;
		}

		d_library_edit_dialog->Set_Target_Name(target_name);
		d_library_edit_dialog->Set_Library_Name(library_name);
		d_library_edit_dialog->Set_Library_Type(type);

		if(QDialog::Accepted == d_library_edit_dialog->exec())
		{
			data = d_library_edit_dialog->Data();
			
			target_name = data.target_name;
			library_name = data.library_name;
			
			if(!Library_Exists(target_name,library_name))
			{
				items[0]->setText(0,target_name);
				items[0]->setText(2,library_name);
				
				switch(data.type)
				{
					case TTypes::TYPE_DYLIB:
						items[0]->setText(1,QStringLiteral("Dylib"));
						break;
						
					case TTypes::TYPE_PLUGIN:
						items[0]->setText(1,QStringLiteral("Plugin"));
						break;
				}
				
				this->Update_Library_Data();
			}
		}
	}
	else if(items.size() > 1)
	{
		d_library_edit_dialog->Reset(TTypes::EDITOR_EDIT_LIBRARY_VERSION,this->Target_Names());
		
		target_name = QString();
		library_name = QStringLiteral("<library_name>.so.1.2.3");
		type = TTypes::TYPE_DYLIB;

		d_library_edit_dialog->Set_Target_Name(target_name);
		d_library_edit_dialog->Set_Library_Name(library_name);
		d_library_edit_dialog->Set_Library_Type(type);

		if(QDialog::Accepted == d_library_edit_dialog->exec())
		{
			data = d_library_edit_dialog->Data();
			
			library_match = library_version_expression.match(data.library_name);
			if(library_match.hasMatch())
			{
				matched_libraries_list = library_match.capturedTexts();

				if(matched_libraries_list.size() == 4)
				{
					library_versions[0] = matched_libraries_list[1].toInt();
					library_versions[1] = matched_libraries_list[2].toInt();
					library_versions[2] = matched_libraries_list[3].toInt();
					
					
					for(iter = items.begin();iter != items.end();++iter)
					{
						text = (*iter)->text(2);
					
						library_match = library_name_expression.match(text);
						if(library_match.hasMatch())
						{
							matched_libraries_list = library_match.capturedTexts();

							if(matched_libraries_list.size() == 5)
							{
								library_name = QString("%1.%2.%3.%4").arg(matched_libraries_list[1]).arg(library_versions[0]).arg(library_versions[1]).arg(library_versions[2]);;
								this->Library_Rename(text,library_name);

							}
						}
					
					}
				}
			}

			this->Update_Library_Tree();
		}
	}
}

void TLibrariesWidget::Initialize(void)
{
	TLibrariesWidget::TTarget			target;
	TLibrariesWidget::TLibrary			library;
	
	d_targets.clear();
	d_libraries.clear();
	
	target.target_name = QStringLiteral("lib");
	target.checked = true;
	d_targets.push_back(target);
	
	target.target_name = QStringLiteral("platforms");
	target.checked = true;
	d_targets.push_back(target);
	
	target.target_name = QStringLiteral("xcbglintegrations");
	target.checked = true;
	d_targets.push_back(target);

	target.target_name = QStringLiteral("imageformats");
	target.checked = false;
	d_targets.push_back(target);
	
	target.target_name = QStringLiteral("platformthemes");
	target.checked = false;
	d_targets.push_back(target);
	
	target.target_name = QStringLiteral("printsupport");
	target.checked = false;
	d_targets.push_back(target);
	
	target.target_name = QStringLiteral("sqldrivers");
	target.checked = false;
	d_targets.push_back(target);
	
	target.target_name = QStringLiteral("styles");
	target.checked = false;
	d_targets.push_back(target);
	
	target.target_name = QStringLiteral("bearer");
	target.checked = false;
	d_targets.push_back(target);
	
	target.target_name = QStringLiteral("egldeviceintegrations");
	target.checked = false;
	d_targets.push_back(target);
	
	target.target_name = QStringLiteral("generic");
	target.checked = false;
	d_targets.push_back(target);
	
	target.target_name = QStringLiteral("platforminputcontexts");
	target.checked = false;
	d_targets.push_back(target);
		
	library.target_name = QStringLiteral("lib");
	library.type = TTypes::TYPE_DYLIB;
	library.library_name = QStringLiteral("libQt5Widgets.so.5.15.2");
	library.checked = true;
	d_libraries.push_back(library);
		
	library.target_name = QStringLiteral("lib");
	library.type = TTypes::TYPE_DYLIB;
	library.library_name = QStringLiteral("libQt5Gui.so.5.15.2");
	library.checked = true;
	d_libraries.push_back(library);
		
	library.target_name = QStringLiteral("lib");
	library.type = TTypes::TYPE_DYLIB;
	library.library_name = QStringLiteral("libQt5Core.so.5.15.2");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("lib");
	library.type = TTypes::TYPE_DYLIB;
	library.library_name = QStringLiteral("libQt5DBus.so.5.15.2");
	library.checked = true;
	d_libraries.push_back(library);

	library.target_name = QStringLiteral("lib");
	library.type = TTypes::TYPE_DYLIB;
	library.library_name = QStringLiteral("libQt5XcbQpa.so.5.15.2");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("lib");
	library.type = TTypes::TYPE_DYLIB;
	library.library_name = QStringLiteral("libQt5EglFSDeviceIntegration.so.5.15.2");
	library.checked = true;
	d_libraries.push_back(library);

	library.target_name = QStringLiteral("lib");
	library.type = TTypes::TYPE_DYLIB;
	library.library_name = QStringLiteral("libQt5Concurrent.so.5.15.2");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("lib");
	library.type = TTypes::TYPE_DYLIB;
	library.library_name = QStringLiteral("libQt5EglFsKmsSupport.so.5.15.2");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("lib");
	library.type = TTypes::TYPE_DYLIB;
	library.library_name = QStringLiteral("libQt5Network.so.5.15.2");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("lib");
	library.type = TTypes::TYPE_DYLIB;
	library.library_name = QStringLiteral("libQt5OpenGL.so.5.15.2");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("lib");
	library.type = TTypes::TYPE_DYLIB;
	library.library_name = QStringLiteral("libQt5PrintSupport.so.5.15.2");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("lib");
	library.type = TTypes::TYPE_DYLIB;
	library.library_name = QStringLiteral("libQt5SerialPort.so.5.15.2");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("lib");
	library.type = TTypes::TYPE_DYLIB;
	library.library_name = QStringLiteral("libQt5Sql.so.5.15.2");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("lib");
	library.type = TTypes::TYPE_DYLIB;
	library.library_name = QStringLiteral("libQt5Xml.so.5.15.2");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("platforms");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libqxcb.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("platforms");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libqeglfs.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("platforms");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libqlinuxfb.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("platforms");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libqminimal.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("platforms");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libqminimalegl.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("platforms");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libqoffscreen.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("platforms");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libqvnc.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("imageformats");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libqgif.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("imageformats");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libqico.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("imageformats");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libqjpeg.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("sqldrivers");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libqsqlite.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("xcbglintegrations");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libqxcb-egl-integration.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("xcbglintegrations");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libqxcb-glx-integration.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("bearer");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libqconnmanbearer.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("bearer");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libqgenericbearer.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("bearer");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libqnmbearer.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("egldeviceintegrations");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libqeglfs-emu-integration.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("egldeviceintegrations");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libqeglfs-kms-egldevice-integration.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("egldeviceintegrations");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libqeglfs-kms-integration.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("egldeviceintegrations");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libqeglfs-x11-integration.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("generic");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libqevdevkeyboardplugin.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("generic");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libqevdevmouseplugin.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("generic");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libqevdevtabletplugin.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("generic");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libqevdevtouchplugin.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("generic");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libqtuiotouchplugin.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("platforminputcontexts");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libcomposeplatforminputcontextplugin.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("platforminputcontexts");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libibusplatforminputcontextplugin.so");
	library.checked = true;
	d_libraries.push_back(library);
	
	library.target_name = QStringLiteral("platformthemes");
	library.type = TTypes::TYPE_PLUGIN;
	library.library_name = QStringLiteral("libqxdgdesktopportal.so");
	library.checked = true;
	d_libraries.push_back(library);

}

bool TLibrariesWidget::Target_Exists(
	const QString 						&name) const
{
	std::vector<TLibrariesWidget::TTarget>::const_iterator iter;
	
	for(iter = d_targets.begin();iter != d_targets.end();++iter)
	{
		if((*iter).target_name == name)
		{
			return true;
		}
	}
	
	return false;
}

bool TLibrariesWidget::Library_Exists(
	const QString						&target_name,
	const QString 						&library_name) const
{
	std::vector<TLibrariesWidget::TLibrary>::const_iterator iter;
	
	for(iter = d_libraries.begin();iter != d_libraries.end();++iter)
	{
		if((*iter).target_name == target_name && (*iter).library_name == library_name)
		{
			return true;
		}
	}
	
	return false;
}

bool TLibrariesWidget::Has_Libraries(
	const QString 						&target_name) const
{
	std::vector<TLibrariesWidget::TLibrary>::const_iterator iter;
	
	for(iter = d_libraries.begin();iter != d_libraries.end();++iter)
	{
		if((*iter).target_name == target_name && (*iter).checked == true)
		{
			return true;
		}
	}
	
	return false;
}

QStringList TLibrariesWidget::Target_Names(void) const
{
	std::vector<TLibrariesWidget::TTarget>::const_iterator iter;
	QStringList							names;
	
	for(iter = d_targets.begin();iter != d_targets.end();++iter)
	{
		names.push_back((*iter).target_name);
	}
	
	return names;
}

void TLibrariesWidget::Target_Remove(
	const QString 						&name)
{
	std::vector<TLibrariesWidget::TTarget>::iterator iter;
	
	for(iter = d_targets.begin();iter != d_targets.end();++iter)
	{
		if((*iter).target_name == name)
		{
			d_targets.erase(iter);
			return;
		}
	}
}

void TLibrariesWidget::Library_Remove(
	const QString						&target_name,
	const QString 						&library_name)
{
	std::vector<TLibrariesWidget::TLibrary>::iterator iter;
	
	for(iter = d_libraries.begin();iter != d_libraries.end();++iter)
	{
		if((*iter).target_name == target_name && (*iter).library_name == library_name)
		{
			d_libraries.erase(iter);
			return;
		}
	}
}

void TLibrariesWidget::Library_Rename(
	const QString 						&old_name,
	const QString 						&new_name)
{
	std::vector<TLibrariesWidget::TLibrary>::iterator iter;
	
	for(iter = d_libraries.begin();iter != d_libraries.end();++iter)
	{
		if((*iter).library_name == old_name)
		{
			(*iter).library_name = new_name;
		}
	}
}

void TLibrariesWidget::Update_Target_Tree(void)
{
	QTreeWidgetItem						*item;
	std::vector<TLibrariesWidget::TTarget>::const_iterator	iter;
	
	d_target_tree->clear();
	
	for(iter = d_targets.begin();iter != d_targets.end();++iter)
	{
		item = new QTreeWidgetItem(d_target_tree);
		
		(*iter).checked ? item->setCheckState(0,Qt::Checked) : item->setCheckState(0,Qt::Unchecked);

		item->setText(0,(*iter).target_name);
	}
	
	d_target_tree->sortByColumn(0,Qt::AscendingOrder);
}

void TLibrariesWidget::Update_Target_Data(void)
{
	QTreeWidgetItem						*item;
	TLibrariesWidget::TTarget			target;
	int									count;
	int									cntr;
	
	d_targets.clear();
	count = d_target_tree->topLevelItemCount();
	
	for(cntr = 0;cntr < count;++cntr)
	{
		item = d_target_tree->topLevelItem(cntr);
		
		target.target_name = item->text(0);
		target.checked = (item->checkState(0) == Qt::Checked);
		
		d_targets.push_back(target);
	}
}

void TLibrariesWidget::Update_Library_Tree(void)
{
	std::vector<TLibrariesWidget::TLibrary>::const_iterator	iter;
	QTreeWidgetItem						*item;
	QStringList							active_targets_list;
//	QString								plugin_names;
	
	d_libraries_tree->clear();
		
	for(iter = d_libraries.begin();iter != d_libraries.end();++iter)
	{
		item = new QTreeWidgetItem(d_libraries_tree);
				
		(*iter).checked ? item->setCheckState(0,Qt::Checked) : item->setCheckState(0,Qt::Unchecked);

		item->setText(0,(*iter).target_name);
		
		switch((*iter).type)
		{
			case TTypes::TYPE_DYLIB:
				item->setText(1,QStringLiteral("Dylib"));
				break;
				
			case TTypes::TYPE_PLUGIN:
				item->setText(1,QStringLiteral("Plugin"));
				break;
		}
		
		item->setText(2,(*iter).library_name);
	}
		
	d_libraries_tree->sortByColumn(0,Qt::AscendingOrder);
	
	d_libraries_tree->resizeColumnToContents(0);
	d_libraries_tree->resizeColumnToContents(1);
	d_libraries_tree->resizeColumnToContents(2);
	
	active_targets_list = this->Active_Targets();
	this->Update_Library_Tree_Status(active_targets_list);
}

void TLibrariesWidget::Update_Library_Data(void)
{
	QTreeWidgetItem						*item;
	TLibrariesWidget::TLibrary			library;
	QString								text;
	int									count;
	int									cntr;
		
	d_libraries.clear();
	count = d_libraries_tree->topLevelItemCount();
	
	for(cntr = 0;cntr < count;++cntr)
	{
		item = d_libraries_tree->topLevelItem(cntr);
		
		library.target_name = item->text(0);
		
		text = item->text(1);
		if(text == QStringLiteral("Dylib"))
		{
			library.type = TTypes::TYPE_DYLIB;
		}
		else	// plugin
		{
			library.type = TTypes::TYPE_PLUGIN;
		}
		
		library.library_name = item->text(2);

		library.checked = (item->checkState(0) == Qt::Checked);
		
		d_libraries.push_back(library);
	}
}

void TLibrariesWidget::Update_Library_Tree_Status(
	const QStringList 					&active_targets)
{
	QTreeWidgetItem						*item;
	int									count;
	int									cntr;
	
	count = d_libraries_tree->topLevelItemCount();
	
	for(cntr = 0;cntr < count;++cntr)
	{
		item = d_libraries_tree->topLevelItem(cntr);
		
		if(active_targets.contains(item->text(0)))
		{
			item->setDisabled(false);
		}
		else
		{
			item->setDisabled(true);
		}
	}
}

void TLibrariesWidget::Load_V1(
	TXmlFile 							* const xml_file)
{
	TLibrariesWidget::TTarget			target;
	TLibrariesWidget::TLibrary			library;
	QDomElement							e0;
	QDomElement							e1;
	QDomElement							e2;
	QString								text;
	
	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Targets"));
	if(!e0.isNull())
	{
		e1 = xml_file->Get_Node(&e0,QStringLiteral("Target"));
		while(!e1.isNull())
		{
			e2 = xml_file->Get_Node(&e1,QStringLiteral("Name"));
			target.target_name = xml_file->Read_Text_Node(&e2);
		
			e2 = xml_file->Get_Node(&e1,QStringLiteral("Checked"));
			text = xml_file->Read_Text_Node(&e2);
			
			(text == QStringLiteral("Yes")) ? target.checked = true:target.checked = false;

			d_targets.push_back(target);

			e1 = xml_file->Get_Sibling_Node(&e1,QStringLiteral("Target"),true);
		}
	}
	
	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Libraries"));
	if(!e0.isNull())
	{
		e1 = xml_file->Get_Node(&e0,QStringLiteral("Library"));
		while(!e1.isNull())
		{
			e2 = xml_file->Get_Node(&e1,QStringLiteral("Target_Name"));
			library.target_name = xml_file->Read_Text_Node(&e2);
			
			e2 = xml_file->Get_Node(&e1,QStringLiteral("Name"));
			library.library_name = xml_file->Read_Text_Node(&e2);
			
			e2 = xml_file->Get_Node(&e1,QStringLiteral("Type"));
			text = xml_file->Read_Text_Node(&e2);
			
			(text == QStringLiteral("DYLIB")) ? library.type = TTypes::TYPE_DYLIB:library.type = TTypes::TYPE_PLUGIN;

			e2 = xml_file->Get_Node(&e1,QStringLiteral("Checked"));
			text = xml_file->Read_Text_Node(&e2);
			
			(text == QStringLiteral("Yes")) ? library.checked = true:library.checked = false;

			d_libraries.push_back(library);

			e1 = xml_file->Get_Sibling_Node(&e1,QStringLiteral("Library"),true);
		}
	}
}



