/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QComboBox>
#include <QFrame>
#include <QGridLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QLineEdit>
#include <QPushButton>
#include <QSpacerItem>
#include <QStringList>
#include <QVariant>

#include "libraryeditdialog.h"

TLibraryEditDialog::TLibraryEditDialog(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QDialog(const_cast<QWidget*>(parent),flags)
{
	QFrame								*separator_hline;
	QGridLayout							*dialog_layout;
	QHBoxLayout							*button_hlayout;
	QSpacerItem							*button_hspacer;
	QSpacerItem							*dialog_vspacer;

	this->resize(409,226);

	dialog_layout = new QGridLayout(this);

	d_target_name_label = new QLabel(this);
	d_target_name_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	dialog_layout->addWidget(d_target_name_label,0,0,1,1);

	d_target_name_edit = new QLineEdit(this);
	dialog_layout->addWidget(d_target_name_edit,0,1,1,1);

	d_library_name_label = new QLabel(this);
	d_library_name_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	dialog_layout->addWidget(d_library_name_label,1,0,1,1);

	d_library_name_edit = new QLineEdit(this);
	dialog_layout->addWidget(d_library_name_edit,1,1,1,1);

	d_library_type_label = new QLabel(this);
	d_library_type_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	dialog_layout->addWidget(d_library_type_label,2,0,1,1);

	d_library_type_combo = new QComboBox(this);
	dialog_layout->addWidget(d_library_type_combo,2,1,1,1);

	separator_hline = new QFrame(this);
	separator_hline->setFrameShape(QFrame::HLine);
	separator_hline->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(separator_hline,3,0,1,2);

	dialog_vspacer = new QSpacerItem(0,0,QSizePolicy::Minimum,QSizePolicy::Expanding);
	dialog_layout->addItem(dialog_vspacer,4,0,1,2);
	
	button_hlayout = new QHBoxLayout();

	button_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	button_hlayout->addItem(button_hspacer);

	d_cancel_button = new QPushButton(this);
	d_cancel_button->setAutoDefault(false);
	button_hlayout->addWidget(d_cancel_button);

	d_accept_button = new QPushButton(this);
	d_accept_button->setAutoDefault(false);
	button_hlayout->addWidget(d_accept_button);
	dialog_layout->addLayout(button_hlayout,5,0,1,2);

	// defaults
	d_accept_button->setEnabled(false);
	d_editor_type = TTypes::EDITOR_ADD_TARGET;
	
	d_library_type_combo->addItem(QStringLiteral("Library"),static_cast<int>(TTypes::TYPE_DYLIB));
	d_library_type_combo->addItem(QStringLiteral("Plugin"),static_cast<int>(TTypes::TYPE_PLUGIN));
	
	this->setWindowTitle(QStringLiteral("Library Editor"));
	
	d_target_name_label->setText(QStringLiteral("Target Name:"));
	d_library_name_label->setText(QStringLiteral("Library Name:"));
	d_library_type_label->setText(QStringLiteral("Library Type:"));
	d_accept_button->setText(QStringLiteral("Accept"));
	d_cancel_button->setText(QStringLiteral("Cancel"));
	
	connect(d_target_name_edit,&QLineEdit::textEdited,this,&TLibraryEditDialog::Validate_Entries);
	connect(d_library_name_edit,&QLineEdit::textEdited,this,&TLibraryEditDialog::Validate_Entries);

	connect(d_accept_button,&QPushButton::clicked,this,&TLibraryEditDialog::accept);
	connect(d_cancel_button,&QPushButton::clicked,this,&TLibraryEditDialog::reject);
}

TLibraryEditDialog::~TLibraryEditDialog(void)
{
}

TLibraryEditDialog::TEditData TLibraryEditDialog::Data(void) const
{
	TLibraryEditDialog::TEditData		data;
	
	data.target_name = d_target_name_edit->text();

	switch(d_editor_type)
	{
		case TTypes::EDITOR_ADD_TARGET:
			data.type = TTypes::TYPE_DYLIB;
			break;
			
		case TTypes::EDITOR_ADD_LIBRARY:
		case TTypes::EDITOR_EDIT_LIBRARY:
		case TTypes::EDITOR_EDIT_LIBRARY_VERSION:
			data.library_name = d_library_name_edit->text();
			data.type = static_cast<TTypes::TLibraryType>(d_library_type_combo->currentData(Qt::UserRole).toInt());
			break;
	}
	
	return data;
}

void TLibraryEditDialog::Reset(
	const TTypes::TEditorType 			type,
	const QStringList					&target_names)
{
	d_editor_type = type;
	d_existing_target_names = target_names;
	
	switch(d_editor_type)
	{
		case TTypes::EDITOR_ADD_TARGET:
			this->setWindowTitle(QStringLiteral("Add Target"));
			d_target_name_edit->setEnabled(true);
			d_library_name_edit->setEnabled(false);
			d_library_type_combo->setEnabled(false);
			
			d_target_name_label->setEnabled(true);
			d_library_name_label->setEnabled(false);
			d_library_type_label->setEnabled(false);
			break;
			
		case TTypes::EDITOR_ADD_LIBRARY:
			this->setWindowTitle(QStringLiteral("Add Library"));
			d_target_name_edit->setEnabled(true);
			d_library_name_edit->setEnabled(true);
			d_library_type_combo->setEnabled(true);
			
			d_target_name_label->setEnabled(true);
			d_library_name_label->setEnabled(true);
			d_library_type_label->setEnabled(true);
			break;
			
		case TTypes::EDITOR_EDIT_LIBRARY:
			this->setWindowTitle(QStringLiteral("Edit Library"));
			d_target_name_edit->setEnabled(true);
			d_library_name_edit->setEnabled(true);
			d_library_type_combo->setEnabled(true);
			
			d_target_name_label->setEnabled(true);
			d_library_name_label->setEnabled(true);
			d_library_type_label->setEnabled(true);
			break;
			
		case TTypes::EDITOR_EDIT_LIBRARY_VERSION:
			this->setWindowTitle(QStringLiteral("Edit Library Version"));
			d_target_name_edit->setEnabled(false);
			d_library_name_edit->setEnabled(true);
			d_library_type_combo->setEnabled(false);
			
			d_target_name_label->setEnabled(false);
			d_library_name_label->setEnabled(true);
			d_library_type_label->setEnabled(false);
			break;
	}	
}

void TLibraryEditDialog::Set_Target_Name(
	const QString 						&name)
{
	d_target_name_edit->setText(name);
}

void TLibraryEditDialog::Set_Library_Name(
	const QString 						&name)
{
	d_library_name_edit->setText(name);
}

void TLibraryEditDialog::Set_Library_Type(
	const TTypes::TLibraryType 			type)
{
	switch(type)
	{
		case TTypes::TYPE_DYLIB:
			d_library_type_combo->setCurrentIndex(0);
			break;
			
		case TTypes::TYPE_PLUGIN:
			d_library_type_combo->setCurrentIndex(1);
			break;
	}
}

void TLibraryEditDialog::Validate_Entries(void)
{
	QString								target_name;
	QString								library_name;
	
	switch(d_editor_type)
	{
		case TTypes::EDITOR_ADD_TARGET:
			target_name = d_target_name_edit->text();
			d_accept_button->setEnabled(target_name.length() && (d_existing_target_names.contains(target_name,Qt::CaseInsensitive) == false));
			break;
			
		case TTypes::EDITOR_ADD_LIBRARY:
		case TTypes::EDITOR_EDIT_LIBRARY:
			target_name = d_target_name_edit->text();
			library_name = d_library_name_edit->text();
			d_accept_button->setEnabled(target_name.length() &&
										library_name.contains(QStringLiteral(".so")) &&
										d_existing_target_names.contains(target_name,Qt::CaseInsensitive));
			break;
			
		case TTypes::EDITOR_EDIT_LIBRARY_VERSION:
			library_name = d_library_name_edit->text();
			d_accept_button->setEnabled(library_name.contains(QStringLiteral(".so")));
			break;
	}
}


