/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QAction>
#include <QApplication>
#include <QMenu>
#include <QMenuBar>
#include <QTabWidget>
#include <QVBoxLayout>
#include <QCloseEvent>
#include <QFileDialog>
#include <QToolBar>
#include <QRegularExpression>
#include <QRegularExpressionMatch>
#include <QTextStream>
#include <QSettings>
#include <QScreen>
#include <QTimer>
#include <QFile>
#include <QDir>
#include <QPoint>
#include <QSize>
#include <QRect>

#include "../../core/messagebox.h"

#include "variableswidget.h"
#include "expressionwidget.h"
#include "librarieswidget.h"
#include "requireswidget.h"
#include "changelogwidget.h"
#include "optionswidget.h"

#include "rpmspecseditor.h"

TRPMSpecsEditor::TRPMSpecsEditor(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QMainWindow(const_cast<QWidget*>(parent),flags)
{
	QMenu								*file_menu;
	QMenu								*help_menu;
	QMenuBar							*menu_bar;
	QToolBar							*tool_bar;
	QVBoxLayout							*widget_vlayout;
	QWidget								*central_widget;
	QString								shared_libs_path;
	QDir								settings_dir;
	QPoint								position;
	QSize								size;
	QRect								desktop_rect;
	QRect								window_rect;

	// storage location
	// Win32 in directory of application (even though it doesn't make sense to run this on Windows).
	// *nix in home folder under hidden .name folder
	
#ifdef Q_OS_WIN
	settings_dir.setPath(qApp->applicationDirPath());
#else
	settings_dir = QDir::home();
#endif

	if(!settings_dir.exists(QStringLiteral(".rpmspecseditor")))
	{
		settings_dir.mkdir(QStringLiteral(".rpmspecseditor"));
	}
	
	settings_dir.cd(QStringLiteral(".rpmspecseditor"));
	
	d_data_path = settings_dir.absolutePath();
	
	if(d_data_path.endsWith('/'))
	{
		d_data_path.chop(1);
	}
	
	d_settings = new QSettings(d_data_path + QStringLiteral("/settings.ini"),QSettings::IniFormat,this);
	
	position = d_settings->value(QStringLiteral("Mainwindow_Position"), QPoint(30,30)).toPoint();
	size = d_settings->value(QStringLiteral("Mainwindow_Size"), QSize(1000,800)).toSize();
	
	desktop_rect = QGuiApplication::primaryScreen()->availableGeometry();
	
	window_rect.moveTopLeft(position);
	window_rect.setSize(size);
	
	window_rect = window_rect & desktop_rect;
	
	if(window_rect.isValid() && window_rect.width() > 500 && window_rect.height() > 500)
	{
		this->resize(window_rect.size());
		this->move(window_rect.topLeft());
	}
	else
	{
		this->resize(1000,800);
		this->move(QPoint(30,30));
	}
	
	d_working_path = d_settings->value(QStringLiteral("Working_Path"),QDir::homePath()).toString();
	shared_libs_path = d_settings->value(QStringLiteral("Shared_Libs_Path"),QDir::homePath()).toString();
	
	d_create_action = new QAction(this);
	d_create_action->setIcon(QIcon(QStringLiteral(":/actions/file_create_x32.png")));

	d_open_action = new QAction(this);
	d_open_action->setIcon(QIcon(QStringLiteral(":/actions/file_open_x32.png")));

	d_save_action = new QAction(this);
	d_save_action->setIcon(QIcon(QStringLiteral(":/actions/file_save_x32.png")));

	d_about_action = new QAction(this);
	d_about_action->setIcon(QIcon(QStringLiteral(":/actions/help_about_x32.png")));

	d_quit_action = new QAction(this);

	central_widget = new QWidget(this);
	this->setCentralWidget(central_widget);

	widget_vlayout = new QVBoxLayout(central_widget);

	d_widget_tab = new QTabWidget(central_widget);

	d_variables_widget = new TVariablesWidget();
	d_widget_tab->addTab(d_variables_widget,QStringLiteral("Variables"));
	
	d_expressions_widget = new TExpressionsWidget();
	d_widget_tab->addTab(d_expressions_widget,QStringLiteral("Expressions"));

	d_libraries_widget = new TLibrariesWidget();
	d_widget_tab->addTab(d_libraries_widget,QStringLiteral("Libraries"));

	d_requires_widget = new TRequiresWidget();
	d_widget_tab->addTab(d_requires_widget,QStringLiteral("Requires"));

	d_changelog_widget = new TChangeLogWidget();
	d_widget_tab->addTab(d_changelog_widget,QStringLiteral("Change Log"));

	d_options_widget = new TOptionsWidget();
	d_widget_tab->addTab(d_options_widget,QStringLiteral("Options"));

	widget_vlayout->addWidget(d_widget_tab);
	
	// defaults
	d_msg_box = new TMessageBox(this);
	
	d_write_specfile_timer = new QTimer(this);
	d_write_specfile_timer->setSingleShot(true);
	d_write_specfile_timer->setInterval(100);

	d_variables_widget->Load_Defaults(d_data_path);
	d_expressions_widget->Load_Defaults(d_data_path);
	d_libraries_widget->Load_Defaults(d_data_path);
	d_requires_widget->Load_Defaults(d_data_path);
	d_changelog_widget->Load_Defaults(d_data_path);
	d_options_widget->Load_Defaults(d_data_path);
	
	d_requires_widget->Set_Shared_Libs_Path(shared_libs_path);
	d_options_widget->Set_Shared_Libs_Path(shared_libs_path);

	menu_bar = new QMenuBar(this);
	this->setMenuBar(menu_bar);

	file_menu = new QMenu(menu_bar);
	menu_bar->addMenu(file_menu);
	
	file_menu->addAction(d_create_action);
	file_menu->addSeparator();
	file_menu->addAction(d_open_action);
	file_menu->addAction(d_save_action);
	file_menu->addSeparator();
	file_menu->addAction(d_quit_action);
	
	help_menu = new QMenu(menu_bar);
	menu_bar->addMenu(help_menu);
	
	help_menu->addAction(d_about_action);

	menu_bar->resize(800,22);
	menu_bar->addAction(file_menu->menuAction());
	menu_bar->addAction(help_menu->menuAction());

	tool_bar = new QToolBar(this);
	tool_bar->setIconSize(QSize(32,32));
	
	tool_bar->addAction(d_create_action);
	tool_bar->addAction(d_open_action);
	tool_bar->addAction(d_save_action);
	tool_bar->addSeparator();
	tool_bar->addAction(d_about_action);

	this->addToolBar(Qt::TopToolBarArea,tool_bar);

	this->setWindowTitle(QStringLiteral("RPMBuild Specs Editor"));
	
	d_create_action->setText(QStringLiteral("Create"));
	d_save_action->setText(QStringLiteral("Save"));
	d_open_action->setText(QStringLiteral("Open"));
	d_quit_action->setText(QStringLiteral("Quit"));
	d_about_action->setText(QStringLiteral("About"));
	file_menu->setTitle(QStringLiteral("File"));
	help_menu->setTitle(QStringLiteral("Help"));

	connect(d_create_action,&QAction::triggered,this,&TRPMSpecsEditor::File_Create_Spec_File);
	connect(d_save_action,&QAction::triggered,this,&TRPMSpecsEditor::File_Save_Spec_Data);
	connect(d_open_action,&QAction::triggered,this,&TRPMSpecsEditor::File_Open_Spec_Data);
	connect(d_about_action,&QAction::triggered,this,&TRPMSpecsEditor::Help_About);
	connect(d_quit_action,&QAction::triggered,this,&TRPMSpecsEditor::close);
	
	connect(d_write_specfile_timer,&QTimer::timeout,this,&TRPMSpecsEditor::Write_Spec_File);

	connect(d_requires_widget,&TRequiresWidget::Refresh,this,&TRPMSpecsEditor::Update_Requirements);
	connect(d_options_widget,&TOptionsWidget::Shared_Libs_Path_Changed,this,&TRPMSpecsEditor::Update_Option_Shared_Libs_Path);
}


TRPMSpecsEditor::~TRPMSpecsEditor(void)
{
}
	
void TRPMSpecsEditor::File_Create_Spec_File(void)
{
	QString								file_name;
	int									index;
	
	file_name = QFileDialog::getSaveFileName(this,
											 QStringLiteral("Create Spec File"),
											 d_working_path,
											 QStringLiteral("SPEC (*.spec)"));
	
	if(file_name.length())
	{
		if(!file_name.endsWith(QStringLiteral(".spec"),Qt::CaseInsensitive))
		{
			file_name += QStringLiteral(".spec");
		}
		
		index = file_name.lastIndexOf('/');
		
		if(!(index < 0))
		{
			d_working_path = file_name;
			d_working_path.truncate(index);
		}
		
		d_file_name = file_name;
		d_write_specfile_timer->start();
	}
}

void TRPMSpecsEditor::File_Save_Spec_Data(void)
{
	QString								file_name;
	int									index;
	
	file_name = QFileDialog::getSaveFileName(this,
											 QStringLiteral("Save Spec Data"),
											 d_working_path,
											 QStringLiteral("SPEC (*.spec_data)"));
	
	if(file_name.length())
	{
		if(!file_name.endsWith(QStringLiteral(".spec_data"),Qt::CaseInsensitive))
		{
			file_name += QStringLiteral(".spec_data");
		}
		
		index = file_name.lastIndexOf('/');
		
		if(!(index < 0))
		{
			d_working_path = file_name;
			d_working_path.truncate(index);
		}
		
		this->Write_Spec_Data_File(file_name);
	}
}

void TRPMSpecsEditor::File_Open_Spec_Data(void)
{
	QString								file_name;
	int									index;
	
	file_name = QFileDialog::getOpenFileName(this,
											 QStringLiteral("Open Spec Data"),
											 d_working_path,
											 QStringLiteral("SPEC (*.spec_data)"));
	
	if(file_name.length())
	{

		index = file_name.lastIndexOf('/');
		
		if(!(index < 0))
		{
			d_working_path = file_name;
			d_working_path.truncate(index);
		}
		
		if(!this->Read_Spec_Data_File(file_name))
		{
			d_last_error = QStringLiteral("ERR  Cannot read Spec_Data file.");
		}
	}
}

void TRPMSpecsEditor::Help_About(void)
{
	d_msg_box->setText(QStringLiteral("RPMBuild Specs Editor"));
	d_msg_box->setInformativeText(QStringLiteral("RPMBuild Specs Editor - Version 1.2"));
	d_msg_box->setDetailedText(QString());
	d_msg_box->setStandardButtons(QMessageBox::Ok);
	d_msg_box->setDefaultButton(QMessageBox::Ok);
	d_msg_box->setIcon(QMessageBox::Information);
	
	d_msg_box->exec();
}

void TRPMSpecsEditor::Update_Requirements(void) const
{
	QStringList							active_targets;
	
	active_targets = d_libraries_widget->Active_Targets(true);
	
	d_requires_widget->Refresh_Required_Packages(active_targets);
}

void TRPMSpecsEditor::Update_Option_Shared_Libs_Path(
	const QString 						&path)
{
	d_requires_widget->Set_Shared_Libs_Path(path);
}

void TRPMSpecsEditor::closeEvent(
	QCloseEvent 						*event)
{
	d_variables_widget->Save_Defaults(d_data_path);
	d_expressions_widget->Save_Defaults(d_data_path);
	d_libraries_widget->Save_Defaults(d_data_path);
	d_requires_widget->Save_Defaults(d_data_path);
	d_changelog_widget->Save_Defaults(d_data_path);
	d_options_widget->Save_Defaults(d_data_path);

	d_settings->setValue(QStringLiteral("Mainwindow_Position"), this->pos());
	d_settings->setValue(QStringLiteral("Mainwindow_Size"), this->size());
	d_settings->setValue(QStringLiteral("Working_Path"),d_working_path);
	d_settings->setValue(QStringLiteral("Shared_Libs_Path"),d_requires_widget->Shared_Libs_Path());

	event->accept();
}

void TRPMSpecsEditor::Write_Spec_File(void) const
{
	TVariablesWidget::TVariable			variable;
	TExpressionsWidget::TLink			link;
	QFile								file;
	QRegularExpression					library_expression(QStringLiteral("^([a-z,A-Z,0-9]{1,64})[.so]{4}([0-9]{1,3})[.]([0-9]{1,3})[.]([0-9]{1,3})$"));
	QRegularExpressionMatch				library_match;
	QStringList::const_iterator			str_iter;
	QStringList::const_iterator			list_iter;
	QStringList							active_targets;
	QStringList							matched_libraries_list;
	QStringList							missing_qt_libraries_list;
	QStringList							available_libraries_list;
	QStringList							required_libraries_list;
	QStringList							additional_executables_list;
	QStringList							list;
	QString								build_path;
	QString								cmake_bin_path;
	QString								qmake_bin_path;
	QString								libary_path_name;
	QString								library_path;
	QString								desktop_menu_path;
	QString								target_install_path;
	QString								target_desktop_menu_path;
	QString								matched_library_name;
	QString								text;
	TTypes::TOptionMakeGenerator		generator_type;
	TTypes::TOptionRequies				requires_type;
	int									count;
	int									cntr;
	int									index;
	int									library_versions[3];
	static const int					MIN_COLUMN_WIDTH(40);
	
	d_variables_widget->Update_Variables();
	generator_type = d_options_widget->Generator_Type();
	requires_type = d_options_widget->Requires_Type();

	file.setFileName(d_file_name);
	
	if(file.open(QIODevice::WriteOnly | QIODevice::Text))
	{
		build_path = d_expressions_widget->Build_Path();
		cmake_bin_path = d_expressions_widget->CMake_Bin_Path();
		qmake_bin_path = d_expressions_widget->QMake_Bin_Path();
		library_path = d_expressions_widget->Library_Path();
		desktop_menu_path = d_expressions_widget->Desktop_Menu_Path();
		target_install_path = d_expressions_widget->Target_Install_Path();
		target_desktop_menu_path = d_expressions_widget->Target_Desktop_Menu_Path();
	
		text = d_expressions_widget->Target_Additional_Executables().simplified();
		
		if(text.length())
		{
			additional_executables_list = text.split(' ');
		}
		else
		{
			additional_executables_list.clear();
		}
	
		switch(requires_type)
		{
			case TTypes::OPTION_AUTO_UPDATE_REQUIRES:
				this->Update_Requirements();
				break;
				
			case TTypes::OPTION_MANUAL_UPDATE_REQUIRES:
			case TTypes::OPTION_IGNORE_REQUIRES:
				break;
		}
		
		required_libraries_list = d_requires_widget->Required_Qt_Libraries();
		matched_libraries_list = d_libraries_widget->Enabled_Libraries(TTypes::TYPE_DYLIB);
		
		available_libraries_list.clear();
		for(str_iter = matched_libraries_list.begin();str_iter != matched_libraries_list.end();++str_iter)
		{
			text = (*str_iter);
			index = text.indexOf(QStringLiteral(".so."));
			
			if(!(index < 0))
			{
				text.truncate(index + 3);	// keep extension .so
				available_libraries_list.push_back(text);
			}
		}
		
		available_libraries_list.removeDuplicates();
		
		missing_qt_libraries_list.clear();
		for(str_iter = required_libraries_list.begin();str_iter != required_libraries_list.end();++str_iter)
		{
			if(!available_libraries_list.contains((*str_iter)))
			{
				missing_qt_libraries_list.push_back((*str_iter));
			}
		}
		
		if(missing_qt_libraries_list.size())
		{
			text = QStringLiteral("Library files that may need to be enabled:\n");
			
			for(str_iter = missing_qt_libraries_list.begin();str_iter != missing_qt_libraries_list.end();++str_iter)
			{
				text += QString("%1\n").arg((*str_iter));
			}
			
			d_msg_box->setText(QStringLiteral("Qt Libraries Missing"));
			d_msg_box->setInformativeText(QStringLiteral("One or more needed Qt libary files may not be installed.  Do you want to continue anyway?"));
			d_msg_box->setDetailedText(text);
			d_msg_box->setStandardButtons(QMessageBox::Yes | QMessageBox::No);
			d_msg_box->setDefaultButton(QMessageBox::Yes);
			d_msg_box->setIcon(QMessageBox::Warning);
			
			if(QMessageBox::No == d_msg_box->exec())
			{
				return;
			}
		}

		active_targets = d_libraries_widget->Active_Targets(true);

		count = d_variables_widget->Count();
		
		for(cntr = 0;cntr < count;++cntr)
		{
			variable = d_variables_widget->Variable(cntr);
			
			text = variable.name;
			
			if(text.length() < MIN_COLUMN_WIDTH)
			{
				text.resize(MIN_COLUMN_WIDTH,QChar(' '));
			}
			
			file.write(text.toUtf8());
			
			text = variable.value;
			text.append('\n');
			
			file.write(text.toUtf8());
		}
		
		switch(requires_type)
		{
			case TTypes::OPTION_AUTO_UPDATE_REQUIRES:
			case TTypes::OPTION_MANUAL_UPDATE_REQUIRES:
				
				//requires data
				list = d_requires_widget->Required_Packages();
				
				for(str_iter = list.begin();str_iter != list.end();++str_iter)
				{
					text = QString("Requires:                               %1\n").arg((*str_iter));
					file.write(text.toUtf8());
				}

				break;
				
			case TTypes::OPTION_IGNORE_REQUIRES:
				break;
		}
		file.write("\n");

		
		file.write("%description\n");
		text = d_variables_widget->Description();
		file.write(text.toUtf8());
		file.write("\n\n");

		file.write("%prep\n");
		file.write("%setup -q\n\n");
		file.write("%build\n\n");
		
		count = d_expressions_widget->Symbolic_Links_Count();
		for(cntr = 0;cntr < count;++cntr)
		{
			link = d_expressions_widget->Symbolic_Link(cntr);
			
			if(link.checked)
			{
				text = QString("ln -s %1 %{_builddir}/%2\n").arg(link.path).arg(link.name);
				file.write(text.toUtf8());
			}
		}
		file.write("\n");

		switch(generator_type)
		{
			case TTypes::GENERATOR_QMAKE:
				text = QString("qmake PREFIX=%1\n").arg(build_path);
				break;
				
			case TTypes::GENERATOR_CMAKE:
				text = QString("cmake -DCMAKE_INSTALL_PREFIX:PATH=%1 .\n").arg(build_path);
				break;
		}
		
		file.write(text.toUtf8());
		file.write("make\n\n");

		file.write("%install\n\n");

		text = QString("mkdir -p %1\n").arg(build_path);
		file.write(text.toUtf8());
		
		for(str_iter = active_targets.begin();str_iter != active_targets.end();++str_iter)
		{
			text = QString("mkdir -p %1/%2\n").arg(build_path).arg(*str_iter);
			file.write(text.toUtf8());
		}

		text = QString("mkdir -p %{buildroot}%1/applications\n").arg(target_desktop_menu_path);
		file.write(text.toUtf8());
		
		text = QString("mkdir -p %{buildroot}%1/pixmaps\n").arg(target_desktop_menu_path);
		file.write(text.toUtf8());
		file.write("\n");

		file.write("make install\n\n");
		
		count = d_expressions_widget->Symbolic_Links_Count();
		for(cntr = 0;cntr < count;++cntr)
		{
			link = d_expressions_widget->Symbolic_Link(cntr);
			
			if(link.checked)
			{
				text = QString("rm %{_builddir}/%1\n").arg(link.name);
				file.write(text.toUtf8());
			}
		}
		file.write("\n");

		switch(generator_type)
		{
			case TTypes::GENERATOR_CMAKE:
				
				if(cmake_bin_path != build_path)
				{
					file.write("# moving the executable to the root build path\n");
					text = QString("mv %1/%{name} %2/%{name}\n").arg(cmake_bin_path).arg(build_path);
					file.write(text.toUtf8());
					
					text = QString("rmdir %1\n\n").arg(cmake_bin_path);
					file.write(text.toUtf8());
				}
				break;
				
			case TTypes::GENERATOR_QMAKE:
				
				if(qmake_bin_path != build_path)
				{
					file.write("# moving the executable to the root build path\n");
					text = QString("mv %1/%{name} %2/%{name}\n").arg(qmake_bin_path).arg(build_path);
					file.write(text.toUtf8());
					
					text = QString("rmdir %1\n\n").arg(qmake_bin_path);
					file.write(text.toUtf8());
				}
				break;
		}

		for(str_iter = active_targets.begin();str_iter != active_targets.end();++str_iter)
		{
			list = d_libraries_widget->Active_Libraries(*str_iter);
			
			for(list_iter = list.begin();list_iter != list.end();++list_iter)
			{
				library_match = library_expression.match(*list_iter);
				if(library_match.hasMatch())
				{
					libary_path_name = (*str_iter);
					break;
				}
			}
		}

		if(!libary_path_name.length())
		{
			d_last_error = QStringLiteral("ERR  Library path name not found.");
			
			d_msg_box->setText(QStringLiteral("Path Error"));
			d_msg_box->setInformativeText(QStringLiteral("Library path name not found"));
			d_msg_box->setDetailedText(QString());
			d_msg_box->setStandardButtons(QMessageBox::Ok);
			d_msg_box->setDefaultButton(QMessageBox::Ok);
			d_msg_box->setIcon(QMessageBox::Critical);
			
			d_msg_box->exec();
			return;
		}
		
		text = QString("patchelf --set-rpath '%1/%{name}/%2' %3/%{name}\n").arg(target_install_path).arg(libary_path_name).arg(build_path);
		file.write(text.toUtf8());
		
		for(str_iter = additional_executables_list.begin();str_iter != additional_executables_list.end();++str_iter)
		{
			text = QString("patchelf --set-rpath '%1/%{name}/%2' %3/%4\n").arg(target_install_path).arg(libary_path_name).arg(build_path).arg(*str_iter);
			file.write(text.toUtf8());
		}
		
		for(str_iter = active_targets.begin();str_iter != active_targets.end();++str_iter)
		{
			list = d_libraries_widget->Active_Libraries(*str_iter);
			file.write("\n");
			
			for(list_iter = list.begin();list_iter != list.end();++list_iter)
			{
				text = QString("install %1/%2/%3 %4/%5\n").arg(library_path).arg(*str_iter).arg(*list_iter).arg(build_path).arg(*str_iter);
				file.write(text.toUtf8());

				library_match = library_expression.match(*list_iter);
				if(!library_match.hasMatch())
				{
					text = QString("patchelf --set-rpath '%1/%{name}/%2' %3/%4/%5\n\n")
								.arg(target_install_path)
								.arg(libary_path_name)
								.arg(build_path)
								.arg(*str_iter)
								.arg(*list_iter);
					file.write(text.toUtf8());
				}
			}
		}

		text = QString("install %1/%{name}.desktop %{buildroot}%2/applications/%{name}.desktop\n").arg(desktop_menu_path).arg(target_desktop_menu_path);
		file.write(text.toUtf8());
		
		text = QString("install %1/%{name}.png %{buildroot}%2/pixmaps/%{name}.png\n\n").arg(desktop_menu_path).arg(target_desktop_menu_path);
		file.write(text.toUtf8());

		file.write("%post\n\n");

		text = QString("ln -sf %1/%{name}/%{name} %{_bindir}/%{name}\n").arg(target_install_path);
		file.write(text.toUtf8());
		
		for(str_iter = additional_executables_list.begin();str_iter != additional_executables_list.end();++str_iter)
		{
			text = QString("ln -sf %1/%{name}/%{name} %{_bindir}/%2\n").arg(target_install_path).arg(*str_iter);
			file.write(text.toUtf8());
		}

		file.write("\n");

		for(str_iter = active_targets.begin();str_iter != active_targets.end();++str_iter)
		{
			list = d_libraries_widget->Active_Libraries(*str_iter);
			
			for(list_iter = list.begin();list_iter != list.end();++list_iter)
			{
				library_match = library_expression.match(*list_iter);
				if(library_match.hasMatch())
				{
					matched_libraries_list = library_match.capturedTexts();

					if(matched_libraries_list.size() == 5)
					{
						library_versions[0] = matched_libraries_list[2].toInt();
						library_versions[1] = matched_libraries_list[3].toInt();
						library_versions[2] = matched_libraries_list[4].toInt();
						
						text = QString("ln -sf %1/%{name}/%2/%3 %4/%{name}/%5/%6.so.%7.%8\n").arg(target_install_path).arg(*str_iter).arg(*list_iter).arg(target_install_path).arg(*str_iter).arg(matched_libraries_list[1]).arg(library_versions[0]).arg(library_versions[1]);
						file.write(text.toUtf8());
						
						text = QString("ln -sf %1/%{name}/%2/%3 %4/%{name}/%5/%6.so.%7\n").arg(target_install_path).arg(*str_iter).arg(*list_iter).arg(target_install_path).arg(*str_iter).arg(matched_libraries_list[1]).arg(library_versions[0]);
						file.write(text.toUtf8());
						
						text = QString("ln -sf %1/%{name}/%2/%3 %4/%{name}/%5/%6.so\n").arg(target_install_path).arg(*str_iter).arg(*list_iter).arg(target_install_path).arg(*str_iter).arg(matched_libraries_list[1]);
						file.write(text.toUtf8());
					}
					file.write("\n");
				}
			}
		}
				
		file.write("%postun\n\n");
	
		text = QString("rm -f %{_bindir}/%{name}\n");
		file.write(text.toUtf8());
		
		for(str_iter = additional_executables_list.begin();str_iter != additional_executables_list.end();++str_iter)
		{
			text = QString("rm -f %{_bindir}/%1\n").arg(*str_iter);
			file.write(text.toUtf8());
		}

		for(str_iter = active_targets.begin();str_iter != active_targets.end();++str_iter)
		{
			text = QString("rm -r %1/%{name}/%2\n").arg(target_install_path).arg(*str_iter);
			file.write(text.toUtf8());
		}

		text = QString("rm -r %1/%{name}\n\n").arg(target_install_path);
		file.write(text.toUtf8());


		file.write("%files\n\n");
		file.write("%defattr(-,root,root)\n");
		
		text = QString("%1/%{name}/%{name}\n").arg(target_install_path);
		file.write(text.toUtf8());
		
		for(str_iter = additional_executables_list.begin();str_iter != additional_executables_list.end();++str_iter)
		{
			text = QString("%1/%{name}/%2\n").arg(target_install_path).arg(*str_iter);
			file.write(text.toUtf8());
		}

		for(str_iter = active_targets.begin();str_iter != active_targets.end();++str_iter)
		{
			text = QString("%1/%{name}/%2/*\n").arg(target_install_path).arg(*str_iter);
			file.write(text.toUtf8());
		}
	
		file.write("\n");

		file.write("%defattr(644,root,root)\n");

		text = QStringLiteral("%{_datadir}/applications/%{name}.desktop\n");
		file.write(text.toUtf8());

		text = QStringLiteral("%{_datadir}/pixmaps/%{name}.png\n\n");
		file.write(text.toUtf8());
		
		file.write("%changelog\n");
		text = d_changelog_widget->Changes();
		file.write(text.toUtf8());
		file.write("\n");
	
		file.close();
		
		d_last_error = QStringLiteral("Specs File Created");
		
		d_msg_box->setText(QStringLiteral("The specs file has been created."));
		d_msg_box->setInformativeText(QString());
		d_msg_box->setDetailedText(QString());
		d_msg_box->setStandardButtons(QMessageBox::Ok);
		d_msg_box->setDefaultButton(QMessageBox::Ok);
		d_msg_box->setIcon(QMessageBox::Information);
		
		d_msg_box->exec();
	}
	else
	{
		d_last_error = QStringLiteral("Specs File Creation Error.");
		
		d_msg_box->setText(QStringLiteral("The specs file cannot be opened for writing."));
		d_msg_box->setInformativeText(QString());
		d_msg_box->setDetailedText(QString());
		d_msg_box->setStandardButtons(QMessageBox::Ok);
		d_msg_box->setDefaultButton(QMessageBox::Ok);
		d_msg_box->setIcon(QMessageBox::Critical);
		
		d_msg_box->exec();
	}
}

bool TRPMSpecsEditor::Write_Spec_Data_File(
	const QString 						&file_name) const
{
	TVariablesWidget::TVariable			variable;
	TLibrariesWidget::TLibrary			library;
	TExpressionsWidget::TLink			link;
	QFile								file;
	QRegularExpression					library_expression(QStringLiteral("^([a-z,A-Z,0-9]{1,64}[.so]{3})"));
	QRegularExpressionMatch				library_match;
	QStringList::const_iterator			str_iter;
	QStringList::const_iterator			lib_iter;
	QStringList							active_targets;
	QStringList							active_libraries;
	QStringList							list;
	QString								text;
	int									count;
	int									cntr;
	
	d_variables_widget->Update_Variables();

	file.setFileName(file_name);
	
	if(file.open(QIODevice::WriteOnly | QIODevice::Text))
	{
		file.write("RPMSpecsEditor:Version=2:Type=Spec_Data\n\n");
		
		file.write("Variables_Begin:\n");
		
		count = d_variables_widget->Count();
		
		for(cntr = 0;cntr < count;++cntr)
		{
			variable = d_variables_widget->Variable(cntr);
			
			text = QStringLiteral("[Name]:");
			
			text += variable.name;
			text += QStringLiteral("[Value]:");
				
			text += variable.value;
			text.append('\n');
		
			file.write(text.toUtf8());
		}
		
		file.write("Variables_End:\n\n");

		file.write("Active_Targets_Begin:\n");

		active_targets = d_libraries_widget->Active_Targets();
		
		for(str_iter = active_targets.begin();str_iter != active_targets.end();++str_iter)
		{
			text = QString("%1\n").arg(*str_iter);
			file.write(text.toUtf8());
		}
		file.write("Active_Targets_End:\n\n");
		
		file.write("Active_Libraries_Begin:\n");
		
		for(str_iter = active_targets.begin();str_iter != active_targets.end();++str_iter)
		{
			text = QString("[Target]:%1\n").arg(*str_iter);
			file.write(text.toUtf8());

			active_libraries = d_libraries_widget->Active_Libraries(*str_iter);
				
			for(lib_iter = active_libraries.begin();lib_iter != active_libraries.end();++lib_iter)
			{
				library = d_libraries_widget->Library((*str_iter),(*lib_iter));
				
				switch(library.type)
				{
					case TTypes::TYPE_DYLIB:
						text = QStringLiteral("Dylib");
						break;
						
					case TTypes::TYPE_PLUGIN:
						text = QStringLiteral("Plugin");
						break;
				}
				
				library_match = library_expression.match(*lib_iter);
				if(library_match.hasMatch())
				{
					list = library_match.capturedTexts();
										
					if(list.size() == 2)
					{
						text = QString("[%1]:%2\n").arg(text).arg(list[1]);
						file.write(text.toUtf8());
					}
				}
				else
				{
					text = QString("[%1]:%2\n").arg(text).arg(*lib_iter);
					file.write(text.toUtf8());
				}
			}
		}
		file.write("Active_Libraries_End:\n\n");
		
		file.write("Active_Symbolic_Links_Begin:\n");

		count = d_expressions_widget->Symbolic_Links_Count();
		
		for(cntr = 0;cntr < count;++cntr)
		{
			link = d_expressions_widget->Symbolic_Link(cntr);
			
			if(link.checked)
			{
				text = QString("[Name]:%1[Value]:%2\n").arg(link.name).arg(link.path);
				file.write(text.toUtf8());
			}
		}
		
		file.write("Active_Symbolic_Links_End:\n\n");
		
		file.write("Additional_Executables:");
		text = d_expressions_widget->Target_Additional_Executables();
		file.write(text.toUtf8());
		file.write("\n\n");

		file.write("Description_Begin:\n");
		text = d_variables_widget->Description();
		file.write(text.toUtf8());
		file.write("\nDescription_End:\n\n");

		file.write("ChangeLog_Begin:\n");
		text = d_changelog_widget->Changes();
		file.write(text.toUtf8());
		file.write("\nChangeLog_End:\n");

		file.close();
	}
	
	return true;
}

bool TRPMSpecsEditor::Read_Spec_Data_File(
	const QString 						&file_name)
{
	QFile								file(file_name);
	QTextStream							text_stream;
	QStringList::const_iterator			iter;
	QString								line;
	QString								name;
	QString								value;
	TVariablesWidget::TVariable			variable;
	bool								init(true);
	bool								init_section(false);
	bool								inside_variables(false);
	bool								inside_active_targets(false);
	bool								inside_active_libraries(false);
	bool								inside_description(false);
	bool								inside_changelog(false);
	bool								inside_links(false);
	int									index;

	if(!file.open(QIODevice::ReadOnly))
	{
		d_last_error = QString("ERR:  Cannot open file '%1' for reading.").arg(file_name);
		return false;
	}
	
	text_stream.setDevice(&file);
	
	while(!text_stream.atEnd())
	{
		line = text_stream.readLine().trimmed();
		
		if(line.length() > 0)
		{
			if(!line.startsWith('#'))
			{
				if(init)
				{
					if(line == QStringLiteral("RPMSpecsEditor:Version=1:Type=Spec_Data"))
					{
						d_expressions_widget->Set_Target_Additional_Executables(QString());
						d_expressions_widget->Uncheck_Symbolic_Links();
					}
					else if(line != QStringLiteral("RPMSpecsEditor:Version=2:Type=Spec_Data"))
					{
						d_last_error = QStringLiteral("ERR:  File is not a RPMSpecsEditor spec_data file.");
						return false;
					}
					
					init = false;
				}
				else
				{
					if(line.startsWith(QStringLiteral("Variables_Begin:")))
					{
						init_section = true;
						inside_variables = true;
						
						inside_active_targets = false;
						inside_active_libraries = false;
						inside_description = false;
						inside_changelog = false;
						inside_links = false;
					}
					else if(line.startsWith(QStringLiteral("Variables_End:")))
					{
						inside_variables = false;
						d_variables_widget->Update_Table();
					}
					else if(line.startsWith(QStringLiteral("Active_Targets_Begin:")))
					{
						init_section = true;
						inside_active_targets = true;
						
						inside_variables = false;
						inside_active_libraries = false;
						inside_description = false;
						inside_changelog = false;
						inside_links = false;
					}
					else if(line.startsWith(QStringLiteral("Active_Targets_End:")))
					{
						inside_active_targets = false;
					}
					else if(line.startsWith(QStringLiteral("Active_Libraries_Begin:")))
					{
						init_section = true;
						inside_active_libraries = true;
						
						inside_variables = false;
						inside_active_targets = false;
						inside_description = false;
						inside_changelog = false;
						inside_links = false;
					}
					else if(line.startsWith(QStringLiteral("Active_Libraries_End:")))
					{
						inside_active_libraries = false;
						d_libraries_widget->Update();
					}
					else if(line.startsWith(QStringLiteral("Additional_Executables:")))
					{
						name = line.mid(23); // remove Additional_Executables:
						d_expressions_widget->Set_Target_Additional_Executables(name);
					}
					else if(line.startsWith(QStringLiteral("Description_Begin:")))
					{
						init_section = true;
						inside_description = true;
						
						inside_variables = false;
						inside_active_targets = false;
						inside_active_libraries = false;
						inside_changelog = false;
						inside_links = false;
					}
					else if(line.startsWith(QStringLiteral("Description_End:")))
					{
						inside_description = false;
					}
					else if(line.startsWith(QStringLiteral("ChangeLog_Begin:")))
					{
						init_section = true;
						inside_changelog = true;
						
						inside_variables = false;
						inside_active_targets = false;
						inside_active_libraries = false;
						inside_description = false;
						inside_links = false;
					}
					else if(line.startsWith(QStringLiteral("ChangeLog_End:")))
					{
						inside_changelog = false;
					}
					else if(line.startsWith(QStringLiteral("Active_Symbolic_Links_Begin:")))
					{
						init_section = true;
						inside_links = true;
						
						inside_variables = false;
						inside_active_targets = false;
						inside_active_libraries = false;
						inside_description = false;
						inside_changelog = false;
					}
					else if(line.startsWith(QStringLiteral("Active_Symbolic_Links_End:")))
					{
						inside_links = false;
						d_expressions_widget->Update_Symbolic_Link_Tree();
					}
				else
					{
						if(init_section)
						{
							if(inside_variables)
							{
								d_variables_widget->Clear_Variables();
							}
							else if(inside_active_targets)
							{
								d_libraries_widget->Unselect_Targets();
							}
							else if(inside_active_libraries)
							{
								d_libraries_widget->Unselect_Libaries();
							}
							else if(inside_description)
							{
								d_variables_widget->Clear_Description();
							}
							else if(inside_changelog)
							{
								d_changelog_widget->Clear_Changelog();
							}
							else if(inside_links)
							{
								d_expressions_widget->Uncheck_Symbolic_Links();
							}

							init_section = false;
						}

						if(inside_variables)
						{
							index = line.indexOf(QStringLiteral("[Name]:"));

							if(index < 0)
							{
								d_last_error = QStringLiteral("ERR:  RPMSpecsEditor variable data missing [Name]: value.");
								return false;
							}
							
							name = line.mid(index + 7); // remove [Name]:
							
							index = name.indexOf(QStringLiteral("[Value]:"));

							if(index < 0)
							{
								d_last_error = QStringLiteral("ERR:  RPMSpecsEditor variable data missing [Value]: value.");
								return false;
							}
							
							value = name.mid(index + 8); // remove [Value]:
							name.truncate(index);
							
							variable.name = name;
							variable.value = value;
							
							d_variables_widget->Add_Variable(variable);
						}
						else if(inside_links)
						{
							index = line.indexOf(QStringLiteral("[Name]:"));

							if(index < 0)
							{
								d_last_error = QStringLiteral("ERR:  RPMSpecsEditor link data missing [Name]: value.");
								return false;
							}
							
							name = line.mid(index + 7); // remove [Name]:
							
							index = name.indexOf(QStringLiteral("[Value]:"));

							if(index < 0)
							{
								d_last_error = QStringLiteral("ERR:  RPMSpecsEditor link data missing [Value]: value.");
								return false;
							}
							
							value = name.mid(index + 8); // remove [Value]:
							name.truncate(index);
														
							d_expressions_widget->Add_Symbolic_Link(name,value);
						}
						else if(inside_active_targets)
						{
							value = line;
							
							d_libraries_widget->Select_Target(value);
						}
						else if(inside_active_libraries)
						{
							if(line.startsWith(QStringLiteral("[Target]:")))
							{
								name = line.mid(9); // remove [Target]:
							}
							else if(line.startsWith(QStringLiteral("[Plugin]:")))
							{
								value = line.mid(9); // remove [Plugin]:
								d_libraries_widget->Select_Library(name,value);
							}
							else if(line.startsWith(QStringLiteral("[Dylib]:")))
							{
								value = line.mid(8); // remove [Dylib]:
								d_libraries_widget->Select_Library(name,value);
							}
							
							d_libraries_widget->Select_Library(name,value);
						}
						else if(inside_description)
						{
							d_variables_widget->Add_Description_Line(line);
						}
						else if(inside_changelog)
						{
							d_changelog_widget->Add_Changelog_Line(line);
						}
					}
				}
			}
		}
	}
	
	d_libraries_widget->Update();

	return true;
}



