/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QLabel>
#include <QPlainTextEdit>
#include <QTableWidget>
#include <QHeaderView>
#include <QToolButton>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QSpacerItem>
#include <QIcon>
#include <QFont>
#include <QList>
#include <assert.h>

#include "../../core/xmlfile.h"
#include "../../core/messagebox.h"

#include "variableeditordialog.h"
#include "variableswidget.h"

TVariablesWidget::TVariablesWidget(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QWidget(const_cast<QWidget*>(parent),flags)
{
	QLabel								*description_label;
	QLabel								*variables_label;
	QVBoxLayout							*widget_vlayout;
	QHBoxLayout							*button_hlayout;
	QSpacerItem							*widget_hspacer;
	QFont								font;

	font.setPointSize(15);
	font.setWeight(50);
	font.setItalic(true);
	font.setBold(false);
	font.setUnderline(true);

	widget_vlayout = new QVBoxLayout(this);

	variables_label = new QLabel(this);
	variables_label->setFont(font);
	widget_vlayout->addWidget(variables_label);

	d_variable_table = new QTableWidget(this);
	widget_vlayout->addWidget(d_variable_table);
	
	button_hlayout = new QHBoxLayout();

	d_restore_button = new QToolButton(this);
	d_restore_button->setFixedSize(QSize(16,16));
	button_hlayout->addWidget(d_restore_button);

	d_store_button = new QToolButton(this);
	d_store_button->setFixedSize(QSize(16,16));
	button_hlayout->addWidget(d_store_button);
	
	widget_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	button_hlayout->addItem(widget_hspacer);

	d_moveup_button = new QToolButton(this);
	d_moveup_button->setFixedSize(QSize(16,16));
	button_hlayout->addWidget(d_moveup_button);

	d_movedown_button = new QToolButton(this);
	d_movedown_button->setFixedSize(QSize(16,16));
	button_hlayout->addWidget(d_movedown_button);

	d_add_button = new QToolButton(this);
	d_add_button->setFixedSize(QSize(16,16));
	button_hlayout->addWidget(d_add_button);

	d_remove_button = new QToolButton(this);
	d_remove_button->setFixedSize(QSize(16,16));
	button_hlayout->addWidget(d_remove_button);
	
	widget_vlayout->addLayout(button_hlayout);

	description_label = new QLabel(this);
	description_label->setFont(font);
	widget_vlayout->addWidget(description_label);

	d_description_edit = new QPlainTextEdit(this);
	d_description_edit->setMaximumSize(16777215,100);
	widget_vlayout->addWidget(d_description_edit);
	
	// defaults
	d_variable_editor_dialog = 0;
	d_msg_box = 0;

	d_moveup_button->setEnabled(false);
	d_movedown_button->setEnabled(false);
	d_remove_button->setEnabled(false);

	d_variable_table->horizontalHeader()->setSectionResizeMode(QHeaderView::Stretch);
	
	d_moveup_button->setIcon(QIcon(QStringLiteral(":/gui/edit_up_x16.png")));
	d_movedown_button->setIcon(QIcon(QStringLiteral(":/gui/edit_down_x16.png")));
	d_add_button->setIcon(QIcon(QStringLiteral(":/gui/edit_add_x16.png")));
	d_remove_button->setIcon(QIcon(QStringLiteral(":/gui/edit_remove_x16.png")));
	d_restore_button->setIcon(QIcon(QStringLiteral(":/gui/file_load_x16.png")));
	d_store_button->setIcon(QIcon(QStringLiteral(":/gui/file_save_x16.png")));

	variables_label->setText(QStringLiteral("Variables:"));
	description_label->setText(QStringLiteral("Description:"));
	
	d_moveup_button->setToolTip(QStringLiteral("Move selected variable up"));
	d_movedown_button->setToolTip(QStringLiteral("Move selected variable down"));
	d_add_button->setToolTip(QStringLiteral("Add new variable"));
	d_remove_button->setToolTip(QStringLiteral("Remove selected variables"));
	d_restore_button->setToolTip(QStringLiteral("Load variables from backup"));
	d_store_button->setToolTip(QStringLiteral("Save current variables to backup"));

	
	connect(d_variable_table,&QTableWidget::itemSelectionChanged,this,&TVariablesWidget::Selection_Changed);
	
	connect(d_moveup_button,&QToolButton::clicked,this,&TVariablesWidget::Move_Item_Up);
	connect(d_movedown_button,&QToolButton::clicked,this,&TVariablesWidget::Move_Item_Down);
	connect(d_add_button,&QToolButton::clicked,this,&TVariablesWidget::Add_Item);
	connect(d_remove_button,&QToolButton::clicked,this,&TVariablesWidget::Remove_Item);
	connect(d_restore_button,&QToolButton::clicked,this,&TVariablesWidget::Restore_Alternate);
	connect(d_store_button,&QToolButton::clicked,this,&TVariablesWidget::Store_Alternate);
}

TVariablesWidget::~TVariablesWidget(void)
{
}

bool TVariablesWidget::Save_Defaults(
	const QString 						&file_path)
{
	std::vector<TVariablesWidget::TVariable>::const_iterator iter;
	QString								file_name;
	TXmlFile							xml_file;
	QDomElement							e0;
	QDomElement							e1;
	QDomElement							e2;
	QString								text;
	
	file_name = file_path + QStringLiteral("/variables.xml");
	this->Update_Variable_Data();

	try
	{
		xml_file.Set_File_Name(file_name);
		xml_file.Reset(QStringLiteral("Variable_Data"),1);

		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Description"));
		text = d_description_edit->toPlainText();
		xml_file.Write_Text_Node(&e0,text);

		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Variables"));

		for(iter = d_variables.begin();iter != d_variables.end();++iter)
		{
			e1 = xml_file.Create_Node(&e0,QStringLiteral("Variable"));
			e2 = xml_file.Create_Node(&e1,QStringLiteral("Name"));
			xml_file.Write_Text_Node(&e2,(*iter).name);
			
			e2 = xml_file.Create_Node(&e1,QStringLiteral("Value"));
			xml_file.Write_Text_Node(&e2,(*iter).value);
		}

		xml_file.Write_File();
	}
	catch(TXmlFileException xml_file_exception)
	{
		d_last_error = xml_file_exception.ErrorText();
		return false;
	}

	return true;
}

bool TVariablesWidget::Save_Alternate(
	const QString 						&file_path)
{
	std::vector<TVariablesWidget::TVariable>::const_iterator iter;
	QString								file_name;
	TXmlFile							xml_file;
	QDomElement							e0;
	QDomElement							e1;
	QDomElement							e2;
	QString								text;
	
	file_name = file_path + QStringLiteral("/alternate_variables.xml");
	this->Update_Variable_Data();

	try
	{
		xml_file.Set_File_Name(file_name);
		xml_file.Reset(QStringLiteral("Variable_Data"),1);

		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Description"));
		text = d_description_edit->toPlainText();
		xml_file.Write_Text_Node(&e0,text);

		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Variables"));

		for(iter = d_variables.begin();iter != d_variables.end();++iter)
		{
			e1 = xml_file.Create_Node(&e0,QStringLiteral("Variable"));
			e2 = xml_file.Create_Node(&e1,QStringLiteral("Name"));
			xml_file.Write_Text_Node(&e2,(*iter).name);
			
			e2 = xml_file.Create_Node(&e1,QStringLiteral("Value"));
			xml_file.Write_Text_Node(&e2,(*iter).value);
		}

		xml_file.Write_File();
	}
	catch(TXmlFileException xml_file_exception)
	{
		d_last_error = xml_file_exception.ErrorText();
		return false;
	}

	return true;
}

TVariablesWidget::TVariable TVariablesWidget::Variable(
	const int 							index) const
{	
	assert(static_cast<unsigned int>(index) < d_variables.size());
	
	return d_variables[index];
}

QString TVariablesWidget::Description(void) const
{
	return d_description_edit->toPlainText();
}

bool TVariablesWidget::Load_Defaults(
	const QString 						&file_path)
{
	TXmlFile							xml_file;
	QString								file_name;
	int									version;
	bool								result(true);
	
	file_name = file_path + QStringLiteral("/variables.xml");
	
	d_variables.clear();
	d_description_edit->clear();
	
	d_default_file_path = file_path;

	try
	{
		xml_file.Set_File_Name(file_name);
		xml_file.Open(QStringLiteral("Variable_Data"),&version);

		switch(version)
		{
			case 1:
				this->Load_V1(&xml_file);
				break;
				
			default:
				throw TXmlFileException(QStringLiteral("Variable datafile version not recognized"));
		}
	}
	catch(TXmlFileException xml_file_exception)
	{
		d_last_error = xml_file_exception.ErrorText();
		
		this->Initialize_Variables();
		result = false;
	}
	
	this->Update_Table_Data();
	return result;
 }

bool TVariablesWidget::Load_Alternate(
	const QString 						&file_path)
{
	TXmlFile							xml_file;
	QString								file_name;
	int									version;
	bool								result(true);
	
	file_name = file_path + QStringLiteral("/alternate_variables.xml");
	
	if(!QFile::exists(file_name))
	{
		d_last_error = QStringLiteral("Alternate variable data does not exist");
		return false;
	}
	
	d_variables.clear();
	d_description_edit->clear();

	try
	{
		xml_file.Set_File_Name(file_name);
		xml_file.Open(QStringLiteral("Variable_Data"),&version);

		switch(version)
		{
			case 1:
				this->Load_V1(&xml_file);
				break;
				
			default:
				throw TXmlFileException(QStringLiteral("Variable datafile version not recognized"));
		}
	}
	catch(TXmlFileException xml_file_exception)
	{
		d_last_error = xml_file_exception.ErrorText();
		
		this->Initialize_Variables();
		result = false;
	}
	
	this->Update_Table_Data();
	return result;
}

void TVariablesWidget::Clear_Variables(void)
{
	d_variables.clear();
}

void TVariablesWidget::Add_Variable(
	const TVariablesWidget::TVariable 	&variable)
{
	d_variables.push_back(variable);
}

void TVariablesWidget::Update_Table(void)
{
	this->Update_Table_Data();
}

void TVariablesWidget::Update_Variables(void)
{
	this->Update_Variable_Data();
}

void TVariablesWidget::Clear_Description(void)
{
		d_description_edit->clear();
}

void TVariablesWidget::Add_Description_Line(
	const QString 						&line)
{
	d_description_edit->appendPlainText(line);
}

void TVariablesWidget::Selection_Changed(void)
{		
	QList<QTableWidgetItem*>			selected_items;
	
	d_moveup_button->setEnabled(false);
	d_movedown_button->setEnabled(false);
	d_remove_button->setEnabled(false);
	
	selected_items = d_variable_table->selectedItems();
		
	if(selected_items.size() == 1)
	{
		d_moveup_button->setEnabled(true);
		d_movedown_button->setEnabled(true);
		d_remove_button->setEnabled(true);
	}
}

void TVariablesWidget::Move_Item_Up(void)
{
	QTableWidgetItem					*header_item;
	QTableWidgetItem					*value_item;
	QList<QTableWidgetItem*>			selected_items;
	int									row;
	
	selected_items = d_variable_table->selectedItems();
		
	if(selected_items.size() == 1)
	{
		row = selected_items[0]->row();
		if(row > 0)
		{
			header_item = d_variable_table->takeVerticalHeaderItem(row);
			value_item = d_variable_table->takeItem(row,0);
			
			d_variable_table->removeRow(row);
			--row;
			
			d_variable_table->insertRow(row);
			d_variable_table->setVerticalHeaderItem(row,header_item);
			d_variable_table->setItem(row,0,value_item);
			
			d_variable_table->selectRow(row);
		}
	}
}

void TVariablesWidget::Move_Item_Down(void)
{
	QTableWidgetItem					*header_item;
	QTableWidgetItem					*value_item;
	QList<QTableWidgetItem*>			selected_items;
	int									row;
	int									max_row;
	
	selected_items = d_variable_table->selectedItems();
		
	if(selected_items.size() == 1)
	{
		row = selected_items[0]->row();
		max_row = d_variable_table->rowCount() - 1;
		
		if(row < max_row)
		{
			header_item = d_variable_table->takeVerticalHeaderItem(row);
			value_item = d_variable_table->takeItem(row,0);
			
			d_variable_table->removeRow(row);
			++row;
			
			d_variable_table->insertRow(row);
			d_variable_table->setVerticalHeaderItem(row,header_item);
			d_variable_table->setItem(row,0,value_item);
			
			d_variable_table->selectRow(row);
		}
	}
}

void TVariablesWidget::Add_Item(void)
{
	TVariablesWidget::TVariable			variable;
	
	if(!d_variable_editor_dialog)
	{
		d_variable_editor_dialog = new TVariableEditorDialog(this);
	}
	
	if(QDialog::Accepted == d_variable_editor_dialog->exec())
	{
		variable.name = d_variable_editor_dialog->Name();
		variable.value = d_variable_editor_dialog->Value();
		
		d_variables.push_back(variable);
		
		this->Update_Table_Data();
	}
}

void TVariablesWidget::Remove_Item(void)
{
	QTableWidgetItem					*header_item;
	QTableWidgetItem					*value_item;
	QList<QTableWidgetItem*>			selected_items;
	int									row;
	
	if(!d_msg_box)
	{
		d_msg_box = new TMessageBox(this);
	}
	
	d_msg_box->setText(QStringLiteral("Remove Variables"));
	d_msg_box->setInformativeText(QStringLiteral("Do you want to remove the selected variables?"));
	d_msg_box->setDetailedText(QString());
	d_msg_box->setStandardButtons(QMessageBox::Yes | QMessageBox::No);
	d_msg_box->setDefaultButton(QMessageBox::Yes);
	d_msg_box->setIcon(QMessageBox::Warning);
	
	if(QMessageBox::No == d_msg_box->exec())
	{
		return;
	}

	selected_items = d_variable_table->selectedItems();
		
	if(selected_items.size() == 1)
	{
		row = selected_items[0]->row();
		
		header_item = d_variable_table->takeVerticalHeaderItem(row);
		value_item = d_variable_table->takeItem(row,0);
		d_variable_table->removeRow(row);
		
		delete header_item;
		delete value_item;
		
		this->Update_Variable_Data();
	}
}

void TVariablesWidget::Store_Alternate(void)
{
	this->Save_Alternate(d_default_file_path);
}

void TVariablesWidget::Restore_Alternate(void)
{
	this->Load_Alternate(d_default_file_path);
}

void TVariablesWidget::Initialize_Variables(void)
{
	TVariablesWidget::TVariable			variable;
	
	d_variables.clear();
	
	variable.name = QStringLiteral("%define _topdir");
	variable.value = QStringLiteral("%{getenv:HOME}/rpmbuild");
	d_variables.push_back(variable);
	
	variable.name = QStringLiteral("%define _builddir");
	variable.value = QStringLiteral("%{_topdir}/BUILD");
	d_variables.push_back(variable);
	
	variable.name = QStringLiteral("%define _buildrootdir");
	variable.value = QStringLiteral("%{_topdir}/BUILDROOT");
	d_variables.push_back(variable);

	variable.name = QStringLiteral("%define name");
	variable.value = QStringLiteral("program_name");
	d_variables.push_back(variable);
	
	variable.name = QStringLiteral("%define release");
	variable.value = QStringLiteral("1");
	d_variables.push_back(variable);
	
	variable.name = QStringLiteral("%define version");
	variable.value = QStringLiteral("1.0");
	d_variables.push_back(variable);
	
	variable.name = QStringLiteral("%define buildroot");
	variable.value = QStringLiteral("%{_topdir}/%{name}-%{version}-root");
	d_variables.push_back(variable);
	
	variable.name = QStringLiteral("Name:");
	variable.value = QStringLiteral("%{name}");
	d_variables.push_back(variable);

	variable.name = QStringLiteral("Version:");
	variable.value = QStringLiteral("%{version}");
	d_variables.push_back(variable);
	
	variable.name = QStringLiteral("Release:");
	variable.value = QStringLiteral("%{release}");
	d_variables.push_back(variable);
	
	variable.name = QStringLiteral("Source:");
	variable.value = QStringLiteral("%{name}-%{version}.tar.gz");
	d_variables.push_back(variable);

	variable.name = QStringLiteral("Summary:");
	variable.value = QStringLiteral("summary of the program function");
	d_variables.push_back(variable);
	
	variable.name = QStringLiteral("License:");
	variable.value = QStringLiteral("GPLv3+");
	d_variables.push_back(variable);
	
	variable.name = QStringLiteral("Group:");
	variable.value = QStringLiteral("Applications");
	d_variables.push_back(variable);
	
	variable.name = QStringLiteral("URL:");
	variable.value = QStringLiteral("http://www.website.ca");
	d_variables.push_back(variable);
	
	variable.name = QStringLiteral("Requires:");
	variable.value = QStringLiteral("glibc >= 2.31");
	d_variables.push_back(variable);
	
	d_description_edit->clear();
	d_description_edit->appendPlainText(QStringLiteral("Description of the program"));
}

void TVariablesWidget::Update_Variable_Data(void)
{
	TVariablesWidget::TVariable			variable;
	QTableWidgetItem					*item;
	int									cntr;
	int									count;
	
	this->setFocus();	// makes sure that any active table edit is completed.
	
	d_variables.clear();
	
	count = d_variable_table->rowCount();
	
	for(cntr = 0;cntr < count;++cntr)
	{
		item = d_variable_table->verticalHeaderItem(cntr);
		variable.name = item->text();
		
		item = d_variable_table->item(cntr,0);
		variable.value = item->text();

		d_variables.push_back(variable);
	}
}

void TVariablesWidget::Update_Table_Data(void)
{
	std::vector<TVariablesWidget::TVariable>::const_iterator iter;
	QTableWidgetItem					*item;
	int									cntr;

	d_variable_table->clear();
	
	d_variable_table->setColumnCount(1);
	d_variable_table->setRowCount(d_variables.size());
	
	item = new QTableWidgetItem();
	item->setText(QStringLiteral("Value"));
	d_variable_table->setHorizontalHeaderItem(0,item);
	
	for(iter = d_variables.begin(),cntr = 0;iter != d_variables.end();++iter,++cntr)
	{
		item = new QTableWidgetItem();
		item->setText((*iter).name);
		d_variable_table->setVerticalHeaderItem(cntr,item);

		item = new QTableWidgetItem();
		item->setText((*iter).value);
		d_variable_table->setItem(cntr,0,item);
	}
}

void TVariablesWidget::Load_V1(
	TXmlFile 							* const xml_file)
{
	TVariablesWidget::TVariable			variable;
	QDomElement							e0;
	QDomElement							e1;
	QDomElement							e2;
	QString								text;
	
	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Description"));
	text = xml_file->Read_Text_Node(&e0);
	d_description_edit->setPlainText(text);

	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Variables"));
	if(!e0.isNull())
	{
		e1 = xml_file->Get_Node(&e0,QStringLiteral("Variable"));
		while(!e1.isNull())
		{
			e2 = xml_file->Get_Node(&e1,QStringLiteral("Name"));
			variable.name = xml_file->Read_Text_Node(&e2);
		
			e2 = xml_file->Get_Node(&e1,QStringLiteral("Value"));
			variable.value = xml_file->Read_Text_Node(&e2);

			d_variables.push_back(variable);

			e1 = xml_file->Get_Sibling_Node(&e1,QStringLiteral("Variable"),true);
		}
	}
}
