/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QFile>
#include <QProcess>
#include <QDir>
#include <QTextStream>
#include <assert.h>

// #define DEBUG_DATA_PROCESSING

#include "libcmakelistsimpl.h"

static const int						MAX_PARENTHESIS_LEVEL(10);

TCMakeListsImpl::TCMakeListsImpl(void)
{
	d_scope_names.push_back(QStringLiteral("linux-g++"));
	d_scope_names.push_back(QStringLiteral("linux"));
	d_scope_names.push_back(QStringLiteral("win32"));
	d_scope_names.push_back(QStringLiteral("win32-g++"));
	d_scope_names.push_back(QStringLiteral("win64"));
	d_scope_names.push_back(QStringLiteral("win64-g++"));
	d_scope_names.push_back(QStringLiteral("macx"));
	d_scope_names.push_back(QStringLiteral("macos"));
	d_scope_names.push_back(QStringLiteral("unix"));
	d_scope_names.push_back(QStringLiteral("unix-g++"));
}

TCMakeListsImpl::~TCMakeListsImpl(void)
{
}

bool TCMakeListsImpl::Write_CMakeLists(
	const QString						&file_name)
{
	QStringList::const_iterator			line_iter;
	QFile								file;
	QString								script_file_path;
	QString								text;

#ifdef DEBUG_DATA_PROCESSING
	qWarning("-----------------------------------------");
	qWarning(QString("Output file: '%1'").arg(file_name).toLatin1());
	qWarning("-----------------------------------------");
#endif
	
	if(file.exists())
	{
		d_last_error = QStringLiteral("Output file already exists and will not be replaced.");
		return false;
	}
		
	if(!d_cmake_lines.size())
	{
		d_last_error = QStringLiteral("Output file is empty and will not be created.");
		return true;	// not an error
	}
	
	file.setFileName(file_name);
	
	if(file.open(QIODevice::WriteOnly | QIODevice::Text))
	{
		for(line_iter = d_cmake_lines.begin();line_iter != d_cmake_lines.end();++line_iter)
		{
			text = (*line_iter);
			text.append(QChar('\n'));
			
			file.write(text.toUtf8());
		}

		file.close();
	}
	else
	{
		d_last_error = QString("Cannot open output file '%1' for writing.").arg(file_name);
		return false;
	}
			
	return true;
}

bool TCMakeListsImpl::Load_QMake(
	const QString						&file_name)
{
	TCMakeListsImpl::TEntry				entry;
	std::vector<TCMakeListsImpl::TEntry>::const_iterator entry_iter;
	QStringList::const_iterator 		line_iter;
	QStringList::const_iterator			list_iter;
	QStringList							qmake_lines;
	QStringList							list;
	QStringList							scope_names;
	QFile								file(file_name);
	QDir								dir;
	QByteArray							file_data;
	QString								file_path;
	QString								include_file_name;
	QString								line;
	QString								text;
	TCMakeListsImpl::TProjectType		project_type;
	int									parenthesis_level;
	int									index;
	bool								append(false);
	bool								process_file_list(false);
	bool								remove_mode;
	bool								enable_scope[MAX_PARENTHESIS_LEVEL + 1];
	bool								condition_result;
	bool								condition_result_valid;

#ifdef DEBUG_DATA_PROCESSING
	qWarning("-----------------------------------------");
	qWarning(QString("Input file:      '%1'").arg(file_name).toLatin1());
#endif
	
	d_entries.clear();
	d_cmake_lines.clear();
	
	dir.setPath(file_name);
	if(!dir.exists())
	{
		dir.cdUp();
	}
	file_path = dir.path();

#ifdef DEBUG_DATA_PROCESSING
	qWarning(QString("Input file path: '%1'").arg(file_path).toLatin1());
	qWarning("-----------------------------------------");
#endif
		
	if(file.open(QIODevice::ReadOnly))
	{
		file_data = file.readAll();
		file.close();
		
		QTextStream						text_stream(file_data,QIODevice::ReadOnly);
		
		while(!text_stream.atEnd())
		{
			line = text_stream.readLine().simplified();
			
			if(line.startsWith('#'))
			{
				continue;
			}
			
			// variable replacements
			line.replace(QStringLiteral("$$PWD"),file_path,Qt::CaseInsensitive);
									
			if(append && qmake_lines.size())
			{
				text = qmake_lines[qmake_lines.size()-1];
				
				text += line;
				line = text;
				
				qmake_lines.pop_back();
			}
			
			if(line.length())
			{
				if(line.endsWith('\\'))
				{
					append = true;
					line.chop(1);
					line.append(' ');
				}
				else
				{
					append = false;
				}
				
				qmake_lines.push_back(line);
			}
		}
	}
	else
	{
		d_last_error = QStringLiteral("Cannot open input file.");
		return false;
	}
	
	// set TARGET to current name of project file by default
	index = file_name.lastIndexOf('/');
	
	if(!(index < 0))
	{
		text = file_name.mid(index + 1);
		
		if(text.endsWith(QStringLiteral(".pro"),Qt::CaseInsensitive))
		{
			text.chop(4);
		}
		
		if(text.length())
		{
			entry.scope_name = QString();
			entry.name = QStringLiteral("TARGET");
			
			entry.values.clear();
			entry.values.push_back(text);
			
			this->Add_Entry(entry);
		}
	}
	
	qmake_lines = this->Expand_QMake(qmake_lines);
	
#ifdef DEBUG_DATA_PROCESSING
	qWarning("-----------------------------------------");
	qWarning("Initial processing and expanding of qmake");
	qWarning("-----------------------------------------");

	for(line_iter = qmake_lines.begin();line_iter != qmake_lines.end();++line_iter)
	{
		qWarning((*line_iter).toLatin1());
	}
#endif

	parenthesis_level = 0;
	memset(enable_scope,0,sizeof(enable_scope));
	enable_scope[0] = true;
	condition_result = true;
	condition_result_valid = false;

	entry.scope_name = QString();
	
	for(line_iter = qmake_lines.begin();line_iter != qmake_lines.end();++line_iter)
	{
		assert(parenthesis_level < MAX_PARENTHESIS_LEVEL);
		
		line = (*line_iter);
		
#ifdef DEBUG_DATA_PROCESSING
		text = scope_names.join('|');
		qWarning(QString("parenthesis: %1  enabled: %2  scopes: '%3'  line: '%4'").arg(parenthesis_level).arg(enable_scope[parenthesis_level]).arg(text,-20).arg(*line_iter).toLatin1());
#endif

		if(this->Is_Qt_Scope_Name(line))
		{
			scope_names = this->Get_CMake_Scope_Names(line);
		}
		else
		{
			process_file_list = false;
			
			entry.name.clear();
			entry.values.clear();
			
			if(line[0] == QChar('{'))
			{
				parenthesis_level++;
				
				if(condition_result_valid)
				{
					enable_scope[parenthesis_level] = condition_result;
				}
				else
				{
					enable_scope[parenthesis_level] = true;
				}
			}
			else if(line[0] == QChar('}'))
			{
				parenthesis_level--;
				
				if(!parenthesis_level)
				{
					scope_names.clear();
				}
			}
			else if(line.compare(QStringLiteral("else"),Qt::CaseInsensitive) == 0)
			{
				condition_result_valid = true;
				
				if(enable_scope[parenthesis_level])
				{
					condition_result = !condition_result;
				}
				else
				{
					condition_result = false;
				}
			}
			else if(parenthesis_level == 0 || enable_scope[parenthesis_level] == true)
			{
				condition_result_valid = false;
				
				if(line.startsWith(QStringLiteral("CONFIG")))
				{
					if(line.contains('(') && line.contains(')'))
					{
						list = this->Get_Parameter_Names(line);

						condition_result_valid = true;
						condition_result = true;
						if(list.size() == 2)
						{
							if(!this->Has_Entry_Value(QString(),QStringLiteral("CONFIG"),list[0]))
							{
								condition_result = false;
							}
						}
					}
					else	// version is =, +=, or -=
					{
						line = line.mid(6).trimmed();
						entry.name = QStringLiteral("CONFIG");
						process_file_list = true;
					}
				}
				else if(line.startsWith(QStringLiteral("contains"),Qt::CaseInsensitive))
				{
					list = this->Get_Parameter_Names(line);
										
					condition_result_valid = true;
					condition_result = true;
					if(list.size() == 2)
					{
						if(!this->Has_Entry_Value(QString(),list[0],list[1]))
						{
							condition_result = false;
						}
					}
				}
				else if(line.startsWith(QStringLiteral("DEFINES")))
				{
					line = line.mid(7).trimmed();
					entry.name = QStringLiteral("DEFINES");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("DISTFILES")))
				{
					line = line.mid(9).trimmed();
					entry.name = QStringLiteral("DISTFILES");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("EXTRA_TRANSLATIONS")))
				{
					line = line.mid(18).trimmed();
					entry.name = QStringLiteral("EXTRA_TRANSLATIONS");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("FORMS")))
				{
					line = line.mid(5).trimmed();
					entry.name = QStringLiteral("FORMS");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("HEADERS")))
				{
					line = line.mid(7).trimmed();
					entry.name = QStringLiteral("HEADERS");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("ICON")))
				{
					line = line.mid(4).trimmed();
					entry.name = QStringLiteral("ICON");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("INCLUDE("),Qt::CaseInsensitive))
				{
					text = line.mid(8).trimmed();
					text.chop(1);	// remove trailing brace
					
					dir.setPath(file_path);
					
					include_file_name = QDir::cleanPath(dir.absoluteFilePath(text));

					if(!this->Load_QMake_Include_File(file_path,include_file_name))
					{
						d_last_error = QString("Error processing include file. %1").arg(d_last_error);
						return false;
					}
				}
				else if(line.startsWith(QStringLiteral("INCLUDEPATH")))
				{
					line = line.mid(11).trimmed();
					entry.name = QStringLiteral("INCLUDEPATH");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("LIBS")))
				{
					line = line.mid(4).trimmed();
					entry.name = QStringLiteral("LIBS");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("OBJECTIVE_SOURCES")))
				{
					line = line.mid(17).trimmed();
					entry.name = QStringLiteral("OBJECTIVE_SOURCES");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("OBJECTIVE_HEADERS")))
				{
					line = line.mid(17).trimmed();
					entry.name = QStringLiteral("OBJECTIVE_HEADERS");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("QMAKE_INFO_PLIST")))
				{
					line = line.mid(16).trimmed();
					entry.name = QStringLiteral("QMAKE_INFO_PLIST");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("QMAKE_POST_LINK")))
				{
					line = line.mid(15).trimmed();
					entry.name = QStringLiteral("QMAKE_POST_LINK");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("RESOURCES")))
				{
					line = line.mid(9).trimmed();
					entry.name = QStringLiteral("RESOURCES");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("RC_FILE")))
				{
					line = line.mid(7).trimmed();
					entry.name = QStringLiteral("RC_FILE");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("SOURCES")))
				{
					line = line.mid(7).trimmed();
					entry.name = QStringLiteral("SOURCES");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("SUBDIRS")))
				{
					line = line.mid(7).trimmed();
					entry.name = QStringLiteral("SUBDIRS");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("TARGET")))
				{
					line = line.mid(6).trimmed();
					entry.name = QStringLiteral("TARGET");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("TEMPLATE")))
				{
					line = line.mid(8).trimmed();
					entry.name = QStringLiteral("TEMPLATE");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("QT")))
				{
					line = line.mid(2).trimmed();
					entry.name = QStringLiteral("QT");
					process_file_list = true;
				}

				if(process_file_list)
				{
					remove_mode = false;
					
					if(line.startsWith('+'))	// handle +=
					{
						line = line.mid(1);
					}
					else if(line.startsWith('-'))	// handle -=
					{
						line = line.mid(1);
						remove_mode = true;
					}
				
					if(line.startsWith('='))
					{
						line = line.mid(1).trimmed();
						
						list = line.split(' ');
						
						for(list_iter = list.begin();list_iter != list.end();++list_iter)
						{
							if((*list_iter).length())
							{
								entry.values.push_back(*list_iter);
							}
						}
						
						if(remove_mode)
						{
							this->Remove_Entry_Values(entry,scope_names);
						}
						else
						{
							this->Add_Entry(entry,scope_names);
						}
					}
				}
			}
		}
	}
		
#ifdef DEBUG_DATA_PROCESSING
	qWarning("-----------------------------------------");
	qWarning("Processed entries");
	qWarning("-----------------------------------------");

	for(entry_iter = d_entries.begin();entry_iter != d_entries.end();++entry_iter)
	{
		qWarning(QString("Scope: '%1'  Name: '%2'").arg((*entry_iter).scope_name,-10).arg((*entry_iter).name).toLatin1());
		for(list_iter = (*entry_iter).values.begin();list_iter != (*entry_iter).values.end();++list_iter)
		{
			qWarning(QString("   %1").arg((*list_iter)).toLatin1());
		}
	}
#endif
	
	project_type = this->Get_Project_Type();
	
#ifdef DEBUG_DATA_PROCESSING
	qWarning("-----------------------------------------");
	if(project_type == TCMakeListsImpl::PROJECT_SUBDIRS) qWarning("Project Type: PROJECT_SUBDIRS");
	if(project_type == TCMakeListsImpl::PROJECT_APPLICATION) qWarning("Project Type: PROJECT_APPLICATION");
	if(project_type == TCMakeListsImpl::PROJECT_LIBRARY) qWarning("Project Type: PROJECT_LIBRARY");
	qWarning("-----------------------------------------");

#endif

	if((project_type == TCMakeListsImpl::PROJECT_SUBDIRS && this->Get_Project_SubDir_Dir_Count()) ||
		project_type == TCMakeListsImpl::PROJECT_APPLICATION ||
		project_type == TCMakeListsImpl::PROJECT_LIBRARY)
	{
		if(!this->Initialize_CMakeList_Data(project_type))
		{
			d_last_error = QString("Initialize_CMakeList_Data failed.  %1").arg(d_last_error);
			return false;
		}
		
		if(project_type == TCMakeListsImpl::PROJECT_SUBDIRS)
		{		
			if(!this->Process_SubDirs())
			{
				d_last_error = QString("Process_SubDirs failed.  %1").arg(d_last_error);
				return false;
			}
		}
		else if(project_type == TCMakeListsImpl::PROJECT_APPLICATION || 
			project_type == TCMakeListsImpl::PROJECT_LIBRARY)
		{
			if(!this->Process_Global_Libs())
			{
				d_last_error = QString("Process_Global_Libs failed.  %1").arg(d_last_error);
				return false;
			}
					
			if(!this->Process_Sources(project_type))
			{
				d_last_error = QString("Process_Sources failed.  %1").arg(d_last_error);
				return false;
			}

			if(!this->Process_Linux_Entries(project_type))
			{
				d_last_error = QString("Process_Linux_Entries failed.  %1").arg(d_last_error);
				return false;
			}
			
			if(!this->Process_Apple_Entries(project_type))
			{
				d_last_error = QString("Process_Apple_Entries failed.  %1").arg(d_last_error);
				return false;
			}
			
			if(!this->Process_Unix_Entries(project_type))
			{
				d_last_error = QString("Process_Unix_Entries failed.  %1").arg(d_last_error);
				return false;
			}

			if(!this->Process_Windows_Entries(project_type))
			{
				d_last_error = QString("Process_Windows_Entries failed.  %1").arg(d_last_error);
				return false;
			}
			
			if(!this->Process_CMake_Target_Link_Libraries())
			{
				d_last_error = QString("Process_CMake_Target_Link_Libraries failed.  %1").arg(d_last_error);
				return false;
			}
		}
		
		if(!this->Finalize_CMakeList_Data(project_type))
		{
			d_last_error = QString("Finalize_CMakeList_Data failed.  %1").arg(d_last_error);
			return false;
		}
	}
#ifdef DEBUG_DATA_PROCESSING
	else
	{
		qWarning("WAR:  Qt Template subdirs has no referenced directories");
	}
#endif
	
#ifdef DEBUG_DATA_PROCESSING
	qWarning("-----------------------------------------");
	qWarning("CMake lists data");
	qWarning("-----------------------------------------");

	for(list_iter = d_cmake_lines.begin();list_iter != d_cmake_lines.end();++list_iter)
	{
		qWarning(QString("%1").arg((*list_iter)).toLatin1());
	}
#endif
	
	return true;
}

bool TCMakeListsImpl::Load_QMake_Include_File(
	const QString 						&root_path,
	const QString						&file_name)
{
	TCMakeListsImpl::TEntry				entry;
	QStringList							qmake_lines;
	std::vector<TCMakeListsImpl::TEntry>::const_iterator entry_iter;
	QStringList::const_iterator 		line_iter;
	QStringList::const_iterator			list_iter;
	QStringList							list;
	QStringList							scope_names;
	QFile								file(file_name);
	QDir								dir;
	QByteArray							file_data;
	QString								file_path;
	QString								line;
	QString								text;
	int									parenthesis_level;
	bool								append(false);
	bool								process_file_list(false);
	bool								remove_mode;
	bool								enable_scope[MAX_PARENTHESIS_LEVEL + 1];
	bool								condition_result;
	bool								condition_result_valid;

#ifdef DEBUG_DATA_PROCESSING
	qWarning("-----------------------------------------");
	qWarning(QString(" Include Input file:      '%1'").arg(file_name).toLatin1());
#endif
		
	dir.setPath(file_name);
	if(!dir.exists())
	{
		dir.cdUp();
	}
	file_path = dir.path();

#ifdef DEBUG_DATA_PROCESSING
	qWarning(QString(" Include Input file path: '%1'").arg(file_path).toLatin1());
	qWarning("-----------------------------------------");
#endif
			
	if(file.open(QIODevice::ReadOnly))
	{
		file_data = file.readAll();
		file.close();
		
		QTextStream						text_stream(file_data,QIODevice::ReadOnly);
		
		while(!text_stream.atEnd())
		{
			line = text_stream.readLine().simplified();
			
			if(line.startsWith('#'))
			{
				continue;
			}
			
			// variable replacements
			if(line.contains(QStringLiteral("$$PWD"),Qt::CaseInsensitive))
			{
				dir.setPath(root_path);
				text = QDir::cleanPath(dir.relativeFilePath(file_path));
				
				line.replace(QStringLiteral("$$PWD"),text,Qt::CaseInsensitive);
			}
						
			if(append && qmake_lines.size())
			{
				text = qmake_lines[qmake_lines.size()-1];
				
				text += line;
				line = text;
				
				qmake_lines.pop_back();
			}
			
			if(line.length())
			{
				if(line.endsWith('\\'))
				{
					append = true;
					line.chop(1);
					line.append(' ');
				}
				else
				{
					append = false;
				}
				
				qmake_lines.push_back(line);
			}
		}
	}
	else
	{
		d_last_error = QStringLiteral("Cannot open input file.");
		return false;
	}
			
	qmake_lines = this->Expand_QMake(qmake_lines);
	
#ifdef DEBUG_DATA_PROCESSING
	qWarning("-----------------------------------------");
	qWarning(" Initial processing and expanding of qmake from include file");
	qWarning("-----------------------------------------");

	for(line_iter = qmake_lines.begin();line_iter != qmake_lines.end();++line_iter)
	{
		qWarning((*line_iter).toLatin1());
	}
#endif

	parenthesis_level = 0;
	memset(enable_scope,0,sizeof(enable_scope));
	enable_scope[0] = true;
	condition_result = true;
	condition_result_valid = false;

	entry.scope_name = QString();

	for(line_iter = qmake_lines.begin();line_iter != qmake_lines.end();++line_iter)
	{
		line = (*line_iter);
		
#ifdef DEBUG_DATA_PROCESSING
		text = scope_names.join('|');
		qWarning(QString("parenthesis: %1  enabled: %2  scopes: '%3'  line: '%4'").arg(parenthesis_level).arg(enable_scope[parenthesis_level]).arg(text,-20).arg(*line_iter).toLatin1());
#endif

		if(this->Is_Qt_Scope_Name(line))
		{
			scope_names = this->Get_CMake_Scope_Names(line);
		}
		else
		{
			process_file_list = false;
			
			entry.name.clear();
			entry.values.clear();
			
			if(line[0] == QChar('{'))
			{
				parenthesis_level++;
				
				if(condition_result_valid)
				{
					enable_scope[parenthesis_level] = condition_result;
				}
				else
				{
					enable_scope[parenthesis_level] = true;
				}
			}
			else if(line[0] == QChar('}'))
			{
				parenthesis_level--;
				
				if(!parenthesis_level)
				{
					scope_names.clear();
				}
			}
			else if(line.compare(QStringLiteral("else"),Qt::CaseInsensitive) == 0)
			{
				condition_result_valid = true;
				
				if(enable_scope[parenthesis_level])
				{
					condition_result = !condition_result;
				}
				else
				{
					condition_result = false;
				}
			}
			else if(parenthesis_level == 0 || enable_scope[parenthesis_level] == true)
			{
				if(line.startsWith(QStringLiteral("contains"),Qt::CaseInsensitive))
				{
					list = this->Get_Parameter_Names(line);
										
					condition_result_valid = true;
					condition_result = true;
					if(list.size() == 2)
					{
						if(!this->Has_Entry_Value(QString(),list[0],list[1]))
						{
							condition_result = false;
						}
					}
				}
				else if(line.startsWith(QStringLiteral("DEFINES")))
				{
					line = line.mid(7).trimmed();
					entry.name = QStringLiteral("DEFINES");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("DISTFILES")))
				{
					line = line.mid(9).trimmed();
					entry.name = QStringLiteral("DISTFILES");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("EXTRA_TRANSLATIONS")))
				{
					line = line.mid(18).trimmed();
					entry.name = QStringLiteral("EXTRA_TRANSLATIONS");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("FORMS")))
				{
					line = line.mid(5).trimmed();
					entry.name = QStringLiteral("FORMS");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("HEADERS")))
				{
					line = line.mid(7).trimmed();
					entry.name = QStringLiteral("HEADERS");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("INCLUDEPATH")))
				{
					line = line.mid(11).trimmed();
					entry.name = QStringLiteral("INCLUDEPATH");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("OBJECTIVE_SOURCES")))
				{
					line = line.mid(17).trimmed();
					entry.name = QStringLiteral("OBJECTIVE_SOURCES");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("OBJECTIVE_HEADERS")))
				{
					line = line.mid(17).trimmed();
					entry.name = QStringLiteral("OBJECTIVE_HEADERS");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("RESOURCES")))
				{
					line = line.mid(9).trimmed();
					entry.name = QStringLiteral("RESOURCES");
					process_file_list = true;
				}
				else if(line.startsWith(QStringLiteral("SOURCES")))
				{
					line = line.mid(7).trimmed();
					entry.name = QStringLiteral("SOURCES");
					process_file_list = true;
				}

				if(process_file_list)
				{
					remove_mode = false;
					
					if(line.startsWith('+'))	// handle +=
					{
						line = line.mid(1);
					}
					else if(line.startsWith('-'))	// handle -=
					{
						line = line.mid(1);
						remove_mode = true;
					}
				
					if(line.startsWith('='))
					{
						line = line.mid(1).trimmed();
						
						list = line.split(' ');
						
						for(list_iter = list.begin();list_iter != list.end();++list_iter)
						{
							if((*list_iter).length())
							{
								entry.values.push_back(*list_iter);
							}
						}
						
						if(remove_mode)
						{
							this->Remove_Entry_Values(entry,scope_names);
						}
						else
						{
							this->Add_Entry(entry,scope_names);
						}
					}
				}
			}
		}
	}
	
#ifdef DEBUG_DATA_PROCESSING
	qWarning("-----------------------------------------");
	qWarning(" Processed entries from include file");
	qWarning("-----------------------------------------");

	for(entry_iter = d_entries.begin();entry_iter != d_entries.end();++entry_iter)
	{
		qWarning(QString("Scope: '%1'  Name: '%2'").arg((*entry_iter).scope_name,-10).arg((*entry_iter).name).toLatin1());
		for(list_iter = (*entry_iter).values.begin();list_iter != (*entry_iter).values.end();++list_iter)
		{
			qWarning(QString("    %1").arg((*list_iter)).toLatin1());
		}
	}
#endif
	
	return true;
}

QStringList TCMakeListsImpl::Expand_QMake(
	const QStringList 					&qmake_lines) const
{
	QStringList::const_iterator 		line_iter;
	QStringList							scope_lines;
	QString								line;
	int									index;
	
	for(line_iter = qmake_lines.begin();line_iter != qmake_lines.end();++line_iter)
	{
		line = (*line_iter);

		if(line.length())
		{
			if(line.endsWith('{'))
			{
				line.chop(1);
				line = line.trimmed();
				
				if(line.endsWith(':'))	// cases where scope written as <name>:{ intead of <name>{
				{
					line.chop(1);
					line = line.trimmed();
				}
								
				if(line[0] == QChar('}'))
				{
					line = line.mid(1);
					line = line.trimmed();

					scope_lines.push_back(QStringLiteral("}"));
					scope_lines.push_back(line);
					scope_lines.push_back(QStringLiteral("{"));
				}
				else
				{
					if(line.length())
					{
						scope_lines.push_back(line);
					}

					scope_lines.push_back(QStringLiteral("{"));
				}
			}
			else if(line[0] == QChar('}'))
			{
				line = line.mid(1);
				line = line.trimmed();

				scope_lines.push_back(QStringLiteral("}"));
				
				if(line.length())
				{
					scope_lines.push_back(line);
				}
			}
			else if(line.contains(QChar(':')))
			{
				index = line.indexOf(QChar(':'));
				
				scope_lines.push_back(line.mid(0,index));
				
				scope_lines.push_back(QStringLiteral("{"));
				scope_lines.push_back(line.mid(index+1));
				scope_lines.push_back(QStringLiteral("}"));
			}
			else
			{
				scope_lines.push_back(line);
			}
		}
	}
		
	return scope_lines;
}

bool TCMakeListsImpl::Is_Qt_Scope_Name(
	const QString 						&name) const
{
	QString								text;
	int									index;
	
	text = name;
	index = text.indexOf('|');
	
	// may be a list of scopes.  Only checking first one
	if(!(index < 0))
	{
		text.truncate(index);
		text = text.trimmed();
	}
	
	if(d_scope_names.contains(text.toLower()))
	{
		return true;
	}

	return false;
}

QStringList TCMakeListsImpl::Get_CMake_Scope_Names(
	const QString 						&qmake_scope_name) const
{
	QStringList::const_iterator			iter;
	QStringList							result;
	QStringList							list;
	QString								name;
	
	list = qmake_scope_name.split('|');
		
	for(iter = list.begin();iter != list.end();++iter)
	{
		name = (*iter).trimmed();
		
		if(name.compare(QStringLiteral("linux"),Qt::CaseInsensitive) == 0)
		{
			result.push_back(QStringLiteral("LINUX"));
		}
		else if(name.compare(QStringLiteral("linux-g++"),Qt::CaseInsensitive) == 0)
		{
			result.push_back(QStringLiteral("LINUX"));
		}
		else if(name.compare(QStringLiteral("macx"),Qt::CaseInsensitive) == 0)
		{
			result.push_back(QStringLiteral("APPLE"));
		}
		else if(name.compare(QStringLiteral("macos"),Qt::CaseInsensitive) == 0)
		{
			result.push_back(QStringLiteral("APPLE"));
		}
		else if(name.compare(QStringLiteral("win32"),Qt::CaseInsensitive) == 0)
		{
			result.push_back(QStringLiteral("WIN32"));
		}
		else if(name.compare(QStringLiteral("win32-g++"),Qt::CaseInsensitive) == 0)
		{
			result.push_back(QStringLiteral("WIN32"));
		}
		else if(name.compare(QStringLiteral("win64"),Qt::CaseInsensitive) == 0)
		{
			result.push_back(QStringLiteral("WIN32"));
		}
		else if(name.compare(QStringLiteral("win64-g++"),Qt::CaseInsensitive) == 0)
		{
			result.push_back(QStringLiteral("WIN32"));
		}
		else if(name.compare(QStringLiteral("unix"),Qt::CaseInsensitive) == 0)
		{
			result.push_back(QStringLiteral("UNIX"));
		}
		else if(name.compare(QStringLiteral("unix-g++"),Qt::CaseInsensitive) == 0)
		{
			result.push_back(QStringLiteral("UNIX"));
		}
		else
		{
#ifdef DEBUG_DATA_PROCESSING
			qWarning(QString("WAR:  Qt scope name '%1' unknown.").arg(name).toLatin1());
#endif
		}
	}
	
	result.removeDuplicates();

	return result;
}

QStringList TCMakeListsImpl::Get_Parameter_Names(
	const QString 						&line) const
{
	QStringList::iterator				iter;
	QStringList							list;
	QString								text;
	int									index;
		
	text = line;
	index = text.indexOf('(');
	
	if(!(index < 0))
	{
		text = text.mid(index+1);
		index = text.indexOf(')');
		
		if(!(index < 0))
		{
			text.truncate(index);
			list = text.split(',');
		}
	}
	
	for(iter = list.begin();iter != list.end();++iter)
	{
		(*iter) = (*iter).trimmed();
	}
	
	return list;
}

QString TCMakeListsImpl::Get_Module_Name(
	const QString 						&name) const
{
	QString								result;

	if(name.length())
	{
		if(name.compare(QStringLiteral("core")) == 0)
		{
			result = QStringLiteral("Core");
		}
		else if(name.compare(QStringLiteral("widgets")) == 0)
		{
			result = QStringLiteral("Widgets");
		}
		else if(name.compare(QStringLiteral("gui")) == 0)
		{
			result = QStringLiteral("Gui");
		}
		else if(name.compare(QStringLiteral("network")) == 0)
		{
			result = QStringLiteral("Network");
		}
		else if(name.compare(QStringLiteral("sql")) == 0)
		{
			result = QStringLiteral("Sql");
		}
		else if(name.compare(QStringLiteral("multimedia")) == 0)
		{
			result = QStringLiteral("Multimedia");
		}
		else if(name.compare(QStringLiteral("webkit")) == 0)
		{
			result = QStringLiteral("WebKit");
		}
		else if(name.compare(QStringLiteral("svg")) == 0)
		{
			result = QStringLiteral("Svg");
		}
		else if(name.compare(QStringLiteral("opengl")) == 0)
		{
			result = QStringLiteral("OpenGL");
		}
		else if(name.compare(QStringLiteral("printsupport")) == 0)
		{
			result = QStringLiteral("PrintSupport");
		}
		else if(name.compare(QStringLiteral("script")) == 0)
		{
			result = QStringLiteral("Script");
		}
		else if(name.compare(QStringLiteral("xml")) == 0)
		{
			result = QStringLiteral("Xml");
		}
		else if(name.compare(QStringLiteral("xmlPatterns")) == 0)
		{
			result = QStringLiteral("XmlPatterns");
		}
		else if(name.compare(QStringLiteral("qml")) == 0)
		{
			result = QStringLiteral("Qml");
		}
		else if(name.compare(QStringLiteral("positioning")) == 0)
		{
			result = QStringLiteral("Positioning");
		}
		else if(name.compare(QStringLiteral("quick")) == 0)
		{
			result = QStringLiteral("Quick");
		}
		else if(name.compare(QStringLiteral("sensors")) == 0)
		{
			result = QStringLiteral("Sensors");
		}
		else if(name.compare(QStringLiteral("serialport")) == 0)
		{
			result = QStringLiteral("SerialPort");
		}
		else if(name.compare(QStringLiteral("sql")) == 0)
		{
			result = QStringLiteral("Sql");
		}
		else if(name.compare(QStringLiteral("declarative")) == 0)
		{
			result = QStringLiteral("Declarative");
		}
		else if(name.compare(QStringLiteral("testlib")) == 0)
		{
			result = QStringLiteral("Test");
		}
		else if(name.compare(QStringLiteral("openglwidgets")) == 0)
		{
			result = QStringLiteral("OpenGLWidgets");
		}
		else
		{
			result = name;
			result[0] = name[0].toUpper();
			
#ifdef DEBUG_DATA_PROCESSING
			qWarning(QString("WAR:  Qt Module name unknown.  Convert from '%1' to '%2' assumed.").arg(name).arg(result).toLatin1());
#endif

		}
	}
	
	return result;
}

bool TCMakeListsImpl::Has_Scope(
	const QString 						&name) const
{
	std::vector<TCMakeListsImpl::TEntry>::const_iterator iter;

	for(iter = d_entries.begin();iter != d_entries.end();++iter)
	{
		if((*iter).scope_name == name)
		{
			return true;
		}
	}
	
	return false;
}

bool TCMakeListsImpl::Has_Entry(
	const QString 						&scope_name,
	const QString 						&name) const
{
	std::vector<TCMakeListsImpl::TEntry>::const_iterator iter;

	for(iter = d_entries.begin();iter != d_entries.end();++iter)
	{
		if((*iter).name == name && (*iter).scope_name == scope_name)
		{
			return true;
		}
	}
	
	return false;
}

bool TCMakeListsImpl::Has_Entry_Value(
	const QString 						&scope_name,
	const QString 						&name,
	const QString						&value) const
{
	std::vector<TCMakeListsImpl::TEntry>::const_iterator iter;
	QStringList::const_iterator			list_iter;

	for(iter = d_entries.begin();iter != d_entries.end();++iter)
	{
		if((*iter).name == name && (*iter).scope_name == scope_name)
		{
			for(list_iter = (*iter).values.begin();list_iter != (*iter).values.end();++list_iter)
			{
				if((*list_iter).compare(value,Qt::CaseInsensitive) == 0)
				{
					return true;
				}
			}
			
			return false;
		}
	}
	
	return false;
}

TCMakeListsImpl::TEntry TCMakeListsImpl::Get_Entry(
	const QString 						&scope_name,
	const QString 						&name,
	bool 								* const	valid) const
{
	std::vector<TCMakeListsImpl::TEntry>::const_iterator iter;

	assert(valid);
	
	*valid = false;

	for(iter = d_entries.begin();iter != d_entries.end();++iter)
	{
		if((*iter).name == name && (*iter).scope_name == scope_name)
		{
			*valid = true;
			return (*iter);
		}
	}
	
	return TCMakeListsImpl::TEntry();
}

TCMakeListsImpl::TProjectType TCMakeListsImpl::Get_Project_Type(void) const
{
	TCMakeListsImpl::TEntry				entry;
	bool								valid;
	
	entry = this->Get_Entry(QString(),QStringLiteral("TEMPLATE"),&valid);
	
	if(valid && entry.values.size())
	{
		if(entry.values[0].compare(QStringLiteral("subdirs"),Qt::CaseInsensitive) == 0)
		{
			return TCMakeListsImpl::PROJECT_SUBDIRS;
		}
		else if(entry.values[0].compare(QStringLiteral("app"),Qt::CaseInsensitive) == 0)
		{
			return TCMakeListsImpl::PROJECT_APPLICATION;
		}
		else if(entry.values[0].compare(QStringLiteral("lib"),Qt::CaseInsensitive) == 0)
		{
			return TCMakeListsImpl::PROJECT_LIBRARY;
		}
	}
	
	return TCMakeListsImpl::PROJECT_APPLICATION;	// default to applicataion
}

int TCMakeListsImpl::Get_Project_SubDir_Dir_Count(void) const
{
	TCMakeListsImpl::TEntry				entry;
	bool								valid;
	
	entry = this->Get_Entry(QString(),QStringLiteral("SUBDIRS"),&valid);
	
	if(valid)
	{
		return entry.values.size();
	}
	
	return 0;
}

void TCMakeListsImpl::Add_Entry(
	const TCMakeListsImpl::TEntry 		&entry)
{
	std::vector<TCMakeListsImpl::TEntry>::iterator iter;

	for(iter = d_entries.begin();iter != d_entries.end();++iter)
	{
		if((*iter).scope_name == entry.scope_name &&
			((*iter).name == entry.name || entry.name == QStringLiteral("TARGET")))
		{
			// special case for target
			// only one target entry should exist.
			if((*iter).name == QStringLiteral("TARGET"))
			{
				if((*iter).values.size() && entry.values.size())
				{
					(*iter).values.clear();
					(*iter).values.push_back(entry.values[0]);
					return;
				}
			}
			(*iter).values += entry.values;
			(*iter).values.removeDuplicates();
			return;
		}
	}

	d_entries.push_back(entry);

}

void TCMakeListsImpl::Add_Entry(
	const TCMakeListsImpl::TEntry 		&entry,
	const QStringList					&scope_names)
{
	QStringList::const_iterator 		iter;
	TCMakeListsImpl::TEntry				new_entry;
	
	if(scope_names.size())
	{
		for(iter = scope_names.begin();iter != scope_names.end();++iter)
		{
			new_entry = entry;
			new_entry.scope_name = (*iter);
			
			this->Add_Entry(new_entry);
		}
	}
	else
	{
		this->Add_Entry(entry);
	}
}

void TCMakeListsImpl::Remove_Entry_Values(
	const TCMakeListsImpl::TEntry 		&entry)
{
	std::vector<TCMakeListsImpl::TEntry>::iterator iter;
	QStringList::const_iterator			value_iter;

	for(iter = d_entries.begin();iter != d_entries.end();++iter)
	{
		if((*iter).scope_name == entry.scope_name &&
			(*iter).name == entry.name)
		{
			for(value_iter = entry.values.begin();value_iter != entry.values.end();++value_iter)
			{
				(*iter).values.removeOne(*value_iter);
			}
			
			return;
		}
	}
}

void TCMakeListsImpl::Remove_Entry_Values(
	const TCMakeListsImpl::TEntry 		&entry,
	const QStringList					&scope_names)
{
	QStringList::const_iterator 		iter;
	TCMakeListsImpl::TEntry				new_entry;
	
	if(scope_names.size())
	{
		for(iter = scope_names.begin();iter != scope_names.end();++iter)
		{
			new_entry = entry;
			new_entry.scope_name = (*iter);
			
			this->Remove_Entry_Values(new_entry);
		}
	}
	else
	{
		this->Remove_Entry_Values(entry);
	}
}

bool TCMakeListsImpl::Initialize_CMakeList_Data(
	const TCMakeListsImpl::TProjectType project_type)
{
	TCMakeListsImpl::TEntry				entry;
	QStringList::const_iterator			iter;
	QStringList							list;
	QString								text;
	int									index;
	bool								valid;
	bool								has_core;
	bool								has_widgets;
	bool								has_forms;
	bool								has_resources;
	bool								has_opengl;
	
	d_cmake_lines.clear();
		
	d_cmake_lines.push_back(QStringLiteral("cmake_minimum_required(VERSION 3.16)"));
	d_cmake_lines.push_back(QString());
	
	entry = this->Get_Entry(QString(),QStringLiteral("TARGET"),&valid);
	
	if(valid == false || entry.values.size() == 0)
	{
		d_last_error = QStringLiteral("Missing TARGET entry.");
		return false;
	}

	d_cmake_lines.push_back(QString("project(%1)").arg(entry.values[0]));
	d_cmake_lines.push_back(QString());
	
	if(project_type == TCMakeListsImpl::PROJECT_APPLICATION ||
	   project_type == TCMakeListsImpl::PROJECT_LIBRARY)
	{
		d_cmake_lines.push_back(QStringLiteral("if(WIN32)"));
		d_cmake_lines.push_back(QString());
		d_cmake_lines.push_back(QStringLiteral("    add_definitions(-DUNICODE -D_UNICODE)"));
		d_cmake_lines.push_back(QString());
		d_cmake_lines.push_back(QStringLiteral("    if(MSVC)"));
		d_cmake_lines.push_back(QStringLiteral("	    set(CMAKE_WINDOWS_EXPORT_ALL_SYMBOLS ON)"));
		d_cmake_lines.push_back(QStringLiteral("    endif()"));
		d_cmake_lines.push_back(QString());
		d_cmake_lines.push_back(QStringLiteral("endif()"));
		d_cmake_lines.push_back(QString());
		
		has_forms = this->Has_Entry(QString(),QStringLiteral("FORMS"));
		has_resources = this->Has_Entry(QString(),QStringLiteral("RESOURCES"));
	
		d_cmake_lines.push_back(QStringLiteral("if(DEFINED TARGET_BIN_LOCATION)"));
		d_cmake_lines.push_back(QStringLiteral("    set(CMAKE_RUNTIME_OUTPUT_DIRECTORY ${TARGET_BIN_LOCATION})"));
		d_cmake_lines.push_back(QStringLiteral("    set(CMAKE_LIBRARY_OUTPUT_DIRECTORY ${TARGET_BIN_LOCATION})"));
		d_cmake_lines.push_back(QStringLiteral("else()"));
		d_cmake_lines.push_back(QStringLiteral("    set(CMAKE_RUNTIME_OUTPUT_DIRECTORY ${PROJECT_SOURCE_DIR}/bin)"));
		d_cmake_lines.push_back(QStringLiteral("    set(CMAKE_LIBRARY_OUTPUT_DIRECTORY ${PROJECT_SOURCE_DIR}/lib)"));
		d_cmake_lines.push_back(QStringLiteral("endif()"));
		d_cmake_lines.push_back(QString());

		d_cmake_lines.push_back(QStringLiteral("set(CMAKE_ARCHIVE_OUTPUT_DIRECTORY ${PROJECT_SOURCE_DIR}/lib)"));
		d_cmake_lines.push_back(QString());
		
		d_cmake_lines.push_back(QStringLiteral("set(CMAKE_INCLUDE_CURRENT_DIR ON)"));
		d_cmake_lines.push_back(QStringLiteral("set(CMAKE_AUTOMOC ON)"));
		
		if(has_resources)
		{
			d_cmake_lines.push_back(QStringLiteral("set(CMAKE_AUTORCC ON)"));
		}
		
		if(has_forms)
		{
			d_cmake_lines.push_back(QStringLiteral("set(CMAKE_AUTOUIC ON)"));
			
			entry = this->Get_Entry(QString(),QStringLiteral("FORMS"),&valid);
			
			if(valid)
			{
				list.clear();
				
				for(iter = entry.values.begin();iter != entry.values.end();++iter)
				{
					index = (*iter).lastIndexOf('/');
					
					if(!(index < 0))
					{
						text = (*iter);
						text.truncate(index);
						list.push_back(text);
					}
				}
				
				for(iter = list.begin();iter != list.end();++iter)
				{
					d_cmake_lines.push_back(QString("set(CMAKE_AUTOUIC_SEARCH_PATHS ${PROJECT_SOURCE_DIR}/%1)").arg(*iter));
				}			
			}
		}
		
		d_cmake_lines.push_back(QString());
		
		d_cmake_lines.push_back(QStringLiteral("find_package(QT NAMES Qt6 Qt5)"));
		d_cmake_lines.push_back(QString());
		
		d_cmake_lines.push_back(QStringLiteral("message(STATUS \"Using prefix Qt${QT_VERSION_MAJOR} for components...\")"));
		d_cmake_lines.push_back(QString());

		has_core = false;
		has_widgets = false;
		has_opengl = false;
		entry = this->Get_Entry(QString(),QStringLiteral("QT"),&valid);
		
		if(valid)
		{
			list.clear();
			
			for(iter = entry.values.begin();iter != entry.values.end();++iter)
			{
				if((*iter).compare(QStringLiteral("core")) == 0)
				{
					has_core = true;
				}
				else if((*iter).compare(QStringLiteral("widgets")) == 0)
				{
					has_widgets = true;
				}
				else if((*iter).compare(QStringLiteral("opengl")) == 0)
				{
					has_opengl = true;
				}

				text = this->Get_Module_Name(*iter);
				
				// special cases for Widgets and OpenGLWidgets
				

				if(text == QStringLiteral("OpenGLWidgets"))
				{
					d_cmake_lines.push_back(QStringLiteral("if(${QT_VERSION_MAJOR} GREATER 5)"));
					d_cmake_lines.push_back(QString("    find_package(Qt${QT_VERSION_MAJOR} REQUIRED COMPONENTS %1)").arg(text));
					d_cmake_lines.push_back(QStringLiteral("endif()"));
				}
				else if(text == QStringLiteral("Widgets"))
				{
					d_cmake_lines.push_back(QStringLiteral("if(${QT_VERSION_MAJOR} GREATER 4)"));
					d_cmake_lines.push_back(QString("    find_package(Qt${QT_VERSION_MAJOR} REQUIRED COMPONENTS %1)").arg(text));
					d_cmake_lines.push_back(QStringLiteral("endif()"));
				}
				else
				{
					d_cmake_lines.push_back(QString("find_package(Qt${QT_VERSION_MAJOR} REQUIRED COMPONENTS %1)").arg(text));
				}
				
				list.push_back(QString("Qt${QT_VERSION_MAJOR}::%1").arg(text));
			}
			entry.scope_name = QStringLiteral("cmake");
			entry.name = QStringLiteral("target_link_libraries");
			entry.values = list;

			this->Add_Entry(entry);
		}
	
		// must have core module for features such as moc or rcc
		if(!has_core)
		{
			text = this->Get_Module_Name(QStringLiteral("core"));
					
			d_cmake_lines.push_back(QString("find_package(Qt${QT_VERSION_MAJOR} REQUIRED COMPONENTS %1)").arg(text));
			list.push_back(QString("Qt${QT_VERSION_MAJOR}::%1").arg(text));
				
			entry.scope_name = QStringLiteral("cmake");
			entry.name = QStringLiteral("target_link_libraries");
			entry.values = list;

			this->Add_Entry(entry);
		}
	
		// must have widgets module.  may not be listed for Qt4
		if(!has_widgets)
		{
			text = this->Get_Module_Name(QStringLiteral("widgets"));
					
			d_cmake_lines.push_back(QString("find_package(Qt${QT_VERSION_MAJOR} REQUIRED COMPONENTS %1)").arg(text));
			list.push_back(QString("Qt${QT_VERSION_MAJOR}::%1").arg(text));
				
			entry.scope_name = QStringLiteral("cmake");
			entry.name = QStringLiteral("target_link_libraries");
			entry.values = list;

			this->Add_Entry(entry);
		}
		
		if(has_opengl)
		{
			d_cmake_lines.push_back(QStringLiteral("find_package(OpenGL REQUIRED)"));
			list.push_back(QStringLiteral("OpenGL::GL"));
				
			entry.scope_name = QStringLiteral("cmake");
			entry.name = QStringLiteral("target_link_libraries");
			entry.values = list;

			this->Add_Entry(entry);
		}
	}
	else if(project_type == TCMakeListsImpl::PROJECT_SUBDIRS)
	{
		d_cmake_lines.push_back(QStringLiteral("if(NOT DEFINED TARGET_BIN_LOCATION)"));
		d_cmake_lines.push_back(QStringLiteral("    set(TARGET_BIN_LOCATION ${PROJECT_SOURCE_DIR}/bin)"));
		d_cmake_lines.push_back(QStringLiteral("endif()"));
		d_cmake_lines.push_back(QString());
	}
	
	d_cmake_lines.push_back(QString());

	return true;
}

bool TCMakeListsImpl::Process_Global_Libs(void)
{
	TCMakeListsImpl::TEntry				entry;
	QStringList::const_iterator			iter;
	QStringList							list;
	QString								text;
	QString								path;
	QString								name;	
	bool								valid;
	
	entry = this->Get_Entry(QString(),QStringLiteral("LIBS"),&valid);
	
	// expecting path and name pairs for global scope libs
	if(valid)
	{	
		for(iter = entry.values.begin();iter != entry.values.end();++iter)
		{
			if((*iter).startsWith(QStringLiteral("-L")))
			{
				path = (*iter).mid(2);
			}
			else if((*iter).startsWith(QStringLiteral("-l")))
			{
				name = (*iter).mid(2);
			}
			
			if(path.length() && name.length())
			{
				list.push_back(name);
				
				path.clear();
				name.clear();
			}
		}
		
		if(list.size())
		{
			text = QStringLiteral("if(WIN32)");
			d_cmake_lines.push_back(text);
			
			text = QStringLiteral("    if(MSVC)");
			d_cmake_lines.push_back(text);
				
			for(iter = list.begin();iter != list.end();++iter)
			{
				text = QString("        set_target_properties(%1 PROPERTIES IMPORTED_LOCATION ${PROJECT_SOURCE_DIR}/../lib%1/lib/lib%1.lib)").arg(*iter);
				d_cmake_lines.push_back(text);
			}
			
			text = QStringLiteral("    elseif(CMAKE_COMPILER_IS_GNUCXX)");
			d_cmake_lines.push_back(text);

			for(iter = list.begin();iter != list.end();++iter)
			{
				text = QString("        set_target_properties(%1 PROPERTIES IMPORTED_LOCATION ${PROJECT_SOURCE_DIR}/../lib%1/lib/lib%1.a)").arg(*iter);
				d_cmake_lines.push_back(text);
			}
			
			text = QStringLiteral("    endif()");
			d_cmake_lines.push_back(text);
			
			text = QStringLiteral("else()");
			d_cmake_lines.push_back(text);
			
			for(iter = list.begin();iter != list.end();++iter)
			{
				text = QString("    set_target_properties(%1 PROPERTIES IMPORTED_LOCATION ${PROJECT_SOURCE_DIR}/../lib%1/lib/lib%1.a)").arg(*iter);
				d_cmake_lines.push_back(text);
			}
			
			text = QStringLiteral("endif()");
			d_cmake_lines.push_back(text);
			d_cmake_lines.push_back(QString());

			entry.scope_name = QStringLiteral("cmake");
			entry.name = QStringLiteral("target_link_libraries");
			entry.values = list;
			
			this->Add_Entry(entry);
		}
	}
	
	return true;
}
	
bool TCMakeListsImpl::Process_Sources(
	const TCMakeListsImpl::TProjectType project_type)
{
	TCMakeListsImpl::TEntry				entry;
	QStringList::const_iterator			iter;
	QStringList							list;
	QString								target_name;
	QString								text;
	int									ival;
	bool								valid;
	bool								init;
	bool								has_sources(false);
	bool								has_headers(false);
	bool								has_forms(false);
	bool								has_resources(false);

	entry = this->Get_Entry(QString(),QStringLiteral("TARGET"),&valid);
	
	if(valid == false || entry.values.size() == 0)
	{
		d_last_error = QStringLiteral("Missing TARGET entry.");
		return false;
	}
	
	target_name = entry.values[0];
	
	entry = this->Get_Entry(QString(),QStringLiteral("CONFIG"),&valid);
	if(valid)
	{
		ival = 0;
		for(iter = entry.values.begin();iter != entry.values.end();++iter)
		{
			if((*iter).startsWith(QStringLiteral("c++")))
			{
				text = (*iter).mid(3);	//
				ival = text.toInt();
				break;
			}
		}
		
		if(ival > 0)
		{
			d_cmake_lines.push_back(QString("set(CMAKE_CXX_STANDARD %1)").arg(ival));
			d_cmake_lines.push_back(QStringLiteral("set(CMAKE_CXX_STANDARD_REQUIRED ON)"));
			d_cmake_lines.push_back(QString());
		}
	}

	entry = this->Get_Entry(QString(),QStringLiteral("SOURCES"),&valid);
	
	if(valid && entry.values.size())
	{
		has_sources = true;
		init = true;
		for(iter = entry.values.begin();iter != entry.values.end();++iter)
		{
			if(init)
			{
				text = QString("set(sources %1").arg(*iter);
				init = false;
			}
			else
			{
				text = QString("            %1").arg(*iter);
			}
			d_cmake_lines.push_back(text);
		}
		
		// add ending bracket
		if(!init)
		{
			d_cmake_lines[d_cmake_lines.size() - 1].append(QChar(')'));
		}
		d_cmake_lines.push_back(QString());
	}

	entry = this->Get_Entry(QString(),QStringLiteral("HEADERS"),&valid);
	
	if(valid && entry.values.size())
	{
		has_headers = true;
		init = true;
		for(iter = entry.values.begin();iter != entry.values.end();++iter)
		{
			if(init)
			{
				text = QString("set(headers %1").arg(*iter);
				init = false;
			}
			else
			{
				text = QString("            %1").arg(*iter);
			}
			d_cmake_lines.push_back(text);
		}
		
		// add ending bracket
		if(!init)
		{
			d_cmake_lines[d_cmake_lines.size() - 1].append(QChar(')'));
		}
		d_cmake_lines.push_back(QString());
	}

	entry = this->Get_Entry(QString(),QStringLiteral("RESOURCES"),&valid);
	
	if(valid && entry.values.size())
	{
		has_resources = true;
		init = true;
		for(iter = entry.values.begin();iter != entry.values.end();++iter)
		{
			if(init)
			{
				text = QString("set(resources %1").arg(*iter);
				init = false;
			}
			else
			{
				text = QString("              %1").arg(*iter);
			}
			d_cmake_lines.push_back(text);
		}
		
		// add ending bracket
		if(!init)
		{
			d_cmake_lines[d_cmake_lines.size() - 1].append(QChar(')'));
		}
		d_cmake_lines.push_back(QString());
	}

	entry = this->Get_Entry(QString(),QStringLiteral("FORMS"),&valid);
	
	if(valid && entry.values.size())
	{
		has_forms = true;
		init = true;
		for(iter = entry.values.begin();iter != entry.values.end();++iter)
		{
			if(init)
			{
				text = QString("set(forms %1").arg(*iter);
				init = false;
			}
			else
			{
				text = QString("          %1").arg(*iter);
			}
			d_cmake_lines.push_back(text);
		}
		
		// add ending bracket
		if(!init)
		{
			d_cmake_lines[d_cmake_lines.size() - 1].append(QChar(')'));
		}
		d_cmake_lines.push_back(QString());
	}
/*	
	entry = this->Get_Entry(QString(),QStringLiteral("DEFINES"),&valid);
		
	if(valid)
	{
		for(iter = entry.values.begin();iter != entry.values.end();++iter)
		{
			d_cmake_lines.push_back(QString("target_compile_definitions(%1 PRIVATE %2)").arg(target_name).arg(*iter));
		}
		d_cmake_lines.push_back(QString());
	}
*/	
	if(!this->Process_Scoped_Sources(QStringLiteral("LINUX")))
	{
		return false;
	}
	
	if(!this->Process_Scoped_Sources(QStringLiteral("APPLE")))
	{
		return false;
	}
	
	if(!this->Process_Scoped_Sources(QStringLiteral("UNIX")))
	{
		return false;
	}
	
	if(!this->Process_Scoped_Sources(QStringLiteral("WIN32")))
	{
		return false;
	}
	
	if(project_type == TCMakeListsImpl::PROJECT_APPLICATION)
	{
		text = QString("add_executable(%1").arg(target_name);
		
		if(has_sources)
		{
			text += QStringLiteral(" ${sources}");
		}
		
		if(has_headers)
		{
			text += QStringLiteral(" ${headers}");
		}
		
		if(has_forms)
		{
			text += QStringLiteral(" ${forms}");
		}

		if(has_resources)
		{
			text += QStringLiteral(" ${resources}");
		}
		
		text.append(')');
		
		d_cmake_lines.push_back(text);
		d_cmake_lines.push_back(QString());		
	}
	else if(project_type == TCMakeListsImpl::PROJECT_LIBRARY)
	{
		this->Process_Scoped_Libraries(target_name,has_sources,has_headers,has_resources,has_forms);
	}
	
	entry = this->Get_Entry(QString(),QStringLiteral("INCLUDEPATH"),&valid);
	if(valid)
	{
		for(iter = entry.values.begin();iter != entry.values.end();++iter)
		{
			text = QString("target_include_directories(%1 PRIVATE ${PROJECT_SOURCE_DIR}/%2)").arg(target_name).arg(*iter);
			d_cmake_lines.push_back(text);
		}
		d_cmake_lines.push_back(QString());
	}

	return true;
}

bool TCMakeListsImpl::Process_Scoped_Sources(
	const QString 						&scope_name)
{
	TCMakeListsImpl::TEntry				entry;
	QStringList::const_iterator			iter;
	QStringList							list;
	QString								text;
	bool								valid;
	bool								init;
	bool								has_entries(false);
	
	entry = this->Get_Entry(scope_name,QStringLiteral("SOURCES"),&valid);
	
	if(valid && entry.values.size())
	{
		has_entries = true;
		init = true;
		for(iter = entry.values.begin();iter != entry.values.end();++iter)
		{
			if(init)
			{
				text = (*iter);
				init = false;
			}
			else
			{
				text += QString(" %1").arg(*iter);
			}
		}
			
		list.push_back(QStringLiteral("    if(DEFINED sources)"));
		list.push_back(QString("        list(APPEND sources %1)").arg(text));
		list.push_back(QStringLiteral("    else()"));
		list.push_back(QString("        set(sources %1)").arg(text));
		list.push_back(QStringLiteral("    endif()"));
		list.push_back(QString());
	}

	entry = this->Get_Entry(scope_name,QStringLiteral("HEADERS"),&valid);
	
	if(valid && entry.values.size())
	{
		has_entries = true;
		init = true;
		for(iter = entry.values.begin();iter != entry.values.end();++iter)
		{
			if(init)
			{
				text = (*iter);
				init = false;
			}
			else
			{
				text += QString(" %1").arg(*iter);
			}
		}
			
		list.push_back(QStringLiteral("    if(DEFINED headers)"));
		list.push_back(QString("        list(APPEND headers %1)").arg(text));
		list.push_back(QStringLiteral("    else()"));
		list.push_back(QString("        set(headers %1)").arg(text));
		list.push_back(QStringLiteral("    endif()"));
		list.push_back(QString());
	}

	entry = this->Get_Entry(scope_name,QStringLiteral("RESOURCES"),&valid);
	
	if(valid && entry.values.size())
	{
		has_entries = true;
		init = true;
		for(iter = entry.values.begin();iter != entry.values.end();++iter)
		{
			if(init)
			{
				text = (*iter);
				init = false;
			}
			else
			{
				text += QString(" %1").arg(*iter);
			}
		}
			
		list.push_back(QStringLiteral("    if(DEFINED resources)"));
		list.push_back(QString("        list(APPEND resources %1)").arg(text));
		list.push_back(QStringLiteral("    else()"));
		list.push_back(QString("        set(resources %1)").arg(text));
		list.push_back(QStringLiteral("    endif()"));
		list.push_back(QString());
	}

	entry = this->Get_Entry(scope_name,QStringLiteral("FORMS"),&valid);
	
	if(valid && entry.values.size())
	{
		has_entries = true;
		init = true;
		for(iter = entry.values.begin();iter != entry.values.end();++iter)
		{
			if(init)
			{
				text = (*iter);
				init = false;
			}
			else
			{
				text += QString(" %1").arg(*iter);
			}
		}
			
		list.push_back(QStringLiteral("    if(DEFINED forms)"));
		list.push_back(QString("        list(APPEND forms %1)").arg(text));
		list.push_back(QStringLiteral("    else()"));
		list.push_back(QString("        set(forms %1)").arg(text));
		list.push_back(QStringLiteral("    endif()"));
		list.push_back(QString());
	}
	
	if(has_entries)
	{
		d_cmake_lines.push_back(QString("if(%1)").arg(scope_name));
		d_cmake_lines.push_back(QString());

		d_cmake_lines += list;
		
		d_cmake_lines.push_back(QStringLiteral("endif()"));
		d_cmake_lines.push_back(QString());
	}
	
	return true;
}

bool TCMakeListsImpl::Process_Scoped_Libraries(
	const QString						&target_name,
	const bool 							has_sources,
	const bool 							has_headers,
	const bool 							has_resources,
	const bool 							has_forms)
{
	TCMakeListsImpl::TEntry				entry;
	QStringList::const_iterator			iter;
	QStringList							list;
	QString								default_library_type("STATIC");
	QString								library_type;
	QString								text;
	bool								init(true);
	bool								valid;
		
	// default from global scope
	entry = this->Get_Entry(QString(),QStringLiteral("CONFIG"),&valid);
	if(valid)
	{
		for(iter = entry.values.begin();iter != entry.values.end();++iter)
		{
			if((*iter).compare(QStringLiteral("dll"),Qt::CaseInsensitive) == 0)
			{
				default_library_type = QStringLiteral("SHARED");
			}
			else if((*iter).compare(QStringLiteral("staticlib"),Qt::CaseInsensitive) == 0)
			{
				default_library_type = QStringLiteral("STATIC");
			}
		}
	}
		
	library_type = default_library_type;
	entry = this->Get_Entry(QStringLiteral("LINUX"),QStringLiteral("CONFIG"),&valid);
	if(valid)
	{
		for(iter = entry.values.begin();iter != entry.values.end();++iter)
		{
			if((*iter).compare(QStringLiteral("dll"),Qt::CaseInsensitive) == 0)
			{
				library_type = QStringLiteral("SHARED");
			}
			else if((*iter).compare(QStringLiteral("staticlib"),Qt::CaseInsensitive) == 0)
			{
				library_type = QStringLiteral("STATIC");
			}
		}
		
		if(init)
		{
			d_cmake_lines.push_back(QStringLiteral("if(LINUX)"));
			init = false;
		}
		else
		{
			d_cmake_lines.push_back(QStringLiteral("elseif(LINUX)"));
		}
		
		text = QString("    add_library(%1 %2").arg(target_name).arg(library_type);

		if(has_sources)
		{
			text += QStringLiteral(" ${sources}");
		}
		
		if(has_headers)
		{
			text += QStringLiteral(" ${headers}");
		}
		
		if(has_forms)
		{
			text += QStringLiteral(" ${forms}");
		}

		if(has_resources)
		{
			text += QStringLiteral(" ${resources}");
		}
		
		text.append(')');
		
		d_cmake_lines.push_back(text);
	}
		
	library_type = default_library_type;
	entry = this->Get_Entry(QStringLiteral("APPLE"),QStringLiteral("CONFIG"),&valid);
	if(valid)
	{
		for(iter = entry.values.begin();iter != entry.values.end();++iter)
		{
			if((*iter).compare(QStringLiteral("dll"),Qt::CaseInsensitive) == 0)
			{
				library_type = QStringLiteral("SHARED");
			}
			else if((*iter).compare(QStringLiteral("staticlib"),Qt::CaseInsensitive) == 0)
			{
				library_type = QStringLiteral("STATIC");
			}
		}
		
		if(init)
		{
			d_cmake_lines.push_back(QStringLiteral("if(APPLE)"));
			init = false;
		}
		else
		{
			d_cmake_lines.push_back(QStringLiteral("elseif(APPLE)"));
		}
		
		text = QString("    add_library(%1 %2").arg(target_name).arg(library_type);

		if(has_sources)
		{
			text += QStringLiteral(" ${sources}");
		}
		
		if(has_headers)
		{
			text += QStringLiteral(" ${headers}");
		}
		
		if(has_forms)
		{
			text += QStringLiteral(" ${forms}");
		}

		if(has_resources)
		{
			text += QStringLiteral(" ${resources}");
		}
		
		text.append(')');
		
		d_cmake_lines.push_back(text);
	}
		
	library_type = default_library_type;
	entry = this->Get_Entry(QStringLiteral("UNIX"),QStringLiteral("CONFIG"),&valid);
	if(valid)
	{
		for(iter = entry.values.begin();iter != entry.values.end();++iter)
		{
			if((*iter).compare(QStringLiteral("dll"),Qt::CaseInsensitive) == 0)
			{
				library_type = QStringLiteral("SHARED");
			}
			else if((*iter).compare(QStringLiteral("staticlib"),Qt::CaseInsensitive) == 0)
			{
				library_type = QStringLiteral("STATIC");
			}
		}
		
		if(init)
		{
			d_cmake_lines.push_back(QStringLiteral("if(UNIX)"));
			init = false;
		}
		else
		{
			d_cmake_lines.push_back(QStringLiteral("elseif(UNIX)"));
		}
		
		text = QString("    add_library(%1 %2").arg(target_name).arg(library_type);

		if(has_sources)
		{
			text += QStringLiteral(" ${sources}");
		}
		
		if(has_headers)
		{
			text += QStringLiteral(" ${headers}");
		}
		
		if(has_forms)
		{
			text += QStringLiteral(" ${forms}");
		}

		if(has_resources)
		{
			text += QStringLiteral(" ${resources}");
		}
		
		text.append(')');
		
		d_cmake_lines.push_back(text);
	}
		
	library_type = default_library_type;
	entry = this->Get_Entry(QStringLiteral("WIN32"),QStringLiteral("CONFIG"),&valid);
	if(valid)
	{
		for(iter = entry.values.begin();iter != entry.values.end();++iter)
		{
			if((*iter).compare(QStringLiteral("dll"),Qt::CaseInsensitive) == 0)
			{
				library_type = QStringLiteral("SHARED");
			}
			else if((*iter).compare(QStringLiteral("staticlib"),Qt::CaseInsensitive) == 0)
			{
				library_type = QStringLiteral("STATIC");
			}
		}
		
		if(init)
		{
			d_cmake_lines.push_back(QStringLiteral("if(WIN32)"));
			init = false;
		}
		else
		{
			d_cmake_lines.push_back(QStringLiteral("elseif(WIN32)"));
		}
		
		text = QString("    add_library(%1 %2").arg(target_name).arg(library_type);

		if(has_sources)
		{
			text += QStringLiteral(" ${sources}");
		}
		
		if(has_headers)
		{
			text += QStringLiteral(" ${headers}");
		}
		
		if(has_forms)
		{
			text += QStringLiteral(" ${forms}");
		}

		if(has_resources)
		{
			text += QStringLiteral(" ${resources}");
		}
		
		text.append(')');
		
		d_cmake_lines.push_back(text);
	}
	
	if(!init)
	{
		d_cmake_lines.push_back(QStringLiteral("else()"));
		text = QString("    add_library(%1 %2").arg(target_name).arg(default_library_type);
	}
	else
	{
		text = QString("add_library(%1 %2").arg(target_name).arg(default_library_type);
	}

	if(has_sources)
	{
		text += QStringLiteral(" ${sources}");
	}
	
	if(has_headers)
	{
		text += QStringLiteral(" ${headers}");
	}
	
	if(has_forms)
	{
		text += QStringLiteral(" ${forms}");
	}

	if(has_resources)
	{
		text += QStringLiteral(" ${resources}");
	}
	
	text.append(')');
	
	d_cmake_lines.push_back(text);	

	if(!init)
	{
		d_cmake_lines.push_back(QStringLiteral("endif()"));
		d_cmake_lines.push_back(QString());
	}

	return true;
}

bool TCMakeListsImpl::Process_CMake_Target_Link_Libraries(void)
{
	TCMakeListsImpl::TEntry				entry;
	QStringList::const_iterator			iter;
	QString								text;
	QString								target_name;
	bool								valid;
	
	entry = this->Get_Entry(QString(),QStringLiteral("TARGET"),&valid);
	
	if(valid == false || entry.values.size() == 0)
	{
		d_last_error = QStringLiteral("Missing TARGET entry.");
		return false;
	}
	
	target_name = entry.values[0];

	entry = this->Get_Entry(QStringLiteral("cmake"),QStringLiteral("target_link_libraries"),&valid);
	
	if(valid)
	{	
		for(iter = entry.values.begin();iter != entry.values.end();++iter)
		{
			if((*iter).startsWith(QStringLiteral("Qt${QT_VERSION_MAJOR}")))
			{
				if((*iter).endsWith(QStringLiteral("OpenGLWidgets")))
				{
					d_cmake_lines.push_back(QStringLiteral("if(${QT_VERSION_MAJOR} GREATER 5)"));

					text = QString("    target_link_libraries(%1 PRIVATE %2)").arg(target_name).arg(*iter);
					d_cmake_lines.push_back(text);

					d_cmake_lines.push_back(QStringLiteral("endif()"));
				}
				else if((*iter).endsWith(QStringLiteral("Widgets")))
				{
					d_cmake_lines.push_back(QStringLiteral("if(${QT_VERSION_MAJOR} GREATER 4)"));
					text = QString("    target_link_libraries(%1 PRIVATE %2)").arg(target_name).arg(*iter);
					d_cmake_lines.push_back(text);

					d_cmake_lines.push_back(QStringLiteral("endif()"));
				}
				else
				{
					text = QString("target_link_libraries(%1 PRIVATE %2)").arg(target_name).arg(*iter);
					d_cmake_lines.push_back(text);
				}
			}
			else
			{
				text = QString("target_link_libraries(%1 PUBLIC %2)").arg(target_name).arg(*iter);
				d_cmake_lines.push_back(text);
			}
		}
		
		d_cmake_lines.push_back(QString());
	}
	
	return true;
}

bool TCMakeListsImpl::Process_SubDirs(void)
{
	TCMakeListsImpl::TEntry				entry;
	QStringList::const_iterator			iter;
	bool								valid;
				
	entry = this->Get_Entry(QString(),QStringLiteral("SUBDIRS"),&valid);
	
	if(valid)
	{
		for(iter = entry.values.begin();iter != entry.values.end();++iter)
		{
			d_cmake_lines.push_back(QString("add_subdirectory(%1)").arg(*iter));
		}
	
		d_cmake_lines.push_back(QString());
	}
	
	return true;
}
bool TCMakeListsImpl::Process_Linux_Entries(
	const TCMakeListsImpl::TProjectType project_type)
{
	Q_UNUSED(project_type);
	
	TCMakeListsImpl::TEntry				entry;
	QStringList							list;
	QStringList::const_iterator			iter;
	QString								text;
	QString								target_name;
	QString								name;
	bool								valid;
	bool								init;
	bool								has_entries(false);
	
	entry = this->Get_Entry(QString(),QStringLiteral("TARGET"),&valid);
	
	if(valid == false || entry.values.size() == 0)
	{
		d_last_error = QStringLiteral("Missing TARGET entry.");
		return false;
	}
	
	target_name = entry.values[0];
	
	entry = this->Get_Entry(QStringLiteral("LINUX"),QStringLiteral("QMAKE_POST_LINK"),&valid);

	if(valid)
	{
		has_entries = true;
		init = true;
		for(iter = entry.values.begin();iter != entry.values.end();++iter)
		{
			if(init)
			{
				text = (*iter);
				init = false;
			}
			else
			{
				text.append(QChar(' '));
				text += (*iter);
			}
		}
		
		list.push_back(QString("    add_custom_command(TARGET %1 POST_BUILD COMMAND %2)").arg(target_name).arg(text));
		list.push_back(QString());
	}
	
	if(has_entries)
	{
		d_cmake_lines.push_back(QStringLiteral("if(LINUX)"));
		d_cmake_lines.push_back(QString());

		d_cmake_lines += list;
		
		d_cmake_lines.push_back(QStringLiteral("endif()"));
		d_cmake_lines.push_back(QString());
	}
		
	return true;
}

bool TCMakeListsImpl::Process_Apple_Entries(
	const TCMakeListsImpl::TProjectType project_type)
{
	TCMakeListsImpl::TEntry				entry;
	QStringList							list;
	QStringList::const_iterator			iter;
	QString								text;
	QString								target_name;
	QString								name;
	bool								valid;
	bool								has_entries(false);
	
	entry = this->Get_Entry(QString(),QStringLiteral("TARGET"),&valid);
	
	if(valid == false || entry.values.size() == 0)
	{
		d_last_error = QStringLiteral("Missing TARGET entry.");
		return false;
	}
	
	target_name = entry.values[0];
	
	if(project_type == TCMakeListsImpl::PROJECT_APPLICATION)
	{
		has_entries = true;
	}

	list.push_back(QString("    set_target_properties(%1 PROPERTIES MACOSX_BUNDLE TRUE)").arg(target_name));
	list.push_back(QString("    set_target_properties(%1 PROPERTIES MACOSX_BUNDLE_BUNDLE_NAME %2)").arg(target_name).arg(target_name));
	
	entry = this->Get_Entry(QStringLiteral("APPLE"),QStringLiteral("ICON"),&valid);
	
	if(valid && entry.values.size())
	{
		name = entry.values[0];
		name.push_front(QChar('/'));
		
		list.push_back(QString("    set_target_properties(%1 PROPERTIES MACOSX_BUNDLE_ICON_FILE ${PROJECT_SOURCE_DIR}%2)").arg(target_name).arg(name));
	}
	
	list.push_back(QString());

	if(has_entries)
	{
		d_cmake_lines.push_back(QStringLiteral("if(APPLE)"));
		d_cmake_lines.push_back(QString());

		d_cmake_lines += list;
		
		d_cmake_lines.push_back(QStringLiteral("endif()"));
		d_cmake_lines.push_back(QString());
	}
		
	return true;
}

bool TCMakeListsImpl::Process_Unix_Entries(
	const TCMakeListsImpl::TProjectType project_type)
{
	Q_UNUSED(project_type);
	
	TCMakeListsImpl::TEntry				entry;
	QStringList							list;
	QStringList::const_iterator			iter;
	QString								text;
	QString								target_name;
	QString								name;
	bool								valid;
	bool								has_entries(false);

	entry = this->Get_Entry(QString(),QStringLiteral("TARGET"),&valid);
	
	if(valid == false || entry.values.size() == 0)
	{
		d_last_error = QStringLiteral("Missing TARGET entry.");
		return false;
	}
	
	target_name = entry.values[0];
	
	// currently no special unix entries
	
	if(has_entries)
	{
		d_cmake_lines.push_back(QStringLiteral("if(UNIX)"));
		d_cmake_lines.push_back(QString());

		d_cmake_lines += list;
		
		d_cmake_lines.push_back(QStringLiteral("endif()"));
		d_cmake_lines.push_back(QString());
	}


	return true;
}

bool TCMakeListsImpl::Process_Windows_Entries(
	const TCMakeListsImpl::TProjectType project_type)
{
	Q_UNUSED(project_type);
	
	TCMakeListsImpl::TEntry				entry;
	QStringList							list;
	QStringList::const_iterator			iter;
	QString								text;
	QString								target_name;
	QString								name;
	bool								valid;
	bool								has_entries(false);

	entry = this->Get_Entry(QString(),QStringLiteral("TARGET"),&valid);
	
	if(valid == false || entry.values.size() == 0)
	{
		d_last_error = QStringLiteral("Missing TARGET entry.");
		return false;
	}
	
	target_name = entry.values[0];

	entry = this->Get_Entry(QStringLiteral("WIN32"),QStringLiteral("LIBS"),&valid);
	if(valid)
	{
		has_entries = true;

		for(iter = entry.values.begin();iter != entry.values.end();++iter)
		{			
			if((*iter).startsWith(QStringLiteral("-l")))
			{
				name = (*iter).mid(2);
			}
			
			if(name.length())
			{
				text = QString("    link_libraries(%1)").arg(name);
				list.push_back(text);
	
				name.clear();
			}
		}
		
		list.push_back(QString());
	}
	
	entry = this->Get_Entry(QString(),QStringLiteral("CONFIG"),&valid);
	if(valid)
	{
		for(iter = entry.values.begin();iter != entry.values.end();++iter)
		{
			if((*iter).compare(QStringLiteral("console"),Qt::CaseInsensitive) == 0)
			{
				has_entries = true;

				text = QString("    set_property(TARGET %1 PROPERTY WIN32_EXECUTABLE FALSE)").arg(target_name);
				list.push_back(text);
				
				list.push_back(QString());
				break;
			}
		}
	}

	
	if(has_entries)
	{
		d_cmake_lines.push_back(QStringLiteral("if(WIN32)"));
		d_cmake_lines.push_back(QString());

		d_cmake_lines += list;
		
		d_cmake_lines.push_back(QStringLiteral("endif()"));
		d_cmake_lines.push_back(QString());
	}

	return true;
}

bool TCMakeListsImpl::Finalize_CMakeList_Data(
	const TCMakeListsImpl::TProjectType project_type)
{
	TCMakeListsImpl::TEntry				entry;
	QStringList::const_iterator			iter;
	QStringList							list;
	QString								target_name;
	QString								text;
	bool								valid;
	
	entry = this->Get_Entry(QString(),QStringLiteral("TARGET"),&valid);
	
	if(valid == false || entry.values.size() == 0)
	{
		d_last_error = QStringLiteral("Missing TARGET entry.");
		return false;
	}
	
	target_name = entry.values[0];
	
	if(project_type == TCMakeListsImpl::PROJECT_APPLICATION ||
		project_type == TCMakeListsImpl::PROJECT_LIBRARY)
	{
		entry = this->Get_Entry(QString(),QStringLiteral("DEFINES"),&valid);
		
		if(valid)
		{
			for(iter = entry.values.begin();iter != entry.values.end();++iter)
			{
				d_cmake_lines.push_back(QString("target_compile_definitions(%1 PRIVATE %2)").arg(target_name).arg(*iter));
			}
			d_cmake_lines.push_back(QString());
		}
		
		d_cmake_lines.push_back(QString("set_target_properties(%1 PROPERTIES DEBUG_POSTFIX \"d\")").arg(target_name));
		d_cmake_lines.push_back(QString());
	}
	
	if(project_type == TCMakeListsImpl::PROJECT_APPLICATION)
	{
		d_cmake_lines.push_back(QString("install(TARGETS %1 DESTINATION bin)").arg(target_name));
		d_cmake_lines.push_back(QString());
	}
	else if(project_type == TCMakeListsImpl::PROJECT_LIBRARY)
	{
		d_cmake_lines.push_back(QString("install(TARGETS %1 DESTINATION lib)").arg(target_name));
		d_cmake_lines.push_back(QString());
	}

	return true;
}
