/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QFrame>
#include <QGridLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QLineEdit>
#include <QPushButton>
#include <QSpacerItem>
#include <QToolButton>
#include <QTreeWidget>
#include <QPlainTextEdit>
#include <QCheckBox>
#include <QFileDialog>
#include <QTextStream>
#include <QSettings>
#include <QDragEnterEvent>
#include <QDropEvent>
#include <QMimeData>
#include <QCloseEvent>
#include <QUrl>
#include <QFile>
#include <QDir>
#include <QFont>
#include <QList>
#include <QTextCursor>
#include <assert.h>

#include "../../core/messagebox.h"

#include "sourcetree.h"

TSourceTree::TSourceTree(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QDialog(const_cast<QWidget*>(parent),flags)
{
	QFrame								*option_vline;
	QFrame								*separator_hline1;
	QFrame								*separator_hline2;
	QFrame								*separator_hline3;
	QGridLayout							*dialog_layout;
	QGridLayout							*button_layout;
	QLabel								*information_label;
	QLabel								*information_name_label;
	QLabel								*information_project_label;
	QLabel								*information_version_label;
	QLabel								*output_directory_label;
	QLabel								*output_label;
	QLabel								*sources_label;
	QLabel								*logo_label;
	QLabel								*copyright_label;
	QPushButton							*close_button;
	QSpacerItem							*information_hspacer;
	QSpacerItem							*button_hspacer;
	QTreeWidgetItem						*sources_tree_header_item;
	QFont								copyright_font;
	QFont								text_font;
	QDir								data_path;

	this->setAcceptDrops(true);

	this->resize(901,701);
	
#ifdef Q_OS_WIN
	d_data_path = QApplication::applicationDirPath();
#else
	d_data_path = QDir::homePath();
#endif
	
	data_path.setPath(d_data_path);
	
	if(!data_path.exists(QStringLiteral(".sourcetree")))
	{
		data_path.mkdir(QStringLiteral(".sourcetree"));
	}
	
	data_path.cd(QStringLiteral(".sourcetree"));
	d_data_path = data_path.absolutePath();

	d_settings = new QSettings(d_data_path + QStringLiteral("/sourcetree_settings.ini"),QSettings::IniFormat,this);

	d_default_path = d_settings->value("Default_Path", QDir::homePath()).toString();
	d_target_path = d_settings->value("Target_Path", QDir::homePath()).toString();

	copyright_font.setFamily(QStringLiteral("Helvetica"));
	copyright_font.setPointSize(8);
	copyright_font.setItalic(true);
	
	text_font.setFamily(QStringLiteral("Courier New"));
	text_font.setFixedPitch(true);

	dialog_layout = new QGridLayout(this);

	information_label = new QLabel(this);
	information_label->setAlignment(Qt::AlignLeading|Qt::AlignLeft|Qt::AlignTop);
	dialog_layout->addWidget(information_label,0,0,4,1);

	option_vline = new QFrame(this);
	option_vline->setFrameShape(QFrame::VLine);
	option_vline->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(option_vline,0,1,10,1);

	information_project_label = new QLabel(this);
	dialog_layout->addWidget(information_project_label,0,2,1,1);

	d_information_project_edit = new QLineEdit(this);
	dialog_layout->addWidget(d_information_project_edit,0,3,1,2);

	d_information_project_browse_button = new QToolButton(this);
	dialog_layout->addWidget(d_information_project_browse_button,0,5,1,1);

	information_name_label = new QLabel(this);
	dialog_layout->addWidget(information_name_label,1,2,1,1);

	d_information_name_edit = new QLineEdit(this);
	d_information_name_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_information_name_edit,1,3,1,1);

	information_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	dialog_layout->addItem(information_hspacer,1,4,2,1);

	information_version_label = new QLabel(this);
	dialog_layout->addWidget(information_version_label,2,2,1,1);

	d_information_version_edit = new QLineEdit(this);
	d_information_version_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_information_version_edit,2,3,1,1);

	separator_hline1 = new QFrame(this);
	separator_hline1->setFrameShape(QFrame::HLine);
	separator_hline1->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(separator_hline1,3,2,1,4);

	sources_label = new QLabel(this);
	sources_label->setAlignment(Qt::AlignLeading|Qt::AlignLeft|Qt::AlignTop);
	dialog_layout->addWidget(sources_label,4,0,2,1);

	d_sources_tree = new QTreeWidget(this);
	dialog_layout->addWidget(d_sources_tree,4,2,1,4);

	separator_hline2 = new QFrame(this);
	separator_hline2->setFrameShape(QFrame::HLine);
	separator_hline2->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(separator_hline2,5,2,1,4);

	output_label = new QLabel(this);
	output_label->setAlignment(Qt::AlignLeading|Qt::AlignLeft|Qt::AlignTop);
	dialog_layout->addWidget(output_label,6,0,4,1);

	output_directory_label = new QLabel(this);
	dialog_layout->addWidget(output_directory_label,6,2,1,1);

	d_output_directory_edit = new QLineEdit(this);
	dialog_layout->addWidget(d_output_directory_edit,6,3,1,2);

	d_output_directory_browse_button = new QToolButton(this);
	dialog_layout->addWidget(d_output_directory_browse_button,6,5,1,1);
	
	d_output_readme_check = new QCheckBox(this);
	dialog_layout->addWidget(d_output_readme_check,7,2,1,1);

	d_output_readme_edit = new QLineEdit(this);
	dialog_layout->addWidget(d_output_readme_edit,7,3,1,2);
	
	d_output_readme_browse_button = new QToolButton(this);
	dialog_layout->addWidget(d_output_readme_browse_button,7,5,1,1);

	d_output_toplevel_project_check = new QCheckBox(this);
	dialog_layout->addWidget(d_output_toplevel_project_check,8,2,1,4);
	
	d_log_text = new QPlainTextEdit();
	d_log_text->setFont(text_font);
	d_log_text->setLineWrapMode(QPlainTextEdit::NoWrap);
	d_log_text->setReadOnly(true);
	dialog_layout->addWidget(d_log_text,9,2,1,4);

	separator_hline3 = new QFrame(this);
	separator_hline3->setFrameShape(QFrame::HLine);
	separator_hline3->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(separator_hline3,10,0,1,6);

	button_layout = new QGridLayout();

	button_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	button_layout->addItem(button_hspacer,0,1,1,1);

	d_load_button = new QPushButton(this);
	d_load_button->setAutoDefault(false);
	button_layout->addWidget(d_load_button,0,2,1,1);

	d_create_button = new QPushButton(this);
	d_create_button->setAutoDefault(false);
	button_layout->addWidget(d_create_button,0,3,1,1);

	close_button = new QPushButton(this);
	close_button->setAutoDefault(false);
	button_layout->addWidget(close_button,0,4,1,1);
	
	logo_label = new QLabel(this);
	logo_label->setMinimumSize(100,40);
	logo_label->setMaximumSize(100,40);
	button_layout->addWidget(logo_label,0,0,2,1);
	
	copyright_label = new QLabel(this);
	copyright_label->setFont(copyright_font);
	copyright_label->setAlignment(Qt::AlignBottom|Qt::AlignRight|Qt::AlignTrailing);	// Qt::AlignLeft
	button_layout->addWidget(copyright_label,1,1,1,4);

	dialog_layout->addLayout(button_layout,11,0,1,6);
	
	d_msg_box = new TMessageBox(this);

	QWidget::setTabOrder(d_information_project_edit,d_information_project_browse_button);
	QWidget::setTabOrder(d_information_project_browse_button,d_information_name_edit);
	QWidget::setTabOrder(d_information_name_edit,d_information_version_edit);
	QWidget::setTabOrder(d_information_version_edit,d_sources_tree);
	QWidget::setTabOrder(d_sources_tree,d_output_directory_edit);
	QWidget::setTabOrder(d_output_directory_edit,d_output_directory_browse_button);
	QWidget::setTabOrder(d_output_directory_browse_button,d_output_readme_check);
	QWidget::setTabOrder(d_output_readme_check,d_output_readme_edit);
	QWidget::setTabOrder(d_output_readme_edit,d_output_readme_browse_button);
	QWidget::setTabOrder(d_output_readme_browse_button,d_output_toplevel_project_check);
	QWidget::setTabOrder(d_output_toplevel_project_check,d_load_button);
	QWidget::setTabOrder(d_load_button,d_create_button);
	QWidget::setTabOrder(d_create_button,close_button);
	
	logo_label->setPixmap(QPixmap(QStringLiteral(":/images/logo.png")));
	
	sources_tree_header_item = d_sources_tree->headerItem();

	this->setWindowTitle(QStringLiteral("Source Tree - Version 2.3"));
	
	information_label->setText(QStringLiteral("Information:"));
	d_information_project_browse_button->setText(QStringLiteral("..."));
	information_project_label->setText(QStringLiteral("Qt Project File:"));
	information_name_label->setText(QStringLiteral("Name:"));
	information_version_label->setText(QStringLiteral("Version:"));
	sources_label->setText(QStringLiteral("Sources:"));
	sources_tree_header_item->setText(0,QStringLiteral("Referenced Directories"));
	output_label->setText(QStringLiteral("Output:"));
	output_directory_label->setText(QStringLiteral("Target Directory:"));
	d_output_readme_check->setText(QStringLiteral("Include Readme:"));
	d_output_readme_browse_button->setText(QStringLiteral("..."));
	d_output_toplevel_project_check->setText(QStringLiteral("Create Top-Level Project File"));
	d_output_directory_browse_button->setText(QStringLiteral("..."));
	copyright_label->setText(QStringLiteral("Copyright (C) 2026 Select Calibration Incorporated.  www.selectcalibration.ca"));
	d_load_button->setText(QStringLiteral("Load"));
	d_create_button->setText(QStringLiteral("Create"));
	close_button->setText(QStringLiteral("Close"));
	
	// defaults
	d_output_directory_edit->setText(QDir::toNativeSeparators(d_target_path));
	d_information_version_edit->setText(d_settings->value(QStringLiteral("Version"), QStringLiteral("1.0")).toString());
	d_information_project_edit->setText(d_settings->value(QStringLiteral("Project"), QString()).toString());
	d_output_readme_edit->setText(d_settings->value(QStringLiteral("Output_Readme_File"), QString()).toString());
	d_output_readme_check->setChecked(d_settings->value(QStringLiteral("Output_Readme"), false).toBool());
	d_output_toplevel_project_check->setChecked(d_settings->value(QStringLiteral("Output_Toplevel_Project"), true).toBool());

	d_output_readme_edit->setEnabled(d_output_readme_check->isChecked());
	d_output_readme_browse_button->setEnabled(d_output_readme_check->isChecked());

	d_create_button->setEnabled(false);

	connect(close_button,&QPushButton::clicked,this,&TSourceTree::close);
	connect(d_information_project_browse_button,&QPushButton::clicked,this,&TSourceTree::Browse_Project_File);
	connect(d_output_directory_browse_button,&QPushButton::clicked,this,&TSourceTree::Browse_Output_Directory);
	connect(d_output_readme_check,&QCheckBox::toggled,this,&TSourceTree::Readme_Check_Toggled);
	connect(d_output_readme_browse_button,&QPushButton::clicked,this,&TSourceTree::Browse_Readme_File);
	connect(d_create_button,&QPushButton::clicked,this,&TSourceTree::Create_Output);
	connect(d_load_button,&QPushButton::clicked,this,&TSourceTree::Load);
}

TSourceTree::~TSourceTree(void)
{
}

void TSourceTree::Browse_Project_File(void)
{
	QString								file_name;
	QString								file_filters;
	QDir								dir;
	
	file_filters = QStringLiteral("Qt Project File(*.pro)");
	
	dir.setPath(QDir::fromNativeSeparators(d_information_project_edit->text()));
	
	if(!dir.exists())
	{
		dir.cdUp();
	}
	
	file_name = QFileDialog::getOpenFileName(this,
											 QStringLiteral("Select Qt Project file"),
											 dir.path(),
											 file_filters);
	
	if(file_name.length())
	{		
		dir.setPath(file_name);
		
		if(!dir.exists())
		{
			dir.cdUp();
		}
		
		d_default_path = QDir::cleanPath(dir.path());

		this->Load_Project(file_name);
	}
}

void TSourceTree::Browse_Output_Directory(void)
{
	QString								path;
	
	path = QFileDialog::getExistingDirectory(this,
											 QStringLiteral("Select Target Directory"),
											 d_target_path);
	
	if(path.length())
	{
		d_target_path = path;
		d_output_directory_edit->setText(QDir::toNativeSeparators(path));
	}
}

void TSourceTree::Browse_Readme_File(void)
{
	QString								file_name;
	QString								file_filters;
	
	file_filters = QString("Text files(*.txt *)");
	
	file_name = QFileDialog::getOpenFileName(this,
											 QStringLiteral("Select Readme Text File"),
											 d_default_path,
											 file_filters);
	
	if(file_name.length())
	{
		d_output_readme_edit->setText(QDir::toNativeSeparators(file_name));
	}
}

void TSourceTree::Load(void)
{
	QString								file_name;
	
	file_name = QDir::fromNativeSeparators(d_information_project_edit->text());

	if(file_name.length())
	{
		this->Load_Project(file_name);
	}
	else
	{
		this->Browse_Project_File();
	}
}

void TSourceTree::Readme_Check_Toggled(
	bool								state)
{
	d_output_readme_edit->setEnabled(state);
	d_output_readme_browse_button->setEnabled(state);
}

void TSourceTree::Create_Output(void)
{
	QString								target;
	QDir								dir;
	QString								text;
	QString								source_path;
	QString								target_path;
	QString								source_file;
	QString								target_file;
	std::vector<QString>				target_paths;
	std::vector<QString>::const_iterator iter;
	std::set<QString>::const_iterator	set_iter;
	int									count;
	int									cntr;
	int									index;
	QTreeWidgetItem						*item;
	bool								copy_error;
	
	target = QDir::fromNativeSeparators(d_output_directory_edit->text());
	
	if(!target.endsWith('/'))
	{
		target += '/';
	}
	
	target += d_information_name_edit->text();
	target += '-';
	target += d_information_version_edit->text();
	
	d_msg_box->setText(QStringLiteral("Create Source Tree"));
	d_msg_box->setInformativeText(QStringLiteral("Do you want to create the source tree?"));
	d_msg_box->setDetailedText(QDir::toNativeSeparators(target));
	d_msg_box->setIcon(QMessageBox::Information);
	d_msg_box->setStandardButtons(QMessageBox::Yes|QMessageBox::No);
	d_msg_box->setDefaultButton(QMessageBox::Yes);
	
	if(QMessageBox::No == d_msg_box->exec())
	{
		return;
	}

	if(QFile::exists(target))
	{
		this->Add_Log_Text(QString("ERR:  Source tree already exists and will not be replaced: %1").arg(QDir::toNativeSeparators(target)));

		d_msg_box->setText(QStringLiteral("Existing Source Tree"));
		d_msg_box->setInformativeText(QStringLiteral("The source tree already exists and will not be replaced."));
		d_msg_box->setDetailedText(QDir::toNativeSeparators(target));
		d_msg_box->setIcon(QMessageBox::Critical);
		d_msg_box->setStandardButtons(QMessageBox::Ok);
		d_msg_box->setDefaultButton(QMessageBox::Ok);
		
		d_msg_box->exec();
		
		return;
	}
	
	if(!dir.mkdir(target))
	{
		this->Add_Log_Text(QString("ERR:  Source tree destination cannot be created: %1").arg(QDir::toNativeSeparators(target)));

		d_msg_box->setText(QStringLiteral("Error Creating Target"));
		d_msg_box->setInformativeText(QStringLiteral("The destination cannot be created."));
		d_msg_box->setDetailedText(QDir::toNativeSeparators(target));
		d_msg_box->setIcon(QMessageBox::Critical);
		d_msg_box->setStandardButtons(QMessageBox::Ok);
		d_msg_box->setDefaultButton(QMessageBox::Ok);
		
		d_msg_box->exec();
		
		return;
	}
	
	// create target directories
	text = QDir::fromNativeSeparators(d_information_project_edit->text());
	source_path = Get_Directory_Of_File(text);
	source_path = Get_Directory_Of_File(source_path);	// moves up one directory
		
	count = d_sources_tree->topLevelItemCount();
	
	for(cntr = 0;cntr < count;++cntr)
	{
		item = d_sources_tree->topLevelItem(cntr);
		
		if(Qt::Checked == item->checkState(0))
		{
			text = item->text(0);

			assert(!(text.length() < source_path.length()));	// blowup if this is not true
			
			target_path = target;

			if(text.length() > source_path.length())
			{
				target_path += text.mid(source_path.length());
			}
			
			dir.mkpath(target_path);
		}
	}
	
	copy_error = false;
	
	// populate target directories
	for(set_iter = d_file_list.begin();set_iter != d_file_list.end();++set_iter)
	{
		source_file = (*set_iter);
		
		target_file = target;
		target_file += source_file.mid(source_path.length());
		
		text = Get_Directory_Of_File(target_file);
				
		// only copy files if the target directory exists
		if(dir.exists(text))
		{
			if(!QFile::copy(source_file,target_file))
			{
				this->Add_Log_Text(QString("ERR:  Error copying file: %1 to %2").arg(source_file).arg(target_file));
				copy_error = true;
			}
			else
			{
				if(target_file.endsWith(QStringLiteral(".pro"),Qt::CaseInsensitive))
				{
					if(d_cmake_lists.Load_QMake(target_file))
					{
						index = target_file.lastIndexOf('/');
						
						if(!(index < 0))
						{
							text = target_file.mid(0,index);
							text += QStringLiteral("/CMakeLists.txt");
							
							this->Add_Log_Text(QString("INF:  Creating CMakeLists file: %1").arg(text));

							if(!d_cmake_lists.Write_CMakeLists(text))
							{
								this->Add_Log_Text(QString("ERR:  %1").arg(d_cmake_lists.Last_Error()));
							}
						}
					}
					else
					{
						this->Add_Log_Text(QString("ERR:  %1").arg(d_cmake_lists.Last_Error()));
					}
				}
			}
			
		}
	}
	
	if(d_output_toplevel_project_check->isChecked())
	{
		this->Create_Toplevel_Project(target);
	}

	if(d_output_readme_check->isChecked())
	{
		this->Add_Readme(target);
	}
	
	if(copy_error)
	{
		this->Add_Log_Text(QStringLiteral("ERR:  Error creating source copy"));

		d_msg_box->setText(QStringLiteral("Copy Error"));
		d_msg_box->setInformativeText(QStringLiteral("One or more files could not be copied to the target directory."));
		d_msg_box->setDetailedText(QString());
		d_msg_box->setIcon(QMessageBox::Critical);
		d_msg_box->setStandardButtons(QMessageBox::Ok);
		d_msg_box->setDefaultButton(QMessageBox::Ok);
		
		d_msg_box->exec();
	}
	else
	{
		this->Add_Log_Text(QStringLiteral("INF:  Source Tree Created"));

		d_msg_box->setText(QStringLiteral("Source Tree Created"));
		d_msg_box->setInformativeText(QStringLiteral("The source tree was created."));
		d_msg_box->setDetailedText(QString());
		d_msg_box->setIcon(QMessageBox::Information);
		d_msg_box->setStandardButtons(QMessageBox::Ok);
		d_msg_box->setDefaultButton(QMessageBox::Ok);
		
		d_msg_box->exec();
	}
}

void TSourceTree::dragEnterEvent(
	QDragEnterEvent 					*drag_enter_event)
{
	if(drag_enter_event->mimeData()->hasFormat(QStringLiteral("text/uri-list")))
	{
		drag_enter_event->acceptProposedAction();
	}
}

void TSourceTree::dropEvent(
	QDropEvent 							*drop_event)
{
	QList<QUrl>							urls;
	QString								file_name;
	
	urls = drop_event->mimeData()->urls();
	
	if(urls.size() == 1)
	{
		file_name = (*urls.begin()).toLocalFile();
		
		if(file_name.length())
		{
			this->Load_Project(file_name);
		}
	}
}

void TSourceTree::closeEvent(
	QCloseEvent							*event)
{
	d_settings->setValue(QStringLiteral("Default_Path"), d_default_path);
	d_settings->setValue(QStringLiteral("Target_Path"), d_target_path);
	d_settings->setValue(QStringLiteral("Version"), d_information_version_edit->text());
	d_settings->setValue(QStringLiteral("Project"), d_information_project_edit->text());
	d_settings->setValue(QStringLiteral("Output_Readme_File"), d_output_readme_edit->text());
	d_settings->setValue(QStringLiteral("Output_Readme"), d_output_readme_check->isChecked());
	d_settings->setValue(QStringLiteral("Output_Toplevel_Project"), d_output_toplevel_project_check->isChecked());

	event->accept();
}

void TSourceTree::Load_Project(
	const QString						&file_name)
{
	d_default_path = this->Get_Directory_Of_File(file_name);
		
	d_path_directories.clear();
	d_file_list.clear();
	d_project_list.clear();
	
	this->Clear_Log();
	
	if(this->Load_Qt_Project_File(d_default_path,file_name))
	{
		this->Create_Directory_Entries();
		
		d_information_project_edit->setText(file_name);
		d_information_name_edit->setText(this->Get_Project_Name(file_name));
		d_create_button->setEnabled(d_file_list.size() > 0);
	}
	else
	{
		d_create_button->setEnabled(false);
		
		this->Add_Log_Text(QStringLiteral("ERR:  Cannot load project file."));

		d_msg_box->setText(QStringLiteral("Error"));
		d_msg_box->setInformativeText(QStringLiteral("Cannot Load Project File."));
		d_msg_box->setDetailedText(QString());
		d_msg_box->setIcon(QMessageBox::Critical);
		d_msg_box->setStandardButtons(QMessageBox::Ok);
		d_msg_box->setDefaultButton(QMessageBox::Ok);
		
		d_msg_box->exec();
	}
}

bool TSourceTree::Load_Qt_Project_File(
	const QString						&file_path,
	const QString						&file_name)
{
	std::vector<QString>				sub_directories;
	std::vector<QString>				project_files;
	std::vector<QString>				resource_files;
	std::vector<QString>				include_files;
	std::vector<QString>::const_iterator iter;
	std::vector<QString>::const_iterator resource_iter;
	std::vector<QString>::const_iterator include_iter;
	QString								path;
	QString								sub_path;
	QString								text;
	QString								sub_file_name;
	QString								sub_directory_project_file;
	
	// check if file exists.  prevent recursive loading of the same file
	if(d_file_list.find(file_name) != d_file_list.end())
	{
		this->Add_Log_Text(QString("WAR:  Already loaded project file: %1").arg(file_name));
		return true;
	}
	
	this->Add_Log_Text(QString("INF:  File path: %1").arg(file_path));
	this->Add_Log_Text(QString("INF:  Loading project file: %1").arg(file_name));
		
	if(false == (file_name.endsWith(QStringLiteral(".pro"),Qt::CaseInsensitive) ||
				 file_name.endsWith(QStringLiteral(".pri"),Qt::CaseInsensitive)))
	{
		this->Add_Log_Text(QStringLiteral("ERR:  Only files with PRO or PRI extensions are accepted."));
		return false;
	}
	
	sub_directories = this->Get_Sub_Directories(file_name);

	for(iter = sub_directories.begin();iter != sub_directories.end();++iter)
	{		
		path = file_path;
		path += '/';
		path += (*iter);
		
		d_last_error = QString();
		text = this->Search_Project_Name(path);
		this->Add_Log_Text(d_last_error);
		
		if(text.length())
		{
			sub_file_name = path;
			sub_file_name += '/';
			sub_file_name += text;
			
			this->Load_Qt_Project_File(path,sub_file_name);
		}
	}
	
	d_file_list.insert(file_name);
	
	project_files = Get_Project_Dependencies(file_name);
	
	for(iter = project_files.begin();iter != project_files.end();++iter)
	{
		text = file_path;
		text += '/';
		text += (*iter);
		
		path = QDir::cleanPath(text);
		
		text = Search_Project_Name(path);
		
		if(!text.length())	// move up one folder
		{
			path = this->Get_Directory_Of_File(path);
			text = Search_Project_Name(path);
		}
		
		if(text.length())
		{
			sub_file_name = path;
			sub_file_name += '/';
			sub_file_name += text;
			
			if(sub_file_name.length())
			{
				this->Load_Qt_Project_File(path,sub_file_name);
			}
		}
	}
		
	this->Add_Project_Name(file_name);
	
	project_files = this->Get_File_List_Qt_Project(file_name);
	
	for(iter = project_files.begin();iter != project_files.end();++iter)
	{
		text = file_path;
		text += '/';
		text += (*iter);
		
		sub_file_name = QDir::cleanPath(text);
		d_file_list.insert(sub_file_name);

		path = this->Get_Directory_Of_File(sub_file_name);
		d_path_directories.insert(path);
		
		this->Add_Log_Text(QString("INF:  Referenced file: %1").arg(sub_file_name));

		if(sub_file_name.endsWith(QStringLiteral(".qrc"),Qt::CaseInsensitive))
		{
			resource_files = Get_File_List_Qt_Resource(sub_file_name);
			
			for(resource_iter = resource_files.begin();resource_iter != resource_files.end();++resource_iter)
			{
				text = path;
				text += '/';
				text += (*resource_iter);
				
				sub_file_name = QDir::cleanPath(text);
				d_file_list.insert(sub_file_name);
				
				sub_path = this->Get_Directory_Of_File(sub_file_name);
				d_path_directories.insert(sub_path);
				
				this->Add_Log_Text(QString("INF:  Referenced file: %1").arg(sub_file_name));
			}
		}
		else if(sub_file_name.endsWith(QStringLiteral(".rc"),Qt::CaseInsensitive))
		{
			resource_files = Get_File_List_Win_Resource(sub_file_name);
			
			for(resource_iter = resource_files.begin();resource_iter != resource_files.end();++resource_iter)
			{
				text = path;
				text += '/';
				text += (*resource_iter);
				
				sub_file_name = QDir::cleanPath(text);
				d_file_list.insert(sub_file_name);
				
				sub_path = this->Get_Directory_Of_File(sub_file_name);
				d_path_directories.insert(sub_path);
				
				this->Add_Log_Text(QString("INF:  Referenced file: %1").arg(sub_file_name));
			}
		}
		else if(sub_file_name.endsWith(QStringLiteral(".pri"),Qt::CaseInsensitive))
		{
			include_files = Get_File_List_Qt_Include(sub_file_name);
			
			for(include_iter = include_files.begin();include_iter != include_files.end();++include_iter)
			{
				text = path;
				text += '/';
				text += (*include_iter);
				
				sub_file_name = QDir::cleanPath(text);
				d_file_list.insert(sub_file_name);
				
				sub_path = this->Get_Directory_Of_File(sub_file_name);
				d_path_directories.insert(sub_path);
				
				this->Add_Log_Text(QString("INF:  Referenced file: %1").arg(sub_file_name));
			}
		}
	}
	return true;
}

void TSourceTree::Clear_Log(void)
{
	d_log_text->clear();
}

void TSourceTree::Add_Log_Text(
	const QString						&text)
{
	if(text.length())
	{
		d_log_text->moveCursor(QTextCursor::End);
		d_log_text->appendPlainText(text);
	}
}

void TSourceTree::Scroll_Log_Top(void)
{
	d_log_text->moveCursor(QTextCursor::Start);
}

QString TSourceTree::Get_Directory_Of_File(
	const QString						&file_name) const
{
	QString								path;
	int									index;
	
	path = file_name;
	path.replace('\\','/');
	
	index = path.lastIndexOf('/');
	
	if(!(index < 0))
	{
		path.truncate(index);
	}
	
	return path;
}

QString	TSourceTree::Get_Project_Name(
	const QString						&file_name) const
{
	QString								path;
	QString								name;
	int									index;

	if(file_name.length())
	{
		path = this->Get_Directory_Of_File(file_name);
		
		name = file_name.mid(path.length() + 1);
		index = name.lastIndexOf('.');
		
		if(!(index < 0))
		{
			name.truncate(index);
		}
	}
	
	return name;
}

QString TSourceTree::Search_Project_Name(
	const QString						&path) const
{
	QDir								dir(path);
	QStringList							list;
	QStringList							name_filter;
	
	name_filter.push_back(QStringLiteral("*.pro"));
	
	list = dir.entryList(name_filter);
	
	if(list.size() == 1)
	{
		return list[0];
	}
	else if(list.size() > 1)
	{
		d_last_error = QString("ERR:  Too many project files in directory %1").arg(path);
	}
	else
	{
		d_last_error = QString("ERR:  No project file found in directory %1").arg(path);
	}
	
	return QString();
}

void TSourceTree::Add_Project_Name(
	const QString						&project_name)
{
	std::vector<QString>::const_iterator iter;
		
	for(iter = d_project_list.begin();iter != d_project_list.end();++iter)
	{
		if(project_name.compare(*iter) == 0)
		{
			return;
		}
	}
				
	d_project_list.push_back(project_name);
}

std::vector<QString> TSourceTree::Get_Sub_Directories(
	const QString						&file_name) const
{
	std::vector<QString>				lines;
	std::vector<QString>				sub_directories;
	std::vector<QString>::const_iterator iter;
	QStringList							list;
	QStringList::const_iterator			list_iter;
	QFile								file(file_name);
	QByteArray							file_data;
	QString								text_line;
	QString								text;
	QString								line;
	QString								sub_path;
	bool								append(false);
	
	if(file.open(QIODevice::ReadOnly))
	{
		file_data = file.readAll();
		file.close();
		
		QTextStream						text_stream(file_data,QIODevice::ReadOnly);
		
		while(!text_stream.atEnd())
		{
			text_line = text_stream.readLine().simplified();
			
			// special case variables
			text_line.remove(QStringLiteral("$$PWD/"),Qt::CaseInsensitive);	// $$PWD is always from the current PRO or PRI file

			if(append && lines.size())
			{
				text = lines[lines.size()-1];
				text += text_line;
				
				text_line = text;
				
				lines.pop_back();
			}
			
			if(text_line.length())
			{
				if(text_line.endsWith('\\'))
				{
					append = true;
					text_line.chop(1);
					text_line.append(' ');
				}
				else
				{
					append = false;
				}

				lines.push_back(text_line);
			}
		}
	}
	
	for(iter = lines.begin();iter != lines.end();++iter)
	{
		line = (*iter);
		
		if(line.startsWith(QStringLiteral("SUBDIRS")))
		{
			text = line.mid(7).trimmed();
			
			if(text.startsWith('+'))	// handle +=
			{
				text = text.mid(1);
			}
			
			if(text.startsWith('='))
			{
				text = text.mid(1).trimmed();
				
				list = text.split(' ');
				
				for(list_iter = list.begin();list_iter != list.end();++list_iter)
				{
					if((*list_iter).length())
					{
						sub_directories.push_back(*list_iter);
					}
				}
			}
		}
	}
	
	return sub_directories;
}

std::vector<QString> TSourceTree::Get_File_List_Qt_Project(
	const QString						&file_name) const
{
	std::vector<QString>				lines;
	std::vector<QString>				file_list;
	std::vector<QString>::const_iterator iter;
	QStringList							list;
	QStringList::const_iterator			list_iter;
	QFile								file(file_name);
	QByteArray							file_data;
	QString								text_line;
	QString								text;
	bool								append(false);
	bool								process_file_list(false);
	
	if(file.open(QIODevice::ReadOnly))
	{
		file_data = file.readAll();
		file.close();
		
		QTextStream						text_stream(file_data,QIODevice::ReadOnly);
		
		while(!text_stream.atEnd())
		{
			text_line = text_stream.readLine().simplified();
			text_line.remove(QStringLiteral("$$PWD/"),Qt::CaseInsensitive);	// $$PWD is always from the current PRO or PRI file

			if(append && lines.size())
			{
				text = lines[lines.size()-1];
				text += text_line;
				
				text_line = text;
				
				lines.pop_back();
			}
			
			if(text_line.length())
			{
				if(text_line.endsWith('\\'))
				{
					append = true;
					text_line.chop(1);
					text_line.append(' ');
				}
				else
				{
					append = false;
				}

				lines.push_back(text_line);
			}
		}
	}
	
	for(iter = lines.begin();iter != lines.end();++iter)
	{
		text = (*iter);
		
		process_file_list = false;
		
		if(text.startsWith(QStringLiteral("DISTFILES")))
		{
			text = text.mid(9).trimmed();
			process_file_list = true;
		}
		else if(text.startsWith(QStringLiteral("EXTRA_TRANSLATIONS")))
		{
			text = text.mid(18).trimmed();
			process_file_list = true;
		}
		else if(text.startsWith(QStringLiteral("FORMS")))
		{
			text = text.mid(5).trimmed();
			process_file_list = true;
		}
		else if(text.startsWith(QStringLiteral("HEADERS")))
		{
			text = text.mid(7).trimmed();
			process_file_list = true;
		}
		else if(text.startsWith(QStringLiteral("ICON")))
		{
			text = text.mid(4).trimmed();
			process_file_list = true;
		}
		else if(text.startsWith(QStringLiteral("INCLUDE("),Qt::CaseInsensitive))
		{
			text = text.mid(8).trimmed();
			text.chop(1);	// remove trailing brace

			file_list.push_back(text);
		}
		else if(text.startsWith(QStringLiteral("OBJECTIVE_SOURCES")))
		{
			text = text.mid(17).trimmed();
			process_file_list = true;
		}
		else if(text.startsWith(QStringLiteral("OBJECTIVE_HEADERS")))
		{
			text = text.mid(17).trimmed();
			process_file_list = true;
		}
		else if(text.startsWith(QStringLiteral("QMAKE_INFO_PLIST")))
		{
			text = text.mid(16).trimmed();
			process_file_list = true;
		}
		else if(text.startsWith(QStringLiteral("RC_FILE")))
		{
			text = text.mid(7).trimmed();
			process_file_list = true;
		}
		else if(text.startsWith(QStringLiteral("RESOURCES")))
		{
			text = text.mid(9).trimmed();
			process_file_list = true;
		}
		else if(text.startsWith(QStringLiteral("SOURCES")))
		{
			text = text.mid(7).trimmed();
			process_file_list = true;
		}

		if(process_file_list)
		{			
			if(text.startsWith('+'))	// handle +=
			{
				text = text.mid(1);
			}
			
			if(text.startsWith('='))
			{
				text = text.mid(1).trimmed();
				
				list = text.split(' ');
				
				for(list_iter = list.begin();list_iter != list.end();++list_iter)
				{					
					if((*list_iter).length())
					{
						file_list.push_back(*list_iter);
					}
				}
			}
		}
	}
	
	return file_list;
}

std::vector<QString> TSourceTree::Get_Project_Dependencies(
	const QString						&file_name) const
{
	std::vector<QString>				lines;
	std::vector<QString>				file_list;
	std::vector<QString>::const_iterator iter;
	QStringList							list;
	QStringList::const_iterator			list_iter;
	QFile								file(file_name);
	QByteArray							file_data;
	QString								text_line;
	QString								text;
	bool								append(false);
	
	if(file.open(QIODevice::ReadOnly))
	{
		file_data = file.readAll();
		file.close();
		
		QTextStream						text_stream(file_data,QIODevice::ReadOnly);
		
		while(!text_stream.atEnd())
		{
			text_line = text_stream.readLine().simplified();
			text_line.remove(QStringLiteral("$$PWD/"),Qt::CaseInsensitive);	// $$PWD is always from the current PRO or PRI file

			if(append && lines.size())
			{
				text = lines[lines.size()-1];
				text += text_line;
				
				text_line = text;
				
				lines.pop_back();
			}
			
			if(text_line.length())
			{
				if(text_line.endsWith('\\'))
				{
					append = true;
					text_line.chop(1);
					text_line.append(' ');
				}
				else
				{
					append = false;
				}
				
				lines.push_back(text_line);
			}
		}
	}
	
	for(iter = lines.begin();iter != lines.end();++iter)
	{
		text = (*iter);
		
		if(text.startsWith(QStringLiteral("LIBS")))
		{
			text = text.mid(4).trimmed();
			
			if(text.startsWith('+'))	// handle +=
			{
				text = text.mid(1);
			}
			
			if(text.startsWith('='))
			{
				text = text.mid(1).trimmed();
				
				list = text.split(' ');
				
				for(list_iter = list.begin();list_iter != list.end();++list_iter)
				{
					if((*list_iter).startsWith("-L"))
					{
						text = (*list_iter).mid(2);
						file_list.push_back(text);
					}
				}
			}
		}
	}
	
	return file_list;
}

std::vector<QString> TSourceTree::Get_File_List_Qt_Resource(
	const QString						&file_name) const
{
	std::vector<QString>				lines;
	std::vector<QString>				file_list;
	std::vector<QString>::const_iterator iter;
	QFile								file(file_name);
	QByteArray							file_data;
	QString								text_line;
	QString								text;
	
	if(file.open(QIODevice::ReadOnly))
	{
		file_data = file.readAll();
		file.close();
		
		QTextStream						text_stream(file_data,QIODevice::ReadOnly);
		
		while(!text_stream.atEnd())
		{
			text_line = text_stream.readLine().simplified();
			
			if(text_line.startsWith(QStringLiteral("<file>")) &&
			   text_line.endsWith(QStringLiteral("</file>")))
			{
				text_line = text_line.mid(6);
				text_line.truncate(text_line.length() - 7);
				
				file_list.push_back(text_line);
			}
		}
	}
	
	return file_list;
}

std::vector<QString> TSourceTree::Get_File_List_Qt_Include(
	const QString						&file_name) const
{
	std::vector<QString>				lines;
	std::vector<QString>				file_list;
	std::vector<QString>::const_iterator iter;
	QStringList							list;
	QStringList::const_iterator			list_iter;
	QFile								file(file_name);
	QByteArray							file_data;
	QString								text_line;
	QString								text;
	bool								append(false);
	bool								process_file_list(false);
	
	if(file.open(QIODevice::ReadOnly))
	{
		file_data = file.readAll();
		file.close();
		
		QTextStream						text_stream(file_data,QIODevice::ReadOnly);
		
		while(!text_stream.atEnd())
		{
			text_line = text_stream.readLine().simplified();
			text_line.remove(QStringLiteral("$$PWD/"),Qt::CaseInsensitive);	// $$PWD is always from the current PRO or PRI file

			if(append && lines.size())
			{
				text = lines[lines.size()-1];
				text += text_line;
				
				text_line = text;
				
				lines.pop_back();
			}
			
			if(text_line.length())
			{
				if(text_line.endsWith('\\'))
				{
					append = true;
					text_line.chop(1);
					text_line.append(' ');
				}
				else
				{
					append = false;
				}

				lines.push_back(text_line);
			}
		}
	}
	
	for(iter = lines.begin();iter != lines.end();++iter)
	{
		text = (*iter);
		
		process_file_list = false;
		
		if(text.startsWith(QStringLiteral("FORMS")))
		{
			text = text.mid(5).trimmed();
			process_file_list = true;
		}
		else if(text.startsWith(QStringLiteral("HEADERS")))
		{
			text = text.mid(7).trimmed();
			process_file_list = true;
		}
		else if(text.startsWith(QStringLiteral("RC_FILE")))
		{
			text = text.mid(7).trimmed();
			process_file_list = true;
		}
		else if(text.startsWith(QStringLiteral("RESOURCES")))
		{
			text = text.mid(9).trimmed();
			process_file_list = true;
		}
		else if(text.startsWith(QStringLiteral("SOURCES")))
		{
			text = text.mid(7).trimmed();
			process_file_list = true;
		}

		if(process_file_list)
		{
			if(text.startsWith('+'))	// handle +=
			{
				text = text.mid(1);
			}
			
			if(text.startsWith('='))
			{
				text = text.mid(1).trimmed();
				
				list = text.split(' ');
				
				for(list_iter = list.begin();list_iter != list.end();++list_iter)
				{
					if((*list_iter).length())
					{
						file_list.push_back(*list_iter);
					}
				}
			}
		}
	}
	
	return file_list;
}

std::vector<QString> TSourceTree::Get_File_List_Win_Resource(
	const QString						&file_name) const
{
	std::vector<QString>				lines;
	std::vector<QString>				file_list;
	std::vector<QString>::const_iterator iter;
	QStringList							list;
	QStringList::const_iterator			list_iter;
	QFile								file(file_name);
	QByteArray							file_data;
	QString								text_line;
	QString								text;
	
	if(file.open(QIODevice::ReadOnly))
	{
		file_data = file.readAll();
		file.close();
		
		QTextStream						text_stream(file_data,QIODevice::ReadOnly);
		
		while(!text_stream.atEnd())
		{
			text_line = text_stream.readLine().simplified();
			
			if(text_line.length())
			{
				list = text_line.split(' ');
				
				if(list.size() == 4)
				{
					text = list[1].trimmed();
					
					if(text.compare(QStringLiteral("ICON"),Qt::CaseInsensitive) == 0)
					{
						text = list[3];
						
						if(text.length())
						{
							text.replace('\"',' ');
							
							file_list.push_back(text.trimmed());

						}
					}
				}
			}
		}
	}
	
	return file_list;
}

void TSourceTree::Create_Directory_Entries(void)
{
	QTreeWidgetItem						*sources_tree_item;
	std::set<QString>::const_iterator	iter;
	
	d_sources_tree->clear();
	
	for(iter = d_path_directories.begin();iter != d_path_directories.end();++iter)
	{
		sources_tree_item = new QTreeWidgetItem(d_sources_tree);
		sources_tree_item->setCheckState(0,Qt::Checked);
		
		sources_tree_item->setText(0,(*iter));
	}
}

void TSourceTree::Create_Toplevel_Project(
	const QString						&target_path)
{
	std::vector<QString>::const_iterator project_iter;
	QStringList::const_iterator			list_iter;
	QStringList							list;
	QString								target_file;
	QString								sub_project_name;
	QString								text;
	QDir								dir;
	QFile								file;
	int									index;
	bool								init;
	bool								append;
		
	dir.setPath(target_path);

	// create top level project file
	target_file = target_path;
	target_file += QStringLiteral("/project.pro");
	
	file.setFileName(target_file);
	
	dir.setPath(target_path);
	
	if(file.open(QIODevice::WriteOnly | QIODevice::Text))
	{
		file.write("TEMPLATE = subdirs\n");
				
		list.clear();
		for(project_iter = d_project_list.begin();project_iter != d_project_list.end();++project_iter)
		{			
			sub_project_name = Get_Directory_Of_File(*project_iter);
			
			index = sub_project_name.lastIndexOf('/');
			
			if(!(index < 0))
			{
				sub_project_name = sub_project_name.mid(index + 1);
				
				if(dir.exists(sub_project_name))
				{
					list.push_back(sub_project_name);
				}
			}
		}
		
		list.removeDuplicates();
				
		init = true;
		append = false;
				
		for(list_iter = list.begin();list_iter != list.end();++list_iter)
		{		
			if(append)
			{
				text = " \\\n";
			}
			else
			{
				text = '\n';
			}
			
			if(init)
			{
				text += QStringLiteral("SUBDIRS =  ");
				text += (*list_iter);
				init = false;
			}
			else
			{
				text += QString("           %1").arg(*list_iter);
			}
			
			file.write(text.toLatin1());
			
			append = true;
		}
		
		file.write("\n");		
		file.close();
	}
	
	if(d_cmake_lists.Load_QMake(target_file))
	{
		target_file = target_path;
		target_file += QStringLiteral("/CMakeLists.txt");
		
		this->Add_Log_Text(QString("INF:  Creating CMakeLists file: %1").arg(target_file));

		if(!d_cmake_lists.Write_CMakeLists(target_file))
		{
			this->Add_Log_Text(QString("ERR:  %1").arg(d_cmake_lists.Last_Error()));
		}
	}
	else
	{
		this->Add_Log_Text(QString("ERR:  %1").arg(d_cmake_lists.Last_Error()));
	}
}

void TSourceTree::Add_Readme(
	const QString						&target_path)
{
	QString								source_file;
	QString								source_file_name;
	QString								target_file;
	int									index;
	
	source_file = QDir::fromNativeSeparators(d_output_readme_edit->text());
	
	if(!QFile::exists(source_file))
	{
		return;
	}
	
	index = source_file.lastIndexOf('/');
	
	if(!(index < 0))
	{
		source_file_name = source_file.mid(index);
	}
	else
	{
		source_file_name = QStringLiteral("readme.txt");
	}

	target_file = target_path;
	target_file += '/';
	target_file += source_file_name;
	
	QFile::copy(source_file,target_file);
}

