/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#ifndef MAT4HEADERFILE
#define MAT4HEADERFILE

// #define MAT4_DEBUG

#ifdef MAT4_DEBUG
#include <QString>
#endif

#include "vector3.h"

/*Mat4 Data Layout

  | 0  4  8  12 |   X  Y  Z  O
  | 1  5  9  13 |   X  Y  Z  O
  | 2  6  10 14 |   X  Y  Z  O
  | 3  7  11 15 |   -  -  p  1

  Where 12 = X Origin, 13 = Y Origin, 14 = Z Origin
         3 = X Shear,   7 = Y Shear,  11 = Perspective
  Note: This order is identical to the OpenGL layout
*/

class TMat4
{
public:
// CREATORS
	TMat4(void);
	TMat4(const TMat4&);
	TMat4(const double *);
	TMat4& operator=(const TMat4&);
	~TMat4(void);

// ACCESSORS
	const double& 	operator[](const int offset) const {return d_mat_data[offset];}
	const double* 	Get(void) const {return d_mat_data;}

	TVector3 		X(void) const;
	TVector3 		Y(void) const;
	TVector3 		Z(void) const;
	TVector3 		Origin(void) const;
	double 			Perspective(void) const;

#ifdef MAT4_DEBUG
	QString DebugData(void);
#endif

// MANIPULATORS
	void 			Set(const double*);
	double* 		Get(void) {return d_mat_data;}
	void 			Normal(void);

	double& 		operator[](const int offset) {return d_mat_data[offset];}
	TMat4 			operator*=(const TMat4&);

	void 			Identity(void);
	void 			Transpose(void);
	void			Rotate(const TVector3 &axis,const double &angle);   // angle in radians, rotation is CCW around axis.
																		// axis is in world axis
																		// Example: Rotate(TVector3(0,0,1),0.78) will rotate around the Z axis of the world.
	void			Rotate(const double &alpha,const double &beta,const double &gamma);	// rotate by euler angles
	
	void 			X(const TVector3 &X);
	void 			X(const double &Xv,const double &Yv,const double &Zv);
	void 			Y(const TVector3 &Y);
	void 			Y(const double &Xv,const double &Yv,const double &Zv);
	void 			Z(const TVector3 &Z);
	void 			Z(const double &Xv,const double &Yv,const double &Zv);
	void 			Origin(const TVector3 &O);
	void 			Origin(const double &Xv,const double &Yv,const double &Zv);
	void 			Perspective(const double &p);


private:
	double								d_mat_data[16];
};

inline TVector3 TMat4::X(void) const
{
	return TVector3(d_mat_data[0],d_mat_data[1],d_mat_data[2]);
}

inline TVector3 TMat4::Y(void) const
{
	return TVector3(d_mat_data[4],d_mat_data[5],d_mat_data[6]);
}

inline TVector3 TMat4::Z(void) const
{
	return TVector3(d_mat_data[8],d_mat_data[9],d_mat_data[10]);
}

inline TVector3 TMat4::Origin(void) const
{
	return TVector3(d_mat_data[12],d_mat_data[13],d_mat_data[14]);
}

inline double TMat4::Perspective(void) const
{
	return d_mat_data[11];
}

inline void TMat4::X(
	const TVector3 						&X)
{
	d_mat_data[0] = X.x;
	d_mat_data[1] = X.y;
	d_mat_data[2] = X.z;
}

inline void TMat4::X(
    const double                  		&Xv,
    const double                  		&Yv,
    const double                  		&Zv)
{
	d_mat_data[0] = Xv;
	d_mat_data[1] = Yv;
	d_mat_data[2] = Zv;
}

inline void TMat4::Y(
	const TVector3 						&Y)
{
	d_mat_data[4] = Y.x;
	d_mat_data[5] = Y.y;
	d_mat_data[6] = Y.z;
}

inline void TMat4::Y(
    const double                  		&Xv,
    const double                  		&Yv,
    const double                  		&Zv)
{
	d_mat_data[4] = Xv;
	d_mat_data[5] = Yv;
	d_mat_data[6] = Zv;
}

inline void TMat4::Z(
	const TVector3 						&Z)
{
	d_mat_data[8] = Z.x;
	d_mat_data[9] = Z.y;
	d_mat_data[10] = Z.z;
}

inline void TMat4::Z(
    const double                  		&Xv,
    const double                  		&Yv,
    const double                  		&Zv)
{
	d_mat_data[8] = Xv;
	d_mat_data[9] = Yv;
	d_mat_data[10] = Zv;
}

inline void TMat4::Origin(
	const TVector3 						&O)
{
	d_mat_data[12] = O.x;
	d_mat_data[13] = O.y;
	d_mat_data[14] = O.z;
}

inline void TMat4::Origin(
    const double                  		&Xo,
    const double                  		&Yo,
    const double                  		&Zo)
{
	d_mat_data[12] = Xo;
	d_mat_data[13] = Yo;
	d_mat_data[14] = Zo;
}

inline void TMat4::Perspective(
	const double 						&p)
{
	d_mat_data[11] = p;
}

// Operators
TMat4 		operator*(const TMat4&,const TMat4&);
TVector3 	operator*(const TMat4&,const TVector3&);

#endif
