/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <assert.h>

#include "mat4.h"
#include "openglgeometryfactory.h"

static const double						ZERO_EPSILON(0.000001);

TOpenGLGeometryFactory::TOpenGLGeometryFactory(void)
:d_geometry_resolution(1)
{

}

TOpenGLGeometryFactory::~TOpenGLGeometryFactory(void)
{

}

std::vector<TOpenGLGeometryFactory::TOpenGLTriangleVertex> TOpenGLGeometryFactory::RenderSphere(
	const TVector3						&position,
	const double						&radius)
{
	std::vector<TOpenGLGeometryFactory::TOpenGLTriangleVertex> triangle_points;
	TVector3							p1,p2,p3;
	
	if(d_geometry_resolution < 0 || radius < ZERO_EPSILON)
	{
		return triangle_points;
	}
	
	p1.Set(0,0,1);
	p2.Set(1,0,0);
	p3.Set(0,1,0);
	this->RenderSphereTriangle(p1,p2,p3,position,radius,d_geometry_resolution,&triangle_points);
	
	return triangle_points;
}

std::vector<TOpenGLGeometryFactory::TOpenGLTriangleVertex> TOpenGLGeometryFactory::RenderCylinder(
	const TVector3 						&position,
	const TVector3	 					&axis,
	const double 						&radius,
	const double						&length)
{
	std::vector<TOpenGLGeometryFactory::TOpenGLTriangleVertex> triangle_points;
	std::vector<TOpenGLGeometryFactory::TOpenGLTriangleVertex> plane_points;
	TVector3							p1,p2,p3;
	TMat4								rotation_mat;
	double								half_length(length / 2.0);
	
	if(d_geometry_resolution < 0 || radius < ZERO_EPSILON || axis.Length() < ZERO_EPSILON)
	{
		return triangle_points;
	}
	
	p1 = axis;
	p1.Normal();
	
	p2 = p1 * TVector3(1,0,0);
	
	if(p2.Length() < 0.5)
	{
		p2 = p1 * TVector3(0,1,0);
	}
	p2.Normal();
	
	p3 = p2 * p1;
	
	rotation_mat.Z(p1);
	rotation_mat.Y(p2);
	rotation_mat.X(p3);
	rotation_mat.Transpose();
	
	// First End
	p1.Set(1,0,half_length);
	p3.Set(0,1,half_length);
	p2.Set(0.707107,0.707107,-half_length);
	this->RenderCircleTriangle(p1,p2,p3,position,radius,d_geometry_resolution,rotation_mat,&triangle_points);

	p1.Set(0,0,half_length);
	p1 = position + rotation_mat * p1;
	plane_points = this->RenderCircularPlane(p1,axis,rotation_mat.X(),radius);
	triangle_points.insert(triangle_points.end(),plane_points.begin(),plane_points.end());

	// Second End
	p1.Set(0.707107,0.707107,-half_length);
	p2.Set(-0.707107,0.707107,-half_length);
	p3.Set(0,1,half_length);
	this->RenderCircleTriangle(p1,p2,p3,position,radius,d_geometry_resolution,rotation_mat,&triangle_points);

	p1.Set(0,0,-half_length);
	p1 = position + rotation_mat * p1;
	plane_points = this->RenderCircularPlane(p1,-1 * axis,rotation_mat.X() + rotation_mat.Y(),radius);
	triangle_points.insert(triangle_points.end(),plane_points.begin(),plane_points.end());
	
	return triangle_points;
}

std::vector<TOpenGLGeometryFactory::TOpenGLTriangleVertex> TOpenGLGeometryFactory::RenderPlane(
	const TVector3 						&p1,
	const TVector3 						&p2,
	const TVector3 						&p3,
	const TVector3 						&p4)
{
	std::vector<TOpenGLGeometryFactory::TOpenGLTriangleVertex> triangle_points;
	TVector3							middle_point;
	
	if(d_geometry_resolution < 0 )
	{
		return triangle_points;
	}
	
	middle_point = (p1 + p2 + p3 + p4) / 4.0;
	
	this->RenderPlaneTriangle(p1,p2,middle_point,d_geometry_resolution,&triangle_points);
	this->RenderPlaneTriangle(p2,p3,middle_point,d_geometry_resolution,&triangle_points);
	this->RenderPlaneTriangle(p3,p4,middle_point,d_geometry_resolution,&triangle_points);
	this->RenderPlaneTriangle(p4,p1,middle_point,d_geometry_resolution,&triangle_points);
	
	return triangle_points;
}

std::vector<TOpenGLGeometryFactory::TOpenGLTriangleVertex> TOpenGLGeometryFactory::RenderCircularPlane(
	const TVector3 						&position,
	const TVector3 						&axis,
	const TVector3						&reference_axis,
	const double 						&radius)
{
	std::vector<TOpenGLGeometryFactory::TOpenGLTriangleVertex> triangle_points;
	TVector3							p1,p2,p3;
	TVector3							p_normal;
	TVector3							p_center;
	TMat4								rotation_mat;
	TMat4								rotation_angle_mat;
	double								angle_increment;
	int									fan_vertex_count;
	int									fan_vertex_cntr;
	int									point_count;
	int									cntr;
	TVector3							*left_points;
	TVector3							*right_points;
	TOpenGLGeometryFactory::TOpenGLTriangleVertex	vertex;
	const double						PI(3.14159265358979323846);
	
	if(d_geometry_resolution < 0 || radius < ZERO_EPSILON || axis.Length() < ZERO_EPSILON)
	{
		return triangle_points;
	}
	
	fan_vertex_count = 4 * static_cast<int>(0.5 + pow(2,static_cast<double>(d_geometry_resolution)));
	angle_increment = 2 * PI / static_cast<double>(fan_vertex_count);
	point_count = static_cast<int>(pow(2,static_cast<double>(d_geometry_resolution))) - 1;
	
	p1 = axis;
	p1.Normal();
	
	p2 = p1 * reference_axis;
	
	if(p2.Length() < 0.001)
	{
		p3.Set(reference_axis.y,reference_axis.z,reference_axis.x);
		p2 = p1 * p3;
	}
	p2.Normal();
	
	p3 = p2 * p1;
	
	rotation_mat.Z(p1);
	rotation_mat.Y(p2);
	rotation_mat.X(p3);
	rotation_mat.Transpose();
	
	left_points = new TVector3[1 + point_count];
	right_points = new TVector3[1 + point_count];
	
	rotation_angle_mat.Rotate(TVector3(0,0,1),angle_increment);
	
	for(cntr = 0;cntr <= point_count;++cntr)
	{
		right_points[cntr].x = static_cast<double>(cntr + 1)/static_cast<double>(point_count + 1);
		right_points[cntr].y = 0;
		right_points[cntr].z = 0;
		
		left_points[cntr] = rotation_angle_mat * right_points[cntr];
	}
	
	p_center.Set(0,0,0);
	p_center = rotation_mat * p_center;
	p_center += position;
	p_normal.Set(0,0,1);
	p_normal = rotation_mat * p_normal;
	
	for(fan_vertex_cntr = 0;fan_vertex_cntr < fan_vertex_count;++fan_vertex_cntr)
	{
		vertex.normal = p_normal;
		vertex.vertex = p_center;
		triangle_points.push_back(vertex);
		
		p2 = position + rotation_mat * (right_points[0] * radius);
		p1 = position + rotation_mat * (left_points[0] * radius);

		vertex.normal = p_normal;
		vertex.vertex = p1;
		triangle_points.push_back(vertex);

		vertex.normal = p_normal;
		vertex.vertex = p2;
		triangle_points.push_back(vertex);

		for(cntr = 0;cntr < point_count;++cntr)
		{
			p1 = position + rotation_mat * (right_points[cntr] * radius);
			p3 = position + rotation_mat * (right_points[cntr + 1] * radius);
			p2 = position + rotation_mat * (left_points[cntr + 1] * radius);
			
			vertex.normal = p_normal;
			vertex.vertex = p1;
			triangle_points.push_back(vertex);
			
			vertex.normal = p_normal;
			vertex.vertex = p2;
			triangle_points.push_back(vertex);
			
			vertex.normal = p_normal;
			vertex.vertex = p3;
			triangle_points.push_back(vertex);
			
			p1 = position + rotation_mat * (left_points[cntr + 1] * radius);
			p3 = position + rotation_mat * (left_points[cntr] * radius);
			p2 = position + rotation_mat * (right_points[cntr] * radius);
			
			vertex.normal = p_normal;
			vertex.vertex = p1;
			triangle_points.push_back(vertex);
			
			vertex.normal = p_normal;
			vertex.vertex = p2;
			triangle_points.push_back(vertex);
			
			vertex.normal = p_normal;
			vertex.vertex = p3;
			triangle_points.push_back(vertex);
		}

		for(cntr = 0;cntr <= point_count;++cntr)
		{
			right_points[cntr] = left_points[cntr];
			left_points[cntr] = rotation_angle_mat * right_points[cntr];
		}
	}
	
	delete[] right_points;
	delete[] left_points;
	
	return triangle_points;
}

std::vector<TOpenGLGeometryFactory::TOpenGLTriangleVertex> TOpenGLGeometryFactory::RenderPolygon(
	const TVector3 						&position,
	const TVector3 						&axis,
	const TVector3						&reference_axis,
	const double						&max_radius,
	const double 						&length,
	const int 							&side_count)
{
	std::vector<TOpenGLGeometryFactory::TOpenGLTriangleVertex> triangle_points;
	std::vector<TOpenGLGeometryFactory::TOpenGLTriangleVertex> plane_points;
	TVector3							p1,p2,p3,p4;
	TVector3							pa,pb,pc,pd;
	TVector3							p_normal;
	TVector3							p_center;
	TMat4								rotation_mat;
	double								half_length(length / 2);
	double								angle_increment;
	TMat4								rotation_angle_mat;
	int									face_cntr;
	int									cntr;
	int									point_count;
	TVector3							*left_points;
	TVector3							*right_points;
	TOpenGLGeometryFactory::TOpenGLTriangleVertex		vertex;
	const double						PI(3.14159265358979323846);
	
	if(d_geometry_resolution < 0 || max_radius < ZERO_EPSILON || side_count < 4 || axis.Length() < ZERO_EPSILON || reference_axis.Length() < ZERO_EPSILON)
	{
		return triangle_points;
	}
	
	angle_increment = 2 * PI / static_cast<double>(side_count);
	point_count = static_cast<int>(pow(2,static_cast<double>(d_geometry_resolution))) - 1;
	
	p1 = axis;
	p1.Normal();
	
	p2 = p1 * reference_axis;
	
	if(p2.Length() < 0.001)
	{
		p3.Set(reference_axis.y,reference_axis.z,reference_axis.x);
		p2 = p1 * p3;
	}
	p2.Normal();
	
	p3 = p2 * p1;
	
	rotation_mat.Z(p1);
	rotation_mat.Y(p2);
	rotation_mat.X(p3);
	rotation_mat.Transpose();
	
	rotation_angle_mat.Rotate(TVector3(0,0,1),angle_increment);
	
	p1.Set(max_radius,0,-half_length);
	p2.Set(max_radius,0,half_length);
	
	for(cntr = 0;cntr < side_count;++cntr)
	{
		p3 = rotation_angle_mat * p2;
		p4 = rotation_angle_mat * p1;
		
		pa = position + rotation_mat * p1;
		pb = position + rotation_mat * p2;
		pc = position + rotation_mat * p3;
		pd = position + rotation_mat * p4;
		
		plane_points = this->RenderPlane(pa,pb,pc,pd);
		triangle_points.insert(triangle_points.end(),plane_points.begin(),plane_points.end());
		
		p1 = p4;
		p2 = p3;
	}
	
	angle_increment = 2 * PI / static_cast<double>(side_count);
	
	left_points = new TVector3[1 + point_count];
	right_points = new TVector3[1 + point_count];
	
	rotation_angle_mat.Identity();
	rotation_angle_mat.Rotate(TVector3(0,0,1),angle_increment);
	
	for(cntr = 0;cntr <= point_count;++cntr)
	{
		right_points[cntr].x = static_cast<double>(cntr + 1)/static_cast<double>(point_count + 1);
		right_points[cntr].y = 0;
		right_points[cntr].z = 0;
		
		left_points[cntr] = rotation_angle_mat * right_points[cntr];
	}
	
	p_center.Set(0,0,half_length);
	p_center = rotation_mat * p_center;
	p_center += position;
	p_normal.Set(0,0,1);
	p_normal = rotation_mat * p_normal;
	for(face_cntr = 0;face_cntr < side_count;++face_cntr)
	{
		vertex.normal = p_normal;
		vertex.vertex = p_center;
		triangle_points.push_back(vertex);
		
		p2 = (right_points[0] * max_radius);
		p2.z = half_length;
		p2 = position + rotation_mat * p2;
		
		
		p1 = (left_points[0] * max_radius);
		p1.z = half_length;
		p1 = position + rotation_mat * p1;
		
		vertex.normal = p_normal;
		vertex.vertex = p1;
		triangle_points.push_back(vertex);
		
		vertex.normal = p_normal;
		vertex.vertex = p2;
		triangle_points.push_back(vertex);
		
		for(cntr = 0;cntr < point_count;++cntr)
		{
			p1 = (right_points[cntr] * max_radius);
			p1.z = half_length;
			p1 = position + rotation_mat * p1;
			
			p3 = (right_points[cntr + 1] * max_radius);
			p3.z = half_length;
			p3 = position + rotation_mat * p3;
			
			p2 = (left_points[cntr + 1] * max_radius);
			p2.z = half_length;
			p2 = position + rotation_mat * p2;
			
			vertex.normal = p_normal;
			vertex.vertex = p1;
			triangle_points.push_back(vertex);
			
			vertex.normal = p_normal;
			vertex.vertex = p2;
			triangle_points.push_back(vertex);
			
			vertex.normal = p_normal;
			vertex.vertex = p3;
			triangle_points.push_back(vertex);
			
			p1 = (left_points[cntr + 1] * max_radius);
			p1.z = half_length;
			p1 = position + rotation_mat * p1;
			
			p3 = (left_points[cntr] * max_radius);
			p3.z = half_length;
			p3 = position + rotation_mat * p3;
			
			p2 = (right_points[cntr] * max_radius);
			p2.z = half_length;
			p2 = position + rotation_mat * p2;
			
			vertex.normal = p_normal;
			vertex.vertex = p1;
			triangle_points.push_back(vertex);
			
			vertex.normal = p_normal;
			vertex.vertex = p2;
			triangle_points.push_back(vertex);
			
			vertex.normal = p_normal;
			vertex.vertex = p3;
			triangle_points.push_back(vertex);
		}
		
		for(cntr = 0;cntr <= point_count;++cntr)
		{
			right_points[cntr] = left_points[cntr];
			left_points[cntr] = rotation_angle_mat * right_points[cntr];
		}
	}
	
	p_center.Set(0,0,-half_length);
	p_center = rotation_mat * p_center;
	p_center += position;
	p_normal.Set(0,0,-1);
	p_normal = rotation_mat * p_normal;
	for(face_cntr = 0;face_cntr < side_count;++face_cntr)
	{
		vertex.normal = p_normal;
		vertex.vertex = p_center;
		triangle_points.push_back(vertex);
		
		p1 = (right_points[0] * max_radius);
		p1.z = -half_length;
		p1 = position + rotation_mat * p1;
		
		p2 = (left_points[0] * max_radius);
		p2.z = -half_length;
		p2 = position + rotation_mat * p2;
		
		vertex.normal = p_normal;
		vertex.vertex = p1;
		triangle_points.push_back(vertex);
		
		vertex.normal = p_normal;
		vertex.vertex = p2;
		triangle_points.push_back(vertex);
		
		for(cntr = 0;cntr < point_count;++cntr)
		{
			p1 = (right_points[cntr] * max_radius);
			p1.z = -half_length;
			p1 = position + rotation_mat * p1;
			
			p2 = (right_points[cntr + 1] * max_radius);
			p2.z = -half_length;
			p2 = position + rotation_mat * p2;
			
			p3 = (left_points[cntr + 1] * max_radius);
			p3.z = -half_length;
			p3 = position + rotation_mat * p3;
			
			vertex.normal = p_normal;
			vertex.vertex = p1;
			triangle_points.push_back(vertex);
			
			vertex.normal = p_normal;
			vertex.vertex = p2;
			triangle_points.push_back(vertex);
			
			vertex.normal = p_normal;
			vertex.vertex = p3;
			triangle_points.push_back(vertex);
			
			p1 = (left_points[cntr + 1] * max_radius);
			p1.z = -half_length;
			p1 = position + rotation_mat * p1;
			
			p2 = (left_points[cntr] * max_radius);
			p2.z = -half_length;
			p2 = position + rotation_mat * p2;
			
			p3 = (right_points[cntr] * max_radius);
			p3.z = -half_length;
			p3 = position + rotation_mat * p3;
			
			vertex.normal = p_normal;
			vertex.vertex = p1;
			triangle_points.push_back(vertex);
			
			vertex.normal = p_normal;
			vertex.vertex = p2;
			triangle_points.push_back(vertex);
			
			vertex.normal = p_normal;
			vertex.vertex = p3;
			triangle_points.push_back(vertex);
			
		}
		
		for(cntr = 0;cntr <= point_count;++cntr)
		{
			right_points[cntr] = left_points[cntr];
			left_points[cntr] = rotation_angle_mat * right_points[cntr];
		}
	}
	
	delete[] right_points;
	delete[] left_points;

	return triangle_points;
}

void TOpenGLGeometryFactory::RenderSphereTriangle(
	const TVector3 						&p1,
	const TVector3 						&p2,
	const TVector3 						&p3,
	const TVector3						&position,
	const double						&radius,
	const int 							resolution,
	std::vector<TOpenGLGeometryFactory::TOpenGLTriangleVertex>	* const triangle_points) const
{
	TVector3 							pa,pb,pc,pt;
	TOpenGLGeometryFactory::TOpenGLTriangleVertex vertex;
	
	if(resolution > 0)
	{
		pa = (p1 + p2) / 2;
		pb = (p2 + p3) / 2;
		pc = (p3 + p1) / 2;
		
		this->RenderSphereTriangle(p1,pa,pc,position,radius,resolution - 1,triangle_points);
		this->RenderSphereTriangle(pa,p2,pb,position,radius,resolution - 1,triangle_points);
		this->RenderSphereTriangle(pa,pb,pc,position,radius,resolution - 1,triangle_points);
		this->RenderSphereTriangle(pb,p3,pc,position,radius,resolution - 1,triangle_points);
	}
	else
	{
		pa = p1;
		pb = p2;
		pc = p3;
		
		pa.Normal();
		pb.Normal();
		pc.Normal();
		
		// Draws one triangle in each of the eight quadrants
		
		pt = pa * radius;
		
		vertex.normal = pa;
		vertex.vertex = position + pt;
		triangle_points->push_back(vertex);
		
		pt = pb * radius;
		
		vertex.normal = pb;
		vertex.vertex = position + pt;
		triangle_points->push_back(vertex);
		
		pt = pc * radius;
		
		vertex.normal = pc;
		vertex.vertex = position + pt;
		triangle_points->push_back(vertex);
		
		pt = pa; pa.i = -pt.j; pa.j = pt.i;
		pt = pb; pb.i = -pt.j; pb.j = pt.i;
		pt = pc; pc.i = -pt.j; pc.j = pt.i;
		
		pt = pa * radius;
		
		vertex.normal = pa;
		vertex.vertex = position + pt;
		triangle_points->push_back(vertex);
		
		pt = pb * radius;
		
		vertex.normal = pb;
		vertex.vertex = position + pt;
		triangle_points->push_back(vertex);
		
		pt = pc * radius;
		
		vertex.normal = pc;
		vertex.vertex = position + pt;
		triangle_points->push_back(vertex);
		
		pt = pa; pa.i = -pt.j; pa.j = pt.i;
		pt = pb; pb.i = -pt.j; pb.j = pt.i;
		pt = pc; pc.i = -pt.j; pc.j = pt.i;
		
		pt = pa * radius;
		
		vertex.normal = pa;
		vertex.vertex = position + pt;
		triangle_points->push_back(vertex);
		
		pt = pb * radius;
		
		vertex.normal = pb;
		vertex.vertex = position + pt;
		triangle_points->push_back(vertex);
		
		pt = pc * radius;
		
		vertex.normal = pc;
		vertex.vertex = position + pt;
		triangle_points->push_back(vertex);
		
		pt = pa; pa.i = -pt.j; pa.j = pt.i;
		pt = pb; pb.i = -pt.j; pb.j = pt.i;
		pt = pc; pc.i = -pt.j; pc.j = pt.i;
		
		pt = pa * radius;
		
		vertex.normal = pa;
		vertex.vertex = position + pt;
		triangle_points->push_back(vertex);
		
		pt = pb * radius;
		
		vertex.normal = pb;
		vertex.vertex = position + pt;
		triangle_points->push_back(vertex);
		
		pt = pc * radius;
		
		vertex.normal = pc;
		vertex.vertex = position + pt;
		triangle_points->push_back(vertex);
		
		pa *= -1;
		pb *= -1;
		pc *= -1;
		
		pt = pb * radius;
		
		vertex.normal = pb;
		vertex.vertex = position + pt;
		triangle_points->push_back(vertex);
		
		pt = pa * radius;
		
		vertex.normal = pa;
		vertex.vertex = position + pt;
		triangle_points->push_back(vertex);
		
		pt = pc * radius;
		
		vertex.normal = pc;
		vertex.vertex = position + pt;
		triangle_points->push_back(vertex);
		
		pt = pa; pa.i = -pt.j; pa.j = pt.i;
		pt = pb; pb.i = -pt.j; pb.j = pt.i;
		pt = pc; pc.i = -pt.j; pc.j = pt.i;
		
		pt = pb * radius;
		
		vertex.normal = pb;
		vertex.vertex = position + pt;
		triangle_points->push_back(vertex);
		
		pt = pa * radius;
		
		vertex.normal = pa;
		vertex.vertex = position + pt;
		triangle_points->push_back(vertex);
		
		pt = pc * radius;
		
		vertex.normal = pc;
		vertex.vertex = position + pt;
		triangle_points->push_back(vertex);
		
		pt = pa; pa.i = -pt.j; pa.j = pt.i;
		pt = pb; pb.i = -pt.j; pb.j = pt.i;
		pt = pc; pc.i = -pt.j; pc.j = pt.i;
		
		pt = pb * radius;
		
		vertex.normal = pb;
		vertex.vertex = position + pt;
		triangle_points->push_back(vertex);
		
		pt = pa * radius;
		
		vertex.normal = pa;
		vertex.vertex = position + pt;
		triangle_points->push_back(vertex);
		
		pt = pc * radius;
		
		vertex.normal = pc;
		vertex.vertex = position + pt;
		triangle_points->push_back(vertex);
		
		pt = pa; pa.i = -pt.j; pa.j = pt.i;
		pt = pb; pb.i = -pt.j; pb.j = pt.i;
		pt = pc; pc.i = -pt.j; pc.j = pt.i;
		
		pt = pb * radius;
		
		vertex.normal = pb;
		vertex.vertex = position + pt;
		triangle_points->push_back(vertex);
		
		pt = pa * radius;
		
		vertex.normal = pa;
		vertex.vertex = position + pt;
		triangle_points->push_back(vertex);
		
		pt = pc * radius;
		
		vertex.normal = pc;
		vertex.vertex = position + pt;
		triangle_points->push_back(vertex);
	}
}

void TOpenGLGeometryFactory::RenderCircleTriangle(
	const TVector3 						&p1,
	const TVector3 						&p2,
	const TVector3 						&p3,
	const TVector3 						&position,
	const double 						&radius,
	const int 							resolution,
	const TMat4 						&rotation_mat,
	std::vector<TOpenGLGeometryFactory::TOpenGLTriangleVertex>	* const triangle_points) const
{
	TVector3 							pa,pb,pc,vec,pt;
	TOpenGLGeometryFactory::TOpenGLTriangleVertex vertex;
	
	if(resolution > 0)
	{
		pa = (p1 + p2) / 2;
		pb = (p2 + p3) / 2;
		pc = (p3 + p1) / 2;
		
		this->RenderCircleTriangle(p1,this->Normalize_XY(pa),this->Normalize_XY(pc),position,radius,resolution - 1,rotation_mat,triangle_points);
		this->RenderCircleTriangle(this->Normalize_XY(pa),p2,this->Normalize_XY(pb),position,radius,resolution - 1,rotation_mat,triangle_points);
		this->RenderCircleTriangle(this->Normalize_XY(pa),this->Normalize_XY(pb),this->Normalize_XY(pc),position,radius,resolution - 1,rotation_mat,triangle_points);
		this->RenderCircleTriangle(this->Normalize_XY(pb),p3,this->Normalize_XY(pc),position,radius,resolution - 1,rotation_mat,triangle_points);
	}
	else
	{
		pa.Set(p1.x,p1.y,0);
		pb.Set(p2.x,p2.y,0);
		pc.Set(p3.x,p3.y,0);
		
		pa.Normal();
		pb.Normal();
		pc.Normal();
		
		// Draws one triangle in each of the four quadrants
		
		vec = rotation_mat * pa;
		
		pt = pa * radius;
		pt.z = p1.z;
		pt = rotation_mat * pt;
		pt += position;
		
		vertex.normal = vec;
		vertex.vertex = pt;
		triangle_points->push_back(vertex);
		
		vec = rotation_mat * pb;
		
		pt = pb * radius;
		pt.z = p2.z;
		pt = rotation_mat * pt;
		pt += position;
		
		vertex.normal = vec;
		vertex.vertex = pt;
		triangle_points->push_back(vertex);
		
		vec = rotation_mat * pc;
		
		pt = pc * radius;
		pt.z = p3.z;
		pt = rotation_mat * pt;
		pt += position;
		
		vertex.normal = vec;
		vertex.vertex = pt;
		triangle_points->push_back(vertex);
		
		pt = pa; pa.i = -pt.j; pa.j = pt.i;
		pt = pb; pb.i = -pt.j; pb.j = pt.i;
		pt = pc; pc.i = -pt.j; pc.j = pt.i;
		
		vec = rotation_mat * pa;
		
		pt = pa * radius;
		pt.z = p1.z;
		pt = rotation_mat * pt;
		pt += position;
		
		vertex.normal = vec;
		vertex.vertex = pt;
		triangle_points->push_back(vertex);
		
		vec = rotation_mat * pb;
		
		pt = pb * radius;
		pt.z = p2.z;
		pt = rotation_mat * pt;
		pt += position;
		
		vertex.normal = vec;
		vertex.vertex = pt;
		triangle_points->push_back(vertex);
		
		vec = rotation_mat * pc;
		
		pt = pc * radius;
		pt.z = p3.z;
		pt = rotation_mat * pt;
		pt += position;
		
		vertex.normal = vec;
		vertex.vertex = pt;
		triangle_points->push_back(vertex);
		
		pt = pa; pa.i = -pt.j; pa.j = pt.i;
		pt = pb; pb.i = -pt.j; pb.j = pt.i;
		pt = pc; pc.i = -pt.j; pc.j = pt.i;
		
		vec = rotation_mat * pa;
		
		pt = pa * radius;
		pt.z = p1.z;
		pt = rotation_mat * pt;
		pt += position;
		
		vertex.normal = vec;
		vertex.vertex = pt;
		triangle_points->push_back(vertex);
		
		vec = rotation_mat * pb;
		
		pt = pb * radius;
		pt.z = p2.z;
		pt = rotation_mat * pt;
		pt += position;
		
		
		vertex.normal = vec;
		vertex.vertex = pt;
		triangle_points->push_back(vertex);
		
		vec = rotation_mat * pc;
		
		pt = pc * radius;
		pt.z = p3.z;
		pt = rotation_mat * pt;
		pt += position;
		
		vertex.normal = vec;
		vertex.vertex = pt;
		triangle_points->push_back(vertex);
		
		pt = pa; pa.i = -pt.j; pa.j = pt.i;
		pt = pb; pb.i = -pt.j; pb.j = pt.i;
		pt = pc; pc.i = -pt.j; pc.j = pt.i;
		
		vec = rotation_mat * pa;
		
		pt = pa * radius;
		pt.z = p1.z;
		pt = rotation_mat * pt;
		pt += position;
		
		vertex.normal = vec;
		vertex.vertex = pt;
		triangle_points->push_back(vertex);
		
		vec = rotation_mat * pb;
		
		pt = pb * radius;
		pt.z = p2.z;
		pt = rotation_mat * pt;
		pt += position;
		
		vertex.normal = vec;
		vertex.vertex = pt;
		triangle_points->push_back(vertex);
		
		vec = rotation_mat * pc;
		
		pt = pc * radius;
		pt.z = p3.z;
		pt = rotation_mat * pt;
		pt += position;
		
		vertex.normal = vec;
		vertex.vertex = pt;
		triangle_points->push_back(vertex);
	}
}

void TOpenGLGeometryFactory::RenderPlaneTriangle(
	const TVector3 						&p1,
	const TVector3 						&p2,
	const TVector3 						&p3,
	const int 							resolution,
	std::vector<TOpenGLGeometryFactory::TOpenGLTriangleVertex>	* const triangle_points) const
{
	TVector3 							pa,pb,pc;
	TVector3							v1,v2,p_normal;
	TOpenGLGeometryFactory::TOpenGLTriangleVertex vertex;
	
	if(resolution > 0)
	{
		pa = (p1 + p2) / 2;
		pb = (p2 + p3) / 2;
		pc = (p3 + p1) / 2;
		
		this->RenderPlaneTriangle(p1,pa,pc,resolution - 1,triangle_points);
		this->RenderPlaneTriangle(pa,p2,pb,resolution - 1,triangle_points);
		this->RenderPlaneTriangle(pa,pb,pc,resolution - 1,triangle_points);
		this->RenderPlaneTriangle(pb,p3,pc,resolution - 1,triangle_points);
	}
	else
	{
		pa = p1;
		pb = p2;
		pc = p3;
		
		v1 = pb - pa;
		v2 = pc - pb;
		p_normal = v1 * v2;
		
		if(p_normal.Length() > ZERO_EPSILON)
		{
			p_normal.Normal();
		}
		else
		{
			p_normal.Set(0,0,1);
		}
		
		vertex.normal = p_normal;
		vertex.vertex = pa;
		triangle_points->push_back(vertex);
		
		vertex.normal = p_normal;
		vertex.vertex = pb;
		triangle_points->push_back(vertex);
		
		vertex.normal = p_normal;
		vertex.vertex = pc;
		triangle_points->push_back(vertex);
	}
}


void TOpenGLGeometryFactory::RenderPlaneTriangle(
	const TVector3 						&p1,
	const TVector3 						&p2,
	const TVector3 						&p3,
	const double 						&radius_1,
	const double 						&radius_2,
	const double 						&radius_3,
	const int 							resolution,
	std::vector<TOpenGLGeometryFactory::TOpenGLTriangleVertex>	* const triangle_points) const
{
	TVector3 							pa,pb,pc;
	double								ra,rb,rc;
	TVector3							v1,v2,p_normal;
	TOpenGLGeometryFactory::TOpenGLTriangleVertex vertex;
	
	if(resolution > 0)
	{
		pa = (p1 + p2) / 2;
		pb = (p2 + p3) / 2;
		pc = (p3 + p1) / 2;
		
		ra = (radius_1 + radius_2)/2;
		rb = (radius_2 + radius_3)/2;
		rc = (radius_3 + radius_1)/2;
		
		this->RenderPlaneTriangle(pa,pb,pc,ra,rb,rc,resolution - 1,triangle_points);
		this->RenderPlaneTriangle(p1,pa,pc,radius_1,ra,rc,resolution - 1,triangle_points);
		this->RenderPlaneTriangle(pa,p2,pb,ra,radius_2,rb,resolution - 1,triangle_points);
		this->RenderPlaneTriangle(pb,p3,pc,rb,radius_3,rc,resolution - 1,triangle_points);
	}
	else
	{
		pa = p1;
		pb = p2;
		pc = p3;
		
		pa.Normal();
		pb.Normal();
		pc.Normal();
		
		pa *= radius_1;
		pb *= radius_2;
		pc *= radius_3;
		
		v1 = pb - pa;
		v2 = pc - pb;
		p_normal = v1 * v2;
		
		if(p_normal.Length() > ZERO_EPSILON)
		{
			p_normal.Normal();
		}
		else
		{
			p_normal.Set(0,0,1);
		}
		
		vertex.normal = p_normal;
		vertex.vertex = pa;
		triangle_points->push_back(vertex);
		
		vertex.normal = p_normal;
		vertex.vertex = pb;
		triangle_points->push_back(vertex);
		
		vertex.normal = p_normal;
		vertex.vertex = pc;
		triangle_points->push_back(vertex);
	}
}

TVector3 TOpenGLGeometryFactory::Normalize_XY(
	const TVector3 						&p) const
{
	double								len;
	TVector3							r(p);
	
	len = sqrt(p.x * p.x + p.y * p.y);
	
	r.x /= len;
	r.y /= len;
	
	return r;
}


