/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#ifndef OPENGLWIDGETHEADERFILE
#define OPENGLWIDGETHEADERFILE

#include <QOpenGLWidget>
#include <QOpenGLFunctions>
#include <QPoint>
#include <QRect>
#include <QFont>

#include <vector>

#include "mat4.h"
#include "vector3.h"

class QMouseEvent;
class QKeyEvent;
class QWheelEvent;
class QRubberBand;
class QPaintEvent;
class TOpenGLWidget;

class TOpenGLWidget : public QOpenGLWidget, protected QOpenGLFunctions
{
Q_OBJECT

public:

	// CREATORS
	TOpenGLWidget(const QWidget *parent=0,Qt::WindowFlags flags=Qt::WindowFlags());
	virtual ~TOpenGLWidget(void);

	// ACCESSORS
	static int Get_OpenGL_Version_Major(QString * const version_text,QString * const extensions_text);
	
	// MANIPULATORS
	void Set_Pan(void) {d_pan_mode = true;d_rotate_2d_mode = false;d_rotate_3d_mode = false;}
	void Set_Rotate_2D(void) {d_pan_mode = false;d_rotate_2d_mode = true;d_rotate_3d_mode = false;}
	void Set_Rotate_3D(void) {d_pan_mode = false;d_rotate_2d_mode = false;d_rotate_3d_mode = true;}
	void Scale_To_Fit(void) {d_scale_to_fit_render_mode = true; this->update();}
	 
signals:
	// XYZ model point clicked by user.  If point is within a selection the Id is returned otherwise -1
	void Model_Click_Point(const double&,const double&,const double&,const int);

protected:
	
enum TSelectionItemMode
{
	SELECTION_BOX = 0,
	SELECTION_LINE
};

struct TSelectionItemType
{
	TVector3							min_corner;
	TVector3							max_corner;
	TVector3							axis_1;
	TVector3							axis_2;
	TVector3							axis_3;
	int									selection_id;
	TSelectionItemMode					selection_mode;
};

	virtual void GLInit(void){;}
	virtual void GLResize(const int, const int){;}	// width, height
	virtual void GLRender(void){;}

	virtual void mousePressEvent(QMouseEvent *event);
	virtual void mouseMoveEvent(QMouseEvent *event);
	virtual void mouseReleaseEvent(QMouseEvent *event);
	virtual void keyPressEvent(QKeyEvent *event);
	virtual void keyReleaseEvent(QKeyEvent *event);
	virtual void wheelEvent(QWheelEvent *event);
	
	virtual void initializeGL(void);
	virtual void resizeGL(int Width,int Height);
	virtual void paintGL(void);
	
	int Selection_Items_Count(void) const {return static_cast<int>(d_SelectionItems.size());}
	TSelectionItemType Selection_Item(const int index) const;
	int Find_Selection(const TVector3 &pick_point) const; // returns -1 for no selection
	
	bool Is_Processing_Scale_To_Fit(void) const {return d_processing_scale_to_fit;}
	void Process_Vertex(const TVector3 &pnt);
	inline void Process_Vertex(const double &x,const double &y,const double &z) {this->Process_Vertex(TVector3(x,y,z));}

	void AddSelectionItem(const TSelectionItemType &t);
	void Clear_Selection_Items(void) {d_SelectionItems.clear();}
	
	// These routines should only be called if rendering context is current.
	void Rotate_Model(const TVector3 &axis,const double &angle);
	void Point_Raster_Position(int * const x,int * const y, const TVector3 &point);
	
	TMat4								d_modelview_mat;
	TMat4								d_projection_mat;
	
private:
	
	TMat4								d_modelview_mat_transpose;
	TVector3							d_min_modelview_corner;
	TVector3							d_max_modelview_corner;

	QPoint								d_mouse_down;
	QPoint								d_right_mouse_down;
	bool								d_right_mouse_down_moving;
	bool								d_rubberband_moving;
	QRect								d_rubberband_rect;
	QRubberBand							*d_rubberband;
	
	bool								d_shift_key_down;
	bool								d_ctrl_key_down;

	QPoint								d_viewport_center;
	TVector3							d_rotation_center;
	bool								d_scale_to_fit_render_mode;
	bool								d_processing_scale_to_fit;
	bool								d_initialize_scale_to_fit;

	bool								d_pan_mode;
	bool								d_rotate_2d_mode;
	bool								d_rotate_3d_mode;

	TVector3							d_light_positions[8];

	std::vector<TSelectionItemType>		d_SelectionItems;

	bool Is_Point_Between(const TVector3 &pnt, const TVector3 &start, const TVector3 &end, const TVector3 &axis) const;

	void Convert_To_GLViewport(short *x,short *y);
	void Convert_GLViewport_ToModel(const short x_gl,const short y_gl,double *x,double *y);
	void Process_Mouse_Click(const short x_gl,const short y_gl);
	void Set_Rotation_Center(void);
	double Get_GLPixel_Depth(const short x_gl,const short y_gl);
	void Zoom_To_Rect(const QRect &rect);
	void Click_Zoom(const short x_gl,const short y_gl);
	void Mousewheel_Zoom(const int delta);

	void Begin_Fit_To_Viewport(void);
	void End_Fit_To_Viewport(void);

	// NOT IMPELEMTENTED
	TOpenGLWidget(const TOpenGLWidget&);
	TOpenGLWidget& operator=(const TOpenGLWidget&);
};

#endif
