/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QComboBox>
#include <QGridLayout>
#include <QLabel>
#include <QSizePolicy>
#include <QStackedWidget>
#include <QStringList>
#include <QWidget>

#include "artifactballbarwidget.h"
#include "artifactfreepointswidget.h"
#include "artifactgaugeblockwidget.h"
#include "artifactringgaugewidget.h"
#include "artifactspherewidget.h"
#include "artifactstepgaugewidget.h"

#include "artifactcontainerwidget.h"

static const double						ZERO_EPSILON(0.001);

TArtifactContainerWidget::TArtifactContainerWidget(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QWidget(const_cast<QWidget*>(parent),flags)
{
	QGridLayout							*widget_layout;
	QLabel								*artifact_type;
	QSizePolicy							sizepolicy_minimumexpanding_fixed(QSizePolicy::MinimumExpanding,QSizePolicy::Fixed);

	sizepolicy_minimumexpanding_fixed.setHorizontalStretch(0);
	sizepolicy_minimumexpanding_fixed.setVerticalStretch(0);

	widget_layout = new QGridLayout(this);
	widget_layout->setSpacing(4);

	artifact_type = new QLabel(this);
	widget_layout->addWidget(artifact_type,0,0,1,1);

	d_artifact_type_combo = new QComboBox(this);
	d_artifact_type_combo->setSizePolicy(sizepolicy_minimumexpanding_fixed);
	widget_layout->addWidget(d_artifact_type_combo,0,1,1,1);

	d_widget_stack = new QStackedWidget(this);
	
	d_artifact_freepoints_widget = new TArtifactFreePointsWidget();
	d_widget_stack->addWidget(d_artifact_freepoints_widget);
	d_artifact_type_combo->addItem(QStringLiteral("Free Points"));
	
	d_artifact_ballbar_widget = new TArtifactBallbarWidget();
	d_widget_stack->addWidget(d_artifact_ballbar_widget);
	d_artifact_type_combo->addItem(QStringLiteral("Ballbar"));

	d_artifact_gaugeblock_widget = new TArtifactGaugeBlockWidget();
	d_widget_stack->addWidget(d_artifact_gaugeblock_widget);
	d_artifact_type_combo->addItem(QStringLiteral("Gauge Block"));

	d_artifact_ringgauge_widget = new TArtifactRingGaugeWidget();
	d_widget_stack->addWidget(d_artifact_ringgauge_widget);
	d_artifact_type_combo->addItem(QStringLiteral("Ring Gauge"));

	d_artifact_sphere_widget = new TArtifactSphereWidget();
	d_widget_stack->addWidget(d_artifact_sphere_widget);
	d_artifact_type_combo->addItem(QStringLiteral("Sphere"));

	d_artifact_stepgauge_widget = new TArtifactStepgaugeWidget();
	d_widget_stack->addWidget(d_artifact_stepgauge_widget);
	d_artifact_type_combo->addItem(QStringLiteral("Step Gauge"));
	
	widget_layout->addWidget(d_widget_stack,1,0,1,2);
	
	// defaults
	d_tip_radius = 0.0;
	d_artifact_type_combo->setCurrentIndex(0);
	d_widget_stack->setCurrentIndex(0);

	artifact_type->setText(QStringLiteral("Artifact Type:"));

	connect(d_artifact_type_combo,static_cast<void (QComboBox::*)(int)>(&QComboBox::currentIndexChanged),d_widget_stack,&QStackedWidget::setCurrentIndex);
	
	connect(d_artifact_freepoints_widget,&TArtifactFreePointsWidget::Touch_Point,this,&TArtifactContainerWidget::Add_Manual_Touch);
	connect(d_artifact_ballbar_widget,&TArtifactBallbarWidget::Touch_Point,this,&TArtifactContainerWidget::Add_Manual_Touch);
	connect(d_artifact_gaugeblock_widget,&TArtifactGaugeBlockWidget::Touch_Point,this,&TArtifactContainerWidget::Add_Manual_Touch);
	connect(d_artifact_ringgauge_widget,&TArtifactRingGaugeWidget::Touch_Point,this,&TArtifactContainerWidget::Add_Manual_Touch);
	connect(d_artifact_sphere_widget,&TArtifactSphereWidget::Touch_Point,this,&TArtifactContainerWidget::Add_Manual_Touch);
	connect(d_artifact_stepgauge_widget,&TArtifactStepgaugeWidget::Touch_Point,this,&TArtifactContainerWidget::Add_Manual_Touch);
}

TArtifactContainerWidget::~TArtifactContainerWidget(void)
{
}

bool TArtifactContainerWidget::Save_FreeTouch_Points(
	const QString 						&data_path) const
{
	return d_artifact_freepoints_widget->Save_Touch_Points(data_path);
}

TVector3 TArtifactContainerWidget::Ballbar_Attribute_Position(void) const
{
	return d_artifact_ballbar_widget->Attribute_Position();
}

TVector3 TArtifactContainerWidget::Ballbar_Attribute_Axis(void) const
{
	return d_artifact_ballbar_widget->Attribute_Axis();
}

double TArtifactContainerWidget::Ballbar_Attribute_Length(void) const
{
	return d_artifact_ballbar_widget->Attribute_Length();
}

TVector3 TArtifactContainerWidget::Gaugeblock_Attribute_Position(void) const
{
	return d_artifact_gaugeblock_widget->Attribute_Position();
}

TVector3 TArtifactContainerWidget::Gaugeblock_Attribute_Axis(void) const
{
	return d_artifact_gaugeblock_widget->Attribute_Axis();
}

double TArtifactContainerWidget::Gaugeblock_Attribute_Length(void) const
{
	return d_artifact_gaugeblock_widget->Attribute_Length();
}

TVector3 TArtifactContainerWidget::Ringgauge_Attribute_Position(void) const
{
	return d_artifact_ringgauge_widget->Attribute_Position();
}

TVector3 TArtifactContainerWidget::Ringgauge_Attribute_Axis(void) const
{
	return d_artifact_ringgauge_widget->Attribute_Axis();
}

double TArtifactContainerWidget::Ringgauge_Attribute_Diameter(void) const
{
	return d_artifact_ringgauge_widget->Attribute_Diameter();
}

bool TArtifactContainerWidget::Ringgauge_Attribute_ID(void) const
{
	return d_artifact_ringgauge_widget->Attribute_ID();
}

TVector3 TArtifactContainerWidget::Sphere_Attribute_Position(void) const
{
	return d_artifact_sphere_widget->Attribute_Position();
}

double TArtifactContainerWidget::Sphere_Attribute_Diameter(void) const
{
	return d_artifact_sphere_widget->Attribute_Diameter();
}

TVector3 TArtifactContainerWidget::Stepgauge_Attribute_Position(void) const
{
	return d_artifact_stepgauge_widget->Attribute_Position();
}

TVector3 TArtifactContainerWidget::Stepgauge_Attribute_Axis(void) const
{
	return d_artifact_stepgauge_widget->Attribute_Axis();
}

double TArtifactContainerWidget::Stepgauge_Attribute_Length(void) const
{
	return d_artifact_stepgauge_widget->Attribute_Length();
}

// MANIPULATORS
bool TArtifactContainerWidget::Load_FreeTouch_Points(
	const QString 						&data_path)
{
	return d_artifact_freepoints_widget->Load_Touch_Points(data_path);
}

void TArtifactContainerWidget::Set_Tool(
	const TVector3					 	&offset,
	const double						&tip_diameter)
{
	d_tool_offset = offset;
	d_tip_radius = tip_diameter / 2.0;
	
	d_artifact_ballbar_widget->Set_Tool_Offset(offset);
	d_artifact_gaugeblock_widget->Set_Tool_Offset(offset);
	d_artifact_ringgauge_widget->Set_Tool_Offset(offset);
	d_artifact_sphere_widget->Set_Tool_Offset(offset);
	d_artifact_stepgauge_widget->Set_Tool_Offset(offset);
}

void TArtifactContainerWidget::Set_Head_AB(
	const double 						&head_a,
	const double 						&head_b)
{
	d_artifact_ballbar_widget->Set_Head_AB(head_a,head_b);
	d_artifact_gaugeblock_widget->Set_Head_AB(head_a,head_b);
	d_artifact_ringgauge_widget->Set_Head_AB(head_a,head_b);
	d_artifact_sphere_widget->Set_Head_AB(head_a,head_b);
	d_artifact_stepgauge_widget->Set_Head_AB(head_a,head_b);
}

void TArtifactContainerWidget::Set_Ballbar_Attribute_Position(
	const TVector3 						&pos)
{
	d_artifact_ballbar_widget->Set_Attribute_Position(pos);
}

void TArtifactContainerWidget::Set_Ballbar_Attribute_Axis(
	const TVector3 						&axis)
{
	d_artifact_ballbar_widget->Set_Attribute_Axis(axis);
}

void TArtifactContainerWidget::Set_Ballbar_Attribute_Length(
	const double 						&len)
{
	d_artifact_ballbar_widget->Set_Attribute_Length(len);
}

void TArtifactContainerWidget::Set_Gaugeblock_Attribute_Position(
	const TVector3 						&pos)
{
	d_artifact_gaugeblock_widget->Set_Attribute_Position(pos);
}

void TArtifactContainerWidget::Set_Gaugeblock_Attribute_Axis(
	const TVector3 						&axis)
{
	d_artifact_gaugeblock_widget->Set_Attribute_Axis(axis);
}

void TArtifactContainerWidget::Set_Gaugeblock_Attribute_Length(
	const double 						&len)
{
	d_artifact_gaugeblock_widget->Set_Attribute_Length(len);
}

void TArtifactContainerWidget::Set_Ringgauge_Attribute_Position(
	const TVector3 						&pos)
{
	d_artifact_ringgauge_widget->Set_Attribute_Position(pos);
}

void TArtifactContainerWidget::Set_Ringgauge_Attribute_Axis(
	const TVector3 						&axis)
{
	d_artifact_ringgauge_widget->Set_Attribute_Axis(axis);
}

void TArtifactContainerWidget::Set_Ringgauge_Attribute_Diameter(
	const double 						&diam)
{
	d_artifact_ringgauge_widget->Set_Attribute_Diameter(diam);
}

void TArtifactContainerWidget::Set_Ringgauge_Attribute_ID(
	const bool 							state)
{
	d_artifact_ringgauge_widget->Set_Attribute_ID(state);
}

void TArtifactContainerWidget::Set_Sphere_Attribute_Position(
	const TVector3 						&pos)
{
	d_artifact_sphere_widget->Set_Attribute_Position(pos);
}

void TArtifactContainerWidget::Set_Sphere_Attribute_Diameter(
	const double 						&diam)
{
	d_artifact_sphere_widget->Set_Attribute_Diameter(diam);
}

void TArtifactContainerWidget::Set_Stepgauge_Attribute_Position(
	const TVector3 						&pos)
{
	d_artifact_stepgauge_widget->Set_Attribute_Position(pos);
}

void TArtifactContainerWidget::Set_Stepgauge_Attribute_Axis(
	const TVector3 						&axis)
{
	d_artifact_stepgauge_widget->Set_Attribute_Axis(axis);
}

void TArtifactContainerWidget::Set_Stepgauge_Attribute_Length(
	const double 						&len)
{
	d_artifact_stepgauge_widget->Set_Attribute_Length(len);
}

void TArtifactContainerWidget::Add_Manual_Touch(
	const double 						&x,
	const double 						&y,
	const double 						&z,
	const double 						&i,
	const double 						&j,
	const double 						&k)
{
	TArtifactBallbarWidget				*ballbar_widget;
	TVector3							xyz;
	TVector3							ijk;
	bool								pre_comp_point(false);
	
	ballbar_widget = qobject_cast<TArtifactBallbarWidget*>(this->sender());
	if(ballbar_widget)
	{
		pre_comp_point = true;
	}
	
	xyz.Set(x,y,z);
	ijk.Set(i,j,k);

	if(ijk.Length() < ZERO_EPSILON)
	{
		ijk.Set(0,0,-1);
	}
	else
	{
		ijk.Normal();
	}
	
	xyz -= (ijk * d_tip_radius);
	
	emit Touch(xyz.x,xyz.y,xyz.z,ijk.i,ijk.j,ijk.k,pre_comp_point);
}
