/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QGridLayout>
#include <QLabel>
#include <QLineEdit>
#include <QListWidget>
#include <QListWidgetItem>
#include <cmath>
#include <assert.h>

#include "../../core/mat4.h"
#include "artifactpointtype.h"

#include "artifactspherewidget.h"

TArtifactSphereWidget::TArtifactSphereWidget(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QWidget(const_cast<QWidget*>(parent),flags)
{
	QGridLayout							*gridLayout;
	QLabel								*attributes_label;
	QLabel								*diameter_label;
	QLabel								*pos_x_label;
	QLabel								*pos_y_label;
	QLabel								*pos_z_label;
	QLabel								*position_label;
	QListWidgetItem						*item;

	gridLayout = new QGridLayout(this);
	gridLayout->setSpacing(4);
	gridLayout->setContentsMargins(0,0,0,0);

	position_label = new QLabel(this);
	gridLayout->addWidget(position_label,0,0,1,1);

	pos_x_label = new QLabel(this);
	pos_x_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	gridLayout->addWidget(pos_x_label,0,1,1,1);

	d_pos_x_edit = new QLineEdit(this);
	d_pos_x_edit->setAlignment(Qt::AlignCenter);
	gridLayout->addWidget(d_pos_x_edit,0,2,1,1);

	pos_y_label = new QLabel(this);
	pos_y_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	gridLayout->addWidget(pos_y_label,1,1,1,1);

	d_pos_y_edit = new QLineEdit(this);
	d_pos_y_edit->setAlignment(Qt::AlignCenter);
	gridLayout->addWidget(d_pos_y_edit,1,2,1,1);

	pos_z_label = new QLabel(this);
	pos_z_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	gridLayout->addWidget(pos_z_label,2,1,1,1);

	d_pos_z_edit = new QLineEdit(this);
	d_pos_z_edit->setAlignment(Qt::AlignCenter);
	gridLayout->addWidget(d_pos_z_edit,2,2,1,1);

	attributes_label = new QLabel(this);
	gridLayout->addWidget(attributes_label,3,0,1,1);

	diameter_label = new QLabel(this);
	diameter_label->setAlignment(Qt::AlignRight|Qt::AlignTrailing|Qt::AlignVCenter);
	gridLayout->addWidget(diameter_label,3,1,1,1);

	d_diameter_edit = new QLineEdit(this);
	d_diameter_edit->setAlignment(Qt::AlignCenter);
	gridLayout->addWidget(d_diameter_edit,3,2,1,1);

	d_points_list = new QListWidget(this);
	gridLayout->addWidget(d_points_list,4,0,1,3);

	QWidget::setTabOrder(d_pos_x_edit,d_pos_y_edit);
	QWidget::setTabOrder(d_pos_y_edit,d_pos_z_edit);
	QWidget::setTabOrder(d_pos_z_edit,d_diameter_edit);
	QWidget::setTabOrder(d_diameter_edit,d_points_list);
	
	// defaults
	d_head_a = 0.0;
	d_head_b = 0.0;
	
	item = new QListWidgetItem;
	item->setText(QStringLiteral("SPH1 Sphere Point 1"));
	item->setData(Qt::UserRole,TPointType::SPH1);
	d_points_list->addItem(item);

	item = new QListWidgetItem;
	item->setText(QStringLiteral("SPH2 Sphere Point 2"));
	item->setData(Qt::UserRole,TPointType::SPH2);
	d_points_list->addItem(item);

	item = new QListWidgetItem;
	item->setText(QStringLiteral("SPH3 Sphere Point 3"));
	item->setData(Qt::UserRole,TPointType::SPH3);
	d_points_list->addItem(item);

	item = new QListWidgetItem;
	item->setText(QStringLiteral("SPH4 Sphere Point 4"));
	item->setData(Qt::UserRole,TPointType::SPH4);
	d_points_list->addItem(item);

	item = new QListWidgetItem;
	item->setText(QStringLiteral("SPH5 Sphere Point 5 (Probe Normal)"));
	item->setData(Qt::UserRole,TPointType::SPH5);
	d_points_list->addItem(item);

	position_label->setText(QStringLiteral("Center:"));
	attributes_label->setText(QStringLiteral("Traits:"));
	pos_z_label->setText(QStringLiteral("Z:"));
	pos_y_label->setText(QStringLiteral("Y:"));
	pos_x_label->setText(QStringLiteral("X:"));
	diameter_label->setText(QStringLiteral("Diameter:"));

	connect(d_points_list,&QListWidget::itemDoubleClicked,this,&TArtifactSphereWidget::Touch_Double_Clicked);
}

TArtifactSphereWidget::~TArtifactSphereWidget(void)
{
}

TVector3 TArtifactSphereWidget::Attribute_Position(void) const
{
	TVector3							pos;
	
	pos.x = d_pos_x_edit->text().toDouble();
	pos.y = d_pos_y_edit->text().toDouble();
	pos.z = d_pos_z_edit->text().toDouble();

	return pos;
}

double TArtifactSphereWidget::Attribute_Diameter(void) const
{
	double								diam;
	static const double					MINIMUM_DIAMETER(10.0);
	
	diam = d_diameter_edit->text().toDouble();
	
	if(diam < MINIMUM_DIAMETER)
	{
		diam = MINIMUM_DIAMETER;
		d_diameter_edit->setText(QString::number(diam,'f',4));
	}
	
	return diam;
}

void TArtifactSphereWidget::Set_Attribute_Position(
	const TVector3 						&pos)
{
	d_pos_x_edit->setText(QString("%1").arg(pos.x,0,'f',4));
	d_pos_y_edit->setText(QString("%1").arg(pos.y,0,'f',4));
	d_pos_z_edit->setText(QString("%1").arg(pos.z,0,'f',4));
}

void TArtifactSphereWidget::Set_Attribute_Diameter(
	const double 						&diam)
{
	d_diameter_edit->setText(QString("%1").arg(diam,0,'f',4));
}

void TArtifactSphereWidget::Touch_Double_Clicked(
	QListWidgetItem						*item)
{
	TVector3							sensor_axis;
	TVector3							axis1;
	TVector3							axis2;
	TVector3							axis3;
	TVector3							center_pt;
	TVector3							touch_pt;
	TVector3							touch_pt_dir;
	double								diameter;
	int									point_id;
	static const double					ZERO_EPSILON(0.0001);
	static const double					MIN_A_ANGLE(1.0);
	static const double					DEG2RAD(0.01745329251994);

	diameter = this->Attribute_Diameter();
	center_pt = this->Attribute_Position();
	
	if(d_head_a > MIN_A_ANGLE)
	{
		sensor_axis.x = sin(d_head_b * DEG2RAD) * sin(d_head_a * DEG2RAD);
		sensor_axis.y = -1.0 * cos(d_head_b * DEG2RAD) * sin(d_head_a * DEG2RAD);;
		sensor_axis.z = -1.0 * cos(d_head_a * DEG2RAD);
	}
	else
	{
		sensor_axis = d_tool_offset;
	}

	if(sensor_axis.Length() < ZERO_EPSILON)
	{
		sensor_axis.Set(0,0,-1);
	}
	else
	{
		sensor_axis.Normal();
	}
	
	axis1 = sensor_axis;
	
	if(axis1.x > 0.707107)
	{
		axis2 = axis1 * TVector3(0,0,1);
	}
	else
	{
		axis2 = axis1 * TVector3(1,0,0);
	}
		
	axis3 = axis1 * axis2;
	
	if(axis3.Length() < ZERO_EPSILON)
	{
		if(fabs(axis1.x) < 0.707107)
		{
			axis3.Set(0,1,0);
		}
		else
		{
			axis3.Set(1,0,0);
		}
	}
	else
	{
		axis3.Normal();
	}

	point_id = item->data(Qt::UserRole).toInt();
	
	switch(point_id)
	{
		case TPointType::SPH1:
			touch_pt = center_pt;
			touch_pt -= (axis2 * diameter / 2.0);
			touch_pt_dir = axis2;

			emit Touch_Point(touch_pt.x,touch_pt.y,touch_pt.z,touch_pt_dir.i,touch_pt_dir.j,touch_pt_dir.k);

			break;
			
		case TPointType::SPH2:
			touch_pt = center_pt;
			touch_pt -= (axis3 * diameter / 2.0);
			touch_pt_dir = axis3;

			emit Touch_Point(touch_pt.x,touch_pt.y,touch_pt.z,touch_pt_dir.i,touch_pt_dir.j,touch_pt_dir.k);

			break;
			
		case TPointType::SPH3:
			touch_pt = center_pt;
			touch_pt += (axis2 * diameter / 2.0);
			touch_pt_dir = (-1.0 * axis2);

			emit Touch_Point(touch_pt.x,touch_pt.y,touch_pt.z,touch_pt_dir.i,touch_pt_dir.j,touch_pt_dir.k);

			break;
			
		case TPointType::SPH4:
			touch_pt = center_pt;
			touch_pt += (axis3 * diameter / 2.0);
			touch_pt_dir = (-1.0 * axis3);

			emit Touch_Point(touch_pt.x,touch_pt.y,touch_pt.z,touch_pt_dir.i,touch_pt_dir.j,touch_pt_dir.k);

			break;
			
		case TPointType::SPH5:
			touch_pt = center_pt;
			touch_pt -= (axis1 * diameter / 2.0);
			touch_pt_dir = axis1;

			emit Touch_Point(touch_pt.x,touch_pt.y,touch_pt.z,touch_pt_dir.i,touch_pt_dir.j,touch_pt_dir.k);

			break;

		default:
			break;
	}
	
}

