/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QTimer>
#include <cmath>

#include "machine.h"

TMachine::TMachine(void)
{
	std::random_device					random_device;

	d_move_speed = 500.0;
	d_jog_speed = 500.0;
	d_jog_highspeed = 500.0;			// only used to position machine manual touch points
	d_jog_distance = 50.0;
	d_touch_speed = 4.0;
	d_prehit_distance = 4.0;
	
	d_scale_factor_x = 1.0;
	d_scale_factor_y = 1.0;
	d_scale_factor_z = 1.0;
	d_squareness_xy = 0.0;
	d_squareness_yz = 0.0;
	d_squareness_zx = 0.0;

	d_min_machine = TVector3(0,0,-1000);
	d_max_machine = TVector3(1200,2200,0);
	d_command_executor.Set_Current_Position((d_max_machine + d_min_machine) / 2.0);

	d_head_angle_a = 0.0;
	d_head_angle_b = 0.0;
	d_tool_offset.Set(0,0,0);
	
	d_prbpin_tip_radius = 0.0;
	d_override_tip_radius = 0.0;
	d_active_tip_radius = 0.0;
	d_use_override_tip_radius = false;
	
	d_temperature_x = 20.0;
	d_temperature_y = 20.0;
	d_temperature_z = 20.0;
	d_temperature_part = 20.0;

	d_noise_value = 0.0;
	d_generate_noise = false;
	
	d_estop_status = true;
	
#ifdef Q_OS_WIN
	d_homed_status = true;
#else
	d_homed_status = false;	// testing:  force initial state of machine to be unhomed on macos or gnu/linux
#endif
		
	d_jog_timer = new QTimer;
	d_control_timer = new QTimer;
	
	d_random_engine.seed(random_device());
	
	connect(d_jog_timer,&QTimer::timeout,this,&TMachine::Jog_Timeout);
	connect(d_control_timer,&QTimer::timeout,this,&TMachine::Control_Timeout);
}

TMachine::~TMachine(void)
{
	delete d_jog_timer;
	delete d_control_timer;
}

bool TMachine::Control_Active(void) const
{
	return d_control_timer->isActive();
}

bool TMachine::Jogbox_Active(void) const
{
	return d_jog_timer->isActive();
}

bool TMachine::Add_Manual_Touch_Point(
	const TVector3						&pos,
	const TVector3						&vec)
{
	TVector3							touch_pos(pos);
	TVector3							offset;
	
	if(d_generate_noise)
	{
		offset = vec * (this->Random() * d_noise_value);
		touch_pos += offset;
	}
	
	return this->Manual_Touch(touch_pos,vec);
}

void TMachine::Set_Machine_Limits(
	const TVector3						&min,
	const TVector3						&max)
{
	d_min_machine = min;
	d_max_machine = max;
}
	
void TMachine::Set_Tool_Offset(
	const TVector3 						&offset)
{
	d_tool_offset = offset;
}

void TMachine::Set_Head_AB(
	const double 						&head_a,
	const double 						&head_b)
{
	d_head_angle_a = head_a;
	d_head_angle_b = head_b;
}

void TMachine::Set_Override_Tip(
	const bool							state,
	const double						&tip_diameter)
{
	d_use_override_tip_radius = state;
	d_override_tip_radius = tip_diameter / 2.0;
	
	if(d_use_override_tip_radius)
	{
		d_active_tip_radius = d_override_tip_radius;
	}
	else
	{
		d_active_tip_radius = d_prbpin_tip_radius;
	}
}

void TMachine::Set_Generate_Noise(
	const bool							state,
	const double						&value)
{
	d_generate_noise = state;
	
	if(value > 0.0)
	{
		d_noise_value = value;
	}
	else
	{
		d_noise_value = 0.0;
	}
}

void TMachine::Send_Delayed_Timout(void)
{
	this->Send(d_tx_queue);
	
	d_tx_queue.clear();
}

void TMachine::Control_Timeout(void)
{
	TVector3							pos;
	QByteArray							completion_text;
	TCommandExecutor::TCommandType		command_type;
	
	if(d_command_executor.Command_Status() != TCommandExecutor::COMMAND_IDLE)
	{
		pos = d_command_executor.Execute();
		emit Position_Changed(pos.x,pos.y,pos.z);
	}
		
	if(d_command_executor.Command_Status() == TCommandExecutor::COMMAND_COMPLETE)
	{
		command_type = d_command_executor.Command_Type();
		completion_text = d_command_executor.Completion_Text();

		if(command_type == TCommandExecutor::COMMAND_TOUCH)
		{
			pos = d_command_executor.Target_Position();
			emit Touch_Complete(pos.x,pos.y,pos.z);
		}
		else if(command_type == TCommandExecutor::COMMAND_GETPOS && completion_text.length() > 0)
		{
			pos = d_command_executor.Current_Position();

			completion_text.replace("%1",QString::number(pos.x,'f',5).toUtf8());
			completion_text.replace("%2",QString::number(pos.y,'f',5).toUtf8());
			completion_text.replace("%3",QString::number(pos.z,'f',5).toUtf8());
		}
		
		if(completion_text.length())
		{
			this->Send(completion_text);
		}
		
		if(!d_command_executor.Execute_Next())
		{
			d_control_timer->stop();
		}
	}
}

void TMachine::Jog_Timeout(void)
{
	TVector3							pos;

	if(d_command_executor.Command_Status() != TCommandExecutor::COMMAND_IDLE)
	{
		pos = d_command_executor.Execute();

		emit Position_Changed(pos.x,pos.y,pos.z);
	}
	
	if(d_command_executor.Command_Status() == TCommandExecutor::COMMAND_COMPLETE)
	{
		if(!d_command_executor.Execute_Next())
		{
			d_jog_timer->stop();
		}
	}
}

void TMachine::Send(
	const QByteArray					&bytes,
	const bool							channel_2)
{
	QString								write_text(QString(bytes).trimmed());
	
	if(channel_2)
	{
		emit Write_Data("<2<" + bytes);
	}
	else
	{
		emit Write_Data(bytes);
	}
	
#if DEBUG_MODE_LEVEL_1
	qWarning(QString("SEND: %1").arg(write_text).toLatin1());
#endif
}

void TMachine::Send_Clean(
	const QByteArray					&bytes)
{
	QByteArray							write_bytes;
	int									cntr;
	
	for(cntr = 0;cntr < bytes.size();++cntr)
	{
		if(!(bytes.at(cntr) < 0x20))
		{
			write_bytes.push_back(bytes.at(cntr));
		}
	}
	
	if(write_bytes.size() == 0)
	{
		return;
	}
	
	this->Send(write_bytes);
}

void TMachine::Send_Delayed(
	const QByteArray 					&bytes,
	const unsigned int 					timeout)
{
	d_tx_queue += bytes;
	
	QTimer::singleShot(timeout,this,&TMachine::Send_Delayed_Timout);
}

bool TMachine::Is_Position_Inside_Volume(
	const TVector3						&pos,
	bool								* const x_error,
	bool								* const y_error,
	bool								* const z_error) const
{
	TVector3							machine;
	
	assert(x_error);
	assert(y_error);
	assert(z_error);
	
	*x_error = false;
	*y_error = false;
	*z_error = false;
	
	machine = pos - d_tool_offset;
	
	if(machine.x < d_min_machine.x) (*x_error) = true;
	if(machine.y < d_min_machine.y) (*y_error) = true;
	if(machine.z < d_min_machine.z) (*z_error) = true;
	
	if(machine.x > d_max_machine.x) (*x_error) = true;
	if(machine.y > d_max_machine.y) (*y_error) = true;
	if(machine.z > d_max_machine.z) (*z_error) = true;
	
	return (!((*x_error) || (*y_error) || (*z_error)));
}


TVector3 TMachine::Get_Noise_Offset(
	const TVector3						&vec)
{
	TVector3							offset;
	
	if(d_generate_noise)
	{
		offset = vec * (this->Random() * d_noise_value);
	}
	
	return offset;
}

double TMachine::Random(void)
{
	double								value;
	double								log_value;
	
	std::uniform_real_distribution<double> random_range(-1.0, 1.0);
	
	value = random_range(d_random_engine);
	
	log_value = value * value;
	
	if(value < 0.0)
	{
		log_value *= (-1.0);
	}
	
	return log_value;
}

