/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#ifndef MACHINEHEADERFILE
#define MACHINEHEADERFILE

#include <QObject>
#include <QString>
#include <QByteArray>
#include <random>

#include "../../core/vector3.h"

#include "command_executor.h"

class QTimer;

class TMachine : public QObject
{
Q_OBJECT
	
public:
// CREATORS
	TMachine(void);
	virtual ~TMachine(void);
	
// ACCESSORS
	virtual QString Machine_Description(void) const = 0;
	virtual QByteArray KeyPress_Done(void) const = 0;
	virtual QByteArray KeyPress_DelPnt(void) const = 0;
	
	TVector3 Position(void) const {return d_command_executor.Current_Position();}
		
	TVector3 Tool_Offset(void) const {return d_tool_offset;}
	double Head_A(void) const {return d_head_angle_a;}
	double Head_B(void) const {return d_head_angle_b;}
	double Tip_Radius(void) const {return d_active_tip_radius;}
	double Tip_Diameter(void) const {return d_active_tip_radius * 2.0;}
	double Move_Speed(void) const {return d_move_speed;}
	
	bool EStop(void) const {return d_estop_status;}
	bool Homed_Status(void) const {return d_homed_status;}
	bool Control_Active(void) const;
	bool Jogbox_Active(void) const;
	
	QString Last_Error(void) const {return d_last_error;}

// MANIPULATORS
	virtual void Process_Rx(const QByteArray &new_data) = 0;
		
	virtual void Set_EStop(const bool state) = 0;
	virtual bool Move_To(const TVector3 &pos) = 0;
	virtual bool Manual_Touch(const TVector3 &pos,const TVector3 &vec) = 0;
	virtual bool Jog_Move(const TVector3 &direction) = 0;
	
	void Set_Not_Homed(void) {d_homed_status = false;}
	
	void Set_Current_Position(const TVector3 &pos) {d_command_executor.Set_Current_Position(pos);}
	
	bool Add_Manual_Touch_Point(const TVector3 &pos,const TVector3 &vec);
	void Set_Machine_Limits(const TVector3 &min,const TVector3 &max);
	void Set_Tool_Offset(const TVector3 &offset);
	void Set_Head_AB(const double &head_a,const double &head_b);
	void Set_Override_Tip(const bool state, const double &tip_diameter);
	void Set_Generate_Noise(const bool state, const double &value);
	
	void Set_Temperature_X(const double &value) {d_temperature_x = value;}
	void Set_Temperature_Y(const double &value) {d_temperature_y = value;}
	void Set_Temperature_Z(const double &value) {d_temperature_z = value;}
	void Set_Temperature_Part(const double &value) {d_temperature_part = value;}
	
	void Set_Scale_Factor_X(const double &scale_factor) {d_scale_factor_x = scale_factor;}
	void Set_Scale_Factor_Y(const double &scale_factor) {d_scale_factor_y = scale_factor;}
	void Set_Scale_Factor_Z(const double &scale_factor) {d_scale_factor_z = scale_factor;}
	void Set_Squareness_Error_XY(const double &squareness) {d_squareness_xy = squareness;}
	void Set_Squareness_Error_YZ(const double &squareness) {d_squareness_yz = squareness;}
	void Set_Squareness_Error_ZX(const double &squareness) {d_squareness_zx = squareness;}

signals:
	void Write_Data(const QByteArray&);
	void Log_Rx(const QString&);
	void Log_Text(const QString&);
	
	void Position_Changed(const double&,const double &,const double&);
	void Touch_Complete(const double&,const double&,const double&);
	void Tool_Offset_Changed(const double&,const double&,const double&,const double&);
	void Head_AB_Changed(const double&,const double&);
	
protected:
	TCommandExecutor					d_command_executor;
	
	QByteArray							d_tx_queue;
	
	QTimer								*d_control_timer;
	QTimer								*d_jog_timer;
	
	TVector3							d_min_machine;
	TVector3							d_max_machine;
	TVector3							d_tool_offset;
	double								d_prehit_distance;
	double								d_touch_speed;
	double								d_move_speed;
	double								d_jog_speed;
	double								d_jog_highspeed;
	double								d_jog_distance;
	double								d_head_angle_a;
	double								d_head_angle_b;
	double								d_override_tip_radius;
	double								d_active_tip_radius;
	double								d_prbpin_tip_radius;
	double								d_temperature_x;
	double								d_temperature_y;
	double								d_temperature_z;
	double								d_temperature_part;
	double								d_scale_factor_x;
	double								d_scale_factor_y;
	double								d_scale_factor_z;
	double								d_squareness_xy;
	double								d_squareness_yz;
	double								d_squareness_zx;
	bool								d_use_override_tip_radius;
	bool								d_estop_status;
	bool								d_homed_status;
	
	mutable QString						d_last_error;

	void Send(const QByteArray &bytes,const bool channel_2 = false);
	void Send_Clean(const QByteArray &bytes);	// same as Send but strips out everything below 0x20
	void Send_Delayed(const QByteArray &bytes,const unsigned int timeout);
			
	bool Is_Position_Inside_Volume(const TVector3 &pos,bool * const x_error,bool * const y_error,bool * const z_error) const;
	
	TVector3 Get_Noise_Offset(const TVector3 &vec);

private slots:
	void Send_Delayed_Timout(void);
	
	void Control_Timeout(void);
	void Jog_Timeout(void);
	

private:

	bool								d_generate_noise;
	double								d_noise_value;

	std::mt19937						d_random_engine;

	double Random(void);
	
// NOT IMPLEMENTED
	TMachine(const TMachine&);
	TMachine& operator=(const TMachine&);
};

#endif
