/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QTimer>
#include <QRegularExpression>
#include <QRegularExpressionMatch>
#include <cmath>
#include <assert.h>

#include "machine_ipp.h"

static const double						HEAD_FIXED_OFFSET_Z(78.55);
static const int						TIMER_INTERVAL(40);

TMachineIpp::TMachineIpp(void)
{
	d_session_active = false;
	d_error_active = true;
	d_user_enabled = false;
	
	d_position_daemon = new QTimer;
	
	d_position_time = 1000;
	d_position_distance = 10.0;
	
	connect(d_position_daemon,&QTimer::timeout,this,&TMachineIpp::Position_Report_Time);
	connect(d_jog_timer,&QTimer::timeout,this,&TMachineIpp::Position_Report_Distance);
	connect(d_control_timer,&QTimer::timeout,this,&TMachineIpp::Position_Report_Distance);
}

TMachineIpp::~TMachineIpp(void)
{
	disconnect(d_jog_timer,&QTimer::timeout,this,&TMachineIpp::Position_Report_Distance);
	disconnect(d_control_timer,&QTimer::timeout,this,&TMachineIpp::Position_Report_Distance);

	delete d_position_daemon;
}

void TMachineIpp::Process_Rx(
	const QByteArray					&new_data)
{
	std::vector<TMachineIpp::TDataProperty> properties;
	std::vector<TMachineIpp::TToolData>::const_iterator tool_iter;
	std::vector<TMachineIpp::TAlignmentData>::const_iterator csy_iter;
	TMachineIpp::TAlignmentData			alignment_data;
	TCommandExecutor::TCommand			command;
	std::vector<double>					values;
	QRegularExpression					expression_tool_x(QStringLiteral("[Tool.EOffsetX(]{14}([-]?[0-9.]*)"));
	QRegularExpression					expression_tool_y(QStringLiteral("[Tool.EOffsetY(]{14}([-]?[0-9.]*)"));
	QRegularExpression					expression_tool_z(QStringLiteral("[Tool.EOffsetZ(]{14}([-]?[0-9.]*)"));
	QRegularExpressionMatch				expression_match;
	QStringList							list;
	QString								line;
	QString								command_tag;
	QString								keyword;
	QString								arguments;
	QString								text;
	QString								write_text;
	TVector3							target;
	TVector3							pos;
	TVector3							vec;
	TVector3							p1,p2;
	int									rx_line_index(-1);
	int									index(-1);
	bool								x_error,y_error,z_error;
	bool								exists;
	bool								tool_offset_changed;
	static const double					ZERO_EPSILON(0.00001);
	static const double					DEG2RAD(0.017453292519943);
	
	d_rx_data += new_data;
	
	do
	{
		rx_line_index = d_rx_data.indexOf('\r');
		
		if(!(rx_line_index < 0))
		{
			line = QString::fromLatin1(d_rx_data.mid(0,rx_line_index)).simplified();
			d_rx_data.remove(0,rx_line_index + 1);
			
			// clean text
			while(line.length() && line[0].toLatin1() < 0x20)
			{
				line = line.mid(1);
			}
			
			emit Log_Rx(line);
			
#if DEBUG_MODE_LEVEL_1
			qWarning(QString("RECV: %1").arg(line).toLatin1());
#endif
			
			if(!this->Process_Line(line,&command_tag,&text))
			{
				this->Send("E0000 ! Error(3, 0008,\"\", \"Protocol error\")\r\n");
			}
			else
			{
				index = text.indexOf('(');
				
				if(index < 0)
				{
					this->Send("E0000 ! Error(3, 0509,\"\", \"Bad argument\")\r\n");
				}
				else
				{
					keyword = text.mid(0,index);
					
					arguments = text.mid(index + 1);	// remove open bracket
					arguments.chop(1);					// remove closeing bracket
										
					if(keyword.compare(QStringLiteral("AbortE")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						if(d_command_executor.Command_Status() != TCommandExecutor::COMMAND_IDLE)
						{
							text = QString("%1 ! Error(2, 0006,\"\", \"Transaction aborted (Use ClearAllErrors to Continue\")\r\n").arg(command_tag);
							this->Send(text.toLatin1());
						}
						
						d_command_executor.Abort();
						
						d_jog_timer->stop();
						d_control_timer->stop();
						
						d_error_active = true;
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("AlignPart")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						text = QString("%1 ! Error(2, 2506,\"Controller does not have a rotary table\", \"Part not aligned\")\r\n").arg(command_tag);
						this->Send(text.toLatin1());

						d_error_active = true;
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("AlignTool")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						values = this->Get_Argument_Values(arguments);
						
						if(values.size() == 4)
						{
							p1.x = values[0];
							p1.y = values[1];
							p1.z = values[2];
							
							if(p1.Length() < ZERO_EPSILON)
							{
								p1.Set(0,0,1);
							}
							
							p1.Normal();
							
							text = QString("%1 # %2, %3, %4\r\n").arg(command_tag).arg(p1.x,0,'f',9).arg(p1.y,0,'f',9).arg(p1.z,0,'f',9);
							this->Send(text.toLatin1());
						}
						else if(values.size() == 8)
						{
							p1.x = values[0];
							p1.y = values[1];
							p1.z = values[2];
							
							p2.x = values[3];
							p2.y = values[4];
							p2.z = values[5];
							
							if(p1.Length() < ZERO_EPSILON)
							{
								p1.Set(0,0,1);
							}
							
							if(p2.Length() < ZERO_EPSILON)
							{
								p2.Set(1,0,0);
							}
							
							p1.Normal();
							p2.Normal();
							
							text = QString("%1 # %2, %3, %4, %5, %6, %7\r\n")
									.arg(command_tag)
									.arg(p1.x,0,'f',9)
									.arg(p1.y,0,'f',9)
									.arg(p1.z,0,'f',9)
									.arg(p2.x,0,'f',9)
									.arg(p2.y,0,'f',9)
									.arg(p2.z,0,'f',9);
							this->Send(text.toLatin1());
						}
						else
						{
							text = QString("%1 ! Error(3, 0502,\"Command requires 4 or 8 values\", \"Incorrect arguments\")\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							d_error_active = true;
						}
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("AvrRadius")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						text = QString("%1 # AvrRadius(%2)\r\n").arg(command_tag).arg(d_active_tip_radius,0,'f',5);
						this->Send(text.toLatin1());
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("CalcToolAlignment")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						text = QString("%1 ! Error(2, 1505,\"CalcToolAlignment\", \"Tool is not alignable\")\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						d_error_active = true;
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("CalcToolAngles")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						text = QString("%1 ! Error(2, 1505,\"CalcToolAngles\", \"Tool is not alignable\")\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						d_error_active = true;
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("ClearAllErrors")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						d_error_active = false;
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("ChangeTool")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());

						p1 = d_tool_offset;
						
						d_active_tool = this->Find_Tool(arguments);
						
						d_tool_offset = d_active_tool.e_offset;
						d_prbpin_tip_radius = d_active_tool.tip_diameter / 2.0;
						
						if(d_use_override_tip_radius)
						{
							d_active_tip_radius = d_override_tip_radius;
						}
						else
						{
							d_active_tip_radius = d_prbpin_tip_radius;
						}
						
						d_head_angle_a = d_active_tool.head_a;
						d_head_angle_b = d_active_tool.head_b;

						// takes into account the shift in position from a change in the tool offset
						pos = d_command_executor.Current_Position();
						pos += (d_tool_offset - p1);
						
						d_command_executor.Set_Current_Position(pos);
						
						emit Tool_Offset_Changed(d_tool_offset.x,d_tool_offset.y,d_tool_offset.z,d_active_tip_radius * 2.0);
						emit Head_AB_Changed(d_active_tool.head_a,d_active_tool.head_b);
						emit Position_Changed(pos.x,pos.y,pos.z);
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());

						if(d_position_daemon->isActive())
						{
							this->Position_Report_Time();	// force position after probe change
						}
					}
					else if(keyword.compare(QStringLiteral("DeleteCoordSystem")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						index = arguments.indexOf(',');
						
						if(!(index < 0))
						{
							text = arguments.mid(0,index);
							
							this->Delete_Alignment_Data(text);
							
							// make sure the delete alignment is not the active one.
							if(d_active_alignment.name.compare(text) == 0)
							{
								d_active_alignment = this->Find_Alignment(QStringLiteral("MachineCsy"),&exists);
							}
						}
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("DisableUser")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						d_user_enabled = false;
												
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("EnableUser")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						d_user_enabled = true;
												
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("EndSession")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						d_session_active = false;
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("EnumAllProp")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						if(!arguments.contains("Tool()"))
						{
							text = QString("%1 ! Error(3, 0502,\"EnumAllProp\", \"Incorrect arguments\")\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							d_error_active = true;
						}
						else
						{
							text = QString("%1 # \"Radius\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"ProbeType\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"GoToPar\",\"Property\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"GoToPar.Speed\",\"Property\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"GoToPar.Speed.Min\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"GoToPar.Speed.Max\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"GoToPar.Speed.Act\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"GoToPar.Speed\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"GoToPar.Accel\",\"Property\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"GoToPar.Accel.Min\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"GoToPar.Accel.Max\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"GoToPar.Accel\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"PtMeasPar\",\"Property\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"PtMeasPar.Speed\",\"Property\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"PtMeasPar.Speed.Min\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"PtMeasPar.Speed.Max\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"PtMeasPar.Speed.Act\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"PtMeasPar.Speed\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"PtMeasPar.Accel\",\"Property\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"PtMeasPar.Accel.Min\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"PtMeasPar.Accel.Max\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"PtMeasPar.Accel\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"PtMeasPar.Approach\",\"Property\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"PtMeasPar.Approach.Min\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"PtMeasPar.Approach.Max\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"PtMeasPar.Approach\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"PtMeasPar.Search\",\"Property\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"PtMeasPar.Search.Min\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"PtMeasPar.Search.Max\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"PtMeasPar.Search\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"PtMeasPar.Retract\",\"Property\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"PtMeasPar.Retract.Min\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"PtMeasPar.Retract.Max\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"PtMeasPar.Retract\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"Name\",\"String\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"AvrRadius\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"ER\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"Offset\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());

							text = QString("%1 # \"EOffsetX\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"EOffsetY\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"EOffsetZ\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"Form\",\"Property\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"Form\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"ED\",\"Property\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"ED\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());						}

						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("EnumCoordSystems")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						for(csy_iter = d_alignments.begin();csy_iter != d_alignments.end();++csy_iter)
						{
							if(!(*csy_iter).name_is_enum)
							{
								text = QString("%1 # \"%2\"\r\n").arg(command_tag).arg((*csy_iter).name);
								this->Send(text.toLatin1());
							}
						}
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("EnumProp")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						if(!arguments.contains("Tool()"))
						{
							text = QString("%1 ! Error(3, 0502,\"EnumProp\", \"Incorrect arguments\")\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							d_error_active = true;
						}
						else
						{
							text = QString("%1 # \"Radius\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"ProbeType\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"GoToPar\",\"Property\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"PtMeasPar\",\"Property\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"Name\",\"String\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"AvrRadius\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"ER\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"Offset\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());

							text = QString("%1 # \"EOffsetX\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"EOffsetY\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"EOffsetZ\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());
							
							text = QString("%1 # \"ED\",\"Number\"\r\n").arg(command_tag);
							this->Send(text.toLatin1());

						}
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("EnumTools")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						for(tool_iter = d_tools.begin();tool_iter != d_tools.end();++tool_iter)
						{
							text = QString("%1 # \"%2\"\r\n").arg(command_tag).arg((*tool_iter).name);
							this->Send(text.toLatin1());
						}
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("FindTool")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
												
						d_found_tool = this->Find_Tool(arguments);
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("Get")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						properties = this->Get_Report_Properties(arguments);
						pos = d_command_executor.Current_Position();
						
						pos = this->Convert_Pos_To_Part_Csy(pos);
						
						text = QString("%1 # ").arg(command_tag);
						text += this->Format_Property(properties,pos);
						text += QStringLiteral("\r\n");
						
						this->Send(text.toLatin1());
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("GetCoordSystem")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						text = QString("%1 # CoordSystem(%2)\r\n").arg(command_tag).arg(d_active_alignment.name);
						this->Send(text.toLatin1());
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("GetChangeToolAction")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						text = QString("%1 # GetChangeToolAction(switch,X(0.00000),Y(0.00000),Z(0.00000))\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("GetCoordSystem")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						text = QString("%1 # CoordSystem(%2)\r\n").arg(command_tag).arg(d_active_alignment.name);
						this->Send(text.toLatin1());
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("GetCsyTransformation")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						list = arguments.split(',');
						exists = false;
						
						if(list.size())
						{
							alignment_data = this->Find_Alignment(list[0],&exists);
							
							if(exists)
							{
								text = QString("%1 # GetCsyTransformation(%2,%3,%4,%5,%6,%7)\r\n")
											.arg(command_tag)
											.arg(alignment_data.csy_origin.x,0,'f',5)
											.arg(alignment_data.csy_origin.y,0,'f',5)
											.arg(alignment_data.csy_origin.z,0,'f',5)
											.arg(alignment_data.csy_alpha,0,'f',5)
											.arg(alignment_data.csy_beta,0,'f',5)
											.arg(alignment_data.csy_gamma,0,'f',5);
								this->Send(text.toLatin1());
							}
						}
						
						if(!exists)
						{
							text = QString("%1 ! Error(3, 0502,\"%2\", \"Incorrect arguments\")\r\n").arg(command_tag).arg(list[0]);
							this->Send(text.toLatin1());
						}
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("GetNamedCsyTransformation")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						list = arguments.split(',');
						exists = false;
						
						if(list.size())
						{
							alignment_data = this->Find_Alignment(list[0],&exists);
							
							if(exists)
							{
								text = QString("%1 # GetNamedCsyTransformation(%2,%3,%4,%5,%6,%7)\r\n")
										.arg(command_tag)
										.arg(alignment_data.csy_origin.x,0,'f',5)
										.arg(alignment_data.csy_origin.y,0,'f',5)
										.arg(alignment_data.csy_origin.z,0,'f',5)
										.arg(alignment_data.csy_alpha,0,'f',5)
										.arg(alignment_data.csy_beta,0,'f',5)
										.arg(alignment_data.csy_gamma,0,'f',5);
								this->Send(text.toLatin1());
							}
						}
						
						if(!exists)
						{
							text = QString("%1 ! Error(3, 1013,\"%2\", \"Coordinate system not found\")\r\n").arg(command_tag).arg(list[0]);
							this->Send(text.toLatin1());
						}
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("GetDMEVersion")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						text = QString("%1 # GetDMEVersion(\"1.7\")\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("GetErrorInfo")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						text = QString("%1 # \"Error code information not available\"\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("GetErrStatusE")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						text = QString("%1 # ErrStatus(%2)\r\n").arg(command_tag).arg(d_error_active | d_estop_status);
						this->Send(text.toLatin1());
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("GetMachineClass")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						text = QString("%1 # GetMachineClass(CartCMM)\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("GetNamedCsyTransformation")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						text = QString("%1 # GetNamedCsyTransformation(0.00000,0.00000,0.00000,0.00000,0.00000,0.00000)\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("GetProp")) == 0 || keyword.compare(QStringLiteral("GetPropE")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						if(arguments.compare(QStringLiteral("FoundTool.AvrRadius()")) == 0)
						{
							text = QString("%1 # FoundTool.AvrRadius(%2)\r\n").arg(command_tag).arg(d_found_tool.tip_diameter / 2.0,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("FoundTool.ER()")) == 0)
						{
							text = QString("%1 # FoundTool.ER(%2)\r\n").arg(command_tag).arg(d_found_tool.tip_diameter / 2.0,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("FoundTool.Name()")) == 0)
						{
							text = QString("%1 # FoundTool.Name(\"%2\")\r\n").arg(command_tag).arg(d_found_tool.name);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("FoundTool.Offset()")) == 0)
						{
							text = QString("%1 # FoundTool.Offset(%2,%3,%4)\r\n").arg(command_tag).arg(d_found_tool.offset.x,0,'f',5).arg(d_found_tool.offset.y,0,'f',5).arg(d_found_tool.offset.z + HEAD_FIXED_OFFSET_Z,0,'f',5);
							
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("FoundTool.EOffset()")) == 0)
						{
							text = QString("%1 # FoundTool.EOffset(%2,%3,%4)\r\n").arg(command_tag).arg(d_found_tool.e_offset.x,0,'f',5).arg(d_found_tool.e_offset.y,0,'f',5).arg(d_found_tool.e_offset.z + HEAD_FIXED_OFFSET_Z,0,'f',5);
							
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("FoundTool.PtMeasPar.Approach()")) == 0)
						{
							text = QString("%1 # FoundTool.PtMeasPar.Approach(%2)\r\n").arg(command_tag).arg(d_prehit_distance,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("FoundTool.PtMeasPar.Approach.Min()")) == 0)
						{
							text = QString("%1 # FoundTool.PtMeasPar.Approach.Min(%2)\r\n").arg(command_tag).arg(0.1,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("FoundTool.PtMeasPar.Approach.Max()")) == 0)
						{
							text = QString("%1 # FoundTool.PtMeasPar.Approach.Max(%2)\r\n").arg(command_tag).arg(50.0,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("FoundTool.PtMeasPar.Speed()")) == 0)
						{
							text = QString("%1 # FoundTool.PtMeasPar.Speed(%2)\r\n").arg(command_tag).arg(d_touch_speed,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("FoundTool.PtMeasPar.Speed.Min()")) == 0)
						{
							text = QString("%1 # FoundTool.PtMeasPar.Speed.Min(%2)\r\n").arg(command_tag).arg(0.1,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("FoundTool.PtMeasPar.Speed.Max()")) == 0)
						{
							text = QString("%1 # FoundTool.PtMeasPar.Speed.Max(%2)\r\n").arg(command_tag).arg(20.0,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("FoundTool.PtMeasPar.Accel()")) == 0)
						{
							text = QString("%1 # FoundTool.PtMeasPar.Accel(%2)\r\n").arg(command_tag).arg(200.0,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("FoundTool.PtMeasPar.Accel.Min()")) == 0)
						{
							text = QString("%1 # FoundTool.PtMeasPar.Accel.Min(%2)\r\n").arg(command_tag).arg(10.0,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("FoundTool.PtMeasPar.Accel.Max()")) == 0)
						{
							text = QString("%1 # FoundTool.PtMeasPar.Accel.Max(%2)\r\n").arg(command_tag).arg(500.0,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("FoundTool.GoToPar.Speed()")) == 0)
						{
							text = QString("%1 # FoundTool.GoToPar.Speed(%2)\r\n").arg(command_tag).arg(d_move_speed,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("FoundTool.GoToPar.Speed.Min()")) == 0)
						{
							text = QString("%1 # FoundTool.GoToPar.Speed.Min(%2)\r\n").arg(command_tag).arg(10.0,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("FoundTool.GoToPar.Speed.Max()")) == 0)
						{
							text = QString("%1 # FoundTool.GoToPar.Speed.Max(%2)\r\n").arg(command_tag).arg(500.0,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("FoundTool.GoToPar.Accel()")) == 0)
						{
							text = QString("%1 # FoundTool.GoToPar.Accel(%2)\r\n").arg(command_tag).arg(1000.0,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("FoundTool.GoToPar.Accel.Min()")) == 0)
						{
							text = QString("%1 # FoundTool.GoToPar.Accel.Min(%2)\r\n").arg(command_tag).arg(50.0,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("FoundTool.GoToPar.Accel.Max()")) == 0)
						{
							text = QString("%1 # FoundTool.GoToPar.Accel.Max(%2)\r\n").arg(command_tag).arg(1000.0,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("Machine.Temperature.X()")) == 0)
						{
							text = QString("%1 # Machine.Temperature.X(%2)\r\n").arg(command_tag).arg(d_temperature_x,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("Machine.Temperature.Y()")) == 0)
						{
							text = QString("%1 # Machine.Temperature.Y(%2)\r\n").arg(command_tag).arg(d_temperature_y,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("Machine.Temperature.Z()")) == 0)
						{
							text = QString("%1 # Machine.Temperature.Z(%2)\r\n").arg(command_tag).arg(d_temperature_z,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("Part.Temperature()")) == 0)
						{
							text = QString("%1 # Part.Temperature(%2)\r\n").arg(command_tag).arg(d_temperature_part,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("Tool.AvrRadius()")) == 0)
						{
							text = QString("%1 # Tool.AvrRadius(%2)\r\n").arg(command_tag).arg(d_active_tool.tip_diameter / 2.0,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("Tool.ER()")) == 0)
						{
							text = QString("%1 # Tool.ER(%2)\r\n").arg(command_tag).arg(d_active_tool.tip_diameter / 2.0,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("Tool.Name()")) == 0)
						{
							text = QString("%1 # Tool.Name(\"%2\")\r\n").arg(command_tag).arg(d_active_tool.name);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("Tool.Offset()")) == 0)
						{
							text = QString("%1 # Tool.Offset(%2,%3,%4)\r\n").arg(command_tag).arg(d_active_tool.offset.x,0,'f',5).arg(d_active_tool.offset.y,0,'f',5).arg(d_active_tool.offset.z + HEAD_FIXED_OFFSET_Z,0,'f',5);
							
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("Tool.EOffset()")) == 0)
						{
							text = QString("%1 # Tool.EOffset(%2,%3,%4)\r\n").arg(command_tag).arg(d_active_tool.e_offset.x,0,'f',5).arg(d_active_tool.e_offset.y,0,'f',5).arg(d_active_tool.e_offset.z + HEAD_FIXED_OFFSET_Z,0,'f',5);
							
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("Tool.PtMeasPar.Approach()")) == 0)
						{
							text = QString("%1 # Tool.PtMeasPar.Approach(%2)\r\n").arg(command_tag).arg(d_prehit_distance,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("Tool.PtMeasPar.Approach.Min()")) == 0)
						{
							text = QString("%1 # Tool.PtMeasPar.Approach.Min(%2)\r\n").arg(command_tag).arg(0.1,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("Tool.PtMeasPar.Approach.Max()")) == 0)
						{
							text = QString("%1 # Tool.PtMeasPar.Approach.Max(%2)\r\n").arg(command_tag).arg(50.0,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("Tool.PtMeasPar.Speed()")) == 0)
						{
							text = QString("%1 # Tool.PtMeasPar.Speed(%2)\r\n").arg(command_tag).arg(d_touch_speed,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("Tool.PtMeasPar.Speed.Min()")) == 0)
						{
							text = QString("%1 # Tool.PtMeasPar.Speed.Min(%2)\r\n").arg(command_tag).arg(0.1,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("Tool.PtMeasPar.Speed.Max()")) == 0)
						{
							text = QString("%1 # Tool.PtMeasPar.Speed.Max(%2)\r\n").arg(command_tag).arg(20.0,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("Tool.PtMeasPar.Accel()")) == 0)
						{
							text = QString("%1 # Tool.PtMeasPar.Accel(%2)\r\n").arg(command_tag).arg(200.0,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("Tool.PtMeasPar.Accel.Min()")) == 0)
						{
							text = QString("%1 # Tool.PtMeasPar.Accel.Min(%2)\r\n").arg(command_tag).arg(10.0,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("Tool.PtMeasPar.Accel.Max()")) == 0)
						{
							text = QString("%1 # Tool.PtMeasPar.Accel.Max(%2)\r\n").arg(command_tag).arg(500.0,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("Tool.GoToPar.Speed()")) == 0)
						{
							text = QString("%1 # Tool.GoToPar.Speed(%2)\r\n").arg(command_tag).arg(d_move_speed,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("Tool.GoToPar.Speed.Min()")) == 0)
						{
							text = QString("%1 # Tool.GoToPar.Speed.Min(%2)\r\n").arg(command_tag).arg(10.0,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("Tool.GoToPar.Speed.Max()")) == 0)
						{
							text = QString("%1 # Tool.GoToPar.Speed.Max(%2)\r\n").arg(command_tag).arg(500.0,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("Tool.GoToPar.Accel()")) == 0)
						{
							text = QString("%1 # Tool.GoToPar.Accel(%2)\r\n").arg(command_tag).arg(1000.0,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("Tool.GoToPar.Accel.Min()")) == 0)
						{
							text = QString("%1 # Tool.GoToPar.Accel.Min(%2)\r\n").arg(command_tag).arg(50.0,0,'f',5);
							this->Send(text.toLatin1());
						}
						else if(arguments.compare(QStringLiteral("Tool.GoToPar.Accel.Max()")) == 0)
						{
							text = QString("%1 # Tool.GoToPar.Accel.Max(%2)\r\n").arg(command_tag).arg(1000.0,0,'f',5);
							this->Send(text.toLatin1());
						}
						else
						{
							d_error_active = true;
							text = QString("%1 ! Error(3, 0510, \"%2\", \"Bad property\")\r\n").arg(command_tag).arg(arguments);
							this->Send(text.toLatin1());
						}
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("GetTemperatureAlarmStatus")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						text = QString("%1 # AlarmStatus(GradTx(0.0),GradTy(0.0),GradTz(0.0),TimeAlarmTx(0.0),TimeAlarmTy(0.0),TimeAlarmTz(0.0),Range(0.0))\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("GetXtdErrStatus")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						if(d_homed_status)
						{
							text = QString("%1 # IsHomed(1)\r\n").arg(command_tag);
						}
						else
						{
							text = QString("%1 # IsHomed(0)\r\n").arg(command_tag);
						}
						
						this->Send(text.toLatin1());
						
						text = QString("%1 # IsUserEnabled(%2)\r\n").arg(command_tag).arg(d_user_enabled);
						this->Send(text.toLatin1());
						
						text = QString("%1 # ErrStatus(%2)\r\n").arg(command_tag).arg(d_estop_status);
						this->Send(text.toLatin1());
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("GoTo")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());

						if(d_estop_status)
						{
							this->Send("E0000 ! Error(3, 0500,\"\", \"Emergency stop\")\r\n");
							
							text = QString("%1 %\r\n").arg(command_tag);
							this->Send(text.toLatin1());
						}
						else if(d_error_active)
						{
							this->Send("E0000 ! Error(2, 0514,\"Program in Error state\",\"Use ClearAllErrors to continue.\")\r\n");
							
							text = QString("%1 %\r\n").arg(command_tag);
							this->Send(text.toLatin1());
						}
						else
						{
							if(this->Get_Move_Data(arguments,&pos))
							{
								pos = this->Convert_Pos_From_Part_Csy(pos);
								
								if(this->Is_Position_Inside_Volume(pos,&x_error,&y_error,&z_error))
								{
									command.command_type = TCommandExecutor::COMMAND_MOVE;
									command.target = pos;
									command.speed = d_move_speed;
									command.completion_text = QString("%1 %\r\n").arg(command_tag);
									
									d_command_executor.Add_Command(command);
									
									if(!d_control_timer->isActive())
									{
										d_control_timer->start(TIMER_INTERVAL);
									}
								}
								else
								{
									if(x_error)
									{
										emit Log_Text(QStringLiteral("Machine limit encountered (X Move Out Of Limits)"));
										text = QString("%1 ! Error(3, 2500, \"GoTo\", \"Machine limit encountered (X Move Out Of Limits)\")\r\n").arg(command_tag);
										this->Send(text.toLatin1());
									}
									
									if(y_error)
									{
										emit Log_Text(QStringLiteral("Machine limit encountered (Y Move Out Of Limits)"));
										text = QString("%1 ! Error(3, 2500, \"GoTo\", \"Machine limit encountered (Y Move Out Of Limits)\")\r\n").arg(command_tag);
										this->Send(text.toLatin1());
									}
									
									if(z_error)
									{
										emit Log_Text(QStringLiteral("Machine limit encountered (Z Move Out Of Limits)"));
										text = QString("%1 ! Error(3, 2500, \"GoTo\", \"Machine limit encountered (Z Move Out Of Limits)\")\r\n").arg(command_tag);
										this->Send(text.toLatin1());
									}
									
									text = QString("%1 %\r\n").arg(command_tag);
									this->Send(text.toLatin1());
									
									d_command_executor.Abort();
									
									d_jog_timer->stop();
									d_control_timer->stop();
									
									d_error_active = true;
								}
							}
						}
					}
					else if(keyword.compare(QStringLiteral("Home")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						if(d_estop_status)
						{
							this->Send("E0000 ! Error(3, 0500,\"Home\", \"Emergency stop\")\r\n");
							
							text = QString("%1 %\r\n").arg(command_tag);
							this->Send(text.toLatin1());
						}
						else if(d_error_active)
						{
							this->Send("E0000 ! Error(2, 0514,\"Program in Error state\",\"Use ClearAllErrors to continue.\")\r\n");
							
							text = QString("%1 %\r\n").arg(command_tag);
							this->Send(text.toLatin1());
						}
						else
						{
							pos = d_command_executor.Current_Position();
							p1 = d_min_machine;
							p2 = d_max_machine;
							
							p1.z = p2.z;
							
							command.command_type = TCommandExecutor::COMMAND_MOVE;
							command.target = TVector3(pos.x,pos.y,p1.z + d_tool_offset.z);
							command.speed = d_move_speed;
							command.completion_text = QString();
							
							d_command_executor.Add_Command(command);
							
							command.command_type = TCommandExecutor::COMMAND_MOVE;
							command.target = p1 + d_tool_offset;
							command.speed = d_move_speed;
							command.completion_text = QString("%1 %\r\n").arg(command_tag);
							
							d_command_executor.Add_Command(command);
							
							if(!d_control_timer->isActive())
							{
								d_control_timer->start(TIMER_INTERVAL);
							}
							
							d_homed_status = true;
						}
					}
					else if(keyword.compare(QStringLiteral("IsHomed")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						if(d_homed_status)
						{
							text = QString("%1 # IsHomed(1)\r\n").arg(command_tag);
						}
						else
						{
							text = QString("%1 # IsHomed(0)\r\n").arg(command_tag);
						}

						this->Send(text.toLatin1());
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("IsUserEnabled")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						text = QString("%1 # IsUserEnabled(%2)\r\n").arg(command_tag).arg(d_user_enabled);
						this->Send(text.toLatin1());
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("OnMoveReportE")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						d_position_report_properties = Get_Report_Properties(arguments);
						this->Start_Position_Report(command_tag,arguments);
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						if(d_position_daemon->isActive())
						{
							this->Position_Report_Time();	// force initial position
						}
					}
					else if(keyword.compare(QStringLiteral("OnPtMeasReport")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						d_pt_meas_report_properties = this->Get_Report_Properties(arguments);

						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("OnScanReport")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						d_scan_report_properties = this->Get_Report_Properties(arguments);
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("PtMeas")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());

						if(d_estop_status)
						{
							this->Send("E0000 ! Error(3, 0500,\"\", \"Emergency stop\")\r\n");
							
							text = QString("%1 %\r\n").arg(command_tag);
							this->Send(text.toLatin1());
						}
						else if(d_error_active)
						{
							this->Send("E0000 ! Error(2, 0514,\"Program in Error state\",\"Use ClearAllErrors to continue.\")\r\n");
							
							text = QString("%1 %\r\n").arg(command_tag);
							this->Send(text.toLatin1());
						}
						else
						{
							if(this->Get_Touch_Data(arguments,&pos,&vec))
							{
								pos = this->Convert_Pos_From_Part_Csy(pos);
								vec = this->Convert_Vector_From_Part_Csy(vec);

								if(this->Is_Position_Inside_Volume(pos,&x_error,&y_error,&z_error))
								{
									p1 = pos + d_prehit_distance * vec + d_active_tip_radius * vec;
									
									command.command_type = TCommandExecutor::COMMAND_MOVE;
									command.target = p1;
									command.speed = d_move_speed;
									command.completion_text = QString();
									
									d_command_executor.Add_Command(command);
									
									p2 = pos + d_active_tip_radius * vec + this->Get_Noise_Offset(vec);
									
									command.command_type = TCommandExecutor::COMMAND_TOUCH;
									command.target = pos;
									command.speed = d_touch_speed;
									
									p2 = this->Convert_Pos_To_Part_Csy(p2);
									vec = this->Convert_Vector_To_Part_Csy(vec);

									text = command_tag;
									text += QStringLiteral(" # ");
									text += this->Format_Property(d_pt_meas_report_properties,p2,vec);
									text += QStringLiteral("\r\n");
									
									command.completion_text = text;
									
									d_command_executor.Add_Command(command);
									
									command.command_type = TCommandExecutor::COMMAND_MOVE;
									command.target = p1;
									command.speed = d_move_speed;
									command.completion_text = QString("%1 %\r\n").arg(command_tag);
									
									d_command_executor.Add_Command(command);
									
									if(!d_control_timer->isActive())
									{
										d_control_timer->start(TIMER_INTERVAL);
									}
								}
								else
								{
									if(x_error)
									{
										emit Log_Text(QStringLiteral("Machine limit encountered (X Move Out Of Limits)"));
										text = QString("%1 ! Error(3, 2500, \"PtMeas\", \"Machine limit encountered (X Move Out Of Limits)\")\r\n").arg(command_tag);
										this->Send(text.toLatin1());
									}
									
									if(y_error)
									{
										emit Log_Text(QStringLiteral("Machine limit encountered (Y Move Out Of Limits)"));
										text = QString("%1 ! Error(3, 2500, \"PtMeas\", \"Machine limit encountered (Y Move Out Of Limits)\")\r\n").arg(command_tag);
										this->Send(text.toLatin1());
									}
									
									if(z_error)
									{
										emit Log_Text(QStringLiteral("Machine limit encountered (Z Move Out Of Limits)"));
										text = QString("%1 ! Error(3, 2500, \"PtMeas\", \"Machine limit encountered (Z Move Out Of Limits)\")\r\n").arg(command_tag);
										this->Send(text.toLatin1());
									}
									
									text = QString("%1 %\r\n").arg(command_tag);
									this->Send(text.toLatin1());
									
									d_command_executor.Abort();
									
									d_jog_timer->stop();
									d_control_timer->stop();
									
									d_error_active = true;
								}
							}
						}
					}
					else if(keyword.compare(QStringLiteral("ReadAllTemperatures")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						text = QString("%1 # \"X Axis-1\",%2\r\n").arg(command_tag).arg(d_temperature_x,0,'f',5);
						this->Send(text.toLatin1());
						
						text = QString("%1 # \"X Axis-2\",%2\r\n").arg(command_tag).arg(d_temperature_x,0,'f',5);
						this->Send(text.toLatin1());
						
						text = QString("%1 # \"Y Axis-1\",%2\r\n").arg(command_tag).arg(d_temperature_y,0,'f',5);
						this->Send(text.toLatin1());
						
						text = QString("%1 # \"Y Axis-2\",%2\r\n").arg(command_tag).arg(d_temperature_y,0,'f',5);
						this->Send(text.toLatin1());
						
						text = QString("%1 # \"Z Axis-1\",%2\r\n").arg(command_tag).arg(d_temperature_z,0,'f',5);
						this->Send(text.toLatin1());
						
						text = QString("%1 # \"Z Axis-2\",%2\r\n").arg(command_tag).arg(d_temperature_z,0,'f',5);
						this->Send(text.toLatin1());
						
						text = QString("%1 # \"Part\",%2\r\n").arg(command_tag).arg(d_temperature_part,0,'f',5);
						this->Send(text.toLatin1());
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("SaveNamedCsyTransformation")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						index = arguments.indexOf(',');
						
						if(!(index < 0))
						{
							alignment_data.name = arguments.mid(0,index);
							alignment_data.name_is_enum = false;
							
							text = arguments.mid(index+1);
							
							values = this->Get_Argument_Values(text);
							
							if(values.size() == 6)
							{
								alignment_data.csy_origin.x = values[0];
								alignment_data.csy_origin.y = values[1];
								alignment_data.csy_origin.z = values[2];
								
								alignment_data.csy_alpha = values[3];
								alignment_data.csy_beta = values[4];
								alignment_data.csy_gamma = values[5];
								
								alignment_data.csy_mat.Identity();
								alignment_data.csy_mat.Rotate(alignment_data.csy_alpha * DEG2RAD,
															  alignment_data.csy_beta * DEG2RAD,
															  alignment_data.csy_gamma * DEG2RAD);
								
								this->Update_Alignment_Data(alignment_data);
							}
						}
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("SetCsyTransformation")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						index = arguments.indexOf(',');
						
						if(!(index < 0))
						{
							alignment_data.name = arguments.mid(0,index);
							alignment_data.name_is_enum = true;

							text = arguments.mid(index+1);
							
							values = this->Get_Argument_Values(text);
							
							if(values.size() == 6)
							{
								alignment_data.csy_origin.x = values[0];
								alignment_data.csy_origin.y = values[1];
								alignment_data.csy_origin.z = values[2];
								
								alignment_data.csy_alpha = values[3];
								alignment_data.csy_beta = values[4];
								alignment_data.csy_gamma = values[5];
								
								alignment_data.csy_mat.Identity();
								alignment_data.csy_mat.Rotate(alignment_data.csy_alpha * DEG2RAD,
															  alignment_data.csy_beta * DEG2RAD,
															  alignment_data.csy_gamma * DEG2RAD);
								
								this->Update_Alignment_Data(alignment_data);
							}
						}
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("SetProp")) == 0 || keyword.compare(QStringLiteral("SetPropE")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
												
						if(arguments.startsWith(QStringLiteral("Tool.GoToPar.Approach(")))
						{
							text = arguments.mid(22);	// remove 'Tool.GoToPar.Approach('
							text.chop(1);				// remove trailing ')'
							
							d_prehit_distance = text.toDouble();
						}
						else if(arguments.startsWith(QStringLiteral("Tool.PtMeasPar.Speed(")))
						{
							text = arguments.mid(21);	// remove 'Tool.PtMeasPar.Speed('
							text.chop(1);				// remove trailing ')'
							
							d_touch_speed = text.toDouble();
						}
						else if(arguments.startsWith(QStringLiteral("Tool.GoToPar.Speed(")))
						{
							text = arguments.mid(19);	// remove 'Tool.GoToPar.Speed('
							text.chop(1);				// remove trailing ')'
							
							d_move_speed = text.toDouble();
						}
						else
						{
							tool_offset_changed = false;
							p1 = d_tool_offset;

							expression_match = expression_tool_x.match(arguments);
							if(expression_match.hasMatch())
							{
								list = expression_match.capturedTexts();
																
								if(list.size() == 2)
								{
									d_tool_offset.x = list[1].toDouble();
									tool_offset_changed = true;
								}
							}
							
							expression_match = expression_tool_y.match(arguments);
							if(expression_match.hasMatch())
							{
								list = expression_match.capturedTexts();
								
								if(list.size() == 2)
								{
									d_tool_offset.y = list[1].toDouble();
									tool_offset_changed = true;
								}
							}
							
							expression_match = expression_tool_z.match(arguments);
							if(expression_match.hasMatch())
							{
								list = expression_match.capturedTexts();
								
								if(list.size() == 2)
								{
									d_tool_offset.z = list[1].toDouble();
									d_tool_offset.z -= HEAD_FIXED_OFFSET_Z;
									tool_offset_changed = true;
								}
							}
							
							if(tool_offset_changed)
							{
								// takes into account the shift in position from a change in the tool offset
								pos = d_command_executor.Current_Position();
								pos += (d_tool_offset - p1);
								
								d_command_executor.Set_Current_Position(pos);

								emit Tool_Offset_Changed(d_tool_offset.x,d_tool_offset.y,d_tool_offset.z,d_active_tip_radius * 2.0);
								emit Position_Changed(pos.x,pos.y,pos.z);
								
								this->Set_Active_Tool_EOffset(d_tool_offset);
							}
						}

						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("StartSession")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						d_session_active = true;
						d_error_active = false;
						d_user_enabled = false;
						
						this->Create_Tool_Data();
						d_active_tool = d_tools[0];
						d_found_tool = d_tools[0];

						d_pt_meas_report_properties = this->Get_Report_Properties(QStringLiteral("X(),Y(),Z()"));
						
						this->Initialize_Alignment_Data();
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("StopDaemon")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						if(arguments.compare(d_position_daemon_command_tag) == 0)
						{
							d_position_daemon->stop();
						}
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("StopAllDaemons")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						if(arguments.compare(d_position_daemon_command_tag) == 0)
						{
							d_position_daemon->stop();
						}
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else if(keyword.compare(QStringLiteral("SetCoordSystem")) == 0)
					{
						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						alignment_data = this->Find_Alignment(arguments,&exists);
						
						if(exists)
						{
							d_active_alignment = alignment_data;
						}
						else
						{
							d_active_alignment = this->Find_Alignment(QStringLiteral("MachineCsy"),&exists);
						}
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
					else
					{
#ifdef DEBUG_MODE_LEVEL_0
						qWarning(QString("IGND: %1").arg(line).toLatin1());
#endif

						text = QString("%1 &\r\n").arg(command_tag);
						this->Send(text.toLatin1());
						
						text = QString("%1 %\r\n").arg(command_tag);
						this->Send(text.toLatin1());
					}
				}
			}
		}
		
	}while(!(rx_line_index < 0));
}

QByteArray TMachineIpp::KeyPress_Done(void) const
{
	QByteArray							data("E0000 # KeyPress(\"F1\")\r\n");
	
	return data;
}

QByteArray TMachineIpp::KeyPress_DelPnt(void) const
{
	QByteArray							data("E0000 # KeyPress(\"F2\")\r\n");
	
	return data;
}

void TMachineIpp::Set_EStop(
	const bool 							state)
{
	d_estop_status = state;
	
	if(d_estop_status)
	{
		d_command_executor.Abort();
		
		d_jog_timer->stop();
		d_control_timer->stop();
		
		this->Send("E0000 ! Error(3, 0500,\"\", \"Emergency stop\")\r\n");

		d_error_active = true;
	}
}

bool TMachineIpp::Move_To(
	const TVector3						&pos)
{
	TCommandExecutor::TCommand			command;
	TVector3							target(pos);
	
	if(d_estop_status)
	{
		d_last_error = QStringLiteral("ERR:  Machine in E-Stop");
		return false;
	}

	target -= d_tool_offset;

	if(target.x < d_min_machine.x) target.x = d_min_machine.x;
	if(target.y < d_min_machine.y) target.y = d_min_machine.y;
	if(target.z < d_min_machine.z) target.z = d_min_machine.z;
	if(target.x > d_max_machine.x) target.x = d_max_machine.x;
	if(target.y > d_max_machine.y) target.y = d_max_machine.y;
	if(target.z > d_max_machine.z) target.z = d_max_machine.z;
	
	target += d_tool_offset;
	
	command.command_type = TCommandExecutor::COMMAND_MOVE;
	command.target = pos;
	
	command.speed = d_move_speed;
	
	command.completion_text = QString();
	
	d_command_executor.Add_Command(command);
	
	if(!d_jog_timer->isActive())
	{
		d_jog_timer->start(TIMER_INTERVAL);
	}
	
	return true;
}

bool TMachineIpp::Manual_Touch(
	const TVector3 						&pos,
	const TVector3 						&vec)
{
	TCommandExecutor::TCommand			command;
	QString								text;
	TVector3							touch_pos(pos);
	TVector3							touch_vec(vec);
	TVector3							clear_pos;
	bool								x_error,y_error,z_error;
	static const double					ZERO_EPSILON(0.001);
	
	if(d_estop_status)
	{
		d_last_error = QStringLiteral("ERR:  Machine in E-Stop");
		return false;
	}

	// scale errors
	touch_pos.x *= d_scale_factor_x;
	touch_pos.y *= d_scale_factor_y;
	touch_pos.z *= d_scale_factor_z;
		
	// squareness errors
	touch_pos.x += pos.y * d_squareness_xy;
	touch_pos.y += pos.z * d_squareness_yz;
	touch_pos.x += pos.z * d_squareness_zx;

	
	if(touch_vec.Length() > ZERO_EPSILON)
	{
		touch_vec.Normal();
	}
	else
	{
		d_last_error = QStringLiteral("ERR:  Touch vector not defined");
		return false;
	}
	
	text = QString("Touch %1, %2, %3, %4, %5, %6")
				.arg(touch_pos.x,0,'f',3)
				.arg(touch_pos.y,0,'f',3)
				.arg(touch_pos.z,0,'f',3)
				.arg(touch_vec.i,0,'f',5)
				.arg(touch_vec.j,0,'f',5)
				.arg(touch_vec.k,0,'f',5);
	
	emit Log_Text(text);
	
	clear_pos = touch_pos;
	
	// touch IJK is approach but machine expects surface normal
	touch_vec *= -1.0;
	
	touch_pos = this->Convert_Pos_To_Part_Csy(touch_pos);
	touch_vec = this->Convert_Vector_To_Part_Csy(touch_vec);
	
	text = QString("E0000 # PtMeas(X(%1),Y(%2),Z(%3),IJK(%4,%5,%6)\r\n")
				.arg(touch_pos.x,0,'f',5)
				.arg(touch_pos.y,0,'f',5)
				.arg(touch_pos.z,0,'f',5)
				.arg(touch_vec.i,0,'f',5)
				.arg(touch_vec.j,0,'f',5)
				.arg(touch_vec.k,0,'f',5);
	
	if(this->Is_Position_Inside_Volume(clear_pos,&x_error,&y_error,&z_error))
	{
		command.command_type = TCommandExecutor::COMMAND_MOVE;
		command.target = clear_pos;
		command.speed = d_jog_highspeed;
		command.completion_text = text;
		
		d_command_executor.Add_Command(command);
		
		if(!d_control_timer->isActive())
		{
			d_control_timer->start(TIMER_INTERVAL);
		}
	}
	else
	{
		text = QString();
		
		if(x_error)
		{
			text += QStringLiteral("X");
		}
		
		if(y_error)
		{
			text += QStringLiteral("Y");
		}
		
		if(z_error)
		{
			text += QStringLiteral("Z");
		}
		
		d_last_error = QString("ERR:  Axis %1 outside of machine volume.").arg(text);
		
		
		return false;
	}
	
	return true;
}

bool TMachineIpp::Jog_Move(
	const TVector3						&direction)
{
	TCommandExecutor::TCommand			command;
	TVector3							target(d_command_executor.Current_Position());
	
	if(d_estop_status)
	{
		d_last_error = QStringLiteral("ERR:  Machine in E-Stop");
		return false;
	}
	
	if(d_control_timer->isActive())
	{
		d_last_error = QStringLiteral("ERR:  Machine currently active");
		return false;
	}
	
	target += (d_jog_distance * direction);

	target -= d_tool_offset;
	
	if(target.x < d_min_machine.x) target.x = d_min_machine.x;
	if(target.y < d_min_machine.y) target.y = d_min_machine.y;
	if(target.z < d_min_machine.z) target.z = d_min_machine.z;
	if(target.x > d_max_machine.x) target.x = d_max_machine.x;
	if(target.y > d_max_machine.y) target.y = d_max_machine.y;
	if(target.z > d_max_machine.z) target.z = d_max_machine.z;
	
	target += d_tool_offset;
	
	command.command_type = TCommandExecutor::COMMAND_MOVE;
	command.target = target;
	
	command.speed = d_move_speed;
	
	command.completion_text = QString();
	
	d_command_executor.Add_Command(command);
	
	if(!d_jog_timer->isActive())
	{
		d_jog_timer->start(TIMER_INTERVAL);
	}
	
	return true;
}

void TMachineIpp::Position_Report_Time(void)
{
	QString								text;
	TVector3							pos(d_command_executor.Current_Position());
	
	pos = this->Convert_Pos_To_Part_Csy(pos);

	text = d_position_daemon_command_tag;
	text += QStringLiteral(" # ");
	text += this->Format_Property(d_position_report_properties,pos);
	text += QStringLiteral("\r\n");
	
	d_last_position = pos;
	
	this->Send(text.toLatin1());
}


void TMachineIpp::Position_Report_Distance(void)
{
	TVector3							pnt;
	TVector3							difference;
	QString								text;

	if(d_position_daemon->isActive())
	{
		pnt = d_command_executor.Current_Position();
		
		pnt = this->Convert_Pos_To_Part_Csy(pnt);
		
		difference = pnt - d_last_position;
		
		if(difference.Length() > d_position_distance)
		{
			text = d_position_daemon_command_tag;
			text += QStringLiteral(" # ");
			text += this->Format_Property(d_position_report_properties,pnt);
			text += QStringLiteral("\r\n");
			
			this->Send(text.toLatin1());
			
			d_last_position = pnt;
		}
	}
}

bool TMachineIpp::Process_Line(
	const QString 						&line,
	QString 							* const tag,
	QString 							* const remainder)
{
	assert(tag);
	assert(remainder);
	
	if(line.length() < 7)
	{
		return false;
	}
	
	if(line[0] == QChar('E') || line[0] == QChar('0'))
	{
		*tag = line.mid(0,5);					// E0001 or 00001
		*remainder = line.mid(6).trimmed();
		
		return true;
	}
	
	return false;
}

void TMachineIpp::Create_Tool_Data(void)
{
	TMachineIpp::TToolData				tool_data;
	QString								text;
	double								head_a;
	double								head_b;
	int									cntr_a;
	int									cntr_b;
	int									count_a;
	int									count_b;
	static const double					ROTATIONAL_OFFSET(116.520);
	static const double					RANGE_A(1.832595714594046);		// 105
	static const double					RANGE_B(6.283185307179586);		// 360
	static const double					INCREMENT(0.785398163397448);	// 45
	static const double					RAD2DEG(57.295779513082321);
	
	d_tools.clear();
	
	tool_data.name = QStringLiteral("NoTool");
	tool_data.name_hint = QStringLiteral("NOTOOL");
	tool_data.offset.Set(0,0,0);
	tool_data.e_offset = tool_data.offset;
	tool_data.vector.Set(0,0,-1);
	tool_data.head_a = 0;
	tool_data.head_b = 0;
	tool_data.tip_diameter = 0;
	d_tools.push_back(tool_data);
	
	tool_data.tip_diameter = 4.000;

	tool_data.name = QStringLiteral("RefTool");
	tool_data.name_hint = QStringLiteral("REFTOOL");
	tool_data.offset.Set(0,0,-195.07);
	tool_data.e_offset = tool_data.offset;
	tool_data.vector.Set(0,0,-1);
	tool_data.head_a = 0;
	tool_data.head_b = 0;
	d_tools.push_back(tool_data);
	
	tool_data.name = QStringLiteral("Tool.A0.0_B0.0");
	tool_data.name_hint = QStringLiteral("T1A0B0");
	tool_data.offset.Set(0,0,0 - HEAD_FIXED_OFFSET_Z - ROTATIONAL_OFFSET);
	tool_data.e_offset = tool_data.offset;
	tool_data.vector.Set(0,0,-1);
	tool_data.head_a = 0;
	tool_data.head_b = 0;
	d_tools.push_back(tool_data);
	
	count_a = 1 + static_cast<int>(0.5 + (RANGE_A - INCREMENT) / INCREMENT);
	count_b = 1 + static_cast<int>(0.5 + RANGE_B / INCREMENT);

	head_a = INCREMENT;
	
	for(cntr_a = 0;cntr_a < count_a;++cntr_a)
	{
		head_b = -1.0 * (RANGE_B / 2.0);

		for(cntr_b = 0;cntr_b < count_b;++cntr_b)
		{
			tool_data.head_a = head_a * RAD2DEG;
			tool_data.head_b = head_b * RAD2DEG;
			
			tool_data.name = QString("Tool.A%1_B%2").arg(tool_data.head_a,0,'f',1).arg(tool_data.head_b,0,'f',1);
			tool_data.name_hint = QString("T1A%1B%2").arg(tool_data.head_a,0,'f',0).arg(tool_data.head_b,0,'f',0);
			
			tool_data.vector.x = sin(head_b) * sin(head_a);
			tool_data.vector.y = -1.0 * cos(head_b) * sin(head_a);
			tool_data.vector.z = -1.0 * cos(head_a);
			
			tool_data.offset = tool_data.vector * ROTATIONAL_OFFSET;
			tool_data.offset.z -= HEAD_FIXED_OFFSET_Z;
			
			tool_data.e_offset = tool_data.offset;
			
			d_tools.push_back(tool_data);

			head_b += INCREMENT;
		}
		
		head_a += INCREMENT;
	}
}

void TMachineIpp::Start_Position_Report(
	const QString						&command_tag,
	const QString						&arguments)
{
	QString								argument_text(arguments.trimmed());
	QString								text;
	int									start_index;
	int									end_index;
	double								dval;
	
	d_position_daemon->stop();
	
	argument_text.remove(' ');

	start_index = argument_text.indexOf(QStringLiteral("Time("));
	
	if(!(start_index < 0))
	{
		start_index += 5;	// remove 'Time('
		
		end_index = argument_text.indexOf(QStringLiteral(")"),start_index);
		
		if(end_index > start_index)
		{
			text = argument_text.mid(start_index,end_index - start_index);
			dval = text.toDouble();
			
			d_position_time = static_cast<int>(dval * 1000);
			
		}
	}
	
	start_index = argument_text.indexOf(QStringLiteral("Dis("));
	
	if(!(start_index < 0))
	{
		start_index += 4;	// remove 'Dis('
		
		end_index = argument_text.indexOf(QStringLiteral(")"),start_index);
		
		if(end_index > start_index)
		{
			text = argument_text.mid(start_index,end_index - start_index);
			d_position_distance = text.toDouble();
		}
	}
	
	if(d_position_time < 100) d_position_time = 100;
	if(d_position_distance < 0.1) d_position_distance = 0.1;
	
	d_position_daemon_command_tag = command_tag;
	d_position_daemon->start(d_position_time);
}

void TMachineIpp::Initialize_Alignment_Data(void)
{
	TMachineIpp::TAlignmentData			alignment_data;
	
	d_alignments.clear();
	
	alignment_data.name_is_enum = true;
	alignment_data.csy_alpha = 0.0;
	alignment_data.csy_beta = 0.0;
	alignment_data.csy_gamma = 0.0;
	alignment_data.csy_mat.Identity();
	alignment_data.csy_origin.Set(0,0,0);
	
	alignment_data.name = QStringLiteral("MachineCsy");
	d_alignments.push_back(alignment_data);
	d_active_alignment = alignment_data;
	
	alignment_data.name = QStringLiteral("MovableMachineCsy");
	d_alignments.push_back(alignment_data);
	
	alignment_data.name = QStringLiteral("MultipleArmCsy");
	d_alignments.push_back(alignment_data);
	
	alignment_data.name = QStringLiteral("RotaryTableVarCsy");
	d_alignments.push_back(alignment_data);
	
	alignment_data.name = QStringLiteral("RotaryTableFixCsy");
	d_alignments.push_back(alignment_data);
	
	alignment_data.name = QStringLiteral("PartCsy");
	d_alignments.push_back(alignment_data);
	
	alignment_data.name = QStringLiteral("JogDisplayCsy");
	d_alignments.push_back(alignment_data);
	
	alignment_data.name = QStringLiteral("JogMoveCsy");
	d_alignments.push_back(alignment_data);

	alignment_data.name = QStringLiteral("SensorCsy");
	d_alignments.push_back(alignment_data);

}

void TMachineIpp::Update_Alignment_Data(
	const TAlignmentData				&alignment_data)
{
	std::vector<TMachineIpp::TAlignmentData>::iterator iter;
	
	for(iter = d_alignments.begin();iter != d_alignments.end();++iter)
	{
		if(alignment_data.name.compare((*iter).name) == 0)
		{
			(*iter) = alignment_data;
			return;
		}
	}
	
	d_alignments.push_back(alignment_data);
}

void TMachineIpp::Delete_Alignment_Data(
	const QString						&name)
{
	std::vector<TMachineIpp::TAlignmentData>::iterator iter;
	
	for(iter = d_alignments.begin();iter != d_alignments.end();++iter)
	{
		if(name.compare((*iter).name) == 0)
		{
			d_alignments.erase(iter);
			return;
		}
	}
}

TMachineIpp::TAlignmentData TMachineIpp::Find_Alignment(
	const QString						&name,
	bool								* const exists) const
{
	std::vector<TMachineIpp::TAlignmentData>::const_iterator iter;
	TMachineIpp::TAlignmentData			machine_csy;
	
	assert(exists);
	
	*exists = false;
	
	for(iter = d_alignments.begin();iter != d_alignments.end();++iter)
	{
		if(name.compare((*iter).name) == 0)
		{
			*exists = true;
			return (*iter);
		}
		else if((*iter).name.compare(QStringLiteral("MachineCsy")) == 0)
		{
			machine_csy = (*iter);
		}
	}
	
	return machine_csy;
}

std::vector<TMachineIpp::TDataProperty> TMachineIpp::Get_Report_Properties(
	const QString						&arguments) const
{
	std::vector<TMachineIpp::TDataProperty> properties;
	QStringList							list;
	QStringList::const_iterator			iter;
	QString								argument_text(arguments.trimmed());
	
	argument_text.remove(' ');
	list = argument_text.split(',');
	
	for(iter = list.begin();iter != list.end();++iter)
	{
		if((*iter).compare(QStringLiteral("X()")) == 0)
		{
			properties.push_back(TMachineIpp::PROPERTY_X);
		}
		else if((*iter).compare(QStringLiteral("Y()")) == 0)
		{
			properties.push_back(TMachineIpp::PROPERTY_Y);
		}
		else if((*iter).compare(QStringLiteral("Z()")) == 0)
		{
			properties.push_back(TMachineIpp::PROPERTY_Z);
		}
		else if((*iter).compare(QStringLiteral("Tool.A()")) == 0)
		{
			properties.push_back(TMachineIpp::PROPERTY_TOOL_A);
		}
		else if((*iter).compare(QStringLiteral("Tool.B()")) == 0)
		{
			properties.push_back(TMachineIpp::PROPERTY_TOOL_B);
		}
		else if((*iter).compare(QStringLiteral("IJK()")) == 0)
		{
			properties.push_back(TMachineIpp::PROPERTY_IJK);
		}
		else if((*iter).compare(QStringLiteral("IJKAct()")) == 0)
		{
			properties.push_back(TMachineIpp::PROPERTY_IJKACT);
		}
		else if((*iter).compare(QStringLiteral("ER()")) == 0)
		{
			properties.push_back(TMachineIpp::PROPERTY_ER);
		}
		else if((*iter).compare(QStringLiteral("Q()")) == 0)
		{
			properties.push_back(TMachineIpp::PROPERTY_Q);
		}
		else if((*iter).compare(QStringLiteral("R()")) == 0)
		{
			properties.push_back(TMachineIpp::PROPERTY_R);
		}
	}
	
	return properties;
}

std::vector<double> TMachineIpp::Get_Argument_Values(
	const QString						&arguments) const
{
	std::vector<double>					values;
	QStringList							list;
	QStringList::const_iterator			iter;
	
	list = arguments.split(',');
	
	for(iter = list.begin();iter != list.end();++iter)
	{
		values.push_back((*iter).toDouble());
	}
	
	return values;
}

QString TMachineIpp::Format_Property(
	const std::vector<TDataProperty>	&properties,
	const TVector3						&pos,
	const TVector3						&vec) const
{
	std::vector<TMachineIpp::TDataProperty>::const_iterator iter;
	QString								text;
	
	for(iter = properties.begin();iter != properties.end();++iter)
	{
		switch(*iter)
		{
			case TMachineIpp::PROPERTY_X:
				text += QString("X(%1),").arg(pos.x,0,'f',5);
				break;
				
			case TMachineIpp::PROPERTY_Y:
				text += QString("Y(%1),").arg(pos.y,0,'f',5);
				break;
				
			case TMachineIpp::PROPERTY_Z:
				text += QString("Z(%1),").arg(pos.z,0,'f',5);
				break;
				
			case TMachineIpp::PROPERTY_TOOL_A:
				text += QString("Tool.A(%1),").arg(d_head_angle_a,0,'f',1);
				break;
				
			case TMachineIpp::PROPERTY_TOOL_B:
				text += QString("Tool.B(%1),").arg(d_head_angle_b,0,'f',1);
				break;
				
			case TMachineIpp::PROPERTY_IJK:
				text += QString("IJK(%1,%2,%3),").arg(vec.i,0,'f',9).arg(vec.j,0,'f',9).arg(vec.k,0,'f',9);
				break;
				
			case TMachineIpp::PROPERTY_IJKACT:
				text += QString("IJKAct(%1,%2,%3),").arg(vec.i,0,'f',9).arg(vec.j,0,'f',9).arg(vec.k,0,'f',9);
				break;
				
			case TMachineIpp::PROPERTY_ER:
				text += QString("ER(%1),").arg(d_active_tip_radius,0,'f',5);
				break;
				
			case TMachineIpp::PROPERTY_Q:
				text += QStringLiteral("Q(0),");
				break;
				
			case TMachineIpp::PROPERTY_R:
				text += QStringLiteral("R(0.00000),");
				break;
		}
	}
	
	text.chop(1);	// remove trailing comma
	
	return text;
}

TMachineIpp::TToolData TMachineIpp::Find_Tool(
	const QString						&name) const
{
	std::vector<TMachineIpp::TToolData>::const_iterator iter;
	QRegularExpression					name_exp("[[A]([0-9.]{1,5})[-_. ]{0,1}[B]([0-9.]{1,5})");
	QRegularExpressionMatch				expression_match;
	QStringList							expression_text_list;
	TMachineIpp::TToolData				tool;
	QString								tool_name(name);
	double								tool_a;
	double								tool_b;
	
	tool_name.remove('\"');
	
	tool.name = QStringLiteral("Default");
	tool.head_a = 0;
	tool.head_b = 0;
	tool.tip_diameter = 0;
	tool.offset.Set(0,0,-195.07);
	tool.vector.Set(0,0,-1);
		
	for(iter = d_tools.begin();iter != d_tools.end();++iter)
	{
		if((*iter).name.compare(tool_name) == 0)
		{
			return (*iter);
		}
		
		if(tool_name.contains((*iter).name_hint))
		{
			return (*iter);
		}
	}
	
	// infered based on format Ax.xBx.x
	expression_match = name_exp.match(name);
	if(expression_match.hasMatch())
	{
		expression_text_list = expression_match.capturedTexts();
		
		if(expression_text_list.size() == 3)
		{
			tool_a = expression_text_list[1].toDouble();
			tool_b = expression_text_list[2].toDouble();
			

			return this->Interpolate_Tool(tool_a,tool_b);
		}
	}
	
	return tool;
}

TMachineIpp::TToolData TMachineIpp::Interpolate_Tool(
	const double						&head_a,
	const double						&head_b) const
{
	TMachineIpp::TToolData				tool_data;
	static const double					ROTATIONAL_OFFSET(116.520);
	static const double					RAD2DEG(57.295779513082321);
	
	tool_data.head_a = head_a;
	tool_data.head_b = head_b;
	
	tool_data.name = QString("Tool.A%1_B%2").arg(tool_data.head_a,0,'f',1).arg(tool_data.head_b,0,'f',1);
	tool_data.name_hint = QString("T1A%1B%2").arg(tool_data.head_a,0,'f',0).arg(tool_data.head_b,0,'f',0);
	
	tool_data.vector.x = sin(head_b / RAD2DEG) * sin(head_a / RAD2DEG);
	tool_data.vector.y = -1.0 * cos(head_b /RAD2DEG) * sin(head_a / RAD2DEG);
	tool_data.vector.z = -1.0 * cos(head_a / RAD2DEG);
	
	tool_data.offset = tool_data.vector * ROTATIONAL_OFFSET;
	tool_data.offset.z -= HEAD_FIXED_OFFSET_Z;
	
	tool_data.e_offset = tool_data.offset;

	tool_data.tip_diameter = 4.000;
	
	return tool_data;
}

TVector3 TMachineIpp::Convert_Pos_To_Part_Csy(
	const TVector3						&pos) const
{
	TVector3							new_pos;
	
	new_pos = pos - d_active_alignment.csy_origin;
	new_pos = (d_active_alignment.csy_mat * new_pos);
	
	return new_pos;
}

TVector3 TMachineIpp::Convert_Vector_To_Part_Csy(
	const TVector3						&vec) const
{
	return d_active_alignment.csy_mat * vec;
}

TVector3 TMachineIpp::Convert_Pos_From_Part_Csy(
	const TVector3						&pos) const
{
	TVector3							new_pos;
	TMat4								inv_mat;
	
	inv_mat = d_active_alignment.csy_mat;
	inv_mat.Transpose();
	
	new_pos = (inv_mat * pos);
	new_pos += d_active_alignment.csy_origin;
	
	return new_pos;
}

TVector3 TMachineIpp::Convert_Vector_From_Part_Csy(
	const TVector3						&vec) const
{
	TMat4								inv_mat;

	inv_mat = d_active_alignment.csy_mat;
	inv_mat.Transpose();

	return inv_mat * vec;
}

bool TMachineIpp::Get_Move_Data(
	const QString						&arguments,
	TVector3							* const pos) const
{
	QStringList							list;
	QStringList::const_iterator			iter;
	QString								argument_text(arguments.trimmed());
	QString								text;
	
	assert(pos);
	
	*pos = d_command_executor.Current_Position();
	
	argument_text.remove(' ');
	list = argument_text.split(',');
	
	for(iter = list.begin();iter != list.end();++iter)
	{
		if((*iter).startsWith(QStringLiteral("X(")))
		{
			text = (*iter).mid(2);
			text.chop(1);
			
			(*pos).x = text.toDouble();
		}
		else if((*iter).startsWith(QStringLiteral("Y(")))
		{
			text = (*iter).mid(2);
			text.chop(1);
			
			(*pos).y = text.toDouble();
		}
		else if((*iter).startsWith(QStringLiteral("Z(")))
		{
			text = (*iter).mid(2);
			text.chop(1);
			
			(*pos).z = text.toDouble();
		}
	}
	
	return true;
}

bool TMachineIpp::Get_Touch_Data(
	const QString						&arguments,
	TVector3							* const pos,
	TVector3							* const ijk) const
{
	QStringList							list;
	TVector3							pnt;
	QString								argument_text(arguments.trimmed());
	QString								token;
	QString								text;
	int									index;
	static const double					ZERO_EPSILON(0.001);
	
	assert(pos);
	assert(ijk);
		
	*pos = d_command_executor.Current_Position();
	
	argument_text.remove(' ');
		
	if(!argument_text.length())
	{
		return false;
	}
	
	do
	{
		index = argument_text.indexOf(')');
		token = argument_text;
		
		if(!(index < 0))
		{
			token.truncate(index);
			argument_text.remove(0,index + 1);
		}
		
		if(argument_text.length())
		{
			if(argument_text[0] == QChar(','))
			{
				argument_text.remove(0,1);
			}
		}
		else
		{
			index = -1;
		}
	
		if(token.startsWith(QStringLiteral("X(")))
		{
			text = token.mid(2);
			(*pos).x = text.toDouble();
		}
		else if(token.startsWith(QStringLiteral("Y(")))
		{
			text = token.mid(2);
			(*pos).y = text.toDouble();
		}
		else if(token.startsWith(QStringLiteral("Z(")))
		{
			text = token.mid(2);
			(*pos).z = text.toDouble();
		}
		else if(token.startsWith(QStringLiteral("IJK(")))
		{
			text = token.mid(4);
			
			list = text.split(',');
			
			if(list.size() > 2)
			{
				if(list.size() > 2)
				{
					(*ijk).i = list[0].toDouble();
					(*ijk).j = list[1].toDouble();
					(*ijk).k = list[2].toDouble();
				}
			}
		}
	}
	while(!(index < 0));	

	if(ijk->Length() < ZERO_EPSILON)
	{
		ijk->Set(0,0,1);
	}
	
	ijk->Normal();
	
	
	pnt = (*pos);
		
	// scale errors
	(*pos).x *= d_scale_factor_x;
	(*pos).y *= d_scale_factor_y;
	(*pos).z *= d_scale_factor_z;

	// squareness errors
	(*pos).x += pnt.y * d_squareness_xy;
	(*pos).y += pnt.z * d_squareness_yz;
	(*pos).x += pnt.z * d_squareness_zx;

	return true;
}

void TMachineIpp::Set_Active_Tool_EOffset(
	const TVector3 						&e_offset)
{
	std::vector<TToolData>::iterator 	iter;
	
	for(iter = d_tools.begin();iter != d_tools.end();++iter)
	{
		if((*iter).name.compare(d_active_tool.name) == 0)
		{
			(*iter).e_offset = e_offset;
			break;
		}
	}
	
	d_active_tool.e_offset = e_offset;
}
